/*===========================================================================
  PCI.C

  Functions to handle Peripheral Component Interconnect (PCI) devices

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
  =========================================================================*/

#include <stdio.h>
#include <dos.h>
#include "pci.h"

#define TRUE 1
#define FALSE 0

/* ------------------------------------------------------------------------
   pci_bios_present - determines if the PCI BIOS support is installed.

   This function permits the caller to determine if the PCI BIOS interface
   function is present. It also allows the caller to determine the interface
   version.

   INPUT:
     lppci : far pointer to a PCI_INFO structure which will be filled with
           information about the PCI system.

   OUTPUT:
     Returns TRUE if a PCI BIOS is present, FALSE otherwise.
------------------------------------------------------------------------ */
int pci_bios_present (PCI_INFO far *lppci)
{
    union REGS regs;

    regs.h.ah = PCI_FUNCTION_ID;
    regs.h.al = PCI_BIOS_PRESENT;
    int86 (0x1A, &regs, &regs);

    if (!(regs.h.dl == 'P' && regs.h.dh == 'C')) return FALSE;
    lppci->present_status = regs.h.ah;
    lppci->hw_mechanism = regs.h.al;
    lppci->ver_maj = regs.h.bh;
    lppci->ver_min = regs.h.bl;
    lppci->pci_bus_num = regs.h.cl;
    return TRUE;
}

/* ------------------------------------------------------------------------
   pci_find_device - determines the device numbers assigned to a device.

   This function will return the Device Number, Bus Number and Function
   Number of the Nth Device/Function whose Vendor ID, Device ID, and Index
   Number (N) match the input parameters. The device number and function
   number may be used to access the device's configuration space, which
   contains device information such as the interrupt line number assigned
   to the device.

   INPUT:
     vendor_id : Vender specific identification number.
     device_id : Device identification number.
     index :     Device ndex number.
     lpdevice :  far pointer to a PCI_DEVICE_INFO structure.

   OUTPUT:
     Returns SUCCESSFUL if successful, DEVICE_NOT_FOUND otherwise.
------------------------------------------------------------------------ */
int pci_find_device (unsigned int device_id, unsigned int vendor_id,
                     unsigned int index, PCI_DEVICE_INFO far *lpdevice)
{
    union REGS regs;

    regs.h.ah = PCI_FUNCTION_ID;
    regs.h.al = FIND_PCI_DEVICE;
    regs.x.cx = device_id;
    regs.x.dx = vendor_id;
    regs.x.si = index;
    int86 (0x1A, &regs, &regs);

    if (regs.h.ah == SUCCESSFUL)
    {
        lpdevice->bus_num = regs.h.bh;
        lpdevice->device_num = regs.h.bl;
    }
    return ((int) regs.h.ah);
}


/* ------------------------------------------------------------------------
   pci_read_config_byte - read byte from configuration space.

   This function allows the caller to read a specific byte from the
   configuration space identified by the device and bus number.

   INPUT:
     reg_num :  offset into configuration space for byte to be read.
     lpdevice : far pointer to a PCI_DEVICE_INFO structure. This
                structure must first be filled by the pci_find_device
                function in order to access the correct configuration
                space.

   OUTPUT:
     Returns register value if successful, BAD_REGISTER_NUMBER otherwise.
------------------------------------------------------------------------ */
unsigned char pci_read_config_byte (unsigned int reg_num,
                                    PCI_DEVICE_INFO far *lpdevice)
{
    union REGS regs;
    unsigned char result;

    regs.h.ah = PCI_FUNCTION_ID;
    regs.h.al = READ_CONFIG_BYTE;
    regs.h.bh = lpdevice->bus_num;
    regs.h.bl = lpdevice->device_num;
    regs.x.di = reg_num;
    int86 (0x1A, &regs, &regs);

    if (regs.h.ah == SUCCESSFUL)
    {
        result = regs.h.cl;
    }
    else result = regs.h.ah;

    return result;
}


