/*==========================================================================
* PGLDEMO.C - Copyright (c) 1993-94 ATI Technologies Inc. All rights reserved*
*                                                                          *
* Sample program that uses the ATI PGL functions.                          *
* ======================================================================== */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include <time.h>
#include <malloc.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pgldemo.h"


/*
 *****************************************************************************
 */

/*
 * GLOBAL VARIABLES
 */

#define MAX_STRING_WIDTH    80

#define FULL_STATUS         0
#define DETECTONLY_STATUS   1

int      MaxColor;

PGL_font *smallfont;     // font pointers
PGL_font *normalfont;
PGL_font *largefont;
PGL_font *extrafont;

PGL_modeconfig modeinfo;
PGL_queryheader queryinfo;

int UserMode;            // used to hold mode code selected by user
char ModeTitle[80];      // title string for options screen
int XCenter, YCenter;    // holds coordinates for the center of the screen
int RealBPP;             // real bits per pixel value

int MinRX, MaxRX, MinRY, MaxRY;     // minimum and maximum coords for 
                                    // random demo display area

int *Patterns, TotalPatterns;

long fillpattern[15];

/*
 *****************************************************************************
 */

void get_key_code(int *ascii_code, int *scan_code)
{
    union REGS regs;

    regs.x.ax = 0;
    int86(0x16, &regs, &regs);
    *ascii_code = (int)(regs.h.al);
    *scan_code = (int)(regs.h.ah);
}

/*
 *****************************************************************************
 */

void ComingSoon(char *Banner)
   //
   // ComingSoon - displays the coming soon banner
   //
   {
   // display the Banner
   PGL_setfgcolor(PGL_getcolorcode(YELLOW));
   PGL_text((modeinfo.xres - (strlen(Banner)*largefont->charwidth))/2,
            (modeinfo.yres-20)/2,
            Banner, largefont);
   }

/*
 *****************************************************************************
 */

void InitDemoScreen(char *ModeTitle, char *SecondStr, int Color)
   //
   // InitDemoScreen -- sets up the elements common to both the options and
   //                   demo screens
   //
   {  
   /* draw a full screen white solid line border */
   PGL_setlinetype(SOLID_LINE);
   PGL_setfgcolor(PGL_getcolorcode(WHITE));
   PGL_moveto(0, 0);
   PGL_drawto(modeinfo.xres - 1, 0);
   PGL_drawto(modeinfo.xres - 1, modeinfo.yres - 1);
   PGL_drawto(0, modeinfo.yres - 1);
   PGL_drawto(0, 0);

   /* fill screen (inside border) with BLUE */
   PGL_setfgcolor(PGL_getcolorcode(Color));
   PGL_setbgcolor(PGL_getcolorcode(Color));
   PGL_drawrect(1, 1, modeinfo.xres - 2, modeinfo.yres - 2);

   // display options screen title: mode, pixel depth
   PGL_setfgcolor(PGL_getcolorcode(WHITE));

   PGL_text((modeinfo.xres - (strlen(ModeTitle)*normalfont->charwidth))/2, 1, ModeTitle, normalfont);

   // print the second string
   PGL_text((modeinfo.xres - (strlen(SecondStr)*normalfont->charwidth))/2, 16, SecondStr, normalfont);
   }

/*
 *****************************************************************************
 */
void RandMessage(void)
   //
   // RandMessage -- displays the standard 2 line random demo message
   //
   {
   char SpaceBar[] = "Press the SPACE BAR to clear the display, or P to pause it.";
   char AnyKey[] = "Press any other key to exit.";

   // set the text color
   PGL_setfgcolor(PGL_getcolorcode(DARKGREEN));

   // display the SPACE BAR message
   PGL_text((modeinfo.xres - (strlen(SpaceBar)*normalfont->charwidth))/2,
            modeinfo.yres-28,
            SpaceBar, normalfont);

   // set the text color
   PGL_setfgcolor(PGL_getcolorcode(YELLOW));

   // display the press any other key to quit message
   PGL_text((modeinfo.xres - (strlen(AnyKey)*normalfont->charwidth))/2,
            modeinfo.yres-15,
            AnyKey, normalfont);
   }

/*
 *****************************************************************************
 */

void DemoMsg(char *Message, int Color)
   // 
   // DemoMsg -- prints the string Message at the bottom of the display
   //            followed by the "Press any key" string
   //
   {
   char PressStr[] = "Press Any Key to Continue";

   // set the text color
   PGL_setfgcolor(PGL_getcolorcode(WHITE));

   // display the passed in message
   PGL_text((modeinfo.xres - (strlen(Message)*normalfont->charwidth))/2,
            modeinfo.yres-(normalfont->charheight*2)-2,
            Message, normalfont);

   // display the press any key message
   PGL_text((modeinfo.xres - (strlen(PressStr)*normalfont->charwidth))/2,
            modeinfo.yres-normalfont->charheight-2,
            PressStr, normalfont);

   // wait for a key from the user
   getch();

   // erase message
   PGL_setfgcolor(PGL_getcolorcode(Color));
   PGL_drawrect(2, modeinfo.yres-(normalfont->charheight*2)-2,
                modeinfo.xres-4, (normalfont->charheight*2));
   }

/*
 *****************************************************************************
 */

void ShowOffFont(PGL_font *ShowFont, char *FontName, int RealBPP, int Auto, int DemoColour)
   //
   //
   // ShowFont -- pointer to the font struct holding the data for the BITMAP
   //             font to use.
   //
   // FontName -- pointer to a string describing the font.
   //
   // RealBPP  -- number of bits per pixel
   //
   // Auto     -- determines if auto delay is used (enabled for non-zero value)
   //
   {

   int Color;                 // used for cycling through colors
   int StartX;                // used to calc starting X coordinate
   int StartY;                // used to calc starting Y coordinate
   int DisplayLine;           // counter for current font display line
   int TotalLines;            // total lines of text to display
   int Character;             // counter for characters on display line
   int LineLength;            // holds # of chars per line
   int ClrY;

   unsigned char Start, End, i;        // used for automatic delay time
   int ch;                             // used for Pause check

   unsigned char CharStr[2] = {0,0};   // string for single character printing

   // calculate number of characters displayed per line for this font
   // (leaving a border of 2 characters on either side)
   LineLength = (modeinfo.xres / ShowFont->charwidth) - 4;

   // calculate starting Y position
   StartY = normalfont->charheight * 3;
   ClrY = StartY;

   // calculate number of lines of text
   TotalLines = (modeinfo.yres - StartY - (normalfont->charheight * 4)) / ShowFont->charheight;

   // display name of font 
   PGL_setfgcolor(PGL_getcolorcode(YELLOW));
   PGL_text((modeinfo.xres - (strlen(FontName) * ShowFont->charwidth))/2, 
            StartY, FontName, ShowFont);

   // point to next text line
   StartY += ShowFont->charheight;

   // calculate starting X position for font show off - 256 chars per line
   StartX = (modeinfo.xres - (LineLength * ShowFont->charwidth)) / 2;

   // loop through display lines, cycling through colors
   Color = DemoColour + 1;
   for(DisplayLine = 0; DisplayLine < TotalLines; DisplayLine++)
      {
      // loop through LineLength characters per line
      for(Character=0; Character < LineLength; Character++)
         {
         // set the color
         PGL_setfgcolor(PGL_getcolorcode(Color));

         // check if max color reached
         Color++;
         if (Color > MaxColor)
         {
            Color = 0;
         }
         if (Color == DemoColour)
         {
            Color++;
         }

         // print this character
         PGL_text(StartX + (Character * ShowFont->charwidth),StartY, 
                  CharStr, ShowFont);

         // increment next character in set
         CharStr[0]++;
         }
      // point to next text line
      StartY += ShowFont->charheight;
      }

   if (Auto == 0)
   {
      // wait for key press and then clean up text demo
      DemoMsg("Demonstrating Bitmap Text", DemoColour);
   }
   else
   {
      // use system tick count to wait for 1 second - 18.2 ticks
      Start = *((unsigned char *)(0x0000046cL));
      i = Start;
      End = (unsigned char)(Start + 18);
      while (i != End)
      {
         i = *((unsigned char *)(0x0000046cL));
      }

      // check for P key to pause
      if (kbhit() != 0)
      {
         ch = getch();
         if ((ch == 'p') || (ch == 'P'))
         {
             getch();
         }
      }
   }

   PGL_setfgcolor(PGL_getcolorcode(DemoColour));
   PGL_drawrect(1,
                ClrY,
                modeinfo.xres-2,
                modeinfo.yres-ClrY-2);
   }

/*
 *****************************************************************************
 */
int DemoFont(void)
{

    ShowOffFont(normalfont,"8x14 Bitmap Font",RealBPP,1,DARKBLUE);
    ShowOffFont(extrafont,"7x15 Bitmap Font",RealBPP,1,DARKBLUE);
    ShowOffFont(largefont,"12x20 Bitmap Font",RealBPP,1,DARKBLUE);
    ShowOffFont(smallfont,"8x8 Bitmap Font",RealBPP,1,DARKBLUE);

    return (0);
}

/*
 *****************************************************************************
 */

void SetDemoMode(void)
   //
   // SetDemoMode -- Perform additional steps required when a new mode is 
   //                set in the PGLDEMO program.
   //
   {
   char ModeWeight[80];

   // calculate the center of the screen
   XCenter = modeinfo.xres / 2;
   YCenter = modeinfo.yres / 2;

   // calculate the limits of the random display area for this mode
   MinRX = 3;                          // minimum X coordinate
   MaxRX = modeinfo.xres - 3;          // maximum X coordinate
   MinRY = 30;                         // minimum Y coordinate
   MaxRY = modeinfo.yres - 30;         // maximum Y coordinate

   // set maximum color
   MaxColor = WHITE;

   // set color depth
   RealBPP = modeinfo.bpp;

   // create mode title
   sprintf(ModeTitle,"DISPLAY MODE IS %d X %d, %d BPP ",modeinfo.xres, modeinfo.yres, RealBPP);
   switch(modeinfo.bpp)
   {
      case 4:
      case 8:
        break;

      case 16:
        sprintf(ModeWeight, "(%d)", modeinfo.depth);
        strcat(ModeTitle, ModeWeight);
        break;

      case 24:
        switch(modeinfo.depth)
        {
           case DEPTH_24_RGB:
              sprintf(ModeWeight, "(RGB)");
              break;

           case DEPTH_24_BGR:
              sprintf(ModeWeight, "(BGR)");
              break;
        }
        strcat(ModeTitle, ModeWeight);
        break;

      case 32:
        switch(modeinfo.depth)
        {
           case DEPTH_32_RGBA:
              sprintf(ModeWeight, "(RGBa)");
              break;

           case DEPTH_32_ABGR:
              sprintf(ModeWeight, "(aBGR)");
              break;

           case DEPTH_32_ARGB:
              sprintf(ModeWeight, "(aRGB)");
              break;

           case DEPTH_32_BGRA:
              sprintf(ModeWeight, "(BGRa)");
              break;
        }
        strcat(ModeTitle, ModeWeight);
        break;
   }

   /* set draw mixes - foreground = OVERPAINT, background = destination */
   PGL_setfgmix(S_MIX);
   PGL_setbgmix(D_MIX);
   PGL_setfgcolor(PGL_getcolorcode(WHITE));
   PGL_setbgcolor(PGL_getcolorcode(BLACK));
   }


/*
 *****************************************************************************
 */

void AfterTarga(void)
   {
   unsigned char Start, End, i;        // used for automatic delay time

   // use system tick count to wait for 1 second - 18.2 ticks
   Start = *((unsigned char *)(0x0000046cL));
   i = Start;
   End = (unsigned char)(Start + 18);
   while (i != End)
   {
      i = *((unsigned char *)(0x0000046cL));
   }

   // restore the palette
   if (modeinfo.bpp < 16)
      PGL_initpalette();
   }

/*
 *****************************************************************************
 */

void terminate(int exit_code)
{
    PGL_close();
    exit(exit_code);
}

/*
 *****************************************************************************
 */

void dump_status(int info_flag)
{
    // insure structure is uptodate
    PGL_getmodeconfig(&modeinfo);

    // display hardware status
    printf("Chip ID           : ");
    switch(modeinfo.chip_type)
    {
        case CHIP_GX_ID: printf("GX");      break;
        case CHIP_CX_ID: printf("CX");      break;
        case CHIP_CT_ID: printf("CT");      break;
        case CHIP_DT_ID: printf("DT");      break;
        default:         printf("Unknown (%04Xh)", modeinfo.chip_type);
                         break;
    }
    printf("\nChip version      : %02Xh", modeinfo.chip_version);
    printf("\nIO base address   : %04Xh", modeinfo.io_base_address);
    printf("\nIO base type      : ");
    switch(modeinfo.io_base_type)
    {
        case IO_TYPE_STANDARD: printf("Standard"); break;
        case IO_TYPE_FLOAT   : printf("Float"); break;
        default: printf("Unknown (%02Xh)", modeinfo.io_base_type); break;
    }
    printf("\nROM segment       : %04Xh (instance = %d)",
                                  modeinfo.rom_segment, modeinfo.instance);


    // dump this info only if the structures were filled properly
    if (info_flag == FULL_STATUS)
    {
        // insure structure is uptodate
        PGL_getqueryheader(&queryinfo);

        printf("\nLinear app address: %08lXh", modeinfo.aperture_address);
        printf("\nLinear app size   : ");
        switch(modeinfo.linear_aperture_size)
        {
            case APERTURE_4M_ENABLE: printf("4M");      break;
            case APERTURE_8M_ENABLE: printf("8M");      break;
            default                : printf("Unknown (%d)", modeinfo.linear_aperture_size);
                                     break;
        }
        printf("\nMemory size       : ");
        switch(queryinfo.memory_size)
        {
            case 0:  printf("512K"); break;
            case 1:  printf("1M");   break;
            case 2:  printf("2M");   break;
            case 3:  printf("4M");   break;
            case 4:  printf("6M");   break;
            case 5:  printf("8M");   break;
            default: printf("Unknown (%02Xh)", queryinfo.memory_size);
                     break;
        }
        printf("\nVGA type          : ");
        if (queryinfo.vga_type == 0)
        {
            printf("Disabled");
        }
        else
        {
            printf("Enabled");
        }
        printf("\nBUS type          : ");
        switch(queryinfo.bus_type)
        {
            case BUS_ISA:    printf("ISA");                 break;
            case BUS_EISA:   printf("EISA");                break;
            case BUS_VLB_NM: printf("VLB non-multiplexed"); break;
            case BUS_VLB:    printf("VLB");                 break;
            case BUS_PCI:    printf("PCI");                 break;
            default:         printf("Unknown (%02Xh)", queryinfo.bus_type);
                             break;
        }
        printf("\nDAC type          : ");
        switch(queryinfo.dac_type)
        {
            case DAC_INTERNAL:   printf("Internal");        break;
            case DAC_IBM514:     printf("IBM514");          break;
            case DAC_ATI68860_1: printf("ATI68860 rev B");  break;
            case DAC_ATI68860_2: printf("ATI68860 rev C");  break;
            case DAC_ATI68875:   printf("ATI68875");        break;
            case DAC_ATT20C490:  printf("ATT20C490");       break;
            case DAC_ATT20C498:  printf("ATT20C498");       break;
            case DAC_ATT21C498:  printf("ATT21C498");       break;
            case DAC_BT476:      printf("BT476");           break;
            case DAC_BT481:      printf("BT481");           break;
            case DAC_CH8398:     printf("CH8398");          break;
            case DAC_STG1700:    printf("STG1700");         break;
            case DAC_STG1702:    printf("STG1702");         break;
            case DAC_STG1703:    printf("STG1703");         break;
            case DAC_SC15021:    printf("SC15021");         break;
            case DAC_TVP3026_1:
            case DAC_TVP3026_2:  printf("TVP3026");         break;
            default:             printf("Unknown (%02Xh)", queryinfo.dac_type);
                                 break;
        }
        printf("\nColor Support     : %02Xh (4 & 8 bpp always supported)", queryinfo.color_depth_support);
        if ((queryinfo.color_depth_support & COLORSUPPORT_15) == COLORSUPPORT_15)
        {
            printf("\n  15 bpp (555)");
        }
        if ((queryinfo.color_depth_support & COLORSUPPORT_16) == COLORSUPPORT_16)
        {
            printf("\n  16 bpp (565)");
        }
        if ((queryinfo.color_depth_support & COLORSUPPORT_24_RGB) == COLORSUPPORT_24_RGB)
        {
            printf("\n  24 bpp (RGB)");
        }
        if ((queryinfo.color_depth_support & COLORSUPPORT_24_BGR) == COLORSUPPORT_24_BGR)
        {
            printf("\n  24 bpp (BGR)");
        }
        if ((queryinfo.color_depth_support & COLORSUPPORT_32_RGBA) == COLORSUPPORT_32_RGBA)
        {
            printf("\n  32 bpp (RGBa)");
        }
        if ((queryinfo.color_depth_support & COLORSUPPORT_32_ARGB) == COLORSUPPORT_32_ARGB)
        {
            printf("\n  32 bpp (aRGB)");
        }
        if ((queryinfo.color_depth_support & COLORSUPPORT_32_BGRA) == COLORSUPPORT_32_BGRA)
        {
            printf("\n  32 bpp (BGRa)");
        }
        if ((queryinfo.color_depth_support & COLORSUPPORT_32_ABGR) == COLORSUPPORT_32_ABGR)
        {
            printf("\n  32 bpp (aBGR)");
        }
    }
    printf("\n");
}

/*
 * START OF MAIN ROUTINE
 */

int main(int argc, char **argv)
   {
   char UserModeStr[2] = {0,0};  // used to hold ascii input code
   char MultiplePage[] = "Use PGUP/PGDN keys for other mode choices";
   char RandSelectStr[] = "Select a RANDOM Demo:";
   char StaticSelectStr[] = "Select a STATIC Demo:";
   char *SelectStrPtr;

   int i,j,LoopCount;   // all purpose loop counters
   int OptionX;         // holds x coordinate for options list
   int OptionY;         // holds y coordinate for options list
   int DemoOption;
   int OldMode;         // holds current mode when trying to set new one
   long DemoType;       // random or static demos
   int DemoColour;      // background colour for the demos
   unsigned long ntimes;
   unsigned long textcol;
   int ModesPerPage;
   int ModeIndex;
   int ascii_code;
   int scan_code;
   int multipleflag;

   // options list strings
   char OptA[] = "A. Patterned Lines";
   char OptB[] = "B. Patterned Outline Rectangles";
   char OptC[] = "C. Solid Filled Rectangles";
   char OptD[] = "D. Pattern Filled Rectangles";
   char OptE[] = "E. Patterned Outline Polygons";
   char OptF[] = "F. Solid Filled Polygons";
   char OptG[] = "G. Pattern Filled Polygons";
   char OptH[] = "H. Bitmapped Text";
   char OptI[] = "I. Display A Targa File Image";
   char OptJ[] = "J. Blit To and From A Buffer";
   char OptK[] = "K. Screen to Screen Blit";
   char OptL[] = "L. PGL Palette";
   char OptM[] = "M. Hardware Cursor";

   char *OptList[] = {OptA,OptB,OptC,OptD,OptE,OptF,OptG,OptH,OptI,OptJ,OptK,OptL,OptM};

   char OptR[] = "R. CHANGE RESOLUTION OR PIXEL DEPTH";

   char OptS[] = "S. SHOW ALL DEMOS IN ALL MODES";

   char OptT[] = "T. TOGGLE BETWEEN RANDOM AND STATIC";

   char OptX[] = "X. EXIT PGL DEMO";

   // define pattern list array
   int PatternList[] = {
      DOTTED_LINE,
      SHORT_DASHED_LINE,   
      DASH_DOT_LINE,       
      DOUBLE_DOTTED_LINE,  
      LONG_DASH_LINE,      
      DASH_DOUBLE_DOT_LINE, 
      SOLID_LINE
      };

   // list of demos
   int Quit, DemoCount;
   PFUNC *DemoListPtr;

   // random demo functions
   PFUNC RandDemoList[] = {
      RandLine,
      RandOpenRect,
      RandSolidRect,
      RandPatternRect,
      RandOpenPoly,
      RandSolidPoly,
      RandPatternPoly,
      DemoFont,
      View_Targa,
      RandBufferBlit,
      RandScreenBlit,
      NULL,
      NULL
      };

   // static demo functions
   PFUNC StaticDemoList[] = {
      StaticLine,
      StaticOpenRect,
      StaticSolidRect,
      StaticPatternRect,
      StaticOpenPoly,
      StaticSolidPoly,
      StaticPatternPoly,
      DemoFont,
      View_Targa,
      StaticBufferBlit,
      StaticScreenBlit,
      NULL,
      NULL
      };

   // number of times to call random demo functions when showing all demos
   long DemoTimes[] = {
      0x00007fff,             // RandLine
      0x00001fff,             // RandOpenRect
      0x00001fff,             // RandSolidRect     
      0x00000fff,             // RandPatternRect
      0x00000fff,             // RandOpenPoly
      0x000004ff,             // RandSolidPoly     
      0x000000ff,             // RandPatternPoly
      0x00000001,             // DemoFont
      0x00000001,             // View_Targa
      0x0000000f,             // RandBufferBlit     
      0x00003fff              // RandScreenBlit     
      };

   // initilization functions that must be called before demo is run
   PFUNC InitList[] = {
      NULL,                   // RandLine
      NULL,                   // RandOpenRect
      NULL,                   // RandSolidRect     
      SetPatternFill,         // RandPatternRect
      NULL,                   // RandOpenPoly
      NULL,                   // RandSolidPoly     
      SetPatternFill,         // RandPatternPoly
      NULL,                   // DemoFont
      NULL,                   // View_Targa
      InitBlit,               // buffer blit
      InitBlit                // screen blit
      };

   // cleanup functions that must be called after demo is run
   PFUNC DeInitList[] = {
      NULL,                   // RandLine
      NULL,                   // RandOpenRect
      NULL,                   // RandSolidRect     
      SetSolidFill,           // RandPatternRect
      NULL,                   // RandOpenPoly
      NULL,                   // RandSolidPoly     
      SetSolidFill,           // RandPatternPoly
      NULL,                   // DemoFont
      AfterTarga,             // View_Targa
      NULL,                   // buffer blit
      NULL                    // screen blit
      };

   // screen clearing function called when user pressed space bar in random demo
   PFUNCA ClearList[] = {
      FullScreen,             // RandLine
      FullScreen,             // RandOpenRect
      FullScreen,             // RandSolidRect     
      FullScreen,             // RandPatternRect
      FullScreen,             // RandOpenPoly
      FullScreen,             // RandSolidPoly     
      FullScreen,             // RandPatternPoly
      FullScreen,             // DemoFont
      FullScreen,             // View_Targa
      HalfScreen,             // buffer blit
      HalfScreen              // screen blit
      };


   int retval;
   PGL_version version;
   char PGLDEMOStr[200];

   // hardware cursor strings
   char NoCurMsg[] = "NO MOUSE DRIVER LOADED!";
   char WaitKeyMsg[] = "Press Any Key to Continue";
   char *CurTypes[] =
   {
        "8x8 Cursor size",
        "16x16 Cursor size",
        "32x32 Cursor size",
        "64x64 Cursor size"
   };
   int mouse;
   PGL_palette entry;

   // aperture type control
   int aperture_type;

   // query variables
   int permit_flag;
   int attribute_flags;
   int TotalModes;
   PGL_modedef *ModeList;
   char *ModeStrings;
   char ch;
   int xres_type[] = {640, 800, 1024, 1280, 1600};
   int yres_type[] = {480, 600, 768, 1024, 1200};
   int bpp_type[] = {4, 8, 15, 16, 24, 32};
   int mode_num[] = {MODE_640x480,
                     MODE_800x600,
                     MODE_1024x768,
                     MODE_1280x1024,
                     MODE_1600x1200};

   // misc flags
   int timeout_flag;
   int dumpmode_flag;
   int status_flag;
   int instance;

   // cursor demo variables
   int CurtypeWidth;
   int CurtypeHeight;
   int CurtypeX;
   int CurtypeY;

   // setup version string
   version = PGL_getversion();
   sprintf(PGLDEMOStr, "Demonstration program for the Mach64 PGL library functions,\n"\
                       "Version %d.%d, Copyright (c) ATI Technologies, 1993-95.\n",
                       version.major, version.minor);

   /* set fill pattern structure */
   fillpattern[0] = 0x10;
   fillpattern[1] = 0x38;
   fillpattern[2] = 0x38;
   fillpattern[3] = 0x7c;
   fillpattern[4] = 0x7c;
   fillpattern[5] = 0xfe;
   fillpattern[6] = 0xff;
   fillpattern[7] = 0xfe;
   fillpattern[8] = 0x7c;
   fillpattern[9] = 0x7c;
   fillpattern[10] = 0x38;
   fillpattern[11] = 0x38;
   fillpattern[12] = 0x10;
   fillpattern[13] = 0x10;

   // initialize patterns
   Patterns = PatternList;
   TotalPatterns = sizeof(PatternList) / 2;

   // identify version of PGLDEMO
   printf(PGLDEMOStr);

   // get aperture type if given
   aperture_type = AUTO_APERTURE;
   timeout_flag = 0;
   dumpmode_flag = 0;
   status_flag = 0;
   permit_flag = 0;
   instance = 0;
   attribute_flags = NO_ATTRIBUTES;
   if (argc > 1)
   {
      for (i = 2; i <= argc; i++)
      {
         // look for option delimiters
         if ((*argv[i-1] == '-') || (*argv[i-1] == '/'))
         {
            // help
            if ((strcmpi(argv[i-1] + 1, "h") == 0) ||
                (strcmpi(argv[i-1] + 1, "help") == 0) ||
                (strcmpi(argv[i-1] + 1, "?") == 0))
            {
                printf("Usage: PGLDEMO {- | /}<options>\n");
                printf("<options>:\n");
                printf("  -h          help (this screen)\n");
                printf("  -al         use linear aperture\n");
                printf("  -av         use VGA aperture\n");
                printf("  -i<#>       instance of desired card (multiple cards only)\n");
                printf("  -ml         display mode list (affected by -p option)\n");
                printf("  -p          permit all standard modes in mode list\n");
                printf("  -s          display hardware status\n");
                printf("  -t          enable fifo and idle timeout checking\n");
                return (0);
            }

            // use linear aperture
            if (strcmpi(argv[i-1] + 1, "al") == 0)
            {
                aperture_type = USE_LINEAR_APERTURE;
            }

            // use VGA aperture
            if (strcmpi(argv[i-1] + 1, "av") == 0)
            {
                aperture_type = USE_VGA_APERTURE;
            }

            // get instance number
            if ((*(argv[i-1] + 1) == 'i') || (*(argv[i-1] + 1) == 'I'))
            {
                instance = atoi(argv[i-1] + 2) & 0xff;
            }

            // dump mode list flag
            if (strcmpi(argv[i-1] + 1, "ml") == 0)
            {
                dumpmode_flag = 1;
            }

            // permit all standard modes in mode list
            if (strcmpi(argv[i-1] + 1, "p") == 0)
            {
                permit_flag = 1;
            }

            // display hardware status
            if (strcmpi(argv[i-1] + 1, "s") == 0)
            {
                status_flag = 1;
            }

            // enable fifo and idle timeout checking
            if (strcmpi(argv[i-1] + 1, "t") == 0)
            {
                timeout_flag = 1;
            }
         }
      }
   }

   // check if Mach64 card is installed
   if (PGL_detectmach64(instance) != NO_ERROR)
   {
      printf("This program requires an ATI MACH64 based video adapter.\n");
      if (status_flag == 1)
      {
         // dump only the information filled in the detect routine
         dump_status(DETECTONLY_STATUS);
      }

      exit(EXIT_FAILURE);
   }

   // initialize the random number generator
   srand((unsigned) time( NULL ));

   /*
    * Start up program in default 640x480 8 bit mode
    */

   // setup fonts
   smallfont = PGL_getfixedfont8x8();
   normalfont = PGL_getfixedfont8x14();
   largefont = PGL_getfixedfont12x20();
   extrafont = PGL_getfixedfont7x15();

   // setup fifo and idle timeout control
   if (timeout_flag == 1)
   {
      // application will terminate if a timeout error occurs
      PGL_setfifocontrol(FIFO_CHECK_TIMEOUT, FIFO_ACTION_TERMINATE);
      printf("Timeout terminate enabled ...\n");
   }
   else
   {
      // default after calling PGL_detectmach64()
      PGL_setfifocontrol(FIFO_CHECK_NORMAL, FIFO_ACTION_NONE);
   }

   // setup GUI register access method
   retval = PGL_initaperture(aperture_type);
   if (retval != NO_ERROR)
   {
      printf("Error initializing Mach64 aperture ");
      switch(retval)
      {
          case QUERY_FAILED:
            printf("(QUERY FAILED)\n");
            if (status_flag == 1)
            {
                // dump only the information filled in the detect routine
                dump_status(DETECTONLY_STATUS);
            }
            break;

          case NO_APERTURE:
            printf("(NO APERTURE)\n");
            if (status_flag == 1)
            {
                // dump full status - all structures have been successfully filled
                dump_status(FULL_STATUS);
            }
            break;

          case FAILED_REG_TEST:
            printf("(REGISTER TEST FAILED)\n");
            if (status_flag == 1)
            {
                // dump full status - all structures have been successfully filled
                dump_status(FULL_STATUS);
            }
            break;
      }
      terminate(EXIT_FAILURE);
   }

   // report special aperture selection if applicable
   switch(aperture_type)
   {
      case USE_VGA_APERTURE:    printf("Using VGA aperture ...\n");    break;
      case USE_LINEAR_APERTURE: printf("Using linear aperture ...\n"); break;
   }

   // get PGL's copy of query structure
   PGL_getqueryheader(&queryinfo);

   // dump hardware status and exit if flag is set
   if (status_flag == 1)
   {
      // dump full status - all structures have been successfully filled
      dump_status(FULL_STATUS);

      // exit
      terminate (0);
   }

   // setup modes list
   if (permit_flag == 0)
   {
      // get available modes list
      ModeList = PGL_loadmodelist(&TotalModes);
      if (ModeList == NULL)
      {
         printf("Error retrieving available modes list.\n");
         terminate(EXIT_FAILURE);
      }
   }
   else
   {
      // fill mode list with all standard modes
      TotalModes = 30;
      ModeList = (PGL_modedef *) malloc (TotalModes * sizeof(PGL_modedef));
      if (ModeList == NULL)
      {
         printf("Not enough memory for standard modes list allocation.\n");
         terminate(EXIT_FAILURE);
      }

      // resolution loop
      LoopCount = 0;
      for (i = 0; i < 5; i++)
      {
         // color depth loop
         for (j = 0; j < 6; j++)
         {
            ModeList[LoopCount].xres = xres_type[i];
            ModeList[LoopCount].yres = yres_type[i];
            ModeList[LoopCount].bpp = bpp_type[j];
            ModeList[LoopCount].depth = DEPTH_32_RGBA;
            ModeList[LoopCount].mode_number = mode_num[i];
            LoopCount++;
         }
      }
   }

   // allocate memory for mode description strings
   ModeStrings = (char *) malloc (TotalModes * MAX_STRING_WIDTH);
   if (ModeStrings == NULL)
   {
      printf("Not enough memory to store mode strings.\n");
      terminate(EXIT_FAILURE);
   }

   // build mode strings
   for (i = 0; i < TotalModes; i++)
   {
      if (i < 9)
      {
         ch = (char)('0' + i + 1);
      }
      else
      {
         ch = (char)('A' + i - 9);
      }
      sprintf(ModeStrings + (i * MAX_STRING_WIDTH),
              "%c. - %4d x %4d  %2d bpp", ch,
              ModeList[i].xres, ModeList[i].yres, ModeList[i].bpp);
   }

   // dump available modes and exit if flag is set
   if (dumpmode_flag == 1)
   {
      for (i = 0; i < TotalModes; i++)
      {
         printf("%s\n", ModeStrings + (i * MAX_STRING_WIDTH));
      }

      // exit
      terminate(0);
   }

   UserMode = 1;  // 640x480x8

   // set an accelerator mode
   retval = PGL_initmode(ModeList[UserMode].xres,
                         ModeList[UserMode].yres,
                         ModeList[UserMode].bpp,
                         ModeList[UserMode].depth,
                         ModeList[UserMode].xres,
                         0,
                         attribute_flags,
                         ModeList[UserMode].mode_number,
                         NO_CRT_TABLE,
                         NO_EEPROM_OFFSET);
   if (retval == NO_APERTURE)
   {
      printf("No aperture available.\n");
      terminate(EXIT_FAILURE);
   }
   if (retval != MODE_SUPPORTED)
   {
      printf("Can't set accelerator mode.\n");
      terminate(EXIT_FAILURE);
   }

   /* fill local structure with current mode information */
   PGL_getmodeconfig(&modeinfo);

   /* point to the random demos as the default */
   DemoListPtr = RandDemoList;
   SelectStrPtr = RandSelectStr;
   DemoType = RANDOM_DEMO;
   DemoColour = DARKBLUE;

   /* do the additonal mode setting steps required for PGLDEMO */
   SetDemoMode();

   /* wait for user keystroke */
   do
      {
      // initialize the screen
      InitDemoScreen(ModeTitle,SelectStrPtr, BLACK);

      // center longest option line
      OptionX = (modeinfo.xres - (strlen(OptT)*normalfont->charwidth))/2;

      // display list of options to user
      PGL_setfgcolor(PGL_getcolorcode(YELLOW));
      LoopCount = sizeof(OptList)/sizeof(char *);

      // position list in the Y direction
      OptionY = modeinfo.yres / 8;

      for(i = 0; i < LoopCount; i++)
      {
         PGL_text(OptionX, OptionY+(i*normalfont->charheight), OptList[i], normalfont);
      }

      // display change mode option
      PGL_setfgcolor(PGL_getcolorcode(LIGHTGREEN));
      PGL_text(OptionX, OptionY+((LoopCount+1)*normalfont->charheight), OptR, normalfont);

      // display show room demo option
      PGL_setfgcolor(PGL_getcolorcode(LIGHTCYAN));
      PGL_text(OptionX, OptionY+((LoopCount+3)*normalfont->charheight), OptS, normalfont);

      // display toggle demo option
      PGL_setfgcolor(PGL_getcolorcode(LIGHTRED));
      PGL_text(OptionX, OptionY+((LoopCount+5)*normalfont->charheight), OptT, normalfont);

      // display exit option 
      PGL_setfgcolor(PGL_getcolorcode(WHITE));
      PGL_text(OptionX, OptionY+((LoopCount+7)*normalfont->charheight), OptX, normalfont);

      // get a keystroke from the user
      DemoOption=toupper(getch());

      // see what the user wanted to do
      switch(DemoOption)
         {
         case 'A':     // draw lines
            // initialize the screen
            InitDemoScreen(ModeTitle,OptA, DemoColour);

            // do the demo until user presses a key
            DoDemo(DemoListPtr[0],NULL,NULL,ClearList[0],DemoType,DemoColour);

            break;


         case 'B':     // draw rectangle
            // initialize the screen
            InitDemoScreen(ModeTitle,OptB, DemoColour);

            // do the demo until user presses a key
            DoDemo(DemoListPtr[1],NULL,NULL,ClearList[1],DemoType,DemoColour);
            
            break;


         case 'C':     // draw solid fill rectangle
            // initialize the screen
            InitDemoScreen(ModeTitle,OptC, DemoColour);

            // do the demo until user presses a key
            DoDemo(DemoListPtr[2],NULL,NULL,ClearList[2],DemoType,DemoColour);

            break;


         case 'D':     // draw pattern fill rectangle
            // initialize the screen
            InitDemoScreen(ModeTitle,OptD, DemoColour);

            // do the demo until user presses a key
            DoDemo(DemoListPtr[3],SetPatternFill,SetSolidFill,ClearList[3],DemoType,DemoColour);

            break;


         case 'E':     // draw polygon
            // initialize the screen
            InitDemoScreen(ModeTitle,OptE, DemoColour);

            // do the demo until user presses a key
            DoDemo(DemoListPtr[4],NULL,NULL,ClearList[4],DemoType,DemoColour);

            break;


         case 'F':     // draw solid fill polygon
            // initialize the screen
            InitDemoScreen(ModeTitle,OptF, DemoColour);

            if (modeinfo.bpp == 24)
            {
                ComingSoon("SOLID POLYGONS ARE NOT SUPPORTED IN 24 BPP");

                // wait for a keypress
                DemoMsg("", DARKBLUE);
            }
            else
            {
                // do the demo until user presses a key
                DoDemo(DemoListPtr[5],NULL,NULL,ClearList[5],DemoType,DemoColour);
            }

            break;


         case 'G':     // draw pattern fill polygon
            // initialize the screen
            InitDemoScreen(ModeTitle,OptG, DemoColour);

            if (modeinfo.bpp == 24)
            {
                ComingSoon("PATTERNED POLYGONS ARE NOT SUPPORTED IN 24 BPP");

                // wait for a keypress
                DemoMsg("", DARKBLUE);
            }
            else
            {
                // do the demo until user presses a key
                DoDemo(DemoListPtr[6],SetPatternFill,SetSolidFill,ClearList[6],DemoType,DemoColour);
            }

            break;


         case 'H':     // display bitmap text
            // initialize the screen
            InitDemoScreen(ModeTitle,OptH, DemoColour);

            // show off the fonts
            ShowOffFont(normalfont,"8x14 Bitmap Font",RealBPP,0,DemoColour);
            ShowOffFont(extrafont,"7x15 Bitmap Font",RealBPP,0,DemoColour);
            ShowOffFont(largefont,"12x20 Bitmap Font",RealBPP,0,DemoColour);
            ShowOffFont(smallfont,"8x8 Bitmap Font",RealBPP,0,DemoColour);

            break;


         case 'I':    // display an image -- ATI Logo
            // initialize the screen
            InitDemoScreen(ModeTitle,OptI, DemoColour);

            // load an image from file
            if (View_Targa() != 0)
            {
                ComingSoon("TEST.TGA could not be found!");
            }

            // wait for a keypress
            DemoMsg("", DemoColour);

            // restore the palette
            if (modeinfo.bpp < 16)
               PGL_initpalette();

            break;


         case 'J':    // blit to and from a buffer
            // initialize the screen
            InitDemoScreen(ModeTitle,OptJ, DemoColour);

            DoDemo(DemoListPtr[9],&InitBlit,NULL,ClearList[9],DemoType,DemoColour);

            break;


         case 'K':    // screen to screen blit
            // initialize the screen
            InitDemoScreen(ModeTitle,OptK, DemoColour);

            // do the demo until user presses a key
            DoDemo(DemoListPtr[10],&InitBlit,NULL,ClearList[10],DemoType,DemoColour);

            break;


         case 'L':    // show 16 pgl colours
            // initialize the screen
            InitDemoScreen(ModeTitle,OptL, DemoColour);

            // show the colour demo
            ColourDemo();

            DemoMsg("", DemoColour);

            break;


         case 'M':    // hardware cursor

            // draw pattern for hardware cursor
            switch(modeinfo.bpp)
            {
                case 4:
                    pattern4(modeinfo.xres, modeinfo.yres);
                    textcol = 0x0f;     // WHITE
                    break;

                case 8:
                    pattern8(modeinfo.xres, modeinfo.yres);
                    textcol = 0xff;     // WHITE
                    break;

                case 16:
                    if (modeinfo.depth == 555)
                    {
                        pattern15(modeinfo.xres, modeinfo.yres);
                    }
                    else // 565 weight
                    {
                        pattern16(modeinfo.xres, modeinfo.yres);
                    }
                    textcol = PGL_getcolorcode(WHITE);
                    break;

                case 24:
                    pattern24(modeinfo.xres, modeinfo.yres, modeinfo.depth);
                    textcol = PGL_getcolorcode(WHITE);
                    break;

                case 32:
                    pattern32(modeinfo.xres, modeinfo.yres, modeinfo.depth);
                    textcol = PGL_getcolorcode(WHITE);
                    break;
            }

            // save cursor type string area
            CurtypeWidth = (strlen(CurTypes[1]) * normalfont->charwidth) + 8;
            CurtypeHeight = normalfont->charheight;
            CurtypeX = (modeinfo.xres - CurtypeWidth)/2;
            CurtypeY = 32;
            PGL_ncscreentoscreenblit(CurtypeX, CurtypeY,
                                     CurtypeWidth, CurtypeHeight,
                                     0, modeinfo.yres+7, SAVE_BLIT);

            // display options screen title: mode, pixel depth
            PGL_setfgcolor(textcol);

            PGL_text((modeinfo.xres - (strlen(ModeTitle)*normalfont->charwidth))/2,
                     1, ModeTitle, normalfont);

            // print the second string
            PGL_text((modeinfo.xres - (strlen(OptM)*normalfont->charwidth))/2,
                     16, OptM, normalfont);

            // print wait key message
            PGL_text((modeinfo.xres - (strlen(WaitKeyMsg)*normalfont->charwidth))/2,
                     modeinfo.yres-15,
                     WaitKeyMsg, normalfont);

            // print no mouse driver message if no driver installed
            mouse = InitMouse();
            if (mouse == 0)
            {
                PGL_text((modeinfo.xres - (strlen(NoCurMsg)*largefont->charwidth))/2,
                         (modeinfo.yres/4)+(modeinfo.yres/8)-10,
                         NoCurMsg, largefont);
            }

            // run cursor demos
            for (i = 0; i < 4; i++)
            {
                // restore cursor size string for next size
                PGL_ncscreentoscreenblit(CurtypeX, CurtypeY,
                                         CurtypeWidth, CurtypeHeight,
                                         0, modeinfo.yres+7, RESTORE_BLIT);

                // display cursor size
                PGL_text((modeinfo.xres - (strlen(CurTypes[i])*normalfont->charwidth))/2,
                         32, CurTypes[i], normalfont);

                // run cursor demo for each cursor size
                CursorDemo(i, mouse);
            }

            // reset palette for 4 or 8 bpp modes
            if (modeinfo.bpp < 16)
            {
                entry.red = 0;
                entry.green = 0;
                entry.blue = 0;
                PGL_setpalette(0, entry);
                PGL_clearscreen(1, 1, modeinfo.xres-1, modeinfo.yres-1);
                PGL_initpalette();
            }

            break;


         case 'R':    // change the Mach64 Display Mode

            // center longest option line
            OptionX = (modeinfo.xres - (strlen(ModeStrings)*normalfont->charwidth))/2;

            // position list in the Y direction
            OptionY = modeinfo.yres / 8;

            // determine if one page is enough to display available modes
            ModeIndex = 0;
            if ((modeinfo.yres-(OptionY*2)) < (TotalModes*normalfont->charheight))
            {
                multipleflag = 1;
                ModesPerPage = (modeinfo.yres-(OptionY*2))/normalfont->charheight;
            }
            else
            {
                multipleflag = 0;
                ModesPerPage = TotalModes;
            }

            // initialize the screen
            InitDemoScreen(ModeTitle,OptR, DARKRED);

            // display list of options to user
            PGL_setfgcolor(PGL_getcolorcode(YELLOW));
            for (i = 0; i < ModesPerPage; i++)
            {
                PGL_text(OptionX, OptionY+(i*normalfont->charheight),
                         ModeStrings + (i * MAX_STRING_WIDTH), normalfont);
            }

            // display multiple page message
            if (multipleflag == 1)
            {
                PGL_setfgcolor(PGL_getcolorcode(LIGHTGREEN));
                PGL_text((modeinfo.xres - (strlen(MultiplePage)*normalfont->charwidth))/2,
                         modeinfo.yres - (normalfont->charheight*2),
                         MultiplePage, normalfont);
            }

            // set the new mode
            OldMode = UserMode;
            UserMode = 100;         // initialize user mode value
            do
            {
               // get input from user
               do
               {
                  // wait for a key
                  get_key_code(&ascii_code, &scan_code);

                  UserModeStr[0] = (char)(ascii_code);

                  if (TotalModes != ModesPerPage)
                  {
                     // PGUP key
                     if (scan_code == 0x49)
                     {
                        ModeIndex = 0;
                     }

                     // PGDN key
                     if (scan_code == 0x51)
                     {
                        ModeIndex = ModesPerPage;
                     }

                     // display new page of mode choices
                     if ((scan_code == 0x49) || (scan_code == 0x51))
                     {
                        // initialize the screen
                        InitDemoScreen(ModeTitle,OptR, DARKRED);

                        // display list of options to user
                        PGL_setfgcolor(PGL_getcolorcode(YELLOW));
                        i = ModeIndex;
                        j = 0;
                        while ((i < TotalModes) && (j < ModesPerPage))
                        {
                           PGL_text(OptionX, OptionY+(j*normalfont->charheight),
                                    ModeStrings + (i * MAX_STRING_WIDTH), normalfont);
                           i++;
                           j++;
                        }

                        // display multiple page message
                        if (multipleflag == 1)
                        {
                           PGL_setfgcolor(PGL_getcolorcode(LIGHTGREEN));
                           PGL_text((modeinfo.xres - (strlen(MultiplePage)*normalfont->charwidth))/2,
                                    modeinfo.yres - (normalfont->charheight*2),
                                    MultiplePage, normalfont);
                        }
                     }
                  }

                  // Check for ESC key
                  if (UserModeStr[0] == 0x1b)
                  {
                     // set to default
                     UserMode = 1;
                  }
                  else
                  {
                     if(sscanf(UserModeStr,"%x",&UserMode) == 0 || UserModeStr[0] == 0)
                     {
                        // must be letter higher than F, so special conversion needed
                        strupr(UserModeStr);
                        UserMode = 0xf + (UserModeStr[0] - 'F');
                     }
                  }
               }
               while((UserMode < 1) || (UserMode > TotalModes));

               if (UserMode > 0)
               {
                  UserMode--;
               }

               /* initialize accelerator card mode */

               // check that chosen mode is supported
               if (UserModeStr[0] == 0x1b)    // escape key
               {
                  // leave old mode
                  retval = MODE_SUPPORTED;
               }
               else
               {
                  // only set mode if its different
                  if (UserMode != OldMode)
                  {
                     retval = PGL_initmode(ModeList[UserMode].xres,
                                           ModeList[UserMode].yres,
                                           ModeList[UserMode].bpp,
                                           ModeList[UserMode].depth,
                                           ModeList[UserMode].xres,
                                           0,
                                           attribute_flags,
                                           ModeList[UserMode].mode_number,
                                           NO_CRT_TABLE,
                                           NO_EEPROM_OFFSET);
                  }
                  else
                  {
                     // leave old mode if new mode = old mode
                     retval = MODE_SUPPORTED;
                  }
               }

               // display error if mode is not supported
               if (retval != MODE_SUPPORTED)
               {
                  // restore old mode
                  PGL_initmode(ModeList[OldMode].xres,
                               ModeList[OldMode].yres,
                               ModeList[OldMode].bpp,
                               ModeList[OldMode].depth,
                               ModeList[OldMode].xres,
                               0,
                               attribute_flags,
                               ModeList[OldMode].mode_number,
                               NO_CRT_TABLE,
                               NO_EEPROM_OFFSET);

                  // initialize the screen
                  InitDemoScreen(ModeTitle,OptR, DARKRED);

                  // display list of options to user
                  PGL_setfgcolor(PGL_getcolorcode(YELLOW));
                  i = ModeIndex;
                  j = 0;
                  while ((i < TotalModes) && (j < ModesPerPage))
                  {
                     PGL_text(OptionX, OptionY+(j*normalfont->charheight),
                              ModeStrings + (i * MAX_STRING_WIDTH), normalfont);
                     i++;
                     j++;
                  }

                  // tell user that mode is no good
                  printf("");
                  DemoMsg("***** CAN'T SET THE REQUESTED ACCELERATOR MODE *****", DARKRED);

                  // display multiple page message
                  if (multipleflag == 1)
                  {
                     PGL_setfgcolor(PGL_getcolorcode(LIGHTGREEN));
                     PGL_text((modeinfo.xres - (strlen(MultiplePage)*normalfont->charwidth))/2,
                              modeinfo.yres - (normalfont->charheight*2),
                              MultiplePage, normalfont);
                  }
               }
            }
            while(retval != MODE_SUPPORTED);

            /* fill structure with current mode information */
            PGL_getmodeconfig(&modeinfo);

            /* do the additonal mode setting steps required for PGLDEMO */
            SetDemoMode();

            break;

         case 'S':   // SHOW-ROOM DEMO -- All modes, all demos
            // check if pgldemo is in random demo mode, if not - NO WAY!
            if(DemoListPtr != RandDemoList)
            {
               // initialize the screen
               InitDemoScreen(ModeTitle,OptS, DemoColour);
               // show banner
               ComingSoon("OPTION S IS NOT AVAILABLE FOR STATIC DEMOS.");
               DemoMsg("Select Option T on the main menu to toggle back to random demos.", DARKBLUE);
            }
            else
            {
               // initialize quit variable
               Quit = 0;

               // loop through all of the modes until user presses a key
               while(Quit != 1)
               {
                  // set the mode
                  if (PGL_initmode(ModeList[UserMode].xres,
                                   ModeList[UserMode].yres,
                                   ModeList[UserMode].bpp,
                                   ModeList[UserMode].depth,
                                   ModeList[UserMode].xres,
                                   0,
                                   attribute_flags,
                                   ModeList[UserMode].mode_number,
                                   NO_CRT_TABLE,
                                   NO_EEPROM_OFFSET) == MODE_SUPPORTED)
                  {
                     /* fill structure with current mode information */
                     PGL_getmodeconfig(&modeinfo);

                     /* do the additonal mode setting steps required for PGLDEMO */
                     SetDemoMode();

                     // run through the demos
                     DemoCount = 0;
                     while(DemoListPtr[DemoCount] != NULL && Quit != 1)
                     {
                        // do the random line demo
                        InitDemoScreen(ModeTitle,OptList[DemoCount], DemoColour);

                        ntimes = DemoTimes[DemoCount];
                        if (modeinfo.bpp == 24)
                        {
                            ntimes = ntimes / 3;
                            if (ntimes == 0)
                            {
                                ntimes = 1;
                            }
                        }

                        if (DoDemo(DemoListPtr[DemoCount],InitList[DemoCount],
                              DeInitList[DemoCount],ClearList[DemoCount],
                              ntimes,DemoColour) != 0)
                        {
                           // user interrupted demo
                           Quit = 1;
                        }

                        // point to the next demo
                        ++DemoCount;
                     }
                  }
                  // do the next mode
                  if(++UserMode > TotalModes)
                     // if all have been done, start again at the beginning
                     UserMode = 0;  // must be first mode (640x480x4)

                  // check for ESC key
                  if (kbhit() != 0)
                  {
                     if (getch() == 0x1b)
                     {
                        Quit = 1;
                        PGL_initmode(ModeList[OldMode].xres,
                                     ModeList[OldMode].yres,
                                     ModeList[OldMode].bpp,
                                     ModeList[OldMode].depth,
                                     ModeList[OldMode].xres,
                                     0,
                                     attribute_flags,
                                     ModeList[OldMode].mode_number,
                                     NO_CRT_TABLE,
                                     NO_EEPROM_OFFSET);
                     }
                  }
               }
            }

            break;

         case 'T':   // Toggle Demo Type
            if(DemoListPtr == RandDemoList)
               {
               // set up for static demos
               DemoListPtr = StaticDemoList;
               SelectStrPtr = StaticSelectStr;
               DemoType = STATIC_DEMO;
               DemoColour = BLACK;
               }
            else
               {
               // set up for random demos
               DemoListPtr = RandDemoList;
               SelectStrPtr = RandSelectStr;
               DemoType = RANDOM_DEMO;
               DemoColour = DARKBLUE;
               }

            break;


         default:
            break;
         }
      }
   while(DemoOption != 'X');

   /*
    * PROGRAM IS FINISHED
    */

   /* unload mode list buffer */
   if (permit_flag == 0)
   {
      PGL_unloadmodelist(ModeList);
   }
   else
   {
      if (ModeList != NULL)
      {
         free(ModeList);
      }
   }

   /* deallocate mode strings buffer */
   if (ModeStrings != NULL)
   {
      free(ModeStrings);
   }

   /* close accelerator down and switch to VGA passthrough */
   PGL_closemode();
   PGL_close();

   // identify version of PGLDEMO
   printf(PGLDEMOStr);

   // return to dos
   return(0);
   }

/*
 *****************************************************************************
 */

