/*==========================================================================
* TEXT.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved    *
*                                                                          *
* PGL functions to display text using a given font. Only BITMAP font style *
* format is supported. See LOADFONT.C for routines that load and unload    *
* font structure data.                                                     *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <io.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_text - print a string to the display using a given font structure

  This function prints a string starting at position (x, y) using the font
  data contained in the given font structure. (x, y) is interpreted as the
  top-left point to start drawing. A previous successful call to
  PGL_loadfont() is mandatory for this function is work. If the font
  structure contains invalid information, nothing is displayed. The string
  is assumed to be '\0' terminated.
-------------------------------------------------------------------------- */
void PGL_text(int x, int y, char *string, PGL_font *font)
{
    unsigned int index, string_index;
    unsigned char ch;
    int i, nbits, hostwrts, chkwrts;
    unsigned long hostdata;
    unsigned long temp1, temp2, temp3;

    /* check font structure */
    if ((font->bit_data == NULL) || (font->charwidth < 1) ||
        (font->charheight < 1))
    {
        return;
    }

    // call special routine for text for 24 bpp modes
    if (PGL_modecfg.bpp == 24)
    {
        pgl_text24(x, y, string, font);
        return;
    }

    // save vital regs
    PGL_waitforidle();
    temp1 = regr(DP_PIX_WIDTH);
    temp2 = regr(DP_SRC);
    temp3 = regr(DST_CNTL);

    // determine number of bits needed and 32 host writes required
    nbits = font->charwidth * font->charheight;
    hostwrts = nbits / 32;
    if (nbits > (hostwrts * 32)) hostwrts++;
    if (hostwrts == 0) hostwrts++;

    // setup monochrome HOST to screen blit - X tiling is enabled
    regw(DP_PIX_WIDTH, temp1 & 0x0000ffff);
    regw(DP_SRC, MONO_SRC_HOST | FRGD_SRC_FRGD_CLR);
    regw(DST_CNTL, DST_X_TILE | DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    regw(DST_Y_X, ((unsigned long)(x) << 16) | y);
    regw(DST_HEIGHT, (unsigned long)(font->charheight));

    // character retrivel loop
    chkwrts = 9;
    string_index = 0;
    ch = *((unsigned char *)(string + string_index));
    while (ch != 0)
    {
        regw(DST_WIDTH, (int)(font->charwidth));
        chkwrts++;

        // character draw loop
        index = font->index[ch];
        for (i = 0; i < hostwrts; i++)
        {
            hostdata = *((unsigned long far *)(font->bit_data + index));
            regw(HOST_DATA0, hostdata);
            index += 4;

            // optimize times needed to wait for engine idle
            chkwrts++;
            if (chkwrts > 15)
            {
                PGL_waitforemptyfifo();
                chkwrts = 0;
            }
        }
        string_index++;
        ch = *((unsigned char *)(string + string_index));
        PGL_waitforidle();
    }

    // restore
    PGL_waitforfifo(3);
    regw(DP_PIX_WIDTH, temp1);
    regw(DP_SRC, temp2);
    regw(DST_CNTL, temp3);
}

/* --------------------------------------------------------------------------
  pgl_text24 - print a string to the display using a given font structure
               using 24 bpp mode.

  This function is a special version of PGL_text() to properly output font
  information. Since the engine is actually in 8 bpp with the CRT in 24 bpp,
  each monochrome font data bit must be manually replicated 3 times.
-------------------------------------------------------------------------- */
void pgl_text24(int x, int y, char *string, PGL_font *font)
{
    unsigned int index, string_index;
    unsigned char ch;
    int i, l, nbits, hostwrts, chkwrts, xpos;
    unsigned long hostdata, remainder;
    unsigned long temp1, temp2, temp3;
    unsigned long rotation;

    // save vital regs
    PGL_waitforidle();
    temp1 = regr(DP_PIX_WIDTH);
    temp2 = regr(DP_SRC);
    temp3 = regr(DST_CNTL);

    // determine number of bits needed and 32 host writes required
    nbits = font->charwidth * font->charheight;
    nbits = nbits * 3;
    hostwrts = nbits / 32;
    if (nbits > (hostwrts * 32)) hostwrts++;
    if (hostwrts == 0) hostwrts++;

    // setup monochrome HOST to screen blit - X tiling is enabled
    regw(DP_PIX_WIDTH, BYTE_ORDER_LSB_TO_MSB | SRC_8BPP | DST_8BPP);
    regw(DP_SRC, MONO_SRC_HOST | FRGD_SRC_FRGD_CLR);
    regw(DST_Y_X, ((unsigned long)(x * 3) << 16) | y);
    regw(DST_HEIGHT, (unsigned long)(font->charheight));

    // character retrivel loop
    chkwrts = 8;
    string_index = 0;
    ch = *((unsigned char *)(string + string_index));
    xpos = x;
    rotation = pgl_get24bpprotation(x);
    while (ch != 0)
    {
        regw(DST_CNTL, (temp3 & 0x80) | rotation | DST_X_TILE |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
        regw(DST_WIDTH, (int)(font->charwidth) * 3);
        chkwrts = chkwrts + 2;

        // character draw loop
        index = font->index[ch];
        hostdata = 0;
        l = 0;
        for (i = 0; i < hostwrts; i++)
        {
            switch (l)
            {
                case 0:
                    if (i > 0) index++;
                    hostdata = pgl_expand8to24bpp(*(font->bit_data + index));
                    remainder = pgl_expand8to24bpp(*(font->bit_data + index + 1));
                    hostdata = hostdata | (remainder << 24);
                    break;

                case 1:
                    remainder = pgl_expand8to24bpp(*(font->bit_data + index + 1));
                    hostdata = (hostdata >> 8) | (remainder << 16);
                    break;

                case 2:
                    remainder = pgl_expand8to24bpp(*(font->bit_data + index + 1));
                    hostdata = (hostdata >> 16) | (remainder << 8);
                    break;
            }
            regw(HOST_DATA0, hostdata);
            hostdata = remainder;
            index++;

            // optimize times needed to wait for engine idle
            chkwrts++;
            if (chkwrts > 13)
            {
                PGL_waitforemptyfifo();
                chkwrts = 0;
            }

            // 24 bpp alignment variable handling
            l++;
            if (l > 2)
            {
                l = 0;
            }
        }

        xpos = (int)(xpos + font->charwidth);
        rotation = pgl_get24bpprotation(xpos);

        string_index++;
        ch = *((unsigned char *)(string + string_index));
        PGL_waitforidle();
    }

    // restore
    PGL_waitforfifo(3);
    regw(DP_PIX_WIDTH, temp1);
    regw(DP_SRC, temp2);
    regw(DST_CNTL, temp3);
}

