/*==========================================================================
* LOADFONT.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved*
*                                                                          *
* PGL functions to load and unload fonts for text displaying. Only BITMAP  *
* font style format is supported. See TEXT.C for functions that actually   *
* displays text.                                                           *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <io.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

#define BIT_FONT 1

/* --------------------------------------------------------------------------
  PGL_loadfont - load a font into memory for displaying text
  
  The function loads the font file and fills the PGL_font structure with
  the appropriate font information. Memory is allocated to store the BITMAP
  data. This memory should be released by calling PGL_unloadfont(). If the
  call is successful, font->bit_data will contain a non-NULL data pointer
  and NO_ERROR is returned. If an error occurs, the pointer will be NULL
  and YES_ERROR is returned.

  Note: only the BITMAP format is supported.
-------------------------------------------------------------------------- */
int PGL_loadfont(char *fontfile, PGL_font *font)
{
    int handle;
    unsigned int index, bufsize;
    PGL_char_def_block cdb;
    PGL_fnt_format header;
    unsigned char *tempptr;

    /* configure PGL_text structure for failure result */
    font->charwidth = 0;
    font->charheight = 0;
    font->bit_data = NULL;
    font->unload_ptr = NULL;
    for (index = 0; index < 256; index++)
    {
        font->index[index] = 0;
    }

    /* Open the font file */
    handle = open(fontfile, O_RDONLY | O_BINARY);
    if (handle == -1)
    {
        return (YES_ERROR);
    }

    /* read file header */
    read(handle, &header, sizeof(PGL_fnt_format));

    /* read cdb block */
    lseek(handle, (long)(header.cdb1_offset), SEEK_SET);
    read(handle, &cdb, sizeof(PGL_char_def_block));
    if (cdb.fonttype != BIT_FONT)
    {
        if (handle != 0) close(handle);
        return (YES_ERROR);
    }

    /* read char index table (for 256 characters) */
    lseek(handle, (long)(cdb.index_table_offset), SEEK_SET);
    read(handle, font->index, 256 * sizeof(unsigned int));

    /* determine size of buffer from largest index */
    bufsize = 0;
    for(index = 0; index < 256; index++)
    {
        if (font->index[index] > bufsize)
        {
            bufsize = font->index[index];
        }
    }

    /* allocate buffer and read in bitmap data */
    tempptr = (unsigned char *) malloc (bufsize + 100);
    if (tempptr == NULL)
    {
        if (handle != 0) close(handle);
        return (YES_ERROR);
    }
    lseek(handle, (long)(cdb.char_table_offset), SEEK_SET);
    read(handle, tempptr, bufsize + 100);

    /* fill in font structure for successfully loaded font info */
    font->charwidth = (int)(cdb.charwidth);
    font->charheight = (int)(cdb.charheight);

    /* close the font file handle */
    close(handle);

    /* assign far pointer to bitmap data */
    font->bit_data = (unsigned char far *)(tempptr);
    font->unload_ptr = tempptr;

    return (NO_ERROR);
}

/* --------------------------------------------------------------------------
  PGL_unloadfont - unload a font loaded previously by PGL_loadfont()

  The memory allocated by a successful call to PGL_loadfont() is released
  by this function. For multiple font loads, releasing them should be done
  in reverse order of loading. This method will insure minimum memory
  fragmentation.
-------------------------------------------------------------------------- */
void PGL_unloadfont(PGL_font *font)
{
    if (font->unload_ptr != NULL)
    {
        free(font->unload_ptr);
    }

    font->charwidth = 0;
    font->charheight = 0;
    font->bit_data = NULL;
}

