/*==========================================================================
* DRAWPOLY.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved*
*                                                                          *
* PGL functions to draw filled polygon shapes.                             *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <math.h>
#include <errno.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  pgl_boxsize - get dimensions of polygon size

  The function returns the box dimensions that enclose an "n" sided polygon.
  The vertices given should form an enclosed shape outline. Note that an "n"
  sided polygon requires "n+1" point pairs. The right and bottom coorindates
  are clamped to ENGINE_MAX_X & ENGINE_MAX_Y if this maximum is exceeded -
  see PGL.H. The coordinates returned are inclusive.
-------------------------------------------------------------------------- */
PGL_scissor pgl_boxsize(int n, PGL_point *vertices)
{
    PGL_scissor box;
    int i;

    /* determine box size of shape */
    box.left = ENGINE_MAX_X - 1;
    box.top = ENGINE_MAX_Y - 1;
    box.right = 0;
    box.bottom = 0;
    for (i = 0; i <= n; i++)
    {
        if (vertices[i].x < box.left)
        {
            box.left = vertices[i].x;
        }
        if (vertices[i].x > box.right)
        {
            box.right = vertices[i].x;
            if (box.right > (ENGINE_MAX_X - 1))
            {
                box.right = ENGINE_MAX_X - 1;
            }
        }
        if (vertices[i].y < box.top)
        {
            box.top = vertices[i].y;
        }
        if (vertices[i].y > box.bottom)
        {
            box.bottom = vertices[i].y;
            if (box.bottom > (ENGINE_MAX_Y - 1))
            {
                box.bottom = ENGINE_MAX_Y - 1;
            }
        }
    }

    return (box);
}

/* --------------------------------------------------------------------------
  check_line - checks coordinates of downward lines for device
  coordinate violation. These lines have a valid starting coordinate but
  the ending coordinate is outside the device range. As a result, the line
  is not drawn at all. This routine will adjust the ending coordinate so
  that it is inside the device range and the line's angle will be
  maintained. The starting y coordinate must be smaller then the ending
  y coordinate.
-------------------------------------------------------------------------- */
void check_line(int *startx, int *starty, int *endx, int *endy)
{
    double delx1, dely1, delx2, dely2;
    double angle;
    int temp;

    if ((*startx > ENGINE_MAX_X) || (*starty > ENGINE_MAX_Y)) return;
    if (*endx > ENGINE_MAX_X) return;

    if (*endy > ENGINE_MAX_Y)
    {
        delx1 = (double)(abs(*endx - *startx));
        dely1 = (double)(abs(*endy - *starty));
        dely2 = (double)(abs(ENGINE_MAX_Y - *starty));
        *endy = ENGINE_MAX_Y;
        angle = atan(delx1/dely1);
        delx2 = tan(angle) * dely2;
        temp = (int)(delx2);
        if (*endx > *startx)
        {
            *endx = *startx + temp;
        }
        else
        {
            *endx = *startx - temp;
        }
    }
}

/* --------------------------------------------------------------------------
  pgl_drawoutline - draw an outline for polygon fills
-------------------------------------------------------------------------- */
void pgl_outline(int x1, int y1, int x2, int y2)
{
    /* check vertical lines for invalid engine coordinates */
    if (x1 == x2)
    {
        if (y1 < ENGINE_MIN_Y) y1 = ENGINE_MIN_Y;
        if (y2 < ENGINE_MIN_Y) y2 = ENGINE_MIN_Y;
        if (y1 > ENGINE_MAX_Y) y1 = ENGINE_MAX_Y;
        if (y2 > ENGINE_MAX_Y) y2 = ENGINE_MAX_Y;
    }

    /* draw boundary lines from top to bottom */
    if (y1 > y2)
    {
        check_line(&x2, &y2, &x1, &y1);
        pgl_drawln(x2, y2, x1, y1);
    }
    else
    {
        check_line(&x1, &y1, &x2, &y2);
        pgl_drawln(x1, y1, x2, y2);
    }
}

/* --------------------------------------------------------------------------
  pgl_packhost - pack a variable width fillpattern into 32 bit dword format
                 up to a bit width of 32 bits
-------------------------------------------------------------------------- */
unsigned long pgl_packhost(unsigned long fillpattern)
{
    unsigned long hostdata;
    int reps, i;

    hostdata = fillpattern;
    reps = 32 / PGL_attr.fillpatternwidth;
    if (reps > 1)
    {
        for (i = 0; i < reps; i++)
        {
            hostdata = hostdata << PGL_attr.fillpatternwidth;
            hostdata = hostdata | fillpattern;
        }
    }

    return (hostdata);
}

/* --------------------------------------------------------------------------
  PGL_getnpolylines - determine number of lines needed to setup polygon

  This function is useful for determining if there's enough off-screen memory
  available in lines (in native pitch and color depth). It is dependent on
  the starting source line for the polygon outlines (see PGL_getsrcpoly()
  and PGL_setsrcpoly() functions). Return line requirement is rounded up.
  The same conditions apply for the input parameters as PGL_drawpolygon().
  If the shape has less than 3 sides or if the native color depth is 24 bpp,
  0 is returned.
-------------------------------------------------------------------------- */
int PGL_getnpolylines(int n, PGL_point *vertices)
{
    int recwidth, recheight;
    int reqlines, monopitch;
    PGL_scissor box;

    if (n < 3)
    {
        return (0);
    }

    // no support for 24 bpp
    if (PGL_modecfg.bpp == 24)
    {
        return (0);
    }

    // determine box size of shape and clamp coordinates to mode size
    box = pgl_boxsize(n, vertices);
    if (box.left < 0) box.left = 0;
    if (box.top < 0) box.top = 0;
    if (box.right > (PGL_modecfg.xres-1)) box.right = PGL_modecfg.xres-1;
    if (box.bottom > (PGL_modecfg.yres-1)) box.bottom = PGL_modecfg.yres-1;
    recwidth = box.right - box.left + 1;
    recheight = box.bottom - box.top + 1;

    // determine 1bpp pitch to use (pitch must be a multiple of 64 pixels)
    monopitch = ((recwidth + 63) / 64) * 64;
    if (monopitch == 0)
    {
        monopitch = 64;
    }

    // determine required lines in native pitch and color depth
    reqlines = (recheight / (monopitch / PGL_modecfg.bpp)) + 2;

    return (reqlines);
}

/* --------------------------------------------------------------------------
  PGL_drawpolygon - draw a polygon

  An "n" sided polygon is drawn and filled with the current foreground color,
  mix, and fill pattern. The vertices given should form an enclosed shape
  outline. Note that an "n" sided polygon requires "n+1" point pairs. A
  polygon fill requires that the outline be drawn in "off-screen" memory,
  followed by a blit & fill to the "on-screen" memory destination.

  The following conditions will prevent anything from being drawn:
    - not enough off-screen memory to permit the fill
    - shape has less than 3 sides
    - native color depth is 24 bpp (no polygon support)

  NO_ERROR is returned if successful, YES_ERROR if not.
-------------------------------------------------------------------------- */
int PGL_drawpolygon(int n, PGL_point *vertices)
{
    int i, j, k, adjwidth, reqlines;
    int fillindex, chkwrts;
    int reps, shift, resbits;
    PGL_scissor s_scis;
    PGL_scissor box;
    int recwidth, recheight;
    unsigned long temp1, temp2, temp3, temp4;
    unsigned long temp5, temp6, temp7, temp8;
    unsigned long temp9;
    unsigned long offset, hostdata;
    unsigned long resdata;
    unsigned long monopitch;

    /* polygon must have at least 3 sides to enclose an area */
    if (n < 3)
    {
        return (YES_ERROR);
    }

    // no support for 24 bpp
    if (PGL_modecfg.bpp == 24)
    {
        return (YES_ERROR);
    }

    /* determine box size of shape and clamp coordinates to mode size */
    box = pgl_boxsize(n, vertices);
    if (box.left < 0) box.left = 0;
    if (box.top < 0) box.top = 0;
    if (box.right > (PGL_modecfg.xres-1)) box.right = PGL_modecfg.xres-1;
    if (box.bottom > (PGL_modecfg.yres-1)) box.bottom = PGL_modecfg.yres-1;
    recwidth = box.right - box.left + 1;
    recheight = box.bottom - box.top + 1;

    // determine 1bpp pitch to use (pitch must be a multiple of 64 pixels)
    monopitch = (unsigned long)(((recwidth + 63) / 64) * 64);
    if (monopitch == 0)
    {
        monopitch = 64;
    }

    // determine if there's enough off-screen memory (lines are in 1 bpp)
    reqlines = (recheight / ((int)monopitch / PGL_modecfg.bpp)) + 2;
    if (reqlines > (PGL_modecfg.maxy - PGL_attr.srcpoly_y + 1))
    {
        return (YES_ERROR);
    }

    /* calculate qword offset to start of off-screen memory */
    offset = pgl_getxyoffset(0, PGL_attr.srcpoly_y) / 8;

    // convert monopitch for use in SRC & DST pitch registers (dword)
    monopitch = (monopitch / 8) << 22;

    /* save vital registers */
    PGL_waitforidle();
    temp1 = regr(DP_PIX_WIDTH);
    temp2 = regr(DP_MIX);
    temp3 = regr(DST_CNTL);
    temp4 = regr(SRC_CNTL);
    temp5 = regr(DP_FRGD_CLR);
    temp6 = regr(DP_SRC);
    temp7 = regr(DST_OFF_PITCH);
    temp8 = regr(SRC_OFF_PITCH);
    temp9 = regr(PAT_CNTL);

    /* save clip rectangle */
    PGL_getclipsize(&s_scis);

    /* setup to clear area for outline draws in 1 bpp */
    PGL_waitforfifo(4);
    regw(DST_OFF_PITCH, (monopitch & 0xffc00000) | offset);
    if (PGL_modecfg.bpp == 4)
    {
        regw(DP_PIX_WIDTH, 0);
    }
    else
    {
        regw(DP_PIX_WIDTH, BYTE_ORDER_LSB_TO_MSB);
    }

    /* clear off-screen memory area for outlines */
    regw(DP_SRC, FRGD_SRC_FRGD_CLR);
    regw(DP_MIX, FRGD_MIX_ZERO | BKGD_MIX_ZERO);

    // make sure clear width is a multiple of 32 pixels for pattern fills
    adjwidth = ((recwidth + 31) / 32) * 32;
    if (adjwidth == 0)
    {
        adjwidth = 32;
    }
    pgl_solidrect(0, 0, adjwidth, recheight);

    /* setup for outline draws in 1 bpp */
    PGL_waitforfifo(4);
    regw(DP_MIX, FRGD_MIX_D_XOR_S | BKGD_MIX_ZERO);
    regw(DST_CNTL, DST_POLYGON_ENABLE |
                   DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    regw(SRC_CNTL, SRC_LINE_X_LEFT_TO_RIGHT);
    regw(DP_FRGD_CLR, 1);

    /* set the scissors to off-screen outline draw area */
    PGL_setclipsize(0, 0, recwidth, recheight);

    /* draw outline of shape */
    for (i = 0; i < n; i++)
    {
        pgl_outline(vertices[i].x - box.left,
                    vertices[i].y - box.top,
                    vertices[i+1].x - box.left,
                    vertices[i+1].y - box.top);
    }

    /* set clip rectangle to on-screen fill area */
    PGL_setclipsize(s_scis.left, s_scis.top, s_scis.right, s_scis.bottom);

    /* set attributes for polygon fill in current bpp */
    PGL_waitforfifo(5);
    regw(DST_OFF_PITCH, temp7);
    regw(SRC_OFF_PITCH, (monopitch & 0xffc00000) | offset);
    if (PGL_modecfg.bpp == 4)
    {
        regw(DP_PIX_WIDTH, temp1 & 0x000000ff);
    }
    else
    {
        regw(DP_PIX_WIDTH, (temp1 & 0x000000ff) | BYTE_ORDER_LSB_TO_MSB);
    }
    regw(DP_MIX, temp2);

    // restore application's fill color
    if ((PGL_modecfg.chip_type == CHIP_GX_ID) &&
       ((regr(GEN_TEST_CNTL) & BLOCKWRITE_ENABLE) == BLOCKWRITE_ENABLE))
    {
        // block write fix for Mach64 rev. C
        PGL_waitforfifo(5);
        regw(DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
        regw(DP_FRGD_CLR, temp5);
        regw(DST_Y_X, PGL_modecfg.yres);    // X = 0
        regw(DST_HEIGHT_WIDTH, 0x00010001);
        regw(DST_CNTL, DST_POLYGON_ENABLE |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    }
    else
    {
        regw(DP_FRGD_CLR, temp5);
    }

    /* ---- setup for patterned POLY fill if selected ----

       Three methods depending on type of fill:

       1) solid fill - use foreground and background color registers for
                       solid fills

       2) host fill  - use host register for patterned fills with pattern
                       widths that are not equal to 1, 2, 4, 8 (this
                       is the slowest method)

       3) pattern fill - use pattern registers for patterned fills when the
                         pattern widths equal 1, 2, 4, 8 (this is the
                         optimized version of (2))
    */

    if ((PGL_attr.fillsolid == USERDEFINED_FILL) && (PGL_attr.fastpolyflag == 1))
    {
        // method (3) - use pattern registers to do fill

        PGL_waitforfifo(2);
        regw(PAT_CNTL, PAT_MONO_8x8_ENABLE);
        regw(DP_SRC, MONO_SRC_PATTERN | FRGD_SRC_FRGD_CLR);

        fillindex = 0;
        for (i = 0; i < recheight; i++)
        {
            hostdata = pgl_packhost(PGL_attr.fillpattern[fillindex]);

            fillindex++;
            if (fillindex >= PGL_attr.fillpatternheight)
            {
                fillindex = 0;
            }

            // setup pattern registers for each line
            PGL_waitforfifo(6);

            regw(PAT_REG0, hostdata);
            regw(PAT_REG1, hostdata);

            // perform POLY blit-fill for each line
            regw(SRC_Y_X, i);           // X = 0
            regw(SRC_HEIGHT1_WIDTH1, ((unsigned long)(adjwidth) << 16) | 1);
            regw(DST_Y_X, ((unsigned long)(box.left) << 16) | (box.top + i));
            regw(DST_HEIGHT_WIDTH, ((unsigned long)(adjwidth) << 16) | 1);
        }
    }
    else
    {
        // used for method (1) - SOLID fill case,
        // and setup for method (2) - PATTERNED fill case with host registers
        PGL_waitforfifo(5);

        if (PGL_attr.fillsolid == USERDEFINED_FILL)
        {
            // use monochrome HOST source
            regw(DP_SRC, MONO_SRC_HOST | FRGD_SRC_FRGD_CLR);
        }

        // perform POLY blit-fill for entire polygon
        regw(SRC_Y_X, 0);
        regw(SRC_HEIGHT1_WIDTH1, ((unsigned long)(adjwidth) << 16) | recheight);
        regw(DST_Y_X, ((unsigned long)(box.left) << 16) | box.top);
        regw(DST_HEIGHT_WIDTH, ((unsigned long)(adjwidth) << 16) | recheight);
    }

    if ((PGL_attr.fillsolid == USERDEFINED_FILL) && (PGL_attr.fastpolyflag == 0))
    {
        // fill part of method (3) - host register polygon fill

        // initialize loop variables
        PGL_waitforemptyfifo();
        fillindex = 0;
        chkwrts = 0;
        reps = 32 / PGL_attr.fillpatternwidth;
        shift = ((reps + 1) * PGL_attr.fillpatternwidth) - 32;

        // setup loop width in 32 bit pieces for monochrome HOST register
        k = adjwidth / 32;
        if (k == 0)
        {
            k = 1;
        }

        // height loop
        for (j = 0; j < recheight; j++)
        {
            // pack hostdata at the start of each line
            hostdata = pgl_packhost(PGL_attr.fillpattern[fillindex]);
            resbits = 0;

            // width loop
            for (i = 0; i < k; i++)
            {
                // write 32 bit HOST data
                regw(HOST_DATA0, hostdata);

                // setup host data for next 32 bit write
                resbits = resbits + shift;
                if (resbits > PGL_attr.fillpatternwidth)
                {
                    resbits = resbits - PGL_attr.fillpatternwidth;
                    hostdata = (unsigned long)(pgl_packhost(PGL_attr.fillpattern[fillindex]) << resbits);
                    resdata = (unsigned long)(PGL_attr.fillpattern[fillindex] >> (PGL_attr.fillpatternwidth - resbits));
                    hostdata = hostdata | resdata;
                }
                else
                {
                    hostdata = (unsigned long)(hostdata << shift);
                    resdata = (unsigned long)(PGL_attr.fillpattern[fillindex] >> (PGL_attr.fillpatternwidth - resbits));
                    hostdata = hostdata | resdata;
                }

                // optimize times needed to wait for engine idle
                chkwrts++;
                if (chkwrts > 15)
                {
                    PGL_waitforemptyfifo();
                    chkwrts = 0;
                }
            }
            fillindex++;
            if (fillindex >= PGL_attr.fillpatternheight)
            {
                fillindex = 0;
            }
        }
    }

    /* restore */
    PGL_waitforidle();
    regw(DP_PIX_WIDTH, temp1);
    regw(DP_MIX, temp2);
    regw(DST_CNTL, temp3);
    regw(SRC_CNTL, temp4);
    regw(DP_FRGD_CLR, temp5);
    regw(DP_SRC, temp6);
    regw(DST_OFF_PITCH, temp7);
    regw(SRC_OFF_PITCH, temp8);
    regw(PAT_CNTL, temp9);

    // set X,Y last position
    regw(DST_Y_X, ((unsigned long)(box.right) << 16) | box.bottom);

    return (NO_ERROR);
}

