/*==========================================================================
  STYLE2.C

  Example code to demonstrate draw operations style 2. A CYAN filled
  rectangle is drawn.

  1. At initialization, store engine context(s) at appropriate addresses.
  2. Load desired default draw context.
  3. Setup draw trajectory registers.
  4. Initiate the draw operation.
  5. Repeat steps 2 to 4 for each operation

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"

#define INSTANCE 0

unsigned long context[64];      // context array

/* --------------------------------------------------------------------------
  GETPAGEPTR - calculate 32K read/write pointer from a given memory address.
-------------------------------------------------------------------------- */
unsigned long getpageptr (unsigned long addr)
{
    unsigned long pageptr;

    pageptr = addr / 0x8000;

    return (pageptr);
}

/* --------------------------------------------------------------------------
  GETOFFSET - calculate offset into 32K VGA aperture from the physical
              address and page pointer.
-------------------------------------------------------------------------- */
unsigned long getoffset (unsigned long phyaddr, unsigned long pageptr)
{
    unsigned long ptraddr, offset;

    ptraddr = pageptr * 0x8000;
    offset = phyaddr - ptraddr;

    return (offset);
}

/* --------------------------------------------------------------------------
  UPLOAD_CONTEXT - copy context array to video memory according to given
                   video memory address
-------------------------------------------------------------------------- */
void upload_context (unsigned long address)
{
    unsigned long writepage, writeoffset;
    int i;

    // Use VGA or LINEAR aperture depending on current aperture settings
    if (modeinfo.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // set write pointer to correct page
        writepage = getpageptr (address);
        writeoffset = getoffset (address, writepage);

        // upper 32k VGA aperture pointer is used for overflow
        regw (MEM_VGA_WP_SEL, ((writepage + 1) << 16) | writepage);

        // transfer the 256 bytes of context data to the appropriate address
        for (i = 0; i < 64; i++)
        {
            *((unsigned long far *) (LOW_APERTURE_BASE +
                                     writeoffset)) = context[i];
            writeoffset = writeoffset + 4;
        }
    }
    else // Linear aperture is enabled
    {
        movemem ((void far *) (context), address + modeinfo.aperture_address,
                 256 / 2, MEM_WRITE);
    }
}


/* Main C program */

int main (int argc, char *argv[])
{
    int memcntl, i;
    unsigned long contextaddr;
    unsigned long xres, yres, pitch;
    unsigned long save_dst_cntl;

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // Process the command line arguments to override default resolution
    // and color depth settings.
    process_command_line (argc, argv);

    // determine if VGA aperture or linear aperture will be used for transfer
    if (querydata.vga_type == VGA_ENABLE)
    {
        // The VGA controller is normally set in planar mode. Data transfer
        // through the VGA aperture (low and high 32K pages) requires that the
        // VGA controller be set in a packed pixel mode where the pixel data
        // is arranged contigiously.
        set_packed_pixel ();
    }

    // set an accelerator mode
    if (open_mode (gmode_res, PITCH_XRES, gclr_depth) != NO_ERROR)
    {
        // close the mode if a packed pixel mode was set.
        if (querydata.vga_type == VGA_ENABLE) close_mode ();
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // get modal information
    xres = (unsigned long) (modeinfo.xres);
    yres = (unsigned long) (modeinfo.yres);
    pitch = (unsigned long) (modeinfo.pitch);

    if (modeinfo.bpp == 24)
    {
        pitch = pitch * 3;
    }

    // setup engine context and clear screen
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);


    // Step 1: Store engine context at appropriate address to load later

    // determine top of memory address
    memcntl = ior16 (ioMEM_CNTL);
    switch (memcntl & 7)
    {
        case 0:
            contextaddr = 0x80000;              // 512K
            break;
        case 1:
            contextaddr = 0x100000;             // 1M
            break;
        case 2:
            contextaddr = 0x200000;             // 2M
            break;
        case 3:
            contextaddr = 0x400000;             // 4M
            break;
        case 4:
            contextaddr = 0x600000;             // 6M
            break;
        case 5:
            contextaddr = 0x800000;             // 8M
            break;
    }

    /*
       Fill context for context pointer 4. Each context pointer represents
       256 bytes of video memory. The context load address decreases as the
       context load pointer increases. Context pointer 0 points to the top of
       memory - 256. Some restrictions apply for context pointers 0-3 if the
       linear aperture size equals the memory size (i.e. 4M aperture size,
       4M of video memory). In this case, the memory mapped registers occupy
       1K of memory below the top of aperture. The only method to reach this
       area in this case is to use the VGA paged aperture.
    */
    context[0] = 0xFFFFFFFF;
    context[1] = 0x00000000;
    context[2] = (pitch / 8) << 22;         // destination pitch
    context[3] = 0x00000000;
    context[4] = 0x00000000;
    context[5] = 0x00000000;
    context[6] = 0x00000000;
    context[7] = 0x00000000;
    context[8] = (pitch / 8) << 22;         // source pitch
    context[9] = 0x00000000;
    context[10] = 0x00000000;
    context[11] = 0x00000000;
    context[12] = 0x00000000;
    context[13] = 0x00000000;
    context[14] = 0x00000000;
    if (modeinfo.bpp == 24)
    {
        context[15] = (xres * 3) << 16;               // scissors
    }
    else
    {
        context[15] = xres << 16;               // scissors
    }
    context[16] = yres << 16;
    context[17] = 0x00000000;
    context[18] = 0xFFFFFFFF;
    context[19] = 0xFFFFFFFF;

    // set DP_PIX_WIDTH and CHAIN_MASK according to color depth
    switch (modeinfo.bpp)
    {
        case 4:
            context[20] = 0x00008888;
            context[21] = HOST_4BPP | SRC_4BPP | DST_4BPP;
            break;
        case 16:
            if (modeinfo.depth == 555)
            {
                // 555 color weighting
                context[20] = 0x00004210;
                context[21] = HOST_15BPP | SRC_15BPP | DST_15BPP;
            }
            else
            {
                // 565 color weighting
                context[20] = 0x00008410;
                context[21] = HOST_16BPP | SRC_16BPP | DST_16BPP;
            }
            break;
        case 32:
            context[20] = 0x00008080;
            context[21] = HOST_32BPP | SRC_32BPP | DST_32BPP;
            break;
        default:
        case 8:
        case 24:
            context[20] = 0x00008080;
            context[21] = HOST_8BPP | SRC_8BPP | DST_8BPP;
            break;
    }

    context[22] = FRGD_MIX_S | BKGD_MIX_S;
    context[23] = FRGD_SRC_FRGD_CLR;
    context[24] = 0x00000000;
    context[25] = 0xFFFFFFFF;
    context[26] = 0x00000000;
    if (modeinfo.bpp == 24)
    {
        context[27] = DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT |
                      DST_24_ROTATION_ENABLE;
    }
    else
    {
        context[27] = DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT;
    }
    context[28] = 0x00000000;
    for (i = 29; i < 64; i++)
    {
        context[i] = 0;
    }

    // Upload context

    // Context load address calculation:
    //
    // Address = total video memory - (context pointer + 1) * 0x100)
    //

    // load up first context -> load pointer = 4
    upload_context (contextaddr - ((4 + 1) * 0x100));

    // Step 2: Load default context (load context pointer 4)

    // load all GUI registers
    wait_for_fifo (2);
    regw (CONTEXT_MASK, 0xFFFFFFFF);

    // load using context pointer 4
    regw (CONTEXT_LOAD_CNTL, CONTEXT_LOAD | 4);

    // Step 3: Setup draw trajectory of draw operation - i.e. rectangle fill

    // set foreground color
    wait_for_fifo (1);
    regw (DP_FRGD_CLR, get_color_code (LIGHTCYAN));

    // draw a filled rectangle
    if (modeinfo.bpp == 24)
    {
        wait_for_idle ();
        save_dst_cntl = regr (DST_CNTL);
        wait_for_fifo (2);
        regw (DST_CNTL, save_dst_cntl |
                        (GET24BPPROTATION (modeinfo.xres / 32) << 8));
        regw (DST_X, (modeinfo.xres / 32) * 3);
    }
    else
    {
        wait_for_fifo (1);
        regw (DST_X, modeinfo.xres / 32);
    }
    wait_for_fifo (3);
    regw (DST_Y, modeinfo.yres / 32);
    regw (DST_HEIGHT, modeinfo.yres / 4);

    // Step 4: Initiate the draw operation - DST_WIDTH is a draw initiator
    if (modeinfo.bpp == 24)
    {
        regw (DST_WIDTH, (modeinfo.xres / 2) * 3);
    }
    else
    {
        regw (DST_WIDTH, modeinfo.xres / 2);
    }

    // wait for a carriage return
    getch ();

    if (modeinfo.bpp == 24)
    {
        wait_for_fifo (1);
        regw (DST_CNTL, save_dst_cntl);
    }

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

