/*==========================================================================
  PAN.C

  Example code to show scrolling and panning of the display.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"


// Keyboard scan and ascii codes for demo

#define ESC         0x1B  // ascii code
#define LEFT_ARROW  0x4B  // scan code
#define RIGHT_ARROW 0x4D  // scan code
#define UP_ARROW    0x48  // scan code
#define DOWN_ARROW  0x50  // scan code
#define HOME        0x47  // scan code
#define END         0x4F  // scan code

#define INSTANCE 0

/* --------------------------------------------------------------------------
  GET_KEY - get keyboard scan code using system ROM call

  Upper 8 bits = scan code
  Lower 8 bits = ascii code
-------------------------------------------------------------------------- */
int get_key (void)
{
    union REGS regs;

    regs.x.ax = 0x1000;
    int86 (0x16, &regs, &regs);

    return (regs.x.ax);
}


/* Main C program */

int main (void)
{
    unsigned long offset;
    int color, ch;
    int xindex, yindex, xmax, ymax;
    int max_x, max_y, size_x, size_y;

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // set an accelerator mode of 640x480 with a pitch of 1024
    if (open_mode (MODE_640x480, PITCH_1024, COLOR_DEPTH_8) != NO_ERROR)
    {
        printf ("Error in setting display mode.\n");
        return (1);
    }

    /*
       To scroll and pan the display, the viewable display area must be
       smaller than the CRTC display area. In this case, the viewable area
       is 640x480 while the CRTC display area is 1024x1024.
    */

    // 'desktop' size = 1024 x 1024
    max_x = 1024;
    max_y = 1024;

    // 'viewable screen size = 640 x 480
    size_x = 640;
    size_y = 480;

    // setup engine context and clear screen
    init_engine ();

    // adjust engine scissors to desktop area (1024x1024)
    wait_for_fifo (4);
    regw (SC_LEFT, 0);
    regw (SC_TOP, 0);
    regw (SC_RIGHT, max_x - 1);
    regw (SC_BOTTOM, max_y - 1);

    // clear area
    clear_screen (0, 0, max_x, max_y);

    // Fill area with some full size rectangles (CRTC width)
    for (color = DARKBLUE-1; color < WHITE; color++)
    {
        set_fg_color (get_color_code (color + 1));
        draw_rectangle (color * 30,
                        color * 30,
                        max_x - (color * 60) - 1,
                        max_y - (color * 60) - 1);
    }

    // Scroll and pan around image:
    //
    //   Viewable display is 640x480
    //   Image size is 1024x1024
    //
    //   ESC         - exit
    //   LEFT_ARROW  - move left
    //   RIGHT_ARROW - move right
    //   UP_ARROW    - move up
    //   DOWN_ARROW  - move down
    //   HOME        - move to top-left corner
    //   END         - move to bottom-right corner
    //

    xindex = 0;
    yindex = 0;

    xmax = (max_x - size_x) / 2;     // 2 pixels per step (in 8 bpp)
    ymax = (max_y - size_y) / 2;     // 2 lines per step
    if (modeinfo.bpp == 4)
    {
        xmax = xmax / 2;
    }
    else
    {
        xmax = xmax * (modeinfo.bpp / 8);
    }

    offset = 0;
    ch = 0;
    while ((ch & 0xFF) != ESC)
    {
        // wait for key input
        while (kbhit () == 0) ;
        ch = get_key ();

        // position CRTC offset according to key input
        if ((ch & 0xFF) != ESC)
        {
            switch (ch >> 8)
            {
                case LEFT_ARROW:   // move left 2 pixels
                    if (xindex > 0)
                    {
                        offset = offset - 1;
                        xindex = xindex - 4;
                    }
                    break;

                case RIGHT_ARROW:  // move right 2 pixels
                    if (xindex < xmax)
                    {
                        offset = offset + 1;
                        xindex = xindex + 4;
                    }
                    break;

                case UP_ARROW:     // move up 2 lines
                    if (yindex > 0)
                    {
                        if (modeinfo.bpp == 4)
                        {
                            offset = offset - (256 / 2);
                        }
                        else
                        {
                            offset = offset - (256 * (modeinfo.bpp / 8));
                        }
                        yindex = yindex - 1;
                    }
                    break;

                case DOWN_ARROW:   // move down 2 lines
                    if (yindex < ymax)
                    {
                        if (modeinfo.bpp == 4)
                        {
                            offset = offset + (256 / 2);
                        }
                        else
                        {
                            offset = offset + (256 * (modeinfo.bpp / 8));
                        }
                        yindex = yindex + 1;
                    }
                    break;

                case HOME:         // move to top-left corner (0, 0)
                    offset = 0;
                    xindex = 0;
                    yindex = 0;
                    break;

                case END:          // move to bottom-right corner

                    // calculate dword address of coordinate (1024-640,1024-480)

                    offset = (unsigned long) (max_y - size_y);
                    offset = (unsigned long) (offset * max_x);  // pitch
                    if (modeinfo.bpp == 4)
                    {
                        offset = (unsigned long) (offset / 2);
                        offset = (unsigned long) (offset +
                                                  ((max_x - size_x) / 2));
                    }
                    else
                    {
                        offset = (unsigned long) (offset * (modeinfo.bpp / 8));
                        offset = (unsigned long) (offset +
                                                  ((modeinfo.bpp / 8) *
                                                   (max_x - size_x)));
                    }
                    offset = offset / 8;
                    xindex = xmax;
                    yindex = ymax;
                    break;
            }

            // vary CRTC offset while maintaining mode pitch
            iow32 (ioCRTC_OFF_PITCH,
                   (ior32 (ioCRTC_OFF_PITCH) & 0xFFC00000) | offset);
        }
    }

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

