/*==========================================================================
  DBUF.C

  Example code to show double buffered drawing using the system timer.
  Several image "frames" are loaded into off-screen memory so that they can
  be blited to the screen in sequence. Image tearing is avoided by using
  double page buffering. The page being updated is never the page being
  displayed. In this example, the page buffer is controlled by waiting for a
  system timer tick which yields an approximate frame rate of 18. A page swap
  is done every timer tick. Double buffering does not require waiting for a
  specific vertical line or blanking before drawing. If the frame is more
  than approximately half the vertical sync frequency, image tearing will
  occur. In this case, it will be necessary to wait for a specific vertical
  line before drawing.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"
#include "..\util\vtga.h"
#include "isr8.h"

#define INSTANCE            0
#define BACKGROUND_COLOR    0x11


/* Main C program */

int main (void)
{
    char filename[20];
    TARGA_HEADER header;
    int width, height;
    int srcx, srcy;
    int savex1, savey1;
    int savex2, savey2;
    int savex, savey;
    int i, j;
    int y_draw;
    int frame;
    int frames;
    int framesperwidth;
    int topline;
    int y_update;
    int update_flag;
    int old_flag;
    int step;
    unsigned long offset;
    POINT points[8];

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // check if Mach 64 VGA controller is enabled
    if (querydata.vga_type != VGA_ENABLE)
    {
        printf ("This sample code example requires an enabled mach64 VGA "
                "controller.\n");
        return (1);
    }

    // set an accelerator mode
    if (open_mode (MODE_640x480, PITCH_XRES, COLOR_DEPTH_8) != NO_ERROR)
    {
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // check if color depth is 8 bpp
    if (modeinfo.bpp != 8)
    {
        close_mode ();
        printf ("This example requires an 8 bpp mode.\n");
        return (1);
    }

    // setup engine context and clear screen
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // get targa header information
    if (get_targa_header ("..\\image\\frame1.tga", &header) != SUCCESS)
    {
        close_mode ();
        printf ("Error reading targa file header information.\n");
        return (1);
    }

    // setup image size, source area, and save area
    width = header.width;
    height = header.height;
    srcx = 0;
    srcy = (modeinfo.yres * 2) + height;
    savex1 = 0;
    savey1 = modeinfo.yres * 2;
    savex2 = width;
    savey2 = savey1;
    y_draw = modeinfo.yres - height;
    step = 2;

    // inform ISR where second page begins (QWORD address)

    // calculate byte address
    offset = (unsigned long) (modeinfo.yres);
    offset = (unsigned long) (offset * modeinfo.pitch);
    if (modeinfo.bpp == 4)
    {
        offset = (unsigned long) (offset / 2);
    }
    else
    {
        offset = (unsigned long) (offset * (modeinfo.bpp / 8));
    }

    // convert byte address to qword address
    offset = offset / 8;

    // add current pitch from CRTC_OFF_PITCH to variable
    offset = offset | (ior32 (ioCRTC_OFF_PITCH) & 0xFFC00000);
    setcrtcoffset (offset);

    // determine how large to expand the scissors
    frames = 8;
    framesperwidth = modeinfo.xres / width;
    topline = frames / framesperwidth;
    if ((topline * framesperwidth) != frames)
    {
        topline++;
    }
    topline = ((topline + 1) * height) + (modeinfo.yres * 2);

    // expand scissors
    wait_for_fifo (4);
    regw (SC_LEFT, 0);
    regw (SC_TOP, 0);
    regw (SC_RIGHT, modeinfo.xres - 1);
    regw (SC_BOTTOM, topline - 1);

    // setup background the same as the image background
    set_fg_color (BACKGROUND_COLOR);
    draw_rectangle (0, 0, modeinfo.xres, modeinfo.yres);

    // copy background (first buffer) to second buffer
    blit (0, 0, 0, modeinfo.yres, modeinfo.xres, modeinfo.yres);

    // load source images
    frame = 0;
    i = 0;
    j = 0;
    while (frame < frames)
    {
        // record each frame coordinate
        points[frame].x = srcx + (width * i);
        points[frame].y = srcy + (height * j);

        // load next frame image into video memory
        sprintf (filename, "..\\image\\frame%d.tga", frame + 1);
        if (load_targa (filename, points[frame].x, points[frame].y) != SUCCESS)
        {
            close_mode ();
            printf ("Error loading targa file to memory.\n");
            return (1);
        }

        // adjust location of frame load coordinate as necessary
        frame++;
        i++;
        if (i > ((modeinfo.xres / header.width) - 1))
        {
            i = 0;
            j++;
        }
    }

    // set palette from targa color table (8 bpp)
    if (header.pixel_depth == 8)
    {
        if (set_targa_palette ("..\\image\\frame1.tga") != SUCCESS)
        {
            close_mode ();
            printf ("Error reading targa file color table information.\n");
            return (1);
        }
    }

    // wait for a key to start
    getch ();

    // setup for blits
    wait_for_fifo (3);
    regw (DP_SRC, FRGD_SRC_BLIT);
    regw (SRC_CNTL, 0);
    regw (DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);

    // copy starting postions to buffer save areas
    blit (0, y_draw, savex1, savey1, width, height);
    blit (0, modeinfo.yres + y_draw, savex2, savey2, width, height);

    // setup loop variables
    i = 0;
    frame = 0;
    update_flag = getswapflag ();
    old_flag = update_flag;

    // enable timer ISR for page swaps each timer tick
    inittimerisr ();

    // main draw loop
    while (i < (modeinfo.xres-width))
    {
        // syncronize frame updates with ISR
        while (update_flag == old_flag)
        {
            update_flag = getswapflag ();
        }
        old_flag = update_flag;

        // don't allow ISR to swap page while updating frame
        interrupts_off ();

        // set update and save variables according to active frame
        if (update_flag == 1)
        {
            // display frame 1, update frame 2
            y_update = modeinfo.yres;
            savex = savex2;
            savey = savey2;
        }
        else
        {
            // display frame 2, update frame 1
            y_update = 0;
            savex = savex1;
            savey = savey1;
        }

        // restore current frame from last frame (each page lags by 2 frames)
        if (i > step)
        {
            blit (savex, savey, i - (2 * step), y_update + y_draw,
                  width, height);
        }

        // save current frame
        blit (i, y_update + y_draw, savex, savey, width, height);

        // update current frame
        blit (points[frame].x, points[frame].y, i, y_update + y_draw,
              width, height);

        // allow ISR to swap page
        interrupts_on ();

        // increment image position
        i = i + step;

        // determine next frame image
        frame++;
        if (frame >= frames)
        {
            frame = 0;
        }
    }

    // disable timer ISR
    canceltimerisr ();

    // wait for a carriage return
    getch ();

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

