/*
Scan the body of an ANSI escape or control sequence, starting after the
introductory chars ESC or ESC [ (respectively).  See section IV of the
Ann Arbor manual for details.

If MKSCANCTL is defined, make scanctl (which scans an ANSI control
sequence).  Otherwise make scanesc (which scans an escape sequence).
*/



#ifdef MKSCANCTL
#define SCANNAME scanctl
#define SCANNSTR "scanctl"
#define FHINYBBLE 0x40
#else
#define SCANNAME scanesc
#define SCANNSTR "scanesc"
#define FHINYBBLE 0x30
#endif



int SCANNAME(iop,stp)		/* return number of chars scanned */
register struct IOStruct *iop;	/* IO buffer structure */
char **stp;			/* starting point of scan */
{
    register char *p;		/* for scanning */
    extern char *bufend;	/* first char off end of buffer */
    int shift;			/* no. of chars shifted by FixSplitSeq */
    int nread;			/* no. of chars read by FixSplitSeq */

#   ifdef DEBUG
    fprintf(stderr, "%s(%lx,%lx)\n", SCANNSTR, iop, *stp);
#   endif

    p = *stp;
    *bufend = '\0';

/* only look for parameters if it's scanctl */
#   ifdef MKSCANCTL
    /* parameters */
    while ((*p & 0x70) == 0x30) {
	p++;
    }

    /* if sequence is split across buffers -- should be rare.  It
    should be extremely rare if not impossible to go around this
    loop more than once. */
    while (p >= bufend) {
#	ifdef DEBUG
	fprintf(stderr,
	    "split sequence in param part of %s\n",
	    SCANNSTR);
#	endif
	shift = FixSplitSeq(iop, &nread);
	TSTEXC;
	p -= shift;
	*stp -= shift;
	while ((*p & 0x70) == 0x30) {
	    p++;
	}
    }
#   ifdef DEBUG
    fprintf(stderr, "parameter part: <");
    dumpstr(stderr, *stp, p);
    fprintf(stderr, ">\n");
#   endif
#   endif

    /* intermediate codes */
    while ((*p & 0x70) == 0x20) {
	p++;
    }

    /* if sequence is split across buffers -- should be rare.  It
    should be extremely rare if not impossible to go around this
    loop more than once. */
    while (p >= bufend) {
#	ifdef DEBUG
	fprintf(stderr,
	    "split sequence in intermediate part of %s\n",
	    SCANNSTR);
#	endif
	shift = FixSplitSeq(iop, &nread);
	TSTEXC;
	p -= shift;
	*stp -= shift;
	while ((*p & 0x70) == 0x20) {
	    p++;
	}
    }

#   ifdef DEBUG
    fprintf(stderr, "intermediate part: <");
    dumpstr(stderr, *stp, p);
    fprintf(stderr, ">\n");
#   endif

    /* final code */
    if (((*p & 0x70) >= FHINYBBLE) && (*p != 0x7F)) {
	p++;
    } else {
	/* if sequence is split across buffers -- should be rare. */
	if (p >= bufend) {
#	    ifdef DEBUG
	    fprintf(stderr,
		"split sequence in final part of %s\n",
		SCANNSTR);
#	    endif
	    shift = FixSplitSeq(iop, &nread);
	    TSTEXC;
	    p -= shift;
	    *stp -= shift;
	    if (((*p & 0x70) >= FHINYBBLE) && (*p != 0x7F)) {
		p++;
	    } else {
		p = *stp;
		shift = 0;
		SETEXC(XNOTANSI);
		goto handle;
	    }
	} else {
	    p = *stp;
	    shift = 0;
	    SETEXC(XNOTANSI);
	    goto handle;
	}
    }

#   ifdef DEBUG
    fprintf(stderr, "final part: <");
    dumpstr(stderr, *stp, p);
    fprintf(stderr, ">\n");
#   endif

    handle:
    switch (exception) {
	case XNORMAL:
#	    ifdef DEBUG
	    fprintf(stderr, "%s: normal return\n", SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	case XOKEOF: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: normal EOF return\n", SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	case XBADEOF: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: EOF in mid-sequence\n", SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	case XNOTANSI: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: sequence doesn't follow ANSI pattern\n",
		SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	case XTOOLONG: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: sequence too long\n", SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	case XBADNUM: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: garbage where number expected\n", SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	case XBADSKIP: 
#	    ifdef DEBUG
	    fprintf(stderr,
		"%s: garbage where explicit string expected\n", SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	case XRDERR: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: read error\n", SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	case XTRASTUFF: 
#	    ifdef DEBUG
	    fprintf(stderr,
		"%s: extra chars among sequence parameters\n", SCANNSTR);
#	    endif
	    return(p - *stp);
	    break;
	default: 
	    fprintf(stderr,
		"%s: panic: exception of unknown type\n", SCANNSTR);
	    abort();
	    break;
    }
}


#undef SCANNAME
#undef SCANNSTR
#undef FHINYBBLE
