/*
    This memory allocator is designed to be optimum for programs that
    primarily build data structures and then use them without a lot
    of freeing and subsequent re-use of objects of widely differing
    sizes.  Seperate heaps are supported which may be created and destroyed
    wholesale, allowing entire structures to be freed at once, recovering
    all such freed space.

    The allocator allocates NO per-object overhead -- it is maximally
    space efficient unless fragmented to death.

    The usual interface is NOT used, as the free operation on single
    elements requires that the user supply the size of the object being
    freed.

    Note -- declare your main arena as struct main_arena *, initialized to 0.
    Similarly, declare arenas as struct arena *, initialized to 0.
    Do not allocate space from a non-established arena and do not
    try to establish an already-established arena.  mempool and 
    main_arena_pool both restore the 0 value to the pointer to indicate
    that it is not established.
*/
#ifndef MEM_INCLUDED
#define MEM_INCLUDED
#ifdef c_plusplus
class main_arena;
class arena;

/* Establish a main arena */
void mem_main_arena(main_arena **);

/* Release all space ever allocated by a main arena to the system making
   it available to other main arenas and to malloc.  De-establish the
   main arena. */
void mem_pool_main_arena(main_arena **);

/* Establish an arena under a specified main arena */
void memarena(main_arena *, arena **);

/* Release all space allocated by an arena to its main arena and de-establish
   the arena. */
void mempool(arena **);

/* "malloc" from a specified arena */
void *mem(arena *, unsigned size);

/* "calloc" from a spceified arena */
void *cmem(arena *, unsigned count, unsigned size);

/* "realloc" from a specified arena -- Note that the old size must be
   specified and that the arena had beter be the one from which the
   original space was allocated. */
void *rmem(arena *, void *, unsigned oldsize, unsigned size);

/* "free" to a specified arena -- Note that the size must be specified and
   that the arena had better be the one from which the space was allocated.
   NOTE: if an object was allocated by cmem, with count > 1, cmemfree
   must be used instead of this to free it. */
void memfree(arena *, void *, unsigned size);

/* "free" to a specified arena something that was allocated by cmem. */
void cmemfree(arena *, void *, unsigned count, unsigned size);

/* Report size of arena and number of free bytes */
void mem_report_usage(arena *, unsigned *size, unsigned *free);

/* Report size of main arena and number of free bytes */
void mem_report_main_usage(main_arena *, unsigned *size, unsigned *free);
#else

#ifdef lint
#include "internals.h"
#endif lint

extern void mem_main_arena();
extern void mem_pool_main_arena();
extern void memarena();
extern void mempool();
extern char *mem();
extern char *cmem();
extern char *rmem();
extern void memfree();
extern void mem_report_usage();
extern void mem_report_main_usage();
#endif c_plusplus
#endif MEM_INCLUDED
