#define ERULE_C 1
#include "error.h"
#include "table.h"
#include "dwg_table.h"
#include "erule.h"
#include "extension.h"
#include "add/v_add.h"


erule_table* erule_the_rule_table = 0;
erule_drawing_table erule_the_drawing_table;
heap erule_heap(0);


erule::erule(const string& compile_type)
{
    erule_heap.establish();
    AS(!erule_the_rule_table);
    new erule_table(compile_type);
    AS(erule_the_rule_table);  // Set by constructor
}


void erule::dump(ostream& o) 
{ 
    if (!erule_the_rule_table) return;
    erule_the_rule_table->dump(o);
    erule_the_drawing_table.dump(o);
}



boolean erule::read(char* filename)
{
    AS(erule_the_rule_table);
    return erule_the_rule_table->read(filename);
}


static void
resolve_presence_of_both(erule_rule* rule,
			 erule_drawing_extension** ext,
                         erule_drawing_special** spec)
{
   if (*spec && *ext) {
	int spec_ord = rule->ord(*spec);
	int ext_ord = rule->ord(*ext);
	if (ext_ord < spec_ord) *spec = 0;
	else if (!((*ext)->is_prim())) *ext = 0;
    }
}


erule_selection::erule_selection(const string& drawing, 
				 const string& context_name,
				 boolean non_existence_not_error)
    : _context(context_name)
{
    if (!erule_the_rule_table) new erule("LOGIC");
    boolean alloc = this ? FALSE : TRUE;
    this = this ? this : (erule_selection*)erule_heap.mem(sizeof(*this));
    allocated = alloc;

    _special = 0;  _extension = 0;

    erule_current_drawing = _drawing = erule_the_drawing_table.enter(drawing);
    if (!_drawing) return;

    erule_rule* rule = erule_the_rule_table->enter()->select();
    if (!rule) return;

    _drawing->push_exception(rule, _context);

    _special = _drawing->select_special(rule);
    _extension  = _drawing->select_extension(rule, _special != 0 ||
					     non_existence_not_error);
    resolve_presence_of_both(rule, &_extension, &_special);
    _extension_forced_to_primitive = rule->forced_to_primitive(_extension);

    _drawing->pop_exception(rule, _context);
}


char* erule_selection::
filename(erule_filetype filetype, unsigned page, char* context)
{
    char descrip[1024];
    erule_scalddir dir;
    char* name;

    static char* ftypes[] = { "", "CN", "PE", "PL", "LL", "", "SCHEMA" };

    if (filetype == ERULE_SPECIAL) {
	if (!_special) {
	    fault("A special file has not been selected");  return 0;
	}
    }
    else {
	if (!_extension) {
	    fault("A drawing has not been selected");  return 0;
	}
    }

    const char* exten = extension();
    switch (filetype) {
    case ERULE_CONNECTIVITY:
        sprintf(descrip, "%s.%d.%d", exten, version(), page);
    	dir = _extension->scalddir();
	name = s_file_name(dir.name(), dir.which(), ftypes[filetype],
	                   _drawing->name(), descrip, S_NOCREATE);
	break;
    case ERULE_EXPANSION:
    case ERULE_LIST:
        sprintf(descrip, "%s.%d.%d.%s", exten, version(), page,
	                 context ? context : "");
    	dir = _extension->scalddir();
	name = s_file_name(dir.name(), dir.which(), ftypes[filetype],
	                   _drawing->name(), descrip, S_CREATE_FILE);
	break;
    case ERULE_LINKER_LIST:
        sprintf(descrip, "%s.%d.%s", exten, version(), 
	                 context ? context : "");
    	dir = _extension->scalddir();
	name = s_file_name(dir.name(), dir.which(), ftypes[filetype],
	                   _drawing->name(), descrip, S_CREATE_FILE);
	break;
    case ERULE_SCHEMA:
    	dir = _extension->scalddir();
	name = s_file_name(dir.name(), dir.which(), ftypes[filetype],
	                   _drawing->name(), 0, S_CREATE_FILE);
        break;
    case ERULE_SPECIAL:
    	dir = _special->scalddir();
	name = s_file_name(dir.name(), dir.which(), _special->name(),
	                   _drawing->name(), 0, S_CREATE_FILE);
        break;
    case ERULE_UNDEF_FILETYPE:
    default:
	fault("Unknown erule_filetype");
	return 0;
    }

    if (!name) erule_err.sdl();
    return name;
}


boolean erule_selection::is_primitive()
{
    if (_special) return TRUE;
    if (!_extension) return FALSE;
    return _extension->is_prim() ? TRUE : _extension_forced_to_primitive;
}


int erule_selection::is_simple_primitive()
{
    if (!_extension || !(_extension->is_prim())) return 0;
    if (!(_drawing->is_single_versioned())) return 0;
    erule_drawing_page* page = _extension->first_page();
    if (!page) return 0;
    int pgnum = page->page_number();
    if (_extension->next_page(page)) return 0;

    // So far, so good -- we have PRIM.1.1 with no other versions or pages.
    // Now, we need to check whether the parameterization affects the 
    // selection.

    erule_entry* entry = erule_the_rule_table->enter();  AS(entry);
    erule_rule* rule = entry->rules.first();  AS(rule);
    if (!entry->rules.next(rule)) return pgnum;		// only 1 rule -- ok

    erule_err.disable();
    for (; rule && pgnum; rule = entry->rules.next(rule)) {
	_drawing->push_exception(rule, _context);
	erule_drawing_extension* ext = 
	    _drawing->select_from_single_version(rule);
	erule_drawing_special* spec = _drawing->select_special(rule);
	resolve_presence_of_both(rule, &ext, &spec);
#ifdef DEBUG
	cerr << "ext = " << hex((int)ext) << " _extension = " << 
	    hex((int)_extension) NL;
#endif
        if (ext == _extension) {
	   if (spec != _special || 
		rule->forced_to_primitive(ext) != _extension_forced_to_primitive) {
#ifdef DEBUG
	       cerr << "spec = " << hex((int)spec) << " _special = " << 
		   hex((int)_special) NL;
	       cerr << "rule->forced_to_primitive(ext) = " <<
		   rule->forced_to_primitive(ext) <<
		       " _extension_forced_to_primitive = " <<
			   _extension_forced_to_primitive NL;
#endif
	       pgnum = 0;
	   }

	}
	else pgnum = 0;
	_drawing->pop_exception(rule, _context);
    }
    erule_err.enable();
    return pgnum;
}


boolean erule_selection::same_directory(erule_selection* other)
{
    if (!other) return !this;
    if (!this) return FALSE;
    if (_drawing != other->_drawing) return FALSE;

    erule_scalddir dir1;
    if (_extension) dir1 = _extension->scalddir();
    else if (_special) dir1 = _special->scalddir();
    else return (!(other->_extension) && !(other->_special));

    erule_scalddir dir2;
    if (other->_extension) dir2 = other->_extension->scalddir();
    else if (other->_special) dir2 = other->_special->scalddir();
    else return FALSE;

    return dir1 == dir2;
}


string erule_selection::directory_type()
{
    if (!this) return 0;

    if (_extension) return _extension->scalddir().type();
    else if (_special) return _special->scalddir().type();
    else return 0;
}


boolean erule::except(const string& compile_type, 
		      const string& drawing, const string& context,
                      const string& extension, const string& attribute)
{
    if (!erule_the_rule_table) fault("Out of sequence");

    // Ignore exceptions for other compile types
    if (!erule_the_rule_table->enter(compile_type)) return TRUE;

    erule_drawing_entry* dwg = 
	erule_the_drawing_table.enter_without_directory_read(drawing);
    if (!dwg) return FALSE;

    dwg->enter_exception(context, new erule_extension(extension, attribute));
    return TRUE;
}


