(**)     { ------- Time utility ------- }


(***************************************************************************)
(*                                                                         *)
(*  TIME UTILITIES                                                         *)
(*                                                                         *)
(*                                                                         *)
(*  These utilities are designed to work on the VAX, 370, and UNIX.  They  *)
(*  do not work for the ELXSI and this file should no be included when     *)
(*  makeing a version for that machine.                                    *)
(*                                                                         *)
(***************************************************************************)



{  There are two MACHINE DEPENDENT functions here:                          }
{                                                                           }
{     elapsed_time -- returns an integer which is                           }
{                     the time of day clock in milliseconds                 }
{                                                                           }
{     CPU_time     -- returns an integer which is                           }
{                     the process clock in milliseconds                     }


function elapsed_time: longint;
  { return the current elapsed time in milliseconds }
  const
    elapsed_clock_units = 1000; { system intrinsic units to millisecs }

#if VAX
    offset_to_hours     = 1;   { 1st char of hrs. in time string }
    offset_to_minutes   = 4;   { 1st char of min. in time string }
    offset_to_seconds   = 7;   { 1st char of sec. in time string }
    offset_to_hundredths= 10;  { 1st char of hun. in time string }
#endif
#if UNIX
    offset_to_hours     = 12;    { 1st char of hrs. in time string }
    offset_to_minutes   = 15;    { 1st char of min. in time string }
    offset_to_seconds   = 18;    { 1st char of sec. in time string }
#endif

  var
    current_time: time_string;   { current time in ASCII }
    hours,                       { current number of hours }
    minutes,                     { current number of minutes }
    seconds,                     { current number of seconds }
    hundredths: longint;         { current number of hundredths of seconds }


  function convert (var s: time_string; pos: longint): longint;
    { convert the next two digits of the given time string (from POS) into
      an integer. }
  begin
    convert := 10*(ord(s[pos]) - ord('0')) + (ord(s[pos+1]) - ord('0'));
  end { convert } ;


 begin { elapsed_time }
#if VAX
  TIME(current_time);
#endif
#if UNIX
  seconds := epochsec;   convert_time(seconds, current_time);
#endif
  hours := convert(current_time, offset_to_hours);
  minutes := convert(current_time, offset_to_minutes);
  seconds := convert(current_time, offset_to_seconds);
#if VAX
  hundredths := convert(current_time, offset_to_hundredths);
#endif
#if UNIX
  hundredths := 0;
#endif
  elapsed_time := hours      * (3600 * elapsed_clock_units) +
                  minutes    * (60   * elapsed_clock_units) +
                  seconds    * (1    * elapsed_clock_units) +
                  hundredths * (elapsed_clock_units DIV 100);
end { elapsed_time } ;


function CPU_time: longint;
  { return the current CPU time in milliseconds }
  const
#if VAX
    CPU_clock_units = 1;    { system intrinsic units in milliseconds }
#endif
#if UNIX
    CPU_clock_units = 16.66; { system intinsic units -- 1/60th seconds }
#endif
begin
#if VAX
  CPU_time := CLOCK div CPU_clock_units;
#endif
#if UNIX
  CPU_time := trunc(vclock * CPU_clock_units);
#endif
end { CPU_time } ;


procedure init_time(*var current_elapsed_time,
                         current_CPU_time: longint*);
  { initialize the starting times }
begin
  current_elapsed_time := elapsed_time;
  current_CPU_time := CPU_time;
end { init_time_and_date } ;


procedure init_time_and_date(*var current_elapsed_time,
                                  current_CPU_time: longint;
                              var current_date: time_string*);
  { initialize the starting times and date for this compilation }
  var
    seconds: longint;               { current time in seconds }
begin
  init_time(current_elapsed_time, current_CPU_time);
#if VAX
  DATE(current_date);
#endif
#if UNIX
  seconds := epochsec;  convert_time(seconds, current_date);
#endif
end { init_time_and_date } ;


procedure print_time(var f: textfile; current_time: longint);
  { print the time to the given file.  Output leading zeroes. }
  var
    hours, minutes, seconds,
    hundredths: longint;          { current time specifications }
begin
  { convert an integer (assumed to represent a time in units of
    CLOCK_UNITS) to hours, minutes, etc.  and return as globals. }

  hours := current_time DIV (3600*CLOCK_UNITS);
  current_time := current_time - (hours*3600*CLOCK_UNITS);
  minutes:= current_time DIV (60*CLOCK_UNITS);
  current_time := current_time - (minutes*60*CLOCK_UNITS);
  seconds:= current_time DIV CLOCK_UNITS;
  current_time := current_time - (seconds*CLOCK_UNITS);
  hundredths := current_time DIV (CLOCK_UNITS DIV 100);

  if hours < 10 then write(f, '0', hours:1)
                else write(f, hours:2);
  write(f, ':');
  if minutes < 10 then write(f, '0', minutes:1)
                  else write(f, minutes:2);
  write(f, ':');
  if seconds < 10 then write(f, '0', seconds:1)
                  else write(f, seconds:2);
  write(f, '.');
  if hundredths < 10 then write(f, '0', hundredths:1)
                     else write(f, hundredths:2);
end { print_time } ;


procedure exec_time(*var last_elapsed_time: integer;
                     var last_CPU_time: integer; just_delta: boolean*);
  { display the execution time, both CPU time and elapsed time.  If
    JUST_DELTA, then display only the delta time from last_CPU_time to the
    current CPU time and reset last_elapsed_time and last_CPU_time. }
  var
    current_elapsed_time,               { current elapsed time }
    current_CPU_time:  longint;         { calculated CPU time }


  procedure display_time_summary(var f: textfile);
    { write time summaries to the given file }
  begin
    writeln(f);
    write(f, ' Start time   = ');
    print_time(f, last_elapsed_time);  writeln(f);
    write(f, ' Ending time  = ');
    print_time(f, current_elapsed_time);  writeln(f);
    write(f, ' Elapsed time = ');
    if last_elapsed_time < current_elapsed_time then
      print_time(f, current_elapsed_time - last_elapsed_time)
    else
      print_time(f, (24*60*60*1000+current_elapsed_time) - last_elapsed_time);
    writeln(f);
    write(f, ' CPU time     = ');
    print_time(f, current_CPU_time - last_CPU_time);  writeln(f);
  end { display_time_summary } ;


begin { exec_time }
  current_elapsed_time := elapsed_time;    { get the current elapsed time }
  current_CPU_time     := CPU_time;        { get the current CPU time }
  
  if just_delta then
    begin
      write(monitor, '(');
      print_time(monitor, current_CPU_time - last_CPU_time);
      writeln(monitor, ')');
      write(CmpLog, '(');
      print_time(CmpLog, current_CPU_time - last_CPU_time);
      writeln(CmpLog, ')');
    end
  else
    begin
      display_time_summary(monitor);  writeln(monitor);
      display_time_summary(CmpLog);
      if PrintCmpLst then display_time_summary(CmpLst); { only one goes here }
    end;
  last_elapsed_time := current_elapsed_time;
  last_CPU_time := current_CPU_time;
end { exec_time } ;


(**)     { ------- compile time reporting ------- }


procedure post_compile_time(*var f: textfile; file_kind: file_kind_type*);
  { output the compile time to the specified output file }
  var
    i: 1..TIME_BUFFER_LENGTH;         { index into the compile date }
begin
  if file_kind = data_file then write(f, 'TIME=''');
  write(f, ' Compilation on ');
#if VAX
  write(f, compile_date);
  write(f, ' at ');
  print_time(f, start_elapsed_time);
#endif
#if UNIX
  for i := 1 to TIME_BUFFER_LENGTH do
    if islegal[compile_date[i]] then write(f, compile_date[i]);
#endif

  if file_kind = data_file then write(f, ''';');
  writeln(f);
end { post_compile_time } ;


function get_compile_time: xtring;
  { return the compile time as a string }
  var
    i: 1..TIME_BUFFER_LENGTH;         { index into the compile date }
    temp: xtring;                     { buffer for string building }
begin
  create_a_string(temp, MAX_STRING_LENGTH);  temp^[0] := chr(0);
  
  if add_alpha_to_string(temp, ' Compilation on ') then ;
  if add_char_to_string(temp, ' ') then ;

  for i := 1 to TIME_BUFFER_LENGTH do
    if islegal[compile_date[i]] then
      if add_char_to_string(temp, compile_date[i]) then ;
  get_compile_time := enter_string(temp);
  temp^[0] := chr(MAX_STRING_LENGTH);
  release_string(temp);
end { get_compile_time } ;
