procedure push(def: signal_definition_ptr);
  { push the signal definition (DEF) onto the current stack (ENTRY) }
begin
  if def^.stack <> NIL then assert(194 { non-NIL stack });

  if def^.polarity = COMPLEMENTED then
    begin
      def^.stack := def^.signal^.low_asserted;
      def^.signal^.low_asserted := def;
    end
  else
    begin
      def^.stack := def^.signal^.high_asserted;
      def^.signal^.high_asserted := def;
    end;
end { push } ;
  

procedure push_globals_and_xfaces_onto_signal_stacks;
  { push the global and interface signals of mtree_root onto the signal
    stacks }
  var
    current_signal: signal_definition_ptr;   { current signal to be added }
begin
  current_signal := mtree_root^.signals;
  while current_signal <> NIL do
    begin
      if (current_signal^.scope in [GLOBAL,XINTERFACE]) then
        push(current_signal);

      current_signal := current_signal^.next;
    end;
end { push_globals_and_xfaces_onto_signal_stacks } ;


procedure pop_signal_stacks(list: signal_definition_ptr);
  { pop signals from the stacks }
  var
    current_signal: signal_definition_ptr;   { current signal to be removed }
begin
  current_signal := list;
  while current_signal <> NIL do
    begin
      if current_signal^.polarity = COMPLEMENTED then
        begin
          current_signal^.signal^.low_asserted := current_signal^.stack;
          current_signal^.stack := NIL;
        end
      else
        begin
          current_signal^.signal^.high_asserted := current_signal^.stack;
          current_signal^.stack := NIL;
        end;

      current_signal := current_signal^.next;
    end;
end { pop_signal_stacks } ;


function found_the_signal(signal: signal_descriptor_ptr;
                          var sig_def: signal_definition_ptr): boolean;
  { search the signal table for the specified signal and if not found,
    create a signal entry.  If the signal entry has no signal definition,
    create one.  Return TRUE if signal was found in the table.  A signal is
    found by searching for the signal name (SIGNAL^.SIGNAL_NAME) and a
    matching net identification (SIGNAL^.NET_ID).  The null string is used to
    represent a net id for which there can be no implicitly connected nets
    (such as UNNAMED or NC).
    NOTE: If the signal is an interface signal that is not in the root level
          drawing, the NET_ID field MUST be set to NULLSTRING. }
  var
    i: string_range;             { index into signal name for hash calc }
    sum: natural_number;         { hash sum }
    index: signal_table_range;   { hash table index }
    found_the_def,               { TRUE if def was not replaced }
    found_entry: boolean;        { TRUE if signal name found in table }
    entry: signal_entry_ptr;     { current signal table entry }

   
  procedure create_signal_entry(var entry: signal_entry_ptr);
    { create a new entry in the signal name table as described by the
      global SIGNAL and return it }
    var
      new_entry: signal_entry_ptr;   { entry being created }
  begin
    new_signal_entry(new_entry);
    new_entry^.name := signal^.signal_name;
    
    { insert after current entry or at the head of the list }

    if entry <> NIL then
      begin  new_entry^.next := entry^.next;  entry^.next := new_entry;  end
    else
      begin
        new_entry^.next := signal_table[index];
        signal_table[index] := new_entry
      end;

    entry := new_entry;
  end { create_signal_entry } ;


  function build_new_signal_def(signal: signal_descriptor_ptr;
                                entry: signal_entry_ptr):
                                                        signal_definition_ptr;
    { create a new signal definition as described by SIGNAL and ENTRY }
    var
      def: signal_definition_ptr;    { new signal def to be created }
  begin
    if debug_12 then writeln(outfile, '-create new net signal def');

    new_signal_definition(def);
    def^.signal := entry;

    def^.net_id := signal^.net_id;

    def^.is_const := signal^.is_const;
    def^.kind := signal^.kind;
    def^.polarity := signal^.polarity;

    { make sure that the def refers to a legal bit of the signal }

    if def^.kind = VECTOR then
      begin
        def^.left_index := signal^.bit_subscript^.left_index;
        def^.right_index := def^.left_index;
      end;

    total_number_signals := total_number_signals + 1;

    if def^.is_const then
      def^.scope := GLOBAL
    else
      begin
        def^.scope := signal^.scope;
        case def^.scope of
          LOCAL:      number_local_signals := number_local_signals + 1;
          DECLARED,
          GLOBAL:     number_global_signals := number_global_signals + 1;
          XINTERFACE: number_interface_signals := number_interface_signals+1;
        end;

        if is_unnamed_signal(def^.signal^.name) then
          number_unnamed_signals := number_unnamed_signals + 1;
      end;

    build_new_signal_def := def;
  end { build_new_signal_def } ;


  function get_definition(current_def: signal_definition_ptr;
                          signal: signal_descriptor_ptr;
                          var used_old_def: boolean): signal_definition_ptr;
    { make sure that the given signal definition has the correct NET_ID.
      If it does, all is ok, return that def.  If it does not, search for
      the correct def on the virtual defs of CURRENT_DEF.  If it cannot be
      found, create a new def and add to the virtual def list.  If the old
      def was used (a new one was not created), return USED_OLD_DEF TRUE. }
    var
      net_id: xtring;               { net id for this def }
      def: signal_definition_ptr;   { def to be returned }
      found: boolean;               { TRUE if desired def is found }
  begin
    { if the signal has different KIND than the def, there probably is an
      error.  If so, ENTER_SIG will catch it.  Return the same def. }

    if (signal^.kind <> current_def^.kind) and
       (signal^.kind <> UNDEFINED) and (current_def^.kind <> UNDEFINED) and
       (signal^.scope <> XINTERFACE) then
      begin
        def := current_def;
        found := FALSE;
      end
    else
      begin
        net_id := signal^.net_id;  found := TRUE;

        if current_def^.net_id = net_id then def := current_def
        else
          begin
            if debug_12 then writeln(outfile, 'Get definition search');

            { if this def is not the non-virtual def, find the correct def on
              the mtree node of the current def }

            current_def := virtual_base(current_def);

            { if the current def does not have the correct NET_ID, search for
              the correct one in the list of virtual defs }

            def := current_def^.next_virtual_def;  found := FALSE;
            while (def <> NIL) and not found do
              if def^.net_id = net_id then found := TRUE
              else def := def^.next_virtual_def;

            if not found then
              begin
                def := build_new_signal_def(signal, current_def^.signal);
                def^.node := current_def^.node;

                def^.next_virtual_def := current_def^.next_virtual_def;
                current_def^.next_virtual_def := def;

                def^.is_virtual_base := FALSE;
              end;
          end;
      end;

    if found then used_old_def := TRUE;

    get_definition := def;
  end { get_definition } ;


  function enter_def_into_stack(entry: signal_entry_ptr;
                                var sig_def: signal_definition_ptr): boolean;
    { find the signal (SIGNAL) within the signal stack or create it.  The
      scopes of signals in the stack can only be: DECLARED, LOCAL, INTERFACE,
      GLOBAL, or SIG_CONST.  Return TRUE if the signal def already existed.
      Return SIG_DEF as the signal definition in the stack. }
    var
      top,                          { top def in the sigal stack }
      def: signal_definition_ptr;   { current signal in the stack }
      node: mtree_node_ptr;         { node for the signal definition }
      done: boolean;                { TRUE if done searching for node }
      found_def: boolean;           { TRUE if signal already in the stack }
      stack_is_empty: boolean;      { TRUE if stack currently empty }


    function create_signal_definition(node: mtree_node_ptr):
                                                        signal_definition_ptr;
      { create a new definition and add to the appropriate node on the mtree }   
      var
        sig_def: signal_definition_ptr;       { definition being created }
    begin
      if debug_12 then writeln(outfile, '-create new signal def');

      sig_def := build_new_signal_def(signal, entry);
      sig_def^.is_virtual_base := TRUE;

      if sig_def^.is_const then
        begin
          { put it at the right place in the expansion tree }
  
          sig_def^.node := mtree_root;
          sig_def^.next := mtree_root^.signals;
          mtree_root^.signals := sig_def;
        end
      else
        begin
          { put it at the right place in the expansion tree }
  
          sig_def^.node := node;
          sig_def^.next := node^.signals;
          node^.signals := sig_def;
        end;
  
      create_signal_definition := sig_def;
    end { create_signal_definition } ;


    procedure insert_after(stack_entry, sig_def: signal_definition_ptr); 
      { insert the signal definition into the stack following the given def }
    begin
      if stack_entry = NIL then push(sig_def)
      else
        begin
          sig_def^.stack := stack_entry^.stack;
          stack_entry^.stack := sig_def;
        end; 
    end { insert_after } ;


    procedure generate_error(error_num: error_range);
      { produce the specified error, dump some information, and recover }
    begin
      error(error_num { generate the error });
      error_dump_current_parse_environment;

      error_dump_indent(INDENT);
      error_dump_alpha('Existing Signal:');
      error_dump_CRLF;

      error_dump_signal_def(def);

      error_dump_indent(INDENT);
      error_dump_alpha('New signal:     ');
      error_dump_CRLF;
      error_dump_signal_descriptor(signal);

      sig_def := def;
    end { generate_error } ;


  begin { enter_def_into_stack }
    if debug_12 then
      begin
        writeln(outfile, 'Enter enter_def_into_stack');
        dump_stack(outfile, entry);
      end;

    { determine whether the stack is empty }

    if signal^.polarity = COMPLEMENTED then
      stack_is_empty := (entry^.low_asserted = NIL)
    else if signal^.polarity IN [NORMAL, NO_POLARITY] then
      stack_is_empty := (entry^.high_asserted = NIL)
    else
      begin
        { signal has unknown polarity }

        stack_is_empty := TRUE;
        if entry^.high_asserted <> NIL then
          if entry^.high_asserted^.node^.level >=
                                                current_mtree_node^.level then
            stack_is_empty := FALSE;

        if (entry^.low_asserted <> NIL) and stack_is_empty then
          if entry^.low_asserted^.node^.level >= current_mtree_node^.level then
            stack_is_empty := FALSE;
      end;

    { set default scopes if necessary }

    if signal^.scope = UNKNOWN_SCOPE then
      if scope_is_local then signal^.scope := LOCAL
                        else signal^.scope := GLOBAL;

    found_def := FALSE;
    sig_def := NIL;

    if stack_is_empty then
      begin
        if debug_12 then
          writeln(outfile, 'Stack is empty: ', ord(stack_is_empty):1);

        { either the signal is defined in current drawing or is global }

        case signal^.scope of
          XINTERFACE: if current_mtree_node^.father_node = NIL then
	                node := mtree_root
		      else node := current_mtree_node;
          LOCAL,
          DECLARED:   node := current_mtree_node;
          SIG_CONST,
          GLOBAL:     node := mtree_root;
        end;

        sig_def := create_signal_definition(node);
        push(sig_def);
      end
    else
      begin
        if debug_12 then writeln(outfile, 'Stack is not empty');

        { determine the assertion of a signal that has unknown polarity.  It
          is known that it is impossible for an unknown assertion signal to
          appear in the design in both a high-asserted and low-asserted
          version.  That is ALL unnamed signals are unique.  It is also
          known that the signals assertion must match its polarity (for 
          signals with initially unknown polarity). }

        if signal^.polarity = UNKNOWN_POLARITY then
          if entry^.low_asserted <> NIL then
            if entry^.low_asserted^.node = current_mtree_node then
              begin
                signal^.polarity := entry^.low_asserted^.polarity;
                signal^.low_asserted := (signal^.polarity = COMPLEMENTED);
              end;

        if debug_12 then writeln(outfile, 'pol=', ord(signal^.polarity)); 

        if signal^.polarity = UNKNOWN_POLARITY then
          if entry^.high_asserted <> NIL then
            if entry^.high_asserted^.node = current_mtree_node then
              begin
                signal^.polarity := entry^.high_asserted^.polarity;
                signal^.low_asserted := (signal^.polarity = COMPLEMENTED);
              end;

        if debug_12 then writeln(outfile, 'pol=', ord(signal^.polarity)); 

        { find the proper stack link; determined by the polarity }

        if signal^.polarity = COMPLEMENTED then def := entry^.low_asserted
                                           else def := entry^.high_asserted;
        top := NIL;

        if debug_12 then writeln(outfile, 'scope=', ord(signal^.scope)); 

        case signal^.scope of
          UNKNOWN_SCOPE:
              begin
                assert(188 { this is an error });
                sig_def := def;
              end;

          LOCAL:
              { A signal with local scope is only known within the current
                drawing.  A global with the same name can be referenced by
                a son drawing without conflicting with the local.  A local
                signal may either appear on the top of the stack (local to
                the current drawing) or may appear one level down.  The
                second case is due to a pin name on the body being parsed
                having the same name as the local being looked for. }
              
              begin
                if current_mtree_node^.level < def^.node^.level then
                  repeat
                    top := def;
                    def := def^.stack;

                    done := (def = NIL);
                    if not done then
                      done := (def^.stack = NIL);
                    if not done then
                      if (current_mtree_node^.level >= def^.node^.level) then
                        done := TRUE;
                  until done;

                if (def = NIL) then
                  begin
                    sig_def := create_signal_definition(current_mtree_node);
  
                    { put at the bottom of the stack }

                    insert_after(top, sig_def);
                  end
                else if current_mtree_node = def^.node then
                  case def^.scope of
                    LOCAL:      sig_def := get_definition(def, signal,
                                                         found_def);

                    XINTERFACE: generate_error(162 { conflict! });

                    DECLARED,
                    GLOBAL:     generate_error(163 { conflict! });
                  end
                else if current_mtree_node^.level > def^.node^.level then
                  begin
                    sig_def := create_signal_definition(current_mtree_node);
  
                    { if TOP is NIL, insert_after performs a PUSH }

                    insert_after(top, sig_def);
                  end
                else
                  begin
                    assert(189 { signals are out of order });
                    dump_signal_descriptor(CmpLog, signal);
                    writeln(CmpLog, 'Dump of the stack: current level = ',
                                    current_mtree_node^.level:1);
                    dump_def_stack(CmpLog, def);
                    sig_def := def;
                  end;
              end;

          XINTERFACE:
              { A signal with interface scope is only known within the current
                drawing.  A global with the same name can be referenced by a
                son drawing without conflicting with the interface. An
                interface signal may either appear on the top of the stack
                (local to the current drawing) or may appear one level down.
                The second case is due to a pin name on the body being parsed
                having the same name as the interface signal being searched
                for. }
              
              begin
                if current_mtree_node^.father_node = NIL then
                  node := mtree_root
                else node := current_mtree_node;

                if node^.level < def^.node^.level then
                  repeat
                    top := def;
                    def := def^.stack;

                    done := (def = NIL);
                    if not done then
                      done := (def^.stack = NIL);
                    if not done then
                      if (node^.level >= def^.node^.level) then
                        done := TRUE;
                  until done;

                if (def = NIL) then
                  begin
                    sig_def := create_signal_definition(node);
  
                    { put at the bottom of the stack }

                    insert_after(top, sig_def);
                  end

                else if node = def^.node then
                  case def^.scope of
                    LOCAL:      generate_error(162 { conflict! });

                    XINTERFACE: sig_def := get_definition(def, signal,
                                                          found_def);

                    DECLARED,
                    GLOBAL:     generate_error(164 { conflict! });
                  end
                else if node^.level > def^.node^.level then
                  begin
                    sig_def := create_signal_definition(node);

                    { if TOP is NIL, insert_after performs a PUSH }

                    insert_after(top, sig_def);
                  end
                else
                  begin
                    assert(189 { signals are out of order });
                    dump_signal_descriptor(CmpLog, signal);
                    writeln(CmpLog, 'Dump of the stack: current level = ',
                                    node^.level:1);
                    dump_def_stack(CmpLog, def);
                    sig_def := def;
                  end;
              end;

          GLOBAL:
              begin
                { A signal with global scope is known throughout the design.
                  It is "defined" in the root node. }

                while (def^.stack <> NIL) do def := def^.stack;

                if def^.node = mtree_root then
                  case def^.scope of
                    LOCAL:      generate_error(163 { conflict! });

                    XINTERFACE: generate_error(164 { conflict! });

                    DECLARED,
                    GLOBAL:     sig_def := get_definition(def, signal,
                                                          found_def);
                  end
                else
                  begin
                    sig_def := create_signal_definition(mtree_root);
                    insert_after(def, sig_def);
                  end;
              end;

          DECLARED:
              begin
                { These are not supported and should have been filtered
		  by now. }

                assert(251 { DECLARED sig encountered });
                sig_def := def;
	      end;

          SIG_CONST:
              begin
                { signal is a signal constant. There can only be one }

                sig_def := def;
                found_def := TRUE;
              end;
        end { case } ;
      end { else } ;

    enter_def_into_stack := found_def;

    if debug_12 then
      begin
        writeln(outfile, 'Exit enter_def_into_stack: sig_def=');
        dump_signal_definition(outfile, sig_def);
        dump_stack(outfile, entry);
      end;
  end { enter_def_into_stack } ;


begin { found_the_signal }
  if debug_12 then
    begin
      writeln(outfile, 'Enter found_the_signal:');
      writeln(outfile, '  scope=', ord(signal^.scope):1, '; polarity=',
                       ord(signal^.polarity)); 
      write(outfile, '  Current_mtree_node = ');
      writestring(outfile, current_mtree_node^.macro_name);
      writeln(outfile);
      write(outfile, '  Signal=');
      dump_signal_descriptor(outfile, signal);
      write(outfile, '  Def=');
      dump_signal_definition(outfile, sig_def);
    end;

  if sig_def <> NIL then
    begin
      found_the_def := FALSE;
      sig_def := get_definition(sig_def, signal, found_the_def);
      found_the_signal := found_the_def;
    end
  else
    begin
      { calculate a hash index into the signal table }

      sum := 0;
      for i := 1 to ord(signal^.signal_name^[0]) do
        sum := sum + ord(signal^.signal_name^[i]);
      index := sum MOD (SIGNAL_TABLE_SIZE+1);

      { search for the signal name in the list of signal names }

      entry := signal_table[index];  found_entry := FALSE;
      while (entry <> NIL) and not found_entry do
        if (entry^.name = signal^.signal_name) then
          found_entry := TRUE
        else
          entry := entry^.next;

      if not found_entry then create_signal_entry(entry);

      found_the_signal := enter_def_into_stack(entry, sig_def);
    end;

  if debug_12 then writeln(outfile, 'Exit found_the_signal');
end { found_the_signal } ;


(**)     { ------- creation of a signal instance ------- }


function enter_signal_instance(signal: signal_descriptor_ptr;
                               def: signal_definition_ptr):
                                                          signal_instance_ptr;
  { look for the signal descriptor in the signal table.  If it is not
    there, create a SIGNAL_DEFINITION and SIGNAL_INSTANCE.  If it is there,
    search the SIGNAL_INSTANCEs for the signal descriptor.  If it is not
    there, create a new SIGNAL_INSTANCE.  Return the SIGNAL_INSTANCE.
    NOTE:  this routine expects current_mtree_node to be the node in which
           new signal definitions will be created. }
  var
    sig_def: signal_definition_ptr;  { the instance's definition }
    instance: signal_instance_ptr;   { the instance being created (entered) }
    found: boolean;                  { used to flag instance already exists }


  function create_signal_instance: signal_instance_ptr;
    { create a new instance from the SIGNAL_DESCRIPTOR and add to signal def }
    var
      instance: signal_instance_ptr;  { instance being created }
  begin
    number_signal_instances := number_signal_instances + 1;
    new_signal_instance(instance);
    with instance^ do
      begin
        defined_by := sig_def;
        if sig_def^.scope <> XINTERFACE then
          number_non_interface_signal_instances :=
                                    number_non_interface_signal_instances + 1;

        bit_subscript := copy_bit_subscript(signal^.bit_subscript);

        replication_factor := signal^.replication_factor;

        low_asserted := signal^.low_asserted;

        next := sig_def^.instances;  sig_def^.instances := instance;
      end;

    create_signal_instance := instance;
  end { create_signal_instance } ;


  function descriptor_same_as_instance(signal: signal_descriptor_ptr;
                                       instance: signal_instance_ptr): boolean;
    { compare the signal_descriptor with the signal instance to see if they
      describe the same object. }
  begin
    descriptor_same_as_instance := FALSE;

    if (   (signal^.polarity = UNKNOWN_POLARITY) or
           (signal^.low_asserted = instance^.low_asserted)   ) and
       (signal^.replication_factor = instance^.replication_factor) then
      if signal^.kind = UNDEFINED then
        begin
          if instance^.defined_by^.kind = UNDEFINED then
            descriptor_same_as_instance := TRUE
          else
            { instance is same only if it refers to the entire signal }
            if (instance^.bit_subscript^.left_index = 
                                     instance^.defined_by^.left_index) and
               (instance^.bit_subscript^.right_index =
                                       instance^.defined_by^.right_index) then
              descriptor_same_as_instance := TRUE;
        end
      else
        if signal^.kind = instance^.defined_by^.kind then
          if identical_bit_subscripts(signal^.bit_subscript,
                                      instance^.bit_subscript) then
            descriptor_same_as_instance := TRUE;
  end { descriptor_same_as_instance } ;


  procedure augment_signal_definition(sig_def: signal_definition_ptr;
                                      instance: signal_instance_ptr;
                                      signal: signal_descriptor_ptr);
    { set the bit subscript of the definition to the union of the bit
      subscripts of the instance and the definition. The definition
      is ALWAYS a contiguous subrange. }


    procedure merge_subscript(def: signal_definition_ptr;
                              new_bits: subscript_ptr);
      { merge the bits of the new subscript into the description of the
        entire signal in the signal definition }
      var
        subscript: subscript_ptr;        { current subscript bits }
        left, right: bit_range;          { left and right bits for subscript }
        left_def, right_def: bit_range;  { left and right bits for def }
        unassigned: boolean;             { TRUE if subscript unassigned yet }
    begin
      if def^.kind = UNDEFINED then unassigned := TRUE
                               else unassigned := FALSE;
      left_def := def^.left_index;  right_def := def^.right_index;

      subscript := new_bits;
      while subscript <> NIL do
        begin
          left := subscript^.left_index;  right := subscript^.right_index;

          if unassigned then
            begin
              left_def := left;  right_def := right;  unassigned := FALSE;
            end
          else if left_to_right then
            begin
              if left < left_def then left_def := left
              else if left > right_def then right_def := left;
              if right < left_def then left_def := right
              else if right > right_def then right_def := right;
            end
          else
            begin
              if left > left_def then left_def := left
              else if left < right_def then right_def := left;
              if right > left_def then left_def := right
              else if right < right_def then right_def := right;
            end;

          subscript := subscript^.next;
        end;

      def^.left_index := left_def;  def^.right_index := right_def;
    end { merge_subscript } ;


  begin { augment_signal_definition }
    case signal^.kind of
      UNDEFINED:
          case sig_def^.kind of
            UNDEFINED: { do nothing } ;

            SINGLE:    signal^.kind := SINGLE;

            VECTOR:    begin
                         new_subscript(instance^.bit_subscript);
                         instance^.bit_subscript^.left_index :=
                                                          sig_def^.left_index;
                         instance^.bit_subscript^.right_index :=
                                                         sig_def^.right_index;
                         signal^.kind := VECTOR;
                       end;
          end;

      SINGLE:
          case sig_def^.kind of
            UNDEFINED: sig_def^.kind := SINGLE;

            SINGLE:    { do nothing } ;

            VECTOR:    begin
                         error(130 { scalar reference to vector });
                         error_dump_current_parse_environment;
                         error_dump_signal_instance(instance);
 
                         { no need to release subscript; NIL already }

                         new_subscript(instance^.bit_subscript);
                         instance^.bit_subscript^.left_index :=
                                                          sig_def^.left_index;
                         instance^.bit_subscript^.right_index :=
                                                          sig_def^.left_index;

                         new_subscript(signal^.bit_subscript);
                         signal^.bit_subscript^.left_index :=
                                                          sig_def^.left_index;
                         signal^.bit_subscript^.right_index :=
                                                          sig_def^.left_index;
                         signal^.kind := VECTOR;
                       end;
          end;

      VECTOR:
          case sig_def^.kind of
            UNDEFINED:
                begin
                  merge_subscript(sig_def, instance^.bit_subscript);
                  sig_def^.kind := VECTOR;
                end;

            SINGLE:
                begin
                  error(131 { vector reference to scalar signal });
                  error_dump_current_parse_environment;

                  release_subscript(instance^.bit_subscript);
                  error_dump_signal_instance(instance);

                  release_subscript(signal^.bit_subscript);
                  signal^.kind := SINGLE;
                end;

            VECTOR:
                merge_subscript(sig_def, instance^.bit_subscript);
          end;
    end { case signal^.kind } ;
  end { augment_signal_definition } ;


begin { enter_signal_instance }
  global_found_def_in_enter_signal := FALSE;

  if signal <> NIL then
    begin
      if debug_12 then
        begin
          write(outfile, 'Enter enter_signal_instance: ');
          dump_signal_descriptor(outfile, signal);
        end;

      { intermediate interface signals have NO virtual synonyms.  Root level
        interface signals are treated like local signals.  The NET_ID is
        important. }

      if signal^.scope = XINTERFACE then
        if def = NIL then
          begin
            if current_mtree_node^.father_node <> NIL then 
              signal^.net_id := nullstring;
          end
        else
          if def^.node^.father_node <> NIL then
            signal^.net_id := nullstring;

      sig_def := def;
      if def = NIL then
        found := found_the_signal(signal, sig_def)

      else if (def^.net_id <> signal^.net_id) then
        found := found_the_signal(signal, sig_def)

      else
        found := TRUE;

      { the following is a HACK and a KLUDGE!  See fix_actual_signal }

      global_found_def_in_enter_signal := found;

      if not found then
        begin
          if debug_12 then writeln(outfile, '-a new signal definition');

          instance := create_signal_instance;
          augment_signal_definition(sig_def, instance, signal);
        end
      else
        begin
          instance := sig_def^.instances;  found := FALSE;
          while (instance <> NIL) and not found do
            if descriptor_same_as_instance(signal, instance) then
              found := TRUE
            else
              instance := instance^.next;

          if not found then
            begin
              if debug_12 then writeln(outfile, '-a new signal instance');

              instance := create_signal_instance;
              augment_signal_definition(sig_def, instance, signal);
            end;
        end;

      enter_signal_instance := instance;

      if debug_12 then
        begin
          writeln(outfile, 'Exit enter_signal_instance: signal entered=');
          dump_signal_definition(outfile, sig_def);
        end;
    end;
end { enter_signal_instance } ;


