(**)     { --------------- Property Utilities --------------- }


function copy_property_list(prop: property_ptr): property_ptr;
  { -- make a copy of the given property_list and return it -- }
  var
    PP,                         { current element of prop list }
    last,                       { last element in the list     }
    copied_PP: property_ptr;    { list to be returned          }
begin
  if debug_14 then
    begin
      writeln(outfile, 'Entering copy_property_list: props:');
      dump_property_list(outfile, prop);
    end;

  PP := prop;  last := NIL;  copied_PP := NIL;
  while PP <> NIL do
    begin
      if INHERIT_PIN IN PP^.name^.kind then
        begin
          if last = NIL then
            begin  new_property(last);  copied_PP := last;  end
          else
            begin  new_property(last^.next);  last := last^.next;  end;
          last^.name := PP^.name;    last^.text := PP^.text;
        end;
      PP := PP^.next;
    end;

  copy_property_list := copied_PP;

  if debug_14 then
    begin
      writeln(outfile, 'Exiting copy_property_list: returning prop list:');
      dump_property_list(outfile, copied_PP);
    end;
end { copy_property_list } ;


(**)     { ------- Random Utilities ------- }


function is_NC_instance(SI: signal_instance_ptr): boolean; 
  { -- checks if name of signal is in the "NC" class. The routine
       assumes that all NC names are prefixed with a special char -- }
begin
  is_NC_instance := is_NC_signal(SI^.defined_by^.signal^.name);
end { is_NC_instance } ; 


function no_NC_PCS(PCS: propertied_CS_ptr): boolean;
  { -- TRUE if the no element of the PCS is an NC signal -- }
  var
    found: boolean;    { TRUE if NC is found }
begin
  found := FALSE;
  while (PCS <> NIL) AND NOT found do 
    begin
      found := is_NC_signal(PCS^.instance^.defined_by^.signal^.name);
      PCS := PCS^.next;
    end;

  no_NC_PCS := NOT found;
end { no_NC_PCS } ;


function scope_of_SI(SI: signal_instance_ptr): scope_type;
  { -- Returns the scope of a signal instance -- }
begin
  scope_of_SI := LOCAL;
  if SI <> NIL then
    with SI^.defined_by^ do
      if is_const then scope_of_SI := GLOBAL
                  else scope_of_SI := scope;
end { scope_of_SI } ;


function scope_of_sig_def(sig_def: signal_definition_ptr): scope_type;
  { -- Returns the scope of a signal definition -- }
begin
  scope_of_sig_def := LOCAL;
  if sig_def <> NIL then
    with sig_def^ do
      if is_const then scope_of_sig_def := GLOBAL
                  else scope_of_sig_def := scope;
end { scope_of_sig_def } ;

    
(**)     { ------- Subscript Utilities ------- }


procedure add_subscript_element(var sub: subscript_ptr; b: bit_range);
  { glues a new subscript to the right end of a subscript list }    
begin
  if sub = NIL then
    begin
      new_subscript(sub);
      with sub^ do
        begin  left_index  := b;  right_index := b;  end;
    end
  else
    if sub^.next <> NIL then
      assert(50 {no appending in middle of bit list})
    else
      with sub^ do
        begin
          new_subscript(next);
          next^.left_index  := b;  next^.right_index := b;
        end { with };
end { add_subscript_element } ;


procedure add_subscript(var sub: subscript_ptr; b: bit_range);
{ -- adds a bit to the right end of a subscript list -- }
  var direction: -1..1;          { -1 => sub is right to left     }
                                 { +1 => sub is left to right     }
      last_sub:  subscript_ptr;  { points to last sub in bit list }

begin
  if sub <> NIL then
    begin
      last_sub := sub;  { -- get the last subscript element -- }
      while last_sub^.next <> NIL do last_sub := last_sub^.next;

      with last_sub^ do
        begin
          if (left_index < right_index) then
            direction := 1    { -- subscript increases to the right -- }
          else
            if (left_index > right_index) then
              direction := -1 { -- subscript increases to the left  -- }
            else              { -- sub has one bit, expand in default dir -- }
              if left_to_right then direction := 1
              else
                direction := -1;
        end { with ... };

      if (b = last_sub^.right_index + direction) then
        last_sub^.right_index := last_sub^.right_index + direction
      else
        add_subscript_element(last_sub,b);
    end
  else { sub = NIL }
    add_subscript_element(sub,b);
end { add_subscript } ;

(**)

procedure copy_subscript_element(d_sub_element, s_sub_element: subscript_ptr);
{ -- copies source subscript element into destination subscript element -- }

begin
  if( d_sub_element = NIL) OR (s_sub_element = NIL) then
    assert(51 {oops})
  else
    with d_sub_element^ do
      begin
        next        := s_sub_element^.next;
        left_index  := s_sub_element^.left_index;
        right_index := s_sub_element^.right_index;
      end;
end { copy_subscript_element } ;


function copy_subscript(s_sub: subscript_ptr): subscript_ptr;
  { return copy of subscript }
  var 
    d_sub: subscript_ptr;      { copied subscript }
    new_sub: subscript_ptr;    { last element in copied subscript }
begin
  if s_sub = NIL then d_sub := NIL
  else
    begin
      new_subscript(new_sub);     d_sub := new_sub;   
      copy_subscript_element(new_sub,s_sub);
      s_sub := s_sub^.next;
      while s_sub <> NIL do
	begin
	  new_subscript(new_sub^.next);
	  new_sub := new_sub^.next;
	  copy_subscript_element(new_sub, s_sub);
	  s_sub := s_sub^.next;
	end { while };
    end;
  copy_subscript := d_sub;
end { copy_subscript } ;


(**)     { ------- PCS_Pointer Utilities ------- }


procedure advance_PCS_pointer(var CP: PCS_pointer);
{ -- advances a pointer into a propertied concatenated signal list by 1 signal
   position with understanding of replication factors. For example consider
   a pointer set to the list   X< .. >*3 : Y< .. >*2 : Z< .. >. Advancing it
   once it still points to X, advancing it two more times it points to Y. -- }

begin
  with CP do
    if replication_count <  PCS^.instance^.replication_factor then
       replication_count := replication_count+1
    else
      begin
        PCS := PCS^.next;
        replication_count:=1;
      end;
end { advance_PCS_pointer } ;


procedure init_PCS_pointer(var CP: PCS_pointer;  PCS: propertied_CS_ptr);
  { -- build a new PCS pointer -- }
begin
  CP.PCS := PCS;              { PCS field points to the first SI of the PCS }
  CP.replication_count := 1;  { count field incs over reps of pointed to SI }
end { init_PCS_pointer } ;


(**)     { ------- Width Utilities ------- }


function width_of_subscript_element(sub: subscript_ptr): natural_number;
{ -- returns the  number of bits in a subscript list element -- }
 var width: natural_number;

begin
  if sub = NIL then
    width := 0 {i.e. the signal has no width}
  else
    width_of_subscript_element:= 1 + ABS(sub^.left_index - sub^.right_index);
end { width_of_subscript_element } ;



function width_of_subscript_list(sub: subscript_ptr): natural_number;
{ -- returns the total number of bits in a subscript list -- }
 var width: natural_number;

begin
  width:=0;
  while sub <> NIL do
    with sub^ do
      begin
        width := width + 1 + abs(left_index - right_index);
        sub := next;
      end { with };
  width_of_subscript_list:=width;
end { width_of_subscript_list } ;


function width_of_signal_descriptor(SD: signal_descriptor_ptr): natural_number;
{ -- returns the number of bits in a signal descriptor -- }
  var width: natural_number;

begin
  width := 0;
  if SD = NIL then
    assert(54 {oops!})
  else
    with SD^ do
      case kind of
        SINGLE:    width := 1;
        VECTOR:    width := width_of_subscript_list(bit_subscript);
        UNDEFINED: width := 0;
      end;
  width_of_signal_descriptor := width;
end { width_of_signal_descriptor } ;


function width_of_signal_definition(S: signal_definition_ptr): natural_number;
{ -- returns the number of bits in a signal definition -- }
  var width: natural_number;

begin
  width := 0;
  if S = NIL then
    assert(69 {oops!})
  else
    with S^ do
      case kind of
        SINGLE:    width := 1;
        VECTOR:    width := ABS(S^.left_index - S^.right_index) + 1;
        UNDEFINED: width := 0;
      end;
  width_of_signal_definition := width;
end { width_of_signal_definition };


function size_of_signal_instance(SI: signal_instance_ptr): natural_number;
{ -- returns the number of bits in a signal instance excluding rep factor -- }
  Var width: natural_number;

begin
  width:=0;
  if SI = NIL then
    assert(55 {oops!})
  else 
    with SI^.defined_by^ do
      case kind of
        SINGLE:    width := 1;
        VECTOR:    width := width_of_subscript_list(SI^.bit_subscript);
        UNDEFINED: width := 0;
      end;

  size_of_signal_instance := width;
end { size_of_signal_instance } ;


function width_of_PCS(PCS: propertied_CS_ptr): natural_number;
{ -- returns the number of bits in a propertied CS list -- }
  var w,                        { width of the last instance in the PCS }
      width: natural_number;    { return value }

begin
  width_of_PCS := 0;    width := 0;   w := 17; { - 17 is THE random number - }
  while (PCS <> NIL) AND (w <> 0) do 
    begin
      w := width_of_signal_instance(PCS^.instance);
      width := width + w;
      PCS := PCS^.next;
    end;

  { if all instances are of defined width, then PCS is of defined width }

  if w <> 0 then width_of_PCS := width;
end { width_of_PCS } ;


(**)     { ------- Bit Subscript Location Utilities ------- }


function bit_in_range(b: bit_range;sub: subscript_ptr): boolean;
{ -- checks if bit b lies with the bit range of subscript sub -- }
  var found: boolean;              { => bit was found            }

begin
  found := FALSE;
  if (b = 1) AND (sub = NIL) then
    found := TRUE  { --  sub is scalar -- }
  else
    if sub = NIL then
      begin
        assert(53 {oops!});
        write(CmpLog, 'Looking for: ', b:1, ' in ');
        dump_bit_subscript(CmpLog, sub, VECTOR); writeln(CmpLog);
      end
    else
      with sub^ do
        if left_index < right_index then
          if (b >= left_index) AND (b <= right_index) then
            found := TRUE
          else {nuttin}
        else
          if (b <= left_index) AND (b >= right_index) then
            found := TRUE;
  bit_in_range := found;
end { bit_in_range } ;


function nth_bit_subscript(var n: bit_range; sub: subscript_ptr): boolean;
  { returns TRUE if the "n"th bit exists in a subscript list, if TRUE n is
   set to the "n"th bit, i.e. n enters as an ordinal exits as a cardinal }
 var
   t: natural_number;    { number of bits passed over so far }
begin
  if debug_19 then
    begin
      writeln(outfile, 'Entered nth_bit_subscript: n=', n:1);
      write(outfile, '  sub=');
      dump_bit_subscript(outfile, sub, VECTOR);
    end;

  nth_bit_subscript := FALSE;

  if sub = NIL then
    begin
      if (n = 1) then
        nth_bit_subscript := TRUE { sub is a scalar }
      else
        assert(58 { not allowed });
    end
  else
    if (n < 1) then 
      assert(58 {oops!})
    else
      { check for a non-bit list subscript }

      if sub^.next = NIL then
        begin
          t := 1 + ABS(sub^.left_index - sub^.right_index);
          if n > t then
            begin
              assert(59 { doesn't fit in subscript! });
              write(CmpLog, '  Bit=', t:1, '; Sub=');
              dump_bit_subscript(CmpLog, sub, VECTOR);
              writeln(CmpLog);
            end
          else
            begin
              with sub^ do
                if left_index <= right_index then
                  n := left_index + (n-1)
                else
                  n := left_index - (n-1);
              nth_bit_subscript := TRUE;
            end
        end
      else
        { handle bit lists }

        if (n > width_of_subscript_list(sub)) then
          begin
            assert(59 { doesn't fit in subscript! });
            write(CmpLog, '  Bit=', t:1, '; Sub=');
            dump_bit_subscript(CmpLog, sub, VECTOR);
            writeln(CmpLog);
          end
        else
          begin
            t := 1 + ABS(sub^.left_index - sub^.right_index);
            while (n > t) AND (sub <> NIL) do
              begin
                n := n - t;
                sub := sub^.next;
                t := 1 + ABS(sub^.left_index - sub^.right_index);
              end;
            with sub^ do
              if left_index <= right_index then  n := left_index + (n - 1)
                                           else  n := left_index - (n - 1);
            nth_bit_subscript := TRUE;
          end;
end { nth_bit_subscript } ;


(**)

function nth_bit_of_signal_instance(var  n: bit_range;
                                    var SI: signal_instance_ptr): boolean;
{ -- returns TRUE if the "n"th bit of SI exists. if TRUE n is set to
     the "n"th bit, i.e. n enters as an ordinal exits as a cardinal.
     Note: This procedure differs from nth_bit_subscript in that signal
           instances may have a replication factor                    -- }
   var 
     sub: subscript_ptr;
     t: bit_range;        { number of bits passed over so far }

begin
  if debug_19 then
    begin
      writeln(outfile, 'Entered nth_bit_of_signal_instance: n=', n:1);
      writeln(outfile, '  SI=');
      dump_signal_instance(outfile, SI);
    end;

  nth_bit_of_signal_instance := FALSE;

  if SI = NIL then assert(149);

  sub := SI^.bit_subscript;

  if  (n < 1)  then assert(111 {oops!})
  else
    begin
      t :=  size_of_signal_instance(SI);
      if SI^.replication_factor <> 1 then
        begin
          if n > (t * SI^.replication_factor) then assert(109 { n too big });
          if t = 0 then assert(110 { signal has undefined width })
                   else n := 1 + (n - 1) MOD t; { for rep'ed signals }
        end;
      if (n > t) then assert(111 {oops!})
      else if (n = 1) AND (SI^.defined_by^.kind = SINGLE) then
        nth_bit_of_signal_instance := TRUE { SI is a scalar }
      else if (sub = NIL) then
        assert(163 { vector signal with no bit subscript })
      else
        if sub^.next = NIL then    { check for a non-bit list subscript }
          begin
            t := 1 + ABS(sub^.left_index - sub^.right_index);
            if n > t then
              begin
                assert(59 { doesn't fit in subscript! });
                write(CmpLog, '  Bit=', t:1, '; Sub=');
                dump_bit_subscript(CmpLog, sub, VECTOR);
                writeln(CmpLog);
              end
            else
              with sub^ do
                if left_index <= right_index then n := left_index + (n - 1)
                                             else n := left_index - (n - 1);
            nth_bit_of_signal_instance := TRUE;
          end
        else
          begin    { handle bit lists }
            t := 1 + ABS(sub^.left_index - sub^.right_index);
            while (n > t) AND (sub <> NIL) do
              begin
                n := n - t;   sub := sub^.next;
                t :=  1 + ABS(sub^.left_index - sub^.right_index);
              end;
            with sub^ do
              if left_index <= right_index then n := left_index + (n - 1)
                                           else n := left_index - (n - 1);
            nth_bit_of_signal_instance := TRUE;
          end;
    end;
end { nth_bit_of_signal_instance } ;


(**)


function leading_n_bits_of_subscript(n: natural_number;
                                      sub: subscript_ptr): subscript_ptr;
{ -- returns a pointer to a subscript list which is a copy of the first
     n bits of the argument subscript list -- }
  var t: natural_number;    { width of next element of sub           }
      new_sub,              { next element in the new subscript list }
      new_sub_head: subscript_ptr;    { return subscript list        }
  
  procedure append_new_subscript(var new_sub, sub: subscript_ptr);
  { -- This local utility  appends a copy of the first element
       of sub onto new_sub  -- }
    begin
      if new_sub = NIL then
        begin
          new_subscript(new_sub);    new_sub_head := new_sub;
        end
      else 
        begin
          new_subscript(new_sub^.next);    new_sub:=new_sub^.next;
        end;
      with new_sub^ do
        begin
          left_index := sub^.left_index;    right_index := sub^.right_index;
        end;
    end { append_new_subscript } ;

begin {leading_n_bits_of_subscript}
  leading_n_bits_of_subscript:=NIL;    new_sub := NIL;    new_sub_head := NIL;

  if (n < 1) then
     assert(66 {oops!})
  else
    if (n > width_of_subscript_list(sub)) then
      assert(67 {oops!})
    else
      begin
        t := width_of_subscript_element(sub);
        while (  (n - t) > 0  ) AND (sub <> NIL) do
          begin
            n := n - t;
            append_new_subscript(new_sub,sub);
            sub := sub^.next;
            t := width_of_subscript_element(sub);
          end;
        if  n <> 0 then
          begin
            append_new_subscript(new_sub,sub);
            with new_sub^ do
              if right_index > left_index then        
                right_index := left_index + (n - 1)
              else
                right_index := left_index - (n - 1);
          end;
      end;
  leading_n_bits_of_subscript := new_sub_head;
end { leading_n_bits_of_subscript } ;


procedure invert_bit_subscript_list(var sub: subscript_ptr);
{ -- Reverse  order of bit subscript list AND subranges in the list.
     For example <2..4,8,9,12..9> --> <9..12,9,8,4..2>   --}

  var t: bit_range;               { swap temporary }
      temp_sub: subscript_ptr;    { next subscript element to reverse }
begin
  if sub <> NIL then
    begin
      reverse_bit_subscript(sub);      {-- reverse the elements --}
      temp_sub := sub;
      while temp_sub <> NIL do         {-- reverse the subranges --}
        begin
          with temp_sub^ do
            begin
              t           := left_index;
              left_index  := right_index;
              right_index := t;
            end {with } ;
          temp_sub := temp_sub^.next;
        end {while } ;
    end {if } ;
end { invert_bit_subscript_list } ;


function trailing_n_bits_of_subscript(n: natural_number;
                                    sub: subscript_ptr): subscript_ptr;
{ -- returns subscript list that is last n bits of the argument subscript -- }
  var skip_bits,                    { # of leading bits stripped off so far }
      t: natural_number;            { the width of the current sub element  }
      new_sub_head: subscript_ptr;  { pointer to the return subscript list  }
  
begin {trailing_n_bits_of_subscript}
  if debug_4 then begin
    writeln(outfile, ' Entered trailing_n_bits_of_subscript');
    write(outfile, '   subscript = ');
    dump_bit_subscript(outfile, sub, VECTOR); writeln(outfile);
    writeln(outfile,' n = ',n:1);
  end;

  new_sub_head := NIL;
  if (n < 1) then
     assert(66 {oops!})
  else
    if (n > width_of_subscript_list(sub)) then
      assert(67 {oops!})
    else
      begin
        skip_bits := width_of_subscript_list(sub) - n;
        t := width_of_subscript_element(sub);
        while (skip_bits  >= t) AND (sub <> NIL) do
          begin
            skip_bits := skip_bits - t;
            sub := sub^.next;
            t := width_of_subscript_element(sub);
          end;
  
       if skip_bits = 0 then new_sub_head := copy_subscript(sub)
        else
          begin
            new_subscript(new_sub_head);     new_sub_head^ := sub^;
            with new_sub_head^ do
              if right_index > left_index then        
                left_index := left_index + skip_bits 
              else
                left_index := left_index - skip_bits;
            new_sub_head^.next := copy_subscript(sub^.next);
          end;
      end;
  trailing_n_bits_of_subscript := new_sub_head;

  if debug_4 then begin
    writeln(outfile, ' Exited trailing_n_bits_of_subscript');
    write(outfile, '   subscript = ');
    dump_bit_subscript(outfile, new_sub_head, VECTOR); writeln(outfile);
  end;

end { trailing_n_bits_of_subscript } ;


(**)


procedure add_SD_onto_PCS(SD: signal_descriptor_ptr;
                          var PCS: propertied_CS_ptr;
                          sig_def: signal_definition_ptr);
  { Turn signal descriptor to a signal instance with given signal (SIG_DEF)
    and append to a concatenated signal list. }
  var
    SI: signal_instance_ptr;
    temp_PCS: propertied_CS_ptr;
begin
  if (SD = NIL) or (sig_def = NIL) then
    assert(65 {oops!})
  else
    begin
      SD^.scope := sig_def^.scope;
      SD^.net_id := sig_def^.net_id;
      SI := enter_signal_instance(SD, sig_def);

      if PCS = NIL then
        begin
          new_propertied_CS(PCS);
          PCS^.instance := SI;
        end
      else
        begin
          { go to end of PCS and glue on the new SD }

          temp_PCS := PCS;
          while temp_PCS^.next <> NIL do temp_PCS := temp_PCS^.next;

          new_propertied_CS(temp_PCS^.next);
          temp_PCS^.next^.instance := SI;
        end;
    end;
end { add_SD_onto_PCS } ;


procedure copy_a_bit_of_SI_to_SD(var SD: signal_descriptor_ptr;{ being built }
                                 SI: signal_instance_ptr; { actual           }
                                  b: bit_range);          { bit of actual    }
  { -- Construct a signal descriptor represention of the b bit of a
       signal instance -- }
  var sig_def: signal_definition_ptr;    { signal definition of SI }
begin
   sig_def := SI^.defined_by;

   with SD^ do
     begin
       signal_name := sig_def^.signal^.name;
       polarity := SI^.defined_by^.polarity;
       low_asserted := SI^.low_asserted;
       kind := sig_def^.kind;
       is_const := sig_def^.is_const;
       scope := sig_def^.scope;
     end;

   if SD^.kind = VECTOR then
     add_subscript(SD^.bit_subscript, b);
end { copy_a_bit_of_SI_to_SD } ;


(**)


procedure build_SD_from_SI(var SD: signal_descriptor_ptr;
                               SI: signal_instance_ptr);
{ -- Construct a signal descriptor representation of a signal instance -- }
  var sig_def: signal_definition_ptr;    { signal definition of SI }

begin
  if SI = NIL then
    SD := NIL {release_signal_descriptor(SD)}
  else
    begin
      sig_def := SI^.defined_by;
      with SD^ do
        begin
          signal_name := sig_def^.signal^.name;
          polarity := SI^.defined_by^.polarity;
          low_asserted := SI^.low_asserted;
          replication_factor := SI^.replication_factor;
          kind := sig_def^.kind;
          scope := sig_def^.scope;
          bit_subscript := SI^.bit_subscript;
          properties := NIL;
          net_id := SI^.defined_by^.net_id;
          is_const := sig_def^.is_const;

        end { with ... };
    end { else ... };
end { build_SD_from_SI } ;

function append_bit_to_SD(SD: signal_descriptor_ptr;     { SD to augment  }
                                   b: bit_range):boolean;{ new bit to add }

{ -- Tries to add the bit b to the SD. Returns true iff it could add the bit.
     In order for a bit to be "appendable" the SD must not be replicated
     or its size must be one and we are adding the same bit.              -- }
    var expanded_SD: boolean;    { => was able to expand the SD }
  
begin
  if debug_4 then begin
    writeln(outfile, 'Entered append_bit_to_SD with: ');
    dump_signal_descriptor(outfile, SD);
    writeln(outfile, ' Bit to add = ', b:1);
  end;

  expanded_SD := FALSE;
  case SD^.kind of
    SINGLE: begin
              SD^.replication_factor := SD^.replication_factor + 1;
              expanded_SD := TRUE;
            end;
    VECTOR: begin { first tries to make SD a replicated signal}
              if SD^.bit_subscript^.next = NIL then { no bit list }
                if (b = SD^.bit_subscript^.right_index) then { no subrange }
                  if (b = SD^.bit_subscript^.left_index) then{ & b=the bit }
                    begin
                      SD^.replication_factor := SD^.replication_factor + 1;
                      expanded_SD := TRUE;
                    end;
         { -- if we  couldn't do it - try to make expand SD's subscript -- }
                if (SD^.replication_factor = 1) AND NOT expanded_SD then
                  begin
                    add_subscript(SD^.bit_subscript, b);
                    expanded_SD := TRUE;
                  end
                else
                  {nuttin}
           end;
    UNDEFINED: begin
                 assert(86 {PUNT!});
                 write(CmpLog, ' SD = ');
                 dump_signal_descriptor(CmpLog, SD);
               end;
  end { of case };

  append_bit_to_SD := expanded_SD;

  if debug_4 then
    begin
      write(outfile, 'Exited append_bit_to_SD (');
      if expanded_SD then  writeln(outfile, 'TRUE) with: ')
                     else  writeln(outfile, 'FALSE) with: ');
      dump_signal_descriptor(outfile, SD);
    end;
  
end { append_bit_to_SD } ;

                  
(**)


procedure replicate_PCS(PCS: propertied_CS_ptr;  rep_factor: natural_number);
  { Takes a PCS, FOO, and replication factor k and clobbers FOO to
    FOO : FOO : FOO ...     k repetions in all }
  var
    head_of_the_PCS,                        { first instance of input PCS   }
    tail_of_the_PCS,                        { last instance of input PCS    }
    temp_PCS_head:  propertied_CS_ptr;
    SD:            signal_descriptor_ptr;   { if PCS has one inst we rep only
                                              it. SD is descr. of new inst   }
    i, j,
    size_of_the_PCS: natural_number;        { number of SIs in the PCS      }
begin
  if rep_factor = 0 then
    assert(80 {oops})
  else
    if rep_factor = 1 then
      {nuttin}
    else
      begin
        head_of_the_PCS := PCS;
        tail_of_the_PCS := PCS;
        size_of_the_PCS := 1;

        while tail_of_the_PCS^.next <> NIL do 
          begin
            tail_of_the_PCS := tail_of_the_PCS^.next;
            size_of_the_PCS := size_of_the_PCS + 1;
          end;

        if size_of_the_PCS = 1 then { special case, only one SI in PCS }
          begin
            SD := NIL;
            new_signal_descriptor(SD);
            build_SD_from_SI(SD, PCS^.instance);
            SD^.replication_factor := rep_factor;
            with PCS^ do
              instance := enter_signal_instance(SD, instance^.defined_by);
            release_signal_descriptor(SD);
          end
        else
          for i:= 1 to rep_factor - 1 do
            begin
              temp_PCS_head := head_of_the_PCS;
              for j := 1 to size_of_the_PCS do
                begin
                  new_propertied_CS(tail_of_the_PCS^.next);
                  tail_of_the_PCS := tail_of_the_PCS^.next;
                  tail_of_the_PCS^.instance := temp_PCS_head^.instance;
		  tail_of_the_PCS^.control := temp_PCS_head^.control;
                  temp_PCS_head := temp_PCS_head^.next;
                end;
            end { for ... };
      end { else (rep_factor <> 1) ... };
end { replicate_PCS };


(**)


procedure dump_PCS(var f: textfile; signal: propertied_CS_ptr);
  { dump the propertied concatenated signal to the specified file }
  var
    PCSP: propertied_CS_ptr;    { propertied CS being printed }
begin
  PCSP := signal;
  while PCSP <> NIL do
    begin
      dump_signal_instance(f, PCSP^.instance);
      if PCSP^.properties = NIL then write(f, '- properties = <NIL>')
      else dump_property_list(f, PCSP^.properties);
       
      PCSP := PCSP^.next;
      if PCSP <> NIL then
        begin  writeln(f, ':');  write(f, '    ');  end;
    end;

  writeln(f);
end { dump_PCS } ;


function copy_PCS(sig: propertied_CS_ptr): propertied_CS_ptr;
  { Make a copy of the given concatenated signal and return it }
  var
    PCS,                                    { current element of signal }
    last,                                   { last element in the list  }
    copied_PCS: propertied_CS_ptr;          { signal to return          }
begin
  PCS := sig;  last := NIL;  copied_PCS := NIL;
  while PCS <> NIL do
    begin
      if last = NIL then
        begin  new_propertied_CS(last);  copied_PCS := last;  end
      else
        begin  new_propertied_CS(last^.next);  last := last^.next;  end;

      last^.instance := PCS^.instance;
      last^.control := PCS^.control;
      copy_properties(PCS^.properties, last^.properties);

      PCS := PCS^.next;
    end;

  copy_PCS := copied_PCS;
end { copy_PCS } ;


procedure append_SD_onto_PCS(SD: signal_descriptor_ptr;
                             sig_def: signal_definition_ptr;
                             var PCS: propertied_CS_ptr);
  { Turns SD into an SI and appends it to end of PCS. sig_def will be NIL
    if someone is creating an ersatz signal. A typical way for this to
    occur is if a lookup fails (in resolve_instance for example, the
    actual will be NIL and SD will be an NC SD). SIG_DEF = NIL means that
    the newly created signal instance will have local scope. }
  var 
    temp_PCS: propertied_CS_ptr;     { will point to last instance of PCS }
    SI: signal_instance_ptr;         { SD will become this sig instance   }
begin
  if debug_4 then
    begin
      write(outfile, 'Entered append_SD_onto_PCS');
      write(outfile, '  PCS: ');
      dump_PCS(outfile, PCS);
    end;

  if sig_def = NIL then
    begin
      SD^.scope := LOCAL;
      SD^.net_id := nullstring;
    end
  else
    begin
      SD^.scope := sig_def^.scope;
      SD^.net_id := sig_def^.net_id;
    end;

  SI := enter_signal_instance(SD, sig_def);

  if PCS = NIL then
    begin  new_propertied_CS(PCS);  temp_PCS := PCS;  end
  else
    begin
      { go to end of PCS and glue on the new SD }

      temp_PCS := PCS;
      while temp_PCS^.next <> NIL do temp_PCS := temp_PCS^.next;

      new_propertied_CS(temp_PCS^.next);
      temp_PCS := temp_PCS^.next;
    end;

  temp_PCS^.instance := SI;
  temp_PCS^.properties := copy_property_list(SD^.properties);

  if debug_4 then
    begin
      write(outfile, 'Exiting append_SD_onto PCS ');
      write(outfile, '  PCS: ');
      dump_PCS(outfile, PCS);
      writeln(outfile,'  properties to be assigned');
      dump_property_list(outfile, PCS^.properties);
    end;
end { append_SD_onto_PCS } ;


(**)


function position_of_bit_in_subscript(var n: bit_range;
                                        sub: subscript_ptr): boolean;
{ -- returns TRUE if the bit "n" exists in the subscript list, if TRUE
     n is set to the first occurance of the bit "n", i.e. n enters as
     an cardinal and exits as a ordinal. If the bit "n" does not exist
     in the subscript list, then n is unalterred.  -- }
   var position: bit_range;    { first occurance of bit "n" }
       found: boolean;              { return value               }

begin
  if (sub = NIL) AND (n = 1) then  {sub is part of scalar}
    begin  position := 1;   found := TRUE;    end
  else
    begin  position := 0;   found := FALSE;   end;

  while (sub <> NIL) AND NOT found do
    with sub^ do
      begin
        if left_index < right_index then
          if (n >= left_index) AND (n <= right_index) then
            found := TRUE
          else {nuttin}
        else
          if (n <= left_index) AND (n >= right_index) then
            found := TRUE;
  
        if found then
          if left_index <= right_index then
            position := position + 1 + (n - left_index)
          else
            position := position + 1 - (n - left_index)
        else
          begin
            position := position + 1 + ABS(left_index - right_index);
            sub := sub^.next;
          end;
      end;
  if found then n := position;    position_of_bit_in_subscript := found;
end { position_of_bit_in_subscript } ;
      

function find_subscript(b: bit_range; sub: subscript_ptr): boolean;
{ -- Returns true if bit "b" is found anywhere in a subscript list -- }
  var found: boolean;    { return value }

begin
  found:=FALSE;
  if (sub = NIL) AND (b = 1) then found := TRUE { sub is part of scalar }
  else
    if sub = NIL then
      assert(57 {oops!})
    else
      while (sub <> NIL) AND NOT found do
        if bit_in_range(b, sub) then  found := TRUE
                                else  sub := sub^.next;
  find_subscript := found;
end { find_subscript } ;


function bit_in_bit_property_subscript(b: bit_range;
                                       sub: subscript_ptr): boolean;
  { Returns TRUE if bit "b" is found anywhere in a subscript list.  If the
    subscript is NIL, it is assumed to apply to the entire subscript.  That
    is, any bit is part of the subscript. }
  var
    current_bits: subscript_ptr;    { current bits of the subscript }
    found: boolean;                 { return value }
begin
  found:=FALSE;

  if (sub = NIL) then found := TRUE
  else
    begin
      current_bits := sub;
      while (current_bits <> NIL) and not found do
        if bit_in_range(b, current_bits) then
          found := TRUE
        else
          current_bits := current_bits^.next;
    end;

  bit_in_bit_property_subscript := found;
end { bit_in_bit_property_subscript } ;


(**)     { ------- bit property utilities ------- }


procedure dump_bit_properties(var f: textfile; list: bit_property_ptr);
  { dump the given property list to the given file }
  var
    PP: bit_property_ptr;      { current bit property to be output }
begin
  PP := list;
  while PP <> NIL do
    with PP^ do
      begin
        write(f, '    ');    writealpha(f, name^.name);    write(f, '=');
        writestring(f, text);
        dump_bit_subscript(f,bit_subscript, VECTOR);  writeln(f, ';');
        PP := next;
      end;
  if list = NIL then writeln(f);     { this is a KLUDGE! }
end { dump_bit_properties } ;


procedure copy_props_to_bits_of_def(prop_list: property_ptr;
                                    bits: subscript_ptr;
                                    def: signal_definition_ptr);
  { add the given property list (PROP_LIST) to the given signal def (DEF)
    with the specified properties (BITS). }
  var
    prop: property_ptr;      { current property in source list }
begin
  if debug_7 then
    begin
      writeln(outfile, 'Entering copy_props_to_bits_of_def: props to add=');
      dump_property_list(outfile, prop_list);
      writeln(outfile, '  bits=');
      dump_bit_subscript(outfile, bits, VECTOR);
    end;

  prop := prop_list;
  while prop <> NIL do
    begin
      if INHERIT_PIN IN prop^.name^.kind then
        add_pin_property_to_bit_property(def^.properties,
                                         prop^.name,
                                         prop^.text,
                                         bits)
      else if INHERIT_SIGNAL IN prop^.name^.kind then
        add_property_to_bit_property_list(def^.properties,
                                          prop^.name,
                                          prop^.text,
                                          bits);

      prop := prop^.next;
    end;

  if debug_7 then
    begin
      writeln(outfile, 'Exiting copy_props_to_bits_of_def: def=');
      dump_signal_definition(outfile, def);
    end;
end { copy_props_to_bits_of_def } ;


function bit_properties_of_this_bit(source_props: bit_property_ptr;
                                    bit: bit_range;
                                    which: property_selector): property_ptr;
  { return a property list consisting of the properties associated with
    the given bit.  If the subscript in the bit property is NIL, it implies
    a property of the entire subscript.  If WHICH = NO_FILTERED_PROPERTIES,
    ignore those properties that have the DONT_OUTPUT attribute. }
  var
    current_prop: bit_property_ptr;   { current property in source list }
    prop_list: property_ptr;          { list to be returned }
begin
  { !!!! This is a dumb implementation of this procedures !!!! }

  current_prop := source_props;  prop_list := NIL;
  while current_prop <> NIL do
    begin
      if bit_in_bit_property_subscript(bit, current_prop^.bit_subscript) then
        if (which = ALL_PROPERTIES) or 
           (signal_inheritance_attributes * current_prop^.name^.kind <>
                                                                      []) then
          add_to_prop_list(prop_list, current_prop^.name, current_prop^.text);

      current_prop := current_prop^.next;
    end;

  bit_properties_of_this_bit := prop_list;
end { bit_properties_of_this_bit } ;


function gather_pin_properties(formal_actual_pair: formal_actual_ptr;
                               bit: bit_range): property_ptr;
  { gather properties from the pin (FORMAL_ACTUAL_PAIR) for the given
    BIT (-1 implies all bits of the pin) and return them.
    If the pin is a pin of a plumbing body, gather INHERIT_PIN and
    INHERIT_SIGNAL properties.  If not, gather only INHERIT_PIN properties. }
  label
    90; { return }
  var
    prop_list: property_ptr;    { list of properties to be returned }
    node: mtree_node_ptr;       { node corresponding to this pin's body }
    source_prop:
       subscript_property_ptr;  { current source property list }
    prop: property_ptr;         { current property being copied }
    attributes: name_type_set;  { property attributes to copy }
begin
  if debug_7 or debug_14 then
    write(outfile, 'gather_pin_properties: node=');

  { determine whether this pin is on a plumbing body and determine which
    property attributes are to be searched for. }

  if formal_actual_pair = NIL then
    begin
      assert(133);
      gather_pin_properties := NIL;
      goto 90 { return } ;
    end;
  attributes := [INHERIT_PIN];
  if formal_actual_pair^.formal_parameter <> NIL then
    begin
      node := formal_actual_pair^.formal_parameter^.defined_by^.node;
      if node^.is_plumbing_node then
        attributes := [INHERIT_PIN, INHERIT_SIGNAL];
      if debug_7 or debug_14 then
        print_string(outfile, node^.macro_name);
    end;

  if debug_7 or debug_14 then
    begin
      writeln(outfile);
      if attributes = [INHERIT_PIN, INHERIT_SIGNAL] then
        writeln(outfile, ' Is a plumbing body');
    end;

  prop_list := NIL;

  source_prop := formal_actual_pair^.properties;
  while source_prop <> NIL do
    begin
      if (source_prop^.left_index < 0) OR (bit = -1) OR
         ((bit <= source_prop^.left_index) AND
          (bit >= source_prop^.right_index)) OR
         ((bit >= source_prop^.left_index) AND
          (bit <= source_prop^.right_index)) then
        begin
          prop := source_prop^.properties;
          while prop <> NIL do
            begin
              if attributes * prop^.name^.kind <> [] then
                if INHERIT_PIN IN prop^.name^.kind then
                  add_to_prop_list(prop_list, prop^.name, prop^.text)
                else
                  check_and_add_to_prop_list(prop_list,
                                             prop^.name, prop^.text);

              prop := prop^.next;
            end;
        end;

      source_prop := source_prop^.next;
    end;

  gather_pin_properties := prop_list;

  if debug_7 or debug_14 then
    begin
      writeln(outfile, 'Exiting gather_pin_properties: prop_list=');
      dump_property_list(outfile, prop_list);
    end;
90:
end { gather_pin_properties } ;


(**)     { ------- Subscript Property Utilities ------- }


function all_props_of_this_subscript(props: subscript_property_ptr;
                                     bit: bit_range): property_ptr;
  { return a list of the properties of the given bit of the subscript props }
  var
    prop_list: property_ptr;     { property list being created for return }
    PP: property_ptr;            { current property being copied }
begin
  if debug_14 then
    begin
      writeln(outfile, 'Entering all_props_of_this_subscript: bit=',
                        bit:1);
      writeln(outfile, '  properties of subscript:');
      dump_subscript_property_list(outfile, props);
    end;

  prop_list := NIL;
  while props <> NIL do
    begin
      if (props^.left_index < 0) OR
         ((bit <= props^.left_index) AND (bit >= props^.right_index)) OR
         ((bit >= props^.left_index) AND (bit <= props^.right_index)) then
        begin
          PP := props^.properties;
          while PP <> NIL do
            begin
              if INHERIT_PIN IN PP^.name^.kind then
                add_to_prop_list(prop_list, PP^.name, PP^.text)
              else
                check_and_add_to_prop_list(prop_list, PP^.name, PP^.text);

              PP := PP^.next;
            end;
        end;

      props := props^.next;
    end;

  all_props_of_this_subscript := prop_list;

  if debug_14 then
    begin
      writeln(outfile, 'Exiting all_props_of_this_subscript: props:');
      dump_property_list(outfile, prop_list);
    end;
end { all_props_of_this_subscript } ;


(**)     { ------- Basescript Utilities ------- }


function find_basescript(b: bit_range; BS: basescript_ptr): boolean;
{ -- Returns true if bit "b" is found anywhere in basescript list BS -- }
  var
    
    found: boolean;    { return value }
begin
  found:=FALSE;
  if (BS = NIL) AND (b = 1) then found := TRUE { BS is part of scalar }
  else
    if BS = NIL then
      assert(64 {oops!})
    else
      while (BS <> NIL) AND NOT found do
        if left_to_right AND
          (b >= BS^.left_index) AND (b <= BS^.right_index) then
          found := TRUE
        else if (b <= BS^.left_index) AND (b >= BS^.right_index) then
          found := TRUE
        else
          BS := BS^.next;
  find_basescript := found;
end { find_basescript } ;

function is_base_instance(SI: signal_instance_ptr): boolean;

{ make this guy smarter i.e. handle subranges directly }

  var
    is_base_so_far:  boolean;
    bits_of_SI:      subscript_ptr;
    non_base_bits:   basescript_ptr;
    i,t,
    size_of_SI: bit_range;
begin
  non_base_bits := SI^.defined_by^.synonym_bits;
  bits_of_SI := SI^.bit_subscript;
  size_of_SI := size_of_signal_instance(SI);
  is_base_so_far := FALSE;
  if (non_base_bits = NIL) then
    is_base_so_far := TRUE
  else
    begin
      is_base_so_far := TRUE; i := 0;
        repeat
          begin
            i := i + 1;    t := i;
            if nth_bit_subscript(t, bits_of_SI) then { nuttin }
            is_base_so_far := is_base_so_far AND 
                              NOT find_basescript(t, non_base_bits);
          end;
        until (NOT is_base_so_far) OR (i >= size_of_SI);
    end;
  is_base_instance := is_base_so_far;
end { is_base_instance } ;
(**)

#include "res_param.p"
(**)


procedure dump_basescript(var f: text; BS: basescript_ptr);
  { -- print a single basescript in the specified file -- }
begin
  if BS = NIL then writeln(f, '<NIL>')
  else
    begin
      writeln(f, 'off:', BS^.offset:1, ',left:', BS^.left_index:1,
                 ',right:', BS^.right_index:1, '; Signal Instance:');
      write(f,'    ');
      dump_signal_instance(f, BS^.instance);
    end;
end { dump_basescript };


procedure dump_basescript_list(*var f: text; BS: basescript_ptr*);
  { -- print a list basescripts in the specified file -- }
begin
  if BS = NIL then writeln(f,'<NIL> ')
  else
    while BS <> NIL do
      begin
        dump_basescript(f, BS);

        BS := BS^.next;

        if BS <> NIL then write(f, '      ');
      end;
end { dump_basescript_list } ;


procedure dump_base_descriptor(var f: text; BD: base_descriptor_ptr);
  { -- print out a single base descriptor -- }
begin
  if BD = NIL then writeln(f, '<NIL>')
  else
    begin
      writeln(f, 'offset:', BD^.offset:1, ',width:', BD^.width:1,
                 '; Signal Instance: ');
      write(f, '    ');
      dump_signal_instance(f, BD^.instance);
    end;
end { dump_base_descriptor };


(**)


function is_same_constant(BD1, BD2: base_descriptor_ptr): boolean;
{ -- checks to see if the "b"th bit of two constants are identical -- }
  var
    i: bit_range;                     { increments over bits of constants   }
    matches: boolean;                 { => bits of two consts are the same  }
    sig_def1,                         { sig. def. of virtual instance BD1   }
    sig_def2:  signal_definition_ptr; { sig. def. of virtual instance BD2   }
begin
  if debug_2 then
    begin
      write(outfile, 'Entered is_same_constant with: ');
      dump_base_descriptor(outfile, BD1);
      dump_base_descriptor(outfile, BD2);
    end;

  matches := FALSE;

  sig_def1 := BD1^.instance^.defined_by;
  sig_def2 := BD2^.instance^.defined_by;
  if (BD1^.width <> BD2^.width) OR
     ((BD1^.width + BD1^.offset) > width_of_signal_definition(sig_def1)) OR
     ((BD2^.width + BD2^.offset) > width_of_signal_definition(sig_def2)) then
    assert(84 { incompatible widths } )
  else
    if sig_def1^.is_const then
      if sig_def2^.is_const then
        begin
          matches := TRUE;    i := 0;
          repeat
            i := i + 1;
            if sig_def1^.signal^.name^[i+BD1^.offset] <>
               sig_def2^.signal^.name^[i+BD2^.offset] then matches := FALSE;
          until (i >= BD1^.width) OR (NOT matches);
        end;

  is_same_constant := matches;

  if debug_2 then
    if matches then writeln(outfile, ' Exited is_same_constant=TRUE')
               else writeln(outfile, ' Exited is_same_constant=FALSE');
end { is_same_constant } ;


#include "synonym.p"


(**)     { ------- Base Signal Utilities ------- }


function find_base_of_signal_instance(SI: signal_instance_ptr):
                                                            propertied_CS_ptr;
{ -- Finds the base representation of a signal instance. Since each
     bit (in general) may have a base representation with a different signal,
     the returned goody is a concatenated signal. This routine works by
     finding the base name of SI bit-by-bit (which is the real work). When
     the base for the bit is returned, the routine tries to augment the
     "right-most" signal instance of the return CS to include the new bit.
     If it does not succeed  it starts a new concatenated signal element     }

  var BD,
      base_BD: base_descriptor_ptr;
      clean_SD,                                 { blank SD to reinit temp_SD }
      temp_SD:  signal_descriptor_ptr;          { SD to SI-ify and add to SD }
      last_looked_up_sig_def,                   { def for last base bit      }
      looked_up_sig_def: signal_definition_ptr; { sig def of next base bit   }
      sig_def_bit: bit_range;                   { bit sub of next base bit   }
      PCS: propertied_CS_ptr;                   { base representation of SI  }
      first_time_through_loop: boolean;         { =>1st bit is being debased }
      i,t: bit_range;
      KLUDGE: boolean; { !!!! We have temporarily hacked find_base...
In the case where SI is BASE, the orginal code merely CSifies it and returns
it. This is presently not acceptable because it can result in bit lists in
the CMPEXP file which the Simulator and Timing Verifier cannot handle. To get
around this, we do ALL lookups bit wise, (the bit wise loop is smart enough
not to build bit lists -- though it is trivial to make it smart enough to 
build bit lists). For efficiency find_base_of_signal_instance sets this flag
if SI is base, so that find_base_of_signal "knows" not to really do the
lookup }




  function find_base_of_signal(SI: signal_instance_ptr;
                               bit: bit_range;
                           var base_sig_def: signal_definition_ptr;
                           var base_bit: bit_range): boolean;
    {  A signal (single bit) in general is known by several names. In the
       expantion output, the signal will be known by one name. This routine
       picks that name. Given any name by which a signal is known, it returns
       the "canonical name", referred to as a base name for the signal. -- }
  label 1;
  var
    ans,
    dummy:   boolean;
  begin
    if debug_2 then
      begin
        writeln(outfile, 'Entered find_base_of_signal with: ');
        dump_signal_instance(outfile, SI);
        writeln(outfile, '  bit = ', bit:1);
      end;
  
    if KLUDGE then
      begin
        { KLUDGE was true, so SI is all base }
        base_sig_def  := SI^.defined_by;
        base_bit      := bit;
        ans           := TRUE;
        GOTO 1;
      end;

  { We use two "scratch" base_descriptors, BD and base_BD. These are new'ed
    and released in find_base_of_signal_instance for efficiency            }

    ans := FALSE;
    dummy := position_of_bit_in_subscript(bit, SI^.bit_subscript);

    with BD^ do
      begin
        instance := SI;
        width    := 1;
        offset   := bit - 1;
      end;

    if NOT find_base_of_base_descriptor(BD, base_BD) then
      assert(100 {single bit lookups must succeed})
    else
      with base_BD^ do
        begin
          ans := TRUE;
          base_sig_def := instance^.defined_by;
          base_bit := 1 + offset;
          dummy := nth_bit_of_signal_instance(base_bit, instance);
        end;
  
1:
    find_base_of_signal := ans;

    if debug_2 then
      begin
        writeln(outfile, 'Exited find_base_of_signal (', ord(ans):1, ') with: ');
        write(outfile, '  Signal_def: ');
        dump_signal_definition(outfile, base_sig_def);
        writeln(outfile, '  bit = ', base_bit:1);
      end;
  end { find_base_of_signal } ;


  procedure init;
    { initialize }
  begin
    new_signal_descriptor(temp_SD);
    new_signal_descriptor(clean_SD);

    looked_up_sig_def := NIL;
    last_looked_up_sig_def := NIL;
    PCS := NIL;
    sig_def_bit := 0;
    first_time_through_loop := TRUE;

    base_BD := NIL;
    new_base_descriptor(base_BD);

    BD := NIL;
    new_base_descriptor(BD);

    KLUDGE := is_base_instance(SI);
  end { init } ;


  procedure un_init;
    { release temporaries }
  begin
    release_complete_signal_descriptor(temp_SD);
    release_complete_signal_descriptor(clean_SD);
    release_base_descriptor(base_BD);   release_base_descriptor(BD);
  end { un_init } ;


  procedure copy_sig_def_into_SD(var SD: signal_descriptor_ptr;
                                 sig_def: signal_definition_ptr;
                                 b: bit_range);
    { Build an initial version of a signal descriptor (SD).
      We call this when a (precursor of a ) new element of
      a concatenated signal is started.  Sig_def is looked up,
      b is bit of looked up, p is polarity of looked up. }
  begin
     with SD^ do
       begin
         signal_name := sig_def^.signal^.name;
         kind := sig_def^.kind;
         net_id := sig_def^.net_id;
         polarity := sig_def^.polarity;
         low_asserted := (polarity = COMPLEMENTED);
         properties := NIL;
         is_const := sig_def^.is_const;
       end;

     if SD^.kind = VECTOR then
       add_subscript(SD^.bit_subscript, b);
  end { copy_sig_def_into_SD } ;
    

  function augment_signal_descriptor(SD: signal_descriptor_ptr;
                                     current_sig_def,
                                     new_sig_def: signal_definition_ptr;
                                     b: bit_range):boolean;
    { augment signal_descriptor (SD) built from sign_def (CURRENT_SIG_DEF)
      from looked up sig_def (NEW_SIG_DEF) and looked up bit (B). }
     var expanded_SD: boolean;


    function expand_subscript: boolean;
      { expand the subscript }
      const we_can_have_bit_lists = FALSE; { => bit lists OK in CMPEXP }
      var direction: -1..1;          { -1 => sub is right to left      }
                                     { +1 => sub is left to right      }
          last_sub:  subscript_ptr;  { points to last sub in bit list  }
      begin
        expand_subscript := FALSE;
    
        if SD^.bit_subscript <> NIL then
          begin
            last_sub := SD^.bit_subscript;
            while last_sub^.next <> NIL do last_sub := last_sub^.next;

            with last_sub^ do
              if (left_index < right_index)      then direction := 1 
              else if (left_index > right_index) then direction := -1
              else if (b > right_index) then direction := 1 {use default dir}
              else direction := -1;
       
            if (b = last_sub^.right_index + direction) then
              begin
                last_sub^.right_index := last_sub^.right_index + direction;
                expand_subscript := TRUE;
              end
            else if we_can_have_bit_lists then
              begin
                add_subscript_element(last_sub,b);
                expand_subscript := TRUE;
              end;
           end
         else
           begin
             add_subscript_element(SD^.bit_subscript,b);
             expand_subscript := TRUE;
           end;
      end { expand_subscript } ;


  begin { augment_signal_descriptor }
    if debug_19 then begin
      writeln(outfile, 'Entered augment_signal_descriptor with: ');
      dump_signal_descriptor(outfile, SD);
      write(outfile, ' current and looked up sig_def:');
      dump_signal_definition(outfile, current_sig_def);
      dump_signal_definition(outfile, new_sig_def);
      writeln(outfile, ' Bit to add = ', b:1);
    end;
  
    expanded_SD := FALSE;
    if new_sig_def = current_sig_def then
      case SD^.kind of
        SINGLE: 
          begin
            SD^.replication_factor := SD^.replication_factor + 1;
            expanded_SD := TRUE;
          end { SINGLE };
        VECTOR:
          begin { first tries to make SD a replicated signal}
            if SD^.bit_subscript^.next = NIL then { no bit list }
              if (b = SD^.bit_subscript^.right_index) then { no subrange }
                if (b = SD^.bit_subscript^.left_index) then{ & b=the bit }
                  begin
                    SD^.replication_factor := SD^.replication_factor + 1;
                    expanded_SD := TRUE;
                  end;
        { -- if we  couldn"t do it - try to make expand SD"s subscript -- }
              if (SD^.replication_factor = 1) AND NOT expanded_SD then
                expanded_SD := expand_subscript;
(*                begin
                  add_subscript(SD^.bit_subscript, b);
                  expanded_SD := TRUE;
                end
              else
*)
          end { VECTOR };
        UNDEFINED:
          begin
            assert(87 {PUNT!});
            write(CmpLog, '                 SD = ');
            dump_signal_descriptor(CmpLog, SD);
            write(CmpLog, '        new_sig_def = ');
            dump_signal_definition(CmpLog, new_sig_def);
            write(CmpLog, '    current_sig_def = ');
            dump_signal_definition(CmpLog, current_sig_def);
          end { UNDEFINED };
       end { of case };
    augment_signal_descriptor := expanded_SD;

    if debug_19 then begin
      write(outfile, 'Exited augment_signal_descriptor (');
      writeln(outfile, expanded_SD:5, ') with: ');
      dump_signal_descriptor(outfile, SD);
    end;
  end { augment_signal_descriptor } ;


  procedure baseless_signal(var SD: signal_descriptor_ptr);
    { If we fail to find a base bit for some bit of a signal instance, we
       use NC as the base. This routine cooks up the NC signal descriptor }
  begin
    if debug_2 then
      begin
        writeln(outfile, 'Entered baseless signal with:');
        dump_signal_descriptor(outfile, SD);
      end;

    if is_NC_signal(SD^.signal_name) then
      { -- already working on NC vector, just augment it -- }
      with SD^.bit_subscript^ do right_index:=right_index+1
 
   else
     begin
       { -- add what we got so far, use NC for failed base -- }

       if last_looked_up_sig_def <> NIL then
         add_SD_onto_PCS(SD, PCS, last_looked_up_sig_def);

       { -- setting last... to NIL ensures that when ersatz NC signal
         is glued on the PCS it will get a scope of LOCAL -- }

       last_looked_up_sig_def := NIL;
       first_time_through_loop := FALSE;

       { --  build a new NC signal_descriptor -- }
       SD^ := clean_SD^;
       with SD^ do
         begin
           signal_name := unique_NC_name;
           kind := VECTOR;
           is_const := FALSE;
           add_subscript(bit_subscript,0);
        end;
     end {else } ;

   if debug_2 then
     begin
       writeln(outfile, 'Exited baseless signal:');
       write(outfile,'       SD = ');  dump_signal_descriptor(outfile, SD);
    end;
  end { baseless signal } ;


begin { find_base_of_signal_instance }
  if debug_2 then
    begin
      writeln(outfile, 'Entered find_base_of_signal_instance with:');
      write(outfile, '  SI = ');
      dump_signal_instance(outfile, SI);
      dump_signal_definition(outfile, SI^.defined_by);
      write(outfile, '  non-base bits: ');
      dump_basescript_list(outfile, SI^.defined_by^.synonym_bits);
    end;

  if FALSE then { Oh KLUDGE of KLUDGES }
  else { do it bit wise for now }
    begin        
      init;

      for i := 1 to size_of_signal_instance(SI) do
        begin
          t:=i;
          if NOT nth_bit_subscript(t, SI^.bit_subscript) then
            begin
              assert(82 {oops!});    write(CmpLog,' SI = ');
              dump_signal_instance(CmpLog, SI);
              writeln(CmpLog,' bit = ', t:1, '(i = ', i:1,')');
            end
          else
            if NOT find_base_of_signal(SI, t,
                           looked_up_sig_def, sig_def_bit) then
              begin
                assert(83 {bad SI or undefined bit in SI});
                write(CmpLog,' SI: ');
                dump_signal_instance(CmpLog, SI);
                writeln(CmpLog, '  bit number: ', t:1);
                baseless_signal(temp_SD);
              end
          else
            begin
              if debug_2 then
                begin
                  writeln(outfile, 'Goodies before work in main loop !!!!!');
                  write(outfile, '  temp_SD: ');
                  dump_signal_descriptor(outfile, temp_SD);
                  write(outfile, '  looked_up_sig_def: ');
                  dump_signal_definition(outfile, looked_up_sig_def);
                  write(outfile, '  last_looked_up_sig_def: ');
                  dump_signal_definition(outfile, last_looked_up_sig_def);
                  write(outfile, '  PCS: ');
                  dump_propertied_CS(outfile, PCS);
                end;

              if first_time_through_loop then
                begin
                  copy_sig_def_into_SD(temp_SD, looked_up_sig_def,
                                       sig_def_bit);
                  last_looked_up_sig_def := looked_up_sig_def;
                  first_time_through_loop := FALSE;
                end
              else
                if augment_signal_descriptor(temp_SD,
                                             last_looked_up_sig_def,
                                             looked_up_sig_def,sig_def_bit)
                then {nuttin}
                else
                  begin
                    add_SD_onto_PCS(temp_SD, PCS, last_looked_up_sig_def);
                    temp_SD^ := clean_SD^;
                    copy_sig_def_into_SD(temp_SD, looked_up_sig_def,
                                         sig_def_bit);
                    last_looked_up_sig_def := looked_up_sig_def;
                  end { else ... };

{            if debug_2 then begin
              writeln(outfile,'!!!!! Goodies after work in main loop !!!!!');
              write(outfile, '      temp_SD: ');
                   dump_signal_descriptor(outfile, temp_SD);
              write(outfile, '      looked_up_sig_def: ');
                   dump_signal_definition(outfile, looked_up_sig_def);
              write(outfile, '      last_looked_up_sig_def: ');
                   dump_signal_definition(outfile, last_looked_up_sig_def);
            write(outfile, '     PCS: ');
                   dump_propertied_CS(outfile, PCS);
            end;
}
          end { else  no lookup errors... };
        end { for ... };
      add_SD_onto_PCS(temp_SD, PCS, last_looked_up_sig_def);

{    if debug_2 then begin
      writeln(outfile,'!!!!! Goodies on exit from main loop !!!!!');
      write(outfile, '      temp_SD: ');
           dump_signal_descriptor(outfile, temp_SD);
      write(outfile, '      looked_up_sig_def: ');
           dump_signal_definition(outfile, looked_up_sig_def);
      write(outfile, '      last_looked_up_sig_def: ');
           dump_signal_definition(outfile, last_looked_up_sig_def);
      write(outfile, '     PCS: ');
           dump_propertied_CS(outfile, PCS);
    end;
}
      replicate_PCS(PCS, SI^.replication_factor);

      un_init;
  end { begin  no argument errors  };

  find_base_of_signal_instance := PCS;

  if debug_2 then
    begin
      write(outfile, 'Exited find_base_of_signal_instance');
      dump_propertied_CS(outfile, PCS);
    end;
end { find_base_of_signal_instance } ;


function find_base_of_PCS(input_PCS: propertied_CS_ptr): propertied_CS_ptr;
  { Finds the base representation of a concatenated signal by "debasing" it
    signal instance by signal instance }
  var
    current_base,                       { current part of base signal }
    base_signal: propertied_CS_ptr;     { base signal as determined }
    current_signal: propertied_CS_ptr;  { current signal being debased }
begin
  if debug_2 then
    begin
      write(outfile, 'Entered find_base_of_PCS with: ');
      dump_propertied_CS(outfile, input_PCS);
    end;

  base_signal := NIL;
  if input_PCS <> NIL then
    begin
      current_signal := input_PCS;  current_base := NIL;
      while current_signal <> NIL do
        begin
          if base_signal = NIL then     { first time through the loop }
            begin
              current_base :=
                       find_base_of_signal_instance(current_signal^.instance);
              base_signal := current_base;
            end
          else
            current_base^.next :=
                       find_base_of_signal_instance(current_signal^.instance);

          { find the end of the PCS returned by find_base_of_signal... }

          while current_base^.next <> NIL do
            current_base := current_base^.next;

          current_signal := current_signal^.next;
        end { while } ;
    end;

  find_base_of_PCS := base_signal;

  if debug_2 then
    begin
      write(outfile, 'Exited find_base_of_PCS with: ');
      dump_propertied_CS(outfile, base_signal);
    end;
end { find_base_of_PCS } ;




(**)     { ------- output synonyms for a signal def ------- }


procedure dump_synonyms_of_signal_definition(var f: textfile;
                                             sig_def: signal_definition_ptr);
  { output the synonyms for the signal definition }
  var
    direction: -1..1;
    i: bit_range;
    sub: subscript_ptr;
    SI: signal_instance_ptr;
    PCS: propertied_CS_ptr;
    ersatz_SD: signal_descriptor_ptr;


  function bit_exists: boolean;
    { return TRUE if the bit}
    var
      SI_list: signal_instance_ptr;
      found: boolean;
  begin
    found := FALSE;    SI_list := sig_def^.instances;
    while (SI_list <> NIL) AND NOT found do
      if find_subscript(i, SI_list^.bit_subscript) then found := TRUE
      else SI_list := SI_list^.next;
    bit_exists := found;
  end { bit_exists };


begin { dump_synonyms_of_signal_defintion }

  if left_to_right then direction := 1 else direction := -1;

  sub := NIL;  ersatz_SD := NIL;  new_signal_descriptor(ersatz_SD);

  with ersatz_SD^ do
    begin
      { next     }
           signal_name := sig_def^.signal^.name;
           polarity := sig_def^.polarity;
           low_asserted := (polarity = COMPLEMENTED);
      { replication_factor }
           kind := sig_def^.kind;
      { properties   }
           is_const := sig_def^.is_const;
    end;

  if ersatz_SD^.kind = VECTOR then
    begin
       i := sig_def^.left_index;
       repeat
         begin
           if bit_exists then add_subscript(sub, i);
           i := i + direction;
         end;
       until i = (sig_def^.right_index + direction);
    end;

  ersatz_SD^.bit_subscript := sub;

  ersatz_SD^.scope := sig_def^.scope;
  ersatz_SD^.net_id := sig_def^.net_id;
  SI := enter_signal_instance(ersatz_SD, sig_def);
  writestring(f, sig_def^.signal^.name);
  dump_bit_subscript(f, sub, sig_def^.kind);
  write(f, '=');
  PCS := NIL;

  if sig_def^.synonym_bits <> NIL then
    PCS := find_base_of_signal_instance(SI);
  dump_propertied_CS(f, PCS);

  if PCS = NIL then writeln(f,'<base>')
               else release_propertied_CS(PCS);

  release_complete_signal_descriptor(ersatz_SD);
end { dump_synonyms_of_signal_definition };


(**) { ---------------- New Synonym Code ----------------- }


function base_bit_properties(sig_def: signal_definition_ptr):
                                                     subscript_property_ptr;
  { for each bit of the given signal (SIG_DEF) that is base, return a
    list of its properties. }
  var
    direction: -1..1;                  { index increment direction }
    bit,                               { current bit in the subscript }
    times_through_loop: bit_range;     { # bits in the signal def }
    new_props: property_ptr;           { properties of current bit (BIT) }
    sub_prop: subscript_property_ptr;  { property list being created }
    augmentable: boolean;              { TRUE if we might augment it }
begin
  if debug_7 then
    begin
      writeln(outfile, 'Entering base_bit_properties:  def=');
      dump_signal_definition(outfile, sig_def);
    end;

  { This is a REAL simple implementation of this function, i.e. SLOW

    for each bit of the sig def do
      if the bit is not base do nothing
      if the bit is base do the following
      build the prop list for that bit
      can this new bit augment the last sub prop element
        if not append a new sub prop element
        if so compare new prop list with prop list on last sub prop element
          if =   add bit to last sub prop element
          if <>  append a new sub prop element
                                                                             }

  sub_prop := NIL;
  if sig_def^.kind = SINGLE then bit := -1
  else bit := sig_def^.left_index;

  times_through_loop := width_of_signal_definition(sig_def);

  if sig_def^.left_index > sig_def^.right_index then
    direction := -1
  else if sig_def^.left_index < sig_def^.right_index then
    direction := 1
  else
    direction := 0;

  while times_through_loop > 0 do
    begin
      if is_base_bit(sig_def, bit) then
        begin
          new_props := bit_properties_of_this_bit(sig_def^.properties, bit,
                                                  INHERIT_PROPERTIES);

          augmentable := FALSE;

          if sub_prop <> NIL then
            if bit = (sub_prop^.right_index + direction) then
              if compare_properties(new_props,
                                    sub_prop^.properties) = EQ then
                begin
                  augmentable := TRUE;

                  sub_prop^.right_index := sub_prop^.right_index + direction;

                  release_entire_property_list(new_props);
                end;

          if not augmentable then
            begin
              new_subscript_property(sub_prop);
              with sub_prop^ do
                begin
                  left_index  := bit;
                  right_index := bit;
                  properties  := new_props;
                end;
            end;
        end;

      bit := bit + direction;

      times_through_loop := times_through_loop - 1;
    end { while ... } ;

  if (sig_def^.kind = SINGLE) and (sub_prop <> NIL) then
    begin
      sub_prop^.left_index := -1;
      sub_prop^.right_index := -1;
    end;

  base_bit_properties := sub_prop;

  if debug_7 then
    begin
      writeln(outfile, 'Exiting base_bit_properties: def=');
      dump_signal_definition(outfile, sig_def);
      writeln(outfile, 'props=');
      dump_subscript_property_list(outfile, sub_prop);
    end;
end { base_bit_properties } ;


function de_NC_return_actual(list_of_actuals:actual_list_ptr):actual_list_ptr;
{ -- Takes a list of actual parameters if the first actual contains
     NC signals a new actual with the NC signals replaced (if possible)
     with non-NC names from the left-most actual that has a
     non-NC in that position. If the left most actual has no NC signals
     then NIL is returned.                                                --}
     
  var
    first_actual,
    next_actual: actual_list_ptr;
    head_of_next_return_PCS,
    next_return_PCS,
    return_PCS: propertied_CS_ptr;
    actual_has_NC_signals: boolean;
    SD1,
    SD2,
    temp_SD: signal_descriptor_ptr;
    PCS1,                           { signal that is synonymed to PCS2 }
    PCS2,                           { signal that is synonymed to PCS1 }
    PCS_ptr1,                       { Steps over the elements of PCS1      }
    PCS_ptr2: PCS_pointer;          { Steps over the elements of PCS2      }
    width_of_SD1,
    width_of_SD2: bit_range;
   


  procedure build_SD_from_SI(var SD: signal_descriptor_ptr;
                                 SI: signal_instance_ptr);
    { construct a signal descriptor from the given signal instance }
    var
      sig_def: signal_definition_ptr;
  begin
    if SD = NIL then new_signal_descriptor(SD);

    sig_def := SI^.defined_by;
    with SD^ do 
      begin
        next := NIL;
        signal_name        := sig_def^.signal^.name;
        polarity           := sig_def^.polarity;
        low_asserted       := SI^.low_asserted;
        scope              := sig_def^.scope;
        replication_factor := 1;
        kind               := sig_def^.kind;
        bit_subscript      := SI^.bit_subscript;
        properties         := NIL;
        net_id             := sig_def^.net_id;
        is_const           := sig_def^.is_const;
      end;
  end { build_SD_from_SI } ;

  procedure build_shortened_SD(new_SD, SD: signal_descriptor_ptr;
                               new_width: bit_range);
    { construct a shortened base descriptor }
  begin
    { !!!! Be VERY careful. This routine is only called with temp_SD, and
      is the only routine that builds temp_SD. The subscript of temp_SD is
      always a copy of something and so can be released. Be sure to change
      this routine if this convention is changed!!!!!!!!!!!!!!!!             }
      
    if new_SD^.bit_subscript <> NIL then { this only works with temp_SD }
      release_subscript(new_SD^.bit_subscript);
    new_SD^ := SD^;  
    new_SD^.bit_subscript := leading_n_bits_of_subscript(new_width,
                                                         SD^.bit_subscript);
  end { build_shortened_SD };


  procedure shorten_SD(SD: signal_descriptor_ptr; new_width: bit_range);
    { shorten the signal descriptor }
    var
      subs: subscript_ptr;
  begin
    subs := trailing_n_bits_of_subscript(new_width, SD^.bit_subscript);
  {  release_subscript(SD^.bit_subscript);  }
    SD^.bit_subscript := subs;
  end { shorten_SD };


  procedure init_de_NC_return_actual;
    { initialize }
  begin
    actual_has_NC_signals := NOT no_NC_PCS(list_of_actuals^.signal);
    if actual_has_NC_signals then
      begin
        SD1 := NIL;     SD2 := NIL;    temp_SD := NIL;

        new_signal_descriptor(SD1);
        new_signal_descriptor(SD2);
        new_signal_descriptor(temp_SD);

        first_actual := list_of_actuals;
        next_actual := first_actual^.next; 

        return_PCS := copy_PCS(first_actual^.signal);


        next_return_PCS := NIL;    head_of_next_return_PCS := NIL;
      end
    else
      begin  return_PCS := NIL; next_actual := NIL; end;
  end { init_de_NC_return_actual } ;


  procedure de_init_de_NC_return_actual;
  begin
    if return_PCS <> NIL then
      begin
{
        release_complete_signal_descriptor(temp_SD);
        release_complete_signal_descriptor(SD1);
        release_complete_signal_descriptor(SD2);
}
        if temp_SD^.bit_subscript <> NIL then { this only works with temp_SD }
          release_subscript(temp_SD^.bit_subscript);
        release_signal_descriptor(temp_SD);
        release_signal_descriptor(SD1);
        release_signal_descriptor(SD2);
      end;
  end {  de_init_de_NC_return_actual } ;


  procedure init_next_pass;
    { initialize }
  begin
    build_SD_from_SI(SD1, return_PCS^.instance);
    init_PCS_pointer(PCS_ptr1, return_PCS);
    PCS1 := PCS_ptr1;
   
    build_SD_from_SI(SD2, next_actual^.signal^.instance);
    init_PCS_pointer(PCS_ptr2, next_actual^.signal);
    PCS2 := PCS_ptr2;
  end { init_next_pass } ;


  procedure copy_next_return_PCS_to_return_PCS;
    { copy a PCS }
    var
      last_PCS: propertied_CS_ptr;
  begin
    while return_PCS <> NIL do
      begin
        last_PCS := return_PCS;
        return_PCS := return_PCS^.next;
        release_propertied_CS(last_PCS);
      end;

    return_PCS := head_of_next_return_PCS;
    next_return_PCS := NIL;
  end { copy_next_return_PCS_to_return_PCS } ;
  

  procedure next_propertied_CS(var PCS: propertied_CS_ptr);
    { add a PCS to the list }
  begin
    if PCS = NIL then
      begin  new_propertied_CS(PCS);   head_of_next_return_PCS := PCS; end
    else
      begin  new_propertied_CS(PCS^.next);  PCS := PCS^.next; end;
  end { next_propertied_CS } ;
  

  procedure next_SD(var SD: signal_descriptor_ptr; var cp:PCS_pointer);
    { create a new signal descriptor }
  begin
    if (cp.PCS = NIL) then
      assert(74 {oops!})
    else
      begin
        advance_PCS_pointer(cp);
        if (cp.PCS <> NIL) then  build_SD_from_SI(SD, cp.PCS^.instance);
      end;
  end { next_SD } ;


  procedure head_of_loop_debug_dump;
    { dump some debug info }
  begin
    writeln(outfile, 'Head of loop debug dump');
    write(outfile, '  (new) SD1:');
    dump_signal_descriptor(outfile,SD1);
    write(outfile, '  (new) SD2:');
    dump_signal_descriptor(outfile,SD2);
    writeln(outfile, '  Return_PCS:');
    dump_propertied_CS(outfile, Return_PCS);
    writeln(outfile, '  head_of_next_return_PCS:');
    dump_propertied_CS(outfile, head_of_next_return_PCS);
    writeln(outfile, '  next_return_PCS:');
    dump_propertied_CS(outfile, next_return_PCS);
  end { head_of_loop_debug_dump } ;


  procedure inside_loop_debug_dump;
    { dump some debug info }
  begin
    writeln(outfile, 'Inside loop debug dump');

    writeln(outfile, '  Return_PCS:');
    dump_propertied_CS(outfile, Return_PCS);
    writeln(outfile, '  head_of_next_return_PCS:');
    dump_propertied_CS(outfile, head_of_next_return_PCS);
    writeln(outfile, '  next_return_PCS:');
    dump_propertied_CS(outfile, next_return_PCS);

    write(outfile, '  rest of first PCS (PCS_ptr1.PCS): ');
    dump_PCS(outfile, PCS_ptr1.PCS);
    writeln(outfile, '    PCS_ptr1.replication_count: ',
                     PCS_ptr1.replication_count:1);
      
    write(outfile, '  rest of second PCS (PCS_ptr2.PCS): ');
    dump_PCS(outfile, PCS_ptr2.PCS);
    writeln(outfile, '    PCS_ptr1.replication_count: ',
                     PCS_ptr1.replication_count:1);
  end { inside_loop_debug_dump } ;


  procedure end_of_loop_debug_dump;
    { dump some debug info }
  begin
    writeln(outfile, 'End of loop debug dump');
    writeln(outfile, '  Return_PCS:');
    dump_propertied_CS(outfile, Return_PCS);
    writeln(outfile, '  head_of_next_return_PCS:');
    dump_propertied_CS(outfile, head_of_next_return_PCS);
    writeln(outfile, '  next_return_PCS:');
    dump_propertied_CS(outfile, next_return_PCS);
  end { end_of_loop_debug_dump } ;


begin { de_NC_return_actual }
  if debug_5 then
    begin
      writeln(outfile, 'Entered de_NC_return_actual with:');
      dump_actual_list(outfile, list_of_actuals);
    end;  

  if (list_of_actuals = NIL)  then
    assert(144 {oops!})
  else
    begin
      init_de_NC_return_actual;    { must build initial SDs }
      
      while (next_actual <> NIL) AND actual_has_NC_signals do
        begin
          init_next_pass;

          (*if debug_5 then head_of_loop_debug_dump;*)
  
          while (PCS_ptr1.PCS <> NIL) AND (PCS_ptr2.PCS <> NIL) do
            begin
      
              width_of_SD1 := width_of_signal_descriptor(SD1);
              width_of_SD2 := width_of_signal_descriptor(SD2);
        
              if (width_of_SD1 <= 0) OR (width_of_SD2 <= 0) then
                begin
                  assert(75 { signal with undefined width passed in });
                  dump_PCS(CmpLog, PCS_ptr1.PCS);
                  dump_PCS(CmpLog, PCS_ptr2.PCS);
                  PCS_ptr1.PCS := NIL;
                  PCS_ptr2.PCS := NIL;
                end
              else
                begin
                  if width_of_SD1 > width_of_SD2 then
                    begin { GT }
                      if is_NC_signal(SD1^.signal_name) then
                        begin
                          next_propertied_CS(next_return_PCS);
                          next_return_PCS^.instance :=
                                                enter_signal_instance(SD2,
                                        PCS_ptr2.PCS^.instance^.defined_by);
                        end
                      else
                        begin
                          build_shortened_SD(temp_SD, SD1, width_of_SD2);
                          next_propertied_CS(next_return_PCS);
                          next_return_PCS^.instance :=
                                                enter_signal_instance(temp_SD,
                                        PCS_ptr1.PCS^.instance^.defined_by);
                        end;
                      shorten_SD(SD1, width_of_SD1 - width_of_SD2);
                      next_SD(SD2, PCS_ptr2);
                    end { GT }

                  else if width_of_SD1 < width_of_SD2 then
                    begin { LT }
                      if NOT is_NC_signal(SD1^.signal_name) then
                        begin
                          next_propertied_CS(next_return_PCS);
                          next_return_PCS^.instance :=
                                                enter_signal_instance(SD1,
                                        PCS_ptr1.PCS^.instance^.defined_by);
                        end
                      else
                        begin
                          build_shortened_SD(temp_SD, SD2, width_of_SD1);
                          next_propertied_CS(next_return_PCS);
                          next_return_PCS^.instance :=
                                                enter_signal_instance(temp_SD,
                                        PCS_ptr2.PCS^.instance^.defined_by);
                        end;
                      shorten_SD(SD2, width_of_SD2 - width_of_SD1);
                      next_SD(SD1, PCS_ptr1);
                    end { LT }

                  else
                    begin { EQ }
                      if is_NC_signal(SD2^.signal_name) then
                        begin
                          next_propertied_CS(next_return_PCS);
                          next_return_PCS^.instance :=
                                                enter_signal_instance(SD1,
                                        PCS_ptr1.PCS^.instance^.defined_by);
                        end
                      else
                        begin
                          next_propertied_CS(next_return_PCS);
                          next_return_PCS^.instance :=
                                                enter_signal_instance(SD2,
                                        PCS_ptr2.PCS^.instance^.defined_by);
                        end;
                      next_SD(SD1, PCS_ptr1);
                      next_SD(SD2, PCS_ptr2);
                    end { EQ };
        
                (*if debug_5 then inside_loop_debug_dump;*)
              end {else ... (widths are defined) };
            end {while there are still more instances ... };
  
          { this is not needed since it is detected in synonym_actuals }

          next_actual := next_actual^.next;

          (*if debug_5 then end_of_loop_debug_dump;*)

          copy_next_return_PCS_to_return_PCS;

          actual_has_NC_signals := NOT no_NC_PCS(return_PCS);
        end {while there are still more actuals and some NC signals left } ;
      de_init_de_NC_return_actual;
    end { else ... ( args are OK ) };
  
  if return_PCS = NIL then de_NC_return_actual := NIL
  else
    begin
      next_actual := NIL;
      new_actual_list(next_actual);
      next_actual^.signal := return_PCS;
      de_NC_return_actual := next_actual;
    end;

  if debug_5 then
    begin
      writeln(outfile, 'Exited de_NC_return_actual with: ');
      if return_PCS = NIL then writeln(outfile, '<NIL>')
                          else dump_PCS(outfile, return_PCS);
    end;
end { de_NC_return_actual } ;


procedure synonym_actuals(list_of_actuals:actual_list_ptr);
  { Takes a list of actual parameters and synonyms them (pair wise) bit
    by bit. In addition the def properties are moved to the most base
    def for all the new synonyms created. }
  var
    first_actual,
    next_actual: actual_list_ptr;
    BD1,
    BD2,
    temp_BD: base_descriptor_ptr;
    PCS1,                           { actual to be synonymed to PCS2 }
    PCS2,                           { acutal to be synonymed to PCS1 }
    PCS_ptr1,                       { Steps over the elements of PCS1      }
    PCS_ptr2: PCS_pointer;          { Steps over the elements of PCS2      }
    width_of_BD1,
    width_of_BD2: bit_range;
   

  procedure build_BD_from_SI(var BD: base_descriptor_ptr;
                                 SI: signal_instance_ptr);
    { construct a base descriptor from the signal instance }
  begin
    BD^.instance := SI;
    case SI^.defined_by^.kind of
      SINGLE:    BD^.width := 1;
      VECTOR:    BD^.width := size_of_signal_instance(SI);
      UNDEFINED: BD^.width := 0;
    end { case };
    BD^.offset := 0;
  end { build_BD_from_SI } ;


  procedure build_shortened_BD(new_BD, BD: base_descriptor_ptr;
                               new_width: bit_range);
    { construct a shortened base descriptor }
  begin
    new_BD^ := BD^;     new_BD^.width := new_width;
  end { build_shortened_BD };


  procedure shorten_BD(BD: base_descriptor_ptr; new_width: bit_range);
    { shorten the base descriptor }
  begin
    with BD^ do
      begin
        offset := offset + width - new_width;
        width := new_width;
      end;
  end { shorten_BD };


  procedure init_synonym_actuals;
    { initialize }
  begin
    temp_BD := NIL;   new_base_descriptor(temp_BD);
    BD1     := NIL;   new_base_descriptor(BD1);
    BD2     := NIL;   new_base_descriptor(BD2);

    first_actual := list_of_actuals;     next_actual := first_actual^.next;
  end { init_synonym_actuals } ;


  procedure de_init_synonym_actuals;   
  begin
    release_base_descriptor(temp_BD);
    release_base_descriptor(BD1);
    release_base_descriptor(BD2);
  end { de_init_synonym_actuals } ;


  procedure init_next_pass;
    { initialize }
  begin
    build_BD_from_SI(BD1, first_actual^.signal^.instance);
    init_PCS_pointer(PCS_ptr1, first_actual^.signal);
    PCS1 := PCS_ptr1;
   
    build_BD_from_SI(BD2, next_actual^.signal^.instance);
    init_PCS_pointer(PCS_ptr2, next_actual^.signal);
    PCS2 := PCS_ptr2;
  end { init_next_pass } ;


  procedure next_BD(var BD:base_descriptor_ptr; var cp:PCS_pointer);
    { create a new base descriptor }
  begin
    if (cp.PCS = NIL) then
      assert(74 {oops!})
    else
      begin
        advance_PCS_pointer(cp);
        if (cp.PCS <> NIL) then  build_BD_from_SI(BD, cp.PCS^.instance);
      end;
  end { next_BD } ;


  procedure debug_dump;
    { report some debug info }
  begin
    write(outfile, '  (new) BD1:');
    dump_base_descriptor(outfile,BD1);
    write(outfile, '  (new) BD2:');
    dump_base_descriptor(outfile,BD2);
    writeln(outfile, '  Same stuff next time round');
    write(outfile, '   rest of first PCS (PCS_ptr1.PCS): ');
    dump_PCS(outfile, PCS_ptr1.PCS);
    writeln(outfile, '     PCS_ptr1.replication_count: ',
                     PCS_ptr1.replication_count:1);
      
    write(outfile, '   rest of second PCS (PCS_ptr2.PCS): ');
    dump_PCS(outfile, PCS_ptr2.PCS);
    writeln(outfile, '     PCS_ptr1.replication_count: ',
                     PCS_ptr1.replication_count:1);
  end { debug_dump } ;


  procedure report_length_error;
    { report an error in the mutual widths of the signals being synonymed }
  begin
    error(159 {the two input args are of unequal length});
    error_dump_propertied_CS(PCS1.PCS);
    error_dump_indent(indent);
    error_dump_alpha('Width=          ');
    error_dump_integer(width_of_PCS(PCS1.PCS));
    error_dump_CRLF;

    error_dump_propertied_CS(PCS2.PCS);
    error_dump_indent(indent);
    error_dump_alpha('Width=          ');
    error_dump_integer(width_of_PCS(PCS2.PCS));
    error_dump_CRLF;

    error_dump_macro_def(current_mtree_node^.macro);

    if debug_34 then
      begin
        write(CmpLog, ' PCS1: ');
        dump_PCS(CmpLog, PCS1.PCS);
        writeln(CmpLog, ' Rep count: ', PCS1.replication_count:1);

        write(CmpLog, ' PCS2: ');
        dump_PCS(CmpLog, PCS2.PCS);
        writeln(CmpLog, ' Rep count: ', PCS2.replication_count:1);
      end;
  end { report_length_error } ;


{--- assumes that all parameters in the arguments are resolved out ---}

begin { synonym_actuals }
  if debug_34 then
    begin
      writeln(outfile, 'Entered synonym_actuals with: ');
      dump_actual_list(outfile, list_of_actuals);
    end;  

  if (list_of_actuals = NIL)  then
    assert(141 {oops!})
  else
    begin
      init_synonym_actuals;    { allocate scratch BDs }
      
      while next_actual <> NIL do
        begin
          init_next_pass;  { build PCS's and build initial BDs for this pass }
  
          while (PCS_ptr1.PCS <> NIL) AND (PCS_ptr2.PCS <> NIL) do
            begin
              width_of_BD1 := BD1^.width;   width_of_BD2 := BD2^.width;
        
              if (width_of_BD1 <= 0) OR (width_of_BD2 <= 0) then
                begin
                  assert(75  {signal with undefined width passed in});
                  dump_PCS(CmpLog, PCS_ptr1.PCS);
                  dump_PCS(CmpLog, PCS_ptr2.PCS);
                  PCS_ptr1.PCS := NIL; PCS_ptr2.PCS := NIL;
                end
              else
                begin
                  if width_of_BD1 > width_of_BD2 then
                    begin { GT }
		      build_shortened_BD(temp_BD, BD1, width_of_BD2);
		      synonym_base_descriptors(temp_BD, BD2);
                      shorten_BD(BD1, width_of_BD1 - width_of_BD2);
                      next_BD(BD2, PCS_ptr2);
                    end { GT }
                  else if width_of_BD1 < width_of_BD2 then
                    begin { LT }
		      build_shortened_BD(temp_BD, BD2, width_of_BD1);
		      synonym_base_descriptors(BD1, temp_BD);
                      shorten_BD(BD2, width_of_BD2 - width_of_BD1);
                      next_BD(BD1, PCS_ptr1);
                    end { LT }
                  else
                    begin { EQ }
		      synonym_base_descriptors(BD1, BD2);
                      next_BD(BD1, PCS_ptr1);
                      next_BD(BD2, PCS_ptr2);
                    end { EQ };
        
                if debug_34 then debug_dump;
              end { else ... (widths are defined) };
          end { while next pair are not synonymed } ;
  
          if (PCS_ptr1.PCS <> NIL) OR (PCS_ptr2.PCS <> NIL) then
            report_length_error;

          next_actual := next_actual^.next;
        end {while there are still more actuals to synonym } ;
    end { else ... ( args are OK ) };

  de_init_synonym_actuals;   

  if debug_34 then  writeln(outfile, 'Exited synonym_actuals');
end { synonym_actuals } ;


function create_synonym(list_of_actuals: actual_list_ptr): actual_list_ptr;
  { create synonyms between all of the actual signals in the given list.
    Return an actual containing no NC signals.  If the first actual in the
    list has no NC signals as is, return NIL. }
  var
    actuals: actual_list_ptr;
begin
  if debug_34 then writeln(outfile, 'Entered create_synonym');

  actuals := list_of_actuals;
  synonym_actuals(actuals);

  create_synonym := de_NC_return_actual(actuals);

  if debug_34 then writeln(outfile, 'Exited create_synonym');
end { create_synonym } ;


(**)     { ------- property gathering routines ------- }


function find_base_of_bit_of_signal_instance(SI:  signal_instance_ptr;
                                            which_bit: bit_range;
                                            var base_SI:  signal_instance_ptr;
                                            var base_bit: bit_range): boolean;
  { return the base instance (BASE_SI) and corresponding bit (BASE_BIT) of
    the given instance (SI) adn its bit (WHICH_BIT).  The bit values are
    "ordinals" in the sense that the first bit in the subscript is referred
    to as bit 1. }
const
   MAX_TRIES = 5000; { loop counter for safety in case of non_convergence }
var
  passes: natural_number;
  found_base,
  found_bit:  boolean;
  next_SI: signal_instance_ptr;
  base_BD: base_descriptor_ptr;
  offset_to_next_bit,
  bit_of_interest: bit_range;
begin
  if debug_7 then
    begin
      writeln(outfile, 'Entered find_base_of_bit_of_signal_instance: ');
      write(outfile, '  SI: ');
      dump_signal_instance(outfile, SI);
      writeln(outfile, '  which_bit: ', which_bit: 1);
    end;

  base_BD := NIL;
  new_base_descriptor(base_BD);

  with base_BD^ do
    begin  instance := SI;  width := 1;  offset := which_bit-1;  end;

  found_base := FALSE;  found_bit := FALSE;  passes := 0;
  while (NOT found_base) AND (passes < MAX_TRIES) do
    begin

      { if THE bit of BDs signal instance is not base then
        set base_BD to the more base signal of that bit }

      passes := passes + 1;
      with base_BD^ do
        case instance^.defined_by^.kind of
          SINGLE: begin  bit_of_interest := 1;  found_bit := TRUE;  end;
          VECTOR:
            begin
              bit_of_interest := 1 + offset;
              found_bit := nth_bit_of_signal_instance(bit_of_interest,
                                                      instance);
              if NOT found_bit then
                begin
                  assert(140 { bit must occur on base_BD });
                  writeln(CmpLog,'	ordinal bit number', bit_of_interest:1);
                  write(CmpLog,'	base_BD: ');
                  dump_base_descriptor(CmpLog, base_BD);
                  dump_tree_information(CmpLog, debug_dump_synonyms);
                end;
            end;
          UNDEFINED: assert(139 { bit must occur on base_BD });
        end { case } ;

      if found_bit then
        begin
          { find next more base SI (and right bit) return TRUE if was
            already base }

          found_base :=  get_next_base_bit(base_BD^.instance,
                                           bit_of_interest,
                                           next_SI, offset_to_next_bit);
          if NOT found_base then
            begin
              base_BD^.instance := next_SI;
              base_BD^.offset := offset_to_next_bit;
            end;
        end;
    end { while have not found base and not in trouble ... };

  if passes >= MAX_TRIES then
    begin
      assert(138 { no reason not to find base of a single bit signal});
      find_base_of_bit_of_signal_instance := FALSE;
      base_SI := NIL; base_bit := 0;
    end
  else
    begin
      find_base_of_bit_of_signal_instance := TRUE;

      base_SI := base_BD^.instance;
      base_bit := base_BD^.offset + 1;
      if base_SI^.defined_by^.kind <> SINGLE then
        found_bit := nth_bit_subscript(base_bit, base_SI^.bit_subscript);
    end;

  if debug_7 then
    begin
      writeln(outfile, 'Exited find_base_of_bit_of_signal_instance');
      write(outfile, '  base_SI: ');
      dump_signal_instance(outfile, base_SI);
      writeln(outfile, '  base_bit: ', base_bit:1);
    end;
end { find_base_of_bit_of_signal_instance };


function get_all_properties(SI: signal_instance_ptr;
                            left, right: bit_range;
                            just_pin_props: boolean): subscript_property_ptr;
  { get all properties from the given signal instance.  Return a
    non-overlapping subscript property list.  The bits should be referenced
    to the vector supplied (LEFT and RIGHT).  If JUST_PIN_PROPS, return
    only pin props. }
  var
    base_SI: signal_instance_ptr;       { instance which is base }
    base_bit: bit_range;                { base instance's bit }
    direction: -1..1;                   { subscript increment offset }
    current_bit,                        { current "virtual" bit }
    i,                                  { current index bit into instance }
    next_bit_of_SI: bit_range;          { next index bit into instance }
    end_SPP,                            { end of the subscript property }
    return_SPP: subscript_property_ptr; { properties to be returned }
    next_bits_props: property_ptr;      { properties of the bit }


  procedure remove_non_pin_properties(var prop_list: property_ptr);
    { remove all properties form the given list that do not have the
      INHERIT_PIN attribute. }
    var
      next_prop,                       { next property in source list }
      source_prop,                     { current source property }
      return_props: property_ptr;      { property list to be returned }
  begin
    source_prop := prop_list;  return_props := NIL;
    while source_prop <> NIL do
      begin
        next_prop := source_prop^.next;

        if INHERIT_PIN IN source_prop^.name^.kind then
          begin
            source_prop^.next := return_props;
            return_props := source_prop;
          end
        else
          release_property(source_prop);

        source_prop := next_prop;
      end;

    prop_list := return_props;
  end { remove_non_pin_properties } ;


  procedure append_to_subscript_property_list(new_props: property_ptr;
                                              bit: bit_range);
    { append the current property list (NEW_PROPS) to the global property list
      (return_SPP and end_SPP).  If the current subrange cannot be augmented,
      build a new element for the list. }
    var
      augmentable: boolean;              { TRUE if we might augment it }
  begin
    augmentable := FALSE;

    if (end_SPP <> NIL) and (new_props <> NIL) then
      if bit = (end_SPP^.right_index + direction) then
        if compare_properties(new_props,
                              end_SPP^.properties) = EQ then
            begin
              augmentable := TRUE;

              end_SPP^.right_index := end_SPP^.right_index + direction;

              release_entire_property_list(new_props);
            end;

      if not augmentable and (new_props <> NIL) then
        begin
          if return_SPP = NIL then
            begin
              new_subscript_property(return_SPP);
              end_SPP := return_SPP;
            end
          else
            begin
              new_subscript_property(end_SPP^.next);
              end_SPP := end_SPP^.next;
            end;

          with end_SPP^ do
            begin
              left_index  := bit;
              right_index := bit;
              properties  := new_props;
            end;
        end;

  end { append_to_subscript_property_list } ;
    

begin { get_all_properties }
  if debug_7 then writeln(outfile, 'Entering get_all_properties');

  { for each bit of SI find its base bit
      for that base bit get its properties.
   
        if the property list is the same as the last prop list
          augment the subscript
        else
          add a new subscript property record }

  return_SPP := NIL;
  end_SPP := NIL;

  if left > right then direction := -1 else direction := 1;
  current_bit := left;

  for i := 1 to size_of_signal_instance(SI) do
    begin
      next_bit_of_SI := i;

      if NOT find_base_of_bit_of_signal_instance(SI, next_bit_of_SI,
                                                 base_SI, base_bit) then
        begin assert(137 {oops!}); end
      else
        begin
          next_bits_props := 
                   bit_properties_of_this_bit(base_SI^.defined_by^.properties,
                                              base_bit, INHERIT_PROPERTIES);

          if just_pin_props then
            remove_non_pin_properties(next_bits_props);

          append_to_subscript_property_list(next_bits_props, current_bit);
        end;

      if left <> right then
        current_bit := current_bit + direction;
    end { for every bit of SI ... };

  (****** this code causes calling routine to fail which depends upon the
          subscript being correct  ***************
  if (SI^.defined_by^.kind = SINGLE) and (return_SPP <> NIL) then
    begin
      return_SPP^.left_index := -1;  return_SPP^.right_index := -1;
    end;
   ***************************)

  get_all_properties := return_SPP;

  if debug_7 then
    begin
      writeln(outfile, 'Exiting get_all_properties: props=');
      dump_subscript_property_list(outfile, return_SPP);
    end;
end { get_all_properties } ;


(**)     { ------- gather properties from a subscript property ------- }


function gather_properties_from_subscript(prop_list: subscript_property_ptr;
                                          left, right: bit_range):
                                                       subscript_property_ptr;
  { gather INHERIT_PIN properties from the given subscript property list.
    Return a non-overlapping subscript property list.  The bits should be
    referenced to the vector supplied (LEFT and RIGHT).  The vector also
    specifies which bits of the subscript property to gather. }
  var
    done: boolean;                      { TRUE if property gathering done }
    direction: -1..1;                   { subscript increment offset }
    current_bit: bit_range;             { current "virtual" bit }
    end_SPP,                            { end of the subscript property }
    return_SPP: subscript_property_ptr; { properties to be returned }
    bits_props: property_ptr;           { properties of the bit }


  procedure append_to_subscript_property_list(new_props: property_ptr;
                                              bit: bit_range);
    { append the current property list (NEW_PROPS) to the global property list
      (return_SPP and end_SPP).  If the current subrange cannot be augmented,
      build a new element for the list. }
    var
      augmentable: boolean;              { TRUE if we might augment it }
  begin
    augmentable := FALSE;

    if (end_SPP <> NIL) and (new_props <> NIL) then
      if bit = (end_SPP^.right_index + direction) then
        if compare_properties(new_props,
                              end_SPP^.properties) = EQ then
            begin
              augmentable := TRUE;

              end_SPP^.right_index := end_SPP^.right_index + direction;

              release_entire_property_list(new_props);
            end;

      if not augmentable and (new_props <> NIL) then
        begin
          if return_SPP = NIL then
            begin
              new_subscript_property(return_SPP);
              end_SPP := return_SPP;
            end
          else
            begin
              new_subscript_property(end_SPP^.next);
              end_SPP := end_SPP^.next;
            end;

          with end_SPP^ do
            begin
              left_index  := bit;
              right_index := bit;
              properties  := new_props;
            end;
        end;

  end { append_to_subscript_property_list } ;
    

begin { gather_properties_from_subscript }
  if debug_14 then
    writeln(outfile, 'Entering gather_properties_from_subscript: L=',
                     left:1, '; R=', right:1);

  { for each bit of SI find its base bit
      for that base bit get its properties.
   
        if the property list is the same as the last prop list
          augment the subscript
        else
          add a new subscript property record }

  return_SPP := NIL;
  end_SPP := NIL;

  if left > right then direction := -1 else direction := 1;

  current_bit := left;
  repeat
    bits_props := all_props_of_this_subscript(prop_list, current_bit);

    append_to_subscript_property_list(bits_props, current_bit);

    done := (current_bit = right);

    if not done then current_bit := current_bit + direction;
  until done;

  if (left = -1) and (return_SPP <> NIL) then
    begin
      return_SPP^.left_index := -1;  return_SPP^.right_index := -1;
    end;

  gather_properties_from_subscript := return_SPP;

  if debug_14 then
    begin
      writeln(outfile, 'Exiting gather_properties_from_subscript: props=');
      dump_subscript_property_list(outfile, return_SPP);
    end;
end { gather_properties_from_subscript } ;
