(**)     { ------- Machine dependent procedures ------- }


#include "exception.p"


(**)     { ------- FORWARD procedure declarations ------- }


function rewrite_file(var txtfil: textfile; filename: xtring; 
                      logical: alpha): boolean;
forward;

function rewrite_ds_file(var txtfil: textfile; filename: xtring; 
                      logical: alpha): boolean;
forward;

procedure remove_logical_file(fname: alpha);
forward;

function close_parse_file(which: parse_file_type): boolean;
forward;

procedure close_file(var f: textfile; logical: alpha; file_name: xtring);
forward;



procedure post_compile_time(var f: textfile; file_kind: file_kind_type);
forward;

procedure exec_time(var last_elapsed_time, last_CPU_time: longint;
                    just_delta: boolean);
forward;

function reset_file(filename: xtring; which: parse_file_type): boolean;
forward;

function reset_ds_file(filename: xtring; which: parse_file_type): boolean;
forward;


procedure init_time_and_date(var current_elapsed_time,
                                 current_CPU_time: longint;
                             var current_date: time_string);
forward;


procedure init_time(var current_elapsed_time, current_CPU_time: longint);
forward;

procedure enter_expandable_id(id: name_ptr);
forward;

procedure report_expandable_id_to_ds(id: name_ptr);
forward;

procedure log_property_use(id: name_ptr);
forward;

procedure init_numbered_dictionary(var dic: numbered_dictionary; 
				   token_kind: numbered_token_type);
forward;

function enter_name(name: alpha): name_ptr;
forward;

function make_and_enter_string(name: alpha): xtring;
forward;

function enter_and_release_string(str: xtring): xtring;
forward;

function enter_string(str: xtring): xtring;
forward;

procedure assert(assertion_num: assert_range);
forward;

function alpha_length(name: alpha): id_range;
forward;

procedure error_dump_signal_descriptor(signal: signal_descriptor_ptr);

forward;

procedure new_identifier(var id: identifier_ptr);
forward;

procedure error_dump_current_parse_environment;
forward;

procedure init_error_info;
forward;

function find_property(prop_list: property_ptr;  name: name_ptr;
                       var property: property_ptr): boolean;
forward;

procedure insymbol;
forward;

function avl_compare(obj1, obj2: avl_object_ptr; 
                      kind: avl_type): compare_type;
forward;

procedure dump_text_macros(var f: textfile; list: text_macro_ptr);
forward;

procedure mark_dirty_for_pass_2(page: paged_schema_ptr);
forward;

procedure enter_critical_section;  forward;
procedure exit_critical_section;  forward;
procedure kill_self;  forward;


procedure error(error_num: error_range);  forward;


function select_module(drawing_name, context: xtring; check_for_prim: longint):
  macro_module_ptr;
  { Call er_select() via this, so that we ddon't get bougs warnings about
    using default value for SIZE. }
begin
  selecting_module := TRUE;
  select_module := er_select(drawing_name, context, check_for_prim);
  selecting_module := FALSE;
end;

procedure setup_signal_configuration_from_ds_module; forward;


{ Functions leveraged by expansion control module }

procedure error_without_parse_line(num: longint);  forward;
procedure assert_without_parse_line(num: longint);  forward;
function evaluate_boolean_expr(str: xtring): longint;  forward;
procedure error_dump_integer(int: longint);  forward;
procedure error_dump_string(str: xtring);  forward;
procedure error_dump_standard_indent;  forward;
procedure error_dump_current_context;  forward;
procedure error_dump_crlf;  forward;
procedure parse_error_notifier;  forward;

(**)     { ------- trivia ------- }


function min(a, b: longint): longint;
  { return the min }
begin
  if a < b then   min := a  else  min := b;
end { min } ;


function max(a, b: longint): longint;
  { return the max }
begin
  if a > b then   max := a  else  max := b;
end { max } ;


function et_property_attributes(kind: name_type_set): longint;
  { return the property attribute code for the set using the values
    declared for use with et (the linker). }
  var
    val: longint;  { value for return }
begin
  val := 0;
  if (PERMIT_BODY in kind) then val := val + ET_PERMIT_BODY;
  if (PERMIT_PIN in kind) then val := val + ET_PERMIT_PIN;
  if (PERMIT_SIGNAL in kind) then val := val + ET_PERMIT_SIGNAL;
  if (INHERIT_BODY in kind) then val := val + ET_INHERIT_BODY;
  if (INHERIT_PIN in kind) then val := val + ET_INHERIT_PIN;
  if (INHERIT_SIGNAL in kind) then val := val + ET_INHERIT_SIGNAL;
  if (IS_PARAMETER in kind) then val := val + ET_PARAMETER;
  if (IS_INT_PARAMETER in kind) then val := val + ET_INT_PARAMETER;
  if (DONT_OUTPUT in kind) then val := val + ET_FILTER;
  if (IS_ET_CONTROL in kind) then val := val + ET_CONTROL;
  et_property_attributes := val;
end { et_property_attributes } ;

  
(**)     { ------- I/O utilities ------- }


procedure dump_boolean(var f: textfile; val: boolean);
begin
  if val then write(f, 'T')
         else write(f, 'F');
end { dump_boolean };


function width_of_integer(i: longint): longint;
  { Returns the minimum number of places PASCAL uses to print i }
  var
    width: longint;     { width of the integer i in print positions }
begin
  width := 1;
  if i < 0 then
    begin  width := 2;  i := -1;  end;

  if i < 10 then  { ok as is }
  else if i < 100 then width := width + 1
  else if i < 1000 then width := width + 2
  else if i < 10000 then width := width + 3
  else if i < 100000 then width := width + 4
  else if i < 1000000 then width := width + 5
  else if i < 10000000 then width := width + 6
  else if i < 100000000 then width := width + 7
  else if i < 1000000000 then width := width + 8
  else width := width + 9;

  width_of_integer := width;
end { width_of_integer } ;


procedure dump_string(var f: textfile; str: xtring);
  { dump the given string (STR) to the given file (F) as is }
  var
    hack: string_hack;
begin
  if ord(str^[0]) > 0 then
    begin
      hack.i := ord(str) + 1;  write(f, hack.s^:ord(str^[0]));
    end;
end { dump_string } ;
    

procedure writestring(var f: textfile; str: Xtring);
  { write the given string (STR) to the given file (F) as is with quotes }
begin
  write(f, OUTPUT_QUOTE_CHAR);

  dump_string(f, str);

  write(f, OUTPUT_QUOTE_CHAR);
end { writestring } ;
    

procedure print_string(var f: textfile; str: xtring);
  { print the given string (STR) to the given file (F) }
  var
    i: string_range;
    len: string_range;
    hack: string_hack;
begin
  { NOTE: Illegal characters can ONLY occur in str^[1] -- used there
    to quickly signify an NC. Illegal chars in input files are filtered
    by insymbol. }
  len := ord(str^[0]);
  if (len > 0) and islegal[str^[1]] then                           
    begin
      hack.i := ord(str) + 1;  write(f, hack.s^:len);
    end
  else if (len > 1) then
    begin
      hack.i := ord(str) + 2;  write(f, hack.s^:len-1);
    end;
end { print_string } ;


procedure print_string_repeat_quotes(var f: textfile; str: xtring);
  { print the string (STR) to the given file (F) doubling up any quotes }
  var
    len, start: string_range;
    stop: integer;  { may overflow string_range by 1 }
    hack: string_hack;
    found_quote: boolean;
begin
  len := ord(str^[0]);  
  stop := 1;
  while stop <= len do
    begin
      start := stop;  found_quote := FALSE;
      while (stop <= len) and not found_quote do
	begin
	  found_quote := str^[stop] = OUTPUT_QUOTE_CHAR;
	  stop := stop + 1;
	end;
      hack.i := ord(str) + start;
      write(f, hack.s^:(stop - start));
      if found_quote then write(f, OUTPUT_QUOTE_CHAR:1);
    end;
end { print_string_repeat_quotes } ;


procedure print_string_with_quotes(var f: textfile; str: xtring);
  { write a string to the output file }
begin
  write(f, OUTPUT_QUOTE_CHAR);
  print_string_repeat_quotes(f, str);
  write(f, OUTPUT_QUOTE_CHAR);
end { print_string_with_quotes } ;
    

procedure print_alpha(var f: textfile; name: alpha);
  { print the given alpha (NAME) to the given file (F) }
begin
  write(f, name:alpha_length(name));
end { print_alpha } ;


procedure writealpha(var f: textfile; name: alpha);
  { write an identifier to the output file removing trailing blanks }
begin
  write(f, name:alpha_length(name));
end { writealpha } ;


(**)     { ------- statistics routines ------- }


procedure increment_heap_count(structure: heap_structures;
                               numbytes: natural_number);
  { advance count of the size of elements created on the heap }
begin
  with heap_usage[structure] do
    begin
      number := number + 1;
      size := size + numbytes;
    end;
end { increment_heap_count } ;


procedure report_heap_usage(var f: textfile);
  { report the sizes and number of the structures used }
  const
    struct_length = 19;          { output print width for structure names }
  var
    structure: heap_structures;  { structure to be reported }
    corrected_total,             { heap corrected total }
    total: natural_number;       { total number of bytes of all structures }
    number: natural_number;      { number of elements for current structure }
    size: natural_number;        { size (bytes) of current structure }
    diff_size: natural_number;   { difference between min heap size and elem }


  procedure report_free_list;
    { report the number of free elements in the free structure lists }
    var
      num: natural_number;          { number of free elements in a free list }

      { following vars used to count up elements of free lists }

      SDP: signal_descriptor_ptr;
      DEF: signal_definition_ptr;
      SIP: signal_instance_ptr;
      FP: freeptr;
      j: string_range;
      id: identifier_ptr;
      SP: subscript_ptr;
      PP: property_ptr;
      EP: environment_ptr;
      ILP: invoke_list_ptr;
      BP: basescript_ptr;
      BDP: base_descriptor_ptr;
      SDLP: signal_definition_list_ptr;
      FAP: formal_actual_ptr;
      SSP: simple_signal_ptr;
      SynSP: synonym_signal_ptr;
      SbP: subscript_property_ptr;
      BiP: bit_property_ptr;
      PCSP: propertied_CS_ptr;
      ALP: actual_list_ptr;
      NDP: net_descriptor_ptr;
      NTP: net_table_ptr;
  begin
    num := 0;
    case structure of
      HEAP_SIGNAL_DEFINITION:
        begin
          DEF := free_signal_definitions;
          while DEF <> NIL do
            begin  num := num + 1;  DEF := DEF^.next;  end;
        end;

      HEAP_SIGNAL_DEFINITION_LIST:
        begin
          SDLP := free_signal_definition_lists;
          while SDLP <> NIL do
            begin  num := num + 1;  SDLP := SDLP^.next;  end;
        end;

      HEAP_SIGNAL_INSTANCE:
        begin
          SIP := free_signal_instances;
          while SIP <> NIL do
            begin  num := num + 1;  SIP := SIP^.next;  end;
        end;

      HEAP_SIGNALDESCR:
        begin
          SDP := free_signal_descriptors;
          while SDP <> NIL do
            begin  num := num + 1;  SDP := SDP^.next;  end;
        end;

      HEAP_SIMPLE_SIGNAL:
        begin
          SSP := free_simple_signals;
          while SSP <> NIL do
            begin  num := num + 1;  SSP := SSP^.next;  end;
        end;

      HEAP_INVOKELIST:
        begin
          ILP := free_invoke_lists;
          while ILP <> NIL do
            begin  num := num + 1;  ILP := ILP^.next;  end;
        end;

      HEAP_FORMAL_ACTUAL_LIST:
        begin
          FAP := free_formal_actual_lists;
          while FAP <> NIL do
            begin  num := num + 1;  FAP := FAP^.next;  end;
        end;

      HEAP_FREEELEMENT:
        begin
          FP := free_pointers;
          while FP <> NIL do
            begin  num := num + 1;  FP := FP^.next;  end;
        end;

      HEAP_STRING:
        for j := 1 to 33 do
          begin
            FP := free_strings[j];
            while FP <> NIL do
              begin  num := num + 1;  FP := FP^.next;  end;
          end;

      HEAP_IDENTIFIER:
        begin
          id := free_identifiers;
          while id <> NIL do
            begin  num := num + 1;  id := id^.next;  end;
        end;

      HEAP_SUBSCRIPT:
        begin
          SP := free_subscripts;
          while SP <> NIL do
            begin  num := num + 1;  SP := SP^.next;  end;
        end;

      HEAP_PROPERTY:
        begin
          PP := free_properties;
          while PP <> NIL do
            begin  num := num + 1;  PP := PP^.next;  end;
        end;

      HEAP_ENVIRONMENT_ENTRY:
        begin
          EP := free_environments;
          while EP <> NIL do
            begin  num := num + 1;  EP := EP^.next;  end;
        end;

      HEAP_BASESCRIPT:
        begin
          BP := free_basescripts;
          while BP <> NIL do
            begin  num := num + 1;  BP := BP^.next;  end;
        end;

      HEAP_BASE_DESCRIPTOR:
        begin
          BDP := free_base_descriptors;
          while BDP <> NIL do
            begin  num := num + 1;  BDP := BDP^.next;  end;
        end;

      HEAP_SYNONYM_SIGNAL:
        begin
          SynSP := free_synonym_signals;
          while SynSP <> NIL do
            begin  num := num + 1;  SynSP := SynSP^.next;  end;
        end;

      HEAP_SUBSCRIPT_PROPERTY:
        begin
          SbP := free_subscript_properties;
          while SbP <> NIL do
            begin  num := num + 1;  SbP := SbP^.next;  end;
        end;

      HEAP_BIT_PROPERTY:
        begin
          BiP := free_bit_properties;
          while BiP <> NIL do
            begin  num := num + 1;  BiP := BiP^.next;  end;
        end;

      HEAP_PROPERTIED_CONCATSIG:
        begin
          PCSP := free_propertied_CSs;
          while PCSP <> NIL do
            begin  num := num + 1;  PCSP := PCSP^.next;  end;
        end;

      HEAP_ACTUAL_LIST:
        begin
          ALP := free_actual_lists;
          while ALP <> NIL do
            begin  num := num + 1;  ALP := ALP^.next;  end;
        end;

      HEAP_NET_DESCRIPTOR:
        begin
          NDP := free_net_descriptors;
          while NDP <> NIL do
            begin  num := num + 1;  NDP := NDP^.next;  end;
        end;

      HEAP_NET_TABLE:
        begin
          NTP := free_net_tables;
          while NTP <> NIL do
            begin  num := num + 1;  NTP := NTP^.next;  end;
        end;
    end { case } ;

    writeln(f, ' (', num:1, ')');
  end { report_free_list } ;


begin { report_heap_usage }
  if debug_20 then writeln(outfile, ' enter report_heap_usage');
  total := 0;
  corrected_total := 0;

  page(f);
  writeln(f, ' Heap Statistics');  writeln(f);
  writeln(f, ' # Elements  # Bytes  Structure type (# free)');

  for structure := succ(FIRST_HEAP_STRUCTURE) to pred(LAST_HEAP_STRUCTURE) do
    begin
      number := heap_usage[structure].number;
      size := heap_usage[structure].size;

      write(f, number:7);

      write(f, size:12);

      total := total + size;

      { calculate heap corrected total }

      if number > 0 then
        begin
          corrected_total := corrected_total + size + HEAP_OVERHEAD * number;

          diff_size := ((size+1) DIV number) MOD MINIMUM_HEAP_INCREMENT;
          if diff_size <> 0 then
            corrected_total := corrected_total +
                                (MINIMUM_HEAP_INCREMENT - diff_size) * number;
        end;

      case structure of
        HEAP_BINDINGSLIST:  writeln(f, 'Bindings list':struct_length);

        HEAP_INVOKELIST:    begin
                              write(f, 'Invoke list':struct_length);
                              report_free_list;
                            end;

        HEAP_SIGNALLIST:    writeln(f, 'Signal list':struct_length);

        HEAP_MACRODEF:      writeln(f, 'Macro def':struct_length);

        HEAP_MTREENODE:     writeln(f, 'Mtree node':struct_length);

        HEAP_IDENTIFIER:    begin
                              write(f, 'Identifier':struct_length);
                              report_free_list;
                            end;

        HEAP_FREEELEMENT:   begin
                              write(f, 'Free element':struct_length);
                              report_free_list;
                            end;

        HEAP_STRING:        begin
                              write(f, 'String':struct_length);
                              report_free_list;
                            end;

        HEAP_PROPERTY:      begin
                              write(f, 'Property':struct_length);
                              report_free_list;
                            end;

        HEAP_DIRECTORY:     writeln(f, 'Directory':struct_length);

        HEAP_MACROENTRY:    writeln(f, 'Macro entry':struct_length);

        HEAP_FILELIST:      writeln(f, 'File list':struct_length);

        HEAP_MACROVERSION:  writeln(f, 'Macro version':struct_length);

        HEAP_SIGNALDESCR:   begin
                              write(f, 'Signal descr':struct_length);
                              report_free_list;
                            end;

        HEAP_SIMPLE_SIGNAL: begin
                              write(f, 'Simple signal':struct_length);
                              report_free_list;
                            end;

        HEAP_SYNONYM_SIGNAL:
                            begin
                              write(f, 'Synonym signal':struct_length);
                              report_free_list;
                            end;

        HEAP_SUBSCRIPT:     begin
                              write(f, 'Subscript':struct_length);
                              report_free_list;
                            end;

        HEAP_SIGNAL_DEFINITION_LIST:
                            begin
                              write(f, 'Sig def list':struct_length);
                              report_free_list;
                            end;

        HEAP_SIGNALENTRY:   writeln(f, 'Signal entry':struct_length);

        HEAP_SIGNAL_DEFINITION:
                            begin
                              write(f, 'Signal def':struct_length);
                              report_free_list;
                            end;

        HEAP_SIGNAL_INSTANCE:
                            begin
                              write(f, 'Signal inst':struct_length);
                              report_free_list;
                            end;

        HEAP_FORMAL_ACTUAL_LIST:
                            begin
                              write(f, 'For/act list':struct_length);
                              report_free_list;
                            end;

        HEAP_CLEAR_TEXT_ACTUAL_LIST:
                            writeln(f, 'Clear text act':struct_length);

        HEAP_NAME_ENTRY:    writeln(f, 'Identifier names':struct_length);

        HEAP_ENVIRONMENT_ENTRY:
                            begin
                              write(f, 'Environment elem':struct_length);
                              report_free_list;
                            end;

        HEAP_HASH_STRINGS:  writeln(f, 'Hash Strings':struct_length);

        HEAP_ACTUAL_LIST:   begin
                              write(f, 'Actual lists':struct_length);
                              report_free_list;
                            end;

        HEAP_BASESCRIPT:    begin
                              write(f, 'Base subscripts':struct_length);
                              report_free_list;
                            end;

        HEAP_BASE_DESCRIPTOR:
                            begin
                              write(f, 'Base descriptors':struct_length);
                              report_free_list;
                            end;

        HEAP_SUBSCRIPT_PROPERTY:
                            begin
                              write(f, 'Subscr props':struct_length);
                              report_free_list;
                            end;

        HEAP_BIT_PROPERTY:  begin
                              write(f, 'Bit properties':struct_length);
                              report_free_list;
                            end;

        HEAP_PROPERTIED_CONCATSIG:
                            begin
                              write(f, 'CS with props':struct_length);
                              report_free_list;
                            end;

        HEAP_FILE_LIST:     writeln(f, 'File lists':struct_length);

        HEAP_NET_DESCRIPTOR:
                            begin
                              write(f, 'Net descriptors':struct_length);
                              report_free_list;
                            end;

        HEAP_NET_TABLE:     begin
                              write(f, 'Net tables':struct_length);
                              report_free_list;
                            end;

        HEAP_NAME_LIST:     writeln(f, 'Name lists':struct_length);

        HEAP_AVL:           writeln(f, 'Avl members':struct_length);

	OTHERWISE
	  begin 
            writeln(f, 'Other':struct_length);
	  (*
	    assert(219);
	    writeln(cmplog, ' structure ', ord(structure):1);
	    if debugging then 
              writeln(outfile, ' structure ', ord(structure):1);
	  *)
	  end;
      end { case } ;
    end;

  writeln(f);
  writeln(f, total:19, 'Total':struct_length);
  writeln(f, corrected_total:19);
  writeln(f);

  if PrintCmpLst then
    begin
      page(CmpLst);
      writeln(CmpLst, ' Compiler structures used an estimated ',
                      corrected_total:1, ' bytes of heap storage.');
    end;

  writeln(f);
  writeln(f);
  writeln(f, ' Major static table sizes');  writeln(f);
  writeln(f, ' # Elements  # Bytes    Structure type');

  writeln(f, (SIGNAL_TABLE_SIZE+1):7,
                  (SIGNAL_TABLE_SIZE+1)*POINTER_SIZE:12,
                  'Signal table':struct_length);
  writeln(f, (NAME_TABLE_SIZE+1):7,
                  (NAME_TABLE_SIZE+1)*POINTER_SIZE:12,
                  'Identifier table':struct_length);
  writeln(f, (HASH_STRING_TABLE_SIZE+1):7,
                  (HASH_STRING_TABLE_SIZE+1)*POINTER_SIZE:12,
                  'String table':struct_length);

  total := (SIGNAL_TABLE_SIZE+1 +
            NAME_TABLE_SIZE+1 +
            HASH_STRING_TABLE_SIZE+1) * POINTER_SIZE;

  writeln(f);
  writeln(f, total:19, 'Total':struct_length);
  writeln(f);
  if debug_20 then writeln(outfile, ' exit report_heap_usage');
end { report_heap_usage } ;


(**)     { ------- string package routines ------- }


{
               ***********************************
               *                                 *
               *       String description        *
               *                                 *
               ***********************************


   A string is represented as a pointer to a packed array of char:

       string = ^packed array [0..255] of char;

   Each string, however, is usually less than 256 characters.  The
   actual length of the string is found in the first byte:  string^[0].
   The length of the string is static;  it should not be changed once
   the string has been created.

   Strings can be up to 255 characters long.  The programmer must make
   sure that characters are not written beyond the end of the string.

   Strings are created on the heap in quantized lengths.  There are 33
   different length arrays created.  The create_a_string routine
   creates an array on the heap big enough to support the given string. }



procedure new_free_element(var f: freeptr);
  { create a new free element for released strings }
begin
  new(f);
  increment_heap_count(HEAP_FREEELEMENT, 2*POINTER_SIZE);
  f^.next := NIL;
  f^.str := NIL;
end { new_free_element } ;


procedure create_a_string(var str: xtring; length: string_range);
  { Create a string on the heap of the given length.  This routine uses a
    variant record to represent strings of various lengths with one
    pointer.  First, the free lists are checked for a string of the
    appropriate length.  If none are available, a string is newed from
    the heap.  This scheme works only if the Pascal compiler creates only
    as much space as needed for a variant when the tag field is specified
    in the new.  }
  type
    size_type = (s4,s8,s12,s16,s20,s24,s28,sz32,s36,s40,s44,
                 s48,s52,s56,s60,s64,s68,s72,s76,s80,s84,s88,
                 s92,s96,s100,s120,s140,s160,s180,s200,s220,s240,s256);

    trick_ptr = ^trick_record;
    trick_record = record case size_type of
                     s4: (f4: packed array [0..3] of char);
                     s8: (f8: packed array [0..7] of char);
                     s12: (f12: packed array [0..11] of char);
                     s16: (f16: packed array [0..15] of char);
                     s20: (f20: packed array [0..19] of char);
                     s24: (f24: packed array [0..23] of char);
                     s28: (f28: packed array [0..27] of char);
                     sz32: (f32: packed array [0..31] of char);
                     s36: (f36: packed array [0..35] of char);
                     s40: (f40: packed array [0..39] of char);
                     s44: (f44: packed array [0..43] of char);
                     s48: (f48: packed array [0..47] of char);
                     s52: (f52: packed array [0..51] of char);
                     s56: (f56: packed array [0..55] of char);
                     s60: (f60: packed array [0..59] of char);
                     s64: (f64: packed array [0..63] of char);
                     s68: (f68: packed array [0..67] of char);
                     s72: (f72: packed array [0..71] of char);
                     s76: (f76: packed array [0..75] of char);
                     s80: (f80: packed array [0..79] of char);
                     s84: (f84: packed array [0..83] of char);
                     s88: (f88: packed array [0..87] of char);
                     s92: (f92: packed array [0..91] of char);
                     s96: (f96: packed array [0..95] of char);
                     s100: (f100: packed array [0..99] of char);
                     s120: (f120: packed array [0..119] of char);
                     s140: (f140: packed array [0..139] of char);
                     s160: (f160: packed array [0..159] of char);
                     s180: (f180: packed array [0..179] of char);
                     s200: (f200: packed array [0..199] of char);
                     s220: (f220: packed array [0..219] of char);
                     s240: (f240: packed array [0..239] of char);
                     s256: (f256: packed array [0..255] of char);
                    end;
var
  k: record case boolean of      { "trick" record to fiddle with pointers }
       TRUE:  (tp: trick_ptr);
       FALSE: (ap: xtring);
     end;
  p: trick_ptr;                  { pointer to the created string }
  fp: freeptr;                   { pointer to head of free strings }
  size: 1..33;                   { the size (index into table) of string }

begin
  if length > 100 then size := ((length+1)+420) DIV 20
                  else size := ((length+1) DIV 4) + 1;
  if free_strings[size] <> NIL then
    begin
      str := free_strings[size]^.str;
      fp := free_strings[size]^.next;
      free_strings[size]^.next := free_pointers;
      free_pointers := free_strings[size];
      free_strings[size] := fp;
    end
  else
    begin
      case s_length[size] of
          4: new(p,s4);
          8: new(p,s8);
         12: new(p,s12);
         16: new(p,s16);
         20: new(p,s20);
         24: new(p,s24);
         28: new(p,s28);
         32: new(p,sz32);
         36: new(p,s36);
         40: new(p,s40);
         44: new(p,s44);
         48: new(p,s48);
         52: new(p,s52);
         56: new(p,s56);
         60: new(p,s60);
         64: new(p,s64);
         68: new(p,s68);
         72: new(p,s72);
         76: new(p,s76);
         80: new(p,s80);
         84: new(p,s84);
         88: new(p,s88);
         92: new(p,s92);
         96: new(p,s96);
        100: new(p,s100);
        120: new(p,s120);
        140: new(p,s140);
        160: new(p,s160);
        180: new(p,s180);
        200: new(p,s200);
        220: new(p,s220);
        240: new(p,s240);
        256: new(p,s256);
      end;
      k.tp := p;  str := k.ap;
      increment_heap_count(HEAP_STRING, s_length[size]);
    end;
  str^[0] := chr(length);
end { create_a_string } ;


procedure release_string(var str: xtring);
  { free the storage used by the given string and place on free list }
  var
    size: string_range;     { size (index into table) of the string }
    f: freeptr;             { head of list of free strings }
begin
  if str <> nullstring then
    begin
      if ord(str^[0]) > 100 then size := (ord(str^[0])+420) DIV 20
                            else size := (ord(str^[0]) DIV 4) + 1;
      if free_pointers = NIL then new_free_element(f)
      else
        begin f := free_pointers; free_pointers := free_pointers^.next; end;
      f^.next := free_strings[size];
      free_strings[size] := f;  f^.str := str;
      str := nullstring;
    end;
end { release_string } ;


procedure copy_string(source: xtring;  var dest: xtring);
  { copy from the source to the destination.  The destination string must 
    exist (= nullstring or some other string).  If the source length is not
    equal to the destination length the destination string is "free"d and a
    new string of the proper size is created. }
  var
    pos: string_range;        { index into string for copy }
begin
  if source = NIL then source := nullstring;
  if source^[0] <> dest^[0] then
    begin
      release_string(dest);  create_a_string(dest, ord(source^[0]));
    end;

  for pos := 1 to ord(source^[0]) do  dest^[pos] := source^[pos];
end { copy_string } ;

    
procedure copy_from_string(str: xtring; var name: alpha);
  { copy from a string to an identifier.  Pad with blanks if the string
    has fewer characters than the identifier;  truncate if longer. }
  var
    min,            { smaller of ID_LENGTH and length(string) }
    i: id_range;    { index into the alpha }
begin
  name := null_alpha;
  if ord(str^[0]) < ID_LENGTH then min := ord(str^[0]) else min := ID_LENGTH;
  for i := 1 to min do name[i] := str^[i];
end { copy_from_string } ;


procedure copy_to_string(name: alpha;  var str: xtring);
  { copy from an alpha to a string.  Trailing blanks are deleted. }
  var
    len: id_range;    { length of the identifer }
    i: id_range;      { index into alpha and string for copy }
begin
  len := alpha_length(name);

  if ord(str^[0]) <> len then 
    begin  release_string(str);  create_a_string(str, len);  end;
  for i := 1 to len do  str^[i] := name[i];
end { copy_to_string } ;


function CmpStrLEQ(s1, s2: xtring): boolean;
  { returns TRUE if s1 <= s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] > s2^[0] then 
    begin  min_length := ord(s2^[0]);  CmpStrLEQ := FALSE;  end
  else
    begin  min_length := ord(s1^[0]);  CmpStrLEQ := TRUE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] > s2^[i] then
        begin  CmpStrLEQ := FALSE;  done := TRUE;  end
      else
        if s1^[i] < s2^[i] then
          begin  CmpStrLEQ := TRUE;  done := TRUE;  end;
    end;
end { CmpStrLEQ } ;


function CmpStrLT(s1, s2: xtring): boolean;
  { returns TRUE if s1 < s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] <= s2^[0] then 
    begin  min_length := ord(s1^[0]);  CmpStrLT := TRUE;  end
  else 
    begin  min_length := ord(s2^[0]);  CmpStrLT := FALSE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] > s2^[i] then 
        begin  CmpStrLT := FALSE;  done := TRUE;  end
      else if s1^[i] < s2^[i] then
        begin  CmpStrLT := TRUE;  done := TRUE;  end;
    end;
end { CmpStrLT } ;


function CmpStrGT(s1, s2: xtring): boolean;
  { returns TRUE if s1 > s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] <= s2^[0] then 
    begin  min_length := ord(s1^[0]);  CmpStrGT := FALSE;  end
  else 
    begin  min_length := ord(s2^[0]);  CmpStrGT := TRUE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] < s2^[i] then 
        begin  CmpStrGT := FALSE;  done := TRUE;  end
      else if s1^[i] > s2^[i] then
        begin  CmpStrGT := TRUE;  done := TRUE;  end;
    end;
end { CmpStrGT } ;


function CmpStrEQ(s1, s2: xtring): boolean;
  { returns TRUE if s1 = s2, FALSE otherwise. }
  var
    i: string_range;        { index into the strings }
    done: boolean;          { TRUE if comparison complete }
begin
  CmpStrEQ := FALSE;

  if s2^[0] = s1^[0] then
    begin
      i := 0;  done := FALSE;
      while (i < ord(s1^[0])) and not done do
        begin
          i := i + 1;
          if s1^[i] <> s2^[i] then done := TRUE;
        end;
      if not done then CmpStrEQ := TRUE;
    end;
end { CmpStrEQ } ;


function compare_strings(s1, s2: xtring): compare_type;
  { compare the strings and return the result }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    result: compare_type;   { result of the comparison }
    still_equal: boolean;   { TRUE if strings are equal to current position }
begin
  if s1^[0] = s2^[0] then
    begin  min_length := ord(s1^[0]);  result := EQ;  end
  else if s1^[0] < s2^[0] then
    begin  min_length := ord(s1^[0]);  result := LT;  end
  else 
    begin  min_length := ord(s2^[0]);  result := GT;  end;

  i := 0;  still_equal := TRUE;
  while (i < min_length) and still_equal do
    begin
      i := i + 1;
      if s1^[i] < s2^[i] then
        begin  result := LT;  still_equal := FALSE;  end
      else if s1^[i] > s2^[i] then
        begin  result := GT;  still_equal := FALSE;  end;
    end;

  compare_strings := result;
end { compare_strings } ;


function add_char_to_string(str: xtring;  ch: char): boolean;
  { add the character to the end of the string.  It is assumed that the
    string has been created with length = MAX_STRING_LENGTH and the 
    current length of the string (STR^[0]) is correct.  Always leave the
    last char of the string empty (for path string closing paren). }
begin
  if ord(str^[0])+1 >= MAX_STRING_LENGTH then
    add_char_to_string := FALSE
  else
    begin
      str^[0] := chr(ord(str^[0]) + 1);  str^[ord(str^[0])] := ch;
      add_char_to_string := TRUE;
    end;
end { add_char_to_string } ;


function add_string_to_string(dest, source: xtring): boolean;
  { add the source string to the end of the destination string.  Return
    FALSE if the destination string length is exceeded. It is assumed that
    the destination string has been created as MAX_STRING_LENGTH long and
    its length can be extended that far.  If the resulting string will be
    too long, copy as much as possible. }
  var
    i: string_range;            { index into the strings }
    source_length,              { length of the source string }
    dest_length: string_range;  { length of the destination string }
begin
  add_string_to_string := TRUE;

  if source <> nullstring then
    begin
      dest_length := ord(dest^[0]);  source_length := ord(source^[0]);
      if source_length + dest_length >= MAX_STRING_LENGTH then
        begin
          add_string_to_string := FALSE;
          source_length := MAX_STRING_LENGTH - dest_length - 1;
        end;

      for i := 1 to source_length do dest^[dest_length+i] := source^[i];

      dest^[0] := chr(dest_length + source_length);
    end;
end { add_string_to_string } ;


function add_quoted_string_to_string(dest, source: xtring): boolean;
  { add the source string, quoted, to the end of the destination string.
    Double quotes occurring inside the string.  Return FALSE if the
    destination string length is exceeded. It is assumed that the
    destination string has been created as MAX_STRING_LENGTH long and its
    length can be extended that far.  If the resulting string will be
    too long, copy as much as possible. }
  var
    i: string_range;            { index into source }
    j: string_range;            { index into dest }
    len: string_range;          { length of source }
    overflow: boolean;          { TRUE iff string overflows }
begin
  i := 0;
  j := ord(dest^[0]);
  len := ord(source^[0]);
  overflow := FALSE;

  if j < MAX_STRING_LENGTH then
    begin
      j := j + 1;
      dest^[j] := OUTPUT_QUOTE_CHAR;
    end;
  
  while (j < MAX_STRING_LENGTH) and (i < len) do
    begin
      i := i + 1;
      j := j + 1;
      dest^[j] := source^[i];
      if source^[i] = OUTPUT_QUOTE_CHAR then
        if j < MAX_STRING_LENGTH then
	  begin
	    j := j + 1;
	    dest^[j] := OUTPUT_QUOTE_CHAR;
	  end;
    end;

  if j < MAX_STRING_LENGTH then
    begin
      j := j + 1;
      dest^[j] := OUTPUT_QUOTE_CHAR;
    end
  else overflow := TRUE;

  dest^[0] := chr(j);
  add_quoted_string_to_string := not overflow;
end { add_quoted_string_to_string } ;


function add_substr_to_string(dest, source: xtring;
                              first, len: string_range): boolean;
  { add the specified substring of source to the end of the destination
    string.  Subrange is len chars long beginning with source^[first].
    Return FALSE if the destination string length is exceeded. 
    It is assumed that the destination string has been created as 
    MAX_STRING_LENGTH long and its length can be extended that far.  
    If the resulting string will be too long, copy as much as possible. }
  var
    i: string_range;            { index into the strings }
    offset: string_range;       { offset to first char (first - 1) }
    source_length,              { length of the source string }
    dest_length: string_range;  { length of the destination string }
begin
  add_substr_to_string := TRUE;

  if len > 0 then
    begin
      offset := first - 1;
      dest_length := ord(dest^[0]);  
      source_length := min(len, max(0, ord(source^[0]) - offset));
      if source_length + dest_length >= MAX_STRING_LENGTH then
        begin
          add_substr_to_string := FALSE;
          source_length := MAX_STRING_LENGTH - dest_length - 1;
        end;

      for i := 1 to source_length do 
        dest^[dest_length + i] := source^[i + offset];

      dest^[0] := chr(dest_length + source_length);
   end;
end { add_substr_to_string } ;


function add_alpha_to_string(dest: xtring; ident: alpha): boolean;
  { append the identifier (alpha) to the end of the destination string.
    It is assumed that the string has been created MAX_STRING_LENGTH long
    and its length can be extended to there.  If the resulting string
    exceeds MAX_STRING_LENGTH (leaving the last character empty), do
    as much as possible and return FALSE.  Delete trailing spaces from the
    alpha. }
  var
    i: id_range;                  { next char of the_alpha to copy in }
    identifier_length: id_range;  { length of the identifier }
    dest_length: string_range;    { length of the destination string }
begin
  identifier_length := alpha_length(ident);

  dest_length := ord(dest^[0]);
  if dest_length + identifier_length >= MAX_STRING_LENGTH then
    begin
      add_alpha_to_string := FALSE;
      identifier_length := MAX_STRING_LENGTH - dest_length - 1;
    end
  else add_alpha_to_string := TRUE;

  for i := 1 to identifier_length do
    dest^[dest_length+i] := ident[i];
  dest^[0] := chr(dest_length + identifier_length);
end { add_alpha_to_string } ;


function add_number_to_string(str: xtring; number: longint): boolean;
  { Append the given number (NUMBER) to the end of the input string (STR).
    If the number is < 0 then append a '-' at the start.  It is assumed that
    the input string has been created with length = MAX_STRING_LENGTH and
    that the length of the string (STR^[0]) is the current length.  Always
    leave the last char of the string empty (for path string closing ')'). }


  procedure build_number(n: natural_number);
    { add the given number to the string }
  begin
    if n > 9 then build_number(n DIV 10);
    if ord(str^[0])+1 >= MAX_STRING_LENGTH then
      add_number_to_string := FALSE
    else
      if not add_char_to_string(str, chr((n mod 10) + ord('0'))) then
        add_number_to_string := FALSE;
  end { build_number } ;


begin { add_number_to_string }
  add_number_to_string := TRUE;

  if number < 0 then
    begin
      number := abs(number);
      if not add_char_to_string(str, '-') then
        add_number_to_string := FALSE;
    end;

  build_number(number);
end { add_number_to_string } ;


function number_to_string(numb: longint): xtring;
  { return string representing the number in decimal }
  var
    temp: xtring;   { for building number }
begin
  create_a_string(temp, MAX_STRING_LENGTH);
  temp^[0] := chr(0);
  if add_number_to_string(temp, numb) then ;
  number_to_string := enter_string(temp);
  temp^[0] := chr(MAX_STRING_LENGTH);
  release_string(temp);
end { number_to_string } ;


function string_to_natural_number(str: xtring): natural_number;
  { convert string containing a decimal natural number into that number }
  const
    RADIX = 10;  { decimal number }
  var
    temp: natural_number;        { value of the function to be returned }
    len: string_range;           { length of str }
    i: string_range;             { index into str }
    ch: char;                    { current char }
    next_digit: 0..9;            { numeric value of current digit }
    trailing_junk: boolean;      { TRUE if chars follow the last digit }
    done: boolean;               { TRUE when done with a loop }
begin
  temp := 0;

  { ignore preceeding blanks }
  len := ord(str^[0]);
  i := 0;  done := FALSE;
  while (i < len) and not done do
    begin
      i := i + 1;
      if str^[i] <> ' ' then done := TRUE;
    end;

  { add up the digits }
  if i > 0 then i := i - 1;  done := FALSE;  trailing_junk := FALSE;
  while (i < len) and not done do 
    begin
      i := i + 1;
      ch := str^[i];
      case ch of
	'0': next_digit := 0;  '1': next_digit := 1;
	'2': next_digit := 2;  '3': next_digit := 3;
	'4': next_digit := 4;  '5': next_digit := 5;
	'6': next_digit := 6;  '7': next_digit := 7;
	'8': next_digit := 8;  '9': next_digit := 9;
	OTHERWISE 
	  begin
	    done := TRUE;
	    trailing_junk := TRUE;
	  end;
      end;
      if not done then 
	if (temp > MAXINT div RADIX) or 
	   ((temp = MAXINT div RADIX) and
	    (next_digit > MAXINT mod RADIX)) then
	  begin  assert(240 { ovf });  done := TRUE;   end
	else
	  begin  temp := RADIX * temp + next_digit;  end;
    end { while } ;
  
  { ignore blanks but report other extraneous trailing junk }
  if trailing_junk then 
    begin
      i := i - 1;  done := FALSE;
      while (i < len) and not done do
	begin
	  i := i + 1;
	  if str^[i] <> ' ' then
	    begin
	      done := TRUE;
	      error(50 { extraneous junk at end of number });
	      error_dump_string(str);
	      error_dump_CRLF;
	    end;
	end;
    end;
  
  string_to_natural_number := temp;
end { string_to_natural_number } ;


function upper_case(str: xtring): xtring;
  { upper case the given string }
  var
    i: string_range;         { index into the string }
    new_string: xtring;      { new string to be created }
begin
  create_a_string(new_string, ord(str^[0]));
  for i := 1 to ord(str^[0]) do new_string^[i] := upshift[str^[i]];
  upper_case := enter_and_release_string(new_string);
end { upper_case } ;


function substring(str : xtring; start,len : string_range): xtring;
  { returns the specified substring of str.  substring starts
    at start and is len chars long.  It will be truncated if 
    start+len-1 > length of str.  Returned string is a string table
    entry. }
  var 
    offset: string_range;      { index offset in str }
    temp: xtring;              { temp for building substring }
    i: string_range;           { index into str and temp }
begin
  if start <= ord(str^[0]) then 
    begin
      offset := start - 1;
      if offset+len > ord(str^[0]) then len := ord(str^[0]) - offset;
      if len > 0 then 
	begin
	  create_a_string(temp, len);
	  for i := 1 to len do temp^[i] := str^[i + offset];
          substring := enter_and_release_string(temp);
	end
      else substring := nullstring;
    end
  else substring := nullstring;
end; {substring}


(**)     { ------- name  utilities ------- }


function alpha_length(*name: alpha): id_range*);
  { find the length of an identifier by scanning for its end }
  var
    i: id_range;         { index into the identifier }
    done: boolean;       { TRUE when end of alpha found }
begin
  i := ID_LENGTH;  done := FALSE;
  while (i > 1) and not done do
    if name[i] <> ' ' then done := TRUE else i := i - 1;

  alpha_length := i;
end { alpha_length } ;


(**)     { ------- create an abbreviation ------- }


function concoct_abbrev(name: xtring): xtring;
  { create an abbreviation from the given name }
  var
    i,                        { index into the created abbreviation }
    left,                     { number chars left in abbreviation }
    pos,                      { index into the returned string }
    len: string_range;        { length of the created abbreviation }
    found: boolean;           { TRUE if vowel found in string }
    temp : char_array;        { temp storage for the new name }
    new_string: xtring;       { abbreviation string to be returned }
begin
  { step 1; throw out garbage }

  len := 0;
  for i := 1 to ord(name^[0]) do
    if isupper[name^[i]] or isdigit[name^[i]] then
      begin  len := len + 1;  temp[len] := name^[i];  end;

  { step 2: if nothing left, add an 'A' }

  if len = 0 then
    begin  len := 1;  temp[len] := 'A';  end;

  { step 3; if > 3 characters, remove vowels till 3 or less }

  left := len;  found := TRUE;
  while (left > 3) AND found do
    begin
      i := 1;  found := FALSE;
      while (i <= len) and (not found) do
        if temp[i] IN ['A', 'E', 'I', 'O', 'U'] then
          begin  temp[i] := ' ';  left := left - 1;  found := TRUE;  end
        else i := i + 1;
    end;

  { step 4: if still more than 3, take the first 3 }

  i := len;
  while (i > 1) and (left > 3) do
    begin
      if temp[i] <> ' ' then
        begin  temp[i] := ' ';  left := left - 1;  end;
      i := i - 1;
    end;

  { step 5: create the string and copy }

  create_a_string(new_string, left);  pos := 0;
  for i := 1 to len do
    if temp[i] <> ' ' then
      begin  pos := pos + 1;  new_string^[pos] := temp[i];  end;
  if pos <> left then assert(39 { no way Jose });

  concoct_abbrev := enter_and_release_string(new_string);
end { concoct_abbrev } ;


(**)     { ------- AVL routines ---------------------------- }


{ ../avl/compare.p is included in compiler.p so that it follows
  all of the procedures it uses. }

#include "../avl/heap.p"
#include "../avl/debug.p"
#include "../avl/insert.p"
#include "../avl/traverse.p"



(**)     { ------- continuation char output routines ------- }


   (********************************************************************)
   (*                                                                  *)
   (*  These routines are used for controlling output to a SCALD       *)
   (*  interface file.  They check to make sure that no line exceeds   *)
   (*  MAX_OUTPUT_FILE_LENGTH number of characters.  If it does, a     *)
   (*  SCALD continuation character is output (CONTINUATION_CHAR) and  *)
   (*  the line is continued on the next line.                         *)
   (*                                                                  *)
   (********************************************************************)


procedure init_output_continue;
  { set up the output state variables for continuation output }
begin
  column := 0;
end { init_output_continue } ;


#include "args.p"


(**)     { ------- other utilities ------- }


procedure disp_line(message: message_type);
  { write a message and a pointer to the output file }
  var
    width: string_range;    { position to place circumflex }
begin
  if line_pos > 0 then  width := line_pos  else  width := 1;
  writeln(outfile, '^':width, message:message_length+4);
end { disp_line } ;


procedure output_configuration(var f: textfile; indent: natural_number);
  { output all configuration to the specified file.  Indentation can be
    specified. }
  var
    config_index: configure_types;       { index into configuration table }
    syntax_index: signal_syntax_range;   { index into syntax table }


  procedure dump_char(ch: char);
    { output the character;  print NULL if null }
  begin
    if ch = chr(255) then write(f, '''''') else write(f, '''', ch, '''');
  end { dump_char } ;


begin { output_configuration }
  { output signal syntax order }

  if indent > 0 then write(f, ' ':indent);

  write(f, 'SYNTAX = ');

  for syntax_index := 1 to SYNTAX_TABLE_SIZE do
    if signal_syntax_table[syntax_index] <> NULL_SPECIFIER then
      begin
        write(f, '<');
        writealpha(f, syntax_specifier_names[
                                    signal_syntax_table[syntax_index]]^.name);
        write(f, '> ')
      end;
  writeln(f, ';');

  { output the configurable portions of the signal syntax }

  for config_index := succ(FIRST_CONFIGURE_SPECIFIER) to
                                             pred(LAST_CONFIGURE_SPECIFIER) do
    begin
      if indent > 0 then write(f, ' ':indent);

      print_alpha(f, configure_specifiers[config_index]^.name);

      write(f, ' = ');

      case config_index of
        CONFIGURE_SUBRANGE:
            if subrangesy = COLON then write(f, ''':''')
                                  else write(f, '''..''');

        CONFIGURE_BIT_ORDERING:
            if left_to_right then write(f, 'LEFT_TO_RIGHT')
                             else write(f, 'RIGHT_TO_LEFT');

        CONFIGURE_LOW_ASSERTED:
            dump_char(signal_is_asserted_low_char);

        CONFIGURE_HIGH_ASSERTED:
            dump_char(signal_is_asserted_high_char);

        CONFIGURE_NEGATION:
            dump_char(signal_negation_char);

        CONFIGURE_NAME_PREFIX:
{           dump_char(name_property_prefix_char); }
            write(f, '''!''');

        CONFIGURE_GENERAL_PREFIX:
            dump_char(general_property_prefix_char);

        CONFIGURE_CONCATENATION:
            write(f, ''':''');
      end { case } ;

      writeln(f, ';');
    end;
end { output_configuration } ;


procedure dump_command_line_arguments(var f: textfile);
  { report command line results to f }
  var
    i: natural_number;       { current arg number }
    arg: xtring;             { current arg }
begin
  writeln(outfile, 'Command line arguments:');
  write(outfile, '   ');
  for i := 0 to sargc - 1 do 
    begin
      sargv(i, arg);
      write(f, ' ');  writestring(f, arg);
    end;
  writeln(outfile);
  
  if COMMAND_specified_in_command_line then
    begin
      write(outfile, '    command: ');
      writealpha(outfile, command_value[command]^.name);
      if command = COMPERR_COMMAND then
        begin
	  write(Outfile, ' Severity: ');
	  case specified_severity of
	    NO_SEVERITY: write(Outfile, ' ALL');
	    WARNING_SEVERITY: write(Outfile, ' WARNING');
	    OVERSIGHT_SEVERITY: write(Outfile, ' OVERSIGHT');
	    ERROR_SEVERITY: write(Outfile, ' ERROR');
	    OTHERWISE write(Outfile, ' ???');
	  end;
	end ;
      writeln(Outfile);
    end;
  if ROOT_specified_in_command_line then
    begin
      write(Outfile, '    root_drawing: ');
      writestring(Outfile, root_macro_name);
      writeln(Outfile);
    end;
  if TYPE_specified_in_command_line then
    begin
      write(Outfile, '    compile: ');
      writealpha(Outfile, specified_compile_type^.name);
      writeln(Outfile);
    end;
  if CMPDRAW_specified_in_command_line then
    writeln(Outfile, '    CmpDraw specified');
  if DESIGN_specified_in_command_line then
    writeln(Outfile, '    Design specified');
  if CONTEXT_specified_in_command_line then
    writeln(Outfile, '    context: specified!!! (but not used)');
end { dump_command_line_arguments } ;


(**)     { ------- error environment stack ------- }


procedure init_error_info;
  { initialize the error environment variables }
begin
  current_file_name := nullstring;
  current_macro_def := NIL;
  current_page := 0;
  current_body_node := NIL;
  current_body_name := nullstring;
  current_path_prop := nullstring;
  current_property_name := null_name;
  current_property_value := nullstring;
end { init_error_info } ;


procedure push_error_info;
  { push the current parse environment onto the stack and create a new one }
  var
    current: environment_ptr;     { save space for current environment }
begin
  if free_environments = NIL then
    begin
      new(current);
      increment_heap_count(HEAP_ENVIRONMENT_ENTRY, 4*POINTER_SIZE+INT_SIZE);
    end
  else
    begin
      current := free_environments;  free_environments := current^.next;
    end;

  with current^ do
    begin
      next := environment_stack;
      environment_stack := current;

      file_name := current_file_name;
      macro := current_macro_def;
      page_number := current_page;
      body_node := current_body_node;
      body_name := current_body_name;
      path_name := current_path_prop;
      property_name := current_property_name;
      property_value := current_property_value;
    end;
end { 
push_error_info } ;


procedure pop_error_info;
  { pop the old parse environment from the stack }
  var
    next: environment_ptr;    { next in the environment stack }
begin
  if environment_stack = NIL then assert(161 { parse environment underflow })
  else
    begin
      with environment_stack^ do
        begin
          current_file_name := file_name;
          current_macro_def := macro;
          current_page := page_number;
          current_body_node := body_node;
          current_body_name := body_name;
          current_path_prop := path_name;
          current_property_name := property_name;
          current_property_value := property_value;
        end;

      next := environment_stack^.next;
      environment_stack^.next := free_environments;
      free_environments := environment_stack;
      environment_stack := next;
    end;
end { pop_error_info } ;


(**)     { ------- error output routines ------- }


procedure error_dump_CRLF;
  { print the CRLF to the appropriate files }
begin
  if ok_to_print_error then
    if PrintCmpLst then writeln(CmpLst) else writeln(monitor);

  writeln(CmpLog);
  if debugging then writeln(outfile);
end { error_dump_CRLF } ;


procedure error_dump_indent(indentation: natural_number);
  { outputs a specified number of spaces to the error files.  If
    printing a CmpLst, then 2 extra spaces are printed to
    CmpLog (as PrintCmpLst implies that we are in the process of 
    printing a page and error messages are to be indented under the
    page heading). }
begin
  if ok_to_print_error then
    if PrintCmpLst then write(CmpLst, ' ':indentation)
                   else write(monitor, ' ':indentation);

  if PrintCmpLst then write(CmpLog, ' ':indentation+2)
                else write(CmpLog, ' ':indentation);

  if debugging then write(outfile, ' ':indentation);
end { error_dump_indent } ;


procedure error_dump_char(ch: char);
  { output a character to the error files }
begin
  if ok_to_print_error then
    if PrintCmpLst then write(CmpLst, ch) else write(monitor, ch);

  write(CmpLog, ch);

  if debugging then write(outfile, ch);
end { error_dump_char } ;


procedure error_dump_integer(*int: longint*);
  { print the integer on the error files }
begin
  if ok_to_print_error then
    if PrintCmpLst then write(CmpLst, int:1) else write(monitor, int:1);

  write(CmpLog, int:1);

  if debugging then write(outfile, int:1);
end { error_dump_integer } ;


procedure error_dump_alpha(data: alpha);
  { print an alpha to the error files }
begin
  if ok_to_print_error then
    if PrintCmpLst then writealpha(CmpLst,data) else writealpha(monitor,data);

  writealpha(CmpLog, data);

  if debugging then writealpha(outfile, data);
end { error_dump_alpha } ;


procedure error_dump_line(data: error_message);
  { indented and CRLFed without trailing blanks (but at least one char will
    be written). }
  var
    len: 1..ERROR_MESSAGE_LENGTH;  
    done: boolean;
begin
  error_dump_indent(indent);

  len := ERROR_MESSAGE_LENGTH;  done := FALSE;
  while (len > 1) and not done do
    if data[len] = ' ' then len := len - 1
                       else done := TRUE;

  if ok_to_print_error then
    if PrintCmpLst then writeln(CmpLst,data:len) 
		   else writeln(monitor,data:len);

  writeln(CmpLog, data:len);

  if debugging then writeln(outfile, data:len);
end { error_dump_line } ;


procedure error_dump_string(*str: xtring*);
  { print the given string to the error files }
begin
  if ok_to_print_error then
    if PrintCmpLst then print_string(CmpLst, str)
    else print_string(monitor, str);

  print_string(CmpLog, str);

  if debugging then print_string(outfile, str);
end { error_dump_string } ;


procedure error_dump_property(name: name_ptr;  text: xtring);
  { print the property name and the property value (if the value <> NULL) }
begin
  error_dump_indent(indent);
  error_dump_alpha('Property name=  ');
  error_dump_alpha(name^.name);
  error_dump_CRLF;

  if text <> nullstring then
    begin
      error_dump_indent(indent);
      error_dump_alpha('Property value= ');
      error_dump_string(text);
      error_dump_CRLF;
    end;
end { error_dump_property } ;


procedure error_dump_expanded_value(val: xtring);
  { dump the string to the error files labled "expanded value". }
begin
  error_dump_indent(indent);
  error_dump_alpha('Expanded value= ');
  error_dump_string(val);
  error_dump_CRLF;
end { error_dump_expanded_value } ;


procedure error_dump_ioresult(iores: integer);
  { output an ioresult error message. Do not print anything if
    iores = 0 (no error). }
begin
  if iores <> 0 then
    begin
      error_dump_indent(indent);
      if ok_to_print_error then
        if PrintCmpLst then write_ioresult(CmpLst, iores)
                       else write_ioresult(Monitor, iores);
      write_ioresult(CmpLog, iores);
      if debugging then write_ioresult(Outfile, iores);
      error_dump_CRLF;
    end;
end { error_dump_ioresult } ;


procedure error_dump_file_name(name: xtring);
  { dump the name of the file to the error files }
begin
  if (name <> NIL) then
    begin
      error_dump_indent(indent);
      error_dump_alpha('File name=      ');
      error_dump_string(name);
      error_dump_CRLF;
    end;
end { error_dump_file_name } ;


procedure error_dump_directory_name(name: xtring);
  { dump the name of the directory to the error files }
begin
  if (name <> NIL) then
    begin
      error_dump_indent(indent);
      error_dump_alpha('Directory name= ');
      error_dump_string(name);
      error_dump_CRLF;
    end;
end { error_dump_file_name } ;


procedure error_dump_alpha_file_name(file_name: alpha);
  { dump the name of the file to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('File name=      ');
  error_dump_alpha(file_name);
  error_dump_CRLF;
end { error_dump_alpha_file_name } ;


procedure error_dump_text_macro(name: name_ptr);
  { dump the text macro name to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('Text macro=     ');
  error_dump_alpha(name^.name);
  error_dump_CRLF;

  if PrintCmpErr then
    begin
      write(CmpErr, '  property = ');
      writealpha(CmpErr, name^.name);
      writeln(CmpErr, ';');
    end;
end { error_dump_text_macro } ;


procedure error_dump_macro_def(macro: macro_def_ptr);
  { dump the macro name and its version to the error files }
begin
  if (macro <> NIL) and (macro <> root_macro_def) then
    begin
      error_dump_indent(indent);
      error_dump_alpha('Drawing name=   ');
      error_dump_string(macro^.macro_name);
      if macro^.version <> NIL then
        if (er_extension(macro^.version) <> NIL) then
	  begin
	    error_dump_char('.');
	    error_dump_string(er_extension(macro^.version));
	    error_dump_char('.');
	    error_dump_integer(er_version(macro^.version));
	    if current_page <> 0 then
	      begin
		error_dump_char('.');
		error_dump_integer(current_page);
	      end;
	  end
        else if (er_special(macro^.version) <> NIL) then
	  begin
	    error_dump_CRLF;
	    error_dump_indent(indent);
	    error_dump_alpha('Special file=   ');
	    error_dump_string(er_special(macro^.version));
	  end;
      error_dump_CRLF;
    end;
end { error_dump_macro_def } ;


procedure error_dump_macro_name(name: xtring; version: natural_number);
  { dump the macro name and its version (if <> 0) to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('Drawing name=   ');
  if version = 0 then
    error_dump_string(name)
  else
    begin
      error_dump_string(name);
      error_dump_alpha('.LOGIC.         ');
      error_dump_integer(version);
    end;
  error_dump_CRLF;
end { error_dump_macro_name } ;


procedure error_dump_body_name(body_name: xtring);
  { dump the body name to the error files along with the PATH property }
begin
  if body_name <> nullstring then
    begin
      error_dump_indent(indent);
      error_dump_alpha('Name of body=   ');
      error_dump_string(body_name);
      if current_path_prop <> nullstring then
        begin
          error_dump_char(' ');
          error_dump_alpha('(path prop=     ');
          error_dump_string(current_path_prop);
          error_dump_char(')');
        end;
      error_dump_CRLF;

      if PrintCmpErr then
        if ok_to_print_error then
          begin
            write(CmpErr, '  body = ');
            writestring(CmpErr, body_name);
            writeln(CmpErr, ';');
            if current_path_prop <> nullstring then
              begin
                write(CmpErr, '  path = ');
                writestring(CmpErr, current_path_prop);  writeln(CmpErr, ';');
              end;
          end;
    end;
end { error_dump_body_name } ;


procedure error_dump_body_node(body: mtree_node_ptr);
  { dump the body name to the error files along with the PATH property }
begin
  if body <> NIL then
    begin
      error_dump_indent(indent);
      error_dump_alpha('Name of body=   ');
      error_dump_string(body^.macro_name);
      if body^.called_by <> NIL then
        begin
          error_dump_char(' ');
          error_dump_alpha('(path prop=     ');
          error_dump_string(body^.called_by^.path);
          error_dump_char(')');
        end;
      error_dump_CRLF;
    end;
end { error_dump_body_node } ;


procedure error_dump_page_number(page_number: page_range);
  { dump the current page number to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('Page=           ');
  error_dump_integer(page_number);
  error_dump_CRLF;
end { error_dump_current_page_number } ;


procedure error_dump_current_parse_environment;
  { dump the current parse environment }
begin
  if current_file_name <> nullstring then
    error_dump_file_name(current_file_name);

  if (current_macro_def <> NIL) then error_dump_macro_def(current_macro_def);

  if current_body_node <> NIL then error_dump_body_node(current_body_node)
  else
    begin
      if (current_macro_def = NIL) and (current_page <> 0) then
        error_dump_page_number(current_page);

      if current_body_name <> nullstring then
        error_dump_body_name(current_body_name);
    end;

  if current_property_name <> null_name then
    error_dump_property(current_property_name, current_property_value);
end { error_dump_current_parse_environment } ;


procedure error_dump_signal_name_string(name: xtring);
  { dump the signal name (clear text) to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('Signal name=    ');
  error_dump_string(name);
  error_dump_CRLF;
end { error_dump_signal_name_string } ;


procedure error_dump_pin_name_string(name: xtring);
  { dump the pin name (clear text) to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('Pin name=       ');
  error_dump_string(name);
  error_dump_CRLF;
end { error_dump_pin_name_string } ;




(**)     { ------- general utility routines ------- }


function check_calculation(var val1: longint; val2: longint; 
                           is_sub: boolean): longint;
  { check to see that the calculation specified (val1 := val1+val2; or
    val1 := val1-val2;  does not overflow.  Return the result. }
begin
  if is_sub then
    if val2 > -MAXINT-1 then val2 := -val2
    else if val1 = MAXINT then begin error(24 { ovf });  val2 := 0;  end
         else val1 := val1 + 1;
  if val2 < 0 then
    begin
      if (val1 < 0) and (val1 < -MAXINT-1-val2) then 
        begin  error(24 { ovf });  val2 := 0;  end
    end
  else if (val1 > 0) and (val1 > MAXINT-val2) then
    begin  error(24 { ovf });  val2 := 0;  end;
  check_calculation := val1 + val2;
end { check_calculation } ;


(**)     { ------- memory usage utilities ------- }


  procedure print_histograms(var f: textfile);
    { print a histogram of each of the various tables }
    var
      i: natural_number;           { index into the tables }
      SEP: signal_entry_ptr;       { signal table element }
      NP: name_ptr;                { name table element }
      HSP: hash_string_ptr;        { hash string table element }
      num_elements,                { total number of elements in one bucket }
      max_num_entries,             { maximum entries in any one bucket }
      min_num_entries,             { minimum entries in any one bucket }
      total_entries: natural_number; { total number of entries in the table }
  begin
    if debug_20 then writeln(outfile, ' enter print_histograms');
    if signal_histogram IN histograms then
      begin
        writeln(f);
        writeln(f, '++++++++ Histogram of the signal table ++++++++');
        writeln(f);
        total_entries := 0;  min_num_entries := maxint;  max_num_entries := 0;
        for i := 0 to SIGNAL_TABLE_SIZE do
          begin
            num_elements := 0;  SEP := signal_table[i];
            write(f, i:3, ': ');
            while SEP <> NIL do
              begin
                num_elements := num_elements + 1;
                write(f, '*');
                SEP := SEP^.next;
              end;
            writeln(f);
            if num_elements > max_num_entries then
              max_num_entries := num_elements;
            if num_elements < min_num_entries then
              min_num_entries := num_elements;
            total_entries := total_entries + num_elements;
          end;
        writeln(f);
        writeln(f, 'Total number of entries: ', total_entries:1);
        writeln(f,'Cell lengths: ',min_num_entries:1,'..',max_num_entries:1);
        writeln(f);
      end;

    if name_table_histogram IN histograms then
      begin
        writeln(f);
        writeln(f, '++++++++ Histogram of the identifier name table ++++++++');
        writeln(f);
        total_entries := 0;  min_num_entries := maxint;  max_num_entries := 0;
        for i := 0 to NAME_TABLE_SIZE do
          begin
            num_elements := 0;  NP := name_table[i];
            write(f, i:3, ': ');
            while NP <> NIL do
              begin
                num_elements := num_elements + 1;
                write(f, '*');
                NP := NP^.next;
              end;
            writeln(f);
            if num_elements > max_num_entries then
              max_num_entries := num_elements;
            if num_elements < min_num_entries then
              min_num_entries := num_elements;
            total_entries := total_entries + num_elements;
          end;
        writeln(f);
        writeln(f, 'Total number of entries: ', total_entries:1);
        writeln(f,'Cell lengths: ',min_num_entries:1,'..',max_num_entries:1);
        writeln(f);
      end;

    if string_table_histogram IN histograms then
      begin
        writeln(f);
        writeln(f, '++++++++ Histogram of the string hash table ++++++++');
        writeln(f);
        total_entries := 0;  min_num_entries := maxint;  max_num_entries := 0;
        for i := 0 to HASH_STRING_TABLE_SIZE do
          begin
            num_elements := 0;  HSP := string_table[i];
            write(f, i:3, ': ');
            while HSP <> NIL do
              begin
                num_elements := num_elements + 1;
                write(f, '*');
                HSP := HSP^.next_hash_string;
              end;
            writeln(f);
            if num_elements > max_num_entries then
              max_num_entries := num_elements;
            if num_elements < min_num_entries then
              min_num_entries := num_elements;
            total_entries := total_entries + num_elements;
          end;
        writeln(f);
        writeln(f, 'Total number of entries: ', total_entries:1);
        writeln(f,'Cell lengths: ',min_num_entries:1,'..',max_num_entries:1);
        writeln(f);
      end;
    if debug_20 then writeln(outfile, ' exit print_histograms');
  end { print_histograms } ;


(**)     { ------- signal dump procedure ------- }

   
procedure dump_signal_polarity(var f: textfile; polarity: signal_polarity);
  { print out the polarity of the signal }
begin
  if polarity = COMPLEMENTED then write(f, '-')
  else if (polarity = UNKNOWN_POLARITY) and debugging then write(f, '#UN#')
  else if (polarity = NO_POLARITY) and debugging then write(f, '#NO#');
end { dump_signal_polarity } ;


procedure dump_polarity(var f: textfile; polarity: signal_polarity);
  { print the signal polarity }
begin
  case polarity of
    NORMAL:            write(f, 'NORMAL');
    COMPLEMENTED:      write(f, 'COMPLEMENTED');
    UNKNOWN_POLARITY:  write(f, 'UNKNOWN_POLARITY');
    NO_POLARITY:       write(f, 'NO_POLARITY');
  end;
end { dump_polarity } ;


procedure dump_left_and_right(var f: textfile; left, right: bit_range);
  { output a left and right subscript with the specified bits }
begin
  if left <> -1 then
    begin
      write(f, '<', left:1);
      if left <> right then
        write(f, '..', right:1);
      write(f, '>');
    end;
end { dump_left_and_right } ;


procedure dump_bit_subscript(var f: textfile; bits: subscript_ptr;
                             kind: signal_kind);
  { print the bit subscript }
  var
    SP: subscript_ptr;       { current subscript element }
    first: boolean;          { TRUE if no preceding comma to be printed }
begin
  if bits = NIL then
    if kind = undefined then
      begin
        write(f,'<');
        writealpha(f, UNDEFINED_identifier^.name);  write(f,'>');
      end
    else if kind = vector then write(f, '<none>')
    else { it's a scalar, don't print anything }
  else
    if (kind = undefined) or (kind = single) then write(f, '<BAD>')
    else
      begin
        SP := bits;  first := TRUE;
        write(f, '<');
        while SP <> NIL do
          begin
            if first then first := FALSE
            else write(f, ',');
            write(f, SP^.left_index:1);
            if SP^.left_index <> SP^.right_index then
              write(f, '..', SP^.right_index:1);
            SP := SP^.next;
          end;
        write(f, '>');
      end;
end { dump_bit_subscript } ;


procedure dump_signal_descriptor(var f: textfile;
                                 signal: signal_descriptor_ptr);
  { dump the signal from the given signal descriptor }
  var
    prop: property_ptr;
begin
  while signal <> NIL do 
    begin
      if signal^.polarity IN [NORMAL,COMPLEMENTED] then
        begin
          if signal^.low_asserted then
            begin
              if signal^.polarity = NORMAL then
                write(f, '-');
            end
          else
            if signal^.polarity = COMPLEMENTED then
              write(f, '-');
        end
      else if debugging then
        if signal^.polarity = UNKNOWN_POLARITY then
          write(f, '#UN#')
        else if signal^.polarity = NO_POLARITY then
          write(f, '#NO#');
        
      writestring(f, signal^.signal_name);
        
      dump_bit_subscript(f, signal^.bit_subscript, signal^.kind);

      if signal^.low_asserted then write(f, '*');

      if signal^.replication_factor <> 1 then
        write(f, DEFAULT_GENERAL_PROPERTY_PREFIX_CHAR, 'R ',
                 signal^.replication_factor:1);

      prop := signal^.properties;
      while prop <> NIL do
        begin
	  write(f, DEFAULT_GENERAL_PROPERTY_PREFIX_CHAR);
	  writealpha(f, prop^.name^.name);
	  write(f, '=');
	  writestring(f, prop^.text);
	  prop := prop^.next;
	end;

      if debugging and (signal^.net_id <> nullstring) then
        begin
          write(f, '(');
          dump_string(f, signal^.net_id);
          write(f, ')');
        end;

      if signal^.next <> NIL then writeln(f, ':');
        
      signal := signal^.next;
    end;

  writeln(f);
end { dump_signal_descriptor } ;

        
(**)     { ------- print the macro definition ------- }


procedure print_bindings_list(var f: textfile;  BP: bindings_list_ptr);
  { print the bindings list in a format suitable for print_macro 
    (following). }
  var
    ALP: clear_text_actual_list_ptr;  { list of actuals bound to pin }
begin
  while BP <> NIL do
    begin
      write(f, ' ':8);
      print_string(f, BP^.formal_parameter);

      ALP := BP^.actual_parameter;
      while ALP <> NIL do
	begin
	  if ALP <> BP^.actual_parameter then
	    write(f, ' ':11);

	  write(f, ' = ');
	  print_string(f, ALP^.actual_parameter);
	  writeln(f);

	  ALP := ALP^.next;
	end;

      BP := BP^.next;
    end;

end { print_bindings_list } ;


procedure print_macro(var f: textfile;  macro: macro_def_ptr);
  { print the macro definition to the specified file }
  var
    ILP: invoke_list_ptr;        { current invoked macro element }


  procedure output_signal_list(SLP: signal_list_ptr);
    { output a signal list. }
  begin
    while SLP <> NIL do
      begin
        write(f, ' ':4); writestring(f, SLP^.signal_name); 
        writeln(f);
        SLP := SLP^.next;
      end;
    writeln(f);
  end { output_signal_list } ;


  procedure output_properties(PP: property_ptr; indent: print_width_range);
    { output a property list }
  begin
    if PP = NIL then writeln(f)
    else
      while PP <> NIL do
        begin
          write(f, ' ':indent);
          print_alpha(f, PP^.name^.name);
          write(f, ' = ');
          writestring(f, PP^.text);  writeln(f);
          PP := PP^.next;
        end;
  end { output_properties } ;


begin { print_macro }
  writeln(f);
  writeln(f, 'Dump of the Macro definitions');
  writeln(f);
  writeln(f, '--------------------------------------------------');

  write(f, 'Macro being defined: ');
  writestring(f, macro^.macro_name);
  writeln(f);  writeln(f);

  if macro^.version <> NIL then
    if (er_extension(macro^.version) <> NIL) then
      begin
        write(f, 'Macro version ');
        print_string(f, er_extension(macro^.version));
	writeln(f, '.', er_version(macro^.version):1);
      end;

  writeln(f);
  if macro^.is_leaf_macro then writeln(f, 'This is a LEAF macro')
                          else writeln(f, 'This is not a LEAF macro');

  writeln(f);  
  writeln(f, 'Dump of the parameter list');
  output_signal_list(macro^.params);

  writeln(f, 'Dump of macro properties');
  output_properties(macro^.properties, 4);

  writeln(f, 'Dump of the text macros');
  output_properties(macro^.text_macros, 4);

  if not macro^.is_leaf_macro then
    begin
      writeln(f);
      writeln(f, 'The macros called are as follows:');  
      writeln(f);

      ILP := macro^.invokes;
      while ILP <> NIL do
        begin
          writeln(f, '    **************************************');
          write  (f, '    Macro: ');
          writestring(f, ILP^.macro_name);
          writeln(f);

          writeln(f, '    Dump of the properties');
          output_properties(ILP^.properties, 8);

          writeln(f, '    Dump of the body parameters');
          output_properties(ILP^.parameters, 8);

          writeln(f, '    Formal/Actual parameters');
	  print_bindings_list(f, ILP^.bindings);

          writeln(f);  writeln(f);

          ILP := ILP^.next;
        end;
    end { is_leaf_macro = FALSE } ;
end { print_macro } ;


(**)     { ------- SYMBOL DUMP procedure ------- }


procedure dump_symbol_table(var f: textfile; symbol_table: identifier_ptr);
  { dump the given symbol table to the given file }
  var
    current_id: identifier_ptr;    { current identifier in the table }
begin
  writeln(f, '--- dump of the symbol table ---');

  current_id := symbol_table;
  while current_id <> NIL do
    begin
      print_alpha(f, current_id^.name^.name);
      write(f, ' = "');
      print_string(f, current_id^.definition);
      writeln(f, '"');

      current_id := current_id^.next;
    end;
end { dump_symbol_table } ;


(**)     { ------- dump out all the identifiers ------- }


procedure dump_all_names;
  { dump all the names in the name table to outfile }
  var
    i: name_table_range;      { current index into the name table }
    name: name_ptr;           { element in the list from table }
    form: name_types;         { type of the identifier }
begin
  if debug_20 then writeln(outfile, ' enter dump_all_names');
  if dump_all_names_ok then
    for i := 0 to NAME_TABLE_SIZE do
      begin
        name := name_table[i];
        while name <> NIL do
          begin
            print_alpha(outfile, name^.name);
            writeln(outfile, '(', i:1, ')');

            for form := succ(FIRST_NAME_TYPE) to pred(LAST_NAME_TYPE) do
              if form IN name^.kind then
                case form of
                  RESERVED:          
		    begin
		      write(outfile, '    reserved ');
		      writestring(outfile, name^.definition);
		      writeln(outfile);
		    end;
                  UNRESERVED:
		    begin
		      write(outfile, '    unreserved ');
		      writestring(outfile, name^.definition);
		      writeln(outfile);
		    end;
                  KEY_WORD:          writeln(outfile, '    key word');
                  INHERIT_PIN:       writeln(outfile, '    inherit pin');
                  INHERIT_SIGNAL:    writeln(outfile, '    inherit signal');
                  INHERIT_BODY:      writeln(outfile, '    inherit body');
                  IS_PARAMETER:      writeln(outfile, '    is parameter');
                  IS_INT_PARAMETER:  writeln(outfile, '    is int parameter');
                  PERMIT_BODY:       writeln(outfile, '    permit body');
                  PERMIT_SIGNAL:     writeln(outfile, '    permit signal');
                  PERMIT_PIN:        writeln(outfile, '    permit pin');
                  DONT_OUTPUT:       writeln(outfile, '    filtered');
                  IS_ET_CONTROL:     writeln(outfile, '    et control');
                  OTHERWISE          writeln(outfile, '    attribute number ',
                                             ord(form):1);
                end;

            name := name^.next;
          end;
      end;
  if debug_20 then writeln(outfile, ' enter dump_all_names');
end { dump_all_names } ;


procedure new_identifier(*var id: identifier_ptr*);
  { create an identifier }
begin
  if free_identifiers <> NIL then
    begin  id := free_identifiers;  free_identifiers := id^.next;  end
  else
    begin
      new(id);
      increment_heap_count(HEAP_IDENTIFIER, 3*POINTER_SIZE+BOOL_SIZE);
    end;

  id^.next := NIL;
  id^.name := NIL;
  id^.definition := nullstring;
  id^.resolves := TRUE;
end { new_identifier } ;


procedure release_identifier(var id: identifier_ptr);
  { release the identifier for later reuse }
begin
  if id <> NIL then
    begin
      id^.next := free_identifiers;  free_identifiers := id;  id := NIL;
    end;
end { release_identifier } ;


procedure release_symbol_table(var symbol_table: identifier_ptr);
  { release all elements in the symbol table }
  var
    next,                       { next element in the table }
    element: identifier_ptr;    { current element in the table }
begin
  element := symbol_table;
  while element <> NIL do
    begin
      next := element^.next;
      release_identifier(element);
      element := next;
    end;
end { release_symbol_table } ;


procedure new_directory_list(var element: directory_list_ptr);
  { create a new directory entry }
begin
  if free_directory_lists = NIL then
    begin
      new(element);  
      increment_heap_count(HEAP_DIRECTORY, 3*POINTER_SIZE);
    end
  else
    begin
      element := free_directory_lists;
      free_directory_lists := free_directory_lists^.next;
    end;

  element^.next := NIL;  
  element^.name := nullstring;
  element^.add := nullstring;
end { new_directory_list } ;

procedure release_directory_list(var element: directory_list_ptr);
  { delete and release a directory entry }
  var
    oldnext: directory_list_ptr; { value to be returned }
begin
  if element <> NIL then
    begin
      oldnext := element^.next;
      element^.next := free_directory_lists;
      free_directory_lists := element;
      element := oldnext;
    end;
end { release_directory_list } ;


(**)     { ------- file utilities ------- }


#include "filename.p"
#include "filedesc.p"
#include "vaxio.p"
#include "fileutil.p"
#include "fileio.p"
#include "pipe.p"


function open_a_file(file_name: xtring; which: parse_file_type): boolean;
  { open the specified file for read and parse the first token from it.
    If the FILE_NAME is not empty (NULLSTRING), open the file named.  If
    it is empty, open the file as specified in the file equation. }
  var
    ok: boolean;         { TRUE iff file open successful }
begin
  ok := reset_file(file_name, which);

  if ok then  
    begin
      read_state := FINIT;
      insymbol;
    end;
  open_a_file := ok;
end { open_a_file } ;

function open_a_ds_file(file_name: xtring; which: parse_file_type): boolean;
  { open the specified file for read and parse the first token from it.
    If the FILE_NAME is not empty (NULLSTRING), open the file named.  If
    it is empty, open the file as specified in the file equation. }
  var
    ok: boolean;         { TRUE iff file open successful }
begin
  ok := reset_ds_file(file_name, which);

  if ok then  
    begin
      read_state := FINIT;
      insymbol;
    end;
  open_a_ds_file := ok;
end { open_a_ds_file } ;


function get_file_type: file_types;
  { check to see that the current file has correct type and return type }
  var
    this_file: file_types;      { type of this file }
    found: boolean;             { TRUE if file type found in table }


begin { get_file_type }
  this_file := FIRST_FILE_TYPE;
  if sy <> FILETYPESY then error(85 { expected FILE_TYPE })
  else
    begin
      insymbol;
      if sy = EQUAL then insymbol else error(2 { expected = });
      if sy <> IDENT then error(1 { expected ident })
      else
        begin
          this_file := succ(FIRST_FILE_TYPE);  found := FALSE;
          while (this_file < LAST_FILE_TYPE) and not found do
            if file_type_list[this_file] = id.name then found := TRUE
            else this_file := succ(this_file);

          if not found then this_file := FIRST_FILE_TYPE;
          insymbol;
          if sy = SEMI then insymbol else assert(35 { expected ; });
        end;
    end;

  get_file_type := this_file;
end { get_file_type } ;


(**)     { ------- string hash table routines ------- }


function make_and_enter_string(*name: alpha): string*);
  { convert an alpha into a string and enter it into the hash table }
  var
    temp_string: xtring;        { temporary string }
    original_string: xtring;    { original string value }
begin
  temp_string := nullstring;
  copy_to_string(name, temp_string);

  original_string := temp_string;
  temp_string := enter_string(temp_string);

  if temp_string <> original_string then release_string(original_string);

  make_and_enter_string := temp_string;
end { make_and_enter_string } ;


function enter_and_release_string(*str: string): string*);
  { enter the given string, and release the original pointer }
  var
    new_string: xtring;    { string from the table }
begin
  new_string := enter_string(str);

  if new_string <> str then release_string(str);

  enter_and_release_string := new_string;
end { enter_and_release_string } ;

  
function enter_string(*str: string): string*);
  { enter or find the string in the string hash table and return it.
    If not found in the table, the string's value is copied to a new
    string created afresh. }
  var
    i: string_range;           { index into STR }
    sum: natural_number;       { checksum of the string }
    index: hash_string_range;  { index into the string table }
    last,                      { last element checked in list }
    element: hash_string_ptr;  { element in the list of names }
    compare: compare_type;     { result of string compare }
    done: boolean;             { TRUE when place in table found }


  procedure insert_entry(list_element: hash_string_ptr);
    { insert a new entry after the given list element.  If the list element
      is NIL, insert at the head of the list. }
    var
      new_element: hash_string_ptr; { new element to be placed into the list }
  begin
    new(new_element);
    increment_heap_count(HEAP_HASH_STRINGS, 2*POINTER_SIZE);

    new_element^.str := nullstring;
    copy_string(str, new_element^.str);
    if list_element = NIL then
      begin
        new_element^.next_hash_string := string_table[index];
        string_table[index] := new_element;
      end
    else
      begin
        new_element^.next_hash_string := list_element^.next_hash_string;
        list_element^.next_hash_string := new_element;
      end;

    enter_string := new_element^.str;
  end { insert_entry } ;


begin { enter_string }
  if (str = NIL) then enter_string := nullstring
  else if ord(str^[0]) = 0 then enter_string := nullstring
  else
    begin
      { create a hash index from the specified name }

      sum := 0;  i := 0;
      for i := 1 to ord(str^[0]) do
         sum := sum + ord(str^[i]);

      index := sum MOD (HASH_STRING_TABLE_SIZE+1);

      element := string_table[index];
      if element = NIL then insert_entry(NIL)
      else
        begin
          last := NIL;  done := FALSE;
          repeat
            if element^.next_hash_string = NIL then
              begin
                done := TRUE;
                compare := compare_strings(str, element^.str);
              end
            else
              begin
                compare := compare_strings(str, element^.str);
                if compare <> GT then done := TRUE
                else
                  begin
                    last := element;  element := element^.next_hash_string;
                  end;
              end;
          until done;

          case compare of
            LT:  insert_entry(last);
            EQ:  enter_string := element^.str;
            GT:  insert_entry(element);
          end;
        end;
    end;
end { enter_string } ;


(**)     { ------- identifier name routines ------- }


function compare_identifiers(id1, id2: name_ptr): compare_type;
  { compare the 2 names and return the result }
begin
  if (id1 = NIL) or (id2 = NIL) then 
    begin
      assert(221 { nil name passed });
      writeln(cmplog, ' compare_identifiers');
      if id1 <> NIL then compare_identifiers := GT 
      else if id2 <> NIL then compare_identifiers := LT 
      else compare_identifiers := EQ;
    end
  else if (id1^.name < id2^.name) then compare_identifiers := LT
  else if (id1^.name = id2^.name) then compare_identifiers := EQ
  else compare_identifiers := GT;
end { compare_identifiers } ;


function enter_name(*name: alpha): name_ptr*);
  { enter or find the name in the name hash table and return a pointer }
  var
    i: 0..ID_LENGTH;           { index into NAME }
    sum: natural_number;       { checksum of the name }
    index: name_table_range;   { index into the name table }
    last,                      { last element checked in list }
    element: name_ptr;         { element in the list of names }
    done: boolean;             { TRUE when end of alpha found }


  procedure insert_entry(list_element: name_ptr);
    { insert a new entry after the given list element.  If the list element
      is NIL, insert at the head of the list. }
    var
      new_element: name_ptr;  { new element to be placed into the list }
  begin
    new(new_element);
    increment_heap_count(HEAP_NAME_ENTRY,
                         2*POINTER_SIZE+ALPHA_SIZE+2*INT_SIZE);

    new_element^.name := name;
    new_element^.kind := default_attributes;
    new_element^.definition := nullstring;
    new_element^.sy := NULLSY;

    if list_element = NIL then
      begin
        new_element^.next := name_table[index];
        name_table[index] := new_element;
      end
    else
      begin
        new_element^.next := list_element^.next;
        list_element^.next := new_element;
      end;

    enter_name := new_element;
  end { insert_entry } ;


begin { enter_name }
  { create a hash index from the specified name }

  sum := 0;  i := 0;  done := FALSE;
  while (i < ID_LENGTH) and not done do
    begin
      i := i + 1;
      if name[i] = ' ' then done := TRUE else sum := sum + ord(name[i]);
    end;
  index := sum MOD (name_table_size+1);

  element := name_table[index];
  if element = NIL then insert_entry(NIL)
  else
    begin
      last := NIL;
      while (name > element^.name) and (element^.next <> NIL) do
        begin  last := element;  element := element^.next;  end;
      if name = element^.name then enter_name := element
      else if name < element^.name then insert_entry(last)
      else insert_entry(element);
    end;
end { enter_name } ;


function name_from_string(s: xtring): name_ptr;
  { assume that resulting id is legal }
  var
    n: alpha;
begin
  if (s = NIL) then name_from_string := null_name
  else
    begin
      copy_from_string(s, n);
      name_from_string := enter_name(n);
    end;
end { name_from_string } ;


(**)     { ------- property utilities ------- }


procedure new_property(var p: property_ptr);
  { create a new property element }
begin
  if free_properties <> NIL then
    begin
      p := free_properties;  free_properties := p^.next;
      if not p^.free then
        begin
          writeln(Monitor, 'ASSERT: Non-free property in free list');
          p:=NIL; p^.next:=NIL;  { blow up }
        end;
    end
  else
    begin
      new(p);
      increment_heap_count(HEAP_PROPERTY, 3*POINTER_SIZE);
    end;

  p^.next := NIL;  p^.text := nullstring;
  p^.free := FALSE;
end { new_property } ;


procedure release_property(var p: property_ptr);
  { release a property element for later reuse }
begin
  if debug_29 then writeln(outfile, 'release_property ',ord(p):1);
  if p <> NIL then
    begin
      if p^.free then
        if debug_30 then
          begin
            writeln(Monitor, 'ASSERT: Releasing free property to free list');
            p := NIL;  p^.next := NIL; { blow up }
          end
        else p := NIL { and do nothing else -- a HORRIBLE band-aid }
      else
        begin
          p^.free := TRUE;
          p^.next := free_properties;  free_properties := p;  p := NIL;
        end;
    end;
end { release_property } ;


procedure release_entire_property_list(var list: property_ptr);
  { release every element of the given property list }
  var
    next: property_ptr;    { next property to be released }
begin
  while list <> NIL do
    begin
      next := list^.next;
      release_property(list);
      list := next;
    end;
end { release_entire_property_list } ;


procedure delete_property(var p: property_ptr);
  { releases p, returning p^.next }
  var
    next: property_ptr;  { saves return value }
begin
  next := p^.next;
  p^.next := free_properties;
  free_properties := p;
  p := next;
end { delete_property } ;


procedure add_to_prop_list(var prop_list: property_ptr;
                           property_name: name_ptr;
                           property_value: xtring);
  { add the given property to head of prop_list }
  var
    P: property_ptr;      { new property created }
begin
  new_property(P); 
  P^.name := property_name;  P^.text := property_value;
  P^.next := prop_list;  prop_list := P;
end { add_to_prop_list } ;


function find_property(*prop_list: property_ptr;  name: name_ptr;
                       var property: property_ptr): boolean*);
  { look for the given property in an unordered list of properties }
  var
    prop: property_ptr;       { current property element }
    found: boolean;           { TRUE when property found }
begin
  property := NIL;  prop := prop_list;  found := FALSE;
  while (prop <> NIL) and not found do
    if prop^.name = name then
      begin  property := prop;  found := TRUE;  end
    else  prop := prop^.next;

  find_property := found;
end { find_property } ;


function find_bit_property(prop_list: bit_property_ptr;  name: name_ptr;
                           var property: bit_property_ptr): boolean;
  { look for the given property in an unordered list of properties }
  var
    prop: bit_property_ptr;   { current property element }
    found: boolean;           { TRUE when property found }
begin
  property := NIL;  prop := prop_list;  found := FALSE;
  while (prop <> NIL) and not found do
    if prop^.name = name then
      begin  property := prop;  found := TRUE;  end
    else  prop := prop^.next;

  find_bit_property := found;
end { find_bit_property } ;


procedure check_and_add_to_prop_list(var dest: property_ptr;
                                     name: name_ptr;  text: xtring);
  { add the given property to the DEST property list.  If the property is
    already in the list, add it only if the property values are different. }
  var
    prop: property_ptr;    { property returned from search }
    skip_it: boolean;      { TRUE iff same name, value alread in list }
    found: boolean;        { Prevents extra call to find_property }
begin
  prop := dest;  skip_it := FALSE;
  repeat
    found := find_property(prop, name, prop);
    if found then
      if prop^.text = text then skip_it := TRUE
                           else prop := prop^.next;
  until skip_it or not found;
  if not skip_it then add_to_prop_list(dest, name, text);
end { check_and_add_to_prop_list } ;


function end_of_property_list(list: property_ptr): property_ptr;
  { find the end of the list and return a pointer to it }
  var
    PP,                         { current property list element }
    last: property_ptr;         { previous property list element }
begin
  PP := list;  last := NIL;
  while PP <> NIL do
    begin  last := PP;  PP := PP^.next;  end;
  end_of_property_list := last;
end { end_of_property_list } ;


function compare_properties(list1, list2: property_ptr): compare_type;
  { compare the two property lists and return a compare type: EQ if
    the property lists are the same; GT if they are not }
  var
    p1, p2: property_ptr;      { current properties for compare }
    equal_so_far: boolean;     { TRUE if property lists are equal so far }
begin
  p1 := list1;  p2 := list2;  compare_properties := GT;  equal_so_far := TRUE;
  while (p1 <> NIL) and (p2 <> NIL) and equal_so_far do
    if p1^.name <> p2^.name then equal_so_far := FALSE
    else if p1^.text <> p2^.text then equal_so_far := FALSE
    else
      begin  p1 := p1^.next;  p2 := p2^.next;  end;
  if equal_so_far then
    if p1 <> p2 then compare_properties := GT
                else compare_properties := EQ;
end { compare_properties } ;


procedure copy_properties(source_list: property_ptr;
                          var dest_list: property_ptr);
  { copy the source list to the destination list }
  var
    source: property_ptr;      { source property to copy }
begin
  source := source_list;
  while source <> NIL do
    begin
      add_to_prop_list(dest_list, source^.name, source^.text);
      source := source^.next;
    end;
end { copy_properties } ;


procedure copy_inherit_pin_properties(var dest_list: property_ptr;
                                      source_list: property_ptr);
  { copy properties from the source list to the dest list if they have the
    INHERIT_PIN attribute.  Duplicate properties in the dest list are OK. }
  var
    source_prop: property_ptr;       { current source property }
begin
  source_prop := source_list;
  while source_prop <> NIL do
    begin
      if INHERIT_PIN IN source_prop^.name^.kind then
        add_to_prop_list(dest_list, source_prop^.name, source_prop^.text);

      source_prop := source_prop^.next;
    end;
end { copy_inherit_pin_properties } ;


procedure copy_specific_properties(which: name_type_set;
                                   var dest: property_ptr;
                                   source: property_ptr);
  { copy only those properties with the specified attributes from the SOURCE
    property list to the DEST property list.  If more than one attribute is
    specified, any attribute present will cause the property to be copied.
    If the property already exists, it is not copied again if its value is
    the same as the one already present. }
  var
    current,                  { current property to be copied }
    prop: property_ptr;       { property returned from search }
begin
  current := source;
  while current <> NIL do
    begin
      if (current^.name^.kind * which) <> [] then
        if find_property(dest, current^.name, prop) then
          if prop^.text = current^.text then
            { do not copy, leave as is }
          else
            add_to_prop_list(dest, current^.name, current^.text)
        else
          add_to_prop_list(dest, current^.name, current^.text);
      current := current^.next;
    end;
end { copy_specific_properties } ;


procedure dump_property_list(var f: textfile; list: property_ptr);
  { dump the given property list to the given file }
  var
    PP: property_ptr;      { current property to be output }
begin
  PP := list;
  while PP <> NIL do
    with PP^ do
      begin
        write(f, '    ');
        writealpha(f, name^.name);
        write(f, '=');
        writestring(f, text);
        if next = NIL then writeln(f, ';') else writeln(f, ',');

        PP := next;
      end;

  if list = NIL then writeln(f);     { this is a KLUDGE! }
end { dump_property_list } ;


(**)     { ------- heap management routines ------- }


{----------------------------------------------------------------------------}
{  The NEW procedures fall into two types:  those that create a new element  }
{  of the particular type and return it, and those that create the element   }
{  and return it as the first element of the list passed in.  Every NEW      }
{  procedure increments a count of the heap space used so that a structure   }
{  by structure accounting can be reported.                                  }
{                                                                            }
{  The RELEASE procedures take the given element and add it to a list of     }
{  free elements.  The associated NEW procedures will remove the element     }
{  from the free list rather than creating one from the heap.  The RELEASE   }
{  procedures come in three forms:  RELEASE_<structure> releases the single  }
{  element adn adds it to the free list.  RELEASE_ENTIRE_<structure> adds    }
{  all of the elements in a list of elements headed by the given element.    }
{  RELEASE_COMPLETE_<structure> releases all elements of the list headed     }
{  by the given element and all of the fields of the element as well.  This  }
{  never includes STRINGs, NAME_PTRs, SIGNAL_DEFINITIONs,                    }
{  or SIGNAL_INSTANCEs.                                                      }
{  Some of the RELEASE <structure> routines delete the passed element from   }
{  its list (in other words, they return the old value of element^.next)     }
{  others don't do this.  Some of these also release all of their fields     }
{  (with the same exceptions as the RELEASE_COMPLETE) routines.              }
{----------------------------------------------------------------------------}


procedure new_subscript(var subscript_list: subscript_ptr);
  { create a new subscript element and add to head of the list }
  var
    SP: subscript_ptr;     { new subscript being created }
begin
  if free_subscripts <> NIL then
    begin  SP := free_subscripts;  free_subscripts := SP^.next;  end
  else
    begin
      new(SP);  increment_heap_count(HEAP_SUBSCRIPT, POINTER_SIZE+2*INT_SIZE);
    end;

  SP^.left_index := 0;
  SP^.right_index := 0;
  SP^.next := subscript_list;  subscript_list := SP;
end { new_subscript } ;


procedure release_subscript(var sub: subscript_ptr);
  { release a subscript element for later reuse }
begin
  if sub <> NIL then
    begin
      sub^.next := free_subscripts;  free_subscripts := sub;  sub := NIL;
    end;
end { release_subscript } ;


procedure release_entire_subscript(var sub: subscript_ptr);
  { release all the subscripts in the list }
  var
    next: subscript_ptr;    { next subscript to release }
begin
  while sub <> NIL do
    begin  next := sub^.next;  release_subscript(sub);  sub := next;  end;
end { release_entire_subscript } ;


procedure new_subscript_property(var list: subscript_property_ptr);
  { return a new element and add to the beginning of the list }
  var
    sub: subscript_property_ptr;     { new element being created }
begin
  if free_subscript_properties <> NIL then
    begin
      sub := free_subscript_properties;
      free_subscript_properties := sub^.next;
    end
  else
    begin
      new(sub);
      increment_heap_count(HEAP_SUBSCRIPT_PROPERTY,
                           2*POINTER_SIZE+2*INT_SIZE);
    end;

  sub^.left_index := -1;
  sub^.right_index := -1;
  sub^.properties := NIL;
  sub^.next := list;  list := sub;
end { new_subscript_property } ;


procedure release_subscript_property(var sub: subscript_property_ptr);
  { add the given element to the free list }
begin
  if sub <> NIL then
    begin
      sub^.next := free_subscript_properties;
      free_subscript_properties := sub;
      sub := NIL;
    end;
end { release_subscript_properties } ;


procedure release_entire_subscript_property(var sub: subscript_property_ptr);
  { add the given element to the free list }
begin
  if sub <> NIL then
    begin
      sub^.next := free_subscript_properties;
      free_subscript_properties := sub;
      sub := NIL;
    end;
end { release_entire_subscript_property } ;


procedure new_bit_property(var list: bit_property_ptr);
  { create a new element and add to the beginning of the list }
  var
    prop: bit_property_ptr;     { new element to be created and returned }
begin
  if free_bit_properties <> NIL then
    begin
      prop := free_bit_properties;  free_bit_properties := prop^.next;
    end
  else
    begin
      new(prop);
      increment_heap_count(HEAP_BIT_PROPERTY, 4*POINTER_SIZE);
    end;

  prop^.bit_subscript := NIL;
  prop^.name := null_name;
  prop^.text := nullstring;
  prop^.next := list;  list := prop;
end { new_bit_property } ;


procedure release_bit_property(var prop: bit_property_ptr);
  { add the given element to the free list }
begin
  if prop <> NIL then
    begin
      prop^.next := free_bit_properties;
      free_bit_properties := prop;
      prop := NIL;
    end;
end { release_bit_property } ;


procedure release_entire_bit_property_list(var list: bit_property_ptr);
  { release every element of the given property list }
  var
    next: bit_property_ptr;    { next property to be released }
begin
  while list <> NIL do
    begin
      next := list^.next;  release_bit_property(list);  list := next;
    end;
end { release_entire_bit_property_list } ;


procedure new_basescript(var basescript_list: basescript_ptr);
  { create a new basescript element and add to head of the list }
  var
    BSP: basescript_ptr;     { new subscript being created }
begin
  if free_basescripts <> NIL then
    begin  BSP := free_basescripts;  free_basescripts := BSP^.next;  end
  else
    begin
      new(BSP);
      increment_heap_count(HEAP_BASESCRIPT, 2*POINTER_SIZE + 3*INT_SIZE);
    end;

  with BSP^ do
    begin
      next := basescript_list;
      instance := NIL;
      left_index := 0;
      right_index := 0;
      offset := 0;
    end;
  basescript_list := BSP;
end { new_basescript } ;


procedure release_basescript(var BSP: basescript_ptr);
  { release a basescript element for later reuse }
begin
  if BSP <> NIL then
    begin
      BSP^.next := free_basescripts;  free_basescripts := BSP;  BSP := NIL;
    end;
end { release_basescript } ;


procedure release_entire_basescript_list(var list: basescript_ptr);
  { release all of the basescripts in the given list }

  var
    next: basescript_ptr;     { next element in the list }

begin
  while list <> NIL do
    begin
      next := list^.next;
      release_basescript(list);
      list := next;
    end;
end { release_entire_basescript_list } ;


procedure new_base_descriptor(var BD: base_descriptor_ptr);
  { create a new base descriptor }
begin
  if free_base_descriptors <> NIL then
    begin
      BD := free_base_descriptors;  free_base_descriptors := BD^.next;
    end
  else
    begin
      new(BD);
      increment_heap_count(HEAP_BASE_DESCRIPTOR, 2*POINTER_SIZE+2*INT_SIZE);
    end;

  with BD^ do
    begin
      width := 0;
      offset := 0;
      next := NIL;
      instance := NIL;
    end;
end { new_base_descriptor } ;


procedure release_base_descriptor(var BD: base_descriptor_ptr);
  { release a base descriptor for later reuse }
begin
  if BD <> NIL then
    begin
      BD^.next := free_base_descriptors;  free_base_descriptors := BD;
      BD := NIL;
    end;
end { release_basescript } ;


procedure new_signal_descriptor(var signal: signal_descriptor_ptr);
  { create a new signal descriptor }
begin
  if free_signal_descriptors <> NIL then
    begin
      signal := free_signal_descriptors;
      free_signal_descriptors := signal^.next;
    end
  else
    begin
      new(signal);
      increment_heap_count(HEAP_SIGNALDESCR,
                           5*POINTER_SIZE+2*BOOL_SIZE+4*INT_SIZE);
    end;

  with signal^ do
    begin
      next := NIL;
      signal_name := nullstring;
      polarity := NORMAL;
      low_asserted := FALSE;
      scope := UNKNOWN_SCOPE;
      replication_factor := 1;
      kind := UNDEFINED;
      bit_subscript := NIL;
      properties := NIL;
      net_id := nullstring;
      is_const := FALSE;
    end;
end { new_signal_descriptor } ;


procedure release_signal_descriptor(var signal: signal_descriptor_ptr);
  { release a signal descriptor for later reuse }
begin
  if signal <> NIL then
    begin
      signal^.next := free_signal_descriptors;
      free_signal_descriptors := signal;
      signal := NIL;
    end;
end { release_signal_descriptor } ;


procedure release_entire_signal_descriptor(var list: signal_descriptor_ptr);
  { release all signal descriptors in a concatenated list }
  var
    next,                            { next signal to release }
    sig: signal_descriptor_ptr;      { signal to be released }
begin
  sig := list;
  while sig <> NIL do
    begin
      next := sig^.next;  release_signal_descriptor(sig);  sig := next;
    end;
end { release_entire_signal_descriptor } ;


procedure release_complete_signal_descriptor(var sig: signal_descriptor_ptr);
  { release the signal and all of its fields }
begin
  release_entire_property_list(sig^.properties);
  release_entire_subscript(sig^.bit_subscript);
  release_signal_descriptor(sig);
end { release_complete_signal_descriptor } ;


procedure release_complete_SD_list(var sig: signal_descriptor_ptr);
  { release the signal and all of its fields }
  var
    next: signal_descriptor_ptr;
begin
  while sig <> NIL do
    begin
      release_entire_property_list(sig^.properties);
      release_entire_subscript(sig^.bit_subscript);
      next := sig^.next;
      release_signal_descriptor(sig);
      sig := next;
    end;
end { release_complete_SD_list } ;


procedure new_simple_signal(var signal: simple_signal_ptr);
  { create a new simple signal on the heap and return it }
begin
  if free_simple_signals = NIL then
    begin
      new(signal);
      increment_heap_count(HEAP_SIMPLE_SIGNAL, 3*POINTER_SIZE+2*INT_SIZE);
    end
  else
    begin
      signal := free_simple_signals;  free_simple_signals := signal^.next;
    end;

  with signal^ do
    begin
      polarity := NORMAL;
      signal_name := nullstring;
      kind := SINGLE;
      bit_subscript := NIL;
      next := NIL;
    end;
end { new_simple_signal } ;


procedure release_simple_signal(var signal: simple_signal_ptr);
  { add the given signal list to the free list }
begin
  if signal <> NIL then
    begin
      signal^.next := free_simple_signals;  free_simple_signals := signal;
      signal := NIL;
    end;
end { release_simple_signals } ;


procedure new_signal_entry(var entry: signal_entry_ptr);
  { create a new signal entry }
begin
  if free_signal_entrys = NIL then
    begin
      new(entry);
      increment_heap_count(HEAP_SIGNALENTRY, 4*POINTER_SIZE);
    end
  else
    begin
      entry := free_signal_entrys;
      free_signal_entrys := free_signal_entrys^.next;
    end;

  entry^.name := nullstring;
  entry^.high_asserted := NIL;
  entry^.low_asserted := NIL;
  entry^.next := NIL;
end { new_signal_entry } ;


procedure new_signal_definition(var sig: signal_definition_ptr);
  { create a new signal definition and return it }
begin
  if free_signal_definitions <> NIL then
    begin
      sig := free_signal_definitions;
      free_signal_definitions := sig^.next;
    end
  else
    begin
      new(sig);
      increment_heap_count(HEAP_SIGNAL_DEFINITION,
                           9*POINTER_SIZE+3*BOOL_SIZE+6*INT_SIZE);
    end;

  with sig^ do
    begin
      next := NIL;
      stack := NIL;
      signal := NIL;
      is_virtual_base := TRUE;
      next_virtual_def := NIL;
      net_id := nullstring;
      left_index := 0;
      right_index := 0;
      properties := NIL;
      polarity := NORMAL;
      instances := NIL;
      kind := UNDEFINED;
      synonym_bits := NIL;
      is_const := FALSE;
      if scope_is_local then scope := LOCAL else scope := GLOBAL;
      node := NIL;
    end;
end { new_signal_definition } ;


procedure release_signal_definition(var SDP: signal_definition_ptr);
  { release the specified signal definition and add to the free list }
begin
  if SDP <> NIL then
    begin
      SDP^.next := free_signal_definitions;
      free_signal_definitions := SDP;
    end;

  SDP := NIL;
end { release_signal_definition } ;


procedure new_signal_definition_list(var list: signal_definition_list_ptr);
  { create a new signal definition list element and add to head of list }
  var
    SDLP: signal_definition_list_ptr;    { signal definition being created }
begin
  if free_signal_definition_lists <> NIL then
    begin
      SDLP := free_signal_definition_lists; 
      free_signal_definition_lists := SDLP^.next;
    end
  else
    begin
      new(SDLP);
      increment_heap_count(HEAP_SIGNAL_DEFINITION_LIST, 2*POINTER_SIZE);
    end;

  SDLP^.definition := NIL;
  SDLP^.next := list;  list := SDLP;
end { new_signal_definition_list } ;


procedure release_signal_definition_list(var SDLP: signal_definition_list_ptr);
  { release the signal definition list element and add to the free list }
begin
  if SDLP <> NIL then
    begin
      SDLP^.next := free_signal_definition_lists;
      free_signal_definition_lists := SDLP;
      SDLP := NIL;
    end;
end { release_signal_definition_list } ;


procedure release_entire_signal_definition_list
                                       (var list: signal_definition_list_ptr);
  { release all of the elements of the given list }
  var
    next: signal_definition_list_ptr;    { next in the list }
begin
  while list <> NIL do
    begin
      next := list^.next;
      release_signal_definition_list(list);
      list := next;
    end;
end { release_entire_signal_definition_list } ;


procedure new_signal_instance(var instance: signal_instance_ptr);
  { create a new signal instance }
begin
  if free_signal_instances <> NIL then
    begin
      instance := free_signal_instances;
      free_signal_instances := instance^.next;
    end
  else
    begin
      new(instance);
      increment_heap_count(HEAP_SIGNAL_INSTANCE, 3*POINTER_SIZE
                           +BOOL_SIZE+INT_SIZE);
    end;

  with instance^ do
    begin
      next := NIL;
      defined_by := NIL;
      replication_factor := 1;
      bit_subscript := NIL;
      low_asserted := FALSE;
    end;
end { new_signal_instance } ;


procedure release_signal_instance(var instance: signal_instance_ptr);
  { add the given signal signal instance to the free list }
begin
  if instance <> NIL then
    begin
      instance^.next := free_signal_instances;
      free_signal_instances := instance;
      instance := NIL;
    end;
end { release_signal_instance } ;


procedure release_complete_signal_def(var def: signal_definition_ptr);
  { free all the signal instances of the given def.  Release the properties
    of each instance as well as the subscripts.   Release the signal def and
    its properties.  If def is virtual base then release all virtual defs }
  var
    instance,                    { current signal instance being released }
    next: signal_instance_ptr;   { next signal instance to be released }
    vdef: signal_definition_ptr;      { current def/virtual def for release }
    next_vdef: signal_definition_ptr; { next virtual def }

begin
  if def <> NIL then
    begin
      if def^.is_virtual_base then
        begin
          vdef := def^.next_virtual_def;
          while vdef <> NIL do
            begin
              next_vdef := vdef^.next;
              vdef^.is_virtual_base := FALSE; { redundant }
              release_complete_signal_def(vdef);
              vdef := next_vdef;
            end;
        end;
      instance := def^.instances;
      while instance <> NIL do
        begin
          release_entire_subscript(instance^.bit_subscript);

          next := instance^.next;
          release_signal_instance(instance);
          instance := next;
        end;

      def^.instances := NIL;

      release_entire_basescript_list(def^.synonym_bits);
      release_entire_bit_property_list(def^.properties);
      release_signal_definition(def);
    end;
end { release_complete_signal_def } ;


procedure new_synonym_signal(var list: synonym_signal_ptr);
  { create a new signal on the heap and add to the head of the list }
  var
    signal: synonym_signal_ptr;     { new signal to be created }
begin
  if free_synonym_signals = NIL then
    begin
      new(signal);
      increment_heap_count(HEAP_SYNONYM_SIGNAL, 4*POINTER_SIZE+ENUM_SIZE);
    end
  else
    begin
      signal := free_synonym_signals;
      free_synonym_signals := signal^.next;
    end;

  with signal^ do
    begin
      signal_name := nullstring;
      polarity := NORMAL;
      def := NIL;
      used_bits := NIL;
      next := list;  list := signal;
    end;
end { new_synonym_signal } ;


procedure release_synonym_signal(var signal: synonym_signal_ptr);
  { release the given signal (and its used_bits list) and add 
    to the free list }
begin
  if signal <> NIL then
    begin
      release_entire_subscript(signal^.used_bits);
      signal^.next := free_synonym_signals;
      free_synonym_signals := signal;
      signal := NIL;
    end;
end { release_synonym_signal } ;


procedure new_formal_actual_list(var list: formal_actual_ptr);
  { create a new formal/actual list element and add to list }
  var
    FAP: formal_actual_ptr;     { new formal/actual list element }
begin
  if free_formal_actual_lists <> NIL then
    begin
      FAP := free_formal_actual_lists;  free_formal_actual_lists := FAP^.next;
    end
  else
    begin
      new(FAP);
      { assume 2 booleans together are efficiently allocated to share
        the same word }
      increment_heap_count(HEAP_FORMAL_ACTUAL_LIST,
                           5*POINTER_SIZE+2*INT_SIZE+BOOL_SIZE);
    end;

  with FAP^ do
    begin
      next := list;  list := FAP;
      formal_parameter := NIL;
      pin_name := NIL;
      polarity := NORMAL;
      actual_parameter := NIL;
      width := 0;
      properties := NIL;
      uses_NAC := FALSE;
      is_NWC_pin := FALSE;
    end;
end { new_formal_actual_list } ;


procedure release_formal_actual_list(var FAP: formal_actual_ptr);
  { add the given formal/actual list to the free list }
begin
  if FAP <> NIL then
    begin
      FAP^.next := free_formal_actual_lists;
      free_formal_actual_lists := FAP;
      FAP := NIL;
    end;
end { release_formal_actual_list } ;


procedure new_clear_text_actual_list(var list: clear_text_actual_list_ptr);
  { create a new clear text actual list element and add to the given list }
  var
    CTALP: clear_text_actual_list_ptr;       { new element being created }
begin
  new(CTALP);
  increment_heap_count(HEAP_CLEAR_TEXT_ACTUAL_LIST, 4*POINTER_SIZE);

  with CTALP^ do
    begin
      actual_parameter := nullstring;
      properties := NIL;
      net_id := nullstring;
      next := list;  list := CTALP;
    end;
end { new_clear_text_actual_list } ;


procedure new_actual_list(var list: actual_list_ptr);
  { create a new element and add to the head of the given list }
  var
    ALP: actual_list_ptr;      { new element being created }
begin
  if free_actual_lists = NIL then
    begin
      new(ALP);
      increment_heap_count(HEAP_ACTUAL_LIST,
                           2*POINTER_SIZE+INT_SIZE+BOOL_SIZE);
    end
  else
    begin
      ALP := free_actual_lists;  free_actual_lists := ALP^.next;
    end;

  ALP^.signal := NIL;
  ALP^.width_is_unknown := FALSE;
  ALP^.assertion_state := ASSERTION_KNOWN;
  ALP^.next := list;  list := ALP;
end { new_actual_list } ;


procedure release_actual_list(var ALP: actual_list_ptr);
  { release the given element }
begin
  if ALP <> NIL then
    begin
      ALP^.next := free_actual_lists;
      free_actual_lists := ALP;
      ALP := NIL;
    end;
end { release_actual_list } ;


procedure new_bindings_list(var bindings_list: bindings_list_ptr);
{ create a new bindings element and add to head of the list }
  var
    BLP: bindings_list_ptr;     { element being created }
begin
  if free_bindings_lists = NIL then
    begin
      new(BLP);
      increment_heap_count(HEAP_BINDINGSLIST, 4*POINTER_SIZE);
    end
  else
    begin
      BLP := free_bindings_lists;
      free_bindings_lists := free_bindings_lists^.next;
    end;
  with BLP^ do
    begin
      formal_parameter := nullstring;
      pin_properties := NIL;
      actual_parameter := NIL;
      next := bindings_list;  bindings_list := BLP;
    end;
end { new_bindings_list } ;


procedure new_invoke_list(var invoke_list: invoke_list_ptr);
  { create a new invoke element and add to the head of the list }
  var
    ILP: invoke_list_ptr;       { element being created }
begin
  if free_invoke_lists <> NIL then
    begin  ILP := free_invoke_lists;  free_invoke_lists := ILP^.next;  end
  else
    begin
      new(ILP);
      increment_heap_count(HEAP_INVOKELIST, 6*POINTER_SIZE+INT_SIZE);
    end;

  with ILP^ do
    begin
      macro_name := nullstring;
      parameters := NIL;
      properties := NIL;
      bindings := NIL;
      path := nullstring;
      page_number := 1;
    end;

  ILP^.next := invoke_list;  invoke_list := ILP;
end { new_invoke_list } ;


procedure release_invoke_list(var list: invoke_list_ptr);
  { release the given invoke list and add to the free list }
begin
  if list <> NIL then
    begin
      list^.next := free_invoke_lists;
      free_invoke_lists := list;
      list := NIL;
    end;
end { release_invoke_list } ;


procedure new_signal_list(var signal_list: signal_list_ptr);
  { create a new signal list element and add to the head of the list }
  var
    SLP: signal_list_ptr;       { element being created }
begin
  if free_signal_lists = NIL then
    begin
      new(SLP);  increment_heap_count(HEAP_SIGNALLIST, 2*POINTER_SIZE);
    end
  else
    begin
      SLP := free_signal_lists;
      free_signal_lists := free_signal_lists^.next;
    end;

  SLP^.signal_name := nullstring;
  SLP^.next := signal_list;  signal_list := SLP;
end { new_signal_list } ;

    
procedure new_macro_def(var MDP: macro_def_ptr);
  { create a new macro definition }
begin
  if free_macro_defs = NIL then
    begin
      new(MDP);
      increment_heap_count(HEAP_MACRODEF, 9*POINTER_SIZE+2*BOOL_SIZE);
    end
  else
    begin
      MDP := free_macro_defs;
      free_macro_defs := free_macro_defs^.next;
    end;

  with MDP^ do
    begin
      next := NIL;
      macro_name := nullstring;
      version := NIL;
      occurances := NIL;
      params := NIL;
      properties := NIL;
      text_macros := NIL;
      written_with_GED := TRUE;
      is_leaf_macro := FALSE;
      invokes := NIL;
    end;
end { new_macro_def } ;
        

procedure new_mtree_node(var mtree_list: mtree_node_ptr);
  { create a new mtree node and add to the head of the list }
  var
    new_mtree: mtree_node_ptr;     { node being created }
begin 
  if free_mtree_nodes = NIL then
    begin
      new(new_mtree);
      increment_heap_count(HEAP_MTREENODE, 
                           10*POINTER_SIZE+3*BOOL_SIZE+2*INT_SIZE);
    end
  else
    begin
      new_mtree := free_mtree_nodes;
      free_mtree_nodes := free_mtree_nodes^.next;
    end;

  total_number_nodes := total_number_nodes + 1;

  with new_mtree^ do
    begin
      macro := NIL;
      macro_name := nullstring;
      called_by := NIL;
      next_node_with_same_mdef := NIL;
      father_node := NIL;
      level := 0;
      x_value := 0;
      params := NIL;
      signals := NIL;
      symbol_table := NIL;
      is_plumbing_node := FALSE;
      is_cardinal_tap := FALSE;
      is_leaf_node := FALSE;
      uses_SIZE_property := FALSE;
      son := NIL;
    end;

  new_mtree^.next := mtree_list;  mtree_list := new_mtree;
end { new_mtree_node } ;


procedure new_propertied_CS(var PCSP: propertied_CS_ptr);
  { create a new propertied concatenated signal return it }
begin
  if free_propertied_CSs <> NIL then
    begin
      PCSP := free_propertied_CSs;
      free_propertied_CSs := PCSP^.next;
    end
  else
    begin
      new(PCSP);
      increment_heap_count(HEAP_PROPERTIED_CONCATSIG,
                           3*POINTER_SIZE+INT_SIZE);
    end;

  PCSP^.next := NIL;
  PCSP^.instance := NIL;
  PCSP^.properties := NIL;
  PCSP^.control := NORMAL_SIGNAL;
end { new_propertied_CS } ;


procedure release_propertied_CS(var CSP: propertied_CS_ptr);
  { release a propertied concatenated signal for later reuse }
begin
  if CSP <> NIL then
    begin
      CSP^.next := free_propertied_CSs;
      free_propertied_CSs := CSP;
      CSP := NIL;
    end;
end { release_propertied_CS } ;


procedure release_entire_propertied_CS(var list: propertied_CS_ptr);
  { release all elements in a propertied concatenated signal list }
  var
    next: propertied_CS_ptr;    { next element to be released }
begin
  while list <> NIL do
    begin
      next := list^.next;
      release_propertied_CS(list);  list := next;
    end;
end { release_entire_propertied_CS } ;


procedure release_complete_propertied_CS(var list: propertied_CS_ptr);
  { release all elements in the list as well as its fields }
  var
    next_signal: propertied_CS_ptr;    { next signal in the list }
begin
  while list <> NIL do
    begin
      release_entire_property_list(list^.properties);

      next_signal := list^.next;
      release_propertied_CS(list);
      list := next_signal;
    end;
end { release_complete_propertied_CS } ;


procedure release_complete_actual_list(var list: actual_list_ptr);
  { release all elements in the given list and all of its fields as well }

  var
    next_actual: actual_list_ptr;    { next actual in the list }
begin
  while list <> NIL do
    begin
      release_complete_propertied_CS(list^.signal);

      next_actual := list^.next;
      release_actual_list(list);
      list := next_actual;
    end;
end { release_complete_actual_list } ;


procedure release_complete_formal_actual_list(var list: formal_actual_ptr);
  { release all of the formal actual elements in the given list and all of
    its fields as well. }
  var
    next_formal_actual: formal_actual_ptr;  { next pair in list }
begin
  while list <> NIL do
    begin
      release_complete_actual_list(list^.actual_parameter);

      release_entire_subscript_property(list^.properties);

      next_formal_actual := list^.next;
      release_formal_actual_list(list);
      list := next_formal_actual;
    end;
end { release_complete_formal_actual_list } ;


procedure new_file_list(var list: file_list_ptr);
  { create a new element and add to the head of the list }
  var
    FLP: file_list_ptr;     { new element created for the list }
begin
  new(FLP);
  increment_heap_count(HEAP_FILE_LIST, 2*POINTER_SIZE);

  FLP^.next := list;  list := FLP;
  FLP^.file_name := nullstring;
end { new_file_list } ;


procedure new_net_descriptor(var net: net_descriptor_ptr);
  { create a new net descriptor, init, and return }
begin
  if free_net_descriptors <> NIL then
    begin
      net := free_net_descriptors;
      free_net_descriptors := net^.next;
    end
  else
    begin
      new(net);
      increment_heap_count(HEAP_NET_DESCRIPTOR, 4*POINTER_SIZE);
    end;

  net^.next := NIL;
  net^.net_name := nullstring;
  net^.properties := NIL;
  net^.net_id := nullstring;
end { new_net_descriptor } ;


procedure release_net_descriptor(var net: net_descriptor_ptr);
  { release the given net and add to free list }
begin
  if net <> NIL then
    begin
      net^.next := free_net_descriptors;
      free_net_descriptors := net;
      net := NIL;
    end;
end { release_net_descriptor } ;


procedure new_net_table(var table: net_table_ptr);
  { create a new net table type, init, insert, and return }
  var
    i: net_group_range;      { index into the net table }
    newone: net_table_ptr;   { new element }
begin
  if free_net_tables <> NIL then
    begin
      newone := free_net_tables;
      free_net_tables := newone^.next;
    end
  else
    begin
      new(newone);
      increment_heap_count(HEAP_NET_TABLE, (NET_GROUP_SIZE+3)*POINTER_SIZE);
    end;

  newone^.next := table;  table := newone;
  table^.group_number := 0;
  for i := 0 to NET_GROUP_SIZE do
    table^.nets[i] := NIL;
end { new_net_table } ;


procedure release_net_table(var table: net_table_ptr);
  { release the given net table }
begin
  if table <> NIL then
    begin
      table^.next := free_net_tables;
      free_net_tables := table;
      table := NIL;
    end;
end { release_net_table } ;


procedure release_entire_net_table(var table: net_table_ptr);
  { release all elements in the given table and add to free list }
  var
    index: net_group_range;    { index into the net table }
    next: net_table_ptr;       { next table in the list }
begin
  while table <> NIL do
    begin
      next := table^.next;
      for index := 0 to NET_GROUP_SIZE do
        release_net_descriptor(table^.nets[index]);
      release_net_table(table);

      table := next;
    end;
end { release_entire_net_table } ;


(**)     { ------- symbol table routines ------- }


function found_id(node: mtree_node_ptr; name: name_ptr;
                  var id: identifier_ptr): boolean;
  { look for the given id (NAME) in the symbol table of the given node (NODE)
    and return a pointer to it if found.  Return TRUE if found.  Return FALSE
    if not present in the table and return ID = NIL.  It is assumed that the
    symbol table is not sorted in any way. }
  var
    current_id: identifier_ptr;  { current ID in the symbol table }
    found: boolean;              { TRUE if name found in symbol table }
begin
  id := NIL;  found := FALSE;
  if node <> NIL then
    begin
      current_id := node^.symbol_table;
      while (current_id <> NIL) and not found do
        if current_id^.name = name then found := TRUE
        else
          current_id := current_id^.next;

      id := current_id;
    end;

  found_id := found;
end { found_id } ;
      
  
procedure enter_id(node: mtree_node_ptr; name: name_ptr;
                   var id: identifier_ptr);
  { enter the given name (NAME) into the symbol table of the given node
    (NODE).  Return a pointer to the symbol table entry (ID).  If the name
    is already in the table, return ID = NIL. }
begin
  if found_id(node, name, id) then id := NIL
  else
    begin
      new_identifier(id);
      id^.name := name;
      id^.next := node^.symbol_table;
      node^.symbol_table := id;
    end;
end { enter_id } ;


function search_id(id_name: name_ptr): xtring;
  { search the current symbol table (CURRENT_MTREE_NODE) for the given name.
    If name is not found, return nullstring else return a pointer to its
    value. }
  var
    id: identifier_ptr;            { entry in symbol table }
    val: xtring;                   { return value }


  function add_default_SIZE_to_symbol_table: xtring;
    { report an error that the SIZE property was not found and add its
      default value to the given symbol table.  Also, if this is not the
      root drawing, add SIZE to the parameter list of the invoke (so that
      it will become part of the context of the invoke).  Issue the
      appropriate message. }
    var
      id: identifier_ptr;     { entry in the symbol table }
      curr: mtree_node_ptr;   { follows path up to root }
  begin
    enter_id(current_mtree_node, SIZE_prop_name, id);

    id^.definition := default_SIZE_string;

    { If the drawing is size replicated then we only need to handle the
      error message and parameter addition for one replication of the node;
      so we pick the direct descendents of mtree_root for this treatment. }

    curr := current_mtree_node;
    while curr^.father_node <> NIL do curr := curr^.father_node;
    if curr = mtree_root then
      if (current_mtree_node^.called_by <> NIL) then
        begin
          add_to_prop_list(current_mtree_node^.called_by^.parameters,
                           SIZE_prop_name, default_SIZE_string);
	  if not selecting_module then
            begin
              error(196 { SIZE property was not found });
              error_dump_current_parse_environment;
	    end;
        end
      else 
        if not selecting_module then
	  error(108 { No SIZE in context for SIZEd drawing });
	  { don't emit message while processing selection expressions }

    add_default_SIZE_to_symbol_table := default_SIZE_string;
  end { add_default_SIZE_to_symbol_table } ;


begin { search_id }
  val := nullstring;

  if found_id(current_mtree_node, id_name, id) then val := id^.definition

  else if parameter_attributes * id_name^.kind <> [] then
    begin
      if id_name = SIZE_prop_name then
        val := add_default_SIZE_to_symbol_table;
    end;

  if val = nullstring then
    if UNRESERVED in id_name^.kind then val := id_name^.definition;

  { if this ID is SIZE then flag same in current symbol table }

  if id_name = SIZE_prop_name then
    current_mtree_node^.uses_SIZE_property := TRUE;

  search_id := val;
end { search_id } ;


#include "parseutil.p"
#include "numbtoken.p"


(**)     { ------- display error summaries ------- }


procedure display_error_summaries;
  { display error information for the entire compile on each output file }


  procedure display_error_message(var f: textfile);
    { display the error total with attention to number }
  begin
    if num_errors = 0 then
      writeln(f, ' No errors detected')
    else if num_errors = 1 then
      writeln(f, ' 1 error detected')
    else
      writeln(f, ' ', num_errors:1, ' errors detected');
  end { display_error_message } ;


  procedure display_warning_message(var f: textfile);
    { display the warning total with attention to number }
  begin
    if num_warnings = 0 then
      writeln(f, ' No warnings detected')
    else if num_warnings = 1 then
      writeln(f, ' 1 warning detected')
    else
      writeln(f, ' ', num_warnings:1, ' warnings detected');
  end { display_warning_message } ;


  procedure display_oversight_message(var f: textfile);
    { display the oversight total with attention to number }
  begin
    if num_oversights = 0 then
      writeln(f, ' No oversights detected')
    else if num_oversights = 1 then
      writeln(f, ' 1 oversight detected')
    else
      writeln(f, ' ', num_oversights:1, ' oversights detected');
  end { display_oversight_message } ;


  procedure display_baseball_message(var f: textfile);
    { display a Baseball message if appropriate }
  begin
    if produce_amusing_messages then
      if (num_errors = 0) and (num_oversights = 0) and (num_warnings = 0) then
        writeln(f, ' No men left on base')
      else if num_errors >= max_errors then
        writeln(f, ' Extreme bogosity in this design')
      else if num_errors = 1 then
        writeln(f, ' 1 error?  Close, but no cigar');
  end { display_baseball_message } ;


begin { display_error_summaries }
  if debug_20 then writeln(outfile, ' enter display_error_summaries');
  if PrintCmpLst then
    begin
      writeln(CmpLst);
      writeln(CmpLst);
      display_error_message(CmpLst);
      display_oversight_message(CmpLst);
      display_warning_message(CmpLst);
      display_baseball_message(CmpLst);
    end;

  writeln(monitor);
  write(monitor,'  ');  display_error_message(monitor);
  write(monitor,'  ');  display_oversight_message(monitor);
  write(monitor,'  ');  display_warning_message(monitor);
  write(monitor,'  ');  display_baseball_message(monitor);

  writeln(CmpLog);
  write(CmpLog,'  ');  display_error_message(CmpLog);
  write(CmpLog,'  ');  display_oversight_message(CmpLog);
  write(CmpLog,'  ');  display_warning_message(CmpLog);
  write(CmpLog,'  ');  display_baseball_message(CmpLog);

  if PrintCmpErr then 
    if last_error <> 0 then writeln(CmpErr, 'end_error;');
  if debug_20 then writeln(outfile, ' exit display_error_summaries');
end { display_error_summaries } ;


(**)     { ------- parameter sorting ------- }


{ Parameter lists are sorted so that SIZE is first, followed by the other
  parameters in leicographic order of parameter name.  Note that no
  name can appear more than once in a parameter list. }

function compare_parameter_names(name1, name2: name_ptr): compare_type;
begin
  case compare_identifiers(name1, name2) of
    LT: 
      if (name2 = SIZE_prop_name) then 
        compare_parameter_names := GT
      else compare_parameter_names := LT;
    GT: 
      if (name1 = SIZE_prop_name) then 
        compare_parameter_names := LT
      else compare_parameter_names := GT;
    EQ: compare_parameter_names := EQ;
  end;
end { compare_parameter_names } ;
      

procedure sort_param_properties(var head: property_ptr);
  { Sorts the (parameter) properties in compare_parameter_names order 
    The sort is order N for the following 2 cases.
      1. already in order.
      2. in opposite order.
    other cases are not expected (and n-squared), but are quietly handled. }
  var
    next_element: property_ptr;  { next element to be checked }
    parent: property_ptr;        { parent of next_element }
    insert_point: property_ptr;  { new parent of next_element }
    comparison: compare_type;    { relation of elements being checked }
begin
  parent := head;
  if parent = NIL then next_element := NIL
		  else next_element := parent^.next;
  while next_element <> NIL do
    begin
      case compare_parameter_names(next_element^.name, parent^.name) of
	EQ: 
	  begin
	  { assert(ASSERT_DUPLICATE_PARAMETERS); }{ who cares - just delete it}
	    delete_property(parent^.next);
	  end;
        GT: parent := next_element;
        LT: { out of order - move it }
	  begin
	    parent^.next := next_element^.next;
	    case compare_parameter_names(next_element^.name, head^.name) of
	      EQ:
		begin
		  next_element^.next := NIL;
		  delete_property(next_element);
		end;
	      LT:
		begin
		  next_element^.next := head;
		  head := next_element;
		end;

	      GT:
		begin
		  insert_point := head;
		  comparison := 
		    compare_parameter_names(next_element^.name, 
		                            insert_point^.next^.name);
		  while comparison = GT do
		    begin
		      insert_point := insert_point^.next;
		      comparison := 
		        compare_parameter_names(next_element^.name,
			                        insert_point^.next^.name);
		    end;
		  if comparison = EQ then
		    begin
		      next_element^.next := NIL;
		      release_property(next_element);
		    end
		  else
		    begin 
		      next_element^.next := insert_point^.next;
		      insert_point^.next := next_element;
		    end;
		end { GT } ;
	    end { case } ;
	  end { LT -> out of order } ;
      end { case } ;
      next_element := parent^.next;
    end { while } ;
end { sort_param_properties } ;


(**)     { ------- error routine ------- }


procedure error(*error_num: error_range*);
  { write out offending line and display error }


  procedure dump_error_message(error_num: error_range);
    { display the error message to the appropriate files }


    procedure write_error(var f: textfile; num: error_range;
                          indent: boolean);
      { write the error message (and possibly the parse string) to the file }
    begin
      writeln(f);
      if (num IN parse_errors) and not ignore_parse_errors then
        print_input_line(f, num, indent);
      if indent then write(f, '  ');
      if command = COMPERR_COMMAND then write(f, ' <COMPERR>')
      else write(f, ' #', num_errors:1);
      writeln(f, ' ERROR(', num:1, '): ', error_strings[num]);
    end { write_error } ;


    procedure write_oversight(var f: textfile; num: error_range;
                              indent: boolean);
      { write the oversight message (and possibly the parse string) to file }
    begin
      writeln(f);
      if (num IN parse_errors) and not ignore_parse_errors then 
        print_input_line(f, num, indent);
      if indent then write(f, '  ');
      if command = COMPERR_COMMAND then write(f, ' <COMPERR>')
      else write(f, ' #', num_oversights:1);
      writeln(f, ' OVERSIGHT(', num:1, '): ', error_strings[num]);
    end { write_oversight } ;

    
    procedure write_warning(var f: textfile; num: error_range; 
                            indent: boolean);
      { write the warning message (and possibly the parse string) to the file }
    begin
      writeln(f);
      if (num IN parse_errors) and not ignore_parse_errors then
        print_input_line(f, num, indent);
      if indent then write(f, '  ');
      if command = COMPERR_COMMAND then write(f, ' <COMPERR>')
      else write(f, ' #', num_warnings:1);
      writeln(f, ' WARNING(', num:1, '): ', error_strings[num]);
    end { write_warning } ;

    
  begin { dump_error_message }
    if (error_num IN warning_errors) then
      begin
        if display_warnings and not (error_num IN suppress_errors) then
          begin
            num_warnings := num_warnings + 1;
	    if command = COMPERR_COMMAND then indent := 11
	    else indent := width_of_integer(num_warnings) + 3;
            errors_encountered := errors_encountered + [error_num];
            if PrintCmpLst then write_warning(CmpLst, error_num, FALSE)
                           else write_warning(monitor, error_num, FALSE);
          end;
        write_warning(CmpLog, error_num, PrintCmpLst);
        if debug_25 then
          begin
            writeln(Outfile, 'at error:');
            dump_parse_state(outfile, TRUE);
          end;
        if debugging then write_warning(outfile, error_num, FALSE);
      end
    else if (error_num IN oversight_errors) then
      begin
        if display_oversights and not (error_num IN suppress_errors) then
          begin
            num_oversights := num_oversights + 1;
	    if command = COMPERR_COMMAND then indent := 11
	    else indent := width_of_integer(num_oversights) + 3;
            errors_encountered := errors_encountered + [error_num];
            if PrintCmpLst then write_oversight(CmpLst, error_num, FALSE)
                           else write_oversight(monitor, error_num, FALSE);
          end;
        write_oversight(CmpLog, error_num, PrintCmpLst);
        if debug_25 then
          begin
            writeln(Outfile, 'at error:');
            dump_parse_state(outfile, TRUE);
          end;
        if debugging then write_oversight(outfile, error_num, FALSE);
      end
    else
      begin
	errors_encountered := errors_encountered + [error_num];
	num_errors := num_errors + 1;
	if command = COMPERR_COMMAND then indent := 11
	else indent := width_of_integer(num_errors) + 3;

	if PrintCmpLst then write_error(CmpLst, error_num, FALSE)
		       else write_error(monitor, error_num, FALSE);

	write_error(CmpLog, error_num, PrintCmpLst);
	if debug_25 then
	  begin
	    writeln(Outfile, 'at error:');
	    dump_parse_state(outfile, TRUE);
	  end;
	if debugging then write_error(outfile, error_num, FALSE);

	if (error_num IN echo_to_monitor_errors) and PrintCmpLst then
	      write_error(monitor, error_num, PrintCmpLst);
      end;
  end { dump_error_message } ;


  procedure output_graphics_editor_error(num: error_range);
    { output some stuff to the graphics editor }
    var
      stack_pos: stack_index_range;   { index into the parse stack }
      done: boolean;                  { TRUE when correct signal found }
  begin
    if not (num IN warning_errors) then
      begin
        if last_error <> 0 then writeln(CmpErr, 'end_error;');
        writeln(CmpErr, 'error');
        writeln(CmpErr, '  message = ''#', num_errors:1, ' ERROR(',
                        num:1, '): ', error_strings[num], ''';');
        if (num IN parse_errors) and not ignore_parse_errors then
          begin
            write(CmpErr, '  signal = ');
            if how_to_parse = parse_transparently then
              begin
                stack_pos := stack_top;  done := FALSE;
                while (stack_pos > 0) and not done do
                  if stack[stack_pos].how <> parse_transparently then
                    done := TRUE
                  else
                    stack_pos := stack_pos - 1;
                print_string(CmpErr, stack[stack_pos].str);
              end
            else
              print_string(CmpErr, instring);

            writeln(CmpErr, ';');
          end;
      end;
  end { output_graphics_editor_error } ;


begin { error }
  dump_error_message(error_num);

  { output some graphics editor stuff }
  if PrintCmpErr then output_graphics_editor_error(error_num);

  last_error := error_num;
  ok_to_print_error := not(error_num IN suppress_errors) and
                   (not(error_num IN warning_errors) or display_warnings) and
                   (not(error_num IN oversight_errors) or display_oversights);
  if (error_num IN parse_errors) and not ignore_parse_errors then
    error_dump_current_parse_environment;
end { error } ;
    
            
procedure error_dump_context(context: property_ptr; extra_indent: integer);
  { identify this context for the current error message }
  var
    param: property_ptr;  { current parameter }
begin
  if context = NIL then
    begin
      error_dump_indent(indent + extra_indent);
      error_dump_alpha('No parameters.  ');
      error_dump_CRLF;
    end;
  param := context;
  while param <> NIL do
    begin
      error_dump_indent(indent + extra_indent);
      error_dump_alpha(param^.name^.name);
      error_dump_char('=');
      error_dump_char(OUTPUT_QUOTE_CHAR);
      error_dump_string(param^.text);
      error_dump_char(OUTPUT_QUOTE_CHAR);
      error_dump_CRLF;
      param := param^.next;
    end;
end { error_dump_context } ;


procedure error_without_parse_line(*num: longint*);
  { print error message, but don't do special "parse error" processing.
    This is for use by C++ or C modules that handle parse errors separately. }
  var
    save: boolean;
begin
  save := ignore_parse_errors;
  ignore_parse_errors := TRUE;
  error(num);
  ignore_parse_errors := save;
end { error_without_parse_line } ;


procedure assert_without_parse_line(*num: longint*);
  { print assert message, but don't do special "parse error" processing.
    This is for use by C++ or C modules that handle parse errors separately. }
  var
    save: boolean;
begin
  save := ignore_parse_errors;
  ignore_parse_errors := TRUE;
  assert(num);
  ignore_parse_errors := save;
end { assert_without_parse_line } ;


procedure error_dump_standard_indent;
  { for use by C++ stuff so it does not need to directly access "indent" }
begin
    error_dump_indent(indent);
end { error_dump_standard_indent } ;


procedure error_dump_current_context;
  { for use by C++ stuff so it does not need to directly access the PASCAL
    data structures. }
  var
    props: property_ptr;


  function props_from_symbol_table(ids: identifier_ptr): property_ptr;
    var
      head: property_ptr;
  begin
    head := NIL;
    while ids <> NIL do
      begin
        add_to_prop_list(head, ids^.name, ids^.definition);
	ids := ids^.next;
      end;
    sort_param_properties(head);
    props_from_symbol_table := head;
  end { props_from_symbol_table } ;


begin
    if current_mtree_node^.father_node = NIL then
      error_dump_context(specified_context, 8 { length of "Drawing=" })
    else
      begin { An error on a plumbing body -- probably a selection error }
	{ Assume that the drawing has not been read, thus the symbol table
	  reflects just the parameters in the instance. }

        props := props_from_symbol_table(current_mtree_node^.symbol_table);
	error_dump_context(props, 8 { length of "Drawing=" });
	release_entire_property_list(props);
      end;
end { error_dump_current_context } ;


(**)     { ------- assertion failure routine ------- }


procedure assert(*assertion_num: assert_range*);
  { display the assertion failure message to the appropriate files }

  
  procedure write_assertion_message(var f: textfile;  num: assert_range;
                                    indent: boolean);
    { display the message }
  begin
    if (num in parse_asserts) and not ignore_parse_errors then
      print_input_line(f, 0, indent);
    if indent then write(f, '  ');
    writeln(f, ' ASSERTION FAILURE(', num:1, '): ', assert_strings[num]);
    if not ignore_parse_errors then error_dump_current_parse_environment;

    if produce_amusing_messages then
      begin
        if indent then write(f, '  ');
        if (num MOD 4) = 0 then
         writeln(f, ' There''s something wicked in the state of Denmark')
        else if (num MOD 4) = 1 then
          writeln(f, ' Bogons have been sighted!')
        else if (num MOD 4) = 2 then
          writeln(f, ' Extreme brain damage has been detected!')
        else if (num MOD 4) = 3 then
          writeln(f, ' Help me Landru! Help me Landru!');
      end;
  end { write_assertion_message } ;


begin { assert }
  error(187);      { tell the normal world about internal screw up }
  write_assertion_message(CmpLog, assertion_num, PrintCmpLst);
  if (command = COMPERR_COMMAND) and PrintCmpLst then
    write_assertion_message(CmpLst, assertion_num, FALSE);
  if debugging then write_assertion_message(outfile, assertion_num, FALSE);
end { assert } ;


(**)     { ------- Close all files used by Compiler ------- }


procedure close_all_files;
  { close the files still open }
begin
  if debug_20 then writeln(outfile, ' enter close_all_files -- bye bye');
  if PrintCmpErr then 
    begin
      writeln(CmpErr, 'END.');
      close_file(CmpErr, CmpErr_file_name, nullstring);
    end;
  if not debugging then
    if file_exists(temp_file_name) then if remove_file(temp_file_name) then ;
  close_file(monitor, MONITOR_FILE_NAME, nullstring);
  close_file(CmpLog, CMPLOG_FILE_NAME, nullstring);
  if debugging then close_file(outfile, DEBUG_FILE_NAME, nullstring);
end { close_all_files } ;


#include "schema.p"
#include "make.p"
#include "module.p"
#include "cleanup.p"
#include "readged.p"
#include "readpage.p"


(**)     { ------- scope utilities ------- }


function determine_scope(scope_value: xtring): scope_type;
  { determine the scope from the value given (SCOPE_VALUE) }
  var
    scope: scope_type;      { scope as determined }
    found: boolean;         { TRUE if entry in table is found }
begin
  scope := succ(FIRST_SCOPE);  found := FALSE;
  while (scope < LAST_SCOPE) and not found do
    if scope_table[scope] = scope_value then found := TRUE
    else scope := succ(scope);

  if scope = LAST_SCOPE then scope := UNKNOWN_SCOPE;

  determine_scope := scope;
end { determine_scope } ;


(**)     { ------- signal utilities ------- }


function create_a_subscript(width: natural_number): subscript_ptr;
  { create a bit subscript of the specified width }
  var
    sub: subscript_ptr;       { subscript being created }
begin
  sub := NIL;  new_subscript(sub);
  if left_to_right then
    begin  sub^.left_index := 0;  sub^.right_index := width-1;  end
  else
    begin  sub^.left_index := width-1;  sub^.right_index := 0;  end;

  create_a_subscript := sub;
end { create_a_subscript } ;


function unique_NC_name: xtring;
  { create a unique name for an NC signal }
  var
    number: alpha;                      { used to create string to return }
    pos: id_range;                      { index into NUMBER }
    temp: xtring;                       { string to be returned }


  procedure convert_number_to_char(n: longint);
    { add the given number to the name buffer recursively }
  begin
    if n > 9 then convert_number_to_char(n div 10);
    pos := pos + 1;  number[pos] := chr((n mod 10)+ord('0'));
  end { convert_number_to_char } ;


begin { unique_NC_name }
  number := special_NC_name;  number[1] := chr(NC_prefix_char);  pos := 3;
  convert_number_to_char(unique_NC_number);
  unique_NC_number := unique_NC_number + 1;

  temp := make_and_enter_string(number);
  unique_NC_name := temp;
end { unique_NC_name } ;


function is_NC_signal(signal_name: xtring): boolean;
  { check the signal to see if it is an NC signal and return TRUE if so }
begin
  is_NC_signal := (signal_name^[1] = chr(NC_prefix_char));
end { is_NC_signal } ;


function is_NC_def(def: signal_definition_ptr): boolean;
  { return TRUE if the signal definition describes an NC signal }
begin
  is_NC_def := is_NC_signal(def^.signal^.name);
end { is_NC_def } ;


function is_unnamed_signal(signal_name: xtring): boolean;
  { return TRUE if the signal is an UNNAMED signal }
  var
    i: id_range;           { index into alpha }
    alpha_name: alpha;     { alpha UNNAMED signal name }
    done: boolean;         { TRUE if '$' is found in name }
begin
  is_unnamed_signal := FALSE;
  if ord(signal_name^[0]) > UNNAMED_SIGNAL_NAME_LENGTH then
    begin
      alpha_name := NULL_ALPHA;
      i := 1;  done := FALSE;
      while (i <= UNNAMED_SIGNAL_NAME_LENGTH) and not done do
        if signal_name^[i] = '$' then done := TRUE
        else
          begin  alpha_name[i] := signal_name^[i];  i := i + 1;  end;
      is_unnamed_signal := (alpha_name = UNNAMED_signal) or
                           (alpha_name = short_UNNAMED_signal);
    end;
end { is_unnamed_signal } ;


function is_strange_signal(signal: signal_descriptor_ptr): boolean;
  { return TRUE if the signal is UNNAMED or NC }
  var
    strange: boolean;
begin
  strange := is_NC_signal(signal^.signal_name);
  if not strange then
    strange := is_unnamed_signal(signal^.signal_name);

  is_strange_signal := strange;
end { is_strange_signal } ;


#include "expandtm.p"
#include "expprop.p"


(**)     { ------- parse a signal name and return descriptor ------- }


function parse_signal_name(signal_name: xtring; use_NAC: boolean): 
                                                signal_descriptor_ptr;
  { parse the specified signal into its various components and return a
    descriptor.  It assumed that the context of the signal is active
    so that all expression and text macro evaluations can be correctly
    performed. }
  var
    property_name: name_ptr;         { current property name }
    property_value: xtring;          { current property value }
    negated,                         { TRUE if signal is negated }
    done: boolean;                   { TRUE when entire string is parsed }
    bit_subscript: subscript_ptr;    { temp bit subscript }
    last,                            { last descriptor created }
    SDP: signal_descriptor_ptr;      { signal descriptor to be returned }
    syntax_index: signal_syntax_range;  { index into syntax table }
    found_scope: boolean;            { TRUE if SCOPE property found }
    scope: scope_type;               { scope of signal }
    rep_prop_value: xtring;          { value of the replication property }
    found_no_assertion: boolean;     { TRUE if no assertion char found }
    is_NC: boolean;                  { TRUE if signal is NC }
    is_UNNAMED: boolean;             { TRUE if signal is UNNAMED }


  function get_property(terminal: symbols): boolean;
    { parse the property decoding special ones and return TRUE if no error.
      Assume that identifiers not properly followed by strings are undefined
      macros - thus needing to be logged as MUST_BE_DEFINED expandable_ids }
    var
      ok: boolean;                   { function value to be returned }
  begin
    ok := FALSE;

    if sy <> IDENT then
      begin  error(1 { expected identifier });  skip([terminal]);  end
    else
      begin
        property_name := id.name;  insymbol;

        if sy = EQUAL then insymbol else error(2 { expected = });

        if sy <> STRINGS then
          begin  
            error(33 { expected a string });  
            enter_expandable_id(property_name);
            skip([terminal]);  
          end
        else
          begin
            property_value := lex_string;  insymbol;  ok := TRUE;

            { expand property value TMs using % mechanism }

            property_value := expand_property_value_TMs(current_mtree_node,
                                                        property_name,
                                                        property_value);
          end;
      end;

    get_property := ok;
  end { get_property } ;


  procedure convert_into_binary(var signal_name: xtring);
    { convert CONST_VAL into an equivalent binary representation in string }
    var
      val: longint;                  { value of the constant }
      pos: string_range;             { index into the string being created }
  begin
    if const_width = 0 then assert(31 { illegal width });

    create_a_string(signal_name, const_width);
    for pos := 1 to const_width do signal_name^[pos] := '0';
    val := const_val;  pos := const_width;
    while (val > 0) and (pos > 0) do
      begin
        signal_name^[pos] := chr((val MOD 2) + ord('0'));
        val := val DIV 2;  pos := pos - 1;
      end;

    signal_name := enter_and_release_string(signal_name);
  end { convert_into_binary } ;


  procedure get_signal_name(SDP: signal_descriptor_ptr);
    { parse the signal name.  It may be a name or a constant.  If the
      signal name = NC, generate a unique name for it. }
  begin
    if sy = STRINGS then     { it's a signal name }
      begin
        SDP^.is_const := FALSE;      { just to make sure }
        if lex_string = NC_signal then
          begin
            SDP^.signal_name := unique_NC_name;
            SDP^.kind := UNDEFINED;
            SDP^.polarity := NO_POLARITY;
            scope := LOCAL;
            is_NC := TRUE;
          end
        else
          SDP^.signal_name := lex_string;
          if is_unnamed_signal(SDP^.signal_name) then
            begin
              scope := LOCAL;
              SDP^.kind := UNDEFINED;
              SDP^.polarity := UNKNOWN_POLARITY;
              is_UNNAMED := TRUE;
            end;

        insymbol;
      end

    else if (sy = SIGNALCONST) or (sy = CONSTANT) then
      begin
        SDP^.is_const := TRUE;
        scope := SIG_CONST;

        convert_into_binary(SDP^.signal_name);     { uses CONST_VAL }
        if ord(SDP^.signal_name^[0]) > 1 then
          begin
            SDP^.kind := VECTOR;
            SDP^.bit_subscript :=
                              create_a_subscript(ord(SDP^.signal_name^[0]));
          end;
        insymbol;
      end

    else error(83 { expected signal name or constant });
  end { get_signal_name } ;


  procedure process_replication(SDP: signal_descriptor_ptr);
    { if a replication factor was specified, set the corresponding field
      in SDP^.  If the signal is an NC change the KIND field to VECTOR
      and represent the "replicated" NC as a vector of that many bits --
      this insures that they are not synonymed together. }
      
    var
      temp: longint;           { value of expression }
  begin
    if rep_prop_value <> nullstring then
      begin
        parse_string(rep_prop_value, PARSE_SEPARATELY);
        temp := expression(NO_RELOPS);
        if sy <> ENDOFDATASY then
          begin
            error(58 { extraneous junk });
            error_dump_indent(indent);
            error_dump_alpha('Processing REP= ');
            error_dump_string(signal_name);
            error_dump_CRLF;
          end;

        if (temp <= 0) or (temp > MAX_BIT_VALUE) then
          begin  error(84 { too big });  temp := 1;  end;

        pop_parsed_string(rep_prop_value);

	if is_NC then
	  begin
	    SDP^.kind := VECTOR;
	    SDP^.bit_subscript := create_a_subscript(temp);
	  end
	else SDP^.replication_factor := temp;
      end;

  end { process_replication } ;


  procedure fix_UNNAMED_signal_name;
    { add a unique number to the end of the signal name to make unique }
    var
      temp: xtring;          { temporary string for new name creation }
      i: string_range;       { index into the signal name }
      length: string_range;  { length of the original signal name }
  begin
    length := ord(SDP^.signal_name^[0]);
    create_a_string(temp, length+2);
    for i := 1 to length do
      temp^[i] := SDP^.signal_name^[i];

    temp^[length+1] := '$';
    temp^[length+2] := 'A';

    SDP^.signal_name := enter_and_release_string(temp);
  end { fix_UNNAMED_signal_name } ;


  procedure fix_name_starting_with_paren;
    { Signal name begins with '('.  Issue error message and prepend the
      signal class 'ILL$' }
    var
      temp: xtring;          { temporary string for new name creation }
      length: string_range;  { length of the original signal name }
  begin
    length := min(ord(SDP^.signal_name^[0]) + 4, MAX_STRING_LENGTH);
    create_a_string(temp, length);
    temp^[0] := chr(0);
    if add_alpha_to_string(temp, 'ILL$            ') then ;
    if add_string_to_string(temp, SDP^.signal_name) then ;

    error(54 { Don't start signal name with '(' });
    error_dump_current_parse_environment;
    error_dump_signal_descriptor(SDP);
    error_dump_indent(INDENT);
    error_dump_alpha('Name changed to:');
    error_dump_string(temp);
    error_dump_CRLF;

    SDP^.signal_name := enter_and_release_string(temp);
  end { fix_name_starting_with_paren } ;


  procedure fix_complemented_constant(signal: signal_descriptor_ptr);
    { complement the bits in the constant signal name }
    var
      i: string_range;        { index into the signal name }
      new_name: xtring;       { new name for the signal }
      length: string_range;   { length of the signal name }
  begin
    length := ord(signal^.signal_name^[0]);
    create_a_string(new_name, length);

    for i := 1 to length do
      if signal^.signal_name^[i] = '0' then new_name^[i] := '1'
                                       else new_name^[i] := '0';

    signal^.signal_name := enter_and_release_string(new_name);
  end { fix_complemented_constant } ;


begin { parse_signal_name }
  if debug then disp_line('enter parse_signa');

  { care must be taken to ensure that text macros are handled correctly when
    parsing a signal.  Text macros (without the delimiter characters) are
    permitted in only two places:  within bit subscripts and within general
    property specifications.  The global flag ALLOW_TM_EXPANSION controls
    whether the input parser expands identifiers (that are text macros) or
    not.  It is initially set OFF and is set ON when a subscript is parsed
    (after which it is set off again) and is set ON when a general property
    prefix is found. }

  allow_TM_expansion := FALSE;

  parse_string(signal_name, PARSE_SEPARATELY);

  done := FALSE;  last := NIL;

  repeat
    new_signal_descriptor(SDP);  
    if last = NIL then parse_signal_name := SDP else last^.next := SDP;  
    last := SDP;

    SDP^.kind := SINGLE;
    scope := UNKNOWN_SCOPE;  found_scope := FALSE;
    rep_prop_value := nullstring;
    found_no_assertion := FALSE;
    is_NC := FALSE;
    is_UNNAMED := FALSE;

    negated := FALSE;
    for syntax_index := 1 to SYNTAX_TABLE_SIZE do
      case signal_syntax_table[syntax_index] of

        NEGATION_SPECIFIER:
            begin
              negated := (sy = signal_negation_symbol);
              if negated then insymbol;
            end;

        NAME_SPECIFIER:
            begin
              get_signal_name(SDP);
            end;

        SUBSCRIPT_SPECIFIER:
            begin
              allow_TM_expansion := TRUE;
              bit_subscript := bit_selector;
              allow_TM_expansion := FALSE;
              if SDP^.is_const then
                if bit_subscript <> NIL then
                  begin
                    error(96 { can't have bit subscript on constant! });
                    { NOTE: release that subscript! }
                  end
                else { it's OK.  leave the subscript alone }
              else SDP^.bit_subscript := bit_subscript;

              if SDP^.bit_subscript <> NIL then
                if SDP^.kind = UNDEFINED then assert(153 { UNNAMED w/ width })
                else SDP^.kind := VECTOR;
            end;

        ASSERTION_SPECIFIER:
            begin
              if sy = signal_is_asserted_low_symbol then
                begin
                  SDP^.low_asserted := TRUE;
                  insymbol;
                end
              else if sy = signal_is_asserted_high_symbol then
                begin
                  SDP^.low_asserted := FALSE;
                  insymbol;
                end
              else if signal_is_asserted_high_symbol <> NULLSY then
                if not allow_missing_high_assertion then
                  found_no_assertion := TRUE;
            end;
    
        PROPERTY_SPECIFIER:
            begin
              allow_TM_expansion := TRUE;
              allowed_key_words := allowed_key_words - config_keysys;

              while sy = general_property_prefix_symbol do
                begin
                  insymbol;
                  if get_property(general_property_prefix_symbol) then
                    if property_name = SCOPE_prop_name then
                      SDP^.scope := determine_scope(property_value)
                    else if property_name = REPLICATION_prop_name then
                      rep_prop_value := property_value
                    else if property_name = NO_ASSERT_prop_name then
                      begin
                        use_NAC := TRUE;
                        add_to_prop_list(SDP^.properties, property_name,
                                                          property_value);
                      end
                    else
                      add_to_prop_list(SDP^.properties, property_name,
                                                        property_value);
                end;
              allow_TM_expansion := FALSE;
              allowed_key_words := signal_keysys;
            end;

        null_specifier: ;

      end { case } ;
      
    process_replication(SDP);

    if SDP^.low_asserted then negated := not negated;  

    if (is_UNNAMED or is_NC) and not bubble_check then
      begin
        SDP^.low_asserted := FALSE;
        SDP^.polarity := NO_POLARITY;
      end;

    if SDP^.polarity = NORMAL then
      if negated then
        SDP^.polarity := COMPLEMENTED
      else
        SDP^.polarity := NORMAL;

    { add unique number to end of UNNAMED signals if low asserted.  The new
      6.0 editor never outputs a low asserted unnamed signal. }

    if is_UNNAMED and SDP^.low_asserted then
      fix_UNNAMED_signal_name;

    { make sure that GED assigned assertions for UNNAMEDs are ignored }

    if SDP^.polarity IN [UNKNOWN_POLARITY, NO_POLARITY] then
      SDP^.low_asserted := FALSE;

    if SDP^.is_const then
      begin
        if SDP^.polarity = COMPLEMENTED then fix_complemented_constant(SDP);

        if not const_bubble_check then SDP^.polarity := NO_POLARITY;
      end;

    if scope <> UNKNOWN_SCOPE then
      begin
        if SDP^.scope <> UNKNOWN_SCOPE then
          begin
            error(165 { this signal cannot have SCOPE property });
            error_dump_current_parse_environment;
            error_dump_signal_descriptor(SDP);
          end;
        SDP^.scope := scope;
      end;

    if found_no_assertion and not (is_NC or is_UNNAMED or SDP^.is_const or
       use_NAC) then
      begin
        error(36 { must have high assertion char });
        error_dump_indent(indent);
        error_dump_alpha('High assert char');
        error_dump_alpha(' = "            ');
        error_dump_char(signal_is_asserted_high_char);
        error_dump_char('"');
        error_dump_CRLF;
      end;
      
    if (SDP^.signal_name <> nullstring) then
      if (SDP^.signal_name^[1] = '(') then
        fix_name_starting_with_paren;

    if sy = concatenation_symbol then insymbol else done := TRUE;
  until done;

  allow_TM_expansion := default_TM_expansion;

  if sy <> ENDOFDATASY then error(179 { unexpected junk });
  pop_parsed_string(signal_name);

  if debug then disp_line('parse_signal_name');
end { parse_signal_name } ;
    
#include "ext_ds.p"
  
#include "bitunion.p"
#include "directives.p"
#include "textmacro.p"
#include "attributes.p"
#include "interrupt.p"
#include "init.p"

{------------------------ end procedures --------------------------}

