(**)     { ------- PASS 1 status report ------- }


procedure display_PASS_1_status(node: mtree_node_ptr);
  { display the status of the expansion so that errors can be associated }
  const
    SEPARATION = 4;                { separation between path and macro names }
  var
    position,                      { position in output line }
    path_end: print_width_range;   { position of last char in path name }
    path_prop: property_ptr;       { PATH property }

 
  procedure display_parameters(initial_position: print_width_range);
    { display the parameters for the current macro }
    var
      id: identifier_ptr;           { parameter entry in symbol table }
      first: boolean;               { TRUE if first parameter not yet printed }
      length: id_range;             { length of the parameter name }
      param: property_ptr;          { current parameter to be printed }
      position: print_width_range;  { current character position in CmpLst }
      text: xtring;                 { parameter value to be printed }
      OK_to_output: boolean;        { TRUE if parameter OK to output }
  begin
    position := initial_position;
    first := TRUE;  param := node^.called_by^.parameters;
    while param <> NIL do
      begin
        if not found_id(node, param^.name, id) then
          assert(48 { it has to be there! })
        else
          begin
            text := id^.definition;

            { do not output default SIZE or TIMES properties }

            OK_to_output := TRUE;

            if (param^.name = SIZE_prop_name) then
              if text = default_SIZE_string then
                OK_to_output := FALSE;

            if (param^.name = TIMES_prop_name) then
              if text = default_SIZE_string then
                OK_to_output := FALSE;

            if OK_to_output then
              begin
                length := alpha_length(param^.name^.name);

                if (length+ord(text^[0])+3) >= (print_width-position-1) then
                  begin
                    if first then writeln(CmpLst) else writeln(CmpLst, ';');
                    position := path_end + SEPARATION + DEFAULT_INDENT;
                    write(CmpLst, ' ':position-1);
                    if first then write(CmpLst, '[');
                  end
                else
                  if first then write(CmpLst, ' [')
                           else write(CmpLst, '; ');
                first := FALSE;
                print_alpha(CmpLst, param^.name^.name);
                write(CmpLst, '=');
                print_string(CmpLst, text);
                position := position + length + ord(text^[0]) + 3;
              end;
          end;

        param := param^.next;
      end;

    if first then  writeln(CmpLst)  else  writeln(CmpLst, ']');
  end { display_parameters } ;


begin { display_PASS_1_status }
  if PrintCmpLst and (PATH_NAMES IN reports_to_generate) then
    begin
      path_end := (print_width * 2 DIV 3);
      if find_property(node^.called_by^.properties, PATH_prop_name, 
                       path_prop) then
        position := print_string_formatted(CmpLst, path_prop^.text, 2, 
                                           path_end, PAD, noCRLF)
      else
        position := print_string_formatted(CmpLst, nullstring, 2, 
                                           path_end, PAD, noCRLF);
      write(CmpLst, '''':SEPARATION+1);
      position := print_string_formatted(CmpLst, node^.macro_name,
                                         path_end + SEPARATION + 1,
                                         print_width-2, noPAD, noCRLF);
      write(CmpLst, '''');
      display_parameters(position+1);
    end;

  if debug_13 then
    if path_prop = NIL then writeln(outfile, '<Nil> path')
    else
      begin
        writestring(outfile, path_prop^.text);  writeln(outfile);
      end;
end { display_PASS_1_status } ;


(**)     { ------- Process drawings ------- }


function PASS1(node: mtree_node_ptr;
               level: level_range;
               is_plumbing_drawing: boolean): boolean;

  { This routine does most of the work of the Compiler.  It processes
    a single drawing and calls itself recursively to process drawings
    called in the current drawing.  A tree of nodes is created where each
    node corresponds to a drawing instance.  The node for the current
    drawing is given by NODE.  The depth within the tree is specified
    by LEVEL with the root node of the design at level 1.  SIZE replication
    is performed.  The following steps are repeated for each value of X.

       1.  Parse the pin names and signals attached to them for each
           body in the drawing.

       2.  Hierarhically process each of the "plumbing" drawings.  These
           drawings are used only to synonym signals together.

       3.  Resolve the widths and assertions of all of the signals in the
           drawing.  This includes those signals local to any "plumbing"
           drawings called from this drawing.  Check to make sure that all
           signals are resolved and report any errors.

       4.  Create all synonyms specified by the actual lists.

       5.  Hierarchically process the rest of the bodies in the drawing.
           If single level compilation is specified, do not process the
           bodies; turn them into primitives.

    "Plumbing" drawings are not permitted to expand to real primitives.
    These drawings are detected by the presence of the NWC property on at
    least one of their pins or the presence of the BODY_TYPE=PLUMBING
    property on the body.  If the drawing does expand to primitives,
    return TRUE.  Hierarchical "plumbing" drawings are permitted.  }
  type
    id_definition_type = (EXPAND_THESE_PARAMETERS, THESE_TMS_ARE_LITERAL);
  var
    macro: macro_def_ptr;            { macro definition for this node }
    expands_to_primitives: boolean;  { TRUE if drawing expands to primitives }
    current_node: mtree_node_ptr;    { current SIZE replicated node }

    x: natural_number;             { X iteration variable }
    x_first: longint;              { initial value for X }
    x_step: longint;               { iteration step value for X }
    size: longint;                 { value of SIZE for the invoked son macro }
    done: boolean;                 { TRUE if SIZE expansion is complete }

    id: identifier_ptr;            { symbol table entry for X }


  procedure add_properties_to_symbol_table(
    node: mtree_node_ptr;  property_list: property_ptr;
    expansion_directive: id_definition_type);
    { add the given properties to the symbol table of the given node }
    var
      id: identifier_ptr;      { symbol table entry for current parameter }
      property: property_ptr;  { current property to be added }


    function evaluate_and_replace(name: name_ptr; definition: xtring): xtring;
      { parse the input string and return a string = value of the expr. }
      var
        val: longint;                { expression value }
    begin
      if definition = nullstring then
        begin
          error(118 { expression value is empty });  val := 0;
          error_dump_current_parse_environment;
          error_dump_property(name, nullstring);
          id^.resolves := FALSE;
        end
      else
        begin
          parse_string(definition, PARSE_SEPARATELY);

          val := expression(NO_RELOPS);

          if sy <> ENDOFDATASY then
            begin
              error(58 { extraneous junk });
              error_dump_property(name, definition);
            end;

          pop_parsed_string(definition);;
        end;

      evaluate_and_replace := number_to_string(val);
    end { evaluate_and_replace } ;


  begin { add_properties_to_symbol_table }
    property := property_list;
    while property <> NIL do
      begin
        enter_id(node, property^.name, id);

        if id = NIL then 
	  case expansion_directive of
            EXPAND_THESE_PARAMETERS: assert(154 { should have been fixed on input });
            THESE_TMS_ARE_LITERAL: { OK, let the parameter win } ;
	   end
        else
          case expansion_directive of
            EXPAND_THESE_PARAMETERS:
              if IS_INT_PARAMETER in property^.name^.kind then
                id^.definition := 
                  evaluate_and_replace(property^.name, property^.text)
              else
                id^.definition := 
                  expand_text_macros(property^.name, property^.text);
            THESE_TMS_ARE_LITERAL: id^.definition := property^.text;
          end;

        property := property^.next;
      end;
  end { add_properties_to_symbol_table } ;


  procedure add_local_TMs_to_symbol_table(node: mtree_node_ptr);
  begin
    { If this is the root node, check local text macros defined on this
      page against their previous defs and, if changed, mark affected
      pages dirty and release any previously defined list of local TMs }

    if debug_13 or debug_23 or debug_24 then
      begin
        writeln(outfile, ' add_local_TMs_to_symbol_table');
      end;

    if node^.father_node <> NIL then
      begin
        assert(0);
	writeln(CmpLog, ' add_local_TMs... called on a plumbing node');
      end;

    if node = mtree_root then
      begin
        if not check_local_TMs(schema_of_drawing_being_compiled,
                               paged_schema_of_this_page.local_text_macros,
			       old_schema_page) then
          schema_of_drawing_being_compiled.local_TMs_defined := FALSE;

        if current_compiled_context^.dirty_for_pass_2 then
          schema_of_drawing_being_compiled.local_TMs_defined := FALSE;
      end;

    if not schema_of_drawing_being_compiled.local_TMs_defined then
      compute_current_local_TM_context(old_schema_page, TRUE);

    add_properties_to_symbol_table(
      node, 
      schema_of_drawing_being_compiled.local_TMs,
      THESE_TMS_ARE_LITERAL);
  end { add_local_TMs_to_symbol_table } ;


  procedure create_current_context(node: mtree_node_ptr; macro: macro_def_ptr);
    { init the given node to create the current "context" or scope 
      (node^.father node MUST already be set) }
  begin
    node^.macro := macro;

    current_mtree_node := node;
    node^.level := level;

    if macro <> NIL then
      begin
        node^.next_node_with_same_mdef := macro^.occurances;
        macro^.occurances := node;

        if node^.is_leaf_node then
          number_leaf_nodes := number_leaf_nodes + 1;

        if node^.father_node = NIL then add_local_TMS_to_symbol_table(node)
        else add_properties_to_symbol_table(node, macro^.text_macros,
                                            THESE_TMS_ARE_LITERAL);
      end;
  end { create_current_context } ;


  procedure init_node(node, father_node: mtree_node_ptr; 
                      invoke: invoke_list_ptr);
    { initialize the given node (NODE) whose father is FATHER invoked by
      INVOKE.  The INVOKE may be NIL if this is the root node (SIZE rep of
      the current drawing.  In this case, FATHER will also be NIL. }
  begin
    node^.father_node := father_node;
    node^.called_by := invoke;

    if father_node = NIL then
      node^.level := 1
    else
      node^.level := node^.father_node^.level+1;

    if invoke <> NIL then
      begin
        node^.macro_name := invoke^.macro_name;

        add_properties_to_symbol_table(node, invoke^.parameters,
                                       EXPAND_THESE_PARAMETERS);
      end;
  end { init_node } ;


  procedure free_formal_parameters(node: mtree_node_ptr);
    { release the formal parameter signal definitions and signal instances of
      the sons of the given node.  The formal parameters are found in the 
      signals list of the node.  The formal/actual parameter list is also
      released.  Do not release formal/actual lists for leaf
      nodes since these are needed for PASS 3.  For leaf
      nodes, replace the formal parameter signal instance with a simple
      signal description. }
    var
      son: mtree_node_ptr;                   { son of the given node }
      next,                                  { next signal in the list }
      previous,                              { previous signal in the list }
      signal: signal_definition_ptr;         { current signal in the list }
      next_formal_actual_pair,               { next formal/actual param }
      formal_actual_pair: formal_actual_ptr; { formal/actual pair }
      pin_name: simple_signal_ptr;           { pin name representation }
  begin
    if debug_13 then
      begin
        write(outfile, 'Entering free_formal_parameters: father node=');
        print_string(outfile, node^.macro_name);
        writeln(outfile);
      end;

    son := node^.son;
    while son <> NIL do
      begin
       if debug_13 then
          begin
            write(outfile, 'processing son=');
            print_string(outfile, son^.macro_name);
            writeln(outfile);
          end;

        { if the node son is not a primitive, 
          then release the formal/actual pairs.  Otherwise, assign a simple
          signal description of the pin so we can still release the pin name
          signal definition. }

        if son^.is_plumbing_node then
          free_formal_parameters(son);

        if not (son^.is_leaf_node) then
          begin
            if debug_13 then
              begin
                writeln(outfile, 'releasing f/a=');
                dump_formal_actual_list(outfile, son^.params);
              end;

            release_complete_formal_actual_list(son^.params)
          end
        else
          begin
            if debug_13 then writeln(outfile, 'node is LEAF');

            formal_actual_pair := son^.params;
            while formal_actual_pair <> NIL do
              begin
                next_formal_actual_pair := formal_actual_pair^.next;

                new_simple_signal(pin_name);
                with formal_actual_pair^.formal_parameter^ do
                  begin
                    pin_name^.polarity := defined_by^.polarity;
                    pin_name^.signal_name := defined_by^.signal^.name;
                    pin_name^.kind := defined_by^.kind;
                    pin_name^.bit_subscript := bit_subscript;
                  end;
                formal_actual_pair^.pin_name := pin_name;

                { set fields to NIL so they won't be released }
                formal_actual_pair^.formal_parameter^.bit_subscript := NIL;
                formal_actual_pair^.formal_parameter := NIL;

                formal_actual_pair := next_formal_actual_pair;
              end { while } ;
          end;

        { release the formal signal definitions and instances }

        if debug_13 then writeln(outfile, 'Starting to release INT sigs');

        signal := son^.signals;  previous := NIL;
        while signal <> NIL do
          begin
            { signal cannot be constant since ALL constants are at root }

            if signal^.is_const then assert(174 { not here! });
            if signal^.scope = XINTERFACE then
              begin
                next := signal^.next;

                { if the def has NIL instances, it was already released }

                if signal^.instances <> NIL then
                  release_complete_signal_def(signal);

                if previous = NIL then son^.signals := next
                                  else previous^.next := next;
                signal := next;
              end
            else
              begin  previous := signal;  signal := signal^.next;  end;
          end;

        son := son^.next;
      end;

    if debug_13 then
      writeln(outfile, 'Exiting free_formal_parameters');
  end { free_formal_parameters } ;


  procedure evaluate_X_loop_controls(node: mtree_node_ptr;
                                     var x_first, x_step, size: longint);
    { evaluate X_first, X_STEP and SIZE, check to see that they are in range,
      and return the values.  Here are the rules:

      Plumbing drawings cannot be SIZE replicated in the Compiler.

      A SIZE parameter will be in the symbol table if:
         1.  A SIZE parameter was attached to the drawing's body.
         2.  The drawing's body used SIZE in a pin name.

      If SIZE is not in the symbol table:
         1. Set SIZE default to 1;  generate no error message.
         2. Set X_first to 0;  generate no error message.
         3. Set X_STEP to SIZE;  generate no error message.

      If SIZE is in the symbol table:
         1. Use specified SIZE value;  check value for > 0.
         2. If no X_first appears, default to 0;  generate no error message.
         3. If X_first appears, use specifed value;  check for >= 0.
         2. If no X_STEP appears, default to SIZE;  generate no error message.
         3. If X_STEP appears, use specified value;  check for > 0.

      If SIZE is in the symbol table but was NOT used in the body:
         1. Generate an error message.
         2. Set SIZE, X_first, and X_STEP to defaults.  }
    var
      id: identifier_ptr;      { symbol table entry }
      prop: property_ptr;      { property returned from search }
  begin
    { set up the defaults }

    size := 1;
    x_first := 0;
    x_step := size;

    if found_id(node, SIZE_prop_name, id) then
      begin
        push_error_info;
        current_property_name := SIZE_prop_name;
        if node^.father_node <> NIL then
          begin
            current_macro_def := node^.father_node^.macro;
            current_body_node := node;
          end
        else
          current_macro_def := node^.macro;

        if not node^.uses_SIZE_property then
          if node^.called_by <> NIL then
            if not find_property(node^.called_by^.properties,
                                 HAS_FIXED_SIZE_prop_name, prop) then
              begin
                error(141 { SIZE property appears but not used on body });
                error_dump_current_parse_environment;
              end;

        size := evaluate_string(SIZE_string);

        if size < 0 then
          begin
            error(185 { must be >= 0 });
            error_dump_current_parse_environment;
            size := 1;
          end;

        x_step := size;

        if found_id(node, X_first_identifier, id) then
          begin
            current_property_name := X_first_identifier;

            x_first := evaluate_string(X_first_string);
            if x_first < 0 then
              begin
                error(183 { illegal });
                error_dump_current_parse_environment;
                x_first := 0;
              end;
          end;

        if found_id(node, X_STEP_identifier, id) then
          begin
            current_property_name := X_STEP_identifier;

            x_step := evaluate_string(X_STEP_string);
            if x_step <= 0 then
              begin
                error(186 { illegal });
                error_dump_current_parse_environment;
                x_step := size;
              end;
          end;

        if is_plumbing_drawing and (x_step <> size) then
          begin
            error(160 { plumbing drawings cannot be SIZE replicated });
            current_property_name := null_name;
            error_dump_current_parse_environment;
          end;

        pop_error_info;
      end;

  end { evaluate_X_loop_control } ;


  function node_is_FLAG_node(node: mtree_node_ptr): boolean;
    { return TRUE if the given node has the BODY_TYPE='FLAG' property }
    var
      prop: property_ptr;       { property returned from search }
      result: boolean;          { value to be returned }
  begin
    result := FALSE;

    if node^.macro <> NIL then
      if find_property(node^.macro^.properties,
                       BODY_TYPE_prop_name, prop) then
        if prop^.text = FLAG_BODY_string then result := TRUE;

    if not result then
      if node^.called_by = NIL then
        begin
          assert(0 { must not be });
          writeln(CmpLog, ' node^.called_by is NIL in node_is_flag_node');
        end
      else
        if find_property(node^.called_by^.properties,
                         BODY_TYPE_prop_name, prop) then
          if prop^.text = FLAG_BODY_string then result := TRUE;

    node_is_FLAG_node := result;
  end { node_is_FLAG_node } ;


  function node_is_MEMORY_prim(node: mtree_node_ptr): boolean;
    { return TRUE if the given node is the incredibly anomalous
      simulator MEMORY primitive }
    var
      vers: macro_module_ptr;  { version for node (if interesting) }
      retval: boolean;
  begin
    { node_is_MEMORY_prim := 
        node's name is MEMEORY and
        it is a primitive and
        it comes from a sim directory
      A MEMORY prim was read if found, because it looks like a plumbing node }

    retval := FALSE;
    if node^.macro_name = MEMORY_prim_name then
      if node^.macro <> NIL then
	begin
	  vers := node^.macro^.version;
	  if vers <> NIL then
	    begin
      	      if er_issimple(vers) <> 0 then
		retval := 
		  name_from_string(er_dirtype(vers)) = SIM_compile_type;
	      if debug_13 then 
	        writeln(outfile, '(MEMORY dirtype = SIM) is ', retval);
	    end
	  else if debug_13 then writeln(outfile, 'MEMORY version is NIL');
	end
      else if debug_13 then writeln(outfile, 'MEMORY macro is NIL');
    node_is_MEMORY_prim := retval;
  end { node_is_MEMORY_prim } ;


  function is_a_plumbing_body(invoke: invoke_list_ptr): boolean;
    { return TRUE if the invocation has a BODY_TYPE=PLUMBING property }
    var
      prop: property_ptr;     { property returned from search }
  begin
    is_a_plumbing_body := FALSE;

    if find_property(invoke^.properties, BODY_TYPE_prop_name, prop) then
      if prop^.text = PLUMBING_string then
        is_a_plumbing_body := TRUE;
  end { is_a_plumbing_body } ;


  procedure process_all_bodies_in_the_drawing(node: mtree_node_ptr);
    { process all of the bodies in the current drawing instance }
    var
      invoke: invoke_list_ptr;        { invocation info for the current body }
      uses_NWC: boolean;              { TRUE if body uses NWC property }
      macro: macro_def_ptr;           { current macro def }
      son: mtree_node_ptr;            { current son of the node }
  begin
    if not node^.is_leaf_node and dumpsigdeflist_ok then
      dump_list_of_signal_definitions(outfile, node^.signals);

    macro := node^.macro;
    expands_to_primitives := FALSE;

    (**************************************************)
    (*                                                *)
    (*  parse all bodies and process plumbing bodies  *)
    (*                                                *)
    (**************************************************)

    invoke := macro^.invokes;
    while invoke <> NIL do
      begin
        if debug_13 then
          begin
            write(outfile, 'Starting compile of body (', level:1, '): ');
            writestring(outfile, invoke^.macro_name);
            writeln(outfile);
          end;

        new_mtree_node(node^.son);
        son := node^.son;
        init_node(son, node, invoke);

        push_error_info;
        current_body_name := invoke^.macro_name;
        current_page := invoke^.page_number;
        current_mtree_node := son;
        current_path_prop := invoke^.path;

        { evaluate the pin names and signals connected to the body }

        evaluate_bindings(son^.params, invoke^.bindings, 
	                  uses_NWC, son^.is_cardinal_tap);

        { process all plumbing bodies }

        if is_a_plumbing_body(invoke) or uses_NWC then
          son^.is_plumbing_node := TRUE;

        if son^.is_plumbing_node and not son^.is_cardinal_tap then
	  begin
            if PASS1(son, level+1, TRUE) then expands_to_primitives := TRUE;
	  end
        else
          pop_signal_stacks(son^.signals);

        current_mtree_node := node;

        pop_error_info;

        invoke := invoke^.next;
      end { while invoke <> NIL } ;

    if not is_plumbing_drawing then
      begin
        (**************************************************)
        (*                                                *)
        (*  resolve widths and assertions of all signals  *)
        (*                                                *)
        (**************************************************)

        if resolve_all_signal_widths(node) then;
        if resolve_all_signal_assertions(node) then;

        check_for_unresolved_signals(node);


        (***********************************)
        (*                                 *)
        (*  assign all specified synonyms  *)
        (*                                 *)
        (***********************************)

        assign_synonyms(node);


        (********************************************)
        (*                                          *)
        (*  copy properties onto formal parameters  *)
        (*                                          *)
        (********************************************)

        copy_properties_from_actuals(node);


        (***********************************)
        (*                                 *)
        (*  remove INHERIT_PIN properties  *)
        (*                                 *)
        (***********************************)

        remove_all_pin_properties(node);
      end;


    (*********************************)
    (*                               *)
    (*  process non-plumbing bodies  *)
    (*                               *)
    (*********************************)

    if debug_13 then
      writeln(outfile, 'Pass 1 (', level:1,
                       '): processing non-plumbing drawings');

    son := node^.son;
    while son <> NIL do
      begin
        if debug_13 then
          begin
            write(outfile, 'Processing drawing: ');
            writestring(outfile, son^.macro_name);
            if son^.is_plumbing_node then write(Outfile, ' PLUMBING');
            if son^.is_cardinal_tap then write(Outfile, ' -- CARDINAL TAP');
            write(outfile, ' Called by: ');
            if son^.father_node = NIL then write(outfile, '<NIL>')
            else
              writestring(outfile, son^.father_node^.macro_name);
	    writeln(Outfile);
          end;

        current_body_name := son^.called_by^.macro_name;
        current_page := son^.called_by^.page_number;

        if son^.is_plumbing_node then
          { Check for the 2 exceptions to the "NWC implies plumbing" rule }
          begin
	    if not son^.is_cardinal_tap then
	      if node_is_flag_node(son) or node_is_MEMORY_prim(son) then
		begin
		  son^.is_leaf_node := TRUE;  son^.is_plumbing_node := FALSE;
		  number_leaf_nodes := number_leaf_nodes + 1;
		  expands_to_primitives := TRUE;
		  if debug_13 then
		    writeln(outfile, 'Is a FLAG or MEMORY node');
		end;
          end
        else
          begin
            { Non-plumbing bodies don't get expanded }
            son^.is_leaf_node := TRUE;
            number_leaf_nodes := number_leaf_nodes + 1;
            expands_to_primitives := TRUE;
          end;

        son := son^.next;
      end;

    (***************************************)
    (*                                     *)
    (*  synonym all "virtual" signal defs  *)
    (*                                     *)
    (***************************************)

    (* This is now being done in pass1 main body -- it is postponed on the
       base node until after all x-replications are done for the sake of
       interface signals and globals which are created only for the base
       node. *)
(*
    if net_processing and not is_plumbing_drawing then
      synonym_virtual_defs(node);
*)

    (*********************************************)
    (*                                           *)
    (*  check that plumbing is correctly formed  *)
    (*                                           *)
    (*********************************************)

    if (node^.father_node <> NIL) and expands_to_primitives then
      if node^.is_plumbing_node then
        begin
          error(221 { this isn't permitted });
          error_dump_macro_def(node^.macro);
        end;

    if not is_plumbing_drawing then free_formal_parameters(node);

    if debug_13 then
      begin
        write(outfile, 'Exiting process: node=');
        print_string(outfile, node^.macro_name);
        writeln(outfile);
        dump_list_of_signal_definitions(outfile, node^.signals);
      end;
  end { process_all_bodies_in_the_drawing } ;


  procedure fix_signals_for_node(current_node, base_node: mtree_node_ptr);
    { fix up the signal stacks for the given node }
  begin
    pop_signal_stacks(current_node^.signals);
    if current_node <> base_node then current_node^.params := NIL;
  end { fix_signals_for_node } ;


  procedure create_X_replicated_node(var current_node: mtree_node_ptr);
    { create a "sibling" node of current_node for SIZE replication.
      the created sibling should immediately follow the old current_node
      in the list. mtree_root is the first replication of the node. }


    procedure init_context(node: mtree_node_ptr);
      { Insert the current context paramters into the symbol table for the
        node }
      var
        parm: property_ptr;        { current parm of context }
        temp: identifier_ptr;      { new ID }
    begin
      parm := specified_context; { global }
      with node^ do while parm <> NIL do
        begin
          temp := NIL;  new_identifier(temp);
          temp^.next := symbol_table;  symbol_table := temp;
          temp^.name := parm^.name;  temp^.definition := parm^.text;

          parm := parm^.next;
        end;
    end { init_context } ;


  begin { create_X_replicated_node }
    if current_node^.father_node <> NIL then
      begin
        assert(0 { should never happen });
        writeln(CmpLog, ' X replicating non-root-level node');
      end
    else
      begin
        if current_node = mtree_root then
          push_globals_and_xfaces_onto_signal_stacks;
        new_mtree_node(current_node^.next);
        current_node := current_node^.next;
      end;

    init_context(current_node);
    init_node(current_node, NIL, mtree_root^.called_by);
    create_current_context(current_node, mtree_root^.macro);

    current_node^.params := mtree_root^.params;
  end { create_X_replicated_node } ;


  procedure insure_unique_path_values;
    { insure that all PATH property values for current_mtree_node are
      unique.  (Change those that are not by appending a colon followed
      by a unique number.) }
    var
      invoke: invoke_list_ptr;   { current invoke }
      path: xtring;              { original path name of current invoke }


    function path_value_is_new(invoke: invoke_list_ptr): boolean;
      { return TRUE iff the path value has not yet been seen on another
        invoke }
      var
        object: avl_object_ptr;
        found: avl_ptr;
    begin
    { object.tag := AVL_INVOKE;                       }(*AVL*)
      object.invoke := invoke;
      found := avl_insert(object, invoke_path_table, AVL_INVOKE);
      path_value_is_new := (found^.object.invoke = invoke);
    end { path_value_is_new } ;


    function tweak_path(path: xtring): xtring;
      { return path with appended colon followed by unique_path_number
         and increment the number }
      var
        len: natural_number; { length of string if not truncated }
        temp: xtring;        { temp string for building new path name }
    begin
      len := ord(path^[0]) + 1 + width_of_integer(unique_path_number);
      create_a_string(temp, min(MAX_STRING_LENGTH, len));
      temp^[0] := chr(0);
      if len > MAX_STRING_LENGTH then
        begin
          { Use a truncated version of the original PATH property value --
            be sure to restore the string!! }
          path^[0] := chr(ord(path^[0]) - (len - MAX_STRING_LENGTH));
          if add_string_to_string(temp, path) then { can't fail } ;
          path^[0] := chr(ord(path^[0]) + (len - MAX_STRING_LENGTH));
        end
      else if add_string_to_string(temp, path) then { can't fail } ;
      if add_char_to_string(temp, ':') then { can't fail } ;
      if add_number_to_string(temp, unique_PATH_number) then { can't fail } ;
      { it is assumed that memory overflow due to number of records allocated
        would occur long before the following could overflow }
      unique_PATH_number := unique_PATH_number + 1;
      tweak_path := enter_and_release_string(temp);
    end { tweak_path } ;


  begin { insure_unique_path_values }
    if current_mtree_node^.macro <> NIL then
      begin
        invoke := current_mtree_node^.macro^.invokes;
        while (invoke <> NIL) do
          begin
            if not path_value_is_new(invoke) then
              begin
                path := invoke^.path;

                current_path_prop := path;  { for error message }
                error(176 { path name is not unique });
                error_dump_body_name(invoke^.macro_name);

                while not path_value_is_new(invoke) do
                  invoke^.path := tweak_path(path);

                error_dump_indent(indent);
                error_dump_alpha('Concocted path n');
                error_dump_alpha('ame:            ');
                error_dump_char(' ');
                error_dump_string(invoke^.path);
                error_dump_CRLF;
              end;
            invoke := invoke^.next;
          end;
      end;
    current_path_prop := NIL;
  end { insure_unique_path_values } ;


begin { PASS1 }
  if debug_13 then
    begin
      writeln(outfile, '------- Entering PASS 1(', level:1, ') -------');
      dump_formal_actual_list(outfile, node^.params);
    end;

  expands_to_primitives := FALSE;

  current_mtree_node := node;

  if (node^.father_node = NIL) then
    macro := read_macro_page(node^.macro_name)
  else
    macro := read_macro_def(node^.macro_name); { plumbing body }

  { create the current environment for error reporting }

  push_error_info;
  current_macro_def := macro;

  create_current_context(node, macro);
  if (node = mtree_root) then insure_unique_path_values;
  
  if node^.is_leaf_node then expands_to_primitives := TRUE
  else
    { check for an empty drawing }
    if macro = NIL then
      begin
        number_terminal_nodes := number_terminal_nodes + 1;
      end
    else if (macro^.invokes = NIL) then
      begin
        number_terminal_nodes := number_terminal_nodes + 1;
      end;

  evaluate_X_loop_controls(node, x_first, x_step, size);
  x := x_first;

  current_node := node;

  if debug_13 then
    begin
      write(outfile, 'IO signals for the node: ');
      print_string(outfile, node^.macro_name);
      writeln(outfile);
      dump_list_of_signal_definitions(outfile, node^.signals);
    end;

  done := FALSE;
  repeat
    current_node^.x_value := x;

    if current_node^.father_node <> NIL then
      display_PASS_1_status(current_node);

    if debug_at_path or undebug_at_path then check_path_debug(current_node);

    if macro = NIL then
      begin
        fix_signals_for_node(current_node, node);
        done := TRUE;
      end
    else
      begin
        { add the given value of X to the current node's symbol table }

        enter_id(current_node, X_identifier, id);
        if id = NIL then assert(33 { X already in the symbol table })
        else
          id^.definition := number_to_string(x);

        process_all_bodies_in_the_drawing(current_node);
        if (node <> current_node { which also implies not plumbing }) and
           net_processing then
          synonym_virtual_defs(current_node);

        fix_signals_for_node(current_node, node);

	if dumptree_ok then
	  begin
	    write(outfile, '--- Dump of the symbol table at end:');
	    writestring(outfile, node^.macro_name);
	    writeln(outfile);
	    dump_symbol_table(outfile, current_node^.symbol_table);
	  end;

        x := x + x_step;

        if x >= (x_first + size) then done := TRUE
        else create_X_replicated_node(current_node);
      end;
  until done;

  if net_processing and not is_plumbing_drawing then 
    synonym_virtual_defs(node);

  pop_error_info;

  PASS1 := expands_to_primitives;

  if debug_13 then
    begin
      writeln(outfile, '------- Exiting PASS 1(', level:1, ') -------');
      writeln(outfile);  writeln(outfile);
    end;
end { PASS1 } ;



