procedure mark_dirty(page: paged_schema_ptr);
  { Mark all results for the page as dirty. }
  var
    context: compiled_context_list_ptr;  { known contexts for this page }
begin
  context := page^.compiled_contexts;
  while context <> NIL do
    begin
      context^.dirty := TRUE;
      context := context^.next;
    end;
end { mark_dirty } ;


procedure mark_dirty_for_pass_2(*page: paged_schema_ptr*);
  { Mark all results for the page as dirty by reason of changes to
    local text macros in other pages. }
  var
    context: compiled_context_list_ptr;    { known contexts for this page }
begin
  context := page^.compiled_contexts;
  while context <> NIL do
    begin
      context^.dirty_for_pass_2 := TRUE;
      context := context^.next;
    end;
end { mark_dirty_for_pass_2 } ;


procedure reverse_numbered_token_lists(var lst: numbered_token_list_ptr);
  { reverse the list }
  var
    p: numbered_token_list_ptr;    { former parent of current }
    n: numbered_token_list_ptr;    { save lst^.next }
begin
  p := NIL;
  while lst <> NIL do
    begin
      n := lst^.next;  lst^.next := p;  p := lst;  lst := n;
    end;
  lst := p;
end { reverse_numbered_token_lists } ;
  

procedure dump_paged_schema_spec(var f: textfile; page: paged_schema_ptr);
  { report drawing type, version, and page to file f }
begin
  writealpha(f, page^.drawing_type^.identifier_p^.name);
  write(f, '.', page^.version:1, '.', page^.page:1);
end { dump_paged_schema_spec } ;


procedure dump_changed_ids(var f: textfile; 
                           changed_ids: numbered_token_list_ptr);  
  { report the changed ids to file f }
  var
    cid: numbered_token_list_ptr;  { current id }
begin
  write(f, '    changed_ids:');
  if changed_ids = NIL then write(f, ' <none>');
  cid := changed_ids;
  while cid <> NIL do
    begin
      write(f, ' ');
      writealpha(f, cid^.token^.identifier_p^.name);
      cid := cid^.next;
    end;
  writeln(f);
end { dump_changed_ids } ;

      
function changed_TMs(list1, list2: text_macro_ptr): 
  numbered_token_list_ptr;
  { return a list of the identifiers whose definitions differ
    between the 2 lists.  The text macro lists are assumed
    sorted (based on id number of name). The resulting list is returned
    sorted in the same order. }
  var
    tm1: text_macro_ptr;           { current list1 element }
    tm2: text_macro_ptr;           { current list2 element }
    head: numbered_token_list_ptr; { list for return }
    

  procedure insert_tm_name(tm: text_macro_ptr);
    { insert then tm name to the head of the list }
  begin
    new_numbered_token_list(head);
    head^.token := tm^.name;
  end { insert_tm_name } ;


begin { changed_TMs }
  tm1 := list1;  tm2 := list2;  head := NIL;
  while (tm1 <> NIL) or (tm2 <> NIL) do
    if tm1 = NIL then 
      begin
	insert_tm_name(tm2);  tm2 := tm2^.next;
       end
    else if tm2 = NIL then
      begin
	insert_tm_name(tm1);  tm1 := tm1^.next;
      end
    else
      begin
        if tm1^.name^.number = tm2^.name^.number then
	  begin
	    if tm1^.text^.string_p <> tm2^.text^.string_p then
	      insert_tm_name(tm1);
	    tm1 := tm1^.next;  tm2 := tm2^.next;
	  end
        else if tm1^.name^.number < tm2^.name^.number then
	  begin
	    insert_tm_name(tm1);  tm1 := tm1^.next;
	  end
        else
	  begin
	    insert_tm_name(tm2);  tm2 := tm2^.next;
	  end;
      end;
  reverse_numbered_token_lists(head);
  changed_TMs := head;
end { changed_TMs } ;


function changed_global_TMs(tm_list: text_macro_ptr): numbered_token_list_ptr;
  { check the (global) TM list against the currently defined
    global text macros and return a list of ids whose definitions
    have changed.  Return the list in the same order as input list. }
  var
    tm: text_macro_ptr;            { current element of tm_list }
    head: numbered_token_list_ptr; { list to be returned }


  procedure insert_tm_name(tm: text_macro_ptr);
    { insert then tm name to the head of the list }
  begin
    new_numbered_token_list(head);
    head^.token := tm^.name;
  end { insert_tm_name } ;


begin
  tm := tm_list;  head := NIL;
  while tm <> NIL do
    begin
      if (tm^.name^.identifier_p^.definition <> tm^.text^.string_p) then
        insert_tm_name(tm)
      else
        if tm^.reserved then
          begin
            if not (RESERVED in tm^.name^.identifier_p^.kind) then
              insert_tm_name(tm);
	  end
        else if tm^.text^.string_p <> nullstring then
          begin
            if (tm^.name^.identifier_p^.kind * [RESERVED,UNRESERVED]) <>
	       [UNRESERVED] then insert_tm_name(tm);
	  end;
      tm := tm^.next;
    end;
  reverse_numbered_token_lists(head);
  changed_global_TMs := head;
end { changed_global_TMs } ;


function match_an_id(ids: numbered_token_list_ptr;
                     exp_ids: expandable_id_ptr): boolean;
  { return TRUE iff any id in ids is also in exp_ids.  Assume that
    both lists are sorted on id number }
  var
    id: numbered_token_list_ptr;  { current element of ids }
    exp: expandable_id_ptr;       { current element of exp_ids }
    result: boolean;              { answer so far }
begin
  result := FALSE;  id := ids;  exp := exp_ids;
  while not result and (id <> NIL) and (exp <> NIL) do
    begin
      if id^.token^.number < exp^.name^.number then id := id^.next
      else if id^.token^.number = exp^.name^.number then result := TRUE
      else exp := exp^.next;
    end;
  match_an_id := result;
end { match_an_id } ;


procedure check_global_TMs(var schema: schema_definition);
  { checks the global text macros recorded in the schema against
    the currently defined text macros.  Any pages using TMs which 
    have changed are marked as dirty }
  var
    changed_ids: numbered_token_list_ptr;  { ids whose defs have changed }
    page: paged_schema_ptr;                { page being checked }
begin
  if debug_23 then
    begin
      write(outfile, 'check_global_TMs ');
      writestring(outfile, root_macro_name);
      writeln(outfile);
    end;

  changed_ids := changed_global_TMs(schema.used_global_TMs);
  
  if debug_23 then dump_changed_ids(outfile, changed_ids);

  if changed_ids <> NIL then
    begin
      page := schema.paged_schemas;
      while page <> NIL do
        begin
	  if debug_23 then 
	    begin
	      write(outfile, '    ');
	      dump_paged_schema_spec(outfile, page);
	    end;
	  if match_an_id(changed_ids, page^.expandable_ids) then
	    begin
	      if debug_23 then writeln(outfile, ' marked dirty');
	      mark_dirty(page);
	    end
	  else if debug_23 then writeln(outfile, ' unaffected');
	  page := page^.next;
	end;
      release_all_numbered_token_lists(changed_ids);
    end;
if debug_23 then writeln(outfile, 'exit check_global_TMs');
end { check_global_TMs } ;


procedure check_global_changes(var schema: schema_definition);
  { check directives, global text macros and property attributes.
    (property attributes not yet implemented) }
  var
    page: paged_schema_ptr;                { page being checked }
begin
  if (schema.bubble_check <> bubble_check) or
     (schema.enable_cardinal_tap <> enable_cardinal_tap) then
    begin
      if debug_23 then writeln(outfile, 'Boolean directive changed');
      page := schema.paged_schemas;
      while page <> NIL do
        begin
	  mark_dirty(page);
	  if debug_23 then 
            begin
	      write(outfile, '    ');
	      dump_paged_schema_spec(outfile, page);
	      writeln(outfile, ' marked dirty');
	    end;
	  page := page^.next;
	end;
    end
  else
    begin
      check_global_TMs(schema);
    end;
end { check_global_changes } ;


function check_local_TMs(var schema: schema_definition;
                         new_TMs: text_macro_ptr;
		         old_current_page: paged_schema_ptr): boolean;
  { determine whether there are any differences between the macros
    defined by new_TMs and those defined by old_current_page.  If
    there are, then check all pages of schema (other than old_current_page)
    to see if they use any of the changed macros as expandable ids.  
    Those that do are marked dirty.  Return FALSE iff differences are found. }
  var
    changed_ids: numbered_token_list_ptr;  { ids whose defs have changed }
    page: paged_schema_ptr;                { page being checked }
begin
  if debug_23 then 
    begin
      write(outfile, 'check_local_TMs of ');
      writestring(outfile, root_macro_name);
      write(outfile, ' against ');
      dump_paged_schema_spec(outfile, old_current_page);
      writeln(outfile);
      write(Outfile, ' new list: ');
      dump_text_macros(Outfile, new_TMs);
      write(Outfile, ' old list: ');
      dump_text_macros(Outfile, old_current_page^.local_text_macros);
    end;

  changed_ids := changed_TMs(new_TMs, old_current_page^.local_text_macros);
  check_local_TMs := (changed_ids = NIL);

  if debug_23 then dump_changed_ids(outfile, changed_ids);

  if changed_ids <> NIL then
    begin
      page := schema.paged_schemas;
      while page <> NIL do
        begin
	  if page <> old_current_page then
	    begin
	      if debug_23 then 
		begin
		  write(outfile, '    ');
		  dump_paged_schema_spec(outfile, page);
		end;
	      if match_an_id(changed_ids, page^.expandable_ids) then
	        begin
	          mark_dirty_for_pass_2(page);
		  if debug_23 then
		    writeln(outfile, ' marked dirty (by local TMs)');
	        end
	      else if debug_23 then writeln(outfile, ' is unaffected');
            end;
	  page := page^.next;
	end;
      release_all_numbered_token_lists(changed_ids);
    end;
  if debug_23 then writeln(outfile, 'end check_local_TMs');
end { check_local_TMs } ;
			  

function make_current_page: boolean;
  { perform the "make" on the current page and return TRUE if page
    needs to be made and no fatal errors were detected here.
    date on schema file versus page_being_compiled file and 
    other file dependencies.
    Also, if use_usage_lists, then the old_modules_list is read in. }
  var
    current_time: time_stamp;         { last modify time on current file }
    depends_on: dependency_list_ptr;  { files other than current page that
                                        were read when last compiling it }
    dirty: boolean;                   { TRUE iff page dirty for all 
                                        compiled contexts }
    val: boolean;                     { value to be returned }
    conn_file: xtring;                { connectitivity file name
                                        (in buffer owned by C code) }


  function create_compiled_context(page: paged_schema_ptr; 
				   context: context_definition_ptr):
    compiled_context_list_ptr;
    { find or create the compiled_context for the specified context under
      the page. }
    var
      cont: compiled_context_list_ptr;  { current context in list }
      found: boolean;                { TRUE iff context is found }
  begin
    cont := page^.compiled_contexts;  found := FALSE;
    while (cont <> NIL) and not found do
      if cont^.context = context then found := TRUE
      else cont := cont^.next;
    if not found then
      begin
	if (debug_23) then
	  writeln(Outfile, '    new compiled context');
	new_compiled_context_list(page^.compiled_contexts);
	cont := page^.compiled_contexts;
	cont^.context := context;
	cont^.dirty := TRUE;
	cont^.dirty_for_pass_2 := TRUE;
      end;
    create_compiled_context := cont;
  end { create_compiled_context } ;


begin { make_current_page }
  if debug_23 then 
    begin
      write(outfile, 'make_current_page ');
      writestring(outfile, root_macro_name);
      write(outfile, '.');
      dump_paged_schema_spec(outfile, old_schema_page);
      writeln(outfile);
    end;

  if not old_schema_page^.make_performed then
    begin
      old_schema_page^.make_performed := TRUE;
      dirty := FALSE;
        
      conn_file := er_filename(module_being_compiled, ord(CONNECTIVITY), 
                              page_being_compiled, NIL);
      if (ord(conn_file^[0]) = 0) then
        begin
	  dirty := TRUE;
	  if debug_23 then 
	    writeln(Outfile, '    Can''t concoct _cn file name');
	end
      else if not get_time_stamp(conn_file, current_time) then dirty := TRUE
	{ debug_23 prints result of get_time_stamp }
      else if current_time <> old_schema_page^.last_modified_time then
        begin
          dirty := TRUE;
	  if debug_23 then 
	    writeln(Outfile, '    _cn file modified ', 
	            old_schema_page^.last_modified_time:1, '<>',
		    current_time:1);
	  old_schema_page^.last_modified_time := current_time;
	end;

      depends_on := old_schema_page^.dependencies;
      while not dirty and (depends_on <> NIL) do
        if not file_exists(depends_on^.file_name^.string_p) then
          begin
            dirty := TRUE;
            if debug_23 then
              begin
                write(outfile, '    ');
                writestring(outfile, depends_on^.file_name^.string_p);
                writeln(outfile, ' does not exist');
              end;
          end
        else if not get_time_stamp(depends_on^.file_name^.string_p,
	                           current_time) then dirty := TRUE
        else if current_time <> depends_on^.last_modified_time then
	  begin
            dirty := TRUE;
            if debug_23 then
              begin
                write(outfile, '    ');
                writestring(outfile, depends_on^.file_name^.string_p);
                writeln(outfile, ' has been changed');
              end;
	  end
        else depends_on := depends_on^.next;

      if dirty then mark_dirty(old_schema_page); 
    end;

  current_compiled_context := 
    create_compiled_context(old_schema_page, context_being_compiled);

  if make_pass = MAKE_PASS_1 then
    if not file_exists(page_expansion_file_name) then
      begin
        current_compiled_context^.dirty := TRUE;
        if debug_23 then writeln(outfile, '    no expansion file');
      end;

  if make_pass = MAKE_PASS_1 then val := current_compiled_context^.dirty
  else val := current_compiled_context^.dirty_for_pass_2;

  if debug_23 then
    if val then writeln(Outfile, '    make returns TRUE')
    else writeln(Outfile, '    make returns FALSE');

  make_current_page := val;
end { make_current_page } ;
            
     
        
