procedure new_signal_instance_list(var head: signal_instance_list_ptr);
  { new one of these and insert it into the list before head }
  var
    ptr: signal_instance_list_ptr;
begin
  if free_signal_instance_lists = NIL then new(ptr)
  else 
    begin
      ptr := free_signal_instance_lists;
      free_signal_instance_lists := free_signal_instance_lists^.next;
    end;
  ptr^.instance := NIL;
  ptr^.next := head;
  head := ptr;
end { new_signal_instance_list } ;


procedure record_instance_for_synonyms(*instance: signal_instance_ptr*);
  { add to list of instance pointers }
begin
  new_signal_instance_list(bogus_cmptmp_list);
  bogus_cmptmp_list^.instance := instance;
end { record_instance_for_synonyms } ;


procedure enter_into_synonyms_table(instance: signal_instance_ptr);
  { search for the signal in the signal table.  If not found, enter it.
    The table keeps track of signals by name and polarity.  The subscript
    of the instance is unioned into the existing subscript for the same
    signal (which results in a list of all bits used in the page). }
  var
    signal_name: xtring;                { name of the signal }
    polarity: signal_polarity;          { polarity of signal }
    i: string_range;                    { index into the signal name }
    sum: natural_number;                { checksum of the signal name }
    index: synonym_signal_table_range;  { index into the table }
    signal: synonym_signal_ptr;         { current entry in the table }
    found: boolean;                     { TRUE when signal found in table }
    node: mtree_node_ptr;               { node defining signal }
begin
  signal_name := instance^.defined_by^.signal^.name;
  node := instance^.defined_by^.node;
  if not instance^.defined_by^.is_const then
    if not is_NC_signal(signal_name) then
      begin
        polarity := instance^.defined_by^.polarity;

	signal_name :=
	  X_replicated_name(signal_name, node, instance^.defined_by^.scope);

        if debug_9 then
          begin
            write(outfile, 'Find_or_enter: name=');
            dump_signal_polarity(outfile, polarity);
            writestring(outfile, signal_name);
            dump_bit_subscript(outfile, instance^.bit_subscript, VECTOR);
            writeln(outfile);
          end;

        { calculate a hash index into the table }

        sum := 0;
        for i := 1 to ord(signal_name^[0]) do
          sum := sum + ord(signal_name^[i]);
        index := sum MOD (SYNONYM_SIGNAL_TABLE_SIZE+1);

        { find the signal within the given hash table bucket }

        signal := synonym_signal_table[index];  found := FALSE;
        while (signal <> NIL) and not found do
          begin
            if signal^.signal_name = signal_name then
             if signal^.polarity = polarity then found := TRUE;
            if not found then signal := signal^.next;
          end;

        if not found then
          begin
            new_synonym_signal(synonym_signal_table[index]);
            signal := synonym_signal_table[index];

            signal^.def := virtual_base(instance^.defined_by);
            signal^.signal_name := signal_name;
            signal^.polarity := polarity;
            signal^.used_bits := NIL;
          end;

        if debug_9 then
          begin
            if found then write(outfile, 'found entry ')
            else write(outfile, 'new entry ');
            write(outfile, ord(signal):1); 
	    writeln(outfile);
          end;
        add_to_bit_union(instance^.bit_subscript, signal^.used_bits);
      end;
end { enter_into_synonyms_table } ;


 
procedure output_signal_instance(
  var f: pipe; instance: signal_instance_ptr);
  { print the signal instance to the specified file.  If the current
    line is overflowed (MAX_OUTPUT_FILE_LENGTH), print a continuation
    character and continue on the next line.  If the insance is a
    constant, then that  constant is printed, else 
    instance^.defined_by^.signal_name_token_number is 
    output as a numbered string representing the signal name. 
    Note: replication is done by concatennation. }
  var
    rep_count: replication_range;  { counts up to the replication 
                                     of the instance }
    name: xtring;                  { signal name }


  procedure dump_constant(var f: pipe; instance: signal_instance_ptr);
    { dump out a constant }
    var
      signal_name: xtring;       { name of the signal (constant digits) }
      length: string_range;      { length of the constant 'signal name' }
      bit: -1..MAX_BIT_VALUE;    { bit being printed }
      sub: subscript_ptr;        { current subscript }
      replication: replication_range;    { replication of the signal }


    procedure write_bit(bit: bit_range);
      { write out the bit.  NOTE: the sepcomp version never replicates a
        bit with REP. }
      var
        ch: char;        { constant 'digit' to be output }
    begin
      if left_to_right then ch := signal_name^[bit+1]
                       else ch := signal_name^[length-bit];

      if ch = '0' then
        output_numbered_token(f, expansion_0_string^, STRING_NUMBER)
      else
        output_numbered_token(f, expansion_1_string^, STRING_NUMBER);
    end { write_bit } ;


  begin { dump_constant }
    signal_name := instance^.defined_by^.signal^.name;
    length := ord(signal_name^[0]);
    replication := instance^.replication_factor;
    repeat
      sub := instance^.bit_subscript;
      if sub = NIL then write_bit(0)    { it's a scalar }
      else
        while sub <> NIL do
          begin
            bit := sub^.left_index;
            if bit <= sub^.right_index then
              repeat
                write_bit(bit);  bit := bit + 1;
              until bit > sub^.right_index
            else
              repeat
                write_bit(bit);  bit := bit - 1;
              until bit < sub^.right_index;
            sub := sub^.next;
          end;

      replication := replication - 1;
    until replication <= 0;
  end { dump_constant } ;


begin { output_signal_instance }
  if instance <> NIL then
    with instance^.defined_by^ do
      begin
        name := X_replicated_name(signal^.name, node, scope);
        if is_const then
          dump_constant(f, instance)
        else
          
          begin
            rep_count := 0;
            repeat
              rep_count := rep_count + 1;
              if polarity = COMPLEMENTED then 
                pipe_dump_char(f, '-');
              lookup_and_print_string_continue(f, name, 
                                               expansion_string_dictionary);
              output_bit_subscript(f, instance^.bit_subscript, kind);

            until rep_count >= instance^.replication_factor;
          end;
      end;
end { output_signal_instance } ;


procedure output_synonyms(var f: pipe);
  { output all the signals in the synonym table to the file }
  var
    index: synonym_signal_table_range; { index into the table }
    signal: synonym_signal_ptr;        { current signal in bucket }
    inst: signal_instance_list_ptr;    { current inst in bogus_cmptmp_list }

  procedure dump_synonym_table(var f: textfile);
    { output the synonym table to the given file }
    var
      index: synonym_signal_table_range;    { index into the table }
      signal: synonym_signal_ptr;           { current signal in bucket }
  begin
    writeln(f, 'Dump of the synonym table');

    for index := 0 to SYNONYM_SIGNAL_TABLE_SIZE do
      begin
        signal := synonym_signal_table[index];
        while signal <> NIL do
          begin
	    dump_signal_polarity(f, signal^.polarity);
            writestring(f, signal^.signal_name);
	    
            print_signal_scope(f, signal^.def^.scope);

            print_alpha(f, 'Used bits:      ');
            dump_bit_subscript(f, signal^.used_bits, VECTOR);
	    writeln(f);

            signal := signal^.next;
          end;
      end;
  end { dump_synonym_table } ;

  
  procedure output_base_signal(signal_name: xtring;
                               polarity: signal_polarity;
                               scope: scope_type;
                               left_bit, right_bit: bit_range;
                               instance: signal_instance_ptr);

    { output the given signal (SIGNAL_NAME, LEFT, RIGHT) with its base
      signal equivalent (INSTANCE, SIGNAL) to the file as expansion data. }
    var
      actual_signal: propertied_CS_ptr;    { actual signal }
      base_signal: propertied_CS_ptr;      { actual base signal }
      left, right: bit_range;              { current subscript }
      width: bit_range;                    { width of signal }
      left_to_right: boolean;              { TRUE if order L-to-R }


    procedure print_the_signal(left, right: bit_range;
                               instance: signal_instance_ptr);
      { print the given signal to the output file.  The signal is
        specified by the global vars SIGNAL_NAME and POLARITY, and
        the parameters LEFT and RIGHT.  Use sepcomp expansion syntax. }
      var
        replication: replication_range;    { temp store of replication }
    begin
      pipe_dump_CRLF(f);
      pipe_dump_char(f, '*');  { exp file signal symonym prefix }

      { output the local name first }

      if polarity = COMPLEMENTED then
        pipe_dump_char(f, '-');
      if is_NC_signal(signal_name) then 
        output_numbered_token(f, expansion_NC_string^, STRING_NUMBER)
      else
        lookup_and_print_string_continue(f, signal_name, 
                                         expansion_string_dictionary);

      if left <> -1 then
        output_left_and_right(f, left, right);

      output_signal_scope(f, scope);

      { if signal is scalar, make sure replication of base is 1.  This
        is a KLUDGE to handle NC signals which are scalars. }

      replication := base_signal^.instance^.replication_factor;
      if left = -1 then
        base_signal^.instance^.replication_factor := 1;

      { output the base signal name }

      output_signal_instance(f, base_signal^.instance);

      { restore replication (see above) }

      if left = -1 then
        base_signal^.instance^.replication_factor := replication;
    end { print_the_signal } ;


  function same_signal(left, right: bit_range): boolean;
    { return TRUE if the given bits of the actual to be output are their
      own base.  NOTE: it is assumed that this function is called only
      on non-LOCAL signals.  It will not work on X-replicated signals. }
    var
      instance: signal_instance_ptr;    { base signal instance }
  begin
    instance := base_signal^.instance;
    if (signal_name = instance^.defined_by^.signal^.name) and
       (polarity = instance^.defined_by^.polarity) then
      if instance^.bit_subscript <> NIL then
	same_signal := (instance^.bit_subscript^.left_index = left) and
	               (instance^.bit_subscript^.right_index = right)
      else same_signal := TRUE
    else same_signal := FALSE;
  end { same_signal } ;

     
  begin { output_base_signal }

    
    actual_signal := NIL;
    new_propertied_CS(actual_signal);
    actual_signal^.instance := instance;

    { determine the base name for the signal }

    base_signal := find_base_of_PCS(actual_signal);


    if base_signal^.next = NIL then
      begin
        if debug_9 then
          begin
            write(outfile, 'Output of the signal: ');
            writestring(outfile, signal_name);
            writeln(outfile, '<', left_bit:1, '..', right_bit:1, '>');
          end;

        if scope = LOCAL then
          print_the_signal(left_bit, right_bit, base_signal^.instance)
        else 
          if not same_signal(left_bit, right_bit) then
            print_the_signal(left_bit, right_bit, base_signal^.instance)
	  else 
	    if debug_9 then 
	      writeln(outfile, '(skipped -- non-local signal is own base)');
      end
    else
      begin
        left := left_bit;  left_to_right := (left_bit < right_bit);
        while base_signal <> NIL do
          begin
            width := 
                  width_of_subscript(base_signal^.instance^.bit_subscript,
                                base_signal^.instance^.defined_by^.kind) *
                                base_signal^.instance^.replication_factor;
            if left_to_right then
              right := left + width - 1
            else
              right := left - width + 1;

            if debug_9 then
              begin
                write(outfile, 'Output of the signal: ');
                writestring(outfile, signal_name);
                writeln(outfile, '<', left:1, '..', right:1, '>');
              end;

	    if scope = LOCAL then
              print_the_signal(left, right, base_signal^.instance)
	    else if not same_signal(left, right) then
              print_the_signal(left, right, base_signal^.instance)
            else
	      if debug_9 then 
	        writeln(outfile, '(skipped -- non-local signal is own base)');

            if left_to_right then
              left := right + 1
            else
              left := right - 1;

            base_signal := base_signal^.next;
          end;
      end;
  end { output_base_signal } ;


  procedure output_all_used_bits_of_def(signal: synonym_signal_ptr);
    { output all the bits of the given signal.  If the signal is NC,
      ignore it: don't output it.  }
    var
      actual_descriptor:
                signal_descriptor_ptr;  { entire actual description }
      def: signal_definition_ptr;       { actual signal def }
      subscript: subscript_ptr;         { subscript for actual }
      synonym_subscript: subscript_ptr; { current element of used bits }


    procedure output_the_signal;
      { create an instance, and output the signal }
      var
        actual_instance: signal_instance_ptr;  { instance for actual }
        left, right: bit_range;       { subscript for signal }
    begin
      if def^.kind = SINGLE then
        begin
          { pick the first instance (there can be only one) }

          left := -1;  right := -1;
          actual_instance := def^.instances;
        end
      else
        begin
          left := subscript^.left_index;
          right := subscript^.right_index;
          actual_instance := enter_signal_instance(actual_descriptor,
                                                   def);
        end;

      output_base_signal(signal^.signal_name,
                         signal^.polarity,
                         def^.scope,
                         left, right,
                         actual_instance);
   end { output_the_signal } ;


  begin { output_all_used_bits_of_def }
    def := signal^.def;

    if not is_NC_signal(def^.signal^.name) then
      begin
        if debug_9 then
          begin
            write(outfile, 'Def to be output: ');
            dump_signal_definition(outfile, def);
            write(outfile, 'Bits used: ');
            dump_bit_subscript(outfile, signal^.used_bits, 
              def^.kind);
            writeln(outfile);
          end;

        actual_descriptor := NIL;
        new_signal_descriptor(actual_descriptor);
	with actual_descriptor^ do
	  begin
            signal_name := def^.signal^.name;
            kind := def^.kind;
            polarity := def^.polarity;
	    low_asserted := (polarity = COMPLEMENTED);
            scope := def^.scope;
            net_id := def^.net_id;
            subscript := NIL;
            new_subscript(subscript);
            bit_subscript := subscript;
	  end;

        if def^.kind = SINGLE then
          begin
            if signal^.used_bits <> NIL then
              begin
                assert(6 { non-vector has subscripts });
                dump_signal_definition(CmpLog, def);
                writeln(CmpLog, 'Synonym table says has used bits:');
                dump_bit_subscript(CmpLog, signal^.used_bits, 
                                   VECTOR);
                if debug_9 then
                  begin
                    dump_signal_definition(outfile, def);
                    writeln(outfile, 
                           'Synonym table says has used bits:');
                    dump_bit_subscript(outfile, signal^.used_bits, VECTOR);
                  end;
              end;
            subscript^.left_index := def^.left_index;
            subscript^.right_index := def^.right_index;
            output_the_signal;
          end
        else
          begin
            synonym_subscript := signal^.used_bits;
            while synonym_subscript <> NIL do
              with synonym_subscript^ do 
                begin
                  subscript^.left_index := left_index;
                  subscript^.right_index := right_index;
                  output_the_signal;
                  synonym_subscript := next;
                end;
          end;

        release_subscript(actual_descriptor^.bit_subscript);
        release_signal_descriptor(actual_descriptor);
      end;
  end { output_all_used_bits_of_def } ;


  procedure output_for_scope_spec(signal: synonym_signal_ptr);
    { output the signal as though synonymed to itself -- this is done
      merely to inform the linker of the scope of the signal.  This
      gets around what is really a linker bug where the linker gets
      confused if it does not know that a signal is non-LOCAL before
      encountering it on the right side of a synonyming.  (It evaluates
      each synonyming itself, so needs that data. ) }
    var
      subrange: subscript_ptr;      { current subrange of signal^.used_bits }
  begin
    subrange := signal^.used_bits;
    repeat  
      { NOTE: The linker counts on the compiler to pre-define ALL of the
        bits of non-local signals in this section, as the routing that
	parses it is the only one that knows how to associate interface
	signals with the pins they represent and global signals with
	their global page representation.  (See linker data structures) }

      pipe_dump_CRLF(f);
      pipe_dump_char(f, '*');  { exp file signal symonym prefix }

      if signal^.polarity = COMPLEMENTED then
        pipe_dump_char(f, '-');
      if is_NC_signal(signal^.signal_name) then 
        output_numbered_token(f, expansion_NC_string^, STRING_NUMBER)
      else
        lookup_and_print_string_continue(f, signal^.signal_name, 
                                         expansion_string_dictionary);
					  
      if subrange <> NIL then
	output_left_and_right(f, subrange^.left_index,
	                                subrange^.right_index);

      output_signal_scope(f, signal^.def^.scope);
					  
      if signal^.polarity = COMPLEMENTED then
        pipe_dump_char(f, '-');
      if is_NC_signal(signal^.signal_name) then 
        output_numbered_token(f, expansion_NC_string^, STRING_NUMBER)
      else
        lookup_and_print_string_continue(f, signal^.signal_name, 
                                         expansion_string_dictionary);

      if subrange <> NIL then
        begin
	  output_left_and_right(f, subrange^.left_index, 
	                                  subrange^.right_index);
	  subrange := subrange^.next;
	end;
    until subrange = NIL;
  end { output_for_scope_spec } ;


  function HACK1_needed(sig: synonym_signal_ptr): boolean;
    { Return TRUE if sig is LOCAL and has a non-local complement.
      This will cause it to be output for scope specification and will
      prevent the triggering of an order-dependent linker bug.  (If the
      linker sees the local signal as a base before seeing it with its
      scope spec, it will assign it the scope of its complement, which is
      not correct).  The linker bug is fixed in any post 1.1.7 version of
      the linker.  This hack can be removed when all tools with which the
      page compiler will be used have been converted to a post 1.1.7 linker. }


    function has_nonlocal_complement(signal_name: xtring; 
                                     polarity: signal_polarity): boolean;
      { return TRUE iff the given signal has a non-LOCAL complement (by
        looking up same in the synonym table) }
      var
        i: string_range;                    { index into the signal name }
        sum: natural_number;                { checksum of the signal name }
        index: synonym_signal_table_range;  { index into the table }
        signal: synonym_signal_ptr;         { current entry in the table }
        found: boolean;                     { TRUE when signal found }
    begin
      has_nonlocal_complement := FALSE;
      if (polarity = NORMAL) or (polarity = COMPLEMENTED) then
        begin
          if polarity = NORMAL then polarity := COMPLEMENTED
          else if polarity = COMPLEMENTED then polarity := NORMAL;

          { calculate a hash index into the table }

          sum := 0;
          for i := 1 to ord(signal_name^[0]) do
            sum := sum + ord(signal_name^[i]);
          index := sum mod (SYNONYM_SIGNAL_TABLE_SIZE+1);

          { find the signal within the given hash table bucket }

          signal := synonym_signal_table[index];  found := FALSE;
          while (signal <> NIL) and not found do
            begin
              if signal^.signal_name = signal_name then
                if signal^.polarity = polarity then found := TRUE;
              if not found then signal := signal^.next;
            end;

          { check for non-local scope }

          if found then
	    if signal^.def^.scope <> LOCAL then 
	      has_nonlocal_complement := TRUE;
        end;
    end { has_nonlocal_complement } ;


  begin { HACK1_needed }
    HACK1_needed := FALSE;
    if sig^.def^.scope = LOCAL then
      if not is_unnamed_signal(sig^.signal_name) then
        if has_nonlocal_complement(sig^.signal_name, sig^.polarity) then
          HACK1_needed := TRUE;
  end { HACK1_needed } ;


begin { output_synonyms }
  inst := bogus_cmptmp_list;
  while inst <> NIL do
    begin
      enter_into_synonyms_table(inst^.instance);
      if inst^.next = NIL then
        begin
          inst^.next := free_signal_instance_lists;
          free_signal_instance_lists := bogus_cmptmp_list;
          inst := NIL;
          bogus_cmptmp_list := NIL;
        end
      else inst := inst^.next;
    end;

  if debug_9 then dump_synonym_table(outfile);
  for index := 0 to SYNONYM_SIGNAL_TABLE_SIZE do
    begin
      signal := synonym_signal_table[index];
      while signal <> NIL do
        begin
          if (signal^.def^.scope <> LOCAL) or HACK1_needed(signal) then
            output_for_scope_spec(signal);
          signal := signal^.next;
        end;
    end;

  for index := 0 to SYNONYM_SIGNAL_TABLE_SIZE do
    begin
      signal := synonym_signal_table[index];
      while signal <> NIL do
        begin
          output_all_used_bits_of_def(signal);
          signal := signal^.next;
        end;
    end;
  pipe_dump_char(f, ';');
end { output_synonyms } ;


