(**)     { ------- file utilities ------- }


procedure printParseType(var f: textfile;
                         parsefile: parse_file_type);
  { print out the enumerated type "parse_file_type" into the given file f. }
begin
  case parsefile of
    DIRECTIVES_FILE:    write(f, 'DIRECTIVES_FILE');
    STANDARD_FILE:      write(f, 'STANDARD_FILE');
    CMPDRAW_FILE:       write(f, 'CMPDRAW_FILE');
    CMPSCHEM_FILE:      write(f, 'CMPSCHEM_FILE');
    OTHERWISE          begin  write(f, 'UNKNOWN_FILE')  end;
  end;
end { printParseType } ;


function rewrite_file(*var txtfil: textfile; filename: xtring; 
                      logical: alpha): boolean*);
  { rewrite a file of the given name.  Return FALSE if not successful }
  var
    mode: open_mode;     { buffered or unbuffered ? }
    val: boolean;        { return value }
    code: longint;       { system dependent return code }
begin
  if debug_22 then
    begin
      write(outfile, 'rewriting file: ');
      if filename = nullstring then print_alpha(outfile, logical)
      else writestring(outfile, filename);
      writeln(outfile);
    end;

  if debugging then mode := UNBUF_WRITE else mode := WRITE_MODE;

  val := vopen(txtfil, logical, filename, mode, code);
#if UNIX
  if debug_22 and val then 
    writeln(Outfile, '  fd = ', text_file_descr(txtfil):1); 
#endif
  if not val then
    if logical = MONITOR_FILE_NAME then
      begin
#if SVS
        writeln(stderr, 'pcomp: unable to open MONITOR (fatal)'); 
        write_ioresult(stderr, to_ioresult(code)); 
        halt_with_status(FATAL_COMPLETION); 
#endif
        writeln(output, 'pcomp: unable to open MONITOR (fatal)');
        write_ioresult(output, to_ioresult(code));
        halt_with_status(FATAL_COMPLETION);
      end
    else if logical = CMPLOG_FILE_NAME then
      begin
        writeln(Monitor, 'pcomp: unable to open CMPLOG (fatal)');
        write_ioresult(Monitor, to_ioresult(code));
        halt_with_status(FATAL_COMPLETION);
      end
    else
      begin
        error(169 { cannot open this file });
        if filename = nullstring then error_dump_alpha_file_name(logical)
	else error_dump_file_name(filename);
	error_dump_ioresult(to_ioresult(code));
	if debug_22 then writeln(Outfile, '  Original error code ', code:1);
      end;
 
  rewrite_file := val;

end { rewrite_file } ;

{Thie is the Pascal call to be used for rewriting files maintained in the data services
The logical name used here should be defined as a part of the DS environment. Hence 
there are are two environments, one maintained by the data services, the other
being the UNIX environment.}

function rewrite_ds_file(*var txtfil: textfile; filename: xtring; 
                      logical: alpha): boolean*);
  { rewrite a file of the given name.  Return FALSE if not successful }
  var
    mode: open_mode;     { buffered or unbuffered ? }
    val: boolean;        { return value }
    code: longint;       { system dependent return code }
begin
  if debug_22 then
    begin
      write(outfile, 'rewriting ds file: ');
      if filename = nullstring then print_alpha(outfile, logical)
      else writestring(outfile, filename);
      writeln(outfile);
    end;

  if debugging then mode := UNBUF_WRITE else mode := WRITE_MODE;

  val := ds_vopen(txtfil, logical, filename, mode, code);
#if UNIX
  if debug_22 and val then 
    writeln(Outfile, '  fd = ', text_file_descr(txtfil):1); 
#endif
  if not val then
    if logical = MONITOR_FILE_NAME then
      begin
#if SVS
        writeln(stderr, 'pcomp: unable to open MONITOR (fatal)'); 
        write_ioresult(stderr, to_ioresult(code)); 
        halt_with_status(FATAL_COMPLETION); 
#endif
        writeln(output, 'pcomp: unable to open MONITOR (fatal)');
        write_ioresult(output, to_ioresult(code));
        halt_with_status(FATAL_COMPLETION);
      end
    else if logical = CMPLOG_FILE_NAME then
      begin
        writeln(Monitor, 'pcomp: unable to open CMPLOG (fatal)');
        write_ioresult(Monitor, to_ioresult(code));
        halt_with_status(FATAL_COMPLETION);
      end
    else
      begin
        error(169 { cannot open this file });
        if filename = nullstring then error_dump_alpha_file_name(logical)
	else error_dump_file_name(filename);
	error_dump_ioresult(to_ioresult(code));
	if debug_22 then writeln(Outfile, '  Original error code ', code:1);
      end;
 
  rewrite_ds_file := val;

end { rewrite_ds_file } ;


function reset_file(*filename: string; which: parse_file_type): boolean*);
  { reset a file of the given name.  If the name is NULL, open nameless.  If
    no file can be opened, return FALSE.  WHICH specifies which logical 
    file is to be opened. }
  var
    logical: alpha;                     { for passing logical name }
    val: boolean;                       { return value }
    code: longint;                      { system dependent code }


#if UNIX
  procedure dump_fd(var f: inputfile);
    { dump the file descripter to outfile }
  begin
    writeln(Outfile, '  fd = ', cfdsc(f):1); 
  end { dump_fd } ;
#endif


begin { reset_file }
  if current_file <> UNKNOWN_FILE then
    begin
      val := FALSE;
      assert(169 { tried to open two files at once });

      write(CmpLog, ' Current file open = ');
      printParseType(CmpLog, current_file);
      writeln(CmpLog);
      write(CmpLog, ' New file to open = '); 
      printParseType(CmpLog, which);
      writeln(CmpLog);
    end
  else
    begin
      if (debug or debug_22) and (which <> DIRECTIVES_FILE) then 
        begin
          write(outfile, 'reseting ');
          printParseType(outfile, which);
	  if filename <> nullstring then
	    begin
	      write(outfile, ' as ');
	      writestring(outfile, filename);
	    end;
	  writeln(outfile);
        end;

      case which of
        DIRECTIVES_FILE:
          begin
            logical := 'INFILE          ';
#if !UNIX
            val := vopen(infile, logical, filename, READ_MODE, code);
#else
            val := creset(infile, logical, filename, BUFSIZ); 
	    if debug_22 and val then dump_fd(infile); 
#endif
          end;

        STANDARD_FILE:
          begin
            logical := 'CMPSTAN         ';
#if !UNIX
            val := vopen(CmpStan, logical, filename, READ_MODE, code);
#else
            val := creset(CmpStan, logical, filename, BUFSIZ); 
	    if debug_22 and val then dump_fd(CmpStan); 
#endif
          end;

        CMPDRAW_FILE:
          begin
            logical := 'CMPDRAW         ';
#if !UNIX
            val := vopen(CmpDraw, logical, filename, READ_MODE, code);
#else
            val := creset(CmpDraw, logical, filename, BUFSIZ); 
	    if debug_22 and val then dump_fd(CmpDraw);
#endif 
          end;

        CMPSCHEM_FILE:
          begin
            logical := 'CMPSCHEM        ';
#if !UNIX
            val := vopen(CmpSchem, logical, filename, READ_MODE, code);
#else
            val := creset(CmpSchemI, logical, filename, BUFSIZ); 
	    if debug_22 and val then dump_fd(CmpSchemI);
#endif 
          end;

        OTHERWISE
          begin
            val := FALSE;
            assert(170 { attempt to open an unknown file });
          end;
      end { case } ;
    end;
  if val then current_file := which;
  reset_file := val;
end { reset_file } ;

{This is the call to reset files maintained by data services}
function reset_ds_file(*filename: string; which: parse_file_type): boolean*);
  { reset a file of the given name.  If the name is NULL, return FALSE.  If
    no file can be opened, return FALSE.  WHICH specifies which logical 
    file is to be opened. }
  var
    logical: alpha;                     { for passing logical name }
    val: boolean;                       { return value }
    code: longint;                      { system dependent code }


#if UNIX
  procedure dump_fd(var f: inputfile);
    { dump the file descripter to outfile }
  begin
    writeln(Outfile, '  fd = ', cfdsc(f):1); 
  end { dump_fd } ;
#endif



begin { reset_ds_file }
  if current_file <> UNKNOWN_FILE then
    begin
      val := FALSE;
      assert(169 { tried to open two files at once });

      write(CmpLog, ' Current file open = ');
      printParseType(CmpLog, current_file);
      writeln(CmpLog);
      write(CmpLog, ' New file to open = '); 
      printParseType(CmpLog, which);
      writeln(CmpLog);
    end
  else
    begin
      if debug or debug_22 then 
        begin
          write(outfile, 'reseting ');
          printParseType(outfile, which);
	  if filename <> nullstring then
	    begin
	      write(outfile, ' as ');
	      writestring(outfile, filename);
	    end;
	  writeln(outfile);
        end;

      case which of
        DIRECTIVES_FILE:
          begin
            logical := 'INFILE          ';

            val := ds_creset(infile, logical, filename, BUFSIZ); 
	    if debug_22 and val then dump_fd(infile); 
          end;

        STANDARD_FILE:
          begin
            logical := 'CMPSTAN         ';
            val := ds_creset(CmpStan, logical, filename, BUFSIZ); 
	    if debug_22 and val then dump_fd(CmpStan); 
          end;

        CMPDRAW_FILE:
          begin
            logical := 'CMPDRAW         ';
            val := ds_creset(CmpDraw, logical, filename, BUFSIZ); 
	    if debug_22 and val then dump_fd(CmpDraw);
          end;

        CMPSCHEM_FILE:
          begin
            logical := 'CMPSCHEM        ';
            val := ds_creset(CmpSchemI, logical, filename, BUFSIZ); 
	    if debug_22 and val then dump_fd(CmpSchemI);
          end;

        OTHERWISE
          begin
            val := FALSE;
            assert(170 { attempt to open an unknown file });
          end;
      end { case } ;
    end;
  if val then current_file := which;
  reset_ds_file := val;
end { reset_ds_file } ;


function close_parse_file(*which: parse_file_type): boolean*);
  { close the input file }
var           
   val: boolean;       { return value }
begin
  if debug_22 then
    begin
      write(outfile, 'closing file ');
      printParseType(outfile, which);
#ifdef UNIX
      write(outfile, ' file descriptor: ');
      case which of
        DIRECTIVES_FILE:  writeln(outfile, cfdsc(infile):1);
        STANDARD_FILE:    writeln(outfile, cfdsc(CmpStan):1);
        CMPDRAW_FILE:     writeln(outfile, cfdsc(CmpDraw):1);
        CMPSCHEM_FILE:    writeln(outfile, cfdsc(CmpSchemI):1);
        UNKNOWN_FILE:     writeln(outfile);
      end;
#else
      writeln(outfile); 
#endif
    end;

  { NOTE: CMPSCHEM_FILE can be closed out of turn -- it is kept open on
    some platforms until after being written to preserve file locking.
    In fact it is never the "current_file" when it is actually closed. }

  if (which <> current_file) and (which <> CMPSCHEM_FILE) then
    begin
      assert(171 { tried to close wrong file! });
      writeln(CmpLog, 'Current_file = ', ord(current_file),
                      ';  File to be closed = ', ord(which));
      val := FALSE;
    end
  else
    begin
      case which of
        DIRECTIVES_FILE:  
#if UNIX
          val := cfclose(infile);
#else
          val := vclose(infile);
#endif
        STANDARD_FILE:    
#if UNIX
          val := cfclose(CmpStan);
#else
          val := vclose(CmpStan);
#endif
        CMPDRAW_FILE:     
#if UNIX
          val := cfclose(CmpDraw);
#else
          val := vclose(CmpDraw);
#endif
        CMPSCHEM_FILE:     
#if UNIX
          val := cfclose(CmpSchemI);
#else
          val := vclose(CmpSchem);
#endif
        UNKNOWN_FILE:
	  begin
	    assert(172 { attempt to close an unknown file });
	    val := FALSE;
	  end;
      end;

    end;
  current_file := UNKNOWN_FILE;
  close_parse_file := val;
end { close_parse_file } ;


procedure close_file(*var f: textfile; logical: alpha; file_name: xtring*);
  { close the given file }
begin
  if debug_22 then 
    begin
      write(outfile, 'close file ');
      if file_name = nullstring then writealpha(outfile, logical)
      else writestring(outfile, file_name);
#ifdef UNIX
      write(outfile, ' file descriptor: ', text_file_descr(f):1);
#endif 
      writeln(outfile);
    end;

  if not vclose(f) then
    begin
      error(168 { cannot close specified file });
      if file_name = nullstring then
        error_dump_alpha_file_name(logical)
      else error_dump_file_name(file_name);
    end;
end { close_file } ;

procedure close_ds_file(var f: textfile; file_name: xtring);
  { close the given ds file }
begin
  if debug_22 then 
    begin
      write(outfile, 'close ds file ');
      writestring(outfile, file_name);
#ifdef UNIX
{      write(outfile, ' file descriptor: ', text_file_descr(f):1);}
#endif 
      writeln(outfile);
    end;

  if not ds_vclose(f) then
    begin
      error(168 { cannot close specified file });
      error_dump_file_name(file_name);
    end;
end { close_ds_file } ;


function rewrite_locked_file(which: parse_file_type;
                             filename: xtring;  logical: alpha): boolean;
  { Rewrite the parse file (which is open, though it may not be current_file)
    without losing the lock on it.  filename (or logical) describe the
    open file accurately. }
#if VAX
  var 
    code: longint;             { return code }
    fab: fab_ptr;              { File access block }
#endif
begin
  if debug_22 then 
    begin
      writeln(outfile, 'rewrite_locked_file ');
      writealpha(outfile, logical);
      write(outfile, ' as ');
      writestring(outfile, filename);
      writeln(outfile);
    end;

  if which <> CMPSCHEM_FILE then
    begin
      assert(0 { should never happen });
      write(CmpLog, ' Rewrite_locked_file ');
      writealpha(CmpLog, logical);
      write(CmpLog, ' as ');
      writestring(CmpLog, filename);
      writeln(CmpLog, ' (not CMPSCHEM file!!)');
    end;

  { The UNIX lock is good til undone by efs_unlock -- we can open and close
    the file with impunity }

#if UNIX
  if close_parse_file(CMPSCHEM_FILE) then ;
  if rewrite_file(CmpSchem, filename, logical) then
    rewrite_locked_file := TRUE
  else rewrite_locked_file := FALSE; 
#endif
#if VAX
  { The VAX lock is maintained by NOT closing the file before rewriting it }

  rewrite_locked_file := TRUE;
  rewrite(CmpSchem, ERROR := CONTINUE);
  code := status(CmpSchem);
  if code > 0 then
    begin
      rewrite_locked_file := FALSE;
      error(169 { cannot open this file });
      if filename = nullstring then
        error_dump_alpha_file_name(logical)
      else error_dump_file_name(filename);
      if debugging then
        begin
          error_dump_indent(indent);
          error_dump_alpha('status code=    ');
          error_dump_integer(code);
          error_dump_CRLF;
          fab := PAS$FAB(CmpSchem);
          if fab <> NIL then
            begin
              error_dump_indent(indent);
              error_dump_alpha('rms code=       ');
              error_dump_integer((fab^.FAB$L_STS)::longint);
              error_dump_CRLF;
            end;
        end;
      close(CmpSchem);
    end;
#endif
end { rewrite_locked_file } ;


procedure remove_logical_file(*fname: alpha*);
  { delete the specified file, if it exists }
begin
  if not delete_logical_file(fname) { external C function } then ;
end { remove_locical_file } ;





