function primitive_directive_found(drawing: xtring): boolean;
  var
    obj: avl_object_ptr;
begin
{ obj.tag := AVL_STRING;                                         }(*AVL*)
  obj.str := drawing;
  primitive_directive_found := 
    avl_find(obj, force_primitives, AVL_STRING) <> NIL;
end { primitive_directive_found } ;


procedure reverse_file_list(var head: file_list_ptr);
  var
    previous: file_list_ptr;  { former parent of head }
    next: file_list_ptr;      { former child of head }
begin
  previous := NIL;
  while head <> NIL do 
    begin
      next := head^.next;
      head^.next := previous;
      previous := head;
      head := next;
    end;
  head := previous;
end { reverse_file_list } ;


(**)     { ------- Expansion control setup ------- }


procedure read_expansion_rules;
  var
    f: file_list_ptr;
    ok: boolean;
begin
  reverse_file_list(expansion_rules_file);
  ok := TRUE;

  f := expansion_rules_file;
  while f <> NIL do
    begin
      if not er_read(f^.file_name) then ok := FALSE;
      f := f^.next;
    end;
  if not ok then error(255 { a fatal error });
end { read_expansion_rules } ;


(**)     


procedure read_master_libraries;
  var
    f: file_list_ptr;
begin
  f := master_library_file;
  while f <> NIL do
    begin
      f := f^.next;
    end;
end { read_master_libraries } ;


procedure PREDS_read_and_check_configuration_file;
  { if configuration directives found, generate configuration file
    otherwise, read configuration from it.  }
  var
    index: signal_syntax_range;      { index into signal syntax table }
    found: boolean;                  { TRUE when name specifier }
    error_occurred: boolean;         { TRUE if error occurred during parse }


  function determine_symbol(var sym: symbols; var ch: char;
                             var name: name_ptr): boolean;
    { determine the symbol, character, or ID from the input file and return
      them.  If an error occurs, return FALSE. }
    var
      specified_character: xtring;  { character specified for symbol }
      ok: boolean;                  { TRUE if all is ok }
  begin
    ok := FALSE;

    sym := NULLSY;  ch := chr(255);  name := null_name;

    create_a_string(specified_character, 1);

    if sy <> STRINGS then
      begin  error(33 { expected a string });  skip([SEMI]);  end

    else if ord(lex_string^[0]) > 1 then
      begin  error(102 { only 1 char });  skip([SEMI]);  end

    else
      begin
        if ord(lex_string^[0]) = 0 then
          begin  sym := NULLSY;  ch := chr(255);  ok := TRUE;  end
        else
          begin
            specified_character^[1] := lex_string^[1];
            ch := specified_character^[1];

            parse_string(specified_character, PARSE_SEPARATELY);
            sym := sy;
            if sym = IDENT then name := id.name;
            pop_parsed_string(specified_character);

            if sym in forbidden_symbols then error(99 { illegal symbol })
                                        else ok := TRUE;
          end;

        insymbol;    { eat the specification }
      end;

    if not ok then error_occurred := TRUE;

    release_string(specified_character);

    determine_symbol := ok;
  end { determine_symbol } ;


  procedure process_syntax;
    { process the signal syntax specification.  The form is:

            <specifier> <specifier> ...      

      where <specifier> is one of the following:

            <negation>
            <assertion>
            <name>
            <subscript>

      <name> and <subscript> MUST always appear. }
    var
      syntax_index: 0..SYNTAX_TABLE_SIZE;   { index into syntax descriptor }
      new_table: signal_syntax_table_type;  { new syntax table }
      encountered: syntax_specifier_set;    { specifiers found }
      specifier: syntax_specifier_type;     { current specifier }
      no_errors: boolean;                   { TRUE if no errors found }


    function get_specifier: syntax_specifier_type;
      { get the syntax specifier and return }
      var
        i: syntax_specifier_type;    { index into specifier table }
        found: boolean;              { TRUE when specifier found }
    begin
      if sy = LESSTHAN then insymbol else error(10 { expected < });

      i := succ(FIRST_SYNTAX_SPECIFIER);  found := FALSE;
      while (i <= pred(LAST_SYNTAX_SPECIFIER)) and not found do
        if syntax_specifier_names[i] = id.name then found := TRUE
        else i := succ(i);

      if not found then
        begin
          error(67 );
	  error_dump_line('Unknown signal syntax specification         ');
          no_errors := FALSE;
          i := NULL_SPECIFIER;
          error_occurred := TRUE;
        end;

      get_specifier := i;
      insymbol;

      if sy = GREATERTHAN then insymbol else error(11 { expected > });
    end { get_specifier } ;


  begin { process_syntax }
    for syntax_index := 1 to SYNTAX_TABLE_SIZE do
      new_table[syntax_index] := null_specifier;

    { read in all the syntax elements }

    syntax_index := 0;  encountered := [];  no_errors := TRUE;
    repeat
      if syntax_index < SYNTAX_TABLE_SIZE then
        begin
          specifier := get_specifier;

          if specifier IN encountered then
            begin  
	       error(67 );
	       error_dump_line('Signal syntax element found twice           ');
	       no_errors := FALSE;  
	    end
          else
            begin
              encountered := encountered + [specifier];
              syntax_index := succ(syntax_index);
              new_table[syntax_index] := specifier;
            end;
        end;
    until (sy <> LESSTHAN) or (syntax_index >= SYNTAX_TABLE_SIZE);

    { general properties must be at the end of the signal }

    new_table[syntax_index] := PROPERTY_SPECIFIER;
    
    if not (NAME_SPECIFIER IN encountered) then
      begin  
        error(67 );
	error_dump_line('Every syntax MUST have a name portion       ');
	no_errors := FALSE;
      end;

    if not (SUBSCRIPT_SPECIFIER IN encountered) then
      begin
        error(67 );
	error_dump_line('Every syntax MUST have a subscript          ');
	no_errors := FALSE;
      end;

    { check the form for temporary restrictions }

    syntax_index := 1;
    if not (new_table[syntax_index] IN [ASSERTION_SPECIFIER,
                                        NEGATION_SPECIFIER,
                                        NAME_SPECIFIER]) then
      begin  
	error(67 );
	error_dump_line('Illegal form for signal syntax              ');
	no_errors := FALSE;  
      end
    else
      begin
        if new_table[syntax_index] <> NAME_SPECIFIER then
          syntax_index := syntax_index + 1;

        if new_table[syntax_index] <> NAME_SPECIFIER then
          begin  
	    error(67 );
	    error_dump_line('Illegal form for signal syntax              ');
	    no_errors := FALSE;  
	  end;
      end;

    if no_errors then signal_syntax_table := new_table
                 else error_occurred := TRUE;
  end { process_SYNTAX_directive } ;


  procedure process_low_assertion;
    { process the low assertion specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym = IDENT then
          begin
            name^.kind := name^.kind + [KEY_WORD];
            name^.sy := LOW_ASSERTED_SY;
            sym := LOW_ASSERTED_SY;
          end;

        signal_is_asserted_low_symbol := sym;
        signal_is_asserted_low_char := ch;
      end;
  end { process_low_assertion } ;


  procedure process_high_assertion;
    { process the high assertion specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym = IDENT then
          begin
            name^.kind := name^.kind + [KEY_WORD];
            name^.sy := HIGH_ASSERTED_SY;
            sym := HIGH_ASSERTED_SY;
          end;

        signal_is_asserted_high_symbol := sym;
        signal_is_asserted_high_char := ch;
      end;
  end { process_high_assertion } ;


  procedure process_negation;
    { process the negation specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym = IDENT then
          begin
            name^.kind := name^.kind + [KEY_WORD];
            name^.sy := NEGATION_SY;
            sym := NEGATION_SY;
          end;

        signal_negation_symbol := sym;
        signal_negation_char := ch;
      end;
  end { process_negation } ;


  procedure process_name_prefix;
    { process the name prefix specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym IN expression_symbols + [IDENT] then
          begin  error(99 { illegal symbol });  error_occurred := TRUE;  end
        else
          begin
            name_property_prefix_symbol := sym;
            name_property_prefix_char := ch;
          end;
      end;
  end { process_name_prefix } ;


  procedure process_general_prefix;
    { process the general prefix specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym IN expression_symbols + [IDENT] then
          begin  error(99 { illegal symbol });  error_occurred := TRUE;  end
        else
          begin
            general_property_prefix_symbol := sym;
            general_property_prefix_char := ch;
          end;
      end;
  end { process_general_prefix } ;


  procedure process_bit_ordering;
    { process the bit ordering configuration specification }
  begin
    if sy <> IDENT then
      begin
        error(1 { expected IDENT });
        skip([SEMI]);
        error_occurred := TRUE;
        end
    else
      begin    
        if id.name = L_to_R_specifier then left_to_right := TRUE
        else if id.name = R_to_L_specifier then left_to_right := FALSE
        else
          begin  error(52 { invalid });  error_occurred := TRUE;  end;

        insymbol;    { eat the specification }
      end;
  end { process_bit_ordering } ;


  procedure process_subrange;
    { process the subrange configuration specification }
    var
      specifier: xtring;          { character(s) specified }
      sym: symbols;               { symbol specified }
  begin
    specifier := nullstring;
    if sy <> STRINGS then
      begin
        error(33 { expected a string });
        skip([SEMI]);
        error_occurred := TRUE;
      end
    else
      begin
        specifier := lex_string;

        parse_string(specifier, PARSE_SEPARATELY);
        sym := sy;
        pop_parsed_string(specifier);

        if sym = DOTDOTSY then
          begin
            subrangesy := DOTDOTSY;
            fieldsy := COLON;
            insymbol;    { eat the .. }
          end
        else if sym = COLON then
          begin
            subrangesy := COLON;
            fieldsy := COLONCOLONSY;
            insymbol;    { eat the : }
          end
        else
          begin
            error(67 { not permitted });
 	    error_dump_line('Subrange symbol must be .. or :             ');
            skip([SEMI]);
            error_occurred := TRUE;
          end;
      end;
  end { process_subrange } ;

    
  procedure parse_configuration_specification;
    { parse the signal syntax configuration file }
    var
      config_spec: configure_types;    { index into table of configure types }
      found: boolean;                  { TRUE if configuration found }
      specifier: name_ptr;             { configuration specifier from file }
  begin
    while sy = IDENT do
      begin
        specifier := id.name;

        config_spec := succ(FIRST_CONFIGURE_SPECIFIER);  found := FALSE;
        while (config_spec < LAST_CONFIGURE_SPECIFIER) and not found do
          if configure_specifiers[config_spec] = specifier then found := TRUE
          else config_spec := succ(config_spec);

        if not found then
          if (specifier^.name = 'SIGNAL_SYNTAX   ') or
             (specifier^.name = 'SYNTAX          ') then
            begin
              insymbol;     { eat the configuration name }
              if sy = EQUAL then insymbol;
              process_syntax;
            end
          else
            begin
              error(69 { unknown signal syntax specification });
              skip([SEMI]);
              error_occurred := TRUE;
            end
        else
          begin
            insymbol;    { eat the configuration name }
            if sy = EQUAL then insymbol;

            case config_spec of
              CONFIGURE_SUBRANGE:
                  process_subrange;

              CONFIGURE_BIT_ORDERING:
                  process_bit_ordering;

              CONFIGURE_LOW_ASSERTED:
                  process_low_assertion;

              CONFIGURE_HIGH_ASSERTED:
                  process_high_assertion;

              CONFIGURE_NEGATION:
                  process_negation;

              CONFIGURE_NAME_PREFIX:
                  process_name_prefix;

              CONFIGURE_GENERAL_PREFIX:
                  process_general_prefix;

              CONFIGURE_CONCATENATION:
                  skip([SEMI]);
            end { case } ;
          end;

        if sy = SEMI then insymbol else error(12 { expected ; });
      end;
  end { parse_configuration_specification } ;


begin { PREDS_read_and_check_configuration_file }
  allowed_key_words := directory_keysys;

  if not open_a_file(configuration_file, STANDARD_FILE) then
    begin
      error(207 { cannot open the configuration file });
      error_dump_file_name(configuration_file);
    end

  else
    begin
      if get_file_type <> CONFIGURATION_SPEC then
        begin
          error(67 { problem with syntax spec });
	  error_dump_line('Illegal file type for configuration file    ');
          error_dump_file_name(configuration_file);
          error_occurred := TRUE;
        end
      else
        begin
          error_occurred := FALSE;

          parse_configuration_specification;

          if sy = ENDSY then insymbol else error(40 { expected END });
          if sy <> PERIOD then error(37 { expected . });
        end;


      if error_occurred then
        begin
          error(157 { error occurred in this file });
          error_dump_file_name(configuration_file);
        end;

      if not close_parse_file(STANDARD_FILE) then
        begin
          error(168 { cannot close the file });
          error_dump_file_name(configuration_file);
        end;
    end;
  allowed_key_words := [];

  { set up sets of special characters }

  found := FALSE;  
  is_signal_name_terminator[general_property_prefix_char] := TRUE;
  is_signal_name_terminator[concatenation_char          ] := TRUE;
  is_signal_name_terminator[chr(EOL)                    ] := TRUE;

  for index := 1 to SYNTAX_TABLE_SIZE do
    if not found then
      if signal_syntax_table[index] = name_specifier then
        found := TRUE
      else
    else
      case signal_syntax_table[index] of
        NEGATION_SPECIFIER:
            is_signal_name_terminator[signal_negation_char] := TRUE;

        SUBSCRIPT_SPECIFIER:
            is_signal_name_terminator['<'] := TRUE;

        ASSERTION_SPECIFIER:
            begin
              is_signal_name_terminator[signal_is_asserted_low_char] := TRUE;
              is_signal_name_terminator[signal_is_asserted_high_char] := TRUE;
            end;

        PROPERTY_SPECIFIER,
        NULL_SPECIFIER: ;
      end;
end { PREDS_read_and_check_configuration_file } ;


(**)     { ------- read in the compiler directives ------- }


procedure PREDS_read_compiler_directives_file;
  { read in the file containing compiler directives and root macro
    name specification. }
  var
    directive: directive_type;          { directive being parsed }
    temp: longint;                      { temp for expression values }
    nil_ptr: file_list_ptr;             { NIL to implement evacuate }
    found_debug_directive: boolean;     { TRUE if debugging on }
    directives_encountered: 
                        directive_set;  { all directives encountered }
    dummy_boolean: boolean;             { For on/off directives not used
                                          by compiler (linker only) }
    old_umask: longint;                 { For return from umask call }


  function find_directive(name: name_ptr; var directive: directive_type):
                                                                    boolean;
    { search the directive table for the specified name and return its
      directive type if found. }
    var
      search: directive_type;       { directive search index }
      found: boolean;               { TRUE when directive found }
  begin
    search := succ(FIRST_DIRECTIVE);  found := FALSE;
    while (search < LAST_DIRECTIVE) and not found do
      if name = compiler_directive[search] then found := TRUE
      else search := succ(search);

    if (search IN debug_directives) and not found_debug_password then
      begin  found := FALSE;  search := LAST_DIRECTIVE;  end;

    find_directive := found;  directive := search;
  end { find_directive } ;


  function ON_or_OFF(specifier: name_ptr; default: boolean): boolean;
    { if specifier = 'ON' then return TRUE else if specifier = 'OFF' then
      return FALSE, else return the default. }
  begin
    if specifier = ON_specifier then ON_or_OFF := TRUE
    else if specifier = OFF_specifier then ON_or_OFF := FALSE
    else
      begin  error(52 { invalid specifier });  ON_or_OFF := default;  end;
  end { ON_or_OFF } ;


  function local_or_global(specifier: name_ptr; default: boolean): boolean;
    { if specifier = 'LOCAL' then return TRUE else if specifier = 'GLOBAL'
      then return FALSE, else return the default. }
  begin
    if specifier = LOCAL_specifier then local_or_global := TRUE
    else if specifier = GLOBAL_specifier then local_or_global := FALSE
    else
      begin  error(52 { invalid });  local_or_global := default;  end;
  end { local_or_global } ;


  procedure process_PICK_directive;
    label 
      90; { return }
    var
      comptype: name_ptr;
      extension: name_ptr;
      attribute: name_ptr;
      dwgname: xtring;


    procedure add_exception(comptype, ext, att: name_ptr);
      var
        t: selection_exception_ptr;
    begin
      new(t);
      with t^ do
        begin
	  compile_type := comptype;
	  extension := ext;
	  attribute := att;
	  drawings := NIL;
	  next := selection_exceptions;
	  selection_exceptions := t;
	end;
    end;


  procedure add_drawing(dname, context: xtring);
    var
      t: drawing_list_ptr;
  begin
    new(t);
    t^.drawing := dname;
    t^.context := context;
    t^.next := selection_exceptions^.drawings;
    selection_exceptions^.drawings := t;
  end;
  

  begin { process_pick_directive }
    if sy <> LPAREN then
      begin
	error(15 { expected LPAREN });  
	skip([SEMI]);
	goto 90 { return };
      end;
    insymbol;
    if sy <> IDENT then
      begin
	error(1 { expected id });  
	skip([SEMI]);
	goto 90 { return };
      end;
    comptype := id.name; 
    if (comptype = PRIM_extension_name) then 
      begin
        error(66);  
	error_dump_indent(indent);
	error_dump_alpha('LOGIC assumed   ');
	error_dump_CRLF;
	comptype := LOGIC_compile_type;
      end
    else if (comptype = PART_extension_name) then 
      begin
        error(216);  { Already states that LOGIC is assumed }
	comptype := LOGIC_compile_type;
      end;
    insymbol;

    if sy <> RPAREN then
      begin
	error(7 { expected RPAREN });  
	skip([SEMI]);
	goto 90 { return };
      end;
    insymbol;

    if sy <> IDENT then
      begin
	error(1 { expected id });  
	skip([SEMI]);
	goto 90 { return };
      end;
    extension := id.name;  
    if extension = PART_extension_name then extension := PRIM_extension_name;
    insymbol;

    if sy <> LPAREN then 
      begin
	attribute := null_name;
	if extension = PRIM_extension_name then
	  error(26 { need directory type }); { attribute will be fixed later }
      end
    else
      begin
	insymbol;
	if sy <> IDENT then
	  begin
	    error(1 { expected id });  
	    skip([SEMI]);
	    goto 90 { return };
	  end;
	attribute := id.name;
	if attribute = PART_extension_name then
	  attribute := PRIM_extension_name;
	if extension = PRIM_extension_name then
	  begin
	    if attribute = PRIM_extension_name then
	      begin
	        error(28 { not legal SCALD dir type });
		attribute := null_name;  { fix it later }
              end;
	  end
        else
	  begin
	    if (attribute <> SPECIAL_specifier) and 
	       (attribute <> PRIMITIVE_specifier) then 
              begin
		error(27 { expected SPECIAL or PRIMITIVE });
		attribute := null_name;
	      end;
	  end;
	insymbol;

	if sy <> RPAREN then
	  begin
	    error(7 { expected RPAREN });  
	    skip([SEMI]);
	    goto 90 { return };
	  end;
	insymbol;
      end;

    add_exception(comptype, extension, attribute);

    repeat
      if sy <> STRINGS then
	begin
	  error(33 { expected a string });  
	  skip([SEMI]);
	  goto 90 { return };
	end;
      dwgname := lex_string;
      insymbol;
      if sy = MINUS then
	begin
	  insymbol;
	  if sy <> STRINGS then
	    begin
	      error(33 { expected a string });  
	      skip([SEMI]);
	      goto 90 { return };
	    end;
	  add_drawing(dwgname, lex_string);
	  insymbol;
	end
      else add_drawing(dwgname, NIL);
      if sy = COMMA then insymbol
      else if sy <> SEMI then goto 90 { return -- error handled elsewhere };
    until sy = SEMI;
  90:
  end { process_PICK_directive } ;


  procedure process_pick_list;
    var
      t: selection_exception_ptr;
      dwg: drawing_list_ptr;


    procedure reverse(var head: selection_exception_ptr);
      var
	previous: selection_exception_ptr;  { former parent of head }
	next: selection_exception_ptr;      { former child of head }
    begin
      previous := NIL;
      while head <> NIL do 
	begin
	  next := head^.next;
	  head^.next := previous;
	  previous := head;
	  head := next;
	end;
      head := previous;
    end { reverse } ;


  begin { process_pick_list }
    reverse(selection_exceptions);
    t := selection_exceptions;
    while t <> NIL do with t^ do
      begin
        if (extension = PRIM_extension_name) and (attribute = NULL_name) then
	  attribute := specified_compile_type;
        dwg := drawings;
	while dwg <> NIL do
	  begin
	    er_except(compile_type^.name, dwg^.drawing, dwg^.context,
	              extension^.name, attribute^.name);
	    dwg := dwg^.next;
	  end;
        t := next;
      end;
  end { process_pick_list } ;


  procedure process_OUTPUT_file_list;
    { process the output file list and set the control flags }
    var
      file_name: output_file_names;      { name of output file to create }
      done: boolean;                     { if all output controls read }
      found: boolean;                    { TRUE if file name found }
  begin 
    if not (OUTPUT_DIRECTIVE in directives_encountered) then
      files_to_generate := [];

    done := FALSE;
    while not done and (sy <> SEMI) do
      begin
        if sy <> IDENT then
          begin  error(1 { expected ident });  skip([SEMI,COMMA,IDENT]);  end
        else
          begin
            file_name := succ(first_file_name);  found := FALSE;
	    if id.name <> null_name then
	      while (file_name < last_file_name) and not found do
		if output_file[file_name] = id.name then found := TRUE
		else file_name := succ(file_name);

            if found then
	      begin
	        if (file_name in SCALD_only_output_files) then
		  error(3 { SCALD compiler only });
	      end
            else
              begin
                error(97 { unknown output file name });
                file_name := first_file_name;
              end;

            files_to_generate := files_to_generate + [file_name];
            insymbol;
          end;
        if sy = COMMA then insymbol else done := TRUE;
      end;
  end { process_OUTPUT_file_list } ;


  procedure process_shareable_directive;
    { process the directive and set the umask accordingly. Note that
      SHAREABLE DEFAULT; means to do nothing to the umask. }
    var
       oldumask: longint;
       val: shareable_value;
  begin
    if sy <> IDENT then error(1 { Expected identifier })
    else
      begin
        val := succ(FIRST_SHAREABLE_VALUE);
        while (val < LAST_SHAREABLE_VALUE) and 
              (shareable.values[val] <> id.name) do val := succ(val);
        if val = LAST_SHAREABLE_VALUE then
	  error(17 { unknown SHAREABLE spec })
        else 
          if val <> DEFAULT_SHARING then
	    oldumask := umask(shareable.umasks[val]);
	specified_shareable_value := val;
	insymbol;
      end;
  end { process_shareable_directive } ;


  function add_name_to_library_list(lib_name, add_name: xtring): boolean;
    { check the given name against the current list of libraries.
      If it exists, return FALSE.  If it doesn't exist, add it to the
      list and return TRUE. }
    var
      already_exists: boolean;         { TRUE if file already in the list }
      element,                         { current directory entry }
      last: directory_list_ptr;        { last library visited in the list }
  begin { add_name_to_library_list }
    element := library_list_root;  last := NIL;
    already_exists := FALSE;
    while (element <> NIL) and not already_exists do
      if element^.name = lib_name then 
	already_exists := TRUE
      else
        begin  last := element;  element := element^.next;  end;
    if not already_exists then
      begin
        new_directory_list(element);
        element^.name := lib_name;
        element^.add := add_name;
        if last=NIL then library_list_root := element
                    else last^.next:=element;
      end;
    add_name_to_library_list := not already_exists;
  end { add_name_to_library_list } ;


  procedure process_DIRECTORY_directive;
    { process the scald directory and its shadow directory The procedure
      reads the directories specified in the directory directive and 
      supplies the add package with the scald directory and its associated
      shadow directory. Each scald directory can be associated with only
      one shadow directory. Any additional add directory is ignored. The
      error handling in this code is very gentle. It only reminds the user
      of any error in the lexical manipulation and tries to recover 
      gracefully from the error condition without much ado. Any errors in
      the specification is dealt by the add package }
    var
      done: boolean;              { TRUE if all directory names read }
      scaldir: xtring;            { scald directory }
      add : xtring;             { add directory }
      dir: directory_list_ptr;    { current directory in list }
      last: directory_list_ptr;   { last directory in list }
      found: boolean;             { TRUE if found in list }

  begin { process_DIRECTORY_directive }
    done := FALSE;
    repeat
      scaldir := NIL;
      add := NIL;
      if sy = STRINGS then
        begin
          scaldir := lex_string;
          insymbol;
        end
      else 
        begin
	  error(93 { expected file name });
          skip([SEMI,COMMA]);
        end;

      if ((sy = MINUS) and (scaldir <> NIL) and not done) then
        begin
          insymbol;
          if sy = STRINGS then
            begin
              add := lex_string;
              insymbol;
            end
          else
            begin
	      error(91 { expected shadow directory name });
              skip([SEMI,COMMA]);
            end;
        end;
      
      if sy <> COMMA then done := TRUE
      else insymbol;

      if scaldir <> NIL then
        begin
	  found := FALSE;
	  dir := directory_list_root;  last := NIL;
	  while (dir <> NIL)  and not found do
	    begin
	      if (dir^.name = scaldir) then
		begin
		  found := TRUE;
                  error(87 { file already exists });
		end
	      else
	        begin
		  last := dir;
		  dir := dir^.next;
		end;
            end;
	  if not found then 
	    begin
	      new_directory_list(dir);
	      if last = NIL then directory_list_root := dir
	      else last^.next := dir;
	      dir^.name := scaldir;
	      dir^.add := add;
            end;
        end;
    until done;
  end { process_DIRECTORY_directive } ;
          

  procedure process_DIRECTORY_list;
    { process the list of directories -- inform the A.D.D. package about
      each and report errors }
    var
      element: directory_list_ptr;      { current directory name }
      add_name: xtring;                  { name of add directory }
  begin
    element := directory_list_root;
    while element <> NIL do
      begin
        add_name := element^.add;
	if add_name = nullstring then add_name := NIL;
        element := element^.next;
      end;
  end  { process_DIRECTORY_list };
      

  procedure process_LIBRARY_list;
    { process the LIBRARY directive.  For each library identifier, check
      the master library description and, if found, add the corresponding
      SCALD directory file name to the directory list. }
    var
      element: directory_list_ptr;       { current library name }
      lib_name,                          { name of library }
      add_name: xtring;                  { name of add directory }
  begin
    element := library_list_root;
    while element <> NIL do
      begin
        lib_name := element^.name;
	if lib_name = nullstring then lib_name := NIL; { add package kink }
        add_name := element^.add;
	if add_name = nullstring then add_name := NIL; { add package kink }
        element := element^.next;
      end;
  end  { process_LIBRARY_list };
      

  procedure process_LIBRARY_directive;
    { add each library name to the list of libraries }
    var
      done: boolean;                    { TRUE when done with directive }
      library_name: xtring;             { name of library being specified }
      add: xtring;                    { specified ADD }
  begin
    done := FALSE;
    repeat
      library_name := NIL;
      add := NIL;

      if sy = IDENT then
        begin
          library_name := make_and_enter_string(id.name^.name);
          insymbol; { eat the identifier }
        end
      else if sy = STRINGS then
        begin
          library_name := lex_string;
          insymbol; { eat the the string }
        end
      else
        begin
          error(4 { expected a string or identifier });  
          skip([SEMI,COMMA,STRINGS,IDENT]);
        end;


      if ((sy = MINUS) and (library_name <> NIL)) then 
               { check if there is any shadow directory }
        begin
	  upper_case_strings := FALSE;
          insymbol; { eat the '-' and get ADD name -- don't upshift it }
	  upper_case_strings := TRUE;
          if sy = STRINGS then
            begin
              add := lex_string;
              insymbol; { eat the shadow string }
            end
          else
            error(93 { expected add_dir });
        end;

      if library_name <> NIL then
        begin
          if not add_name_to_library_list(library_name, add) then 
	    error(68 { lib already there });
	end;

      if sy = COMMA then insymbol else done := TRUE;
    until done;
  end { process_LIBRARY_directive } ;


  procedure get_file_name_list(var list: file_list_ptr);
    { read a list of files from the input.  Check to see if the file has
      already been specified.  If not, add it to the list.  Otherwise,
      generate an error. }
    var
      done: boolean;                { TRUE when done with directive }
      file_name: xtring;            { name of file being specified }
      current_file: file_list_ptr;  { current file in the list }
      found: boolean;               { TRUE if entry found in library }
  begin    
    done := FALSE;
    repeat
      if sy <> STRINGS then
        begin
          error(33 { expected a string });  skip([SEMI,COMMA,STRINGS]);
        end
      else
        begin
          file_name := lex_string;
          current_file := list;  found := FALSE;
          while (current_file <> NIL) and not found do
            if current_file^.file_name = file_name then found := TRUE
            else current_file := current_file^.next;

          if found then
            error(38 { file was already specified })
          else
            begin
              new_file_list(list);
              list^.file_name := file_name;
            end;

          insymbol;     { eat the file name }
        end;

      if sy = COMMA then insymbol else done := TRUE;
    until done;
  end { get_file_name_list } ;


  procedure process_debug_controls;
    { process the debug control specifiers }
    var
      control: debug_controls;       { control specified }
      done: boolean;                 { TRUE when all debug controls found }
      found: boolean;                { TRUE when control name found in table }


    procedure process_histograms;
      { process the histogram specifiers }
      var
        hist: histogram_types;       { histogram to be printed }
        done: boolean;               { TRUE if all histograms have been read }
        found: boolean;              { TRUE if specifier found }
    begin
      insymbol;   { eat the PRINTHISTOGRAMS }
      if sy = LPAREN then insymbol else error(15 { expected ( });

      done := FALSE;
      repeat
        if sy <> IDENT then
          begin  error(1 { expected ident });  skip([SEMI,COMMA]);  end
        else
          begin
            hist := succ(FIRST_HISTOGRAM);  found := FALSE;
            while (hist < LAST_HISTOGRAM) and not found do
              if histogram_specifiers[hist] = id.name then found := TRUE
              else hist := succ(hist);

            if found then histograms := histograms + [hist]
            else error(42 { unknown histogram });
          end;

        insymbol;
        if sy = COMMA then insymbol else done := TRUE;
      until done;

      if sy <> RPAREN then error(7 { expected ) });
    end { process_histograms } ;


  begin { process_debug_controls }
    done := FALSE;
    while (sy = IDENT) and not done do
      begin
        control := succ(FIRST_CONTROL);  found := FALSE;
        while (control < LAST_CONTROL) and not found do
          if debug_control_specifier[control] = id.name then found := TRUE
          else control := succ(control);

        case control of
          CONTROL_DUMPTREE:          dumptree_ok := TRUE;
          CONTROL_DUMPSIGNALS:       dumpsignals_ok := TRUE;
          CONTROL_PRINTMACROS:       printmacros_ok := TRUE;
          CONTROL_PRINTDIRECTORY:    printdirectory_ok := TRUE;
          CONTROL_PRINTHISTOGRAMS:   process_histograms;
          CONTROL_DUMPSIGDEFLIST:    dumpsigdeflist_ok := TRUE;
          CONTROL_DUMP_ALL_NAMES:    dump_all_names_ok := TRUE;
          CONTROL_ERULE_XFACE:       
	    begin  
	      er_debug;  trace_erule_xface := TRUE;
	    end;
          LAST_CONTROL,
          FIRST_CONTROL:             error(42 { unknown debug control });
        end;

        insymbol;

        if sy = COMMA then insymbol else done := TRUE;
      end;

    if not found_debug_directive then
      begin
        debugging := TRUE;
        if not rewrite_file(Outfile, nullstring, DEBUG_FILE_NAME) then
          halt_with_status(FATAL_COMPLETION);
      end;
    found_debug_directive := TRUE;
  end { process_debug_controls } ;


  procedure process_debug;
    { process the DEBUG directive }
    var
      debug_number: natural_number;     { debug flag number }
      done: boolean;                    { TRUE when all debug COMMAnds read }
  begin
    if not found_debug_directive then
      begin
        debugging := TRUE;
        if not rewrite_file(Outfile, nullstring, DEBUG_FILE_NAME) then
          halt_with_status(FATAL_COMPLETION);
      end;
    found_debug_directive := TRUE;

    done := FALSE;
    repeat
      if sy = IDENT then
        begin  debug := ON_or_OFF(id.name, debug);  insymbol;  end
      else if sy = CONSTANT then
        begin
          debug_number := const_val;  insymbol;
          case debug_number of
            1:   debug_1 := TRUE;
            2:   debug_2 := TRUE;
            3:   debug_3 := TRUE;
            4:   debug_4 := TRUE;
            5:   debug_5 := TRUE;
            6:   debug_6 := TRUE;
            7:   debug_7 := TRUE;
            8:   debug_8 := TRUE;
            9:   debug_9 := TRUE;
            10:  debug_10 := TRUE;
            11:  debug_11 := TRUE;
            12:  debug_12 := TRUE;
            13:  debug_13 := TRUE;
            14:  debug_14 := TRUE;
            15:  debug_15 := TRUE;
            16:  debug_16 := TRUE;
            17:  debug_17 := TRUE;
            18:  debug_18 := TRUE;
            19:  debug_19 := TRUE;
            20:  debug_20 := TRUE;
            21:  debug_21 := TRUE;
            22:  debug_22 := TRUE;
            23:  debug_23 := TRUE;
            24:  debug_24 := TRUE;
            25:  debug_25 := TRUE;
            26:  debug_26 := TRUE;
            27:  debug_27 := TRUE;
            28:  debug_28 := TRUE;
            29:  debug_29 := TRUE;
            30:  debug_30 := TRUE;
            31:  debug_31 := TRUE;
            32:  debug_32 := TRUE;
            33:  debug_33 := TRUE;
            34:  debug_34 := TRUE;
            35:  debug_35 := TRUE;
            36:  debug_36 := TRUE;
            37:  debug_37 := TRUE;
            38:  debug_38 := TRUE;
            39:  debug_39 := TRUE;
            40:  debug_40 := TRUE;
          end;
        end 
      else
        begin
          error(1 { expected ident });  skip([SEMI, IDENT]);
        end;

      if sy = COMMA then insymbol else done := TRUE;
    until done;
  end { process_debug } ;


  procedure process_path_debug;
    { process the DEBUG_AT_PATH directive }
    var
      done: boolean;               { TRUE when command completed }
  begin
    if not found_debug_directive then
      begin
        debugging := TRUE;
        if not rewrite_file(Outfile, nullstring, DEBUG_FILE_NAME) then
          halt_with_status(FATAL_COMPLETION);
      end;
    found_debug_directive := TRUE;

    if sy <> STRINGS then
      begin  error(33 { expected a string });  skip([SEMI]);  end
    else
      begin
        path_for_debug := lex_string;  insymbol;
        done := FALSE;
        repeat
          if sy <> CONSTANT then
            begin
              error(52 { invalid });  skip([SEMI,IDENT]);
            end
          else
            if (const_val < 0) or (const_val > MAX_DEBUG_FLAG_NUMBER) then
              begin
                error(52 { not correct });  skip([SEMI,IDENT]);
              end
            else
              begin
                debug_flags := debug_flags + [const_val];
                insymbol;
                debug_at_path := TRUE;
              end;

          if sy = COMMA then insymbol else done := TRUE;
        until done;
      end;
  end { process_path_debug } ;


  procedure process_SUPPRESS_directive;
    { process the SUPPRESS directive }
    var
      done: boolean;            { TRUE when all numbers read in }
  begin
    done := FALSE;
    while ((sy = CONSTANT) or (sy = IDENT)) and not done do
      begin
        if sy = IDENT then
          if id.name = ALL_identifier then
            begin
              suppress_errors := warning_errors + oversight_errors;
              insymbol;
            end
          else
            begin
              error(52 { unexpected specification });
              skip([COMMA,SEMI,CONSTANT,IDENT]);
            end
        else
          if not (const_val IN warning_errors+oversight_errors) then
            begin
              error(92 { invalid number });  skip([SEMI,COMMA]);
            end
          else
            begin
              suppress_errors := suppress_errors + [const_val];  insymbol;
            end;

        if sy = COMMA then insymbol else done := TRUE;
      end;
  end { process_SUPPRESS_directive } ;


  procedure process_COMPILE_directive(compile_type: name_ptr);
    { process the COMPILE directive and set up the extension for compile }
  begin
    if not TYPE_specified_in_command_line then
      begin
        if (compile_type = PART_extension_name) then
          begin
            error(216 { PART not allowed; LOGIC assumed });
            compile_type := LOGIC_compile_type;
          end;

        if (compile_type = PRIM_extension_name) then
	  error(66 { not allowed });
	specified_compile_type := compile_type;
      end;
  end { process_COMPILE_directive } ;


  procedure process_PRIMITIVE_directive;
    { process the primitive directive - create list of macro names.  If
      the given macro name has a path name in front of it, just save
      the path and make only that instance of the macro a primitive. }
    var
      done: boolean;          { TRUE when all macro names have been read }
      obj: avl_object_ptr;
  begin
{   obj.tag := AVL_STRING;                                         }(*AVL*)
    done := FALSE;
    repeat
      if sy <> STRINGS then
        begin  error(33 { expected string });  skip([SEMI,COMMA]);  end
      else
        begin
          if lex_string^[1] = '(' then 
            begin
              error(3 { SCALD compiler only });
	      error_dump_indent(indent);
	      error_dump_alpha(' ValidCOMPILER d');
	      error_dump_alpha('oes not support ');
	      error_dump_alpha(' path names on t');
	      error_dump_alpha('the PRIMITIVE di');
	      error_dump_alpha('rective         ');
	      error_dump_CRLF;
            end
          else
	    begin
	      obj.str := lex_string;
	      if (avl_insert(obj, force_primitives, AVL_STRING) <> NIL) then ;
            end;

          insymbol;
        end;

      if sy = COMMA then insymbol else done := TRUE;
    until done;
  end { process_PRIMITIVE_directive } ;


  procedure process_PASSWORD_directive;
    { process the SECRET SCALD PASSWORDS!!!!! }
  begin
    if sy = EQUAL then insymbol;     { allow space or equal }
    if sy <> IDENT then error(1 { expected ident })
    else
      if id.name^.name = CONFIGURE_PASSWORD then
        { do nothing }
      else if id.name^.name = DEBUG_PASSWORD then
        found_debug_password := TRUE;

    insymbol;    { eat the symbol }
  end { process_PASSWORD_directive } ;


  procedure process_filters;
    { parse but ignore this PASS_PROPERTY or FILTER_PROPERTY directive. }
    var
      done: boolean;         { TRUE when done processing input list }
  begin
    done := FALSE;
    while (sy = IDENT) and not done do
      begin
        insymbol;     { eat the identifier }
        if sy = COMMA then insymbol else done := TRUE;
      end;
  end { process_filters } ;


  procedure process_report_directive;
    { process the REPORT directive }
    var
      report: report_types;    { current report from the table }
      found: boolean;          { TRUE if report id found in the table }
      add_to_list: boolean;    { TRUE if report to be added to the list }
  begin
    while (sy = IDENT) or (sy = minus) do
      begin
        if sy = minus then
          begin  add_to_list := FALSE;  insymbol;  end
        else
          add_to_list := TRUE;

        report := succ(FIRST_REPORT_TYPE);  found := FALSE;
        while (report < LAST_REPORT_TYPE) and not found do
          if report_type_table[report] = id.name then found := TRUE
          else report := succ(report);

        if not found then
          if id.name <> ALL_identifier then
            error(31 { unknown report specification })
          else
            if add_to_list then
	      begin
                reports_to_generate := 
                           [succ(FIRST_REPORT_TYPE)..pred(LAST_REPORT_TYPE)];
              end
            else
              reports_to_generate := []
        else
          if add_to_list then
            if (report in SCALD_only_reports) then error(3 { can't do it })
            else reports_to_generate := reports_to_generate + [report]
          else
            reports_to_generate := reports_to_generate - [report];

        insymbol;    { eat the report name }

        if sy = COMMA then insymbol;
      end;
  end { process_report_directive } ;


#include "commandarg.p"


  procedure welcome;
    { display a welcome message to indicate start of compiler directive parse }
  begin
    if PrintCmpLst then
      begin
        writeln(CmpLst);
        writeln(CmpLst, ' *************************************');
        writeln(CmpLst, ' *  Reading the compiler directives  *');
        writeln(CmpLst, ' *************************************');
        writeln(CmpLst);
      end;
    writeln(monitor);  writeln(monitor, ' Reading the compiler directives.');
    writeln(CmpLog);  writeln(CmpLog, ' Reading the compiler directives.');
  end { welcome } ;


begin { PREDS_read_compiler_directives_file }
  welcome;

  process_command_line_arguments;

  { set up the default master library }

  new_file_list(master_library_file);
  master_library_file^.file_name := standard_library_file_name;

  allowed_key_words := directives_keysys;
  if not open_a_file(nullstring, DIRECTIVES_FILE) then
    error(135 { file does not exist! })
  else
    begin
      directives_encountered := [];

      found_debug_directive := FALSE;

      while sy = IDENT do
        begin
          if not find_directive(id.name, directive) then
            begin
              error(51 { unknown directive });
              skip([SEMI] + directives_keysys);
            end

          else if (directive IN directives_encountered) and
                  (directive IN one_time_directives) then
            begin
              error(45 { directive specified more than once });
              skip([SEMI]);
            end

          else
            begin
              if directive IN file_name_directives then
                upper_case_strings := FALSE;

              insymbol;     { eat the directive symbol }

              if sy = EQUAL then insymbol;    { allow an equal }
              case directive of
                ALLOW_PART_NAME_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        allow_PART_NAME_property := ON_or_OFF(id.name,
                                                     allow_PART_NAME_property)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                AMUSING_MESSAGES_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        produce_amusing_messages := ON_or_OFF(id.name,
                                                     produce_amusing_messages)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                BUBBLECHECK_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        bubble_check := ON_or_OFF(id.name, bubble_check)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                COMMAND_DIRECTIVE:
                    begin
                      if sy <> ident then error(1 { expected ident })
                      else process_COMMAND_directive(id.name);
                      insymbol;
                    end;

                COMPILE_DIRECTIVE:
                    begin
                      if sy <> ident then error(1 { expected ident })
                      else process_COMPILE_directive(id.name);
                      insymbol;
                    end;

                CONFIG_FILE_DIRECTIVE:
                    begin
                      if sy <> STRINGS then error(33 { expected name })
                      else configuration_file := lex_string;
                      insymbol;
                    end;

                CONST_BUBBLE_CHK_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        const_bubble_check :=
                                        ON_or_OFF(id.name, const_bubble_check)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

(************************ obsolete for now
                CONTEXT_DIRECTIVE:
                    begin
                      if sy = STRINGS then
                        if CONTEXT_specified_in_command_line then
                          begin   end
                        else
                          context_being_compiled := lex_string
                      else
                        error(33);
                      insymbol;
                    end;
**************************)

                DEBUG_DIRECTIVE:
                    process_debug;

                DEBUG_AT_PATH_DIRECTIVE:
                    process_path_debug;

                DEBUG_CONTROL_DIRECTIVE:
                    process_debug_controls;

                DECLARE_BODIES_DIRECTIVE:
                    begin
                      if sy = IDENT then
		        begin
                          if ON_or_OFF(id.name, FALSE) then
			    error(3 { SCALD compiler only });
			end
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                DEFAULT_FILTER_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        dummy_boolean := ON_or_OFF(id.name, FALSE)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                DEFAULT_L_OR_G_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        scope_is_local := local_or_global(id.name,
                                                          scope_is_local)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                DIRECTORY_DIRECTIVE:
                     process_directory_directive; 

                ENABLE_CARDINAL_TAP_DIRECTIVE:
                    begin
                      if sy = IDENT then
		        begin
                          enable_cardinal_tap := 
			    ON_or_OFF(id.name, enable_cardinal_tap);
                        end
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                ERROR_HELP_DIRECTIVE:
                    begin
                      if sy <> IDENT then error(1 { expected ident })
		      else
		        begin
                          display_error_doc := ON_or_OFF(id.name,
                                                         display_error_doc);
                          insymbol;
			end;
                    end;

                EVACUATE_DIRECTIVE:
                    begin
                      writeln(monitor);  writeln(monitor);
                      writeln(monitor,
                                    ' Evacuate!?  In our moment of triumph?');
                      writeln(monitor,
                                  ' I think you overestimate their chances.');
                      writeln(monitor);  writeln(monitor);
                      nil_ptr := NIL;  nil_ptr := nil_ptr^.next;
                    end;

                EXPANSION_RULES_DIRECTIVE:
                    get_file_name_list(expansion_rules_file);

                FILTER_PROPERTY_DIRECTIVE:
                    process_filters;

                HIERARCHICAL_NWC_DIRECTIVE:
                    begin
                      if sy = IDENT then
		        begin
                          if ON_or_OFF(id.name, FALSE) then
			    error(3 { SCALD compiler only });
                        end
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                LIBRARY_DIRECTIVE:
                    process_LIBRARY_directive;

                LOCALLY_GLOBAL_DIRECTIVE:
                    begin
                      if sy = IDENT then
		        begin
                          if not ON_or_OFF(id.name, TRUE) then
			    error(3 { SCALD compiler only });
			end
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;


                MASTER_LIBRARY_DIRECTIVE:
		    get_file_name_list(master_library_file);

                MAX_ERROR_DIRECTIVE:
                    begin
                      temp := expression(no_relops);
                      if temp <= 0 then error(178 { ridiculous! })
                      else max_errors := temp;
                    end;

                NET_PROCESSING_DIRECTIVE:
                    begin
                      if sy = IDENT then
		        begin
                          if not ON_or_OFF(id.name, TRUE) then
			    error(3 { SCALD compiler only });
			end
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                OUTPUT_DIRECTIVE:
                    process_OUTPUT_file_list;

                OVERSIGHT_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        display_oversights := ON_or_OFF(id.name,
                                                        display_oversights)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                PAGE_SYNONYM_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        dummy_boolean := ON_or_OFF(id.name, FALSE)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                PASS_PROPERTY_DIRECTIVE:
                    process_filters;

                PASSWORD_DIRECTIVE:
                    process_PASSWORD_directive;

                PERMIT_NO_ASSERTION_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        allow_missing_high_assertion := ON_or_OFF(id.name,
                                                 allow_missing_high_assertion)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                PICK_DIRECTIVE:
                    process_PICK_directive;

                PRIMITIVE_DIRECTIVE:
		    process_PRIMITIVE_directive;

                PROPERTY_DIRECTIVE:
                    get_file_name_list(property_file);

                PRINT_WIDTH_DIRECTIVE:
                    begin
                      temp := expression(no_relops);
                      if (temp<MIN_PRINT_WIDTH) or (temp>MAX_PRINT_WIDTH) then
                        error(94 { invalid print width value })
                      else print_width := temp;
                    end;

                REPORT_DIRECTIVE:
                    process_report_directive;

		REPORT_UNKASSERT_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        report_unknown_assertions :=
			  ON_or_OFF(id.name, report_unknown_assertions)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                ROOT_DIRECTIVE:
                    begin
                      if sy = STRINGS then
                        if ROOT_specified_in_command_line then
                          { ignore this one }
                        else
                          root_macro_name := lex_string
                      else
                        error(33 { expected name string });
                      insymbol;
                    end;

                SHADOW_ROOT_DIRECTIVE:
                    begin
                      if sy = STRINGS then shadow_root := lex_string
                      else
                        error(33 { expected a string });
                      insymbol;
                    end;

                SHAREABLE_DIRECTIVE:
		    process_shareable_directive;

                SINGLE_LEVEL_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        single_level_compile := 
			  ON_or_OFF(id.name, single_level_compile)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                SUPPRESS_DIRECTIVE:
                    process_suppress_directive;

                TEXT_MACRO_DIRECTIVE:
                    get_file_name_list(text_macro_file);

                TOKENIZE_PARAMS_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        tokenize_params :=
                          ON_or_OFF(id.name, tokenize_params)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                WARN_DIRECTIVE:
                    begin
                      if sy = IDENT then
                        display_warnings := ON_or_OFF(id.name,
                                                      display_warnings)
                      else
                        error(1 { expected ident });
                      insymbol;
                    end;

                OTHERWISE
                    begin
                      error(51 { unknown directive });
                      skip([SEMI]+directives_keysys);
                    end;
              end { case } ;

              directives_encountered := directives_encountered + [directive];

              upper_case_strings := TRUE;
            end { else } ;

          if sy = SEMI then insymbol else error(12 { expected ; });
        end { while } ;

      if sy = ENDSY then insymbol else error(40 { expected END });
      if sy <> PERIOD then error(37 { expected . });

      if not close_parse_file(DIRECTIVES_FILE) then
        begin
          error(168 { cannot close file });
          error_dump_alpha_file_name('DIRECTIVES FILE ');
        end;
      
      setup_signal_configuration_from_ds_module;

      parse_errors := parse_errors - [68,87,103]; { directory, lib errors }

      { Be sure to set the umask BEFORE specifying the shadow_root }

      if not (SHAREABLE_DIRECTIVE in directives_encountered) then
        old_umask := umask(shareable.umasks[GROUP_SHARING]);

      if shadow_root <> nullstring then
	begin
	end;

      reverse_file_list(expansion_rules_file);
      reverse_file_list(master_library_file);
      {read_master_libraries;}

      {process_library_list;}
      {process_directory_list;}
    end { else not file open } ;

  er_init(evaluate_boolean_expr,
	  error_without_parse_line, 
	  assert_without_parse_line, 
	  error_dump_integer, 
	  error_dump_standard_indent, 
	  error_dump_string, 
	  error_dump_current_context,
	  error_dump_crlf,
	  parse_error_notifier);

  er_type(specified_compile_type^.name);
  if init_expansion_rules <> 0 then
    begin
    end;
  read_expansion_rules;
  process_pick_list;

  allowed_key_words := [];

  files_to_generate := 
    files_to_generate - [CHIPS_FILE,CMPHIER_FILE,CMPEXP_FILE,CMPSYN_FILE];

  write(monitor, '   Compiler directives read ');
  write(CmpLog, '   Compiler directives read ');
  exec_time(last_elapsed_time, last_CPU_time, TRUE);
end { PREDS_read_compiler_directives_file } ;    
    
    
(**)     { ------- output list of all directive values ------- }


procedure report_compiler_directives(var f: textfile);
  { report the values of all of the compiler directives to the given file }
  var
    directive: directive_type;    { current directive to report }


  procedure print_name(var f: textfile; directive: directive_type);
    { print the name of the compiler directive }
  begin
    write(f, ' ');
    print_alpha(f, compiler_directive[directive]^.name);
    write(f, ' ');
  end { print_name } ;


  procedure report_directories(var f: textfile);
    { report the directories being used }
    var
      dir: directory_list_ptr;          { current directory in list }
      first_yet_to_be_output: boolean;  { TRUE if first not yet output }
  begin
    dir := directory_list_root;  first_yet_to_be_output := TRUE;
    while dir <> NIL do
      begin
        if first_yet_to_be_output then print_name(f, DIRECTORY_DIRECTIVE)
	else 
	  begin
	    writeln(f, ',');
	    write(f, ' ':11);
	  end;
        first_yet_to_be_output := FALSE;
	writestring(f, dir^.name);
	if dir^.add <> NIL then 
	  begin
            write(f, '-');
            writestring(f, dir^.add);
	  end;
        dir := dir^.next;
      end;

    if not first_yet_to_be_output then writeln(f, ';');
  end { report_directories } ;


  procedure report_libraries(var f: textfile);
    { report the libraries used }
    var
      first_yet_to_be_output: boolean;  { TRUE if first not yet output }
      library: directory_list_ptr;      { current element in the library }
  begin
    library := library_list_root;  first_yet_to_be_output := TRUE;
    while library <> NIL do
      begin
        if first_yet_to_be_output then  print_name(f, LIBRARY_DIRECTIVE)
        else
          begin  writeln(f, ',');  write(f, ' ':9);  end;
        first_yet_to_be_output := FALSE;
        writestring(f, library^.name);
	if library^.add <> NIL then
	  begin
            write(f, '-');
            writestring(f, library^.add);
	  end;
        library := library^.next;
      end;

    if not first_yet_to_be_output then writeln(f, ';');
  end { report_libraries } ;


  procedure report_output_directive(var f: textfile);
    { report the output files to be created }
    var
      file_name: output_file_names;   { name of the output file }
      first: boolean;                 { TRUE if first element in the list }
  begin
    print_name(f, OUTPUT_DIRECTIVE);  first := TRUE;
    for file_name := succ(FIRST_FILE_NAME) to pred(LAST_FILE_NAME) do
      if file_name IN files_to_generate then
        begin
          if first then first := FALSE else write(f, ', ');
          writealpha(f, output_file[file_name]^.name);
        end;

    writeln(f, ';');
  end { report_output_directive } ;


  procedure report_REPORT_directive(var f: textfile);
    { report the report directive }
    var
      first: boolean;            { TRUE if first report yet to be generated }
      report: report_types;      { current report type }
  begin
    if reports_to_generate <> [] then
      begin
        first := TRUE;
        print_name(f, REPORT_DIRECTIVE);
        for report := succ(FIRST_REPORT_TYPE) to pred(LAST_REPORT_TYPE) do
          if report IN reports_to_generate then
            begin
              if first then first := FALSE else write(f, ', ');
              writealpha(f, report_type_table[report]^.name);
            end;

        writeln(f, ';');
      end;
  end { report_REPORT_directive } ;


  procedure report_shareable_directive(var f: textfile);
    { report whether to be shared  by owner, group or by all }
  begin
    print_name(f, SHAREABLE_DIRECTIVE);
    writealpha(f,shareable.values[specified_shareable_value]^.name);
    writeln(f, ';');
  end;

  procedure report_SUPPRESS_directive(var f: textfile);
    { report the suppressed error messages }
    var
      first: boolean;        { TRUE if first time through loop }
      error: error_range;    { current error candidate }
  begin
    if suppress_errors <> [] then
      begin
        print_name(f, SUPPRESS_DIRECTIVE);

        first := TRUE;
        for error := 0 to MAX_ERROR_NUM do
          if error IN suppress_errors then
            if first then
              begin  write(f, error:1);  first := FALSE;  end
            else
              write(f, ',', error:1);

        writeln(f, ';');
      end;
  end { report_SUPPRESS_directive } ;


  procedure report_bool_directive(directive: directive_type; val: boolean);
    { report the value of an ON/OFF directive }
  begin
    print_name(f, directive);
    if val then writeln(f, 'ON;') else writeln(f, 'OFF;');
  end { report_bool_directive } ;
   

  procedure report_int_directive(directive: directive_type; val: longint);
    { report value of integer directive }
  begin
    print_name(f, directive);
    writeln(f, val:1, ';');
  end { report_int_directive } ;


  procedure report_string_directive(directive: directive_type; val: xtring);
    { report value of string directive }
  begin
    print_name(f, directive);
    writestring(f, val);
    writeln(f, ';');
  end { report_string_directive } ;


  procedure report_file_list_directive(directive: directive_type;
                                       file_name_list: file_list_ptr);
    { report a list of files for the given directive }
    var
      current_file: file_list_ptr;     { current file in the list }
  begin
    current_file := file_name_list;
    while current_file <> NIL do
      begin
        report_string_directive(directive, current_file^.file_name);

        current_file := current_file^.next;
      end;
  end { report_file_list_directive } ;


  procedure report_COMPILE_directive(var f: textfile);
    { report the value of the COMPILE directive }
    var
      id_size: id_range;    { size of compile directive }
  begin
    print_name(f, COMPILE_DIRECTIVE);
    id_size := alpha_length(specified_compile_type^.name);
    writeln(f, specified_compile_type^.name:id_size, ';');
  end { report_COMPILE_directive } ;


  procedure report_DEBUG_directive(var f: textfile);
    { report the debug directive }
  begin
    print_name(f, DEBUG_DIRECTIVE);
    if debug then write(f, 'ON') else write(f, 'OFF');

    if debug_1  then write(f, ',1');
    if debug_2  then write(f, ',2');
    if debug_3  then write(f, ',3');
    if debug_4  then write(f, ',4');
    if debug_5  then write(f, ',5');
    if debug_6  then write(f, ',6');
    if debug_7  then write(f, ',7');
    if debug_8  then write(f, ',8');
    if debug_9  then write(f, ',9');
    if debug_10 then write(f, ',10');
    if debug_11 then write(f, ',11');
    if debug_12 then write(f, ',12');
    if debug_13 then write(f, ',13');
    if debug_14 then write(f, ',14');
    if debug_15 then write(f, ',15');
    if debug_16 then write(f, ',16');
    if debug_17 then write(f, ',17');
    if debug_18 then write(f, ',18');
    if debug_19 then write(f, ',19');
    if debug_20 then write(f, ',20');
    if debug_21 then write(f, ',21');
    if debug_22 then write(f, ',22');
    if debug_23 then write(f, ',23');
    if debug_24 then write(f, ',24');
    if debug_25 then write(f, ',25');
    if debug_26 then write(f, ',26');
    if debug_27 then write(f, ',27');
    if debug_28 then write(f, ',28');
    if debug_29 then write(f, ',29');
    if debug_30 then write(f, ',30');
    if debug_31 then write(f, ',31');
    if debug_32 then write(f, ',32');
    if debug_33 then write(f, ',33');
    if debug_34 then write(f, ',34');
    if debug_35 then write(f, ',35');
    if debug_36 then write(f, ',36');
    if debug_37 then write(f, ',37');
    if debug_38 then write(f, ',38');
    if debug_40 then write(f, ',40');

    writeln(f, ';');
  end { report_DEBUG_directive } ;


  procedure report_DEBUG_AT_PATH_directive(var f: textfile);
    { report the directive assignments }
    var
      first: boolean;          { TRUE if first yet to be output }
      i: debug_flag_range;     { debug flag number (0=DEBUG ON) }
  begin
    if debug_at_path then
      begin
        print_name(f, DEBUG_AT_PATH_DIRECTIVE);
        writestring(f, path_for_debug);
        write(f, ' ');

        first := TRUE;
        for i := 0 to MAX_DEBUG_FLAG_NUMBER do
          if i IN debug_flags then
            begin
              if first then first := FALSE else write(f, ',');

              if i = 0 then write(f, 'ON') else write(f, i:1);
            end;

        writeln(f, ';');
      end;
  end { report_DEBUG_AT_PATH_directive } ;


  procedure report_DEBUG_CONTROL_directive(var f: textfile);
    { report the debug control directive }
    var
      first: boolean;      { TRUE if first yet to be output }
      i: debug_controls;   { index into control table }


    procedure print_control(control: debug_controls);
      { print the given control to the output file }
    begin
      if first then first := FALSE
      else
        begin
          writeln(f, ',');
          write(f, ' ':15);
        end;

      writealpha(f, debug_control_specifier[control]^.name);
    end { print_control } ;


    procedure report_histograms(control: debug_controls);
      { report the histogram specification }
      var
        hist: histogram_types;      { index into the histogram table }
        first: boolean;             { TRUE if first yet to be output }
    begin
      if histograms <> [] then
        begin
          print_control(control);

          write(f, '(');

          first := TRUE;
          for hist := succ(FIRST_HISTOGRAM) to pred(LAST_HISTOGRAM) do
            if hist IN histograms then
              begin
                if first then first := FALSE else write(f, ',');

                writealpha(f, histogram_specifiers[hist]^.name);
              end;

          write(f, ')');
        end;
    end { report_histograms } ;


  begin { report_DEBUG_CONTROL_directive }
    print_name(f, DEBUG_CONTROL_DIRECTIVE);

    first := TRUE;
    for i := succ(FIRST_CONTROL) to pred(LAST_CONTROL) do
      case i of
        control_dumptree:        if dumptree_ok then print_control(i);
        control_dumpsignals:     if dumpsignals_ok then print_control(i);
        control_printmacros:     if printmacros_ok then print_control(i);
        control_printdirectory:  if printdirectory_ok then print_control(i);
        control_printhistograms: report_histograms(i);
        control_dumpsigdeflist:  if dumpsigdeflist_ok then print_control(i);
        control_dump_all_names:  if dump_all_names_ok then print_control(i);
        control_erule_xface:  if trace_erule_xface then print_control(i);
      end;

    writeln(f, ';');
  end { report_DEBUG_CONTROL_directive } ;


  procedure report_filter_directives(var f: textfile);
    { report the PASS and FILTER directives }


    procedure output_property_list(which: xtring);
      { output the properties from the list as specified by WHICH }
      var
        prop: property_ptr;    { current property in the list }
        first: boolean;        { TRUE if first prop yet to be printed }
    begin
      prop := properties_assigned_filters;  first := TRUE;
      while prop <> NIL do
        begin
          if prop^.text = which then
            begin
              if first then first := FALSE
              else
                begin
                  writeln(f, ',');
                  if which = PASS_string then
                    write(f, ' ':15)
                  else
                    write(f, ' ':17);
                end;

              writealpha(f, prop^.name^.name);
            end;

          prop := prop^.next;
        end;

      writeln(f, ';');
    end { output_property_list } ;


  begin { report_filter_directives }
    print_name(f, PASS_PROPERTY_DIRECTIVE);
    output_property_list(PASS_string);

    print_name(f, FILTER_PROPERTY_DIRECTIVE);
    output_property_list(FILTER_string);
  end { report_filter_directives } ;


begin { report_compiler_directives }
  writeln(f);
  writeln(f, ' ------ Global Compiler Directives ------');
  writeln(f);

  for directive := succ(FIRST_DIRECTIVE) to pred(LAST_DIRECTIVE) do
    case directive of
      ALLOW_PART_NAME_DIRECTIVE:
          if not allow_PART_NAME_property then
            report_bool_directive(ALLOW_PART_NAME_DIRECTIVE,
                                  allow_PART_NAME_property);

      AMUSING_MESSAGES_DIRECTIVE:
          if produce_amusing_messages then
            report_bool_directive(AMUSING_MESSAGES_DIRECTIVE,
                                  produce_amusing_messages);

      BUBBLECHECK_DIRECTIVE:
          report_bool_directive(BUBBLECHECK_DIRECTIVE, bubble_check);

      COMPILE_DIRECTIVE:
          report_COMPILE_directive(f);

      CONFIG_FILE_DIRECTIVE:
          if configuration_file <> default_configuration_file then
            report_string_directive(CONFIG_FILE_DIRECTIVE, configuration_file);

      CONST_BUBBLE_CHK_DIRECTIVE:
          report_bool_directive(CONST_BUBBLE_CHK_DIRECTIVE,
                                const_bubble_check);

      DEBUG_DIRECTIVE:
          if found_debug_password then
            report_debug_directive(f);

      DEBUG_AT_PATH_DIRECTIVE:
          if found_debug_password then
            report_debug_at_path_directive(f);

      DEBUG_CONTROL_DIRECTIVE:
          if found_debug_password then
            report_debug_control_directive(f);

      DECLARE_BODIES_DIRECTIVE: { nothing -- not supported here } ;

      DEFAULT_L_OR_G_DIRECTIVE:
          if not scope_is_local then
            begin
              print_name(f, DEFAULT_L_OR_G_DIRECTIVE);
              if scope_is_local then writeln(f, 'LOCAL;')
                                else writeln(f, 'GLOBAL;');
            end;

      DIRECTORY_DIRECTIVE:
          report_directories(f);

      ENABLE_CARDINAL_TAP_DIRECTIVE:
          if not enable_cardinal_tap then
	    report_bool_directive(ENABLE_CARDINAL_TAP_DIRECTIVE, 
				  enable_cardinal_tap);

      ERROR_HELP_DIRECTIVE:
          report_bool_directive(ERROR_HELP_DIRECTIVE, display_error_doc);

      EVACUATE_DIRECTIVE:  ;

      EXPANSION_RULES_DIRECTIVE:
          if expansion_rules_file <> NIL then
            report_file_list_directive(EXPANSION_RULES_DIRECTIVE, 
	                               expansion_rules_file);

      FILTER_PROPERTY_DIRECTIVE:
          report_filter_directives(f);

      LIBRARY_DIRECTIVE:
          report_libraries(f);

      MASTER_LIBRARY_DIRECTIVE:
          if master_library_file <> NIL then
            report_file_list_directive(MASTER_LIBRARY_DIRECTIVE, 
	                               master_library_file);

      MAX_ERROR_DIRECTIVE:
          report_int_directive(MAX_ERROR_DIRECTIVE, max_errors);

      NET_PROCESSING_DIRECTIVE:
          if not net_processing then
            report_bool_directive(NET_PROCESSING_DIRECTIVE, net_processing);

      OUTPUT_DIRECTIVE:
          report_output_directive(f);

      OVERSIGHT_DIRECTIVE:
          report_bool_directive(OVERSIGHT_DIRECTIVE, display_oversights);

      PASS_PROPERTY_DIRECTIVE:  ;

      PASSWORD_DIRECTIVE:  ;

      PERMIT_NO_ASSERTION_DIRECTIVE:
          report_bool_directive(PERMIT_NO_ASSERTION_DIRECTIVE,
                                allow_missing_high_assertion);

      PROPERTY_DIRECTIVE:
          if property_file <> NIL then
            report_file_list_directive(PROPERTY_DIRECTIVE, property_file);

      PRINT_WIDTH_DIRECTIVE:
          report_int_directive(PRINT_WIDTH_DIRECTIVE, print_width);

      REPORT_DIRECTIVE:
          report_REPORT_directive(f);

      SHADOW_ROOT_DIRECTIVE:
          if shadow_root <> nullstring then
            report_string_directive(SHADOW_ROOT_DIRECTIVE, shadow_root);

      SHAREABLE_DIRECTIVE:
	  report_shareable_directive(f);

      SUPPRESS_DIRECTIVE:
          report_suppress_directive(f);

      TEXT_MACRO_DIRECTIVE:
          if text_macro_file <> NIL then
            report_file_list_directive(TEXT_MACRO_DIRECTIVE, text_macro_file);

      TOKENIZE_PARAMS_DIRECTIVE:
          if tokenize_params then
            report_bool_directive(TOKENIZE_PARAMS_DIRECTIVE,
                                  tokenize_params);

      WARN_DIRECTIVE:
          report_bool_directive(WARN_DIRECTIVE, display_warnings);

      OTHERWISE
            ;

    end { case directive of } ;

  writeln(f);
  writeln(f);
  writeln(f, ' Signal syntax:');
  writeln(f);

  output_configuration(f, 3);

  writeln(f);
end { report_compiler_directives } ;


