procedure evaluate_bindings(var formal_actual_list: formal_actual_ptr;
                            clear_text_bindings_list: bindings_list_ptr;
                            var uses_NWC: boolean;
			    var is_cardinal_tap: boolean);
  { parse each of the pins on the body and the actual signals connected
    to each.  The pairs are given by CLEAR_TEXT_BINDINGS_LIST and the
    parsed formal/actual pairs are returned in FORMAL_ACTUAL_LIST.  If any
    pin on the body uses the NWC property, return USES_NWC TRUE. 
    If any pin has the BN property, return is_cardinal_tap TRUE. }

  type
    cardinal_tap_pin = record
      clear_text_binding: bindings_list_ptr;
      binding: formal_actual_ptr;
      tap: subscript_ptr;
    end;
      
  var
    formal: signal_descriptor_ptr;       { current formal of the pair }
    formal_def: signal_definition_ptr;   { signal def for formal }
    formal_instance: signal_instance_ptr;{ instance of def }
    current_element: formal_actual_ptr;  { element in formal/actual list }
    formal_is_low_asserted: boolean;     { TRUE if formal parameter has * }
    actual_signal: clear_text_actual_list_ptr;  { elem of synonymed actuals }
    clear_text_binding: bindings_list_ptr; { the current binding pair }
    dummy: boolean;                      { dummy var parameter }
    directives: record
                  inherit_assertion,     { TRUE iff assertion to be inherited }
                  check_bubble,          { TRUE iff bubbles to be checked }
		  tap_property_found,    { TRUE if CARDINAL_TAP... found }
                  check_width: boolean;  { TRUE iff width to be checked }
		  cardinal_tap: xtring;  { value of tap prop if found }
                end;
    cardinal_tap : cardinal_tap_pin;     { The tap pin (if found) }
    error_on_body: boolean;              { TRUE iff any non-bubble error
                                           has been found }


  procedure dump_clear_text_binding(var f: textfile; bnd: bindings_list_ptr);
    { dump for debugging purposes }
    var
      actual: clear_text_actual_list_ptr;


    procedure dump_props(var f: textfile; props: property_ptr);
    begin
      while props <> NIL do
        begin
	  write(f, DEFAULT_GENERAL_PROPERTY_PREFIX_CHAR);
	  writealpha(f, props^.name^.name);
	  write(f, '=');
	  writestring(f, props^.text);
	  props := props^.next;
	end;
    end { dump_props } ;


  begin { dump_clear_text_binding }
    write(f, 'formal ');
    if bnd = NIL then writeln(f, '<NIL>')
    else
      begin
	writestring(f, bnd^.formal_parameter);
	dump_props(f, bnd^.pin_properties);
	writeln(f);
	actual := bnd^.actual_parameter;
	while actual <> NIL do
	  begin
	    write(f, '    actual ');
	    writestring(f, actual^.actual_parameter);
	    write(f, '(');
	    writestring(f, actual^.net_id);
	    write(f, ')');
	    dump_props(f, actual^.properties);
	    writeln(f);
	    actual := actual^.next;
	  end;
      end;
  end { dump_clear_text_binding } ;


  procedure dump_bindings_list(var f: textfile; list: bindings_list_ptr);
  begin
    while list <> NIL do 
      begin
        dump_clear_text_binding(f, list);
	list := list^.next;
      end;
  end { dump_bindings_list } ;


  function parse_the_subscript(script: xtring):subscript_ptr;
    { parse script, allowing brackets or no brackets }
  begin
    current_property_name := CARDINAL_TAP_prop_name;
    current_property_value := script;

    parse_string(script, PARSE_SEPARATELY);
    if sy = LESSTHAN then
      begin
        insymbol;
	parse_the_subscript := parse_bit_subscript;
	if sy = GREATERTHAN then insymbol
	else 
	  begin  error(11 { expected > });  skip(signal_name_end_sys);  end;
      end
    else parse_the_subscript := parse_bit_subscript;
    if sy <> ENDOFDATASY then error(29 { junk after bit list });
    pop_parsed_string(script);

    current_property_name := null_name;
    current_property_value := nullstring;
  end { parse_the_subscript } ;


  function name_of_pin_is_OK: boolean;
    { check the formal parameter to see that it is of the correct form }

    var
      OK: boolean;     { FALSE if error detected }


    procedure get_formal_directives;
      { read the properties of the formal and return the directives.  Remove
        all "directives" properties from the formal's property list. }
      var
        new_list,               { list with no "directives" properties }
        next,                   { next element in the property list }
        previous,               { previous element of the property list }
        prop: property_ptr;     { current property element }
    begin
      directives.inherit_assertion := FALSE;
      directives.check_bubble := TRUE;
      directives.check_width := TRUE;
      directives.tap_property_found := FALSE;
      directives.cardinal_tap := nullstring;

      formal_is_low_asserted := FALSE;

      prop := formal^.properties;  previous := NIL;  new_list := NIL;
      while prop <> NIL do
        begin
          next := prop^.next;

          if prop^.name = NO_WIDTH_prop_name then
            directives.check_width := FALSE
          else if prop^.name = NO_BUBBLE_prop_name then
            directives.check_bubble := FALSE
          else if prop^.name = NO_ASSERT_prop_name then
            directives.inherit_assertion := TRUE
          else if prop^.name = BUBBLED_prop_name then 
            formal_is_low_asserted := TRUE
          else if (prop^.name = CARDINAL_TAP_prop_name) and
	          enable_cardinal_tap then
	    begin
	      directives.tap_property_found := TRUE;
	      is_cardinal_tap := TRUE;
              directives.cardinal_tap := prop^.text;
              directives.check_width := FALSE;
	    end
          else
            begin
              if previous = NIL then
                begin
                  formal^.properties := next;
                  prop^.next := new_list;
                  new_list := prop;
                  prop := NIL;
                end
              else
                begin
                  previous^.next := next;
                  prop^.next := new_list;
                  new_list := prop;
                  prop := previous;
                end;
            end;

          previous := prop;  prop := next;
        end { while } ;

      release_entire_property_list(formal^.properties);
      formal^.properties := new_list;

      if enable_cardinal_tap then
	if find_property(clear_text_binding^.pin_properties, 
			 CARDINAL_TAP_prop_name, prop) then
	  begin
	    directives.tap_property_found := TRUE;
	    is_cardinal_tap := TRUE;
	    directives.cardinal_tap := prop^.text;
	    directives.check_width := FALSE;
  
	    if find_property(prop^.next, CARDINAL_TAP_prop_name, prop) then
	      begin
		error(69 { ill-formed cardinal tap });
		error_dump_indent(indent);
		error_dump_alpha('More than one   ');
		error_dump_char(' ');
		error_dump_alpha(CARDINAL_TAP_prop_name^.name);
		error_dump_char(' ');
		error_dump_alpha('property on pin ');
		error_dump_CRLF;
		error_dump_pin_name_string(clear_text_binding^.
					   formal_parameter);
		error_on_body := TRUE;
	      end;
	  end;
    end { get_formal_directives } ;


  begin { name_of_pin_is_OK }
    OK := TRUE;

    { pin name cannot have a replication property }

    if formal^.replication_factor <> 1 then
      begin
        error(113 { replication not allowed on formal });
        error_dump_current_parse_environment;
        error_dump_formal(formal);
        formal^.replication_factor := 1;
	error_on_body := TRUE;
      end;

    { make sure that the pin name has the correct scope }

    if formal^.scope <> UNKNOWN_SCOPE then
      begin
        error(165 { pin name has incorrect scope });
        error_dump_current_parse_environment;
        error_dump_formal(formal);
	error_on_body := TRUE;
      end;
    formal^.scope := XINTERFACE;

    { pin name cannot be a concatenated signal }

    if formal^.next <> NIL then
      begin
        error(132 { concatenated signals not permitted as formals });
        error_dump_current_parse_environment;
        error_dump_formal(formal);
        release_entire_signal_descriptor(formal^.next);
	error_on_body := TRUE;
      end;

    { pin name cannot be specified with a bit list }

    if formal^.bit_subscript <> NIL then with formal^.bit_subscript^ do
      if (next <> NIL) or
         ((left_index < right_index) and not left_to_right) or
         ((left_index > right_index) and left_to_right) then
        begin
          error(218 { bit lists are not permitted });
          error_dump_current_parse_environment;
          error_dump_formal(formal);
          release_entire_subscript(formal^.bit_subscript^.next);
	  error_on_body := TRUE;
        end;

    { pin name cannot be a constant }

    if formal^.is_const then
      begin
        error(212 { const not permitted as formal });
        error_dump_current_parse_environment;
        error_dump_formal(formal);
        OK := FALSE;
      end

    { pin name cannot be an NC signal }

    else if is_NC_signal(formal^.signal_name) then
      begin
        error(119 { NC not permitted as a pin name });
        error_dump_current_parse_environment;
        error_dump_formal(formal);
        OK := FALSE;
      end

    { pin name cannot have a timing assertion }

    else if signal_name_has_timing_assertion(formal^.signal_name) then
      begin
        error(189 { timing assertions not permitted on pin names });
        error_dump_current_parse_environment;
        error_dump_formal(formal);
        OK := FALSE;
      end

    { pin name cannot use negation }

    else if ((formal^.polarity = COMPLEMENTED) and not formal^.low_asserted) or
            ((formal^.polarity = NORMAL) and formal^.low_asserted) then
      begin
        error(204 { negation is not permitted on pin names });
        error_dump_current_parse_environment;
        error_dump_formal(formal);
        OK := FALSE;
      end;

    get_formal_directives;

    { pin names with NWC must be scalar }

    if (formal^.kind <> SINGLE) and not directives.check_width then
      begin
        error(202 { \NWC only allowed on non_vector formals });
        error_dump_current_parse_environment;
        error_dump_formal(formal);
        release_entire_subscript(formal^.bit_subscript);
	error_on_body := TRUE;
      end;

    if not directives.check_width then formal^.kind := UNDEFINED;

    name_of_pin_is_OK := OK;
    if not ok then error_on_body := TRUE;
  end { name_of_pin_is_OK } ;


  function found_formal_instance(var instance: signal_instance_ptr): boolean;

    { check the formal subscript against other subscripts of this formal.
      On entry, FORMAL (the descriptor) and FORMAL_DEF are valid and global.
      Return INSTANCE = NIL if this is a new formal instance else
      return the old instance.  Return CURRENT_ELEMENT set to the old
      instance if found in the list. }

    var
      current_formal: formal_actual_ptr;  { list of the formals }
      found: boolean;                     { TRUE when formal instance found }
      ok: boolean;                        { TRUE if all is OK }
      expanded_props: property_ptr;       { expanded version of attached pin
                                            properties }


    function identical_formal_subscripts(sub1, sub2: subscript_ptr): boolean;
      { Return TRUE if the two subscripts are identical.  It is known that
        both subscripts are simpe subranges. }
    begin
      if (sub1 = NIL) or (sub2 = NIL) then
        identical_formal_subscripts := (sub1 = NIL) and (sub2 = NIL)
      else
        identical_formal_subscripts :=
                                     (sub1^.left_index = sub2^.left_index) and
                                     (sub1^.right_index = sub2^.right_index);
    end { identical_formal_subscripts } ;


    function formal_bits_in_common(sub1, sub2: subscript_ptr): boolean;
      { return TRUE if the two subscripts have any bits in common.  It is
        known that both subscripts are simple subranges and are in
	the configured bit order. }
      var
        found_common_bit: boolean;  { TRUE if a common bit found }
    begin
      found_common_bit := FALSE;
      if (sub1 <> NIL) and (sub2 <> NIL) then
        if left_to_right then
          found_common_bit := (sub2^.left_index <= sub1^.right_index) and
                               (sub2^.right_index >= sub1^.left_index)
        else 
          found_common_bit := (sub2^.right_index <= sub1^.left_index) and
                               (sub2^.left_index >= sub1^.right_index);

      formal_bits_in_common := found_common_bit;
    end { formal_bits_in_common } ;


  begin { found_formal_instance }
    if debug_3 then
      begin
        write(outfile, 'Enter formal_subscript: descrip=');
        dump_signal_descriptor(outfile, formal);
      end;

    instance := NIL;
    ok := TRUE;

    { search for the formal parameter in the formal/actual list }

    current_formal := formal_actual_list;  found := FALSE;
    while (current_formal <> NIL) and not found do
      begin
        if current_formal^.formal_parameter^.defined_by = formal_def then
          if formal^.kind = UNDEFINED then
            found := TRUE
          else if identical_formal_subscripts(formal^.bit_subscript,
                         current_formal^.formal_parameter^.bit_subscript) then
            found := TRUE
          else if formal_bits_in_common(formal^.bit_subscript,
                         current_formal^.formal_parameter^.bit_subscript) then
            begin
              error(199 { incompatible! });
              error_dump_current_parse_environment;
              error_dump_formal(formal);
              error_dump_signal_instance(current_formal^.formal_parameter);
              ok := FALSE;
            end;

        if not found then current_formal := current_formal^.next;
      end { while } ;

    if ok then
      begin
        if found then     { it's a previously parsed formal }
          begin
            instance := current_formal^.formal_parameter;
            formal^.kind := formal_def^.kind;

            current_element := current_formal;

            if directives.tap_property_found or
	       (cardinal_tap.binding = current_element) then
	      begin
	        error(69 { ill-formed cardinal tap });
		error_dump_indent(indent);
		error_dump_alpha('Pass-through pin');
		error_dump_alpha('s not allowed on');
		error_dump_alpha(' cardinal taps  ');
		error_dump_CRLF;
		error_dump_pin_name_string(clear_text_binding^.
		                           formal_parameter);
		ok := FALSE;
	      end
          end
        else
          begin
            { create a new formal/actual list element }

            new_formal_actual_list(formal_actual_list);
            current_element := formal_actual_list;

            instance := enter_signal_instance(formal, formal_def);
            current_element^.formal_parameter := instance;
            if formal_def^.kind <> UNDEFINED then
              current_element^.width :=
                                   width_of_subscript(instance^.bit_subscript,
                                                      formal_def^.kind);

            { assign the bubble state for the formal parameter }

            if not directives.check_bubble then
              current_element^.polarity := NO_POLARITY
            else if directives.inherit_assertion then
              current_element^.polarity := UNKNOWN_POLARITY
            else if formal_is_low_asserted then
              current_element^.polarity := COMPLEMENTED
            else
              current_element^.polarity := NORMAL;

            if directives.tap_property_found then
	      if cardinal_tap.binding <> NIL then
	        begin
		  error(69 { Ill-formed tap body });
		  error_dump_indent(indent);
		  error_dump_alpha('Can''t have more ');
		  error_dump_alpha(' than one pin   ');
		  error_dump_alpha(' with the       ');
        	  error_dump_char(' ');
		  error_dump_alpha(CARDINAL_TAP_prop_name^.name);
		  error_dump_alpha(' property       ');
		  error_dump_CRLF;
		  error_dump_pin_name_string(cardinal_tap.
		                             clear_text_binding^.
					     formal_parameter);
		  error_dump_pin_name_string(clear_text_binding^.
					     formal_parameter);
		  ok := FALSE;
		end
	      else
		begin
		  if debug_3 then
		    begin
		      write(Outfile, 'Is a cardinal tap of ');
		      writestring(Outfile, directives.cardinal_tap);
		      writeln(Outfile);
		    end; 
		  cardinal_tap.clear_text_binding := clear_text_binding;
		  cardinal_tap.binding := current_element;
		  cardinal_tap.tap := 
		    parse_the_subscript(directives.cardinal_tap);
                  if cardinal_tap.tap = NIL then ok := FALSE;
		  { error already issued (?) }
		end;
          end;

        { copy the unique properties attached to the pin to the formal/actual
          element.  First the properties attached with the PROPERTY command
          followed by the properties specified as part of the pin name. }

        if ok then
	  begin
	    expanded_props := 
	      expand_property_list_TMs(current_mtree_node, 
				       clear_text_binding^.pin_properties);
	    copy_unique_properties_to_subscript(current_element^.properties,
					      expanded_props);
	    copy_unique_properties_to_subscript(current_element^.properties,
						formal^.properties);
	    release_entire_property_list(expanded_props);
	  end;
      end;
    
    found_formal_instance := ok;
    if not ok then error_on_body := TRUE;

    if debug_3 then
      begin
        write(outfile, 'Formal_subscript: found=', found, ';  inst=');
        dump_signal_instance(outfile, instance);
      end;
  end { found_formal_instance } ;


(**)     { ------- parse the actual signal ------- }


  function generate_unique_net_id: xtring;
    { return a unique net id }
    var
      net_id: xtring;
  begin
    create_a_string(net_id, MAX_STRING_LENGTH);
    net_id^[0] := chr(0);
    if add_char_to_string(net_id, VIRTUAL_PREFIX_CHAR) then;
    unique_NET_ID_number := unique_NET_ID_number + 1;
    if add_number_to_string(net_id, unique_NET_ID_number) then;
    generate_unique_net_id := enter_string(net_id);
    net_id^[0] := chr(MAX_STRING_LENGTH);
    release_string(net_id);
  end { generate_unique_net_id } ;


  procedure parse_the_actual(formal_actual_pair: formal_actual_ptr;
                             clear_text_signal: clear_text_actual_list_ptr;
			     raw_parsed_actual: signal_descriptor_ptr;
			     assertion_control: control_type);
    { parse the given actual and return a concatenated signal list.
      (If raw_parsed_actual is specified, use it as the parsed value of
      the signal and release it when done -- calling procedure must beware.
      This is a hack to allow this procedure to be used to finish processing
      a tapped subrange produced by a cardinal bit tap which may contain
      a reference to a formal parameter. }

    var
      next,                             { temp signal storage for release }
      sig: signal_descriptor_ptr;       { current actual }
      def: signal_definition_ptr;       { definition for the actual }
      end_of_signal,                    { end of the formal PCS }
      signal,                           { signal within the formal PCS }
      last,                             { last concatenated signal read }
      current,                          { current concatenated signal }
      PCSP: propertied_CS_ptr;          { signals to be returned }
      found_signal: boolean;            { TRUE if this part of actual
                                          was found }
      found_net: boolean;               { TRUE if this net has been seen 
                                          before }
      first_signame: boolean;           { TRUE until found_net has been set }
      old_node: mtree_node_ptr;         { temporary storage for current node }
      net_id: xtring;                   { signal's net id }
      was_fixed: boolean;               { TRUE if assertion was fixed }
      assertion_is_unknown: boolean;    { TRUE if one or more signals no ass }
      uses_NAC: boolean;                { TRUE if this pin uses NAC }
      control: control_type;            { assertion control for signal }
      expanded_props: property_ptr;     { attached props with expanded values }
      properties: property_ptr;         { list of all name props on signal }


    function fix_actual_signal(actual: propertied_CS_ptr;
                               formal_net: xtring): boolean;
      { fix the signal instance in the given signal.  We need to create a
        new net specific def for the actual signal so that there is a unique
        place to hold INHERIT_PIN properties.  The net id value is made
        from the formal's net id and the actual's net id. }
      var
        new_net_id: xtring;             { net id for new def }
        signal: signal_descriptor_ptr;  { description of new signal }
        def: signal_definition_ptr;     { actual signal definition }


    begin { fix_actual_signal }
      if debug_14 or debug_15 then
        begin
          writeln(outfile, 'Entering fix_actual_signal: current instance=');
          dump_signal_instance(outfile, actual^.instance);
          write(outfile, '  formal_net=');
          writestring(outfile, formal_net);
          writeln(outfile);
        end;

      def := actual^.instance^.defined_by;

      new_net_id := generate_unique_net_id;

      signal := NIL;
      new_signal_descriptor(signal);
      with signal^ do
        begin
          signal_name := def^.signal^.name;
          polarity := def^.polarity;
          scope := def^.scope;
          low_asserted := actual^.instance^.low_asserted;
          kind := def^.kind;
          bit_subscript := copy_bit_subscript(actual^.instance^.bit_subscript);
          is_const := def^.is_const;
          net_id := new_net_id;
          replication_factor := actual^.instance^.replication_factor;
        end;

      actual^.instance := enter_signal_instance(signal, def);

      release_signal_descriptor(signal);

      fix_actual_signal := global_found_def_in_enter_signal;

      if debug_14 or debug_15 then
        begin
          writeln(outfile, 'Exiting fix_actual_signal: new_def=');
          dump_signal_definition(outfile, actual^.instance^.defined_by);
        end;
    end { fix_actual_signal } ;


    function make_NC_from_signal(sig: signal_descriptor_ptr):
                                                            propertied_CS_ptr;
      { create an NC signal of the same width as given signal and return it }

      var
        width: bit_range;                { width of NC signal to build }
        NC_signal: propertied_CS_ptr;    { signal to be returned }

    begin
      width := width_of_subscript(sig^.bit_subscript, sig^.kind);
      new_propertied_CS(NC_signal);
      NC_signal^.instance := build_NC_signal(width);
      NC_signal^.control := IGNORE_ALL;

      make_NC_from_signal := NC_signal;
    end { make_NC_from_signal } ;


  begin { parse_the_actual }
    if debug_3 then
      begin
        write(outfile, 'Entering parse_the_actual: ');
        dump_formal_actual(outfile, formal_actual_pair);
	if raw_parsed_actual <> NIL then
	  begin
	    write(Outfile, 'Parsing tapped bit list ');
	    case assertion_control of
	      IGNORE_ALL: writeln(Outfile, 'IGNORING assertion.');
	      NORMAL_SIGNAL: writeln(Outfile, 'checking assertion.');
	      OTHERWISE writeln(Outfile, 'with BOGUS assertion handling.');
	    end;
	  end;
      end;

    { on entry, current_mtree_node = body's node }

    old_node := current_mtree_node;
    current_mtree_node := current_mtree_node^.father_node;

    assertion_is_unknown := FALSE;
    first_signame := TRUE;
    found_net := FALSE;

    if debug_3 then
      begin
        write(outfile, 'Current_mtree_node: ');
        writestring(outfile, old_node^.macro_name);
        writeln(outfile);

        write(outfile, 'Father node: ');
        writestring(outfile, current_mtree_node^.macro_name);
        writeln(outfile);
      end;

    { process each signal in the concatenated signal }

    PCSP := NIL;  last := NIL;
    if raw_parsed_actual <> NIL then sig := raw_parsed_actual
    else sig := parse_signal_name(clear_text_signal^.actual_parameter, FALSE);

    net_id := clear_text_signal^.net_id;
    if (net_id = nullstring) then
      begin
        assert(200 { every signal must have one });
        write(CmpLog, ' Macro name=');
        dump_string(CmpLog, current_mtree_node^.macro_name);
        writeln(CmpLog);
        write(CmpLog, ' Body Name=');
        dump_string(CmpLog, old_node^.macro_name);
        writeln(CmpLog);
        write(CmpLog, ' Signal=');
        dump_signal_descriptor(CmpLog, sig);
      end;

    while sig <> NIL do
      begin
        if debug_3 then
          begin
            write(outfile, 'Actual being processed: ');
            dump_signal_descriptor(outfile, sig);
          end;

        end_of_signal := NIL;

        { find the signal in the signal table }

        def := NIL;  current := NIL;

        if is_strange_signal(sig) then sig^.net_id := nullstring

        { intermediate interface signals have NO virtual synonyms.  Root level
          interface signals are treated like local signals.  The NET_ID is
          important. }

        else if (sig^.scope = XINTERFACE) and
                (current_mtree_node^.father_node <> NIL) then
          sig^.net_id := nullstring

        else
          begin
            sig^.net_id := net_id;
            if (net_id^[1] = VIRTUAL_PREFIX_CHAR) and 
	       (raw_parsed_actual = NIL) then
              current_mtree_node^.macro^.written_with_GED := FALSE;
          end;

        found_signal := found_the_signal(sig, def);

        control := assertion_control;
        if control = NORMAL_SIGNAL then
	  if not check_pin_name_assertion(formal_actual_pair, NIL,
					  def, sig, was_fixed) then
	    begin
	      if bubble_check then
		begin
		  error(201 { bubble check error });
		  error_dump_macro_def(current_mtree_node^.macro);
		  error_dump_body_name(old_node^.macro_name);
		  error_dump_formal_instance
			(formal_actual_pair^.formal_parameter);
		  error_dump_signal_descriptor(sig);
		end;
	      control := IGNORE_ALL;
	    end;

        if (def^.scope = XINTERFACE) and (def^.node^.father_node <> NIL) then

          { check to see that the interface signal really exists on body }

          if not found_signal then
            begin
              error(145 { pin name does not exist });
              error_dump_current_parse_environment;
              error_dump_signal_descriptor(sig);
              error_dump_all_pin_names(def^.node^.params);
              if debug_3 then
                dump_list_of_signal_definitions(outfile, def^.node^.signals);
              new_propertied_CS(current);
              current^.instance := fix_weird_formal(def, sig, NIL);
	      error_on_body := TRUE;
            end

          { make sure reference and pin are same KIND }

          else if (def^.kind = SINGLE) and (sig^.kind = VECTOR) then
            begin
              error(215 { formal is scalar and instance is vector });
              error_dump_current_parse_environment;
              error_dump_signal_descriptor(sig);
              current := make_NC_from_signal(sig);
	      error_on_body := TRUE;
            end
          else if (def^.kind = VECTOR) and (sig^.kind = SINGLE) and
                  not current_mtree_node^.is_plumbing_node then
            begin
              error(211 { formal is vector and instance is scalar });
              error_dump_current_parse_environment;
              error_dump_signal_descriptor(sig);
              current := make_NC_from_signal(sig);
	      error_on_body := TRUE;
            end

          { don't permit subscripts on NWC signals }

          else if (def^.kind = UNDEFINED) and (sig^.kind = VECTOR) then
            begin
              error(198 { subscript on undefined formal is illegal });
              error_dump_current_parse_environment;
              error_dump_signal_descriptor(sig);
              current := make_NC_from_signal(sig);
	      error_on_body := TRUE;
            end

          { all is OK.  Find the signals connected to the pin }

          else
            begin
              current := resolve_instance_of_parameter(def, sig, uses_NAC);

              if uses_NAC then
                control := IGNORE_PIN
              else
                control := IGNORE_ALL;

              if debug_3 and (current = NIL) then
                writeln(outfile, 'RESOLVE returned NIL!!!!!');

              { copy properties from the instance (signal descriptor) to
                the PCS representing the formal parameter. }

              signal := current;
              while signal <> NIL do
                begin
                  { if the signal has instance specific properties (from the
                    formal parameter) create a new signal def to carry them }

                  if (signal^.properties <> NIL) or
                     (sig^.properties <> NIL) or
                     (clear_text_signal^.properties <> NIL) then
                    found_signal := fix_actual_signal(signal, net_id);

                  (* NOTE: the current mechanism for fix_actual_signal
		     ALWAYS creates a new actual, so found_signal will
		     be TRUE if there were properties on the (interface)
		     actual.  Therefore we cannot use it to determine
		     whether or not the net has been seen. *)
                  (*
		  if first_signame not is_NC_signal(sig^.signal_name) then
		    begin
		      found_net := found_signal;
		      first_signame := FALSE;
		    end;
                  *)

                  def := signal^.instance^.defined_by;
                  if signal^.control <> IGNORE_ALL then
                    signal^.control := control;

                  if signal^.instance^.defined_by^.polarity =
                                                         UNKNOWN_POLARITY then
                    assertion_is_unknown := TRUE;


                  { copy properties from the resolved formal parameter (pin)
                    to the actual signal.  This is correct for INHERIT_SIGNAL
                    properties but is not correct for INHERIT_PIN properties
                    which must be handled via come other mechanism. }

                  if (signal^.properties <> NIL) (* and not found_net *) then
                    begin
                      if debug_14 then
			writeln(outfile, 'Interface \\signal properties');

                      copy_inherit_pin_properties_to_BP(def^.properties,
                                                        signal^.properties,
                                            signal^.instance^.bit_subscript);
                      copy_unique_properties_to_bit_property(def^.properties,
                                                         signal^.properties,
                                             signal^.instance^.bit_subscript);
                    end;


                  (*  This is not needed  **********
                  signal^.properties := NIL;

                  add_pin_properties_to_subscript_property
                                              (formal_actual_pair^.properties,
                                               -1, -1,
                                               signal^.properties);
                   ********************)


                  { copy the properties on the formal parameter to the actual
                    signals.  If inherit pin copy to the actual PCS otherwise
                    copy to the actual's signal def. }

                  if (sig^.properties <> NIL) (* and not found_net *) then
                    begin
                      if debug_14 then writeln(outfile, '\\Signal properties');

                      copy_inherit_pin_properties_to_BP(def^.properties,
                                                        sig^.properties,
                                             signal^.instance^.bit_subscript);
                      copy_unique_properties_to_bit_property(def^.properties,
                                                             sig^.properties,
                                             signal^.instance^.bit_subscript);
                    end;

                  { copy the signal properties attached to the signal with
                    the PROPERTY command.  Inherit pin properties are
                    attached to the actual's PCS and others to the actual's
                    signal definition. }

                  if (clear_text_signal^.properties <> NIL) (* and
                     not found_net *) then
                    begin
                      if debug_14 then
                        writeln(outfile, 'properties attached to net');

                      expanded_props := expand_property_list_TMs(
                        current_mtree_node, clear_text_signal^.properties);
                      copy_inherit_pin_properties_to_BP(def^.properties,
                                             expanded_props,
                                             signal^.instance^.bit_subscript);
                      copy_unique_properties_to_bit_property(def^.properties,
                                             expanded_props,
                                             signal^.instance^.bit_subscript);
                      release_entire_property_list(expanded_props);
                    end;

                  end_of_signal := signal;
                  signal := signal^.next;
                end;
            end

        else
          begin
            { signal is not an interface.  Add to the actual CS being built }

            new_propertied_CS(current);
            properties := sig^.properties;
            sig^.properties := NIL;
            current^.instance := enter_signal_instance(sig, def);
            current^.control := control;

            if def^.polarity = UNKNOWN_POLARITY then
              assertion_is_unknown := TRUE;

            { copy properties with inherit pin attribute to the actual's PCS.
              Other properties are copied to the actual's signal def. }

	    if first_signame and not is_NC_signal(sig^.signal_name) then
              begin
		found_net := found_signal;
		first_signame := FALSE;
	      end;

            if (properties <> NIL) and
	       (not found_net or is_NC_signal(sig^.signal_name)) then
              begin
                if debug_14 then writeln(outfile, '\\Signal properties');

                copy_inherit_pin_properties_to_BP
                                   (current^.instance^.defined_by^.properties,
                                    properties,
                                    current^.instance^.bit_subscript);
                copy_unique_properties_to_bit_property(def^.properties,
                                                       properties,
                                            current^.instance^.bit_subscript);
              end;

            { copy properties attached to the signal with the PROPERTY
              command.  Properties with the inherit pin property are copied
              to the actual's PCS and others to the actual's signal def. }

            if (clear_text_signal^.properties <> NIL) and 
	       (not found_net or is_NC_signal(sig^.signal_name)) then
              begin
                if debug_14 then writeln(outfile, 'Clear text properties');

                expanded_props := expand_property_list_TMs(
                  current_mtree_node, clear_text_signal^.properties);
                copy_inherit_pin_properties_to_BP
                                  (current^.instance^.defined_by^.properties,
                                   expanded_props,
                                   current^.instance^.bit_subscript);
                copy_unique_properties_to_bit_property(def^.properties,
                                            expanded_props,
                                            current^.instance^.bit_subscript);
                release_entire_property_list(expanded_props);
              end;

            if (current_mtree_node^.father_node = NIL) or
	      (current^.instance^.defined_by^.scope = GLOBAL) then
              record_instance_for_synonyms(current^.instance);
          end;

        if last = NIL then PCSP := current else last^.next := current;
        if end_of_signal = NIL then
          last := current
        else
          last := end_of_signal;

        next := sig^.next;

        release_complete_signal_descriptor(sig);

        sig := next;
      end { while } ;

    formal_actual_pair^.actual_parameter^.signal := PCSP;

    current_mtree_node := old_node;

    { set up assertion knowledge (ASSERTION_IS_UNKNOWN) in the actual.
      Width knowledge (WIDTH_IS_UNKNOWN) is set up in a procedure that
      processes the actual after parsing. }

    if assertion_is_unknown then
      formal_actual_pair^.actual_parameter^.assertion_state :=
                                                             ASSERTION_UNKNOWN
    else
      formal_actual_pair^.actual_parameter^.assertion_state :=
                                                              ASSERTION_KNOWN;

    if debug_7 or debug_3 then
      begin
        writeln(outfile, 'Exiting parse_the_actual:');
        dump_actual_list_element(outfile,
                                 formal_actual_pair^.actual_parameter);
        dump_formal_actual(outfile, formal_actual_pair);
      end;
  end { parse_the_actual  } ;


#include "bittap.p"


begin { evaluate_bindings }
  if debug_3 or debug_35 then
    begin
      write(outfile, 'Enter evaluate_bindings: ');
      print_string(outfile, current_mtree_node^.macro_name);
      writeln(outfile);
      dump_bindings_list(Outfile, clear_text_bindings_list);
    end;

  uses_NWC := FALSE;
  is_cardinal_tap := FALSE;
  error_on_body := FALSE;

  cardinal_tap.clear_text_binding := NIL;
  cardinal_tap.binding := NIL;
  cardinal_tap.tap := NIL;

  if formal_actual_list <> NIL then assert(41 { better not happen! });

  clear_text_binding := clear_text_bindings_list;
  while clear_text_binding <> NIL do
    begin
      formal := parse_signal_name(clear_text_binding^.formal_parameter, 
                                  FALSE);
      formal^.net_id := nullstring;

      if name_of_pin_is_OK then
        begin
          formal_def := NIL;
          if found_the_signal(formal, formal_def) then ;

          if found_formal_instance(formal_instance) then
            begin
              { place signals into the formal/actual lists but do not create
                any synonyms. }

              actual_signal := clear_text_binding^.actual_parameter;
              while actual_signal <> NIL do
                begin
                  new_actual_list(current_element^.actual_parameter);

                  parse_the_actual(current_element, actual_signal, 
		                   NIL, NORMAL_SIGNAL);

                  if check_and_fix_formal_actual(current_element, FALSE,
                                                 dummy) then;

                  if directives.inherit_assertion then
                    current_element^.uses_NAC := TRUE;

                  if not directives.check_width then
                    current_element^.is_NWC_pin := TRUE;

                  actual_signal := actual_signal^.next;
                end;
            end;
        end;

      { restriction for \NWC and \NAC use }

      if not directives.check_width then uses_NWC := TRUE;

{      release_subscript_list(formal^.bit_subscript);}
      release_signal_descriptor(formal);

      clear_text_binding := clear_text_binding^.next;
    end { while } ;

  if cardinal_tap.binding <> NIL then
    begin
      if not error_on_body then process_cardinal_tap(cardinal_tap);
      release_entire_subscript(cardinal_tap.tap);
    end;

  if debug_3 or debug_7 or debug_35 then
    begin
      writeln(outfile, 'Exiting evaluate_bindings:');
      dump_formal_actual_list(outfile, formal_actual_list);
    end;
end { evaluate_bindings } ;


