function found_actual_assertion(actual_list: actual_list_ptr;
                                var signal_inst: signal_instance_ptr):
                                                                      boolean;
  { search the given actual lists for a signal with known assertion and
    return it.  If none is found, return FALSE otherwise return TRUE
    and the signal defining the polarity.  If the actual signal's polarity
    is NO_POLARITY, ignore that signal. }
  var
    actual: actual_list_ptr;     { current actual in the list }
    signal: propertied_CS_ptr;   { current signal in the actual }
    found: boolean;              { TRUE if assertion is found }
begin
  actual := actual_list;  found := FALSE;  signal_inst := NIL;
  while (actual <> NIL) and not found do
    begin
      signal := actual^.signal;
      while (signal <> NIL) and not found do
        begin
          if signal^.control <> IGNORE_ALL then
            begin
              signal_inst := signal^.instance;

              if signal_inst^.defined_by^.polarity 
	        in [NORMAL, COMPLEMENTED] then found := TRUE;
            end;

          if not found then signal := signal^.next;
        end;

      if not found then actual := actual^.next;
    end;

  found_actual_assertion := found;

  if not found then signal_inst := NIL;

  if debug_11 then
    begin
      writeln(outfile, 'Exiting found_actual_assertion: inst=');
      dump_signal_instance(outfile, signal_inst);
    end;
end { found_actual_assertion } ;


procedure fix_polarity(sig_def: signal_definition_ptr; low_asserted: boolean);
  { fix the polarity of the given signal (DEF) to the given polarity }
  var
    def: signal_definition_ptr; { sig_def or one of its "virtual" siblings }
    inst: signal_instance_ptr;  { current instance }
begin
  if debug_11 then
    begin
      write(outfile, 'Entering fix_polarity: low_ass=');
      dump_boolean(outfile, low_asserted);
      writeln(outfile);
      write(outfile, '  Def=');
      dump_signal_definition(outfile, sig_def);
    end;

  { sig def can be virtual, or have virtual siblings due to the action
    of fix_actual_signal (within evaluate_bindings) }

  def := virtual_base(sig_def);
  if def^.polarity <> UNKNOWN_POLARITY then assert(187 { error! })
  else
    begin
      if debug_11 then
        begin
	  write(outfile, '    Fixing virtual base');
	  dump_virtual_spec(outfile, def);
	  writeln(outfile);
	end;

      if low_asserted then
        begin
          def^.polarity := COMPLEMENTED;
          inst := def^.instances;
          while inst <> NIL do
            begin
              inst^.low_asserted := TRUE;
              inst := inst^.next;
            end;
        end
      else
        def^.polarity := NORMAL;

      { fix up the stack link (if there is one).  Move the low asserted
        signal from the high asserted stack to the low asserted stack. }

      {***** may be anywhere in the stack; insert at proper place *****}

      if debug_11 and low_asserted then
        writeln(outfile, '  Inserting low asserted into the stack');

      if low_asserted then
        with def^.signal^ do
          if high_asserted = def then
            begin
              high_asserted := def^.stack;
              def^.stack := low_asserted;
              low_asserted := def;
    
              if debug_11 then dump_stack(outfile, def^.signal);
            end
          else 
            begin
              assert(192 { this cannot be! });
              dump_stack(CmpLog, def^.signal);
              if debugging then dump_stack(Outfile, def^.signal);
            end;
    end;

  def := def^.next_virtual_def;
  while def <> NIL do
    begin
      if debug_11 then
        begin
	  write(outfile, '    Fixing virtual def ');
	  dump_virtual_spec(outfile, def);
	  writeln(outfile);
	end;

      if def^.polarity <> UNKNOWN_POLARITY then assert(187 { error! })
      else
        begin
          if low_asserted then
            begin
              def^.polarity := COMPLEMENTED;
              inst := def^.instances;
              while inst <> NIL do
                begin
                  inst^.low_asserted := TRUE;
                  inst := inst^.next;
                end;
            end
          else
            def^.polarity := NORMAL;
        end;
      def := def^.next_virtual_def;
    end;   

  if debug_11 then writeln(outfile, 'Exiting fix_polarity');
end { fix_polarity } ;


function check_pin_name_assertion(formal_actual_pair: formal_actual_ptr;
                                  signal_inst: signal_instance_ptr;
                                  signal_def: signal_definition_ptr;
                                  signal_desc: signal_descriptor_ptr;
                                  var was_fixed: boolean): boolean;
  { check the assertion of the given signal with the given pin.  For
    a pin the polarity has the following meaning:
       NORMAL           = high asserted
       COMPLEMENTED     = low asserted
       UNKNOWN_POLARITY = NAC with none connected yet or UNNAMED unknown
       NO_POLARITY      = NBC or NC signal
    Check and/or coerce the polarity of the pin and actual signal.  If
    an error is detected, return FALSE.  If a signal's assertion is
    fixed, return WAS_FIXED TRUE.  The actual signal can be described 
    either by a signal instance OR by signal definion and descriptor. }
  var
    found_error: boolean;           { TRUE if error was detected }
    def: signal_definition_ptr;     { the signal definition }
    low_asserted: boolean;          { TRUE iff inst (or descriptor) is low
                                      asserted }
begin
  if debug_11 then
    begin
      writeln(outfile, 'Enter check_pin_name_ass: f/a=');
      dump_formal_actual(outfile, formal_actual_pair);
      if signal_inst <> NIL then
        begin
          writeln(outfile, '  sig inst=');
          dump_signal_instance(outfile, signal_inst);
        end
      else
        begin
          writeln(outfile, '  sig def=');
          dump_signal_definition(outfile, signal_def);
          writeln(outfile, '  sig desc=');
          dump_signal_descriptor(outfile, signal_desc);
        end;
    end;

    if signal_inst = NIL then
      begin
        def := signal_def;
        low_asserted := signal_desc^.low_asserted;
      end
    else
      begin
        def := signal_inst^.defined_by;
        low_asserted := signal_inst^.low_asserted;
      end;

  found_error := FALSE;  was_fixed := FALSE;

  { check the polarity of the pin with that of the signal or given }

  if def^.polarity <> NO_POLARITY then
    case formal_actual_pair^.polarity of
      NORMAL:
          if def^.polarity = UNKNOWN_POLARITY then
            begin
              fix_polarity(def, ASSERT_HIGH);
              if signal_desc <> NIL then
                begin
                  signal_desc^.low_asserted := FALSE;
                  signal_desc^.polarity := NORMAL;
                end;
              was_fixed := TRUE;

              if debug_11 then writeln(outfile, 'fixed signal def');
            end
          else
            if low_asserted then found_error := TRUE;

      COMPLEMENTED:
          if def^.polarity = UNKNOWN_POLARITY then
            begin
              fix_polarity(def, ASSERT_LOW);
              if signal_desc <> NIL then
                begin
                  signal_desc^.low_asserted := TRUE;
                  signal_desc^.polarity := COMPLEMENTED;
                end;
              was_fixed := TRUE;

              if debug_11 then writeln(outfile, 'fixed signal def');
            end
          else
            if not low_asserted then found_error := TRUE;

      UNKNOWN_POLARITY:
          begin
            if (def^.polarity in [NORMAL,COMPLEMENTED]) then
              begin
                if low_asserted then
                    formal_actual_pair^.polarity := COMPLEMENTED
                else
                  formal_actual_pair^.polarity := NORMAL;
              end;

              was_fixed := TRUE;

              if debug_11 then writeln(outfile, 'fixed formal polarity');
          end;

      NO_POLARITY:
          { anything goes } ;
    end { case } ;

  check_pin_name_assertion := not found_error;

  if debug_11 then
    begin
      write(outfile, 'Exiting check_pin_name_assertion=');
      dump_boolean(outfile, not found_error);
      write(outfile, '; was_fixed=');
      dump_boolean(outfile, was_fixed);
      writeln(outfile);
    end;
end { check_pin_name_assertion } ;


function check_mutual_signal_assertions(defining_inst,
                                        signal_inst: signal_instance_ptr;
                                        var was_fixed: boolean): boolean;
  { check the assertion of the given defining signal with the def.  A signal's
    polarity has the following meaning:
       NORMAL           = assertion determined by low_asserted flag
       COMPLEMENTED     = assertion determined by low_asserted flag
       UNKNOWN_POLARITY = UNNAMED unknown -- coerce to match assertion
                          of defining inst (if it has one)
       NO_POLARITY      = NC signal -- no assertion
    If an error is detected, return FALSE.  If the def's assertion is
    fixed, return WAS_FIXED TRUE.  It is assumed that the defining signal's
    assertion is not unknown. }
  var
    found_error: boolean;           { TRUE if error was detected }
begin
  if debug_11 then
    begin
      writeln(outfile, 'Enter check_mutual_signal_assertions; defining=');
      dump_signal_instance(outfile, defining_inst);
      writeln(outfile, '  sig inst=');
      dump_signal_instance(outfile, signal_inst);
    end;

  found_error := FALSE;  was_fixed := FALSE;

  if signal_inst^.defined_by^.polarity <> NO_POLARITY then
    case defining_inst^.defined_by^.polarity of
      NORMAL,COMPLEMENTED:
          if signal_inst^.defined_by^.polarity = UNKNOWN_POLARITY then
            begin
              fix_polarity(signal_inst^.defined_by, 
	                   defining_inst^.low_asserted);
              was_fixed := TRUE;

              if debug_11 then writeln(outfile, 'fixed signal def');
            end
          else
            if signal_inst^.low_asserted <> defining_inst^.low_asserted then
	      found_error := TRUE;

      UNKNOWN_POLARITY:
          begin
            assert(199 { this better not happen });
            write(CmpLog, ' defining def=');
            dump_signal_instance(CmpLog, defining_inst);
          end;

      NO_POLARITY:
          { anything goes } ;
    end { case } ;

  check_mutual_signal_assertions := not found_error;

  if debug_11 then
    begin
      write(outfile, 'Exiting check_mutual_signal_assertions=');
      dump_boolean(outfile, not found_error);
      write(outfile, '; was_fixed=');
      dump_boolean(outfile, was_fixed);
      writeln(outfile);
    end;
end { check_mutual_signal_assertions } ;


function check_and_fix_signal_assertions(node: mtree_node_ptr):  boolean;
  { check the assertions of all of the signals on the given formal/actual
    list.  Fix those assertions that can be fixed.  If any assertion was
    fixed, return TRUE.

    ASSUMPTION: if unknown_polarity = FALSE, ALL signals in the actual have
                known assertion.  }
  type
    bubble_error_type = (PIN_AND_SIGNAL, MUTUAL_SIGNALS);

  var
    this_one_fixed,                        { TRUE if this signal was fixed }
    one_was_fixed: boolean;                { TRUE if one signal was fixed }
    formal_actual_pair: formal_actual_ptr; { current formal/actual pair }
    actual: actual_list_ptr;               { current actual on formal }
    signal: propertied_CS_ptr;             { current piece of actual }
    signal_inst: signal_instance_ptr;      { defining signal polarity }
    one_is_unknown: boolean;               { TRUE if at least one is unknown }
    assertion_inst:
                   signal_instance_ptr;    { assertion determining instance }


  function has_unknown_assertion(signal: propertied_CS_ptr): boolean;
    { return TRUE if the given signal has unknown assertion }
  begin
    has_unknown_assertion := signal^.instance^.defined_by^.polarity =
                                                             UNKNOWN_POLARITY;
  end { has_unknown_assertion } ;


  procedure generate_bubble_error(which: bubble_error_type);
    { display a bubble error message and related stuff }
    var
      old_path_prop: xtring;      { original PATH property value }
      prop: property_ptr;         { property returned from search }
      polarity: signal_polarity;  { correct polarity }
  begin
    if bubble_check then
      begin
        old_path_prop := current_path_prop;
        if find_property(node^.called_by^.properties,
                         PATH_prop_name, prop) then
          current_path_prop := prop^.text;
    
        if which = PIN_AND_SIGNAL then
          begin
            error(201 { signal fails bubble check });
            polarity := formal_actual_pair^.polarity;
          end
        else
          begin
            error(149 { synonyms must use single assertion });
            if (assertion_inst^.low_asserted) then polarity := COMPLEMENTED
                                              else polarity := NORMAL;
          end;

        error_dump_macro_def(node^.father_node^.macro);
        error_dump_body_node(node);
        error_dump_formal_instance(formal_actual_pair^.formal_parameter);
        error_dump_signal_instance(signal^.instance);

        error_dump_indent(INDENT);
        error_dump_alpha('Should be:      ');
        error_dump_char(' ');
        if polarity = COMPLEMENTED then error_dump_alpha('low asserted    ')
                                   else error_dump_alpha('high asserted   ');
        error_dump_CRLF;
    
        current_path_prop := old_path_prop;
      end;

    signal^.control := IGNORE_ALL;
  end { generate_bubble_error } ;


begin { check_and_fix_signal_assertions }
  if debug_11 then
    begin
      writeln(outfile, 'Entering check_and_fix_signal_assertion');
      dump_formal_actual_list(outfile, node^.params);
    end;

  one_was_fixed := FALSE;

  formal_actual_pair := node^.params;
  while formal_actual_pair <> NIL do
    begin
      if debug_11 then
        begin
          write(outfile, '--Checking ');
          dump_formal_actual(outfile, formal_actual_pair);
        end;

      {------------------------------------}
      {  fix formal parameter's assertion  }
      {------------------------------------}

      signal_inst := NIL;

      if formal_actual_pair^.polarity = UNKNOWN_POLARITY then
        begin
          if debug_11 then writeln(outfile, '--fixing formal assertion');

          if found_actual_assertion(formal_actual_pair^.actual_parameter,
                                    signal_inst) then
            if not check_pin_name_assertion(formal_actual_pair,
                                            signal_inst, NIL, NIL,
                                            this_one_fixed) then
              begin
                assert(198 { this cannot happen });
                write(CmpLog, 'Formal/actual=');
                dump_formal_actual(CmpLog, formal_actual_pair);
                write(CmpLog, 'Signal inst=');
                dump_signal_instance(CmpLog, signal_inst);
              end
            else
              if this_one_fixed then
                one_was_fixed := TRUE;
        end;

      {-------------------------------------}
      {  fix actual parameters' assertions  }
      {-------------------------------------}

      if found_actual_assertion(formal_actual_pair^.actual_parameter,
                                assertion_inst) then ;

      if (formal_actual_pair^.polarity <> UNKNOWN_POLARITY) and
         (formal_actual_pair^.polarity <> NO_POLARITY)  then
        begin
          if debug_11 then writeln(outfile, '--Checking actual assertions');

          actual := formal_actual_pair^.actual_parameter;
          while actual <> NIL do
            begin
              if debug_11 then
                begin
                  write(outfile, 'Checking actual w/ formal: ');
                  dump_actual_list_element(outfile, actual);
                end;

              { check the assertion of the actual against the assertion of
                the pin to which the actual is connected. }

              if actual^.assertion_state <> ASSERTION_CHECKED then
                begin
                  one_is_unknown := FALSE;

                  signal := actual^.signal;
                  while (signal <> NIL) do
                    begin
                      this_one_fixed := FALSE;
                      if signal^.control = NORMAL_SIGNAL then
                        begin
                          if not check_pin_name_assertion(formal_actual_pair,
                                                 signal^.instance, NIL, NIL,
                                                 this_one_fixed) then
                            generate_bubble_error(PIN_AND_SIGNAL)
                          else if this_one_fixed then
                            one_was_fixed := TRUE;
                        end
                      else if signal^.control <> IGNORE_ALL then
                        one_is_unknown := TRUE;

                      {------------------
                      if not this_one_fixed then
                        if has_unknown_assertion(signal) then
                          one_is_unknown := TRUE;
                       -------------}

                      signal := signal^.next;
                    end;

                  if not one_is_unknown then
                    actual^.assertion_state := ASSERTION_CHECKED;
                end;

              if debug_11 then
                begin
                  write(outfile, 'Checking actual w/ actual: ');
                  dump_actual_list_element(outfile, actual);
                end;

              { check the assertion of the actual with the assertion of the
                other actuals connected to the same pin as this actual. }

              if actual^.assertion_state <> ASSERTION_CHECKED then
                begin
                  if assertion_inst = NIL then
                    if found_actual_assertion(formal_actual_pair^.
                                                             actual_parameter,
                                              assertion_inst) then ;

                  if assertion_inst <> NIL then
                    begin
                      one_is_unknown := FALSE;

                      signal := actual^.signal;
                      while (signal <> NIL) do
                        begin
                          this_one_fixed := FALSE;
                          if signal^.control <> IGNORE_ALL then
                            if not check_mutual_signal_assertions
                                                (assertion_inst,
                                                 signal^.instance,
                                                 this_one_fixed) then
                              generate_bubble_error(MUTUAL_SIGNALS)
                            else if this_one_fixed then
                              one_was_fixed := TRUE;

                          if this_one_fixed then
                            if has_unknown_assertion(signal) then
                              one_is_unknown := TRUE;

                          signal := signal^.next;
                        end;

                      if not one_is_unknown then
                        actual^.assertion_state := ASSERTION_CHECKED;
                    end;
                end;

              actual := actual^.next;
            end { while } ;
        end { if } ;

      formal_actual_pair := formal_actual_pair^.next;
    end { while } ;

  check_and_fix_signal_assertions := one_was_fixed;

  if debug_11 then
    begin
      write(outfile, 'Exiting check_and_fix_signal_assertions: ');
      dump_boolean(outfile, one_was_fixed);
      writeln(outfile);
    end;
end { check_and_fix_signal_assertions } ;

                                          
(**)     { ------- resolve all signal assertions ------- }


function resolve_all_signal_assertions(node: mtree_node_ptr): boolean;
  { resolve the assertions of all of the signals in the current drawing and
    within any plumbing drawings called by this drawing.
    Return TRUE if any signal's assertion is fixed. }
  const
    MAX_PASSES = 500;                   { maximum # passes through signals }
  var
    number_of_passes: 0..MAX_PASSES;    { number of passes through signals }
    son: mtree_node_ptr;                { current son of the given node }
    some_assertion_was_fixed: boolean;  { TRUE if any signal was fixed }
    one_was_fixed: boolean;             { TRUE if at least one was fixed }
begin
  if debug_11 then
    begin
      write(outfile, 'Entering resolve_all_signal_assertions: is_plum=');
      dump_boolean(outfile, node^.is_plumbing_node);
      write(outfile, ' is_tap=');
      dump_boolean(outfile, node^.is_cardinal_tap);
      writeln(outfile);
      write(outfile, '  Node name=');
      writestring(outfile, node^.macro_name);
      writeln(outfile);
    end;

  number_of_passes := 0;             { try MAX_PASSES times to resolve }

  one_was_fixed := FALSE;
  repeat
    son := node^.son;  some_assertion_was_fixed := FALSE;
    while son <> NIL do
      begin
        if son^.is_plumbing_node and not son^.is_cardinal_tap then
          if resolve_all_signal_assertions(son) then
            begin
              one_was_fixed := TRUE;
              some_assertion_was_fixed := TRUE;
            end;

        if check_and_fix_signal_assertions(son) then
          begin
            one_was_fixed := TRUE;
            some_assertion_was_fixed := TRUE;
          end;

        son := son^.next;
      end;

    number_of_passes := number_of_passes + 1;
  until (number_of_passes >= MAX_PASSES) or not some_assertion_was_fixed;

  resolve_all_signal_assertions := one_was_fixed;

  if debug_11 then
    begin
      write(outfile, 'Exiting resolve_all_signal_assertions:');
      dump_boolean(outfile, one_was_fixed);
      writeln(outfile);
    end;
end { resolve_all_signal_assertions } ;


