function min(a,b: longint): longint;
  { return the minimum }
begin
  if a < b then min := a
           else min := b;
end { min } ;


procedure halt_with_status(halt_code: longint);
  { halt the program generating some termination status }

#if VAX
  procedure SYS$EXIT(%IMMED halt_code: integer);               EXTERN;
#endif

begin
#if VAX
  SYS$EXIT(halt_code); 
#endif
#if SVS
  halt(halt_code); 
#endif
#if SUN || PMAX
  exit(halt_code);
#endif
end { halt_with_status } ;

{ ------- Time utility ------- }

function elapsed_time: longint;
  { return the current elapsed time in milliseconds }
  const
    elapsed_clock_units = 1000; { system intrinsic units to millisecs }

#if VAX
    offset_to_hours     = 1; 
    offset_to_minutes   = 4; 
    offset_to_seconds   = 7; 
    offset_to_hundredths= 10;
#endif
#if UNIX
    offset_to_hours     = 12;    { 1st char of hrs. in time string }
    offset_to_minutes   = 15;    { 1st char of min. in time string }
    offset_to_seconds   = 18;    { 1st char of sec. in time string }
#endif

  var
    current_time: time_string;   { current time in ASCII }
    hours,                       { current number of hours }
    minutes,                     { current number of minutes }
    seconds,                     { current number of seconds }
    hundredths: longint;         { current number of hundredths of seconds }


  function convert (var s: time_string; pos: longint): longint;
    { convert the next two digits of the given time string (from POS) into
      an integer. }
  begin
    convert := 10*(ord(s[pos]) - ord('0')) + (ord(s[pos+1]) - ord('0'));
  end { convert } ;


 begin { elapsed_time }
#if VAX
  TIME(current_time);
#endif
#if UNIX
  seconds := epochsec;   convert_time(seconds, current_time);
#endif
  hours := convert(current_time, offset_to_hours);
  minutes := convert(current_time, offset_to_minutes);
  seconds := convert(current_time, offset_to_seconds);
#if VAX
  hundredths := convert(current_time, offset_to_hundredths);
#endif
#if UNIX
  hundredths := 0;
#endif
  elapsed_time := hours      * (3600 * elapsed_clock_units) +
                  minutes    * (60   * elapsed_clock_units) +
                  seconds    * (1    * elapsed_clock_units) +
                  hundredths * (elapsed_clock_units DIV 100);
end { elapsed_time } ;


function CPU_time: longint;
  { return the current CPU time in milliseconds }
  const
#if VAX
    CPU_clock_units = 1;   { system intrinsic units in milliseconds }
#endif
#if UNIX
    CPU_clock_units = 16.66; { system intinsic units -- 1/60th seconds }
#endif
begin
#if VAX
  CPU_time := CLOCK div CPU_clock_units;
#endif
#if UNIX
  CPU_time := trunc(vclock * CPU_clock_units);
#endif
end { CPU_time } ;


procedure print_time(var f: textfile; current_time: longint);
  { print the time to the given file.  Output leading zeroes. }
  var
    hours, minutes, seconds,
    hundredths: longint;          { current time specifications }
begin
  { convert an integer (assumed to represent a time in units of
    CLOCK_UNITS) to hours, minutes, etc.  and return as globals. }

  hours := current_time DIV (3600*CLOCK_UNITS);
  current_time := current_time - (hours*3600*CLOCK_UNITS);
  minutes:= current_time DIV (60*CLOCK_UNITS);
  current_time := current_time - (minutes*60*CLOCK_UNITS);
  seconds:= current_time DIV CLOCK_UNITS;
  current_time := current_time - (seconds*CLOCK_UNITS);
  hundredths := current_time DIV (CLOCK_UNITS DIV 100);

  if hours < 10 then write(f, '0', hours:1)
                else write(f, hours:2);
  write(f, ':');
  if minutes < 10 then write(f, '0', minutes:1)
                  else write(f, minutes:2);
  write(f, ':');
  if seconds < 10 then write(f, '0', seconds:1)
                  else write(f, seconds:2);
  write(f, '.');
  if hundredths < 10 then write(f, '0', hundredths:1)
                     else write(f, hundredths:2);
end { print_time } ;

procedure exec_time(var last_elapsed_time: longint;
                     var last_CPU_time: longint; just_delta: boolean);
  { display the execution time, both CPU time and elapsed time.  If
    JUST_DELTA, then display only the delta time from last_CPU_time to the
    current CPU time and reset last_elapsed_time and last_CPU_time. }
  var
    current_elapsed_time,               { current elapsed time }
    current_CPU_time:  longint;         { calculated CPU time }


  procedure display_time_summary(var f: textfile);
    { write time summaries to the given file }
  begin
    writeln(f);
    write(f, ' Start time   = ');
    print_time(f, last_elapsed_time);  writeln(f);
    write(f, ' Ending time  = ');
    print_time(f, current_elapsed_time);  writeln(f);
    write(f, ' Elapsed time = ');
    if last_elapsed_time < current_elapsed_time then
      print_time(f, current_elapsed_time - last_elapsed_time)
    else
      print_time(f, (24*60*60*1000+current_elapsed_time) - last_elapsed_time);
    writeln(f);
    write(f, ' CPU time     = ');
    print_time(f, current_CPU_time - last_CPU_time);  writeln(f);
  end { display_time_summary } ;


begin { exec_time }
  current_elapsed_time := elapsed_time;    { get the current elapsed time }
  current_CPU_time     := CPU_time;        { get the current CPU time }
  
  if just_delta then
    begin
      write(monitor, '(');
      print_time(monitor, current_CPU_time - last_CPU_time);
      writeln(monitor, ')');
    end
  else
      display_time_summary(monitor);  writeln(monitor);
  last_elapsed_time := current_elapsed_time;
  last_CPU_time := current_CPU_time;
end { exec_time } ;


procedure post_compile_time(var f: textfile; file_kind: file_kind_type);
  { output the compile time to the specified output file }
  var
    i: 1..TIME_BUFFER_LENGTH;         { index into the compile date }
begin
  if file_kind = data_file then write(f, 'TIME=''');
  write(f, ' Compilation on ');
#if VAX
  write(f, compile_date);
  write(f, ' at ');
  print_time(f, start_elapsed_time);
#endif
#if UNIX
  for i := 1 to TIME_BUFFER_LENGTH do
    if compile_date[i] in legal_chars then
      write(f, compile_date[i]);
#endif

  if file_kind = data_file then write(f, ''';');
  writeln(f);
end { post_compile_time } ;


procedure init_time_and_date(var current_elapsed_time,
                                  current_CPU_time: longint;
                              var current_date: time_string);
  { initialize the starting times and date for this compilation }
  var
    seconds: longint;               { current time in seconds }
begin
  current_elapsed_time := elapsed_time;
  current_CPU_time := CPU_time;
#if VAX
  DATE(current_date);
#endif
#if UNIX
  seconds := epochsec;  convert_time(seconds, current_date);
#endif
end { init_time_and_date } ;


{__________    error handling routines  ______________ }


procedure dump_string(var f: textfile; str: xtring);
  { dump the given string (STR) to the given file (F) as is }
  var
    i: string_range;            { index into the string }
begin
  for i := 1 to ord(str^[0]) do write(f, str^[i]);
end { dump_string } ;
    

procedure print_input_line(var f: textfile; error_num: error_range);
  { print the input parse line to the given file }
  var
    base_of_stack: stack_index_range;  { position in stack of base string }
    position: string_range;            { position to print pointer (^) }
    base_pos,                          { starting position of current string }
    curr_pos: string_range;            { current output string length }


  procedure output_parse_string(stack_pos: natural_number);
    { output the parse string, expanding current text macro }
    var
      start: string_range;        { start of last half of parse string }
      i: string_range;            { index into the string }
  begin
    if (stack_pos > stack_top) or (how_to_parse = PARSE_SEPARATELY) then
      begin
        dump_string(f, instring);
        curr_pos := curr_pos + ord(instring^[0]);
      end
    else
      with stack[stack_pos] do
        begin
          if stack_pos = base_of_stack then
            begin
              dump_string(f, str);
              writeln(f);  write(f, ' ');
            end;

          for i := 1 to last_pos do write(f, str^[i]);

          curr_pos := curr_pos + last_pos;
          i := last_pos+1;
          while (i < line_pos) and (str^[i] = ' ') do
            begin  write(f, ' ');  curr_pos := curr_pos + 1;  i := i+1;  end;

          base_pos := curr_pos;
          output_parse_string(stack_pos+1);

          if (state = FGOT_CHAR) and (pos < ord(str^[0])) then start := pos
                                                          else start := pos+1;
          for i := start to ord(str^[0]) do
            begin  write(f, str^[i]);  curr_pos := curr_pos + 1;  end;
        end;
  end { output_parse_string } ;


begin { print_input_line }
  curr_pos := 0;  base_pos := 0;  base_of_stack := stack_top;

  write(f, ' ');
  output_parse_string(base_of_stack);
  writeln(f);

  { LAST_SYM_POS points to last character preceding current symbol.  If the
    the linker does not understand the current symbol, the pointer should
    point to the 1st place in the symbol (e.g., expected > ), hence, need to
    use LAST_SYM_POS+1.  If READ_STATE = FINPUT, then LINE_POS points to the
    last character read in and should be used as is.  If READ_STATE = 
    FGOT_CHAR, then LINE_POS points to the character following the last char
    read in.  If, however, LINE_POS points to the last position in the string,
    use it as is. }

  if error_num IN scan_past_errors then position := last_sym_pos+1
  else
    if read_state = FINPUT then position := line_pos
    else
      if line_pos = ord(instring^[0]) then position := line_pos
      else if line_pos > 1 then position := line_pos-1
                           else position := line_pos;

  position := position {in line} + base_pos {text macro} + 1 {leading space};
  if position > 1 then writeln(f, error_position_char:position)
                  else writeln(f, ' ', error_position_char);
end { print_input_line } ;

function alpha_length(name: alpha): id_range;
  { find the length of an identifier by scanning for its end }
  var
    i: id_range;         { index into the identifier }
    done: boolean;       { TRUE when end of alpha found }
begin
  i := ID_LENGTH;  done := FALSE;
  while (i > 1) and not done do
    if name[i] <> ' ' then done := TRUE else i := i - 1;

  alpha_length := i;
end { alpha_length } ;


procedure writealpha(var f: textfile; name: alpha);
  { write an identifier to the output file removing trailing blanks }
begin
  write(f, name:alpha_length(name));
end { writealpha } ;


procedure print_string(var f: textfile; str: xtring);
  { print the given string (STR) to the given file (F) }
  var
    hack: string_hack;
begin
  if ord(str^[0]) > 0 then
    begin
      hack.i := ord(str) + 1;  write(f, hack.s^:ord(str^[0]));
    end;
end { print_string } ;


procedure error_dump_CRLF;
  { print the CRLF to the appropriate files }
begin
  if ok_to_print_error then writeln(monitor);
end { error_dump_CRLF } ;


procedure error_dump_indent(indentation: natural_number);
  { outputs a specified number of spaces to the error files }
begin
  if ok_to_print_error then write(monitor, ' ':indentation);
end { error_dump_indent } ;


procedure error_dump_integer(int: longint);
  { print the integer on the error files }
begin
  if ok_to_print_error then write(monitor, int:1);
end { error_dump_integer } ;


procedure error_dump_line(data: error_message);
  { indented and CRLFed without trailing blanks (but at least one char will
    be written). }
  var
    len: 1..ERROR_MESSAGE_LENGTH;  
    done: boolean;
begin
  error_dump_indent(indent);
  len := ERROR_MESSAGE_LENGTH;  done := FALSE;
  while (len > 1) and not done do
    if data[len] = ' ' then len := len - 1
                       else done := TRUE;
  if ok_to_print_error then writeln(monitor,data:len);
end { error_dump_line } ;


procedure error_dump_string(str: xtring);
  { print the given string to the error files }
begin
  if ok_to_print_error then print_string(monitor, str);
end { error_dump_string } ;


procedure error_dump_alpha(data: alpha);
  { print an alpha to the error files }
begin
  if ok_to_print_error then writealpha(monitor,data);
end { error_dump_alpha } ;


procedure error_dump_file_name(name: xtring);
  { dump the name of the file to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('File name=      ');
  error_dump_string(name);
  error_dump_CRLF;
end { error_dump_file_name } ;


procedure error_dump_alpha_file_name(file_name: alpha);
  { dump the name of the file to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('File name=      ');
  error_dump_alpha(file_name);
  error_dump_CRLF;
end { error_dump_alpha_file_name } ;

procedure display_error_summaries;
  { display error information for the entire compile on each output file }


  procedure display_error_message(var f: textfile);
    { display the error total with attention to number }
  begin
    if num_errors = 0 then
      writeln(f, ' No errors detected')
    else if num_errors = 1 then
      writeln(f, ' 1 error detected')
    else
      writeln(f, ' ', num_errors:1, ' errors detected');
  end { display_error_message } ;


  procedure display_warning_message(var f: textfile);
    { display the warning total with attention to number }
  begin
    if num_warnings = 0 then
      writeln(f, ' No warnings detected')
    else if num_warnings = 1 then
      writeln(f, ' 1 warning detected')
    else
      writeln(f, ' ', num_warnings:1, ' warnings detected');
  end { display_warning_message } ;


  procedure display_oversight_message(var f: textfile);
    { display the oversight total with attention to number }
  begin
    if num_oversights = 0 then
      writeln(f, ' No oversights detected')
    else if num_oversights = 1 then
      writeln(f, ' 1 oversight detected')
    else
      writeln(f, ' ', num_oversights:1, ' oversights detected');
  end { display_oversight_message } ;


  procedure display_baseball_message(var f: textfile);
    { display a Baseball message if appropriate }
  begin
    if produce_amusing_messages then
      if (num_errors = 0) and (num_oversights = 0) and (num_warnings = 0) then
        writeln(f, ' No men left on base')
      else if num_errors >= max_errors then
        writeln(f, ' Extreme bogosity in this design')
      else if num_errors = 1 then
        writeln(f, ' 1 error?  Close, but no cigar');
  end { display_baseball_message } ;


begin { display_error_summaries }

  writeln(monitor);
  display_error_message(monitor);
  display_oversight_message(monitor);
  display_warning_message(monitor);
  display_baseball_message(monitor);


end { display_error_summaries } ;


procedure error(error_num: error_range);
  { write out offending line and display error }


  procedure dump_error_message(error_num: error_range);
    { display the error message to the appropriate files }


    procedure write_error(var f: textfile; num: error_range);
      { write the error message (and possibly the parse string) to the file }
    begin
      writeln(f);
      if (num IN parse_errors) then print_input_line(f, num);
      writeln(f, ' #', num_errors:1, ' ERROR(', num:1, '): ',
              error_strings[num]);
    end { write_error } ;


    procedure write_oversight(var f: textfile; num: error_range);
      { write the oversight message (and possibly the parse string) to file }
    begin
      writeln(f);
      if (num IN parse_errors) then print_input_line(f, num);
      writeln(f, ' #', num_oversights:1, ' OVERSIGHT(', num:1, '): ',
              error_strings[num]);
    end { write_oversight } ;

    
    procedure write_warning(var f: textfile; num: error_range);
      { write the warning message (and possibly the parse string) to the file }
    begin
      writeln(f);
      if (num IN parse_errors) then print_input_line(f, num);
      writeln(f, ' #', num_warnings:1, ' WARNING(', num:1, '): ',
              error_strings[num]);
    end { write_warning } ;

    
  begin { dump_error_message }
    if (error_num IN warning_errors) then
      begin
        num_warnings := num_warnings + 1;
        if display_warnings and not (error_num IN suppress_errors) then
          begin
            errors_encountered := errors_encountered + [error_num];
            write_warning(monitor, error_num);
          end;
      end
    else if (error_num IN oversight_errors) then
      begin
        num_oversights := num_oversights + 1;
        if display_oversights and not (error_num IN suppress_errors) then
          begin
            errors_encountered := errors_encountered + [error_num];
            write_oversight(monitor, error_num);
          end;
      end
    else
      begin
        errors_encountered := errors_encountered + [error_num];
        num_errors := num_errors + 1;

        write_error(monitor, error_num);


        if (error_num IN echo_to_monitor_errors) then
          write_error(monitor, error_num);
      end;
  end { dump_error_message } ;



begin { error }
  dump_error_message(error_num);


  if num_errors >= max_errors then { abort the compilation }
    begin
      dump_error_message(208 { error overflow });
      display_error_summaries;
      exec_time(start_elapsed_time, start_CPU_time, FALSE);
      halt_with_status(FATAL_COMPLETION);
    end;
  last_error := error_num;
  ok_to_print_error := not(error_num IN suppress_errors) and
                   (not(error_num IN warning_errors) or display_warnings) and
                   (not(error_num IN oversight_errors) or display_oversights);
end { error } ;
    
procedure error_dump_ioresult(iores: integer);
  { outputs an ioresult message -- UNIX only }
begin
  error_dump_indent(indent);
  if ok_to_print_error then
  write_ioresult(Monitor, iores);
  error_dump_CRLF;
end { error_dump_ioresult } ;



{ --------- string processing routines ----------- }




procedure create_a_string(var str: xtring; length: string_range);
  { Create a string on the heap of the given length.  This routine uses a
    variant record to represent strings of various lengths with one
    pointer.  First, the free lists are checked for a string of the
    appropriate length.  If none are available, a string is newed from
    the heap.  This scheme works only if the Pascal compiler creates only
    as much space as needed for a variant when the tag field is specified
    in the new.  }
  type
    size_type = (s4,s8,s12,s16,s20,s24,s28,sz32,s36,s40,s44,
                 s48,s52,s56,s60,s64,s68,s72,s76,s80,s84,s88,
                 s92,s96,s100,s120,s140,s160,s180,s200,s220,s240,s256);

    trick_ptr = ^trick_record;
    trick_record = record case size_type of
                     s4: (f4: packed array [0..3] of char);
                     s8: (f8: packed array [0..7] of char);
                     s12: (f12: packed array [0..11] of char);
                     s16: (f16: packed array [0..15] of char);
                     s20: (f20: packed array [0..19] of char);
                     s24: (f24: packed array [0..23] of char);
                     s28: (f28: packed array [0..27] of char);
                     sz32: (f32: packed array [0..31] of char);
                     s36: (f36: packed array [0..35] of char);
                     s40: (f40: packed array [0..39] of char);
                     s44: (f44: packed array [0..43] of char);
                     s48: (f48: packed array [0..47] of char);
                     s52: (f52: packed array [0..51] of char);
                     s56: (f56: packed array [0..55] of char);
                     s60: (f60: packed array [0..59] of char);
                     s64: (f64: packed array [0..63] of char);
                     s68: (f68: packed array [0..67] of char);
                     s72: (f72: packed array [0..71] of char);
                     s76: (f76: packed array [0..75] of char);
                     s80: (f80: packed array [0..79] of char);
                     s84: (f84: packed array [0..83] of char);
                     s88: (f88: packed array [0..87] of char);
                     s92: (f92: packed array [0..91] of char);
                     s96: (f96: packed array [0..95] of char);
                     s100: (f100: packed array [0..99] of char);
                     s120: (f120: packed array [0..119] of char);
                     s140: (f140: packed array [0..139] of char);
                     s160: (f160: packed array [0..159] of char);
                     s180: (f180: packed array [0..179] of char);
                     s200: (f200: packed array [0..199] of char);
                     s220: (f220: packed array [0..219] of char);
                     s240: (f240: packed array [0..239] of char);
                     s256: (f256: packed array [0..255] of char);
                    end;
var
  k: record case boolean of      { "trick" record to fiddle with pointers }
       TRUE:  (tp: trick_ptr);
       FALSE: (ap: xtring);
     end;
  p: trick_ptr;                  { pointer to the created string }
  fp: freeptr;                   { pointer to head of free strings }
  size: 1..33;                   { the size (index into table) of string }

begin
  if length > 100 then size := ((length+1)+420) DIV 20
                  else size := ((length+1) DIV 4) + 1;
  if free_strings[size] <> NIL then
    begin
      str := free_strings[size]^.str;
      fp := free_strings[size]^.next;
      free_strings[size]^.next := free_pointers;
      free_pointers := free_strings[size];
      free_strings[size] := fp;
    end
  else
    begin
      case s_length[size] of
          4: new(p,s4);
          8: new(p,s8);
         12: new(p,s12);
         16: new(p,s16);
         20: new(p,s20);
         24: new(p,s24);
         28: new(p,s28);
         32: new(p,sz32);
         36: new(p,s36);
         40: new(p,s40);
         44: new(p,s44);
         48: new(p,s48);
         52: new(p,s52);
         56: new(p,s56);
         60: new(p,s60);
         64: new(p,s64);
         68: new(p,s68);
         72: new(p,s72);
         76: new(p,s76);
         80: new(p,s80);
         84: new(p,s84);
         88: new(p,s88);
         92: new(p,s92);
         96: new(p,s96);
        100: new(p,s100);
        120: new(p,s120);
        140: new(p,s140);
        160: new(p,s160);
        180: new(p,s180);
        200: new(p,s200);
        220: new(p,s220);
        240: new(p,s240);
        256: new(p,s256);
      end;
      k.tp := p;  str := k.ap;
    end;
  str^[0] := chr(length);
end { create_a_string } ;


procedure new_free_element(var f: freeptr);
  { create a new free element for released strings }
begin
  new(f);
  f^.next := NIL;
  f^.str := NIL;
end { new_free_element } ;





procedure release_string(var str: xtring);
  { free the storage used by the given string and place on free list }
  var
    size: string_range;     { size (index into table) of the string }
    f: freeptr;             { head of list of free strings }
begin
  if str <> nullstring then
    begin
      if ord(str^[0]) > 100 then size := (ord(str^[0])+420) DIV 20
                            else size := (ord(str^[0]) DIV 4) + 1;
      if free_pointers = NIL then new_free_element(f)
      else
        begin f := free_pointers; free_pointers := free_pointers^.next; end;
      f^.next := free_strings[size];
      free_strings[size] := f;  f^.str := str;
      str := nullstring;
    end;
end { release_string } ;


procedure copy_string(source: xtring;  var dest: xtring);
  { copy from the source to the destination.  The destination string must 
    exist (= nullstring or some other string).  If the source length is not
    equal to the destination length the destination string is "free"d and a
    new string of the proper size is created. }
  var
    pos: string_range;        { index into string for copy }
begin
  if source^[0] <> dest^[0] then
    begin
      release_string(dest);  create_a_string(dest, ord(source^[0]));
    end;

  for pos := 1 to ord(source^[0]) do  dest^[pos] := source^[pos];
end { copy_string } ;

function compare_strings(s1, s2: xtring): compare_type;
  { compare the strings and return the result }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    result: compare_type;   { result of the comparison }
    still_equal: boolean;   { TRUE if strings are equal to current position }
begin
  if s1^[0] = s2^[0] then
    begin  min_length := ord(s1^[0]);  result := EQ;  end
  else if s1^[0] < s2^[0] then
    begin  min_length := ord(s1^[0]);  result := LT;  end
  else 
    begin  min_length := ord(s2^[0]);  result := GT;  end;

  i := 0;  still_equal := TRUE;
  while (i < min_length) and still_equal do
    begin
      i := i + 1;
      if s1^[i] < s2^[i] then
        begin  result := LT;  still_equal := FALSE;  end
      else if s1^[i] > s2^[i] then
        begin  result := GT;  still_equal := FALSE;  end;
    end;

  compare_strings := result;
end { compare_strings } ;

  
function enter_string(str: xtring): xtring;
  { enter or find the string in the string hash table and return it.
    If not found in the table, the string's value is copied to a new
    string created afresh. }
  var
    i: string_range;           { index into STR }
    sum: natural_number;       { checksum of the string }
    index: hash_string_range;  { index into the string table }
    last,                      { last element checked in list }
    element: hash_string_ptr;  { element in the list of names }
    compare: compare_type;     { result of string compare }
    done: boolean;             { TRUE when place in table found }


  procedure insert_entry(list_element: hash_string_ptr);
    { insert a new entry after the given list element.  If the list element
      is NIL, insert at the head of the list. }
    var
      new_element: hash_string_ptr; { new element to be placed into the list }
  begin
    new(new_element);

    new_element^.str := nullstring;
    copy_string(str, new_element^.str);
    if list_element = NIL then
      begin
        new_element^.next_hash_string := string_table[index];
        string_table[index] := new_element;
      end
    else
      begin
        new_element^.next_hash_string := list_element^.next_hash_string;
        list_element^.next_hash_string := new_element;
      end;

    enter_string := new_element^.str;
  end { insert_entry } ;


begin { enter_string }
  if ord(str^[0]) = 0 then enter_string := nullstring
  else
    begin
      { create a hash index from the specified name }

      sum := 0;  i := 0;
      for i := 1 to ord(str^[0]) do
         sum := sum + ord(str^[i]);

      index := sum MOD (HASH_STRING_TABLE_SIZE+1);

      element := string_table[index];
      if element = NIL then insert_entry(NIL)
      else
        begin
          last := NIL;  done := FALSE;
          repeat
            if element^.next_hash_string = NIL then
              begin
                done := TRUE;
                compare := compare_strings(str, element^.str);
              end
            else
              begin
                compare := compare_strings(str, element^.str);
                if compare <> GT then done := TRUE
                else
                  begin
                    last := element;  element := element^.next_hash_string;
                  end;
              end;
          until done;

          case compare of
            LT:  insert_entry(last);
            EQ:  enter_string := element^.str;
            GT:  insert_entry(element);
          end;
        end;
    end;
end { enter_string } ;

procedure copy_to_string(name: alpha;  var str: xtring);
  { copy from an alpha to a string.  Trailing blanks are deleted. }
  var
    len: id_range;    { length of the identifer }
    i: id_range;      { index into alpha and string for copy }
begin
  len := alpha_length(name);

  if ord(str^[0]) <> len then 
    begin  release_string(str);  create_a_string(str, len);  end;
  for i := 1 to len do  str^[i] := name[i];
end { copy_to_string } ;


function enter_and_release_string(str: xtring): xtring;
  { enter the given string, and release the original pointer }
  var
    new_string: xtring;    { string from the table }
begin
  new_string := enter_string(str);

  if new_string <> str then
    release_string(str);

  enter_and_release_string := new_string;
end { enter_and_release_string } ;

function make_and_enter_string(name: alpha): xtring;
  { convert an alpha into a string and enter it into the hash table }
  var
    temp_string: xtring;        { temporary string }
    original_string: xtring;    { original string value }
begin
  temp_string := nullstring;
  copy_to_string(name, temp_string);

  original_string := temp_string;
  temp_string := enter_string(temp_string);

  if temp_string <> original_string then
    release_string(original_string);

  make_and_enter_string := temp_string;
end { make_and_enter_string } ;

function enter_name(name: alpha): name_ptr;
  { enter or find the name in the name hash table and return a pointer }
  var
    i: 0..ID_LENGTH;           { index into NAME }
    sum: natural_number;       { checksum of the name }
    index: name_table_range;   { index into the name table }
    last,                      { last element checked in list }
    element: name_ptr;         { element in the list of names }
    done: boolean;             { TRUE when end of alpha found }


  procedure insert_entry(list_element: name_ptr);
    { insert a new entry after the given list element.  If the list element
      is NIL, insert at the head of the list. }
    var
      new_element: name_ptr;  { new element to be placed into the list }
  begin
    new(new_element);

    new_element^.name := name;
    new_element^.kind := default_attributes;
    new_element^.definition := nullstring;
    new_element^.sy := NULLSY;

    if list_element = NIL then
      begin
        new_element^.next := name_table[index];
        name_table[index] := new_element;
      end
    else
      begin
        new_element^.next := list_element^.next;
        list_element^.next := new_element;
      end;

    enter_name := new_element;
  end { insert_entry } ;


begin { enter_name }
  { create a hash index from the specified name }

  sum := 0;  i := 0;  done := FALSE;
  while (i < ID_LENGTH) and not done do
    begin
      i := i + 1;
      if name[i] = ' ' then done := TRUE else sum := sum + ord(name[i]);
    end;
  index := sum MOD (name_table_size+1);

  element := name_table[index];
  if element = NIL then insert_entry(NIL)
  else
    begin
      last := NIL;
      while (name > element^.name) and (element^.next <> NIL) do
        begin  last := element;  element := element^.next;  end;
      if name = element^.name then enter_name := element
      else if name < element^.name then insert_entry(last)
      else insert_entry(element);
    end;
end { enter_name } ;


procedure fix_parse_stack;
  { remove wasted "virtually" popped signals from the top of the stack. 
    This is used by insymbol to clean up before returning. When done,
    parse_stack_pointer = stack_top OR parse_stack_pointer = stack_top + 1. }
begin
  if parse_stack_pointer <= 0 then 
    begin
      error(187 { assertion} );
      parse_stack_pointer := 1;
    end;
  while (stack_top > parse_stack_pointer) do
    begin
      error(187 { assertion} );
      stack_top := stack_top - 1;
    end;
end { fix_parse_stack } ;

