#ifndef HEAPH
#define HEAPH
/*
    Multiple heap memory allocator.  
    Currently this is implemented as calls to the ValidLINKER allocator,
    implemented in C.
*/


#ifndef BASICTYPES_H
#include "basictypes.h"
#endif
#ifndef FAULTH
#include "fault.h"
#endif
#include <string.h>

// ValidLINKER allocator entry points.
#include "lib/mem/mem.h"

class mainarena {
    friend class heap;
    struct main_arena *marena;

    void establish() { if (!marena) mem_main_arena(&marena); }
public:
    mainarena() { marena = 0; }
    void pool() { if (marena) mem_pool_main_arena(&marena); }
};
    

class heap {
    arena *arena;
    mainarena *marena;
    void *offset(void *p, int n) { return (void *)((char *)p + (n)); }
    
public:
    // Leaving the primary arena 0 results in allocation from a common primary arena
    heap(mainarena *m = 0) { arena = 0;  marena = m; }
    ~heap() { pool(); }
    //
    // establish() and pool() are virtual so that derived classes can be made
    // that establish and destroy other structures related to a heap whenever
    // the heap is establishd and pooled.
    virtual void establish(); // allocates the arena space
    virtual void pool();  // release the entire heap
private:
    void check() { if (!arena) establish(); }

public:
    /*
     * NOTE: Do not mix the following 2 groups of calls -- if an object
     * is allocated with a "mem" function, it must be freed with a memfree
     * call (or cmemfree call if it was allocated by cmem with count > 1).
     * If an object is allocated by a "malloc" call it must be freed by
     * using free.
     *
     * The "mem" group.  Again note that within this group cmemfree
     * should always be used when memory has been allocated by cmem
     * with a count > 1 (It is also safe to use cmemfree on space
     * allocated by mem -- just set the count to 1).  memfree can
     * be used on space allocated by mem (or rmem) or by cmem with
     * a count of 1.
     */

    void *mem(unsigned size)
	{ check();  return ::mem(arena, size); }
    void *rmem(void *ptr, unsigned oldsize, unsigned newsize)
	{ AS(arena);  return ::rmem(arena, ptr, oldsize, newsize); }
    void memfree(void *ptr, unsigned size)
	{ AS(arena);  ::memfree(arena, ptr, size); }

    void *cmem(unsigned count, unsigned size)
	{ check();  return ::cmem(arena, count, size); }
    void cmemfree(void *ptr, unsigned count, unsigned size)
        { AS(arena);  ::cmemfree(arena, ptr, count, size); }

    char* strcpy(const char* s);  // allocates strlen(s) +1 and copies
    void strfree(char* s) { if (s) memfree(s, strlen(s) + 1); }

    /*
     * The "malloc" group -- these must be similarly used together.
     */
    void *malloc(unsigned size)
    {
	check();  
	unsigned *p = (unsigned *)::mem(arena, size + 4);
	*p = size + 4;
	return offset(p, 4);
    }
	
    void *realloc(void *ptr, unsigned size)
    {
	AS(arena); 
        unsigned *p = (unsigned *)offset(ptr, -4);
	p = (unsigned *)::rmem(arena, p, *p, size + 4);
	*p = size + 4;
	return offset(p, 4);
    }
	
    void free(void *ptr)
    {
	AS(arena); 
        unsigned *p = (unsigned *)offset(ptr, -4);
	::memfree(arena, p, *p);
    }

    void *calloc(unsigned count, unsigned size);
};


#define NULL_HEAP (*(heap*)0)
#endif HEAPH
