#ifndef AVLBASEH
#define AVLBASEH
// Class avl -- overhead fields for manipulating AVL (balanced binary) tree.
// This class is used to derive other classes which have different kinds of
// keys.  There are no functions here that are callable by the final end
// user -- all are for use in implementing the derived classes only.

// Assumes that pointers are 32 bits, with at most a 30 bit address
// space (using the low order bits).  The 2 MSBs are used to store the
// balance factor.


#include "fault.h"


const AVL_BALANCE_BITS = 0x1;
enum avl_balance { AVL_UNDEF=0x0,
                   AVL_LEFT=0x1,
		   AVL_RIGHT=0x2,
		   AVL_EVEN=0x3,
		 };

class avlpath;

class avlbase {
    avlbase *l;
    avlbase *r;

    avl_balance left_on() { return ((long)l & AVL_BALANCE_BITS) ? AVL_LEFT : 0; }
    avl_balance right_on() { return ((long)r & AVL_BALANCE_BITS) ? AVL_RIGHT : 0; }
    avl_balance balance()
    {
	return left_on() | right_on();
    }
    void set_balance(avl_balance b=AVL_UNDEF)
    { 
	l = (avlbase *)((b & AVL_LEFT) | ((int)l & ~AVL_BALANCE_BITS));
	r = (avlbase *)((b & AVL_RIGHT) | ((int)r & ~AVL_BALANCE_BITS));
    }
    void set_left(avlbase *val)
        { l = (avlbase *)(balance() | ((int)val & ~AVL_BALANCE_BITS)); }
    void set_right(avlbase *val)
        { r = (avlbase *)(balance() | ((int)val & ~AVL_BALANCE_BITS)); }

public:
    avlbase() { l = r = 0; }
    void operator=(avlbase&) { fault("avlbase"); } // meaningless

    avlbase* left()
        { return (avlbase*)((int)l & ~AVL_BALANCE_BITS); }
    avlbase* right()
        { return (avlbase*)((int)r & ~AVL_BALANCE_BITS); }
    void insert(avlbase**, avlpath&);
    void pick(avlbase**, avlpath&);
    int in_tree()
        { return balance() != AVL_UNDEF; }
};


class avl_traversal_stack {
    avlbase *stack[32];
    avlbase **top;	// points to last pushed element
#ifdef DEBUGD
    avlbase **limit;	// points to last legal element of stack;
#endif DEBUGD
protected:
    avlbase *tree;	// root of tree being iterated
    void clear()
        { top = stack - 1; }

    void push(avlbase *next)
    {
#ifdef DEBUGD
	if (top == limit) fault("ovf");
#endif DEBUGD
	*++top = next;
    }

    avlbase *pop()
    {
	return top < stack ? 0 : *top--;
    }
public:
    avl_traversal_stack()
    {
	tree = 0;
	clear();
#ifdef DEBUGD
	limit = stack + 31;
#endif DEBUGD
    }

};


class avl_preorder : public avl_traversal_stack {
    void init(avlbase *);
public:
    avl_preorder(avlbase *root = 0)
        { init(root); }
    avlbase *next();
    avlbase *first(avlbase *root = 0)
        { init(root);  return next(); }
};


class avl_inorder : public avl_traversal_stack {
    void init(avlbase *);
public:
    avl_inorder(avlbase *root = 0)
        { init(root); }
    avlbase *next();
    avlbase *first(avlbase *root = 0)
        { init(root);  return next(); }
};


class avlpath_itr;
class avlpath {
    friend avlpath_itr; // defined and used only in avlbase.c

    int path[32];	// negative -> go left, 0 stop, positive -> go right
    int *top;		// points to last pushed element
#ifdef DEBUGD
    int *limit;		// last legal entry of path
#endif DEBUGD

public:
    void operator=(avlpath&)
	{ fault("avlpath"); }

    avlpath()
    {
#ifdef DEBUGD
        limit = path + 31; 
#endif DEBUGD
	top = path - 1;
    }

    void push(int direction)
    {
#ifdef DEBUGD
	if (top == limit) fault("ovf");
#endif DEBUGD
	*++top = direction;
    }
};


#endif AVLBASE_INCL
