#ifndef AVLBASEH
#define AVLBASEH
// Class avl -- overhead fields for manipulating AVL (balanced binary) tree.
// This class is used to derive other classes which have different kinds of
// keys.  There are no functions here that are callable by the final end
// user -- all are for use in implementing the derived classes only.

// Assumes that pointers are 32 bits, with at most a 30 bit address
// space (using the low order bits).  The 2 MSBs are used to store the
// balance factor.


#include "fault.h"
#include "giterator.h"

class avlpath;

#define USE_2_MSBS 1 /* Most of our platforms can use this */
#if convex
#undef USE_2_MSBS
#define USE_1_LSB 1
#endif


#ifdef USE_2_MSBS
// Machines that use <= 30 bit pointers

const AVL_BALANCE_BITS = 0xC0000000;
enum avl_balance { AVL_UNDEF =0, 
                   AVL_LEFT =0x80000000, 
		   AVL_RIGHT=0x40000000,
		   AVL_EVEN=0xC0000000 };


class avlbase {
    friend class avlroot_base;
    avlbase *l;
    avlbase *r;

    avl_balance balance()
        { return (int)l & AVL_BALANCE_BITS; }
    void set_balance(avl_balance b=0)
        { l = (avlbase *)(b | ((int)l & ~AVL_BALANCE_BITS)); }
    void set_left(avlbase *val)
    { 
	AS(((int)val & ~AVL_BALANCE_BITS) == (int)val);
	l = (avlbase *)(balance() | ((int)val & ~AVL_BALANCE_BITS));
    }
    void set_right(avlbase *val)
        { r = val; }
    void init() { l = r = 0; }
public:
    avlbase() { init(); }
    void operator=(avlbase&) { fault("avlbase"); } // meaningless

    avlbase* left()
        { return (avlbase*)((int)l & ~AVL_BALANCE_BITS); }
    avlbase* right()
        { return r; }
    int in_tree()
        { return balance() != AVL_UNDEF; }
};


#endif
#ifdef USE_1_LSB
// Machines that use word-aligned 32 bit pointers

const AVL_BALANCE_BITS = 0x1;
enum avl_balance { AVL_UNDEF=0x0,
                   AVL_LEFT=0x1,
		   AVL_RIGHT=0x2,
		   AVL_EVEN=0x3,
		 };

class avlbase {
    friend avlroot_base;
    avlbase *l;
    avlbase *r;

    avlbase* ptr(avlbase* p)
        { return (avlbase*)((int)p & ~AVL_BALANCE_BITS); }
    int bal(avlbase* p) { return (int)p & AVL_BALANCE_BITS; }
    avl_balance left_on() { return bal(l) ? AVL_LEFT : 0; }
    avl_balance right_on() { return bal(r) ? AVL_RIGHT : 0; }
    avl_balance balance()
    {
	return left_on() | right_on();
    }
    void set_balance(avl_balance b=AVL_UNDEF)
    { 
	l = (avlbase *)(((b & AVL_LEFT) ? AVL_BALANCE_BITS : 0) | (int)ptr(l));
	r = (avlbase *)(((b & AVL_RIGHT) ? AVL_BALANCE_BITS : 0) | (int)ptr(r));
    }
    void set_left(avlbase *val)
	{ l = (avlbase *)(bal(l) | (int)ptr(val)); }
    void set_right(avlbase *val)
	{ r = (avlbase *)(bal(r) | (int)ptr(val)); }

    void init() { l = r = 0; }
public:
    avlbase() { init(); }
    void operator=(avlbase&) { fault("avlbase"); } // meaningless

    avlbase* left()
	{ return ptr(l); }
    avlbase* right()
	{ return ptr(r); }
    int in_tree()
        { return balance() != AVL_UNDEF; }
};
#endif


class avlroot_base : public collection_base {
    friend class avl_traversal_stack;
    avlbase* root;
protected:
    avlbase* tree_root() { return root; }
public:
    avlroot_base() { root = 0; }
    void insert(avlbase*, const avlpath&);
    void pick(avlbase*, const avlpath&);
    unsigned long num_entries();
    void clear() { root = 0; }
};


class avl_traversal_stack : public iterator_base {
    avlbase *stack[32];
    avlbase **top;	// points to last pushed element
#ifdef DEBUGD
    avlbase **limit;	// points to last legal element of stack;
#endif DEBUGD
protected:
    void clear()
        { top = stack - 1; }

    avlbase* tree_root() { return ((avlroot_base*)&collection())->tree_root(); }

    void push(avlbase *next)
    {
#ifdef DEBUGD
	if (top == limit) fault("ovf");
#endif DEBUGD
	*++top = next;
    }

    avlbase *pop()
    {
	return top < stack ? 0 : *top--;
    }
public:
    avl_traversal_stack(const avlroot_base& b) : (b)
    {
	clear();
#ifdef DEBUGD
	limit = stack + 31;
#endif DEBUGD
    }

};


class avl_preorder : public avl_traversal_stack {
public:
    void init();
    avl_preorder(const avlroot_base& b) : (b)
        { init(); }
    avlbase* operator()();
    
    avlbase *next() { return (*this)(); }
    avlbase *first() { init();  return (*this)(); }
};


class avl_inorder : public avl_traversal_stack {
public:
    /* GITERATOR */
    void init();
    avl_inorder(const avlroot_base& b) : (b)
        { init(); }
    avlbase* operator()();

    /* backward compatibility */
    avlbase *next() { return (*this)(); }
    avlbase *first() { init();  return (*this)(); }
};


class avlpath_itr;
class avlpath {
    friend avlpath_itr; // defined and used only in avlbase.c

    int path[32];	// negative -> go left, 0 stop, positive -> go right
    int *top;		// points to last pushed element
#ifdef DEBUGD
    int *limit;		// last legal entry of path
#endif DEBUGD

public:
    void operator=(avlpath&)
	{ fault("avlpath"); }

    avlpath()
    {
#ifdef DEBUGD
        limit = path + 31; 
#endif DEBUGD
	top = path - 1;
    }

    void push(int direction)
    {
#ifdef DEBUGD
	if (top == limit) fault("ovf");
#endif DEBUGD
	*++top = direction;
    }
};


#endif AVLBASE_INCL
