/* ShellRegExp.c -- implementation of Shell regular expressions */

#include "Object.hxx"
#include "String.hxx"
#include "RegExp.hxx"
#include "ShellRegExp.hxx"

#define	THIS	ShellRegExp
#define	BASE	RegExp

DEFINE_CLASS(ShellRegExp,Object,1,NULL,NULL);

String MATCHALL("*");

String
ShellRegExp::trShellExp(const String& shellExp) {
    /*
     *--------------------------------------------------------------------------
     * Translate Shell Regular Expression to that of "ex", used by 
     * re_comp() && re_exec.
     * This routines owes much to CompileRegExp(pattern) from IC group.
     *
     * Table 5.2 Shell Pattern matching rules 
     * (p. 137 The Unix Programming Env, Kernighan && Pike)
     *
     * Shell (p.137)   	meaning				RegExp equiv. (p. 105)
     *--------------------------------------------------------------------------
     * 1. "*"		Match any string		".*"
     * 2. "?"		Match any single char   	"."
     * 3. [ccc]		Match any chars in ccc  	same
     * 4. "..."		Match exactly,	        	same
     *                  quotes special chars.
     *                  equiv to '...'
     * 5. \c		Match c literally		same
     * 6. a|b		In "case expressions" only.    	same
     *                  matches either a or b
     * 7. /		In filenames, matched only      ??? - do nothing,
     *                  by an explicit '/' in the             match like any other.
     *                  expression.
     *                  in "case expressions", matched
     *                  like any other character.
     * 8. '.'		as the first character in a     '\.' - turn off special 
     *                  filename, is matched only by an        meaning.
     *                  explicit '.' in the expression.
     *                  I think we're ok on this one.
     *                  ./abc is anchored by '^', so to
     *                  force an explicit match.
     *--------------------------------------------------------------------------
     */
    String munged("^");  	// anchor first char with '^'
    bool firstChar = YES;
    int i = 0;
    while (i < shellExp.size()) {
	switch (shellExp[i]) {
	    /*
	     * The following special SHELL meta characters must be translated
	     * to their RegExp equivalents.
	     */
	    case '\\': {
		/*
		 * We need to handle this as if it were different -
		 * otherwise we might translate escaped chars.
		 */
		if (i < (shellExp.size() - 1)) { // i is not last char
		    munged &= shellExp(i, 2);    // give it the escaped chr - 
		    i++; 			 // compenstate for extra char.
	        } else {
		    munged &= shellExp(i, 1);    // last char.
	        }
		break;
	    }
	    case '*': {
		munged &= ".*";
		break;
	    }
	    case '?': {
		munged &= ".";
		break;
	    }
	    case '.': {
		munged &= "\\.";
		break;
	    }
	   /*
	    * The following characters have NO special meaning to the Shell,
	    * but they will to RegExp, so turn off their special meaning.
	    */
	    case '^': {
		munged &= "\\^";
		break;
	    }
	    case '$': {
		munged &= "\\$";
		break;
	    }
	    case '+': {
		munged &= "\\+";
		break;
	    }
	    default: {
	       munged &= shellExp(i, 1);
	    }
	}
	firstChar = NO;
	i++; // skip past current char.
    }
    munged &= "$";	// anchor last char, to match literals.
#ifdef DEBUGSHELLREGEXP
    cout << "trShellEuxp.shellExp: " << shellExp << " => " << munged << "\n";
#endif DEBUGSHELLREGEXP
    return munged;
}


ShellRegExp::ShellRegExp(istream& strm, ShellRegExp& where) : (strm, where) {
    this = &where;
}

ShellRegExp::ShellRegExp(fileDescTy& fd, ShellRegExp& where) : (fd, where) {
    this = &where;
}

void
ShellRegExp::operator=(const ShellRegExp& d) {
    BASE::operator=(d);
}

bool
ShellRegExp::operator==(const ShellRegExp& d) {
    return BASE::operator==(d);
}

bool
ShellRegExp::isEqual(const Object &Ob) {
    return Ob.isSpecies(class_ShellRegExp) && *this == *(ShellRegExp*)&Ob;
}

void
ShellRegExp::storer(ostream& strm) {
    BASE::storer(strm);
}

void
ShellRegExp::storer(fileDescTy& fd) {
    BASE::storer(fd);
}

