	page	80,132
	title	MONO10 - Monochrome INT10 Routines

;
;  MONO10 - Monochrome INT10 Routines
;
;  Routines to replace the BIOS Video Display Routines for a monochrome
;  monitor. Also includes some support for ANSI escape sequences.
;

mono_base	equ	0B000h		;Segment for monochrome display buffer
mono_6845	equ	03B4h		;Port for monochrome 6845
mono_mode	equ	7		;Mode for monochrome display
mono_enable	equ	29h		;Bits to enable monochrome display
mono_cols	equ	80		;Number of display columns
mono_regen_b	equ	4096		;Regen length in bytes
mono_regen_w	equ	2048		;Regen length in words

abs0		segment	at 0
		org	74h
parm_ptr	dd	?		;pointer to video parameter table
abs0		ends

data		segment	at 40h		;BIOS data area; maintained for
					;compatibility.
		org	49h
crt_mode	db	?
crt_cols	dw	?
crt_len		dw	?
crt_start	dw	?
cursor_posn	dw	8 dup(?)
cursor_mode	dw	?
active_page	db	?
addr_6845	dw	?
crt_mode_set	db	?
crt_palette	db	?
data		ends

code	segment
	assume	cs:code
	org	100h
start:	jmp	main

escmode	db	0			;1 if escape found
numparm	dw	0			;# of parameters so far
parmstk	db	4 dup (0)		;parameters converted to binary
byte_ten db	10			;for multiply instr
savedcur dw	0			;saved cursor position
dispattr db	07h			;display attribute
wrapmode db	0			;1 = wrap at end of line; 0 = ignore

brtab	dw	set_mode		; 0 = set mode
	dw	set_ctype		; 1 = set cursor type
	dw	set_cpos		; 2 = set cursor position
	dw	read_cursor		; 3 = read cursor position
	dw	read_lpen		; 4 = read light pen position
	dw	act_disp_page		; 5 = select active display page
	dw	scroll_up		; 6 = scroll up
	dw	scroll_down		; 7 = scroll down
	dw	read_ac_current		; 8 = read attr/char
	dw	write_ac_current	; 9 = write attr/char
	dw	write_c_current		;10 = write character only
	dw	set_color		;11 = set color palette
	dw	write_dot		;12 = write dot
	dw	read_dot		;13 = read dot
	dw	write_tty		;14 = write tty
	dw	video_state		;15 = current video state

position:
; calculate offset into buffer
; ah = row, al = col (0,0 = upper left)
; mult ah by 80 then add al; multiply result by 2
	assume	ds:data
	push	bx			;save work reg
	push	ax			;save al (column)
	mov	al,ah			;set ax to row
	mul	byte ptr crt_cols	;ax = row * 80
	pop	bx			;get column
	xor	bh,bh			;bx = column
	add	ax,bx			;add column into partial result
	shl	ax,1			;mult by 2 for byte offset
	pop	bx			;restore work reg
	ret

out_cx:
; write word value in cx to 6845 register.
; ah has 6845 register number
	mov	dx,mono_6845		;6845 port address
	mov	al,ah			;get register number
	out	dx,al			;set up for register
	inc	dx			;data address
	mov	al,ch			;output the data byte
	out	dx,al
	dec	dx			;back to register number port
	mov	al,ah			;get register number again
	inc	al			;inc for 2nd half
	out	dx,al			;set up for register
	inc	dx			;data address
	mov	al,cl			;2nd data byte
	out	dx,al
	ret

set_cursor_posn:
; ax has row/col to put cursor
	call	position		;calc buffer addr
	mov	cx,ax			;copy buffer addr
	sar	cx,1			;convert from byte to word offset
	mov	ah,14			;cursor register number
	call	out_cx			;set cursor position register
	ret

int10:					;Replacement for INT10
	sti				;allow interrupts
	cld				;forward direction
	push	es			;save regs
	push	ds
	push	dx
	push	cx
	push	bx
	push	si
	push	di
	push	ax

	cmp	ah,15			;is request code valid?
	ja	int_return1		;no: do nothing

	mov	al,ah			;copy request code
	xor	ah,ah			;ax = request code
	sal	ax,1			;get offset into branch table
	mov	si,ax			;put offset into index register

	mov	ax,40h			;segment for BIOS data area
	mov	ds,ax
	assume	ds:data

	mov	ax,mono_base		;segment for mono card
	mov	es,ax

	pop	ax			;restore value in al
	jmp	cs:brtab[si]

int_return1:
	pop	ax			;restore regs before returning
set_color:
act_disp_page:
read_lpen:
write_dot:
read_dot:
int_return:
	pop	di
	pop	si
	pop	bx
	pop	cx
	pop	dx
	pop	ds
	pop	es
	iret

set_cpos:
	cmp	bh,0			;page 0?
	jne	int_return		;no: return
	mov	cursor_posn,dx		;save cursor position
	mov	ax,dx			;yes: set up row/col
	call	set_cursor_posn		;set the cursor position
	jmp	int_return

read_cursor:
	cmp	bh,0			;page 0?
	jne	int_return		;no: return
	mov	dx,cursor_posn		;return cursor position
	mov	cx,cursor_mode		; and cursor mode
	pop	di			;restore regs
	pop	si
	pop	bx
	pop	ax			;discard saved cx and dx
	pop	ax
	pop	ds
	pop	es
	iret

set_mode:
	mov	crt_mode,mono_mode	;Display mode
	mov	dx,mono_6845		;Port addr for 6845
	mov	addr_6845,dx		;Save for compatibility
	push	ds			;Save data segment pointer
	push	dx
	add	dx,4			;Reset Video port addr
	mov	al,1			;Reset monochrome display
	out	dx,al			;Reset Video
	pop	dx			;Restore 6845 port addr
	sub	ax,ax
	mov	ds,ax			;set up for ABS0 segment
	assume	ds:abs0
	lds	bx,parm_ptr		;get video parms table addr
	assume	ds:nothing
	add	bx,30h			;get addr of monochrome parms
	mov	cx,10h			;length of table entry
; ah has register number (beginning with 0)
; cx has loop counter
out_init:
	mov	al,ah			;register number
	out	dx,al			;set 6845 register number
	inc	dx			;point to data port
	mov	al,byte ptr [bx]	;get value from table
	out	dx,al			;send data to 6845
	inc	ah			;next register
	inc	bx			;next data value
	dec	dx			;reset to set register number
	loop	out_init		;do the next value
	pop	ds			;restore data segment pointer
	assume	ds:data

; fill monochrome buffer with blanks

	xor	di,di			;pointer within buffer
	mov	crt_start,di		;save for compatibility
	mov	active_page,0		;for compatibility
	mov	cx,mono_regen_w		;monochrome buffer size
	mov	ax,0720h		;blank, normal display attribute
	rep	stosw			;clear the buffer

; enable video, etc.

	mov	cursor_mode,0B0Ch	;save cursor mode
	mov	dx,mono_6845+4		;Video Enable port
	mov	al,mono_enable		;bits to enable monochrome display
	out	dx,al			;enable the display
	mov	crt_mode_set,al		;save enable bits for compatibility
	mov	crt_cols,mono_cols	;save for compatibility
	mov	crt_len,mono_regen_b	;save regen length for compatibility
	mov	cursor_posn,0		;clear page 0 cursor position

; set up overscan register

	inc	dx			;dx = mono_6845+5
	mov	al,30h			;overscan value
	out	dx,al			;set overscan register
	mov	crt_palette,al		;save for compatibility
	jmp	int_return

set_ctype:
	mov	ah,10			;6845 register for cursor set
	mov	cursor_mode,cx		;save cursor mode in data area
	call	out_cx			;output cx register to 6845
	jmp	int_return		;return

video_state:
	mov	ah,mono_cols		;return # columns on the screen
	mov	al,mono_mode		;return video mode
	mov	bh,0			;active display page
	pop	di			;restore regs
	pop	si
	pop	cx			;discard saved bx
	pop	cx
	pop	dx
	pop	ds
	pop	es
	iret

scroll_setup:
; set up SI, DI for scroll
; on entry, AX has the upper left corner address (lower right for scroll down)
	call	position		;convert row/col in ax to buffer offset
	mov	di,ax			;set up to location
	mov	si,ax			;set up from location
	sub	dx,cx			;dx = # rows, # cols in block
	inc	dh			;adjust for zero relative
	inc	dl
	xor	ch,ch			;clear high byte of count reg
	mov	bp,mono_cols*2		;# bytes on a line
	mov	al,bl			;get # of lines to scroll
	mul	byte ptr crt_cols	;find offset to from address
	add	ax,ax			;*2 for attribute byte
	push	es			;address the display buffer
	pop	ds
	cmp	bl,0			;0 = clear the area of the screen
	ret

move_row:
; move the row from [si] to [di]
	mov	cl,dl			;# of columns to move
	push	si			;save regs
	push	di
	rep	movsw			;move the row
	pop	di			;restore regs
	pop	si
	ret

clear_row:
; clear the row at [di]
	mov	cl,dl			;# of columns to clear
	push	di			;save reg
	rep	stosw			;clear the row
	pop	di			;restore reg
	ret

clear_scr:
; clear the entire screen
; bh has the attribute
	xor	di,di			;set to start of screen
	mov	cx,mono_regen_w		;number of words on the screen
	mov	ah,bh			;attribute
	mov	al,' '			;char = blank
	rep	stosw			;clear the screen
	ret

scroll_clear:
	call	clear_scr
	jmp	int_return

scroll_up_one:
	xor	di,di			;move to upper left
	mov	si,mono_cols*2		;from second line
	mov	cx,24*mono_cols		;# of words to move
	push	es			;address display buffer
	pop	ds
	rep	movsw			;move the lines up one
	mov	cx,mono_cols		;# of words to clear
	mov	ah,bh			;attribute of blank line
	mov	al,' '
	rep	stosw			;clear the last line
	jmp	int_return

scroll_up:
; al = number of rows to scroll (0 means clear area of screen)
; cx = row/col of upper left corner
; dx = row/col of lower right corner
; bh = attribute for blank lines
; ds = BIOS data area pointer
; es = display buffer pointer

	cmp	cx,0			;from upper left...
	jne	scroll_up_010
	cmp	dx,184Fh		;..to lower right?
	jne	scroll_up_010
	cmp	al,0			;clear all screen?
	je	scroll_clear
	cmp	al,1			;scroll 1 line
	je	scroll_up_one
scroll_up_010:
	mov	bl,al			;copy line count to bl
	push	bx			;save attribute, line count
	mov	ax,cx			;copy upper left row/col
	call	scroll_setup		;set up SI,DI for scrolling
	jz	scroll_up_070		;branch if area to be cleared
	add	si,ax			;calculate from address
	mov	ah,dh			;# of rows in the block
	sub	ah,bl			;# of rows to be moved
scroll_up_020:
	call	move_row		;move 1 row
	add	si,bp			;incr for next row
	add	di,bp
	dec	ah
	jnz	scroll_up_020		;do the next row
; clear the rows
scroll_up_030:
	pop	ax			;get attribute into ah
	mov	al,' '			;set to clear the row
scroll_up_040:
	call	clear_row		;clear the row
	add	di,bp			;incr for the next row
	dec	bl			;counter of lines to scroll
	jnz	scroll_up_040		;do the next row
	jmp	int_return
scroll_up_070:
	mov	bl,dh			;number of rows to clear
	jmp	scroll_up_030		;clear the area

scroll_down:
; al = number of rows to scroll (0 means clear area of screen)
; cx = row/col of upper left corner
; dx = row/col of lower right corner
; bh = attribute for blank lines
; ds = BIOS data area pointer
; es = display buffer pointer

	std				;reverse direction of moves
	mov	bl,al			;copy line count to bl
	push	bx			;save attribute, line count
	mov	ax,dx			;copy lower right row/col
	call	scroll_setup		;set up SI,DI for scrolling
	jz	scroll_down_070		;branch if area to be cleared
	sub	si,ax			;calculate from address
	mov	ah,dh			;# of rows in the block
	sub	ah,bl			;# of rows to be moved
scroll_down_020:
	call	move_row		;move 1 row
	sub	si,bp			;decr for next row
	sub	di,bp
	dec	ah
	jnz	scroll_down_020		;do the next row
; clear the rows
scroll_down_030:
	pop	ax			;get attribute into ah
	mov	al,' '			;set to clear the row
scroll_down_040:
	call	clear_row		;clear the row
	sub	di,bp			;decr for the next row
	dec	bl			;counter of lines to scroll
	jnz	scroll_down_040		;do the next row
	jmp	int_return
scroll_down_070:
	mov	bl,dh			;number of rows to clear
	jmp	scroll_down_030		;clear the area

read_ac_current:
	mov	ax,cursor_posn		;get page 0 cursor position
	call	position		;calc buffer offset
	mov	si,ax			;copy buffer offset into index reg
	push	es			;address the display buffer
	pop	ds
	lodsw				;get the char/attr
	jmp	int_return

write_ac_current:
	push	ax			;save char to write (al)
	mov	ax,cursor_posn		;get page 0 cursor position
	call	position		;calc buffer offset
	mov	di,ax			;copy buffer offset into dest index
	pop	ax			;al = char to write
	mov	ah,bl			;ah = attribute
	rep	stosw			;store char/attr in display buffer
	jmp	int_return

write_c_current:
	push	ax			;save char to write (al)
	mov	ax,cursor_posn		;get page 0 cursor position
	call	position		;calc buffer offset
	mov	di,ax			;copy buffer offset into dest index
	pop	ax			;al = char to write
write_c_loop:
	stosb				;store char only in display buffer
	inc	di			;incr past attribute
	loop	write_c_loop		;write the next char
	jmp	int_return

write_tty:
; al = char to write
	push	ax			;all regs saved/restored
	mov	dx,cursor_posn		;get page 0 cursor position
	cmp	al,1Bh			;Escape?
	je	tty_start_esc1
	cmp	cs:escmode,0		;Processing escape sequence?
	jne	tty_esc_continue1
tty_continue:
	cmp	al,8			;back space?
	je	tty_bs			;yes: branch
	cmp	al,0Dh			;carriage return?
	je	tty_cr			;yes: branch
	cmp	al,0Ah			;line feed?
	je	tty_lf			;yes: branch
	cmp	al,07h			;bell?
	je	tty_bel			;yes: branch

; write the character to the screen

	push	ax			;save char to write
	mov	ax,dx			;copy cursor position
	call	position		;get buffer location
	mov	di,ax			;copy posn to destination index reg
	pop	ax			;get char to write
	mov	ah,dispattr		;attribute for the character
	stosw				;put char/attr in display buffer
	inc	dl			;incr column
	cmp	dl,mono_cols		;wrap to next line?
	jne	tty_setcursor
	cmp	cs:wrapmode,1		;wrap to next line?
	jne	tty_return		;no: leave cursor at end of line
	xor	dl,dl			;wrap to column 1
	cmp	dh,24			;scroll required?
	jne	tty_incrow

; scroll required

tty_scroll:
	mov	ah,2			;set cursor
	int	10h			;recursive call to set cursor
	mov	bh,dispattr		;attribute to fill blank lines
	mov	ax,0601h		;scroll 1 line
	sub	cx,cx			;upper left corner
	mov	dx,184Fh		;lower right corner (24,79)
tty_final:
	int	10h			;scroll the lines
tty_return:
	pop	ax
	jmp	int_return

tty_incrow:
	inc	dh			;next row
tty_setcursor:
	mov	ah,2
	jmp	tty_final

tty_start_esc1:
	jmp	tty_start_esc

tty_esc_continue1:
	jmp	tty_esc_continue

tty_bs:
	cmp	dl,0			;already at front of line?
	je	tty_setcursor		;yes: set cursor and return
	dec	dl			;no: back up 1 space
	jmp	tty_setcursor		;set cursor and return

tty_cr:
	mov	dl,0			;go to the start of the line
	jmp	tty_setcursor		;set cursor and return

tty_lf:
	cmp	dh,24			;scroll required?
	jne	tty_incrow		;no: incr row number
	jmp	tty_scroll		;yes: scroll the screen

tty_bel:
	mov	bl,2			;count for beep
	mov	al,10110110b		;timer 2
	out	43h,al			;write timer mode reg
	mov	ax,533h			;divisor for 1000 hz
	out	42h,al			;write timer 2 count - lsb
	mov	al,ah
	out	42h,al			;write timer 2 count - msb
	in	al,61h			;get current portb setting
	mov	ah,al			;save current setting
	or	al,03h			;turn speaker on
	out	61h,al
	sub	cx,cx			;set for delay loop
tty_bel_loop:
	loop	tty_bel_loop		;delay a while
	dec	bl			;dec delay count
	jnz	tty_bel_loop		;loop if delay not expired
	mov	al,ah			;restore portb setting
	out	61h,al			;turn off sound
	jmp	tty_return

tty_start_esc:
; ESC found; start escape sequence
	mov	escmode,1		;initialize data areas
	mov	numparm,0
	mov	parmstk+0,0
	mov	parmstk+1,0
	mov	parmstk+2,0
	mov	parmstk+3,0
	jmp	tty_return		;swallow the escape char

tty_esc_continue:
; process char in an escape sequence
	cmp	al,' '			;valid char in escape sequence?
	jbe	tty_esc_end1		;no: branch
	cmp	al,'['			;escape flag char?
	je	tty_return		;yes: swallow it
	cmp	al,'='			;used in some escape sequences
	je	tty_return		;..ignore it
	cmp	al,'?'			;used in some escape sequences
	je	tty_return		;..ignore it
	cmp	al,';'			;next parameter?
	je	tty_esc_nextparm	;..yes: branch
	cmp	al,'0'			;is it a digit?
	jb	tty_esc_end1		;..no: branch
	cmp	al,'9'			;it is a digit?
	ja	tty_esc_end1		;..no: branch
	mov	si,numparm		;get index to parameter
	mov	bl,al			;save input char
	mov	al,cs:parmstk[si]	;get current value of parameter
	mul	cs:byte_ten		;mult prev value by 10
	sub	bl,'0'			;convert digit to binary
	add	al,bl			;add to previous value
	mov	cs:parmstk[si],al	;and save value
	jmp	tty_return

tty_esc_end1:
	jmp	tty_esc_end

tty_esc_nextparm:
	cmp	numparm,4		;max parms?
	je	tty_return1
	inc	numparm
tty_return1:
	jmp	tty_return

tty_esc_end:
	mov	escmode,0		;end of escape mode
	cmp	al,'H'			;set cursor position?
	je	tty_esc_setcur1
	cmp	al,'f'			;set cursor position?
	je	tty_esc_setcur1
	cmp	al,'A'			;cursor up?
	je	tty_esc_curup1
	cmp	al,'B'			;cursor down?
	je	tty_esc_curdown1
	cmp	al,'C'			;cursor right?
	je	tty_esc_curright1
	cmp	al,'D'			;cursor left?
	je	tty_esc_curleft1
	cmp	al,'s'			;save cursor posn?
	je	tty_esc_savecur1
	cmp	al,'r'			;restore cursor posn?
	je	tty_esc_restcur1
	cmp	al,'J'			;erase in display?
	je	tty_esc_erase_scr1
	cmp	al,'K'			;erase in line?
	je	tty_esc_erase_line1
	cmp	al,'m'			;set graphics rendition?
	je	tty_esc_sgr1
	cmp	al,'h'			;set mode?
	je	tty_esc_set_mode1
	cmp	al,'l'			;reset mode?
	je	tty_esc_reset_mode1
	jmp	tty_continue		;else display char

tty_esc_setcur1:
	jmp	tty_esc_setcur

tty_esc_curup1:
	jmp	tty_esc_curup

tty_esc_curdown1:
	jmp	tty_esc_curdown

tty_esc_curright1:
	jmp	tty_esc_curright

tty_esc_curleft1:
	jmp	tty_esc_curleft

tty_esc_savecur1:
	jmp	tty_esc_savecur

tty_esc_restcur1:
	jmp	tty_esc_restcur

tty_esc_erase_scr1:
	jmp	tty_esc_erase_scr

tty_esc_erase_line1:
	jmp	tty_esc_erase_line

tty_esc_sgr1:
	jmp	tty_esc_sgr

tty_esc_set_mode1:
	jmp	tty_esc_set_mode

tty_esc_reset_mode1:
	jmp	tty_esc_reset_mode

tty_esc_setcur:
	cmp	parmstk+0,0		;adjust row if zero
	jne	tty_esc_setcur_010
	inc	parmstk+0
tty_esc_setcur_010:
	cmp	parmstk+1,0		;adjust col if zero
	jne	tty_esc_setcur_020
	inc	parmstk+1
tty_esc_setcur_020:
	mov	dh,parmstk+0		;get new row
	mov	dl,parmstk+1		;get new col
	jmp	tty_setcursor

tty_esc_curup:
	cmp	parmstk+0,0		;adjust count if zero
	jne	tty_esc_curup_010
	inc	parmstk+0
tty_esc_curup_010:
	sub	dh,parmstk+0		;adjust row number
	jns	tty_esc_curup_020	;skip if result not negative
	xor	dh,dh			;else set to top row
tty_esc_curup_020:
	jmp	tty_setcursor

tty_esc_curdown:
	cmp	parmstk+0,0		;adjust count if zero
	jne	tty_esc_curdown_010
	inc	parmstk+0
tty_esc_curdown_010:
	add	dh,parmstk+0		;adjust row number
	cmp	dh,24			;past bottom of screen?
	jbe	tty_esc_curdown_020	;skip if not past bottom
	mov	dh,24			;else set to bottom row
tty_esc_curdown_020:
	jmp	tty_setcursor

tty_esc_curright:
	cmp	parmstk+0,0		;adjust count if zero
	jne	tty_esc_curright_010
	inc	parmstk+0
tty_esc_curright_010:
	add	dl,parmstk+0		;adjust column number
	cmp	dl,79			;past end of line?
	jbe	tty_esc_curright_020	;skip if not off end
	mov	dl,79			;else set to bottom row
tty_esc_curright_020:
	jmp	tty_setcursor

tty_esc_curleft:
	cmp	parmstk+0,0		;adjust count if zero
	jne	tty_esc_curleft_010
	inc	parmstk+0
tty_esc_curleft_010:
	sub	dl,parmstk+0		;adjust column number
	jns	tty_esc_curleft_020	;skip if result not negative
	xor	dl,dl			;else set to left column
tty_esc_curleft_020:
	jmp	tty_setcursor

tty_esc_savecur:
	mov	savedcur,dx		;save cursor position
	jmp	tty_return

tty_esc_restcur:
	mov	dx,savedcur		;retore cursor position
	jmp	tty_setcursor

tty_esc_erase_scr:
	cmp	parmstk+0,2		;erase all of screen?
	jne	tty_continue1		;no: error
	mov	bh,dispattr		;set up display attribute
	call	clear_scr
	xor	dx,dx			;cursor posn to home
	jmp	tty_setcursor

tty_continue1:
	jmp	tty_continue
tty_esc_erase_line:
	cmp	parmstk+0,0		;erase to end of line?
	jne	tty_continue1		;no: error
	mov	ax,dx			;get cursor position
	call	position		;convert into buffer offset
	mov	di,ax			;put offset into destination index
	mov	cx,mono_cols		;calc # columns to clear
	mov	dh,0
	sub	cx,dx			;cx has # columns to clear
	mov	ah,dispattr		;attribute
	mov	al,' '			;char = blank
	rep	stosw			;clear rest of line
	jmp	tty_return

tty_esc_sgr:
	mov	cx,numparm
	inc	cx
	xor	si,si
tty_esc_sgr_loop:
	mov	al,cs:parmstk[si]	;get parameter value
	cmp	al,0			;reset?
	je	tty_esc_sgr_reset
	cmp	al,1			;bold on?
	je	tty_esc_sgr_bold
	cmp	al,4			;underscore on?
	je	tty_esc_sgr_under
	cmp	al,5			;blink on?
	je	tty_esc_sgr_blink
	cmp	al,7			;reverse video?
	je	tty_esc_sgr_reverse
	cmp	al,8			;invisible?
	je	tty_esc_sgr_invis
tty_esc_sgr_next:
	inc	si
	loop	tty_esc_sgr_loop
tty_return2:
	jmp	tty_return

tty_esc_sgr_reset:
	mov	dispattr,07h		;set to default attribute
	jmp	tty_esc_sgr_next

tty_esc_sgr_bold:
	or	dispattr,08h		;turn on high intensity
	jmp	tty_esc_sgr_next

tty_esc_sgr_under:
	or	dispattr,01h		;turn on underscore
	jmp	tty_esc_sgr_next

tty_esc_sgr_blink:
	or	dispattr,80h		;turn on blink
	jmp	tty_esc_sgr_next

tty_esc_sgr_reverse:
	and	dispattr,0F8h		;turn on reverse video
	or	dispattr,70h
	jmp	tty_esc_sgr_next

tty_esc_sgr_invis:
	and	dispattr,88h		;make char invisible
	jmp	tty_esc_sgr_next

tty_esc_set_mode:
	cmp	cs:parmstk+0,7		;set wrap mode?
	jne	tty_return2
	mov	cs:wrapmode,1		;set wrap mode
	jmp	tty_return

tty_esc_reset_mode:
	cmp	cs:parmstk+0,7		;reset wrap mode?
	jne	tty_return2
	mov	cs:wrapmode,0		;reset wrap mode
	jmp	tty_return
	
resend	equ	$			;End of resident code

msg	db	0Dh,0Ah,'MONO10 Installed OK.',0Dh,0Ah,'$'

main:
	mov	ah,25h			;Set Vector
	mov	al,10h			;Vector number
	mov	dx,offset int10		;DS:DX has new vector value
	int	21h			;Set the Vector

	mov	ah,09h			;Print String
	mov	dx,offset msg		;DS:DX has message address
	int	21h

	mov	ax,word ptr ds:[2Ch]	;Get environment address
	mov	es,ax
	mov	ah,49h			;Free Allocated Memory
	int	21h			;Release the environment block

	mov	ax,offset resend	;# bytes that must be resident
	add	ax,15			;Round to # paragraphs
	mov	cl,4
	shr	ax,cl			;ax has # paragraphs to be resident
	mov	dx,ax
	mov	ah,31h			;Terminate and Stay Resident
	mov	al,0			;Return Code
	int	21h

code	ends
	end	start
