/*
   Add or remove directories from DOS search path.
     
   Syntax:
     
        addpath dir1 dir2 ... - dir3 dir4 ... - dir5 dir6
     
   Each appearance of "-" toggles the state of the add/delete flag.
   The initial state is "add".
     
   Compile with Microsoft C compact or large memory model.  Must use
   -Zp option.  Version 3 of MSC will also require -Ze.
     
   cl -Zp -AC addpath.c
*/
#include <dos.h>
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
     
#define MAX_PATH 1023
     
char oldpath[MAX_PATH+1];
char newpath[MAX_PATH+1];
char temppath[MAX_PATH+1];
     
/*
   envptr - returns pointer to command.com's copy of the environment
*/
char *
envptr(size)
int *size;
{
    struct INT_VEC {
        unsigned short off, seg;
    } far *intvecs;
     
    struct MCB {
        char id;
        unsigned int owner;
        unsigned int size;
    } far *mcb;
     
    intvecs = (struct INT_VEC far *) 0L;
    mcb = (struct MCB far *) (((long) (intvecs[0x2e].seg - 1)) << 16);
    mcb = (struct MCB far *) (((long) (FP_SEG(mcb) + mcb->size + 1)) << 16);
    *size = mcb->size * 16;
    return ((char far *) ((long) (FP_SEG(mcb) + 1) << 16));
}
     
/*
   msetenv - place an environment variable in command.com's copy of
             the envrionment.
*/
int
msetenv(var,value)
char *var, *value;
{
    char *env1, *env2;
    char *cp;
    int size;
    int l;
     
    env1 = env2 = envptr(&size);
    l = strlen(var);
    strupr(var);
     
    /*
       Delete any existing variable with the name (var).
    */
    while (*env2) {
        if ((strncmp(var,env2,l) == 0) && (env2[l] == '=')) {
            cp = env2 + strlen(env2) + 1;
            memcpy(env2,cp,size-(cp-env1));
        }
        else {
            env2 += strlen(env2) + 1;
        }
    }
     
    /*
       If the variable fits, shovel it in at the end of the envrionment.
    */
    if (strlen(value) && (size-(env2-env1)) >= (l + strlen(value) + 3)) {
        strcpy(env2,var);
        strcat(env2,"=");
        strcat(env2,value);
        env2[strlen(env2)+1] = 0;
        return(0);
    }
     
    /*
       Return error indication if variable did not fit.
    */
    return(-1);
}
     
/*
   addpath - add directory to path
     
   Adds the directory (p) to the path if it is not already in the
   path.
*/
void
addpath(p)
char *p;
{
    int flag = 1;
    char *cp;
     
    strcpy(temppath,newpath);
    newpath[0] = 0;
    cp = strtok(temppath,";");
    while(cp) {
        if (strcmp(p,cp) == 0) {
            flag = 0;
        }
        if (newpath[0]) strcat(newpath,";");
        strcat(newpath,cp);
        cp = strtok(NULL,";");
    }
    if (flag) {
        if ((strlen(newpath) + strlen(p) + 1) > MAX_PATH) {
            fprintf(stderr, "cannot add %s: path too long\n", p);
        }
        else {
            if (newpath[0]) strcat(newpath,";");
            strcat(newpath,p);
        }
    }
}
     
/*
   subpath - remove directory from path
     
   Removes the directory (p) from the path.
*/
void
subpath(p)
char *p;
{
    char *cp;
     
    strcpy(temppath,newpath);
    newpath[0] = 0;
    cp = strtok(temppath,";");
    while(cp) {
        if (strcmp(p,cp)) {
            if (strlen(newpath)) strcat(newpath,";");
            strcat(newpath,cp);
        }
        cp = strtok(NULL,";");
    }
}
     
/*
   translate - resolve path string to fully qualified path string.
     
   Uses undocumented DOS function 0x60 to translate path string.
*/
void
translate(dst,src)
char *dst, *src;
{
    union REGS regs;
    struct SREGS sregs;
     
    regs.x.ax = 0x6000;
    regs.x.si = FP_OFF(src);
    sregs.ds = FP_SEG(src);
    regs.x.di = FP_OFF(dst);
    sregs.es = FP_SEG(dst);
    intdosx(&regs,&regs,&sregs);
}
     
/*
   Check DOS version number
*/
int
checkversion(ver)
int ver;
{
    union REGS regs;
    int iver;
     
    regs.x.ax = 0x3000;
    intdos(&regs, &regs);
    iver = regs.h.al * 100 + regs.h.ah;
     
    if (iver == ver)
        return(0);
    else
        if (iver < ver)
            return(-1);
        else
            return(1);
}
     
main(ac,av)
int ac;
char *av[];
{
    int subflag = 0;
    int i;
    char path[128];
     
    if (checkversion(310) < 0) {
        fprintf(stderr, "version 3.10 or later DOS required\n");
        exit(1);
    }
     
    strcpy(newpath,strcpy(oldpath,getenv("PATH")));
     
    for (i=1; i<ac; i++) {
        if (strcmp(av[i],"-")) {
            translate(path,av[i]);
            if (subflag)
                subpath(path);
            else
                addpath(path);
        }
        else {
            subflag = !subflag;
        }
    }
    if (strcmp(oldpath,newpath)) {
        if (msetenv("PATH",newpath)) {
            fprintf(stderr, "new path too large, path unchanged\n");
            msetenv("PATH",oldpath);
            exit(1);
        }
    }
    exit(0);
}
