/**********************************************************************

	SE1.C           Simple EDIT

			A text editor
			based on Simple Software
			principles and conventions.

			(Part 1)

			Copyright (c) 1988, Ted A. Campbell

			Bywater Software
			Box 4023
			Duke Station,
			Durham, NC  27706

			The use of Simple EDIT is governed by the
			"Simple Software User Agreement" (see file
			SS.DOC).

**********************************************************************/

#include "se.h"

#ifdef  FAST_VIDEO
#include "stdio.h"
#include "fv.h"
#else
#include "curses.h"
#endif

#include "kb.h"

#ifdef	UINCLUDE
#include "ctype.h"
#include "stdlib.h"
#include "malloc.h"
#endif

extern FILE *se_fp;

/**********************************************************************

	GLOBAL DATA STRUCTURES

	The following data strutures define window sizes as used by
	se, and indicate current cursor position relative to the
	window.  LINEOFFSET and COLOFFSET define the top line
	and left-hand column of the Simple EDIT window relative to the
	absolute screen (character) coordinates.  se_lines and
	se_columns then define the size of the Simple EDIT window itself,
	so that, for instance, LINEOFFSET + se_lines would yield
	the bottom line of the Simple EDIT window in absolute screen
	coordinates, and COLOFFSET + se_cols would yield the right-
	most column of the Simple EDIT window in abslute screen
	coordinates.  se_lpos and se_cpos() define the current cursor
	position relative to the Simple EDIT window.

*********************************************************************/

unsigned char se_lines;                /* Lines in window            */
unsigned char se_cols;                 /* Columns in window          */
unsigned char se_lpos;                 /* Line position of the
					   current character         */
unsigned int se_alc;                   /* Absolute line count        */

/**********************************************************************

	The following global data structures define an edit buffer
	of LINESIZE characters, and se_bpos gives the position of
	the cursor in the edit buffer.

**********************************************************************/

int se_bpos;                          /* Position in edit buffer of
					   the current character     */
char se_tbuf[ LINESIZE ];             /* Temporary (scratch) buffer  */

unsigned se_imode;                     /* Insert mode:  0 = over-write
					   mode; 1 = insert mode */

/**********************************************************************

	The following data structures contain information relevant
	to path- and filename assignments.

**********************************************************************/

unsigned char se_file[ 40 ];
FILE *se_fp;

/**********************************************************************

	structure:  se_line

	The following structure defines a line as used in the Simple
	EDIT edit buffer.  The structure includes a pointer to the
	structure for the previous line, a pointer to the structure for
	the next line, and a character array which holds the actual
	text of the line.

**********************************************************************/

struct se_line *se_current, *se_btop, *se_bbot;

/**********************************************************************

	FUNCTION:  main()

	DESCRIPTION:  As in any C program, main() is the function
	from which all other functions are called.  In the case
	of Simple EDIT, main() checks for an argument, and if
	there is none, it prompts for the name of a file to
	edit.  Having a valid filename, the function then sets
	up the program, calls se_edit() to edit the document, and
	then calls se_exit() to deinitialize the program.

	INPUT:  The only input associated with the function is
	an argument in the form of a command line string which
	is taken to be the name of the file to be edited.

	RETURNS:  None.

**********************************************************************/

main( argc, argv )
	int argc;
	char *argv[];
	{
	se_setup();
	if ( argc < 2 )                         /*  Are there argu-  */
		{                               /*  ments?           */
		se_castr( 0, 0, "File to edit:  ", SE_NORMAL );
		move( 0, 15 );
		refresh( );                     /*  name             */
		se_str( se_file );
		}
	else
		{
		strcpy( se_file, argv[ 1 ] );   /*  Yes, save the    */
		}                               /*  filename         */
	se_edit();
	se_exit();
	}

/**********************************************************************

	FUNCTION:  se_setup()

	DESCRIPTION:  This function performs all setup routines and
	initializations for Simple EDIT.

	INPUT:  None.

	RETURNS:  None.

**********************************************************************/

se_setup()
	{
	initscr();
	kb_init();

	se_lines   = 24 - LINEOFFSET - 1;
	se_cols    = 80 - COLOFFSET - 1;

	/***    Allocate memory for buffer beginning and ending
		pointers.  ***/

	if ( ( se_btop = (struct se_line *) calloc( 1, sizeof( struct se_line ))) == 0 )
		{
		error( "No memory for setup." );
		return SS_ERROR;
		}
	if ( ( se_bbot = (struct se_line *) calloc( 1, sizeof( struct se_line ))) == 0 )
		{
		error( "No memory for setup." );
		return SS_ERROR;
		}
	return SS_NOMINAL;
	}

/**********************************************************************

	FUNCTION:  se_exit()

	DESCRIPTION:  This clears the screen, closes any open files,
	and performs other tasks necessary prior to leaving the program.

	INPUT:  None.

	RETURNS:  None.

**********************************************************************/

se_exit()
	{
	clear();
	move( 0, 0 );
	refresh( );
	kb_deinit();
	endwin();
	return SS_NOMINAL;
	}

/**********************************************************************

	FUNCTION:  se_edit()

	DESCRIPTION:  This function sets up a file for editing.
	It attempts to open the file in se_file; if the open
	fails, the function presumes that the file is new,
	and proceeds on that presumption.  If the open succeeds,
	the function calls se_read to read the file into memory.
	Then (in either case) it edits the file.

	INPUT:  None.  The function presumes that all parameters
	have been previously set, including a filename to be
	edited in se_file.

	RETURNS:  The function returns SS_ERROR in the case of
	an error (failure to allocate memory for edit).

**********************************************************************/

se_edit()
	{

	/***    Draw the screen for editing  ***/

	clear();
	se_castr( 3,  0, "---------1---------2---------3---------4", SE_SPECIAL );
	se_castr( 3, 40, "---------5---------6---------7---------8", SE_SPECIAL );
	sprintf( se_tbuf, "  SE v %s ", VERSION );
	se_castr( 2, 68, se_tbuf, SE_SPECIAL );
	kb_dmenu();

	/***    Set up initial parameters  ***/

	se_imode = 1;
	se_bpos = 0;
	se_lpos = 0;
	se_alc = 0;

	/***    Open the file for editing:
		read into buffer if it exists ***/

	if ( ( se_fp = fopen( se_file, "r" )) != 0 )
		{
		se_read();
		fclose( se_fp );
		se_screen( se_btop->next, 0 );
		}

	/***    New file:  set up the first line  ***/

	else
		{
		sprintf( se_tbuf, "Editing new file:  %s", se_file );
		message( se_tbuf );
		if ( (se_current = (struct se_line *) calloc( 1, sizeof( struct se_line )))
			== 0 )
			{
			error( "Memory allocation error" );
			return SS_ERROR;
			}
		se_current->next = se_bbot;
		se_current->prev = se_btop;
		se_bbot->prev = se_current;
		se_btop->next = se_current;
		se_current->characters[ 0 ] = LF;
		}

	se_lc();
	se_im();
	se_edadr( 0, 0 );
	refresh();

	while( se_getch() != SE_EXIT )
		{
		}
	}



/**********************************************************************

	FUNCTION:  se_getch()

	DESCRIPTION:  This is the basic function that is called by
	se_edit().  It gets a character from the console, and
	processes it, returning SE_EXIT if the character has
	indicated an exit from editing mode.

	INPUT:  None.

	RETURNS:  The function returns SE_EXIT if the exit
	character has been returned by se_control; otherwise
	it returns SS_NOMINAL or one of the other control
	characters;

**********************************************************************/

se_getch()
	{
	unsigned int character;

	/*** Get a character  ***/

	character = kb_ci();

	/***    Erase the message area  ***/

	message( " " );

	/***    Is the character printable? -- process it  ***/

	if ( ( (character > 0x1f ) && ( character < 0x80) )
		|| ( character == TAB))
		{
		if ( se_imode == 1 )
			{
			se_insert( character );
			}
		else
			{
			se_over( character );
			}
		return SS_NOMINAL;
		}

	/***    Process non-printable (control) character  ***/

	else
		{
		return ( se_control( character ) );
		}

	}

/**********************************************************************

	FUNCTION:  se_read()

	DESCRIPTION:  This function reads a file pointed to by se_file
	into memory, then positions se_current to the first line of the
	file.

	INPUT:  The filename to be read is presumed to be in the
	string location se_file.

	RETURNS:  In case of error (no memory), SS_ERROR is
	returned.

**********************************************************************/

se_read()
	{
	register int c;
	register unsigned int counter;
	struct se_line *oldline, *nextline;
	unsigned char endofline;

	counter = 0;
	endofline = 0;

	/***    Get the first line ready  ***/

	if ( ( se_current = (struct se_line *) calloc( 1, sizeof( struct se_line ))) == 0 )
		{
		error( "No more memory for editing." );
		return SS_ERROR;
		}
	se_current->prev = se_btop;
	se_current->next = se_bbot;
	se_btop->next = se_current;
	se_bbot->prev = se_current;

	/***    Message to user.  ***/

	sprintf( se_tbuf, "Reading file %s; please wait.", se_file );
	message( se_tbuf );

	while( feof( se_fp ) == 0 )
		{

		/***    Get one character from the file.  ***/

		c = fgetc( se_fp );

		/***    If necessary, allocate a new line.  ***/

		if ( ( endofline == 1 ) && ( feof( se_fp ) == 0 ) )
			{
			oldline  = se_current;
			nextline = se_current->next;
			if ( ( se_current = (struct se_line *) calloc( 1, sizeof (
				struct se_line ))) == 0 )
				{
				se_current = oldline;
				error( "No more memory for editing." );
				return SS_ERROR;
				}
			else
				{
				nextline->prev    = se_current;
				oldline->next     = se_current;
				se_current->next = se_bbot;
				se_current->prev = oldline;
				counter = 0;
				}
			endofline = 0;
			}

		switch ( c )
			{
			case 0:
			case CR:           /*** Discard CR ***/
				break;
			case EOF:
				se_current->characters[ counter ] = LF;
				break;
			case LF:
				se_current->characters[ counter ] = LF;
				endofline = 1;
				break;
			default:
				se_current->characters[ counter ] = c;
				++counter;
				break;
			}

		}

	se_current = se_btop->next;

	}

/*********************************************************************

	FUNCTION:  se_screen()

	DESCRIPTION:  This function fills the screen beginning with
	the line to which 'lpointer' points, which is drawn at
	visible line number 'line'.  The remainder of the visible
	screen is filled in from this point.

	INPUT:  <startline> is a pointer to the se_line structure
	denoting the first line of the screen to be drawn; <line>
	is an integer indicating the visible screen line on which
	the drawing of the screen should begin.

	RETURNS:  None.

**********************************************************************/

se_screen( startline, line )
	struct se_line *startline;
	unsigned int line;
	{
	register unsigned int lcounter;
	struct se_line *lpointer;

	/***    Initialize local variables  ***/

	lcounter = line;
	lpointer = startline;

	/***    Draw the new lines   ***/

	while( ( lcounter < se_lines ) && ( lpointer != se_bbot ) )
		{
		se_displ( lpointer, lcounter, 0 );
		lpointer = lpointer->next;
		++lcounter;
		}

	/***    Fill out any blank lines  ***/

#ifdef  FAST_VIDEO
	se_cfill( lcounter + LINEOFFSET,      COLOFFSET,
		  se_lines - 1 + LINEOFFSET,  COLOFFSET + se_cols - 1,
		  ' ' );
#else
	while( lcounter + LINEOFFSET < se_lines + LINEOFFSET )
		{
		se_edadr( lcounter, 0 );
		clrtoeol();
		++lcounter;
		}
#endif

	se_edadr( lcounter, 0 );
	}

/**********************************************************************

	FUNCTION:  se_insert()

	DESCRIPTION:  This function inserts a printable character in
	the current position in the edit buffer and updates the screen.

	INPUT:  <character> is the character to be inserted.

	RETURNS:  None.

**********************************************************************/

se_insert( character )
	unsigned char character;
	{
	unsigned char current;
	register unsigned int bpointer;
	current = se_current->characters[ se_bpos ];
	bpointer = 0;

	/***    Find the end of the string in the buffer  ***/

	while ( se_current->characters[ se_bpos + bpointer ] != LF )
		{
		++bpointer;
		}

	/***    Add a line terminator one position beyond the end  ***/

	se_current->characters[ se_bpos + bpointer + 1 ] = LF;

	/***    Now go down the string advancing each character  ***/

	while ( bpointer > 0 )
		{
		se_current->characters[ se_bpos + bpointer ] =
			se_current->characters[ se_bpos + bpointer - 1 ];
		--bpointer;
		}

	/***    Add the new character  ***/

	se_current->characters[ se_bpos ] = character;

	/***    All buffer work is now complete; now it only remains
		to update the screen */

	/***    Display the end of the line  ***/

	if ( ( se_bpos < 2 ) || ( character == TAB ) )
		{
		se_displ( se_current, se_lpos, 0 );
		}
	else
		{
		se_displ( se_current, se_lpos, se_bpos - 2 );
		}

	if ( character == TAB )
		{
		++se_bpos;
		se_edadr( se_lpos, se_cpos( se_current, se_bpos ) );
		se_lc();
		}
	else
		{
		se_right();
		}
	}

/**********************************************************************

	FUNCTION:  se_over()

	DESCRIPTION:  This function overwrites a printable character
	in the current position in the edit buffer and updates the
	screen.

	INPUT:  <character> is the character which will overwrite
	the current character.

	RETURNS:  None.

**********************************************************************/

se_over( character )
	unsigned char character;
	{
	unsigned char current;
	current = se_current->characters[ se_bpos ];
	se_current->characters[ se_bpos ] = character;

	/*** Detect end of line                                      */
	if ( current == LF )
		{
		se_current->characters[ se_bpos + 1 ] = LF;
		}

	/***    Display the end of the line                     ***/
	se_displ( se_current, se_lpos, se_bpos - 1 );
	se_right();
	}

