/*************************************************************************

	KB_IBMPC.C   -- Keyboard (KB) Subroutines
			for Simple Software.

			Implementation for the IBM PC and Compatibles
			utilizing BIOS keyboard routines.

			This implementation contains code specific
			to the Microsoft Quick C (tm) compiler.

			The use of Simple Software is governed by the
			"Simple Software User Agreement" (see file
			SS.DOC).

			Copyright (c) 1988, Ted A. Campbell

			Bywater Software
			Box 4023
			Duke Station,
			Durham, NC  27706

**************************************************************************/

#include "stdio.h"
#include "kb.h"
#include "bios.h"

#ifndef CR
#define CR      0x0d
#endif

/*************************************************************************

	FUNCTION:       kb_init()

	DESCRIPTION:    This function should perform any initialization
			necessary for the keyboard system.

	INPUT:          none.

	RETURNS:        none.

**************************************************************************/

kb_init()
	{
	}

/*************************************************************************

	FUNCTION:       kb_deinit()

	DESCRIPTION:    This function should perform any necessary
			deinitialization, that is, return the keyboard
			to its default state when a Simple Software
			program is to be exited.

	INPUT:          none.

	RETURNS:        none.

**************************************************************************/

kb_deinit()
	{
	}

/*************************************************************************

	FUNCTION:       kb_cis()

	DESCRIPTION:    This function determines whether a character is
			ready from the console.  The function is used
			especially in telecommunications programs,
			where it is necessary to poll the keyboard
			without locking up the program waiting for a
			response.

	INPUT:          none.

	RETURNS:        The function returns 0 if no character is
			available and 1 if a character is available.

**************************************************************************/

kb_cis()
     {
     return _bios_keybrd( _KEYBRD_READY );
     }

/*************************************************************************

	FUNCTION:       kb_ci()

	DESCRIPTION:    This function returns a single character from
			the keyboard.  If a character is not available
			it waits.  The function should be able to
			recognize any special keys, and return the
			appropriate Simple Software KB conventions
			designated for them.

	INPUT:          none.

	RETURNS:        The function returns the ASCII code for the
			key pressed, or the Simple Software KB convention
			(see kb.h) for a function or other special key.

**************************************************************************/

kb_ci()
     {
     unsigned int ax;
     unsigned char ah, al;
     while( kb_cis() == 0 )
	  {
	  }
     ax = _bios_keybrd( _KEYBRD_READ );
     ah = ax / 256;
     al = ax % 256;

     if ( al != 0 )
	  {
	  switch( al )
		{
		case 0x02:              /* Control-B    */
			return KB_LEFT;
		case 0x04:              /* Control-D    */
			return KB_D_CHAR;
		case 0x06:              /* Control-F    */
			return KB_RIGHT;
		case 0x0e:              /* Control-N    */
			return KB_DOWN;
		case 0x10:              /* Control-P    */
			return KB_UP;
		case 0x1b:
			return ibm_esc();
		default:
			return al;
		}
	  }

     else
	  {
	  switch( ah )
		{
		case 0x50:
			return KB_DOWN;
		case 0x48:
			return KB_UP;
		case 0x4b:
			return KB_LEFT;
		case 0x4d:
			return KB_RIGHT;
		case 0x73:
			return KB_W_LEFT;
		case 0x74:
			return KB_W_RIGHT;
		case 0x49:
			return KB_P_UP;
		case 0x51:
			return KB_P_DOWN;
		case 0x53:
			return KB_D_CHAR;
		case 0x52:
			return KB_INSERT;
		case 0x4f:
			return KB_END;
		case 0x47:
			return KB_HOME;
		case 0x3b:                      /* FK 1         */
		case 0x54:
			return KB_HELP;         /* Help         */
		case 0x3c:                      /* FK 2         */
		case 0x55:
			break;                  /* not used     */
		case 0x3d:                      /* FK 3         */
		case 0x56:
			return KB_FIND;         /* Find         */
		case 0x3e:                      /* FK 4         */
		case 0x57:
			return KB_REPLACE;      /* Replace      */
		case 0x3f:                      /* FK 5         */
		case 0x58:
			return KB_D_WORD;       /* Delete word  */
		case 0x40:                      /* FK 6         */
		case 0x59:
			return KB_D_LINE;       /* Delete line  */
		case 0x41:                      /* FK 7         */
		case 0x5a:
			break;                  /* not used     */
		case 0x42:                      /* FK 8         */
		case 0x5b:
			return KB_ABANDON;
		case 0x43:                      /* FK 9         */
		case 0x5c:
			return KB_SAVE;
		case 0x44:                      /* FK 10        */
		case 0x5d:
			return KB_EXIT;
		default:
			return 0;
		}
	  return 0;
	  }
     }

/*************************************************************************

	FUNCTION:       ibm_esc()

	DESCRIPTION:    This function handles escape sequences from the
			keyboard.  It calls the kb_emenu() function
			(defined by the application program), accepts
			a new command letter, then replace the menu
			with kb_dmenu (the default menu defined by
			the application program).

	INPUT:          none.

	RETURNS:        The function returns the Simple Software
			convention for a special key sequence (see
			kb.h), or 0.

**************************************************************************/

ibm_esc()
	{
	register int c;
	kb_emenu();
	c = getch();
	kb_dmenu();
	switch ( toupper( c ) )
		{
		case 'B':
			return KB_W_LEFT;
		case 'F':
			return KB_W_RIGHT;
		case 'P':
			return KB_P_UP;
		case 'N':
			return KB_P_DOWN;
		case 'H':
			return KB_HOME;
		case 'E':
			return KB_END;
		case 'D':
			return KB_D_WORD;
		case 'I':
			return KB_INSERT;
		case 'M':
			return KB_FIND;
		case 'R':
			return KB_REPLACE;
		case 'S':
			return KB_SAVE;
		case 'Q':
			return KB_ABANDON;
		case 'X':
			return KB_EXIT;
		default:
			message( "Unrecognized Escape Sequence" );
			return 0;
		}
	}
