/*
 * protocol
 * this is the master protocol program for an NTR rje package, for
 * communication between a pdp-11 running UNIX v6 and a UNIVAC 1100 computer.
 * The entire set of programs must run setuid root, to allow manipulation
 * of files belonging to users, and to restrict access to the communication
 * line.
 * Communication is via a du-11 with dedicated message buffers in the du-11
 * handler, and the du-11 used as a message oriented device.
 *
 * For further discussion of the organization of the rje package see
 * rje-tech-doc.
 *
 * Written by David E. Miran
 * Wisconsin State Hygiene Lab
 * 465 Henry Mall
 * Madison, Wi 53706
 *
 *  October 27, 1982
 *
 */

#include	<stdio.h>
#include	"rjedef.h"
#include	"cvtabs.h"
#include	"rjefiles.h"
#include	"rjeforms.h"

int	ack_send;		/* ack # to send with next msg 0=none */
int	bufseen	0;		/* 1=yes, we have already read a buffer from sender and looked at it */
int	clkcnt	0;		/* count of clock ticks */
int	clkid	0;		/* pid of clock process */
int	commin;			/* file id of communications input line */
int	commout;		/* file id of comm output line */
char	cur_ack;		/* current recd ack number (on mesg) */
char	curjob[7];		/* job currently being transmitted */
int	curstat	TERMK;		/* current site status - internal and externally available */
int	devstate[16];	/* device states */
int	devtype[16] 
		{ DEVCONS, NODEV, DEVCR, DEVPU, DEVLP, DEVLP,
		  NODEV, NODEV, NODEV, NODEV, NODEV, NODEV, NODEV,
		  NODEV, NODEV, DEVSITE };
int	eojflg;			/* set if next ack will release curent job */
int	fid;			/* spare file id */
char	jtext[20]	"text.xxxxxx\0\0\0\0";
char	loginmsg[8]   "93HFL0 ";	/* login message (must be fieldata) */
char	*lstype;		/* loc of last record sent to U1100 */
int	lssize	0;		/* size of last record sent to U1100 */
int	l_ack_rec;		/* last ack # recd from U1100 */
int	mtype;			/* last message type sent,0=ctrl,1=data*/
char	nakbuf[ICSIZE];		/* buffer in which to build a nak */
char	nak2buf[ICSIZE];	/* buffer in which to build a nak2 */
char	nak_send;		/* type of nak to send (0=none) */
char	nak_seq;		/* nak sequence number */
int	nrd;			/* number of chars read */
int	protpid;		/* process id of ntr */
int	rdevpnt	0;		/* receiving device pointer - for rotation among eligible devices */
int	reclist[NRECDEV] {0, 3, 4, 5};  /* devices to request output for */
long int rectime, curtime, difftime;	/* time of last input, current time, difference */
					/* for detecting timeout of rje */
int	retrycnt	0;		/* count of retransmit tries */
int	retransf	0;	/* 1 if retransmission needed */
int	rjestate;		/* state of rje system */
char	rbuf[512], sbuf[512];	/* output to receiver, input from sender */
int	rtrace		0;	/* set if receive trace is on */
int	rtracfid	0;	/* receive trace file id */
int	setsusp		0;	/* suspend site on getting ack (of init reader) */
char	sibuf[OSIZE];		/* for output from Univac */
char	signon[12];		/* place to build a signon message */
int	smesgno		0;	/* site - last message number sent */
char	sobuf[IDSIZE];		/* for input to Univac */
int	termed	1;		/* set if site is currently terminated */
int	terming	0;		/* set if site is in process of terminating (comptm, comqtm, rqterm) */
int	tracid;			/* trace file id */
int	umesgno		0;	/* U1100 - last correct message number recd */
int	whonext	0;		/* 0=request output, 1=transmit data */
int	wtack	0;		/* 1 if waiting for an ack */
int	wtlog	0;		/* are we waiting for login response - 1=yes */
int	wt_trmak	0;	/* waiting for ack of a terminate message - TERMST, RQTERM */
int	wtcterm		0;	/* waiting on term READER at end of job */
int	xtry		0;	/* set if we have tried one extra send of message */
struct icmesg *icptr;		/* pointer to an input control message */
struct idmesg *idptr;		/* pointer to an input data message */
struct omesg *optr;		/* pointer to an output (from 1100) message */
struct locmesg *imptr, *omptr;  /* pointer to internal messages */

char partab[] {
	0200,0000,0000,0200,0000,0200,0200,0000,
	0000,0200,0200,0000,0200,0000,0000,0200,
	0000,0200,0200,0000,0200,0000,0000,0200,
	0200,0000,0000,0200,0000,0200,0200,0000,
	0000,0200,0200,0000,0200,0000,0000,0200,
	0200,0000,0000,0200,0000,0200,0200,0000,
	0200,0000,0000,0200,0000,0200,0200,0000,
	0000,0200,0200,0000,0200,0000,0000,0200,
	0000,0200,0200,0000,0200,0000,0000,0200,
	0200,0000,0000,0200,0000,0200,0200,0000,
	0200,0000,0000,0200,0000,0200,0200,0000,
	0000,0200,0200,0000,0200,0000,0000,0200,
	0200,0000,0000,0200,0000,0200,0200,0000,
	0000,0200,0200,0000,0200,0000,0000,0200,
	0000,0200,0200,0000,0200,0000,0000,0200,
	0200,0000,0000,0200,0000,0200,0200,0000
};

char ascfd(), fdasc();

main(argc,argv)
char **argv;
{
extern errno;
extern int rjeown, consfid, logfid;
extern int backflg, scneed;
extern int stwork(), shutdown(), clktick();
register int i;

	setbuf(stdout,NULL);
	rjestate = 0;
	close(0);	close(2);	/* not used - need fids elsewhere */
	protpid=getpid();
	if ((fid = creat(pidfile, 0644)) < 0) {
		printf("cannot register protpid\n");
		exit(1);
		}
	if ((write(fid, &protpid, 2)) < 0) {
		printf("register of protpid failed\n");
		exit(1);
		}
	close(fid);
	nice(20);
	signal(CLOCK,&clktick);
	signal(WORK,&stwork);
	signal(QUIT,&shutdown);
/* spawn the clock process for 30 second probes */
	clkid = fork();
	if (clkid == 0) {  /* child process (clock) */
		execl("/rje/bin/rjeclock","rjeclock", 0);
		exit(0);
	}
	if (clkid < 0) clkid = 0;  /* no clock process if fork fails */
	/* check options and turn on any traces wanted */
	/* options allowed
	 *	i trace input to U1100
	 *	o trace output from U1100
	 *	c only write control portion of images into trace file
	 *	r separate received input trace
	 *	t test mode (for extended tracing
	 */
		while (--argc)
			switch ( **++argv ) {
			case 'i':
				rjestate =| ITRACE; break;
			case 'o':
				rjestate =| OTRACE; break;
			case 'c':
				rjestate =| CTRACE; break;
			case 't':
				rjestate =| TESTMD; break;
			case 'r':
				rtrace = 1;  break;
	
			default:
				printf("unknown flag: %c\n", **argv);
					exit(1);
		}
		if ((tracid = creat(tracefil,0644)) < 0) {
			printf("cannot create trace file\n");
			exit(1);
		}
		if ((rtracfid = creat(rtrcfile, 0644)) < 0) {
			printf("cannot creat receive trace file\n");
			exit(1);
		}
/* if the traces are not turned on close the trace files */
	if ((rjestate & ANYTRACE) == 0) {
		close(tracid);
		tracid = -1;
	}
	if (rtrace == 0) {
		close(rtracfid);
		rtracfid = -1;
	}
	/* start by opening up files, registering ids, and then generate
	 * and send a login message.
	 */
	
	if ((commin = commout = open(dufile, 2)) < 0) {
		printf("ntr: cannot open du-11 communications line.\n");
		exit(1);
	}
	chdir(datafile);
	
	rjeown = SITUID | SITGID<<8;
	consfid = open(console,1);
	if ((logfid = open(logfile, 1)) < 0)
		logfid = creat(logfile, 0644);
	else
		seek(logfid, 0,2);
	buildsgo();  /* build the login message */
	scneed = 1;
	setstat(TERMK);
	goto wt_wake;	/* go wait for a wakeup and login command */

waitin:
	nrd = read(commin, sibuf, OSIZE);
	if (rjestate & TESTMD) printf("read input - %d chars\n",nrd);
	if (nrd < 1) {		/* error or serviced signal */
		if (errno == 4) {		/* serviced signal - see whatsup */
			time(&curtime);	/* check current time */
			difftime = curtime - rectime;	/* vs last input time */
			if (difftime > MAXTIME) rjestate =| TIMOUT;
			if (rjestate & TIMOUT) {
				if (rjestate & TERMED) goto wt_wake; /* not active */
				if (terming)	{  /* complete and term */
					rjestate =| TERMED;
					rjestate =& ~(ACTIVE|IDLE);
					terming = wt_trmak = 0;
					termed = 1;
					goto wt_wake;
				}
				/* if lost communication try one resend of last message */
				/* before backing up processing */
				if (xtry == 0) {  /* first shot */
					xtry = 1;
					write(commout, lstype, lssize);
					genlog("rje: send retry\n");
					goto waitin;
				}
				/* we tried this once and it didn't help - start over */
				setstat(TIMDOUT);
				backflg = 1; /* warn sender to backup */
				rjestate =| FLUSH;  /* clean out stuff from sender */
				close(commin);
				sleep(1);
				if ((commin = commout = open(dufile, 2)) < 0) {
					printf("ntr: cannot re-open du-11 communications line.\n");
					exit(1);
				}
				sleep(1);
				login();	/* initiate login sequence */
				genlog("rje: 1100 timeout - login sent\n");
				setstat(TIMDOUT);
				goto waitin;	/* and wait for response */
				}
			goto waitin;	/* nothing to worry about */
			}
		if (nrd == 0) {
			sleep(2);  /* wait and try again */
			goto waitin;
		}
	}
	rjestate =& ~(WAIT | TIMOUT);
	clkcnt = 0;
	time(&rectime);	/* note time of latest input from U1100 */
	if (rjestate & ANYTRACE) tracegen(3, nrd, sibuf);

	/* now for the hard part - handle protocol */
	/* check for valid parity, lrc, mesg length - return 1 for invalid */
	if (valchk(sibuf, nrd))   {
		if (rjestate & TESTMD) printf("valchk failed\n");
		do_nak1();
		goto waitin;
	}
	if (rjestate & WTNAK2)   {
	/* go process response to our nak1 */
		if (wtnak2()) {
			rjestate =| WAIT;
			goto waitin;
		}
		if (rjestate & WAIT) goto waitin;
		goto whatnext;
	}

/*  we have a good input message (probably) go process it */
	if (do_input()) goto waitin;  /* problems - go wait for more input */
	if (rjestate & WAIT) goto waitin;
	if (termed) goto wt_wake;
whatnext:
	do_next();
	if (termed) goto wt_wake;
	goto waitin;
wt_wake:
	/* wait here if site is not active, until a login message is processed */
	if (rjestate & TESTMD) printf("rje state is termed\n");
	if (rjestate & FREADY) goto whatnext;
	sleep(300);	/* 5 minute nap - will be interrupted by any signal */
	if (termed) goto wt_wake;
	setstat(WTLOG);
	termed = 0;
	login();
	time(&rectime);
	genlog("rje: logging in\n");
	goto waitin;
}

/*setstat - write status into a file for inquiry by status program */

setstat(code)
int	code;
{
int sfid;
	sfid = creat(statfile, 0644);
	write(sfid, &code, 2);
	close(sfid);
	curstat = code;
}
/*
 * input
 * code for rje protocol program which will analyze a message from
 * the U1100, check the ack numbers, and then process the message
 * in an appropriate way.
 *
 */

/* return codes - 0 = do a whatnext,  1 = goto waitin */

do_input()
{

extern backflg;
int nxtmsg, dev, det;
register int i, j;


	if (rjestate & TESTMD) printf("entered do_input\n");

	/* strip out parity bits */
	for (i=0; i<nrd; i++)
		sibuf[i] =& 0177;
	optr = sibuf;
	/* check message number if msg is not ack or nak */
	if (optr->o_msgtyp > NAK2) {
		nxtmsg = umesgno + 1;
		if (nxtmsg > 63) nxtmsg = 1;
		i = optr->o_msgnum;
		if (nxtmsg == i) {
			umesgno = nxtmsg;
			ack_send = i;
		}  else  {
			if (rjestate & TESTMD) printf("bad message #.  Expected %d   got %d\n",nxtmsg,i);
			maklog(0, nxtmsg, i, 1000);
			do_nak1();
			return(1);
		}
	}

	/* process ack (if any included) */
	if (wtack == 0) goto no_ack;  /* we don't expect any */
	if (optr->o_acknum == 0) goto no_ack;  /* didn't get one */
	i = optr->o_acknum;
	l_ack_rec = i;
	if (i == smesgno) {
		/* good ack of last message */
		wtack = 0;
		if (setsusp)	{	/* got ack of init reader - suspend site */
			rjestate =| SUSPND;
			setsusp = 0;
		}
	}  else  {
		/* bad - last message not acked */
		retransf++;
	}
no_ack:

/* input handler - at this point we have a valid message, and have
   processed any accompanying ack
 */

	if (!retransf) retrycnt = 0;
	nak_seq = 0;
	dev = optr->o_device;

	/* now process according to the message type */

	switch (optr->o_msgtyp) {
		case ACK:	/* ack - we have already processed */
			if (wt_trmak) {	/* this ack is to a terminate message */
				wt_trmak = 0;
				terming = 0;
				termed = 1;
				setstat(TERMK);
			}
			break;

		case NAK1:	/* a nak1 from U1100 */
			pro_nak1();
			break;

		case DATA:	/* data from the U1100 - send to receiver program */
			omptr = rbuf;
			omptr->m_type = IDATA;
			omptr->m_dev = dev;
			omptr->m_size = RECSIZE;
			omptr->m_detail = optr->o_detail;
			for (i=0;i<RECSIZE;i++)
				omptr->m_text[i] = optr->o_data[i];
			if (optr->o_detail == DATAEOF) { /* end of data file */
				devstate[dev] =& ~ACTIVE;
				omptr->m_type =| IEOJ;
				for (i=0; i<NRECDEV; i++) {
					j = reclist[i];
					if (devstate[j] & ACTIVE) goto put_out;
				}
				rjestate =& ~RECPRG; /* no output device active */
			}
put_out:
			recv(omptr);	/* process buffer out */
			rjestate =& ~BUFASK;	/* we got the buffer - ask for more */
			break;

		case TERMD:	/* terminate device on order of U1100 */
			devstate[dev] =| DEVTERM;
			devstate[dev] =& ~ACTIVE;
			if (dev == READER)  {
				if (wtcterm) { /* we were waiting for this */
					wtcterm = 0;
				}  else  {  /* suprise - Univac shut us down for errors */
					backflg = 1;
					rjestate =| FLUSH;
					rjestate =& ~JOBPRG;
				}
			}
			for (i=0; i<NRECDEV; i++) {
				j = reclist[i];
				if (devstate[j] & ACTIVE) return(0);
			}
			rjestate =& ~RECPRG; /* no output device active */
			if (dev == CONSOL) {
			/* build a console end of data message */
				omptr = rbuf;
				omptr->m_type = IDATA;
				omptr->m_dev = dev;
				omptr->m_size = 1;
				omptr->m_detail = DATAEOF;
				omptr->m_text[0] = ENDBUF;
				recv(omptr);
			}
			break;

		case SUSPEN:	/* suspend device on order of U1100 */
			if (dev < 15)
				devstate[dev] =| SUSPND;  /* suspend device */
			else
				rjestate =| SUSPND;   /* suspend site */
			break;

		case RESUME:	/* undo suspend of device or site */
			if (dev < 15)
				devstate[dev] =& ~SUSPND;
			else
				rjestate =& ~SUSPND;
			break;

		case TERMST:	/* terminate site on order of U1100 */
			terming = wt_trmak = 0;
			termed = 1;
			setstat(TERMK);
			break;

		case INIT:	/* initialize device or site */
			if (dev == 15) {
				if (wtlog == 0)
					genlog("received unexpected site initialize command\n");
				wtlog = 0;
				rjestate =| ACTIVE | IDLE;
				setstat(ACTIVE);
				l_ack_rec = cur_ack = nak_seq = nak_send = smesgno=0;
				umesgno = optr->o_msgnum & 0177;
				rjestate =& ~(RECPRG|TIMOUT|JOBPRG|BUFASK|TERMED|SUSPND|WTNAK2);
				backflg = 1;
			} else {  /* initialize device  */
				devstate[dev] =& ~SUSPND;
				devstate[dev] =& ~DEVTERM;
				devstate[dev] =| ACTIVE;
				rjestate =| RECPRG;  /* set receive active */
				rjestate =& ~IDLE;
			}
			break;

		case DISPLY:	/* display message from U1100 */
			omptr = rbuf;
			omptr->m_type = IDISP;
			omptr->m_dev = dev;
			omptr->m_detail = optr->o_detail;
			omptr->m_size = 0;
			recv(omptr);
			break;

		default:
		/* unrecognized message type */
			i = optr->o_msgtyp;
			det = optr->o_detail;
			if (rjestate & TESTMD)
				printf("invalid message from U1100 - type, dev, detail= %o %o %o\n",
					i, dev, det);
			maklog(2, i, dev, det);
	}  /* end of switch statement */
	return(0);
}
/*
 * next
 * code for rje protocol program which will review the pending
 * needs for service, select the next thing to do, and do it.
 * runs round robin in each of transmit and request data and
 * alternates between those two types of things.
 *
 */

do_next()
{
register int i, j, dev;
int x, y, z;
char c, *v;
int try_s, try_r;



	try_s = try_r = 0;
whatnext:
	if (rjestate & TESTMD)
		printf("next: rjestate = %o whonext = %d\n", rjestate, whonext);
	if (rjestate & SUSPND) goto lone_ack;
	if (retransf) goto retrans;
	retrycnt = 0;
	if ( !(rjestate & (RECPRG|FREADY|JOBPRG)))  {
		rjestate =| IDLE;
		 goto lone_ack;
	}
trynext:
	if (try_s && try_r) goto lone_ack; /* nothing else to do */
	if (whonext) {
		goto ck_send;
	}  else {
		goto ck_rec;
	}

ck_rec:
/* either nothing to send or not the turn for the sending devices.
   Transmit a request for input for next eligible device.
 */

	whonext = 1;
	try_r = 1;
	if (!(rjestate & RECPRG)) goto trynext; /* no receive in progress */
	if (rjestate & BUFASK) goto trynext;  /* outstanding request */
	i=0;
devsel:
	rdevpnt++;
	if (rdevpnt >= NRECDEV) rdevpnt=0;
	if (i++ > 16) {	/* no receivers ready */
		rjestate =& ~RECPRG;
		goto trynext;
	}
	dev = reclist[rdevpnt];
	if ((devstate[dev] & ACTIVE) == 0) goto devsel;
	if (devstate[dev] & (SUSPND | DEVTERM)) goto devsel;

	/* found an eligible device - generate request for output */

	buildcm(REQ, dev, 1);
	rjestate =| BUFASK;
	return(0);

ck_send:
	whonext = 0;
	try_s = 1;
	if (wtcterm) goto trynext;  /* last job not yet fully done */
	if (rjestate& (FREADY|JOBPRG)) goto set_send;
	goto trynext;

set_send:

	/* get next message to process */
	imptr = sender();	/* pointer to buffer with message */
	if (rjestate & ANYTRACE)
		tracegen(4, imptr->m_size+5, imptr);
	j = imptr->m_type;
	if (rjestate & FLUSH) {
		if (j&07 == IFLUSH) {
			rjestate =& ~FLUSH;
			goto set_send;
		}
		if (j == IEOF) {
			rjestate =& ~(FLUSH|FREADY|JOBPRG);
			goto trynext;
		}
	goto set_send;
	}
	if (j == IEOF) { /* no more follows */
		rjestate =& ~FREADY;
		x = READER;
		if (devstate[x] & ACTIVE)  {	/* need to terminate reader */
			devstate[x] =& ~ACTIVE;
			buildcm(TERMD, READER, 0);
			if (rjestate & TESTMD)
				printf("prot:next:  got IEOF and TERMD READER\n");
			return(0);
		}
		goto trynext;
	}
	if ( j == IEOJ) {  /* end of job */
		joblog(6, curjob);
		rjestate =& ~JOBPRG;
		for (j=0; j<7; j++)
			jtext[j+5] = curjob[j];
		unlink(jtext);
		chmod(curjob,0654);
		if (rjestate & TESTMD)
			printf("prot:next: got IEOJ\n");
		x = READER;
		if (devstate[x] & ACTIVE)  {	/* need to terminate reader */
			devstate[x] =& ~ACTIVE;
			buildcm(TERMD, READER, 0);
			if (rjestate & TESTMD)
				printf("prot:next:  got IEOF and TERMD READER\n");
			return(0);
		}
		goto set_send;
	}
	j =& 07;  /* mask off special bits */
	if (j == ISHUT) shutdown();
	if (j == ICTRL) {  /* send a control message */
		x=imptr->m_text[0];
		y=imptr->m_dev;
		z=imptr->m_detail;
		if (x == 99) { /* login */
			if (curstat == ACTIVE) goto set_send;
			login();
			genlog("rje: login sent\n");
			setstat(WTLOG);
			termed = terming = 0;
			return(0);
		}
		if (termed) goto set_send; /* only accept login ctrl message if termed */
		buildcm(x, y, z);
		if ((x == TERMST) || (x == RQTERM)) {
			rjestate =& ~(JOBPRG|ACTIVE|WTNAK2|RECPRG|IDLE);
			genlog("rje: terminated by keyin\n");
			terming = wt_trmak = 1;
		}
		if ((x == COMPTM) || (x == COMQTM)) {
			genlog("rje: complete and term in progress\n");
			terming = 1;
		}
		return(0);
	}
	if (j == ISTRT) { /* initialize device for input to 1100 */
		x = imptr->m_dev;
		for (i = 0; i<7; i++)
			curjob[i] = imptr->m_text[i];
		joblog(5, curjob);
		if (devstate[x] & ACTIVE) goto set_send;
		buildcm(INIT, x, 0);
		rjestate =| (WAIT | JOBPRG);
		setsusp = 1;	/* suspend site on ack of the init message */
		rjestate =& ~IDLE;
		devstate[x] =| ACTIVE;
		return(0);
	}
	if (j == IDATA) { /* data transfer to the U1100 */
		if (imptr->m_size != SNDSIZE) {
			printf("prot:next: data message is wrong size = %d\n", imptr->m_size);
			shutdown();
		}
		x = imptr->m_dev;
		y = imptr->m_detail;
		v = &(imptr->m_text[0]);
		builddm(DATA, x, y, v);
		return(0);
	}
	if (j == ILOG) {  /* log message - simply foreward */
		i = imptr->m_size + 5;
		recv(imptr);
		if (rjestate & FREADY) goto set_send; /* go get more */
		goto trynext;  /* see if anything else to do */
	}
	if (j == ITRC) {  /* turn trace modes on or off */
		/* device is the character mode and detail is 1 for on and 0 for off */
		c = imptr->m_dev;
		i = imptr->m_detail;
		switch (c) {
			case 'i':
				if (i) rjestate =| ITRACE;
					else rjestate =& ~ITRACE;
				break;
			case 'o':
				if (i) rjestate =| OTRACE;
					else rjestate =& ~OTRACE;
				break;
			case 'c':
				if (i) rjestate =| CTRACE;
					else rjestate =& ~CTRACE;
				break;
			case 't':
				if (i) rjestate =| TESTMD;
					else rjestate =& ~TESTMD;
				break;
			case 'r':
				rtrace = i;
				break;
			default:
				printf("invalid trace command =%c\n",c);
		}
		if ((rjestate & ANYTRACE) != 0) {
			if (tracid == -1) {
				tracid = open(tracefil, 1);
				lseek(tracid, 0L, 2);
			}
		} else {
			if (tracid != -1) {
				close(tracid);
				tracid = -1;
			}
		}
		if (rtrace != 0) {
			if (rtracfid == -1) {
				rtracfid = open(rtrcfile, 1);
				lseek(rtracfid, 0L, 2);
			}
		} else {
			if (rtracfid != -1) {
				close(rtracfid);
				rtracfid = -1;
			}
		}
	}
	if (rjestate & FREADY) goto set_send;  /* go get more */
	goto trynext;  /* no more - see if anything else to do */

lone_ack:
	sendack();
	return(0);


retrans:
	/* strip out ack riders and resend last message */
	i = 0;
	if (mtype < 0) goto retrlog;  /* no message to retrans */
	i = sobuf[7] & 0177;	/* get message type */
	if (i == 0) {  /* do not retrans ack */
		retransf = 0;
		goto whatnext;
	}

	if (retrycnt++ > RETRYLIM) {
		printf("ntr: exceeded retransmit limit\n\r");
		shutdown();
	}
	if (mtype) {  /* data message to resend */
		idptr = sobuf;
		j = l_ack_rec + 1;
		if (j > 63) j=1;
		if (smesgno != j) goto retrlog;
		idptr->d_acknum = 0200;  /* odd parity 0 */
		idptr->d_lrc = genlrc(&sobuf[4], IDSIZE-5);
		putout(sobuf,IDSIZE);
		x = idptr->d_msgtyp & 0177;
		y = idptr->d_msgnum & 0177;
		z = idptr->d_device & 0177;
		if (rjestate & ANYTRACE)
			tracegen(2, IDSIZE, sobuf);
		if (rjestate & TESTMD)
			printf("retransmitted data message\n");
	}  else  {     /* control message to resend */
		icptr = sobuf;
		j = l_ack_rec + 1;
		if (j > 63) j=1;
		if (smesgno != j) goto retrlog;
		icptr->c_acknum = 0200;  /* odd parity 0 */
		icptr->c_lrc = genlrc(&sobuf[4], ICSIZE-5);
		putout(sobuf, ICSIZE);
		x = icptr->c_msgtyp & 0177;
		y = icptr->c_msgnum & 0177;
		z = icptr->c_device & 0177;
		if (rjestate & ANYTRACE)
			tracegen(2, CSIZE, sobuf);
		if (rjestate & TESTMD)
			printf("retransmitted control message\n");
	}
	retransf = 0;
	rjestate =| WAIT;
	return(0);
retrlog:
	/* retransf was set, but either there was no message to resend (after a login)
	 * or last ack received +1 was not = message number to send so we are permanently
	 * out of sync.  So, reinitialize the site.
	 * Unless the last message has been acked, then ignore
	 */
	retransf = 0;
	if ((l_ack_rec & 0177) == smesgno) goto whatnext;  /* no problem */
	maklog(9, l_ack_rec&0177, smesgno, 1000);
	login();
	return(0);
}
/*
 * build
 * part of the protocol program of the rje package
 * to build various types of messages, and usually send them
 */


/* buildsgo - build a signon (login) message - called once only */

buildsgo()
{
int i;
char c;
char *ptr;
	for(i=0;i<4;i++)
		signon[i]=SYNC;
	for (i=4; i<10;i++) {
		c = loginmsg[i-4];
		signon[i] = ascfd(c);
	}
	signon[10]=EOM;
	ptr= &signon[4];
	setopar(ptr,6);  /* attatch odd parity to message  */
	ptr = &signon[4];
	signon[11]=genlrc(ptr,7);
}

/* buildcm - build a control message - called for all short
	(control) messages except login, nak1, nak2, and ack
 */

buildcm(typ, dev, det)

{

register int i;

	icptr = sobuf;
	smesgno++;
	if (smesgno > 63) smesgno = 1;
	for (i=0;i<4;i++)
		icptr->c_syncs[i] = SYNC;
	icptr->c_soh = SOM;
	icptr->c_etx = EOM;
	icptr->c_msgnum = smesgno;
	icptr->c_acknum = ack_send;
	ack_send = 0;
	icptr->c_msgtyp = typ;
	icptr->c_device = dev;
	icptr->c_detail = det;

	setopar(&sobuf[4], ICSIZE-6);
	icptr->c_lrc = genlrc(&sobuf[4], ICSIZE-5);
	rjestate =| WAIT;
	putout(sobuf, ICSIZE);
	if (rjestate & ANYTRACE)
		tracegen(2, ICSIZE, sobuf);
	if (rjestate & TESTMD)
		printf("sent control message - typ,dev,detail= %d %d %d\n",
			typ, dev, det);
	wtack++;
	mtype = 0;
}

/* builddm - build a data message */

builddm(typ, dev, det, text)
int typ, dev, det;
char text[];

{

register int i;

	idptr = sobuf;
	smesgno++;
	if (smesgno > 63) smesgno = 1;
	for (i=0;i<4;i++)
		idptr->d_syncs[i] = SYNC;
	idptr->d_soh = SOM;
	idptr->d_etx = EOM;
	idptr->d_stx = STX;
	idptr->d_msgnum = smesgno;
	idptr->d_device = dev;
	idptr->d_detail = det;
	idptr->d_msgtyp = typ;
	idptr->d_acknum = ack_send;
	ack_send = 0;

	for (i=0;i<SNDSIZE;i++)
		idptr->d_data[i] = text[i];
/* data is always a fixed size and is preformatted */

	setopar(&sobuf[4], IDSIZE-6);
	idptr->d_lrc = genlrc(&sobuf[4], IDSIZE-5);
	rjestate =| WAIT;
	putout(sobuf, IDSIZE);
	if (rjestate & ANYTRACE)
		tracegen(2, IDSIZE, sobuf);
	if (rjestate & TESTMD)
		printf("sent data message - typ,dev,detail, msg #= %d %d %d %d\n",
			typ, dev, det, smesgno);
	wtack++;
	mtype = 1;
}

/* sendack - send a lone ack message */

sendack()

{

register int i;

	icptr = sobuf;
	for (i=0;i<4;i++)
		icptr->c_syncs[i] = SYNC;
	icptr->c_soh = SOM;
	icptr->c_etx = EOM;
	icptr->c_msgnum = 0;
	icptr->c_device = 0;
	icptr->c_detail = 0;
	icptr->c_msgtyp = ACK;
	icptr->c_acknum = ack_send;
	ack_send = 0;

	setopar(&sobuf[4], ICSIZE-6);
	icptr->c_lrc = genlrc(&sobuf[4], ICSIZE-5);
	rjestate =| WAIT;
	putout(sobuf, ICSIZE);
	if (rjestate & ANYTRACE)
		tracegen(2, ICSIZE, sobuf);
	if (rjestate & TESTMD)
		printf("sent lone ack\n");
	mtype = 0;
}

/* genlrc - generate an lrc and return it*/


genlrc(string,len)
char *string;
int len;
{
register i, sum;

	sum = 0;
	for (i=0;i<len;i++)
		sum =^ *string++;
	return(sum);
}


/* login - send a login message to U1100 */


login()
{
register int i;
extern int jobprg, sjobprg;
	/* reset state of system and send a login message */
	
	rjestate =| WAIT|IDLE;
	rjestate =& ~(ACTIVE|SUSPND|JOBPRG|WTNAK2|RECPRG|BUFASK|TERMED);
	l_ack_rec = cur_ack = nak_seq = nak_send = smesgno = umesgno = 0;
	setsusp = eojflg = retransf = retrycnt = wt_trmak = wtcterm = 0;
	jobprg = sjobprg = 0;
	for (i=0; i<16; i++)  /* reset all devices */
		devstate[i] =& ~(ACTIVE|SUSPND|DEVTERM);
	putout(signon, ICSIZE);  /* send the login */
	if (ANYTRACE & rjestate) {
		tracegen(1, ICSIZE, signon);
		}
	wtlog++;
	wtack++;
	mtype = -1;
	/* clean up any files in progress at login time */
	omptr = rbuf;
	omptr->m_type = IFLUSH;
	omptr->m_size = 0;
	recv(omptr);
}
/*
 * naks
 * code for rje protocol program related to nak transmission
 * or reception and processing
 */

/* generate a nak1 in response to receiving an erroneous message */

do_nak1()
{
register int i;


	if (rjestate & TESTMD) printf("generating a NAK1\n");
	icptr = nakbuf;
	for (i=0; i<4; i++)     icptr->c_syncs[i] = SYNC;
	icptr->c_soh = SOM;
	icptr->c_etx = EOM;
	icptr->c_msgnum = 0;
	icptr->c_acknum = umesgno;
	icptr->c_msgtyp = NAK1;
	icptr->c_device = ++nak_seq;
	icptr->c_detail = 0;

	setopar(&nakbuf[4], ICSIZE-6);
	icptr->c_lrc = genlrc(&nakbuf[4], ICSIZE-5);

	rjestate =| WTNAK2 | WAIT;
	putout(nakbuf, ICSIZE);

	if (rjestate & ANYTRACE)
		tracegen(2, ICSIZE, nakbuf);
	if (rjestate & TESTMD)
		printf("NAK1 sent with seq number %d\n",nak_seq);
	genlog("NAK send\n");
}

/* process a received message when we are in the 'waiting for nak2' state */
/* i.e. we have sent a nak1 in response to a bad message */

wtnak2()
{
	optr = sibuf;

	if (optr->o_msgtyp == NAK1)   {
		pro_nak1();	/* process nak1 and quit waiting on nak2 */
		rjestate =& ~WTNAK2;
		return(0);
	}
	if (optr->o_msgtyp != NAK2) return(1); /* not nak2, ignore */

	/* device is nak sequence number to check */
	if (optr->o_device != nak_seq) return(1); /* not right one */
	rjestate =& ~WTNAK2;  /* no longer waiting for nak2 */

	if (optr ->o_acknum != 0) {
		l_ack_rec = cur_ack = optr->o_acknum;
		if (cur_ack != smesgno) retransf++;
	}
	return(0);
}

/* process a nak1 from u1100 - i.e. one of our messages was
   received in error or this is an idle state timeout handshake */

pro_nak1()
{
register int i;

	optr = sibuf;
	if (optr->o_acknum != 0)
		l_ack_rec = cur_ack = optr->o_acknum;
	if (l_ack_rec != smesgno)
		retransf++;
	icptr = nak2buf;
	if (!(rjestate & (RECPRG|FREADY|JOBPRG))) {
		rjestate =| IDLE;
	}
	if (!(rjestate&IDLE)) maklog(4,optr->o_detail, 1000, 1000);
	if (rjestate & TESTMD) printf("generating a NAK2\n");
	for (i=0; i<4; i++)     icptr->c_syncs[i] = SYNC;
	icptr->c_soh = SOM;
	icptr->c_etx = EOM;
	icptr->c_msgnum = 0;
	icptr->c_acknum = umesgno;
	icptr->c_msgtyp = NAK2;
	icptr->c_device = optr->o_device;
	icptr->c_detail = 0;

	setopar(&nak2buf[4], ICSIZE-6);
	icptr->c_lrc = genlrc(&nak2buf[4], ICSIZE-5);

	putout(nak2buf, ICSIZE);

	if (rjestate & ANYTRACE)
		tracegen(2, ICSIZE, nak2buf);
	if (rjestate & TESTMD)
		printf("NAK2 sent\n");
	rjestate =| WAIT;
	return(0);
}
/*
 * logger
 * code for rje protocol to generate log messages to record errors
 * and important status changes
 */

char	logbuf[132];		/* buffer in which to construct a log message */
char	*errmsg[]	{	/* a set of pointers to numbered messages */
	/*  0 */	"Bad seq from U1100 - expected, got",
	/*  1 */	"Did not get ack - scheduled retransmission of message number",
	/*  2 */	"Invalid message from U1100 - type, device, detail",
	/*  3 */	"",
	/*  4 */	"NAK1 recd-detail=",
	/*  5 */	"Began sending ",
	/*  6 */	"End sending ",
	/*  7 */	"Invalid message from sender to protocol ignored. Type is ",
	/*  8 */	"",
	/*  9 */	"Site out of sync - reinitialized - l-ack-rec, msg to send, ",
			0};

char	*cptr, *lptr;

/* put a character string into a log message and write it to receiver */

genlog(mesg)
char *mesg;
{

register int i;

	omptr = rbuf;
	omptr->m_type = ILOG;
	omptr->m_dev = omptr->m_detail = 0;
	i = 0;
	while (omptr->m_text[i++] = *mesg++);
	omptr->m_size = i-1;
	recv(omptr);
}

/* take a number and convert it to 2 decimal digits and add it to the line */

dig2(dig)
int dig;
{
	if (dig > 999) return(0);
	*lptr++ = ' ';
	*lptr++ = ((dig/10) | 060) & 077;
	*lptr++ = (dig%10) | 060;
	*lptr++ = ' ';
}

/* make a log message from a numbered error message and 3 numbers */

maklog(num, l1, l2, l3)
int num, l1, l2, l3;
{
	cptr = errmsg[num];
	lptr = logbuf;
	while (*lptr++ = *cptr++);
	lptr--;
	dig2(l1);  dig2(l2);  dig2(l3);
	*lptr++ = '\n';
	*lptr = 0;
	genlog(logbuf);
}

/* joblog - make a log entry about a job */

joblog(num, job)
int num;
char job[];
{
register int i;

	cptr = errmsg[num];
	lptr = logbuf;
	while (*lptr++ = *cptr++);
	lptr--;
	*lptr++ = ' ';
	for (i=0; i<6; i++)
		*lptr++ = job[i];
	*lptr++ = '\n';
	*lptr++ = '\0';
	genlog(logbuf);
 }
/* maintain an rje activity debugging trace file */

/* form of trace file is:
 * 2 bytes - size of record, ^, mode, ^, text, ^
 */

tracegen(mode, len, mesg)

/* mode - 1=login, 2=input to U1100, 3= output from U1100, 4=from sender routine */

int mode,len;
char *mesg;
{
char  tracbuf[512];
register int size, i, j;

	if ((mode == 1) | (mode == 2) | (mode == 4))
		if (!(rjestate&ITRACE)) goto notrace;
	if (mode == 3)
		if (!(rjestate&OTRACE)) goto notrace;
	
	tracbuf[2]=tracbuf[4]='^';
	tracbuf[3]= mode;
	size=5;
	if (!(rjestate&CTRACE)) goto notc;
	switch(mode)  {
		case 1:
		case 2:
			for (i = 0; i<7; i++)
				tracbuf[size++] = mesg[i+4];	/* skip syncs */
			if (len > ICSIZE)
				tracbuf[size++] = mesg[len-2];	/* etx */
			tracbuf[size++] = mesg[len-1];	/* lrc */
			break;
		case 3:
			for (i=0; i<7; i++)
				tracbuf[size++] = mesg[i];
			if (len > ICSIZE)
				tracbuf[size++] = mesg[len-2];	/* etx */
			tracbuf[size++] = mesg[len-1];	/* lrc */
			break;
		case 4:
			for (i = 0; i<6; i++)
				tracbuf[size++] = mesg[i];
	}	/* end of switch */
	goto finit;
notc:
	for (i = 0; i<len; i++)
		tracbuf[size++] = mesg[i];
finit:
	tracbuf[size++]='^';
	tracbuf[1]=size;
	tracbuf[0]=size>>8;
	write(tracid, tracbuf, size);
notrace:;
}
/*
 * setpar - various routines to set parity and check
 *	    message validty
 */

/* setopar - attach odd parity to all chars in a string */

setopar(ptr,cnt)
char *ptr;
int cnt;
{
int i;
for (i=0;i< cnt; i++) {
	*ptr =| partab[*ptr];
	ptr++;
	}
}

/* chkopar - check a character string and return 0 if all are
   	     odd parity, 1 if they are not   */

chkopar(ptr, cnt)
char *ptr;
int cnt;
{
register int i, c2;
register char j;
char c3, c4;

	for (i=0;i<cnt;i++) {
		j = *ptr++;
		c2 = j & 0177;
		c3 = partab[c2];
		c4 = j & 0200;
		if (c3 != c4) {
			if (rjestate & TESTMD)
				printf("parity error on char # %d\n",i);
			return(1);
		}
	}
	return(0);
}

/* valchk - check message validity ( size, parity, and lrc) */
/* returns 0 for valid messages and 1 for invalid */

valchk(ptr, nrdx)
char ptr[];
int nrdx;
{
int i;

	if (( nrdx != CSIZE) && ( nrdx != OSIZE))  {
		if (rjestate & TESTMD)
			printf("invalid message size %d\n", nrdx);
		return(1);
	}
	if (chkopar(ptr,nrdx-2))  {
		if (rjestate & TESTMD)
			printf("bad parity on input message\n");
		return(1);
	}
	if ((i = (0177&genlrc(ptr, nrdx-1))) != (ptr[nrdx-1] & 0177)) {
		if (rjestate & TESTMD)
			printf("invalid lrc calc,act = %o  %o\n",i,ptr[nrdx-1]);
		return(1);
	}
	return(0);  /*  all ok */
}
/* chcode - ascii to fd and fd to ascii conversions */


char ascfd(c)
char c;
{
extern char aftab[];
register char xc;
/* convert ascii to fieldata */
xc = c&0177;
return(aftab[xc]);
}
char fdasc(c)
char c;
{
extern char fatab[];
register char xc;
/* convert fieldata to ascii */
xc = c&077;
return(fatab[xc]);
}
/*
 * sigs
 * code for rje protocol program related to processing signals
 * from other programs
 */

/*  clock tick interrupt - used for timeout of U1100 */
clktick()
{

	if (rjestate & TESTMD) printf("got clock tick signal\n");
	if (termed) goto reclk;  /* ignore clock ticks if not logged in */
	clkcnt++;
	if (clkcnt >= NTICK) {
		rjestate =| TIMOUT;
		clkcnt = 0;
	}
reclk:
	signal(CLOCK, &clktick); /* reset handler */
}

/* close down communications in case of an error or a QUIT interrupt */
shutdown()
{

	fid=open(pidfile,2);
	protpid=0;
	write(fid, &protpid, 2);	/* mark ntr gone */
	close(fid);
	if (clkid != 0)  kill(clkid,9);
	setstat(NOTOP);
	printf("rje: shutting down\n\r");
	sleep(2);
	exit(1);
}

/* write a message to the U1100 and save the location and size of the message for possible retransmit */
putout(x, y)
char *x;
int y;
{
	lstype = x;
	lssize = y;
	xtry = 0;
	write(commout, x, y);
}
