/*
Name:
	SUSDO Calendar System Command

Function:
	Suspend dolist item.

Algorithm:
	Scan the user supplied list file for the requested item and extract
	it from the list file. If no user file is supplied, look in the
	users $HOME/exec directory to find the default file. Then use the
	current default dolist. Then append the item to the appropriate
	suspense file. Then sort/list the new list and suspense files.

Parameters:
	item number             the line number of the list item in the list file
	due-date		the due date of the suspense (if a d is entered
				a five day default due date)
	user supplied list file
		or
	default file in $HOME/exec/default

Returns:
	ENOERR	Normal exit
	EFORM	Bad date format or incorrect number of command line arguments

Files and Programs:
	user supplied list file
	$SUSPENSE/sus.ddmmm                             the target suspense file
	TEMPFILE                                        temporary file for extraction

Installation Instructions:
	Compile as follows: cc -o $EXEC.DIR/susdo susdo.c ../library
	Change owner to root
	Deny write permissions except for owner

History:
	Dec 79,	Release 2 Version 1, G. Maltagliati, coded in shell
	Nov 80,	Release 5 Version 1, G. Maltagliati, coded in C for efficiency
	Jul 82, Release 7 Version 1, L. Reed, for Perkin-Elmer
	Aug 82, Release 7 Version 2, J. Neyer...rewrite exec support


*/
#include <stdio.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#define TEMPFILE "/tmp/susdoXXXXXX"
#define USAGE "You must enter:  susdo item-number due-date <list-file>\n"

char id[] = "%W% %H%";

/*

Name:
	main

Function:
	Validate parameters. Open the list file. Find the proper suspense
	file name. Suspend the item. List the dolist and suspense file.

Algorithm:
	Insure that the correct number of arguments were entered on the command
	line. Determine if a user supplied list file was given and open it if
	so. Otherwise, find the default dolist file in $HOME/exec/default
	and open it. Call makesus to form the name of the suspense file from
	the given date. Open the suspense file and do the suspense. Call
	dolist and suslist to perform "DOLIST" on the new do and suspense
	lists. Close the files and exit.

Parameters:
	item number		the line number of the dolist item in the dolist file
	due-date		the due date of the suspense (if a d is entered
				a five day default due date)
	user supplied list file or default list file

Returns:
	ENOERR	Normal exit
	EFORM	Bad date format or incorrect number of command line arguments

Files and Programs:
	user supplied list file or default list file


*/
main (argc, argv, envp)
int     argc;						/* Count of command line arguments */
char   *argv[];						/* Pointer to list of argument pointers */
char   **envp;                  /* the environment */
{


    char listfile[80],
	 *lfile,                             /* Pointer to list file name */
	 *date,
	 *sfile;                                        /* Pointer  to suspense file name */
    char *line;
    FILE *fp;

    struct passwd *user;

    extern char *getline();
    extern char *makesus();
    extern char *mktemp();
    extern char *sprintf();
    extern char *strcat();
    extern char *findenv();

    if (argc < 3 || argc > 4)       {
	fprintf (stderr, USAGE);
	exit(EFORM);
    }
    parse(argc, argv, envp, &date, &user, &lfile);
    if (!lfile)      {
	lfile = listfile;
	strcat (lfile, (findenv("HOME", envp)));        /* Get the HOME directory */
	strcat (lfile, "/");
	strcat (lfile, DEFAULT);
	if ((fp = fopen(lfile, "r")) == NULL)   {       /* Try to open default */
	    fprintf (stderr, "susdo: cannot open default file %s\n", lfile);
	    exit (EACCES);
	}
	if ((line = getline(fp)) == NULL)    {         /* Get the default dolist */
	    fprintf (stderr, "susdo: default file empty\n");
	    exit (EACCES);
	}
	else
	    strcpy(lfile, line);
	if ((fclose(fp)) == EOF)                             {
	    fprintf (stderr,"susdo: cannot close file %s\n", lfile);
	    exit (EACCES);
	}
    }
    if ((sfile = makesus(argv[2], findenv("HOME", envp))) == NULL)     {       /* Create the suspense file name using date */
	fprintf (stderr, "susdo: invalid date format %s\n", argv[2]);
	exit (EACCES);
    }
    dosus(lfile, sfile, argv[1]);
    dolist_call(lfile, findenv("USER", envp));
    exit(ENOERR);
}
/*

Name:
	dosus

Function:
	Remove the item from the dolist and add it to the suspense file.

Algorithm:
	Open a temp file. Extract the do item and append it to suspense file.
	Copy rest of list file to temp. Close the list and temp files.
	Unlink the old list fiel and link the new list file from the temp
	file. Error exit if temp file problems.

Parameters:
	list file               pointer to list file
	suspense file           pointer to suspense file
	item number		the line number of the dolist item in the dolist file

Returns:
	ENOERR	Normal exit
	EACESS  Access error to files.

Files and Programs:
	user supplied list file or default list file
	suspense file


*/
dosus (lfile, sfile, item)
char *lfile, *sfile,
     *item;

{
    char   *tfile,                                      /* Pointer to temp file name */
	   *line,                                       /* Holds extracted line */
	    susp_num[5],
	   *susnum = susp_num,                          /* Number of times item was suspended */
	    com_buf[CMDLEN],
	   *command = com_buf;
    int     index, colcount = 0,
	    tvec[2];                        /* Time system call return info */
    FILE   *tfp, *fp, *sfp;


    if ((fp = fopen(lfile, "r")) == NULL)   {           /* Open dolist */
	fprintf (stderr, "susdo: cannot open dolist %s\n", lfile);
	exit (EACCES);
    }
    tfile = mktemp (TEMPFILE);
    if ((tfp = fopen(tfile, "a")) == NULL)      {
	fprintf(stderr, "susdo: cannot open temp file %s\n", tfile);
	exit (EACCES);
    }
    if ((sfp = fopen(sfile, "a")) == NULL)      {
	fprintf (stderr, "susdo: cannot open suspense file %s\n", sfile);
	exit (EACCES);
    }
    for (index = 1; ((line = getline(fp)) != NULL); index++)
	if (index != atoi(item))        {
	    fputs (line, tfp);
	    fputc ('\n', tfp);
	}
	else    {
	    while (*line != FIELDCHAR || colcount < 2)        {
		fputc(*line, sfp);
		if (*line == FIELDCHAR)
		    colcount++;
		line++;
	    }
	    fputc(*line++, sfp);
	    while (*line != FIELDCHAR) line++;                 /* Eat up old time */
	    time(tvec);                                   /* Get current time */
	    fprintf(sfp , "%s", (cnv_time(tvec)));        /* Put converted time */
	    colcount = 0;
	    while (colcount < 3)        {
		fputc(*line, sfp);
		if (*line == FIELDCHAR)
		    colcount++;
		line++;
	    }
	    while (*line != FIELDCHAR)
		*susnum++ = *line++;
	    *susnum = '\0';
	    susnum = susp_num;
	    colcount = atoi(susnum);
	    colcount++;
	    fputs(itoa(colcount,0), sfp);                /* Increment suspense count */
	    fputs (line, sfp);
	    fputc ('\n', sfp);
	}
    if ((fclose(tfp)) == EOF)     {
	fprintf (stderr, "cannot close file %s\n", tfile);
	exit (EACCES);
    }
    if ((fclose(fp)) == EOF)      {
	fprintf (stderr, "cannot close file %s\n", lfile);
	exit (EACCES);
    }
    if ((fclose(sfp)) == EOF)     {
	fprintf (stderr, "cannot close file %s\n", sfile);
	exit (EACCES);
    }
    sprintf(command, "cp %s %s", tfile, lfile);
    if ((system(command)) != 0)             {                /* Copy the temp file */
	fprintf (stderr, "susdo: error on system call\n");
	exit (ESYSTEM);
    }
    if ((unlink (tfile)) == -1)         {
	fprintf (stderr, "susdo: error unlinking temp file %s\n", tfile);
	exit (EUNLINK);
    }
    return;
}
