#
/*
Name:
	APPOINT Calendar System Command

Function:
	Enters an appointment in a user's calendar file system.

Algorithm:
	If only one appointment is to be entered, display the other appointments
	for that day and ask if the user if he would like to continue. If he does
	ask for the appointment and enter it in the file system. Otherwise, exit.
	If multiple appointments are to be entered, ask the user for the 
	appointment and add the appointment for all days requested.

Parameters:
	m	Indicates multiple appointments
	date	the date the appointment is to be entered. If multiple appointments
		then it is the starting date.
	#_days	the number of consecutive days to add the appointment (for 
		multiple appointments only 
	start-time	the starting time of the appointment
	end-time	the ending time of the appointment (optional)

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or exec/appointments directory
	EFORM	Invalid date or time format
	EBRK	Too many consecutive days (max = 31)

Files and Programs:
	$HOME/exec/appointments/filename	appointments files

History:
	Oct 79,	Release 1 Version 1,G. Maltagliati, coded in shell
	Sep 80,	Release 4 Version 1,Sgt. T. Irish, recoded for efficiency
	Nov 80,	Release 5 Version 1,G. Maltagliati, coded in C for efficiency
	Aug 82, Release 7 Version 2,J, Neyer...re-write exec support


*/
#include <stdio.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#define errormsg(a) write(2,a,sizeof(a)-1)
#define APPOINT "exec/appointments"
#define DATEERR "appoint: invalid date format\n"
#define TIMEERR "appoint: invalid time format\n"
#define TOOMANY "You cannot make an appointment for more than 31 days at a time\n"
#define USAGE "You must enter: appoint <date> <start-time> <end-time>\n		appoint m <start-date> <# of days> <start-time> <end-time>\n"

char id[] = "%W% %H%";


/*

Name:
	main

Function:
	Add an appointment for one day or multiple days to a user's calendar
	file system.

Algorithm:
	Determine if one or multiple appointments are to be entered. Check
	to see that the correct number of arguments have been entered, and then
	check to see if a valid date has been entered. If any errors have been 
	found, note them and exit. Otherwise execute the appropriate subroutine,
	oneappoint for one appointment and multiappoint for multiple appointments,
	and exit.

Parameters:
	m	Indicates multiple appointments
	date	the date the appointment is to be entered. If multiple appointments
		then it is the starting date.
	#_days	the number of consecutive days to add the appointment (for 
		multiple appointments only 
	start-time	the starting time of the appointment
	end-time	the ending time of the appointment (optional)

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or exec/appointments directory
	EFORM	Bad date format

Files and Programs:
	None



*/
main (argc, argv, envp)
int     argc;						/* Count of command line arguments */
char   *argv[];						/* Pointer to list of argument pointers */
char   **envp;						/* Pointer to list of environment parameters */
{
    char    date[15],					/* Holds the standardized date (ddmmm) */
           *month;					/* Pointer to month number */
    if (chdir (findenv("HOME",envp)))
    {
	printf ("appoint: bad home directory");
	exit (ENOENT);
    }
    if (chdir (APPOINT))		/* move to appoints directory */
    {
	printf ("appoint: bad directory--%s\n", APPOINT);
	exit (ENOENT);
    }
    if (argc >= 3)					/* Check the number of arguments */
	if (*argv[1] != 'm')
/* Only one appointment is to be entered. Now check the date */
	    if (month = valid_date (argv[1], date, 0))
		oneappoint (date, argc, argv[2], argv[3]);
							/* Valid date */
	    else					/* Invalid date */
	    {
		errormsg (DATEERR);
		exit (EFORM);
	    }
	else						/* Multiple date appointments */
	    if (argc >= 5)				/* Were the correct number of arguments entered */
		if (month = valid_date (argv[2], date, 0))
							/* Check date */
		    multiappoint (date, atoi (month), argc, atoi (argv[3]), argv[4], argv[5]);
							/* Valid date */
		else					/* Invalid date */
		{
		    errormsg (DATEERR);
		    exit (EFORM);
		}
	    else					/* Not enough arguments were entered */
	    {
		errormsg (USAGE);
		exit (EFORM);
	    }
    else						/* Not enough arguments were entered */
    {
	errormsg (USAGE);
	exit (EFORM);
    }
    exit (ENOERR);
}
/*

Name:
	oneappoint

Function:
	Add one appointment to a user's calendar file system

Algorithm:
	Standardize the times to format hhmm and ask for the appointment
	text.  Then call writeappoint to add it.

Parameters:
	date	date of the appointment
	argc	used to find if an end time was entered
	stime	the starting time of the appointment
	etime	the ending time of the appointment

Returns:
	EFORM	Bad time format

Files and Programs:
	None



*/
    extern char *gettime();
oneappoint (date, argc, stime, etime)
char   *date,						/* Date of the appointment */
       *stime,						/* Start time of the appointment */
       *etime;						/* End time of the appointment */
int     argc;						/* Indicates if an end time was entered */
{
    char    appoint_buf[MAXAP],                             /* Holds the appointment */
	   *appoint = appoint_buf,
	   file[12],					/* Name of the appointment file */
           *file_ptr,					/* Pointer to file */
           *end_ptr;					/* Pointer to outbuf */
    char   store_stm[5],
	   store_etm[5],
	   *time_ptr,
	   temp;
    int    i,
	   ival,
	   time_flag,
	   temp_stm,
	   temp_etm;
    time_flag = 1;
    time_ptr = gettime(stime,time_flag);
    for (i=0; *(time_ptr + i) != '\0'; i++) {
	store_stm[i] = *(time_ptr + i);
	if (store_stm[i] == 'a' || store_stm[i] == 'p')
	   temp = store_stm[i];
    }
	temp_stm = atoi(store_stm);
	if (atoi(store_stm) == 1200 && temp == 'a')
	   temp_stm = 0;
	else
	   if (atoi(store_stm) == 1200 && temp == 'p')
	      temp_stm = 1200;

	if (temp_stm >= 0 && temp_stm <= 2400)   {    /* Standardize start time */
	   ival = atoi(store_stm);
	   if (temp == 'p' && ival >= 1300)
	      ival -= 1200;
	end_ptr = copy (itoa(ival, 4), appoint);
	*end_ptr++ = temp;
    }
    else                                                /* Invalid time format */
    {
	errormsg (TIMEERR);
	exit (EFORM);
    }
    if (argc == 4){                                    /* Check to see if an end time was entered */
	time_flag = 0;
	time_ptr = gettime(etime, time_flag);
	for (i=0; *(time_ptr + i) != '\0'; i++){
	    store_etm[i] = *(time_ptr + i);
	    if (store_etm[i] == 'a' || store_etm[i] == 'p')
		temp = store_etm[i];
	}
	temp_etm = atoi(store_etm);
	if (atoi(store_etm) == 1200 && temp == 'a')
	   temp_etm = 0;
	else
	   if (atoi(store_etm) == 1200 && temp == 'p')
	      temp_etm = 1200;
	   if (temp_etm >= 0 && temp_etm <= 2400 && temp_etm > temp_stm)
							   /* Standardize end time */
	   {
	      ival = atoi(store_etm);
	   if (temp == 'p' && ival >= 1300)
	      ival -= 1200;
	    *end_ptr++ = '-';
	    end_ptr = copy (itoa (ival, 4), end_ptr);
	    *end_ptr++ = temp;
	}
	else						/* Invalid time format */
	{
	    errormsg (TIMEERR);
	    exit (EFORM);
	}
   }
/* Add rest of format to outbuf (excluding the actual appointment) */
    *end_ptr++ = ':';
    *end_ptr++ = ' ';
    *end_ptr++ = ' ';
    *end_ptr = '\0';
    file_ptr = copy ("appts.", file);			/* Create the name of the  appointment file */
    copy (date, file_ptr);
    printf("\nEnter appointment on %s for %s\n", date, appoint_buf);
    fgets(end_ptr, MAXAP-12, stdin);
    writeappoint( file, appoint_buf);               /* make the appointment */
    set_alarm(stime, date,appoint_buf);
}
/*

Name:
	multiappoint

Function:
	Add an appointment to a user's appointment file system for a number
	of consecutive days

Algorithm:
	First, standardize the times to the four digit military time form (hhmm).
	Build the appointment record, by first placing the start time, dash, and
	then the end time. Prompt the user for the appointment and append the 
	appointment to the end of the record already created. Find the julian date
	for the start date of the appointments. For num_days number of days, do
	add the appointment to the file.


Parameters:
	date		start date for the range of appointments
	month		the number of the month in date
	argc		indicates if an end time has been entered
	num_days	the number of consecutive days for the appointment to be entered
	stime		the starting time of the appointment
	etime		the ending time of the appointment

Returns:
	EFORM	Bad time format
	EBRK	Too many consecutive days (max = 31)

Files and Programs:
	None


*/
extern char *gettime();
multiappoint (date, month, argc, num_days, stime, etime)
char   *date,						/* Start date for multiple appointments */
       *stime,						/* Start time of the appointment */
       *etime;						/* End time of the appointment */
int     argc,						/* Indicates if an end time has been entered */
        num_days,					/* The number of consecutive days the appointment will be entered */
        month;						/* The number of the month in the starting date */
{
    char    appoint_buf[MAXAP],                             /* Holds the appointment record */
	   *appoint = appoint_buf,
            file[12],					/* Holds the name of the file */
           *file_ptr,					/* Pointer into file */
           *end_ptr,					/* Pointer to outbuf */
            jdate[6],					/* Julian date in ascii */
            newdate[6];					/* Holds new date after julian date has been incremented */
    int     julian,					/* Julian date (integer) */
            year,					/* Year (yy) */
            days;					/* Number of days in the year */
    char    store_stm[5],
	    store_etm[5],
	    *time_ptr,
	    temp;
    int     i,
	    ival,
	    time_flag,
	    temp_stm,
	    temp_etm;

    time_flag = 1;
    time_ptr = gettime(stime, time_flag);

    for (i=0; *(time_ptr + i) != '\0'; i++){
	store_stm[i] = *(time_ptr + i);
	if (store_stm[i] == 'a' || store_stm[i] == 'p')
	    temp = store_stm[i];
    }
	temp_stm = atoi(store_stm);
	if (atoi(store_stm) == 1200 && temp == 'a')
	   temp_stm = 0;
	else
	   if (atoi(store_stm) == 1200 && temp == 'p')
	      temp_stm = 1200;

    if (temp_stm >= 0 && temp_stm <= 2400)   {    /* Standardize start time */
	ival = atoi(store_stm);
	if (temp == 'p' && ival >= 1300)
	   ival -= 1200;
	end_ptr = copy (itoa (ival, 4), appoint);
	*end_ptr++ = temp;
    }
    else						/* Invalid time format */
    {
	errormsg (TIMEERR);
	exit (EFORM);
    }

    if (argc == 6) {                                    /* Check to see if an end time was entered */
	time_flag = 0;
	time_ptr = gettime(etime, time_flag);

	for (i=0; *(time_ptr + i) != '\0'; i++) {
	    store_etm[i] = *(time_ptr + i);
	    if (store_etm[i] == 'a' || store_etm[i] == 'p')
	       temp = store_etm[i];
	}
	temp_etm = atoi(store_etm);
	if (atoi(store_etm) == 1200 && temp == 'a')
	   temp_etm = 0;
	else
	   if (atoi(store_etm) == 1200 && temp == 'p')
	      temp_etm = 1200;

	if (temp_etm >= 0 && temp_etm <= 2400 && temp_etm > temp_stm)
							/* Standardize end time */
	{
	    ival = atoi(store_etm);
	    if (temp == 'p' && ival >= 1300)
	       ival -= 1200;
	    *end_ptr++ = '-';
	    end_ptr = copy (itoa (ival, 4), end_ptr);
	    *end_ptr++ = temp;
	}
	else						/* Invalid time format */
	{
	    errormsg (TIMEERR);
	    exit (EFORM);
	}
   }
/* Finish building record excluding the actual appointment */
    *end_ptr++ = ':';
    *end_ptr++ = ' ';
    *end_ptr++ = ' ';
    *end_ptr = '\0';
    if (num_days > 31)					/* Check to see if the number of days exceeds the max */
    {
	errormsg (TOOMANY);
	exit (EBRK);
    }
    printf ("Enter appointment:\n");
    fgets(end_ptr, MAXAP-12, stdin);
    file_ptr = copy ("appts.", file);
    year = atoi (get_year ());				/* Get the current year */
    julian = atoi (julian_date (date, month, year));    /* Get the julian date */
    if ((year / 4) * 4 == year && year != 0)		/* Find the number of days in the year */
	days = 366;
    else
	days = 365;
    while (num_days-- > 0)
    {
	copy (date, file_ptr);				/* Create filename of appointment file */
	writeappoint (file, appoint_buf);           /* Write appointment to the file */
	set_alarm(stime, date, appoint_buf);
	if (++julian > days)				/* Check for end of year and increment julian date */
	{						/* New year */
	    julian = 1;
	    if (++year == 100)				/* Only occurs if it is the year 2000 */
		year = 0;
	}
/* Create the julian date in the form yyjjj */
	copy (itoa (year, 2), jdate);
	copy (itoa (julian, 3), &jdate[2]);
	if (!valid_date (jdate, newdate, 1))  {               /* Get the new date, format ddmmm */
	   printf(stderr, "appoint: date conversion error\n");
	   exit(EFORM);
	}
	date = newdate;
    }
}
