#
/*
Name:
	ADDSUS Calendar System Command

Function:
	Add items to a user's suspense list.

Algorithm:
	Parse the input arguments. Create the suspense file name using
	user supplied date. Locate the proper list file using the user
	id if passed.
	Write each line to the specified list file. Sort and redisplay
	the list by calling "dolist".

Parameters:
	date
	user id

Returns:
	ENOERR  Normal exit
	EACCES  Unable to open user specified list file

Files and Programs:
	user specified list file        place to store new items
	user id                         authorized user id

Installation Instructions:
	Compile:  cc -o $EXECDIR/addsus addsus.c ../library
	Change owner to root
	Deny write permissions except for owner

History:
	Oct 79,  Release 1 Version 1,G. Maltagliati, coded in shell
	Nov 80,  Release 5 Version 1,G. Maltagliati, coded in C for
		 efficiency.
	Jul 82, Release 7  Version 1, L. Reed for Perkin-Elmer
	Aug 82, Release 7  Version 2, J. Neyer...rewrite exec support

*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#include <pwd.h>
#define ITEM "Item: "
#define ENTERINFO "Enter suspense items. Use q to quit\n"
#define NEWLINE '\n'
#define USAGE "Usage: addsus date <user>\n"

char id[] = "%W% %H%";

/*

Name:
	main

Function:
	Add entered items to the end of a calendar user's list

Algorithm:
	Parse input arguments. Open the user supplied list file.
	Append new lines to the end of the file, until a "q"
	is input. Close the file. Call dolist to sort and display
	the file.

Parameters:
	User supplied list file in argv[1]
	user id

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory
	EACCES  Unable to open list file

Files and Programs:
	user supplied list file         place to store list items
	user id

*/
main( argc, argv, envp)

int argc;
char  *argv[], **envp;
{
    char *date;
    char *lfile;
    FILE *fp;                                           /* List file pointer */
    struct passwd *user;
    extern char *makesus();

    parse(argc, argv, envp, &date, &user, &lfile);
    if (!user)
	user = getpwnam(findenv("USER", envp));
    if ((lfile = makesus(date, user->pw_dir)) == NULL)       {
	fprintf(stderr, "addsus: invalid date %s\n", argv[1]);
	exit(EFORM);
    }
    if ((fp = fopen(lfile, "a")) == NULL)   {         /* Open the list file */
	fprintf (stderr, "addsus: cannot open list file %s\n", lfile);
	exit (EACCES);
    }
    add_items(fp, user->pw_dir);
    fclose(fp);
    dolist_call(lfile, user->pw_name);
    exit(ENOERR);
}

/*

Name:
	add_items

Function:
	Add entered items to the end of a calendar user's list

Algorithm:
	Read the user terminal. Append new lines to the end of the file,
	until a "q" is input.

Parameters:
	list file pointer
	user home directory name

Returns:
	ENOERR	Normal exit

Files and Programs:
	list file                 place to store list items


*/
add_items (fp, user_dir)
FILE *fp;
char *user_dir;
#define BUFSIZE 128
{

    int num_chars,                      /* Number of characters read */
	index,
	tvec[2];                        /* Time system call return info */
    char buffer[BUFSIZE];               /* Item buffer */
    char do_file[80];

    printf(ENTERINFO);
    printf(ITEM);
    while ((num_chars = read (0, &buffer, BUFSIZE)))     /* Get one to-do item */
	if (num_chars > 1)      {                       /* Not a carriage return */
	    if (num_chars == 2 && buffer[0] == 'q')     /* Look for quit */
		break;
	    for (index = 0; buffer[index] == '*'; index++)
		fputc(buffer[index], fp);
	    fputc (FIELDCHAR, fp);
	    for (; buffer[index] != '\n'; index++)
		fputc (buffer[index], fp);                /* Put in item */
	    fputc (FIELDCHAR, fp);
	    time(tvec);                                   /* Get current time */
	    fprintf(fp , "%s", (cnv_time(tvec)));         /* Put converted time */
	    fputs ("::", fp);
	    printf ("Due to: ");
	    num_char = read(0, &buffer, BUFSIZE);          /* Get to whom the suspense is due */
	    for (index = 0; buffer[index] != '\n'; index++)
		fputc(buffer[index], fp);
	    fputs(":::", fp);
	    strcpy(do_file, user_dir);                    /* Put in target dolist name */
	    strcat(do_file, "/exec/dolist/do");
	    fprintf(fp, "%s\n", do_file);
	    printf(ITEM);
	}
    return(ENOERR);
}

