/***********************************************************************
*
* MKDISK - Make a TI990 TILINE disk image file.
*
* Changes:
*      12/17/03   DGP   Original
*      07/15/04   DGP   Corrected DS80 info.
*
***********************************************************************/
 
#include <stdio.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#define MAXDISKS 9

typedef struct
{
   char *model;
   int cyls;
   int heads;
   int sectrk;
   int bytsec;
   int overhead;
} disk_types;


disk_types disks[MAXDISKS] =
{
   { "WD900",  815, 24, 100, 256,  0 },
   { "WD500",  150,  4,  32, 256,  0 },
   { "DS300",  803, 19,  61, 256,  0 },
   { "DS200",  815, 19,  38, 288,  0 },
   { "DS80",   803,  5,  61, 256,  0 },
   { "DS50",   815,  5,  38, 288,  0 },
   { "DS25",   408,  5,  38, 288,  0 },
   { "DS10",   408,  2,  20, 288, 96 },
   { "FD1000",  77,  2,  26, 288,  0 },
};

/***********************************************************************
* dskwriteint - Write an integer.
***********************************************************************/

static int
dskwriteint (FILE *fd, int num)
{

   if (fputc ((num >> 24) & 0xFF, fd) < 0) return (-1);
   if (fputc ((num >> 16) & 0xFF, fd) < 0) return (-1);
   if (fputc ((num >> 8) & 0xFF, fd) < 0) return (-1);
   if (fputc (num & 0xFF, fd) < 0) return (-1);
   return (0);
}

/***********************************************************************
* dskreadint - Read an integer.
***********************************************************************/

static int
dskreadint (FILE *fd)
{
   int r;
   int i;

   r = 0;
   for (i = 0; i < 4; i++)
   {
      int c;
      if ((c = fgetc (fd)) < 0) return (-1);
      r = (r << 8) | (c & 0xFF);
   }
   return (r);
}

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *dfd;
   char *bp;
   char *diskfile;
   char *diskname;
   int i;
   int diskndx;
   int disksize;
   int verbose;

   verbose = FALSE;
   diskfile = NULL;
   diskname = NULL;

   /*
   ** Scan off args 
   */

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'd':
	    i++;
	    diskname = argv[i];
	    break;

	 case 'v':
	    verbose = TRUE;
	    break;

         default:
      USAGE:
	    printf ("usage: mkdisk [-options] disk.fil\n");
            printf (" options:\n");
	    printf ("    -d dskmodel  - Disk model\n");
            printf ("    -v           - Verbose output\n");
	    return (ABORT);
         }
      }

      else
      {
         if (diskfile) goto USAGE;
         diskfile = argv[i];
      }

   }

   if (!diskfile) goto USAGE;
   if (!diskname) diskname = "DS10";

   /*
   ** Check disk model
   */

   for (diskndx = 0; diskndx < MAXDISKS; diskndx++)
   {
      if (!strcmp (diskname, disks[diskndx].model)) break;
   }
   if (diskndx == MAXDISKS)
   {
      printf ("mkdisk: Unknown disk model: %s\n", diskname);
      exit (ABORT);
   }

   /*
   ** Calulate the size of the disk
   */

   disksize = (disks[diskndx].bytsec + disks[diskndx].overhead) *
	disks[diskndx].sectrk * disks[diskndx].heads * disks[diskndx].cyls;

   if (verbose)
   {
      float size;
      size = disksize / 1048576.0;
      printf ("mkdisk: create disk %s on file %s\n", diskname, diskfile);
      printf (
"Geometry: cyls = %d, heads = %d, sec/trk = %d, byte/sec = %d, overhead = %d\n",
		  disks[diskndx].cyls, disks[diskndx].heads,
		  disks[diskndx].sectrk, disks[diskndx].bytsec,
		  disks[diskndx].overhead);
      printf (" Disk size = %6.2f MB\n", size);
   }

   /*
   ** Open the disk image file
   */

   if ((dfd = fopen (diskfile, "wb")) == NULL)
   {
      perror ("mkdisk: Can't open diskfile");
      exit (ABORT);
   }

   /*
   ** Write the disk geometry information
   */

   dskwriteint (dfd, disks[diskndx].cyls);
   dskwriteint (dfd, disks[diskndx].heads);
   dskwriteint (dfd, (disks[diskndx].overhead << 8) | disks[diskndx].sectrk);
   dskwriteint (dfd, disks[diskndx].bytsec);

   /*
   ** Write the disk image
   */

   for (i = 0; i < disksize; i++)
   {
      fputc ('\0', dfd);
   }
   
   fclose (dfd);
   return (0);

}
