/***********************************************************************
*
* fpyfile - Dump file from TI-900 FD800 disk to stdout.
*
* Changes:
*   06/24/03   DGP   Original.
*	
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#define FALSE 0
#define TRUE  1

/*
** Disk geometry stuff
*/

#define MAXAUS 333	/* Max Allocation units (AU) on disk */
#define SECPERAU   6	/* Sectors per AU */

#define SECPERTRK 26	/* Sectors per track */
#define SECLEN 128	/* Sector length */

/*
** Location of things
*/

#define DISKLABELAU 4	/* Disk label AU */
#define DISKDIRAU 5	/* Disk directory AU start */

#define DIRENTSIZE 16	/* Size of a directory entry */
#define DIRENTPSEC 8	/* Directory entries per sector */
#define DIRFILE 0	/* Offset to a file name */
#define DIRFILELEN 7	/* Length of the file name */
#define DIREXT 7	/* Offset to a file name extension */
#define DIREXTLEN 3	/* Length of the file name extension */
#define DIRFCB 10	/* Offset to the FCB location */

#define FCBEXTENT 20	/* Offset to FCB file extents */

/*
** File control codes
*/

#define ENDOFRECORD 0xFF
#define COMPBLANKS 0xFE
#define ENDOFFILE 0xFB
#define BINARYESCAPE 0xFA

static char *ibp;
static int textmode = FALSE;
static int cnt = 0;
static char buf[1024];

/*
** Skew (interleaving) table
*/

static int skew[26] = {
    4,10,16,22, 2, 8,
   14,20, 0, 6,12,18,
   24, 5,11,17,23, 3,
    9,15,21, 1, 7,13,
   19,25
};

static unsigned char datablock[SECLEN];
static unsigned char fcbblock[SECLEN];

#define HEXDUMP(file,ptr,size,offset)	\
{\
   int jjj;\
   int iii;\
   char *tp;\
   char *cp;\
   for (iii = 0, tp = (char *)(ptr), cp = (char *)(ptr); iii < (size); )\
   {\
      fprintf ((file), "%04X  ", iii+offset);\
      for (jjj = 0; jjj < 8; jjj++)\
      {\
	 if (cp < ((char *)(ptr)+(size)))\
	 {\
	    fprintf ((file), "%02.2X", *cp++ & 0xFF);\
	    if (cp < ((char *)(ptr)+(size)))\
	    {\
	       fprintf ((file), "%02.2X ", *cp++ & 0xFF);\
	    }\
	    else\
	    {\
	       fprintf ((file), "   ");\
	    }\
	 }\
	 else\
	 {\
	    fprintf ((file), "     ");\
	 }\
	 iii += 2;\
      }\
      fprintf ((file), "   ");\
      for (jjj = 0; jjj < 8; jjj++)\
      {\
	 if (tp < ((char *)(ptr)+(size)))\
	 {\
	    if (isprint(*tp))\
	       fprintf ((file), "%c", *tp);\
	    else\
	       fprintf ((file), ".");\
	    tp++;\
	    if (tp < ((char *)(ptr)+(size)))\
	    {\
	       if (isprint(*tp))\
		  fprintf ((file), "%c ", *tp);\
	       else\
		  fprintf ((file), ". ");\
	       tp++;\
	    }\
	    else\
	    {\
	       fprintf ((file), "  ");\
	    }\
	 }\
	 else\
	 {\
	    fprintf ((file), "   ");\
	 }\
      }\
      fprintf ((file), "\n");\
   }\
}

/***********************************************************************
* cvtau - Convert AU address into Track/sector
***********************************************************************/

static int
cvtau (int au, int aus, int *ntrk, int *nsec)
{
   int trk;
   int sec;

   if (au < MAXAUS && aus < SECPERAU)
   {
      trk = (au * SECPERAU + aus) / SECPERTRK;
      sec = (au * SECPERAU + aus) % SECPERTRK;
      sec = skew[sec] + 1;
      *ntrk = trk;
      *nsec = sec;
#ifdef DEBUG
      fprintf (stderr, "   trk = %d, sec = %d\n", trk, sec);
#endif
      return (0);
   }
   return (-1);
}

/***********************************************************************
* readdisk - Read the disk
***********************************************************************/

static int
readdisk (FILE *diskfd, int au, int aus, char *buffer)
{
   int diskloc;
   int trk, sec;

#ifdef DEBUG
   fprintf (stderr, "readdisk: au = %d, aus = %d\n", au, aus);
#endif
   cvtau (au, aus, &trk, &sec);
   diskloc = ((SECLEN * SECPERTRK) * trk) + (SECLEN * (sec-1));
   if (fseek (diskfd, diskloc, SEEK_SET) < 0)
   {
      perror("Disk seek error");
      return (-1);
   }

   if (SECLEN != fread (buffer, 1, SECLEN, diskfd))
   {
      perror ("Disk read error");
      return (-1);
   }
#ifdef DEBUG
   fprintf (stderr, "Buffer:\n");
   HEXDUMP (stderr, buffer, SECLEN, 0);
#endif

   return (SECLEN);

}

/***********************************************************************
* getdisk - Get disk data and process into records.
***********************************************************************/

static int
getdisk (FILE *diskfd, int start, int numaus, int offset)
{
   int incompress;
   int binescape;
   int i, j, k;

   incompress = FALSE;
   binescape = FALSE;
   for (i = 0; i < numaus; i++)
   {
      for (j = offset; j < SECPERAU; j++)
      {
	 unsigned char *cp;

	 offset = 0;
         if (readdisk (diskfd, start+i, j, datablock) < 0)
	    return (-1);

	 cp = datablock + 2;
	 for (k = 2; k < SECLEN; k++)
	 {
	    unsigned char ch;

	    ch = *cp++;
	    if (binescape)
	    {
	       binescape = FALSE;
	       if (ch == 0) ch = 0xFA;
	       *ibp++ = ch;
	       cnt++;
	    }
	    else if (incompress)
	    {
	       incompress = FALSE;
	       for (; ch; ch--) { *ibp++ = ' '; cnt++; }
	    }
	    else switch (ch)
	    {
	    case ENDOFRECORD: /* 0xFF - End of record */
	       if (textmode)
	       {
	          for (ibp--; cnt && (*ibp == ' '); ibp--) cnt--;
		  ibp++;
		  cnt++;
	       }
	       else cnt++;
	       *ibp++ = '\n';
	       fwrite (buf, 1, cnt, stdout);
	       ibp = buf;
	       cnt = 0;
	       break;

	    case COMPBLANKS: /* 0xFE - Compressed blank */
	       incompress = TRUE;
	       break;

	    case 0xFC: /* ??? - Can't remember */
	       break;

	    case ENDOFFILE: /* 0xFB - End of file */
	       if (cnt)
	       {
		  *ibp++ = '\n';
		  fwrite (buf, 1, cnt, stdout);
	       }
	       return (EOF);

	    case BINARYESCAPE: /* 0xFA - Binary escape */
	       binescape = TRUE;
	       break;

	    default:
	       if (textmode)
	       {
		  if (isprint(ch) || ch == 0x0C)
		  {
		     *ibp++ = ch;
		     cnt++;
		  }
	       }
	       else
	       {
		  *ibp++ = ch;
		  cnt++;
	       }
	    }
	 }
      }
   }
   return (0);
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *diskfd;
   char *diskname = NULL;
   char *filename = NULL;
   char *bp;
   int done;
   int i, j, k, l;
   char temp[256];

   /*
   ** Process args
   */

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 't':
	    textmode = TRUE;
	    break;

         default:
      USAGE:
	    printf ("usage: fpyfile [-t] ti990.fpy ti990.file\n");
	    return (1);
         }
      }
      else
      {
         if (diskname == NULL) 
	    diskname = argv[i];
         else if (filename == NULL)
	 {
	    filename = argv[i];
	    for (j = 0; j < strlen(filename); j++)
	       if (islower(filename[j])) filename[j] = toupper(filename[j]);
	 }
         else goto USAGE;
      }
   }
   if (diskname == NULL || filename == NULL) goto USAGE;
   
   if ((diskfd = fopen (diskname, "r")) == NULL)
   {
      sprintf (temp, "fpyfile: Can't open disk image: %s", diskname);
      perror (temp);
      return (1);
   }

   ibp = buf;

   /*
   ** Scan directory for filename
   */

   for (i = 0; i < SECPERAU; i++)
   {
      readdisk (diskfd, DISKDIRAU, i, datablock);
      for (j = 0; j < DIRENTPSEC; j++)
      {
	 if (strncmp (&datablock[j*DIRENTSIZE], "..", 2))
	 {
	    char file[DIRFILELEN+1];
	    char ext[DIREXTLEN+1];

	    strncpy (file, &datablock[j*DIRENTSIZE+DIRFILE], DIRFILELEN);
	    file[DIRFILELEN] = '\0';
	    for (l = 0; l < DIRFILELEN; l++) if (file[l] == ' ') file[l] = '\0';
	    strncpy (ext, &datablock[j*DIRENTSIZE+DIREXT], DIREXTLEN);
	    ext[DIREXTLEN] = '\0';
	    for (l = 0; l < DIREXTLEN; l++) if (ext[l] == ' ') ext[l] = '\0';
	    sprintf (temp, "%s.%s", file, ext);

	    /*
	    ** If this is the file, then process it
	    */

	    if (!strcmp (temp, filename))
	    {
	       int fcbaddr;
	       int start, numaus;

	       /*
	       ** Get file control block
	       */

	       fcbaddr = (datablock[j*DIRENTSIZE+DIRFCB] << 8) |
			  datablock[j*DIRENTSIZE+DIRFCB+1];
#ifdef DEBUG
               fprintf (stderr, " read fcb: fcbaddr = %d\n", fcbaddr);
#endif
	       readdisk (diskfd, fcbaddr, 0, fcbblock);
	       fcbaddr = FCBEXTENT;
	       k = 1;

	       /*
	       ** Read the sectors and process
	       */

	       done = FALSE;
	       while (!done)
	       { 
		  start = (fcbblock[fcbaddr] << 8) | fcbblock[fcbaddr+1];
#ifdef DEBUG
		  fprintf (stderr, "   start = %d\n", start);
#endif
		  if (start == 0xFFFF) break; /* Last extent */
		  numaus = (fcbblock[fcbaddr+2] << 8) | fcbblock[fcbaddr+3];
		  fcbaddr += 4;
#ifdef DEBUG
		  fprintf (stderr, "   numaus = %d\n", numaus);
#endif
	          if (getdisk (diskfd, start, numaus, k) == EOF) done = TRUE;
		  k = 0;
	       }
	       return (0);
	    }
	 }
      }
   }

   printf ("File %s not found on disk %s\n", filename, diskname);
   return (1);

}
