/***********************************************************************
*
* simtape.c - TILINE Tape IO processing for the TI 990 Simulator.
*
* Changes:
*   11/18/03   DGP   Original.
*   12/10/03   DGP   Change TPCS usage.
*   12/30/03   DGP   Added non-rom boot code.
*   05/10/04   DGP   Added DEV_NULL support.
*   07/29/04   DGP   Added tapereset function.
*
* Tape record format:
*   4 bytes (little-endian): block length
*   N bytes: raw data for the record
*   4 bytes (little-endian): repeat block length
*
* Tape mark (EOF):
*   4 bytes: zero data
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (_WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#endif

#if defined(UNIX)
#include <sys/time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#endif

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */

extern int run;
extern int idle;
extern int fault;
extern int devcnt;
extern int model;
extern int mapenabled;
extern int pancount;
extern uint32 memlen;
extern char view[MAXVIEW][81];

extern uint8 memory[MEMSIZE];
extern Device devices[MAXDEVICES];

/*
** Command words access macros
*/

#define GETW0(d)   GETMEM0((d)->devaddr+TPCSSTART)
#define GETW7(d)   GETMEM0((d)->devaddr+TPCSSTART+14)

#define PUTW0(d,v) PUTMEM0((d)->devaddr+TPCSSTART,v)
#define PUTW7(d,v) PUTMEM0((d)->devaddr+TPCSSTART+14,v)

#define GETLEN(d)  GETMEM0((d)->devaddr+TPCSSTART+8)
#define GETADDR(d) \
(GETMEMB0((d)->devaddr+TPCSSTART+13)<<16)|GETMEM0((d)->devaddr+TPCSSTART+10)
#define GETCMD(d)  GETMEMB0((d)->devaddr+TPCSSTART+12) & 0xF
#define GETUNIT(d) (GETMEMB0((d)->devaddr+TPCSSTART+12) >> 4) & 0xF

/*
** Command word control bits
*/

#define OLBIT	0x8000
#define BOTBIT	0x4000
#define EORBIT	0x2000
#define EOFBIT	0x1000
#define EOTBIT	0x0800
#define WPBIT	0x0400
#define RWBIT	0x0200
#define TOBIT	0x0100

#define IDLEBIT	0x8000
#define COMPBIT	0x4000
#define ERRBIT	0x2000
#define INTBIT	0x1000
#define sp0BIT	0x0800
#define sp1BIT	0x0400
#define PEFBIT	0x0200
#define ABNBIT	0x0100

#define IPEBIT	0x0080
#define ECEBIT	0x0040
#define HWEBIT	0x0020
#define TMRBIT	0x0010
#define TTEBIT	0x0008
#define TTOBIT	0x0004
#define FMEBIT	0x0002
#define TPEBIT	0x0001

/***********************************************************************
* tapereadint - Read an integer.
***********************************************************************/

static int
tapereadint (FILE *fd)
{
   int r;

   r = fgetc (fd);
   r = r | (fgetc (fd) << 8);
   r = r | (fgetc (fd) << 16);
   r = r | (fgetc (fd) << 24);
   if (feof(fd)) return (-2);
   return (r & 0x7FFF);
}

/***********************************************************************
* tapewriteint - Write an integer.
***********************************************************************/

static void
tapewriteint (FILE *fd, int v)
{
   v &= 0x7FFF;
   fputc (v & 0xFF, fd);
   fputc ((v >> 8) & 0xFF, fd);
   fputc ((v >> 16) & 0xFF, fd);
   fputc ((v >> 24) & 0xFF, fd);
}

/***********************************************************************
* tapeopen - Open tape.
***********************************************************************/

int 
tapeopen (Device *dev)
{
#ifdef DEBUGTAPE
   fprintf (stderr, "tapeopen: ENTERED\n");
   fprintf (stderr, "   file = %s\n", dev->file);
#endif

   if (dev->fd == DEV_NULL)
   {
      PUTW0 (dev, OLBIT | WPBIT);
      PUTW7 (dev, IDLEBIT); /* Mark idle */
      return (0);
   }

   /*
   ** Seek to beginning of the tape file.
   */

   if (fseek (dev->fd, 0, SEEK_SET) < 0)
   {
      sprintf (view[0], "tapeopen: seek failed: %s",
	    strerror (ERRNO));
      sprintf (view[1], "filename: %s", dev->file);
      return (-1);
   }

   PUTW0 (dev, BOTBIT | (dev->info.tapeinfo.writeprotect ? WPBIT : 0));
   PUTW7 (dev, IDLEBIT); /* Mark idle */

   return (0);
}

/***********************************************************************
* tapeputmem - Put data into memory.
***********************************************************************/

static void
tapeputmem (uint32 ma, uint16 v)
{
   if (ma < memlen)
   {
      memory[ma+MSB] = (v >> 8) & 0xFF;
      memory[ma+LSB] = v & 0xFF;
   }
}

/***********************************************************************
* tapegetmem - Get data from memory.
***********************************************************************/

static uint16
tapegetmem (uint32 ma)
{
   if (ma >= memlen) return (0);

   return (((memory[ma+MSB] & 0xFF) << 8)
          | (memory[ma+LSB] & 0xFF));
}

/***********************************************************************
* tapereset - Reset device
***********************************************************************/

void
tapereset (Device *dev)
{
#ifdef DEBUGTAPE
   fprintf (stderr, "tapereset: ENTERED\n");
#endif

   if (dev->fd == DEV_NULL)
   {
      PUTW0 (dev, OLBIT | WPBIT);
   }
   else
   {
      PUTW0 (dev, BOTBIT | (dev->info.tapeinfo.writeprotect ? WPBIT : 0));
   }

   PUTW7 (dev, IDLEBIT | ERRBIT | ABNBIT ); 

   return;
}

/***********************************************************************
* tapedocmd - Do disk command
***********************************************************************/

int 
tapedocmd (Device *dev)
{
   Device *udev;
   int cmd;
   int i;
   int reclen, len;
   size_t cp;
   uint32 addr, saddr;
   uint16 W0, W7;
   uint8 unit;

   /*
   ** Get common command word values
   */

   W0 = GETW0 (dev);
   W7 = GETW7 (dev);

   cmd = GETCMD (dev);
   unit = GETUNIT (dev);
   saddr = addr = GETADDR(dev) /*(GETADDR (dev) + 1) & 0x1FFFFE */;
   len = GETLEN(dev) /*(GETLEN (dev) + 1) & 0xFFFE*/;

#ifdef DEBUGTAPE
   fprintf (stderr,
	    "tapedocmd: ENTRY: cmd = %d, unit = %02X, addr = %06X, len = %d\n",
	    cmd, unit, addr, len);
   fprintf (stderr, "   pc = %04X, wp = %04X, st = %04X\n",
	    pcreg, wpreg, statreg);
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	    dev->devaddr+TPCSSTART+i, GETMEM0 (dev->devaddr+TPCSSTART+i));
#endif

   /*
   ** Find selected unit
   */

   for (i = 0; i < devcnt; i++) /* find selected unit */
      if (devices[i].devaddr == dev->devaddr && devices[i].unit == unit)
	 break;
   if (i == devcnt)
   {
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: unit = %d: Unit not selected\n", unit);
#endif
      /* Drive not ready */
      W0 |= OLBIT | (dev->info.tapeinfo.writeprotect ? WPBIT : 0);
      W7 |= ERRBIT | IDLEBIT;
      PUTW0 (dev, W0);
      PUTW7 (dev, W7);
      return(0);
   }
   udev = &devices[i];

   if (udev->fd == DEV_NULL)
   {
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: unit = %d: Unit is NULL\n", unit);
#endif
      PUTW0 (udev, OLBIT | WPBIT);
      PUTW7 (udev, IDLEBIT);
      return (0);
   }

   /*
   ** Set initial sate
   */

   udev->select = TRUE;
   udev->intenabled = FALSE;

   /*
   ** Process command
   */

   switch (GETCMD (udev))
   {
   case 0: /* NOP */
   case 12: /* NOP */
   case 13: /* NOP */
   case 14: /* NOP */
      break;

   case 1: /* Buffer Sync */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: FORMAT: \n");
#endif
      break;

   case 2: /* Write EOF */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: WRITEEOF: \n");
#endif
      if (!udev->info.tapeinfo.writeprotect)
      {

	 tapewriteint (udev->fd, 0);

	 W0 |= EOTBIT | OLBIT;
      }
      break;

   case 3: /* Record skip backward */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: BCKSKIP: \n");
#endif
      while (len)
      {
	 cp = ftell (udev->fd) - 4;

	 if (fseek (udev->fd, cp, SEEK_SET) < 0)
	 {
	    sprintf (view[0], "tapedocmd: BCKSKIP: seek failed: %s",
		  strerror (ERRNO));
	    sprintf (view[1], "filename: %s", udev->file);
	    return (0);
	 }
	 reclen = tapereadint (udev->fd);
	 if (reclen > 0)
	    cp -= reclen - 4;
	 if (fseek (udev->fd, cp, SEEK_SET) < 0)
	 {
	    sprintf (view[0], "tapedocmd: BCKSKIP: seek failed: %s",
		  strerror (ERRNO));
	    sprintf (view[1], "filename: %s", udev->file);
	    return (0);
	 }
	 len --;
      }
      break;

   case 4: /* Binary read data */
      reclen = tapereadint (udev->fd);
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: READ: reclen = %d\n", reclen);
#endif

      if (reclen > 0)
      {
	 for (i = 0; i < reclen;)
	 {
	    uint16 d, c;

	    d = fgetc (udev->fd);
	    i++;
	    if (i < reclen)
	    {
	       c = fgetc (udev->fd);
	       i++;
	    }
	    else
	       c = 0xFF;
	    d = (d << 8) | c;
	    if (i < len)
	       tapeputmem (addr, d);
	    addr += 2;
	 }
	 if (i < len)
	 {
	    W0 |= EORBIT;
	    W7 |= TPEBIT;
	 }
	 i = tapereadint (udev->fd);
#ifdef DEBUGTAPE
      fprintf (stderr, "   traillen = %d\n", i);
#endif
         if (i != reclen)
	 {
	    W7 |= ERRBIT;
	 }

#ifdef DEBUGTAPEDATA
	 HEXDUMP (stderr, &memory[saddr], reclen, saddr);
#endif
      }
      else if (reclen == 0)
      {
	 W7 |= ERRBIT | TPEBIT;
	 W0 |= EOFBIT;
      }
      else if (reclen == -2)
      {
	 W7 |= ERRBIT | TPEBIT;
	 W0 |= EOTBIT;
      }
      break;

   case 5: /* Record skip forward */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: FWDSKIP: \n");
#endif

      udev->info.tapeinfo.curpos = ftell (udev->fd);

      while (len)
      {
	 reclen = tapereadint (udev->fd);
	 if (reclen > 0)
	 {
	    for (i = 0; i < reclen; i++)
	    {
	       fgetc (udev->fd);
	    }
	    i = tapereadint (udev->fd);
	    if (i != reclen)
	    {
	       W7 |= ERRBIT;
	       break;
	    }
	 }
	 else if (reclen == 0)
	 {
	    W7 |= ERRBIT | TPEBIT;
	    W0 |= EOFBIT;
	    break;
	 }
	 else if (reclen == -2)
	 {
	    W7 |= ERRBIT | TPEBIT;
	    W0 |= EOTBIT;
	    break;
	 }
	 len--;
      }
      break;

   case 6: /* Binary write data */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: WRITE: \n");
#endif
      if (!udev->info.tapeinfo.writeprotect)
      {
#ifdef DEBUGTAPEDATA
	 HEXDUMP (stderr, &memory[saddr], GETLEN(udev), saddr);
#endif
	 tapewriteint (udev->fd, len);

	 for (i = 0; i < len; )
	 {
	    uint16 d;

	    d = tapegetmem (addr);
	    fputc ((d >> 8) & 0xFF, udev->fd);
	    i++;
	    if (i < len)
	    {
	       fputc (d & 0xFF, udev->fd);
	       i++;
	    }
	    addr += 2;
	 }
	 tapewriteint (udev->fd, len);
      }
      break;

   case 7: /* Erase */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: ERASE: \n");
#endif
      break;

   case 8: /* Read transport status */
   case 9: /* Read transport status */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: READST: \n");
#endif
      break;

   case 10: /* Rewind */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: REWIND: \n");
#endif
      if (fseek (udev->fd, 0, SEEK_SET) < 0)
      {
	 sprintf (view[0], "tapedocmd: REWIND: seek failed: %s",
	       strerror (ERRNO));
	 sprintf (view[1], "filename: %s", udev->file);
	 return (0);
      }

      W0 |= (udev->unit << 4) | BOTBIT;
      W7 |= COMPBIT | TPEBIT;
      break;

   case 11: /* Rewind and offline */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: RESTORE: \n");
#endif
      if (fseek (udev->fd, 0, SEEK_SET) < 0)
      {
	 sprintf (view[0], "tapedocmd: RESTORE: seek failed: %s",
	       strerror (ERRNO));
	 sprintf (view[1], "filename: %s", udev->file);
	 return (0);
      }

      W0 |= (udev->unit << 4) | BOTBIT | OLBIT;
      W7 |= COMPBIT | TPEBIT;
      break;

   }

   if (W7 & INTBIT)
      udev->intenabled = TRUE;
   W7 |= COMPBIT | IDLEBIT;
   W0 |= (udev->info.tapeinfo.writeprotect ? WPBIT : 0);
   PUTW0 (udev, W0);
   PUTW7 (udev, W7);

   return (udev->intenabled ? udev->intlvl : 0);
}


/***********************************************************************
* tapeboot - Boot from device.
***********************************************************************/

int
tapeboot (Device *dev)
{
   run = FALSE;
   SET_MASK (0);

#ifdef USETAPEROM

   if (model < 10)
   {
      wpreg = GETMEM0 (0xFFFC) & 0xFFFE;
      pcreg = GETMEM0 (0xFFFE) & 0xFFFE;
   }
   else
   {
      wpreg = GETMEM0 (TPCSSTART+0xFFFC) & 0xFFFE;
      pcreg = GETMEM0 (TPCSSTART+0xFFFE) & 0xFFFE;
   }

#else /* !USETAPEROM */

   {
      FILE *lfd = NULL;
      int lp, clp;
      int i;
      int reclen;
      char inbuf[82];

      if (dev->fd == DEV_NULL)
      {
	 sprintf (view[0], "tapeboot: Can't boot %s device", dev->file);
         run = FALSE;
	 return (-1);
      }

      clp = lp = LOADADDRESS;

#ifdef DEBUGLOADER
      lfd = fopen ("tapeload.log", "w");
      fprintf (lfd, "tapeboot: file = %s, loadpd = %d(%04X)\n",
	       dev->file, lp, lp);
#endif

      if (fseek (dev->fd, 0, SEEK_SET) < 0)
      {
	 sprintf (view[0], "tapeboot: seek failed: %s",
	       strerror (ERRNO));
	 sprintf (view[1], "filename: %s", dev->file);
	 return (-1);
      }
      i = 0;

      while ((reclen = tapereadint (dev->fd)) == 80)
      {
	 if (fread (inbuf, 1, reclen, dev->fd) != reclen)
	 {
	    reclen = 0;
	    break;
	 }
#ifdef DUMPLOADREC
	 fprintf (lfd, "Record: %d\n", ++i);
	 HEXDUMP (lfd, inbuf, 80, 0);
#endif
	 if (tapereadint (dev->fd) != reclen)
	 {
	    reclen = 0;
	    break;
	 }
	 if (loadrec (lfd, inbuf, lp, &clp) == EOF) break;
      }

#ifdef DEBUGLOADER
      fclose (lfd);
#endif
      if (reclen != 80)
      {
	 sprintf (view[0], "tapeboot: read failed: %s",
	       strerror (ERRNO));
	 sprintf (view[1], "filename: %s", dev->file);
         run = FALSE;
	 return (-1);
      }
   }

#endif /* USETAPEROM */

   run = TRUE;

   return (0);
}

