/***********************************************************************
*
* simeops.c - Simulate extended /12 operations for the TI 990 computer.
*
* Changes:
*   06/16/04   DGP   Original.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 opcreg;	/* The old program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */
extern uint32 mpcreg;	/* Mapped PC */
extern int breakaddr;   /* The break point address */
extern long pancount;	/* Instructions executed for panel */
extern unsigned long instcount;

extern int run;
extern int idle;
extern int runled;
extern int idleled;
extern int model;
extern int breakflag;
extern int enablepanel;
extern int mapenabled;
extern int traceenable;
extern int tracestart;
extern int traceend;
extern int tracelim[2];
extern int deferint;
extern int traceit;
extern uint16 mapcrudata;
extern uint16 errcrudata;
extern uint32 memlen;
extern uint32 maplatch;
extern char view[MAXVIEW][81];
extern FILE *tracefd;

extern uint8 memory[MEMSIZE];
extern MapFile mapfile[MAPSIZE];

/***********************************************************************
* getlongreg - Get a long register value.
***********************************************************************/

static uint32
getlongreg (uint16 r)
{
   return (((uint32)GETREG(r) << 16) | (uint32)GETREG(r+1));
}

/***********************************************************************
* getquadreg - Get a quad register value.
***********************************************************************/

static t_uint64
getquadreg (uint16 r)
{
   return (((t_uint64)GETREG(r) << 48) | ((t_uint64)GETREG(r+1) << 32) |
	   ((t_uint64)GETREG(r+2) << 16) | (t_uint64)GETREG(r+3));
}

/***********************************************************************
* putlongreg - Put a long value into a register.
***********************************************************************/

static void
putlongreg (uint16 r, uint32 v)
{
   PUTREG (r, (uint16)((v >> 16) & 0xFFFF));
   PUTREG (r+1, (uint16)(v & 0xFFFF));
}

/***********************************************************************
* putquadreg - Put a quad value into a register.
***********************************************************************/

static void
putquadreg (uint16 r, t_uint64 v)
{
   PUTREG (r, (uint16)((v >> 48) & 0xFFFF));
   PUTREG (r+1, (uint16)((v >> 32) & 0xFFFF));
   PUTREG (r+2, (uint16)((v >> 16) & 0xFFFF));
   PUTREG (r+3, (uint16)(v & 0xFFFF));
}

/***********************************************************************
* getlong - Get a long operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

uint32
getlong (uint16 sr, uint16 st, int inc, int srcdst)
{
   uint16 sa;
   uint32 sval;

   switch (st)
   {
   case 0:

      if (traceit && inc)
	 fprintf (tracefd, " R%d", sr);
      sval = getlongreg(sr);
      break;

   case 1:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d", sr);
      sa = GETREG (sr);
      sval = ((uint32)GETMEM (sa & 0xFFFE, srcdst) << 16) | 
	      (uint32)GETMEM ((sa+2) & 0xFFFE, srcdst);
      break;

   case 2:

      sa = GETINST;
      if (traceit && inc)
      {
	 if (sr == 0)
	    fprintf (tracefd, " @>%X", sa);
	 else
	    fprintf (tracefd, " @>%X(%d)", sa, sr);
      }
      if (inc) pcreg += 2;
      if (sr != 0) (int16)sa += (int16)GETREG (sr);
      sval = ((uint32)GETMEM (sa & 0xFFFE, srcdst) << 16) | 
	      (uint32)GETMEM ((sa+2) & 0xFFFE, srcdst);
      break;

   case 3:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d+", sr);
      sa = GETREG (sr);
      sval = ((uint32)GETMEM (sa & 0xFFFE, srcdst) << 16) | 
	      (uint32)GETMEM ((sa+2) & 0xFFFE, srcdst);
      if (inc) PUTREG (sr, GETREG (sr) + 4);
      break;
   }
   return (sval);
}

/***********************************************************************
* getstack - Get a stack operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

void
getstack (uint16 sr, uint16 st, int inc, int srcdst, uint16 *stk)
{
   uint16 sa;

   switch (st)
   {
   case 0:

      if (traceit && inc)
	 fprintf (tracefd, " R%d", sr);
      stk[0] = GETREG(sr);
      stk[1] = 0x0000;
      stk[2] = 0xFFFF;
      break;

   case 1:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d", sr);
      sa = GETREG (sr) & 0xFFFE;
      stk[0] = GETMEM (sa,     srcdst);
      stk[1] = GETMEM (sa + 2, srcdst);
      stk[2] = GETMEM (sa + 4, srcdst);
      break;

   case 2:

      sa = GETINST;
      if (traceit && inc)
      {
	 if (sr == 0)
	    fprintf (tracefd, " @>%X", sa);
	 else
	    fprintf (tracefd, " @>%X(%d)", sa, sr);
      }
      if (inc) pcreg += 2;
      if (sr != 0) (int16)sa += (int16)GETREG (sr);
      sa &= 0xFFFE;
      stk[0] = GETMEM (sa,     srcdst);
      stk[1] = GETMEM (sa + 2, srcdst);
      stk[2] = GETMEM (sa + 4, srcdst);
      break;

   case 3:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d+", sr);
      sa = GETREG (sr) & 0xFFFE;
      stk[0] = GETMEM (sa,     srcdst);
      stk[1] = GETMEM (sa + 2, srcdst);
      stk[2] = GETMEM (sa + 4, srcdst);
      if (inc) PUTREG (sr, GETREG (sr) + 6);
      break;
   }
   return;
}

/***********************************************************************
* gettbl - Get a table operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

void
gettbl (uint16 sr, uint16 st, int inc, int srcdst, uint16 *tbl)
{
   uint16 sa;

   switch (st)
   {
   case 0:

      if (traceit && inc)
	 fprintf (tracefd, " R%d", sr);
      tbl[0] = GETREG(sr);
      tbl[1] = GETREG(sr+1);
      break;

   case 1:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d", sr);
      sa = GETREG (sr) & 0xFFFE;
      tbl[0] = GETMEM (sa,     srcdst);
      tbl[1] = GETMEM (sa + 2, srcdst);
      break;

   case 2:

      sa = GETINST;
      if (traceit && inc)
      {
	 if (sr == 0)
	    fprintf (tracefd, " @>%X", sa);
	 else
	    fprintf (tracefd, " @>%X(%d)", sa, sr);
      }
      if (inc) pcreg += 2;
      if (sr != 0) (int16)sa += (int16)GETREG (sr);
      sa &= 0xFFFE;
      tbl[0] = GETMEM (sa,     srcdst);
      tbl[1] = GETMEM (sa + 2, srcdst);
      break;

   case 3:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d+", sr);
      sa = GETREG (sr) & 0xFFFE;
      tbl[0] = GETMEM (sa,     srcdst);
      tbl[1] = GETMEM (sa + 2, srcdst);
      if (inc) PUTREG (sr, GETREG (sr) + 4);
      break;
   }
   return;
}

/***********************************************************************
* getlscb - Get a lscb operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

void
getlscb (uint16 sr, uint16 st, int inc, int srcdst, uint16 *lscb)
{
   uint16 sa;

   switch (st)
   {
   case 0:

      if (traceit && inc)
	 fprintf (tracefd, " R%d", sr);
      lscb[0] = GETREG(sr);
      lscb[1] = GETREG(sr+1);
      lscb[2] = GETREG(sr+2);
      lscb[3] = GETREG(sr+3);
      lscb[4] = GETREG(sr+4);
      break;

   case 1:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d", sr);
      sa = GETREG (sr) & 0xFFFE;
      lscb[0] = GETMEM (sa,     srcdst);
      lscb[1] = GETMEM (sa + 2, srcdst);
      lscb[2] = GETMEM (sa + 4, srcdst);
      lscb[3] = GETMEM (sa + 6, srcdst);
      lscb[4] = GETMEM (sa + 8, srcdst);
      break;

   case 2:

      sa = GETINST;
      if (traceit && inc)
      {
	 if (sr == 0)
	    fprintf (tracefd, " @>%X", sa);
	 else
	    fprintf (tracefd, " @>%X(%d)", sa, sr);
      }
      if (inc) pcreg += 2;
      if (sr != 0) (int16)sa += (int16)GETREG (sr);
      sa &= 0xFFFE;
      lscb[0] = GETMEM (sa,     srcdst);
      lscb[1] = GETMEM (sa + 2, srcdst);
      lscb[2] = GETMEM (sa + 4, srcdst);
      lscb[3] = GETMEM (sa + 6, srcdst);
      lscb[4] = GETMEM (sa + 8, srcdst);
      break;

   case 3:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d+", sr);
      sa = GETREG (sr) & 0xFFFE;
      lscb[0] = GETMEM (sa,     srcdst);
      lscb[1] = GETMEM (sa + 2, srcdst);
      lscb[2] = GETMEM (sa + 4, srcdst);
      lscb[3] = GETMEM (sa + 6, srcdst);
      lscb[4] = GETMEM (sa + 8, srcdst);
      if (inc) PUTREG (sr, GETREG (sr) + 10);
      break;
   }
   return;
}

/***********************************************************************
* getquad - Get a quad operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

static t_uint64
getquad (uint16 sr, uint16 st, int inc, int srcdst)
{
   uint16 sa;
   t_uint64 sval;

   switch (st)
   {
   case 0:

      if (traceit && inc)
	 fprintf (tracefd, " R%d", sr);
      sval = getquadreg(sr);
      break;

   case 1:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d", sr);
      sa = GETREG (sr);
      sval = ((t_uint64)GETMEM (sa & 0xFFFE, srcdst) << 48) | 
	     ((t_uint64)GETMEM ((sa+2) & 0xFFFE, srcdst) << 32) |
	     ((t_uint64)GETMEM ((sa+4) & 0xFFFE, srcdst) << 16) |
	      (t_uint64)GETMEM ((sa+6) & 0xFFFE, srcdst);
      break;

   case 2:

      sa = GETINST;
      if (traceit && inc)
      {
	 if (sr == 0)
	    fprintf (tracefd, " @>%X", sa);
	 else
	    fprintf (tracefd, " @>%X(%d)", sa, sr);
      }
      if (inc) pcreg += 2;
      if (sr != 0) (int16)sa += (int16)GETREG (sr);
      sval = ((t_uint64)GETMEM (sa & 0xFFFE, srcdst) << 48) | 
	     ((t_uint64)GETMEM ((sa+2) & 0xFFFE, srcdst) << 32) |
	     ((t_uint64)GETMEM ((sa+4) & 0xFFFE, srcdst) << 16) |
	      (t_uint64)GETMEM ((sa+6) & 0xFFFE, srcdst);
      break;

   case 3:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d+", sr);
      sa = GETREG (sr);
      sval = ((t_uint64)GETMEM (sa & 0xFFFE, srcdst) << 48) | 
	     ((t_uint64)GETMEM ((sa+2) & 0xFFFE, srcdst) << 32) |
	     ((t_uint64)GETMEM ((sa+4) & 0xFFFE, srcdst) << 16) |
	      (t_uint64)GETMEM ((sa+6) & 0xFFFE, srcdst);
      if (inc) PUTREG (sr, GETREG (sr) + 8);
      break;
   }
   return (sval);
}

/***********************************************************************
* putlong - Put a long operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

static void
putlong (uint16 dr, uint16 dt, uint32 dval, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      if (traceit)
	 fprintf (tracefd, " R%d", dr);
      putlongreg (dr, dval);
      break;

   case 1:

      if (traceit)
	 fprintf (tracefd, " *R%d", dr);
      da = GETREG (dr);
      PUTMEM (da, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)(dval & 0xFFFF), srcdst);
      break;

   case 2:

      da = GETINST;
      if (traceit)
      {
	 if (dr == 0)
	    fprintf (tracefd, " @>%X", da);
	 else
	    fprintf (tracefd, " @>%X(%d)", da, dr);
      }
      pcreg += 2;
      if (dr != 0) (int16)da += (int16)GETREG (dr);
      PUTMEM (da, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)(dval & 0xFFFF), srcdst);
      break;

   case 3:

      if (traceit)
	 fprintf (tracefd, " *R%d+", dr);
      da = GETREG (dr);
      PUTREG (dr, GETREG (dr) + 4);
      PUTMEM (da, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)(dval & 0xFFFF), srcdst);
      break;
   }
}

/***********************************************************************
* putstack - Put a stack operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

static void
putstack (uint16 dr, uint16 dt, uint16 *stk, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      if (traceit)
	 fprintf (tracefd, " R%d", dr);
      PUTREG (dr,   stk[0]);
      break;

   case 1:

      if (traceit)
	 fprintf (tracefd, " *R%d", dr);
      da = GETREG (dr) & 0xFFFE;
      PUTMEM (da,   stk[0], srcdst);
      PUTMEM (da+2, stk[1], srcdst);
      PUTMEM (da+4, stk[2], srcdst);
      break;

   case 2:

      da = GETINST;
      if (traceit)
      {
	 if (dr == 0)
	    fprintf (tracefd, " @>%X", da);
	 else
	    fprintf (tracefd, " @>%X(%d)", da, dr);
      }
      pcreg += 2;
      if (dr != 0) (int16)da += (int16)GETREG (dr);
      da &= 0xFFFE;
      PUTMEM (da,   stk[0], srcdst);
      PUTMEM (da+2, stk[1], srcdst);
      PUTMEM (da+4, stk[2], srcdst);
      break;

   case 3:

      if (traceit)
	 fprintf (tracefd, " *R%d+", dr);
      da = GETREG (dr) & 0xFFFE;
      PUTMEM (da,   stk[0], srcdst);
      PUTMEM (da+2, stk[1], srcdst);
      PUTMEM (da+4, stk[2], srcdst);
      PUTREG (dr, GETREG (dr) + 6);
      break;
   }
}

/***********************************************************************
* puttbl - Put a table operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

static void
puttbl (uint16 dr, uint16 dt, uint16 *tbl, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      if (traceit)
	 fprintf (tracefd, " R%d", dr);
      PUTREG (dr,   tbl[0]);
      PUTREG (dr+1, tbl[1]);
      break;

   case 1:

      if (traceit)
	 fprintf (tracefd, " *R%d", dr);
      da = GETREG (dr) & 0xFFFE;
      PUTMEM (da,   tbl[0], srcdst);
      PUTMEM (da+2, tbl[1], srcdst);
      break;

   case 2:

      da = GETINST;
      if (traceit)
      {
	 if (dr == 0)
	    fprintf (tracefd, " @>%X", da);
	 else
	    fprintf (tracefd, " @>%X(%d)", da, dr);
      }
      pcreg += 2;
      if (dr != 0) (int16)da += (int16)GETREG (dr);
      da &= 0xFFFE;
      PUTMEM (da,   tbl[0], srcdst);
      PUTMEM (da+2, tbl[1], srcdst);
      break;

   case 3:

      if (traceit)
	 fprintf (tracefd, " *R%d+", dr);
      da = GETREG (dr) & 0xFFFE;
      PUTMEM (da,   tbl[0], srcdst);
      PUTMEM (da+2, tbl[1], srcdst);
      PUTREG (dr, GETREG (dr) + 4);
      break;
   }
}

/***********************************************************************
* putquad - Put a quad operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

static void
putquad (uint16 dr, uint16 dt, t_uint64 dval, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      if (traceit)
	 fprintf (tracefd, " R%d", dr);
      putquadreg (dr, dval);
      break;

   case 1:

      if (traceit)
	 fprintf (tracefd, " *R%d", dr);
      da = GETREG (dr);
      PUTMEM (da, (uint16)((dval >> 48) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)((dval >> 32) & 0xFFFF), srcdst);
      PUTMEM (da+4, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+6, (uint16)(dval & 0xFFFF), srcdst);
      break;

   case 2:

      da = GETINST;
      if (traceit)
      {
	 if (dr == 0)
	    fprintf (tracefd, " @>%X", da);
	 else
	    fprintf (tracefd, " @>%X(%d)", da, dr);
      }
      pcreg += 2;
      if (dr != 0) (int16)da += (int16)GETREG (dr);
      PUTMEM (da, (uint16)((dval >> 48) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)((dval >> 32) & 0xFFFF), srcdst);
      PUTMEM (da+4, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+6, (uint16)(dval & 0xFFFF), srcdst);
      break;

   case 3:

      if (traceit)
	 fprintf (tracefd, " *R%d+", dr);
      da = GETREG (dr);
      PUTREG (dr, GETREG (dr) + 8);
      PUTMEM (da, (uint16)((dval >> 48) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)((dval >> 32) & 0xFFFF), srcdst);
      PUTMEM (da+4, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+6, (uint16)(dval & 0xFFFF), srcdst);
      break;
   }
}

/***********************************************************************
* cmplongzero - Compare long value to zero.
***********************************************************************/

void
cmplongzero (uint32 val)
{
   if (val > 0) SET_LGT;
   else CLR_LGT;
   if ((int32)val > 0) SET_AGT;
   else CLR_AGT;
   if (val == 0) SET_EQ;
   else CLR_EQ;
}

/***********************************************************************
* cmpquadzero - Compare quad value to zero.
***********************************************************************/

static void
cmpquadzero (t_uint64 val)
{
   if (val > 0) SET_LGT;
   else CLR_LGT;
   if ((t_int64)val > 0) SET_AGT;
   else CLR_AGT;
   if (val == 0) SET_EQ;
   else CLR_EQ;
}

/***********************************************************************
* chkmbzero - Check multibyte value to zero.
***********************************************************************/

static void
chkmbzero (uint8 *dat, uint16 bc)
{
   int i;

   CLR_LGT;
   CLR_AGT;
   SET_EQ;
   for (i = 0; i < bc; i++)
   {
      if (dat[i] != 0)
      {
	 CLR_EQ;
	 break;
      }
   }
   if (!IS_EQ)
   {
      for (i = 0; i < bc; i++)
      {
	 if (dat[i] > 0)
	 {
	    SET_LGT;
	    if (i == 0 && (int8)dat[i] < 0) CLR_AGT;
	    else SET_AGT;
	    break;
	 }
      }
   }
}

/***********************************************************************
* chkmbmemzero - Check multibyte memory value to zero.
***********************************************************************/

static void
chkmbmemzero (uint16 ad, uint16 bc, int srcdst)
{
   int i;

   CLR_AGT;
   CLR_LGT;
   SET_EQ;
   for (i = 0; i < bc; i++)
   {
      if (GETMEMB (ad+i, srcdst) != 0)
      {
	 CLR_EQ;
	 break;
      }
   }
   if (!IS_EQ)
   {
      for (i = 0; i < bc; i++)
      {
	 uint8 dat;

	 dat = GETMEMB (ad+i, srcdst);
	 if (dat > 0)
	 {
	    SET_LGT;
	    SET_AGT;
	    if (i == 0 && (int8)dat < 0) CLR_AGT;
	    break;
	 }
      }
   }
}

/***********************************************************************
* cmpmbstring - Compare multibyte strings.
***********************************************************************/

static void
cmpmbstring (uint8 *sdat, uint8 *ddat, uint16 bc)
{
   int i;

   CLR_AGT;
   CLR_LGT;
   SET_EQ;
   for (i = 0; i < bc; i++)
   {
      if (sdat[i] != ddat[i])
      {
	 CLR_EQ;
	 break;
      }
   }
   if (!IS_EQ)
   {
      for (i = 0; i < bc; i++)
      {
	 if (sdat[i] > ddat[i])
	 {
	    SET_LGT;
	    SET_AGT;
	    if (i == 0 && (int8)sdat[i] < (int8)ddat[i]) CLR_AGT;
	    break;
	 }
	 else if (sdat[i] < ddat[i]) break;
      }
   }
}

/***********************************************************************
* chkvalcc - Check value for cc (condition code).
***********************************************************************/

static int
chkvalcc (uint16 val, uint16 cc, uint16 *lscb)
{
   int result;

   switch (cc)
   {
      case 0: /* EQ */
	 result = (val & lscb[3]) == lscb[2];
	 break;
      case 1: /* NE */
	 result = (val & lscb[3]) != lscb[2];
	 break;
      case 2: /* HE */
	 result = (val & lscb[3]) >= lscb[2];
	 break;
      case 3: /* L */
	 result = (val & lscb[3]) < lscb[2];
	 break;
      case 4: /* GE */
	 result = (int16)(val & lscb[3]) >= (int16)lscb[2];
	 break;
      case 5: /* LT */
	 result = (int16)(val & lscb[3]) < (int16)lscb[2];
	 break;
      case 6: /* LE */
	 result = (val & lscb[3]) <= lscb[2];
	 break;
      case 7: /* H */
	 result = (val & lscb[3]) > lscb[2];
	 break;
      case 8: /* LTE */
	 result = (int16)(val & lscb[3]) <= (int16)lscb[2];
	 break;
      case 9: /* GT */
	 result = (int16)(val & lscb[3]) > (int16)lscb[2];
	 break;
      default: result = FALSE;
   }
   return (result);
}

/***********************************************************************
* chkaddlong - Check addition for carry and overflow
***********************************************************************/

static void
chkaddlong (t_int64 acc, int32 sv1, int32 sv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((acc & 0xFFFFFFFF) < (sv1 & 0xFFFFFFFF)) SET_CARRY;

   if (((sv1 & 0x80000000) == (sv2 & 0x80000000)) &&
       ((sv2 & 0x80000000) != (acc & 0x80000000))) SET_OVER;
}

/***********************************************************************
* chkaddquad - Check addition for carry and overflow
***********************************************************************/

static void
chkaddquad (t_int64 acc, t_int64 sv1, t_int64 sv2)
{
   CLR_CARRY;
   CLR_OVER;

   if (acc < sv1) SET_CARRY;

   if (((sv1 & 0x8000000000000000ULL) == (sv2 & 0x8000000000000000ULL)) &&
       ((sv2 & 0x8000000000000000ULL) != (acc & 0x8000000000000000ULL)))
       SET_OVER;
}

/***********************************************************************
* chksublong - Check subtraction for carry and overflow
***********************************************************************/

static void
chksublong (t_int64 acc, int32 sv1, int32 sv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((uint32)sv2 >= (uint32)sv1) SET_CARRY;

   if (((sv1 & 0x80000000) != (sv2 & 0x80000000)) &&
       ((sv2 & 0x80000000) != (acc & 0x80000000))) SET_OVER;
}

/***********************************************************************
* chksubquad - Check subtraction for carry and overflow
***********************************************************************/

static void
chksubquad (t_int64 acc, t_int64 sv1, t_int64 sv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((t_uint64)sv2 >= (t_uint64)sv1) SET_CARRY;

   if (((sv1 & 0x8000000000000000ULL) != (sv2 & 0x8000000000000000ULL)) &&
       ((sv2 & 0x8000000000000000ULL) != (acc & 0x8000000000000000ULL)))
       SET_OVER;
}

/***********************************************************************
* proc11op - Process type 11 operations.
***********************************************************************/

int
proc11op (uint16 inst)
{
   int i;
   int found;
   uint16 inst1;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 sad, dad;
   uint8 db;
   uint8 sdat[18], ddat[18];

   if (model < 12) return (-1);

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   bc = (inst1 & 0xF000) >> 12;

   if (bc == 0)
   {
      bc = R0 & 0x000F;
      if (bc == 0) bc = 16;
   }
   
   if (inst == 0xC08) /* NRM */
   {
      int j;
      uint8 ocarry, carry;

      if (traceit) fprintf (tracefd, "NRM");
      sad = getaddr (sr, st, FALSE);
      dad = getword (dr, dt, FALSE, DST);
#ifdef DEBUGEINSTR
      fprintf (stderr,
        "INST = NRM sr = %d, st = %d, dr = %d, dt = %d, bc = %d, sad = >%04X\n",
	       sr, st, dr, dt, bc, sad);
      fprintf (stderr, "   dad = >%04X\n", dad);
#endif
      CLR_AGT;
      CLR_LGT;
      CLR_EQ;
      CLR_CARRY;
      j = FALSE;
      for (i = 0; i < bc; i++)
      {
         sdat[i] = GETMEMB (sad+i, SRC);
	 if (sdat[i] != 0) j = TRUE;
      }
#ifdef DEBUGEINSTR
      fprintf (stderr,"   j = %d, stat = >%04X\n   sdat = ", j, statreg);
      for (i = 0; i < bc; i++)
	 fprintf (stderr, "%02X", sdat[i]);
      fprintf (stderr,"\n");
#endif
      if (!j)
      {
	 SET_EQ;
      }
      else
      {

	 j = 0;
	 while (((sdat[0] & 0x80) >> 1) == (sdat[0] & 0x40))
	 {
	    ocarry = 0;
	    for (i = bc; i > 0; i--) 
	    {
	       carry = (sdat[i-1] & 0x80) >> 7;
	       sdat[i-1] = (sdat[i-1] << 1) | ocarry;
	       ocarry = carry;
	    }
	    if (ocarry) SET_CARRY;
	    j++;
	 }
#ifdef DEBUGEINSTR
	 fprintf (stderr,"   sdat = ");
	 for (i = 0; i < bc; i++)
	    fprintf (stderr, "%02X", sdat[i]);
	 fprintf (stderr,"\n");
#endif
	 for (i = 0; i < bc; i++)
	    PUTMEMB (sad+i, sdat[i], SRC);
	 dad += j;
	 if (sdat[0] > 0) SET_LGT;
	 if ((int8)sdat[0] > 0) SET_AGT;
      }
      putaddr (sr, st, bc);
      putword (dr, dt, dad, DST);
#ifdef DEBUGEINSTR
      fprintf (stderr, "   dad = >%04X\n", dad);
      fprintf (stderr,"   stat = >%04X\n", statreg);
#endif
      return (0);
   }

   else switch ((inst & 0x003F) - 0x001E)
   {
   case 0: /* RTO */

      if (traceit) fprintf (tracefd, "RTO");
      sad = getaddr (sr, st, FALSE);
      dad = getword (dr, dt, FALSE, DST);
#ifdef DEBUGEINSTR
      fprintf (stderr,
        "INST = RTO sr = %d, st = %d, dr = %d, dt = %d, bc = %d, sad = >%04X\n",
	       sr, st, dr, dt, bc, sad);
      fprintf (stderr, "   dad = >%04X\n", dad);
      fprintf (stderr, "   dat = ");
#endif
      putaddr (sr, st, bc);
      sad += (bc - 1);
      i = bc * 8;
      SET_EQ;
      for (found = FALSE; !found && bc; bc--)
      {
	 int j;

	 db = GETMEMB (sad, SRC);
#ifdef DEBUGEINSTR
	 fprintf (stderr, "%02X", db);
#endif
	 if (db == 0)
	 {
	    i -= 8;
	 }
	 else for (j = 0; j < 8; j++)
	 {
	    if (db & 0x01)
	    {
	       dad += (i - 1);
	       CLR_EQ;
	       found = TRUE;
	    }
	    else
	    {
	       db >>= 1;
	       i--;
	    }
	 }
	 sad--;
      }

#ifdef DEBUGEINSTR
      fprintf (stderr, "\n   dad = >%04X\n", dad);
#endif
      putword (dr, dt, dad, DST);
      break;

   case 1: /* LTO */

      if (traceit) fprintf (tracefd, "LTO");
      sad = getaddr (sr, st, FALSE);
      dad = getword (dr, dt, FALSE, DST);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	"INST = LTO sr = %d, st = %d, dr = %d, dt = %d, bc = %d, sad = >%04X\n",
	       sr, st, dr, dt, bc, sad);
      fprintf (stderr, "   dad = >%04X\n", dad);
      fprintf (stderr, "   dat = ");
#endif
      putaddr (sr, st, bc);
      i = 0;
      SET_EQ;
      for (found = FALSE; !found && bc; bc--)
      {
	 int j;

	 db = GETMEMB (sad, SRC);
#ifdef DEBUGEINSTR
	 fprintf (stderr, "%02X", db);
#endif
	 if (db == 0)
	 {
	    i += 8;
	 }
	 else for (j = 0; j < 8; j++)
	 {
	    if (db & 0x80)
	    {
	       dad += i;
	       CLR_EQ;
	       found = TRUE;
	    }
	    else
	    {
	       db <<= 1;
	       i++;
	    }
	 }
	 sad++;
      }
#ifdef DEBUGEINSTR
      fprintf (stderr, "\n   dad = >%04X\n", dad);
#endif
      putword (dr, dt, dad, DST);
      break;

   case 2: /* CNTO */

      if (traceit) fprintf (tracefd, "CNTO");
      sad = getaddr (sr, st, FALSE);
      dad = getword (dr, dt, FALSE, DST);
#ifdef DEBUGEINSTR
      fprintf (stderr,
       "INST = CNTO sr = %d, st = %d, dr = %d, dt = %d, bc = %d, sad = >%04X\n",
	       sr, st, dr, dt, bc, sad);
      fprintf (stderr, "   dad = >%04X\n", dad);
      fprintf (stderr, "   dat = ");
#endif
      putaddr (sr, st, bc);
      i = 0;
      for (; bc; bc--)
      {
	 int j;
	 db = GETMEMB (sad, SRC);
#ifdef DEBUGEINSTR
	 fprintf (stderr, "%02X", db);
#endif
	 if (db != 0)
	    for (j = 0; j < 8; j++)
	    {
	       if (db & 0x80)
	       {
		  i++;
	       }
	       db <<= 1;
	    }
	 sad++;
      }
      if (i)
	 CLR_EQ;
      else
         SET_EQ;
      dad += i;
#ifdef DEBUGEINSTR
      fprintf (stderr, "\n   dad = >%04X\n", dad);
#endif
      putword (dr, dt, dad, DST);
      break;

   case 5: /* BDC */

      if (traceit) fprintf (tracefd, "BDC");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = BDC sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      break;

   case 6: /* DBC */

      if (traceit) fprintf (tracefd, "DBC");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = DBC sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      break;

   case 7: /* SWPM */

      if (traceit) fprintf (tracefd, "SWPM");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = SWPM sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      sad = getaddr (sr, st, FALSE);
      dad = getaddr (dr, dt, FALSE);
      for (i = 0; i < bc; i++)
      {
	 ddat[i] = GETMEMB (sad+i, SRC);
	 sdat[i] = GETMEMB (dad+i, DST);
      }
      cmpmbstring (sdat, ddat, bc);
      for (i = 0; i < bc; i++)
      {
	 PUTMEMB (sad+i, sdat[i], SRC);
	 PUTMEMB (dad+i, ddat[i], DST);
      }
      putaddr (sr, st, bc);
      putaddr (dr, dt, bc);
      break;

   case 8: /* XORM */

      if (traceit) fprintf (tracefd, "XORM %d", bc);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = XORM sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      sad = getaddr (sr, st, FALSE);
      dad = getaddr (dr, dt, FALSE);

      for (i = 0; i < bc; i++)
      {
         ddat[i] = GETMEMB (sad+i, SRC) ^ GETMEMB (dad+i, DST);
      }
      chkmbzero (ddat, bc);
      for (i = 0; i < bc; i++)
      {
         PUTMEMB (dad+i, ddat[i], DST);
      }
      putaddr (sr, st, bc);
      putaddr (dr, dt, bc);
      break;

   case 9: /* ORM */

      if (traceit) fprintf (tracefd, "ORM %d", bc);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = ORM sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      sad = getaddr (sr, st, FALSE);
      dad = getaddr (dr, dt, FALSE);

      for (i = 0; i < bc; i++)
      {
         ddat[i] = GETMEMB (sad+i, SRC) | GETMEMB (dad+i, DST);
      }
      chkmbzero (ddat, bc);
      for (i = 0; i < bc; i++)
      {
         PUTMEMB (dad+i, ddat[i], DST);
      }
      putaddr (sr, st, bc);
      putaddr (dr, dt, bc);
      break;

   case 10: /* ANDM */

      if (traceit) fprintf (tracefd, "ANDM %d", bc);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = ANDM sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      sad = getaddr (sr, st, FALSE);
      dad = getaddr (dr, dt, FALSE);

      for (i = 0; i < bc; i++)
      {
         ddat[i] = GETMEMB (sad+i, SRC) & GETMEMB (dad+i, DST);
      }
      chkmbzero (ddat, bc);
      for (i = 0; i < bc; i++)
      {
         PUTMEMB (dad+i, ddat[i], DST);
      }
      putaddr (sr, st, bc);
      putaddr (dr, dt, bc);
      break;

   case 11: /* SM */

      if (traceit) fprintf (tracefd, "SM %d", bc);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = SM sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      if (bc == 4)
      {
	 t_int64 acc;
	 uint32 val;
	 uint32 uv1;
         int32 sv1, sv2;

	 sv1 = (int32)getlong (sr, st, TRUE, SRC);
	 acc = sv1;
	 sv2 = (int32)getlong (dr, dt, FALSE, DST);
	 acc = sv2 - acc;
	 uv1 = acc & 0xFFFFFFFF;
	 acc = (int32)uv1;
	 chksublong (acc, sv1, sv2);
	 val = acc & 0xFFFFFFFF;
	 putlong (dr, dt, val, DST);
	 cmplongzero (val);
      }
      else if (bc == 8)
      {
	 t_int64 acc;
         t_int64 sv1, sv2;

	 sv1 = (t_int64)getquad (sr, st, TRUE, SRC);
	 acc = sv1;
	 sv2 = (t_int64)getquad (dr, dt, FALSE, DST);
	 acc = sv2 - acc;
	 chksubquad (acc, sv1, sv2);
	 putquad (dr, dt, acc, DST);
	 cmpquadzero (acc);
      }
      else
      {
      }
      break;

   case 12: /* AM */

      if (traceit) fprintf (tracefd, "AM %d", bc);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = AM sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      if (bc == 4)
      {
	 t_int64 acc;
	 uint32 val;
	 uint32 uv1;
         int32 sv1, sv2;

	 sv1 = (int32)getlong (sr, st, TRUE, SRC);
	 acc = sv1;
	 sv2 = (int32)getlong (dr, dt, FALSE, DST);
	 acc = acc + sv2;
	 uv1 = acc & 0xFFFFFFFF;
	 acc = (int32)uv1;
	 chkaddlong (acc, sv1, sv2);
	 val = acc & 0xFFFFFFFF;
	 putlong (dr, dt, val, DST);
	 cmplongzero (val);
      }
      else if (bc == 8)
      {
	 t_int64 acc;
         t_int64 sv1, sv2;

	 sv1 = (t_int64)getquad (sr, st, TRUE, SRC);
	 acc = sv1;
	 sv2 = (t_int64)getquad (dr, dt, FALSE, DST);
	 acc = acc + sv2;
	 chkaddquad (acc, sv1, sv2);
	 putquad (dr, dt, acc, DST);
	 cmpquadzero (acc);
      }
      else
      {
      }
      break;

   default:
      return (-1);
   }

   return (0);
}

/***********************************************************************
* proc12op - Process type 12 operations.
***********************************************************************/

int
proc12op (uint16 inst)
{
   int i;
   int tagged;
   int bytechk;
   int srczero, dstzero;
   uint16 inst1;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 ck;
   uint16 ckpt;
   uint16 sad, dad;
   uint16 stk[3];
   uint8 db;

   tagged = 0;
   if (model < 12) return (-1);

   ckpt = inst & 0x000F;

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   bc = (inst1 & 0xF000) >> 12;

   if (bc == 0)
   {
      bc = R0;
      if (bc == 0xFFFF) tagged = 1;
   }
   
   if ((inst & 0x0F00) == 0x0E00)
   {
      switch ((inst & 0x00F0) >> 4)
      {
      case 1: /* SNEB */

	 {
	    int i;
	    uint16 ck;
	    uint16 ad;
	    uint8 cb, mb, db;

	    if (traceit) fprintf (tracefd, "SNEB");
#ifdef DEBUGEINSTR
	    fprintf (stderr,
		  "INST = SNEB sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
		     sr, st, dr, dt, bc);
#endif
	    ad = getaddr (sr, st, TRUE);
	    mb = GETMEMB (ad, SRC);
	    db = GETMEMB (ad+1, SRC);
	    ad = getaddr (dr, dt, FALSE);
	    ck = GETREG (ckpt);
#ifdef DEBUGEINSTR
	    fprintf (stderr,
	    "   mb = >%02X, db = >%02X, ck = >%04X, ad = >%04X, stat = >%04X\n",
		     mb, db, ck, ad, statreg);
#endif
	    if (tagged)
	    {
	       bc = GETMEMB (ad, DST) - 1;
	       ad++;
	    }
	    if (bc == 0)
	    {
	       CLR_AGT;
	       CLR_LGT;
	       SET_EQ;
	       PUTREG (ckpt, 0xFFFF);
	       return (0);
	    }
	    if (ck != 0xFFFF)
	    {
	       ad += ck + 1;
	       bc = bc - ck - 1;
	    }
#ifdef DEBUGEINSTR
	    fprintf (stderr,
		     "   ck = %d, bc = %d, ad = >%04X\n",
		     ck, bc, ad);
#endif
	    putaddr (dr, dt, bc + tagged);
	    for (i = 0; i < bc; i++)
	    {
	       cb = GETMEMB (ad, DST);
#ifdef DEBUGEINSTR
	       fprintf (stderr,
			"   cb = >%02X, cb & mb = >%02X\n",
			cb, cb & mb);
#endif
	       ad++;
	       ck++;
	       if ((cb & mb) != db) 
	       {
	          CLR_EQ;
		  if (db > (cb & mb)) SET_LGT;
		  else CLR_LGT;
		  if ((int8)db > (int8)(cb & mb)) SET_AGT;
		  else CLR_AGT;
#ifdef DEBUGEINSTR
		  fprintf (stderr,
			   "   IS NOT EQ: ck = >%04X, stat = >%04X\n",
			   ck, statreg);
#endif
		  PUTREG (ckpt, ck);
		  return (0);
	       }
	    }
	    SET_EQ;
	    CLR_AGT;
	    CLR_LGT;
#ifdef DEBUGEINSTR
	    fprintf (stderr,
		     "   IS EQ: ck = >%04X, stat = >%04X\n", ck, statreg);
#endif
	    PUTREG (ckpt, 0xFFFF);
	 }
         break;

      case 2: /* CRC */

	 if (traceit) fprintf (tracefd, "CRC");
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = CRC sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
		  sr, st, dr, dt, bc);
#endif
         break;

      case 3: /* TS */

	 if (traceit) fprintf (tracefd, "TS");
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = TS sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
		  sr, st, dr, dt, bc);
#endif
	 sad = getaddr (sr, st, FALSE);
	 dad = getaddr (dr, dt, FALSE);
	 if (tagged)
	 {
	    bc = GETMEMB (dad, DST) - 1;
	    dad++;
	 }
	 CLR_EQ;
	 CLR_AGT;
	 CLR_LGT;
	 if (bc == 0)
	 {
	    return (0);
	 }
	 bytechk = TRUE;
	 for (i = 0; i < bc; i++)
	 {
	    uint8 tb;
	    db = GETMEMB (dad+i, DST);
	    tb = GETMEMB (sad+db, SRC);
	    PUTMEMB (dad+i, tb, DST);
	    if (tb != 0) bytechk = FALSE;
	 }
	 if (bytechk)
	 {
	    SET_EQ;
	 }
	 else
	 {
	    db = GETMEMB (dad, DST);
	    if (db > 0) SET_LGT;
	    if ((int8)db > 0) SET_AGT;
	 }
	 putaddr (sr, st, 256);
	 putaddr (dr, dt, bc + tagged);
         break;

      default:
	 return (-1);
      }
   }
   else switch (((inst & 0x00F0) - 0x0040) >> 4)
   {
   case 0: /* CS */

      if (traceit) fprintf (tracefd, "CS %d", bc);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = CS sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      sad = getaddr (sr, st, FALSE);
      dad = getaddr (dr, dt, FALSE);
      ck = GETREG(ckpt);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	      "   sad = >%04X, dad = >%04X, ck = >%04X, stat = >%04X\n",
	       sad, dad, ck, statreg);
#endif
      CLR_AGT;
      CLR_LGT;
      SET_EQ;
      if (tagged)
      {
	 uint16 sbc, dbc;

	 sbc = GETMEMB (sad, SRC);
	 if (sbc == 0) sbc = 256;
	 else sbc--;
	 dbc = GETMEMB (dad, DST);
	 if (dbc == 0) dbc = 256;
	 else dbc--;
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "   tagged: sbc = %d, dbc = %d\n", sbc, dbc);
#endif
	 if (ck == 0xFFFF && dbc != sbc)
	 {
	    ck = 0;
	    PUTREG (ckpt, ck);
	    CLR_EQ;
	    if (sbc > dbc) SET_LGT;
	    if ((int16)sbc > (int16)dbc) SET_AGT;
#ifdef DEBUGEINSTR
	    fprintf (stderr,"   ck = >%04X\n", ck);
	    fprintf (stderr,"   stat = >%04X\n", statreg);
#endif
	    return (0);
	 }
	 bc = sbc;
	 if (ck == 0)
	 {
	    if (sbc < dbc) bc = sbc;
	    else bc = dbc;
	 }
	 sad++;
	 dad++;
      }
      if (ck != 0xFFFF)
      {
	 sad += ck + 1;
	 dad += ck + 1;
	 bc = bc - ck - 1;
	 if (IS_EQ) bytechk = FALSE;
	 else bytechk = TRUE;
      }
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "   ck = %d, bc = %d\n",
	       ck, bc);
      {
	 int pr;
	 pr = mapaddr (sad, GET_MAP, MEM_NOPROT, FALSE);
	 fprintf (stderr, "SRC:\n");
	 HEXDUMP (stderr, &memory[pr], bc, sad);
	 pr = mapaddr (dad, GET_MAP, MEM_NOPROT, FALSE);
	 fprintf (stderr, "DST:\n");
	 HEXDUMP (stderr, &memory[pr], bc, dad);
      }
#endif
      srczero = dstzero = TRUE;
      for (i = 0; i < bc; i++)
      {
	 uint8 sb, db;

	 ck++;
	 sb = GETMEMB(sad+i,SRC);
	 db = GETMEMB(dad+i, DST);
	 if (sb) srczero = FALSE;
	 if (db) dstzero = FALSE;
	 if (sb != db)
	 {
	    CLR_EQ;
	    if (bytechk)
	    {
	       if (sb > db) SET_LGT;
	       if ((int8)sb > (int8)db) SET_AGT;
	    }
	    ck += tagged;
	    break;
	 }
      }
      if (!IS_EQ)
      {
	 for (i = 0; i < bc; i++)
	 {
	    uint8 sdat, ddat;

	    sdat = GETMEMB(sad+i, SRC);
	    ddat = GETMEMB(dad+i, DST);

	    if (sdat > ddat)
	    {
	       SET_LGT;
	       if (i == 0 && (int8)sdat < (int8)ddat) CLR_AGT;
	       else SET_AGT;
	       break;
	    }
	    else if (sdat < ddat)
	    {
	       if (!srczero && i == 0 && (int8)sdat > (int8)ddat) SET_AGT;
	       break;
	    }
	 }
      }
      else
      {
         ck = 0xFFFF;
      }
      PUTREG (ckpt, ck);
#ifdef DEBUGEINSTR
      fprintf (stderr,"   ck = >%04X\n", ck);
      fprintf (stderr,"   stat = >%04X\n", statreg);
#endif
      putaddr (sr, st, bc + tagged);
      putaddr (dr, dt, bc + tagged);
      break;

   case 1: /* SEQB */

      {
	 int i;
	 uint16 ck;
	 uint16 ad;
	 uint8 cb, mb, db;

	 if (traceit) fprintf (tracefd, "SEQB");
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = SEQB sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
		  sr, st, dr, dt, bc);
#endif
	 ad = getaddr (sr, st, TRUE);
	 mb = GETMEMB (ad, SRC);
	 db = GETMEMB (ad+1, SRC);
	 ad = getaddr (dr, dt, FALSE);
	 ck = GETREG (ckpt);
#ifdef DEBUGEINSTR
	 fprintf (stderr,
	   "   mb = >%02X, db = >%02X, ck = >%04X, ad = >%04X, stat = >%04X\n",
		  mb, db, ck, ad, statreg);
#endif
	 if (tagged)
	 {
	    bc = GETMEMB (ad, DST) - 1;
	    ad++;
	 }
	 if (bc == 0)
	 {
	    CLR_AGT;
	    CLR_LGT;
	    CLR_EQ;
	    PUTREG (ckpt, 0xFFFF);
	    return (0);
	 }
	 if (ck != 0xFFFF)
	 {
	    ad += ck + 1;
	    bc = bc - ck - 1;
	 }
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "   ck = %d, bc = %d, ad = >%04X\n",
		  ck, bc, ad);
#endif
	 putaddr (dr, dt, bc + tagged);
	 for (i = 0; i < bc; i++)
	 {
	    cb = GETMEMB (ad, DST);
#ifdef DEBUGEINSTR
	    fprintf (stderr,
		     "   cb = >%02X, cb & mb = >%02X\n",
		     cb, cb & mb);
#endif
	    ck++;
	    ad++;
	    if ((cb & mb) == db) 
	    {
	       SET_EQ;
	       CLR_AGT;
	       CLR_LGT;
#ifdef DEBUGEINSTR
	       fprintf (stderr,
			"   IS EQ: ck = >%04X, stat = >%04X\n", ck, statreg);
#endif
	       PUTREG (ckpt, ck);
	       return (0);
	    }
	 }
	 CLR_EQ;
	 if (db > (cb & mb)) SET_LGT;
	 else CLR_LGT;
	 if ((int8)db > (int8)(cb & mb)) SET_AGT;
	 else CLR_AGT;
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "   IS NOT EQ: ck = >%04X, stat = >%04X\n", ck, statreg);
#endif
	 PUTREG (ckpt, 0xFFFF);
      }
      break;

   case 2: /* MOVS */

      if (traceit) fprintf (tracefd, "MOVS %d", bc);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = MOVS sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      sad = getaddr (sr, st, FALSE);
      dad = getaddr (dr, dt, FALSE);
      ck = GETREG(ckpt);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	      "   sad = >%04X, dad = >%04X, ck = >%04X, stat = >%04X\n",
	       sad, dad, ck, statreg);
#endif
      CLR_EQ;
      CLR_AGT;
      CLR_LGT;
      if (tagged)
      {
	 bc = GETMEMB (sad++, SRC);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "   tagged: bc = >%02X\n", bc);
#endif
	 PUTMEMB (dad++, bc, DST);
	 if (bc == 1) goto MOVS_EXIT;
	 if (bc == 0) bc = 256;
	 else bc--;
      }
      if (ck != 0xFFFF)
      {
	 sad += ck + 1;
	 dad += ck + 1;
	 bc = bc - ck - 1;
      }
      if (bc == 0)
	 return (0);

      for (i = 0; i < bc; i++)
      {
	 PUTMEMB (dad+i, GETMEMB(sad+i, SRC), DST);
      }
      chkmbmemzero (dad, bc, DST);
#ifdef DEBUGEINSTR
      fprintf (stderr,"   ck = >%04X\n", ck);
      fprintf (stderr,"   stat = >%04X\n", statreg);
#endif
   MOVS_EXIT:
      PUTREG (ckpt, 0xFFFF);
      putaddr (sr, st, bc + tagged);
      putaddr (dr, dt, bc + tagged);
      break;

   case 8: /* MVSR */

      if (traceit) fprintf (tracefd, "MVSR");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = MVSR sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      sad = getaddr (sr, st, FALSE);
      dad = getaddr (dr, dt, FALSE);
      CLR_EQ;
      CLR_AGT;
      CLR_LGT;
      if (tagged)
      {
	 bc = GETMEMB (sad, SRC);
	 if (bc == 1)
	    return (0);
	 PUTMEMB (dad++, GETMEMB(sad++, SRC), DST);
	 bc--;
      }
      if (bc == 0)
	 return (0);
      for (i = 0; i < bc; i++)
      {
	 PUTMEMB (dad+i, GETMEMB(sad+(bc-i), SRC), DST);
      }
      chkmbmemzero (dad, bc, DST);
      PUTREG (ckpt, 0xFFFF);
      putaddr (sr, st, bc + tagged);
      putaddr (dr, dt, bc + tagged);
      break;

   case 9: /* MVSK */

      if (traceit) fprintf (tracefd, "MVSK");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = MVSK sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      getstack (sr, st, TRUE, SRC, stk);
      sad = stk[0];
      dad = getaddr (dr, dt, FALSE);
      CLR_EQ;
      CLR_AGT;
      CLR_LGT;
      if (tagged)
      {
	 bc = GETMEMB (sad, SRC);
	 if (bc == 1)
	    return (0);
	 PUTMEMB (dad++, GETMEMB(sad++, SRC), DST);
	 bc--;
      }
      if (bc == 0)
	 return (0);
      if ((stk[0] + bc + tagged) > stk[3])
      {
         if (IS_OVERI)
	 {
	 }
	 return (0);
      }
      for (i = 0; i < bc; i++)
      {
	 PUTMEMB (dad+i, GETMEMB(sad+i, SRC), DST);
      }
      chkmbmemzero (dad, bc, DST);
      PUTREG (ckpt, 0xFFFF);
      putaddr (dr, dt, bc + tagged);
      break;

   case 10: /* POPS */

      if (traceit) fprintf (tracefd, "POPS");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = POPS sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      getstack (sr, st, FALSE, SRC, stk);
      sad = stk[0];
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       " stk = %04X %04X %04X\n",
	       stk[0], stk[1], stk[2]);
#endif
      pcreg += 2;
      dad = getaddr (dr, dt, FALSE);
      pcreg -= 2;
      CLR_EQ;
      CLR_AGT;
      CLR_LGT;
      if (tagged)
      {
	 bc = GETMEMB (sad, SRC);
	 if (bc == 1)
	    return (0);
	 PUTMEMB (dad++, GETMEMB(sad++, SRC), DST);
	 bc--;
      }
      if (bc == 0)
	 return (0);
      if ((stk[0] + bc + tagged) > stk[3])
      {
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  " stk underflow\n");
#endif
         if (IS_OVERI)
	 {
	 }
	 return (0);
      }
      for (i = 0; i < bc; i++)
      {
	 PUTMEMB (dad+i, GETMEMB(sad+i, SRC), DST);
      }
      chkmbmemzero (dad, bc, DST);
      stk[0] += bc + tagged;
      putstack (sr, st, stk, SRC);
      PUTREG (ckpt, 0xFFFF);
      putaddr (dr, dt, bc + tagged);
      break;

   case 11: /* PSHS */

      if (traceit) fprintf (tracefd, "PSHS");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = PSHS sr = %d, st = %d, dr = %d, dt = %d, bc = %d\n",
	       sr, st, dr, dt, bc);
#endif
      sad = getaddr (sr, st, FALSE);
      getstack (dr, dt, FALSE, DST, stk);
      dad = stk[0];
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       " stk = %04X %04X %04X\n",
	       stk[0], stk[1], stk[2]);
#endif
      CLR_EQ;
      CLR_AGT;
      CLR_LGT;
      if (tagged)
      {
	 bc = GETMEMB (sad, SRC);
	 if (bc == 1)
	    return (0);
	 PUTMEMB (dad-bc, GETMEMB(sad++, SRC), DST);
	 bc--;
      }
      if (bc == 0)
	 return (0);
      if ((stk[0] - bc - tagged) <= stk[1])
      {
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  " stk overflow\n");
#endif
         if (IS_OVERI)
	 {
	 }
	 return (0);
      }
      dad -= bc;
      for (i = 0; i < bc; i++)
      {
	 PUTMEMB (dad+i, GETMEMB (sad+i, SRC), DST);
      }
      chkmbmemzero (dad, bc, DST);
      putaddr (sr, st, bc + tagged);
      stk[0] -= bc + tagged;
      putstack (dr, dt, stk, DST);
      PUTREG (ckpt, 0xFFFF);
      break;

   default:
      return (-1);
   }

   return (0);
}

/***********************************************************************
* proc13op - Process type 13 operations.
***********************************************************************/

int
proc13op (uint16 inst)
{
   int i;
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 ct;
   uint16 sad;
   uint8 carry, ocarry;
   uint8 sdat[18];

   if (model < 12) return (-1);

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ct = (inst1 & 0x03C0) >> 6;
   bc = (inst1 & 0xF000) >> 12;

   if (bc == 0)
   {
      bc = R0 & 0x000F;
      if (bc == 0) bc = 16;
   }
   if (ct == 0)
   {
      ct = (R0 & 0x0F00) >> 8;
   }

   sad = getaddr (sr, st, FALSE);

   for (i = 0; i < bc; i++)
   {
      sdat[i] = GETMEMB (sad+i, SRC);
   }

   switch ((inst & 0x000F) - 0x000C)
   {
   case 0: /* SRAM */

#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = SRAM sr = %d, st = %d, ct = %d, bc = %d\n",
	       sr, st, ct, bc);
#endif
      if (traceit) fprintf (tracefd, "SRAM %d", bc);
      for (; ct; ct--)
      {
	 ocarry = 0;
	 for (i = 0; i < bc; i++)
	 {
	    carry = sdat[i] & 1;
	    if (i == 0) sdat[i] = (int8)sdat[i] >> 1;
	    else sdat[i] = sdat[i] >> 1;
	    sdat[i] |= ocarry << 7;
	    ocarry = carry;
	 }
	 if (carry) SET_CARRY;
	 else CLR_CARRY;
      }
      break;

   case 1: /* SLAM */

#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = SLAM sr = %d, st = %d, ct = %d, bc = %d\n",
	       sr, st, ct, bc);
#endif
      if (traceit) fprintf (tracefd, "SLAM %d", bc);
      for (; ct; ct--)
      {
	 ocarry = 0;
	 for (i = bc; i; i--)
	 {
	    carry = sdat[i-1] & 0x80;
	    sdat[i-1] = sdat[i-1] << 1;
	    sdat[i-1] |= ocarry >> 7;
	    ocarry = carry;
	 }
	 if (carry) SET_CARRY;
	 else CLR_CARRY;
      }
      break;

   default:
      return (-1);
   }

   chkmbzero (sdat, bc);
   for (i = 0; i < bc; i++)
   {
      PUTMEMB (sad+i, sdat[i], SRC);
   }
   putaddr (sr, st, bc);

   return (0);
}

/***********************************************************************
* proc14op - Process type 14 operations.
***********************************************************************/

int
proc14op (uint16 inst)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 ps;
   uint16 ad;
   uint16 wd;
   uint8 bt;
   uint8 val;

   if (model < 12) return (-1);

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ps = (inst1 & 0xFFC0) >> 6;
   if (ps == 0x03FF) ps = R0;

   ad = getaddr (sr, st, FALSE);
   wd = ps / 8;
   bt = 0x80 >> (ps % 8);
   val = GETMEMB (ad+wd, SRC);

   if (val & bt) SET_EQ;
   else CLR_EQ;

   putaddr (sr, st, 1);

   switch ((inst & 0x000F) - 0x0009)
   {
   case 0: /* TMB */

      if (traceit) fprintf (tracefd, "TMB");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = TMB sr = %d, st = %d, ps = %d\n",
	       sr, st, ps);
#endif
      break;

   case 1: /* TCMB */

      if (traceit) fprintf (tracefd, "TCMB");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = TCMB sr = %d, st = %d, ps = %d\n",
	       sr, st, ps);
#endif
      val = val & ~bt;
      PUTMEMB (ad+wd, val, SRC);
      break;

   case 2: /* TSMB */

      if (traceit) fprintf (tracefd, "TSMB");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = TSMB sr = %d, st = %d, ps = %d\n",
	       sr, st, ps);
#endif
      val = val | bt;
      PUTMEMB (ad+wd, val, SRC);
      break;

   default:
      return (-1);
   }

   return (0);
}

/***********************************************************************
* proc15op - Process type 15 operations.
***********************************************************************/

int
proc15op (uint16 inst)
{
   int i;
   int shift;
   uint32 lval;
   uint32 lbit;
   uint32 lmsk;
   uint32 lwrk;
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 ps;
   uint16 wd;
   uint16 sval;
   uint16 sbit;
   uint16 smsk;
   uint16 swrk;

   if (model < 12) return (-1);

   wd = inst & 0x000F;

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ps = (inst1 & 0xF000) >> 12;

   if ((inst & 0x0F00) != 0x0E00) return (-1);

   if (traceit) fprintf (tracefd, "IOF");
#ifdef DEBUGEINSTR
   fprintf (stderr,
	    "INST = IOF sr = %d, st = %d, ps = %d, wd = %d\n",
	    sr, st, ps, wd);
#endif

   if (wd == 0)
   {
      wd = R0 & 0x000F;
      if (wd == 0) wd = 16;
   }
   if (ps == 0)
   {
      ps = (R0 >> 8) & 0x000F;
   }
#ifdef DEBUGEINSTR
   fprintf (stderr, "   ps = %d, wd = %d\n", ps, wd);
#endif
   shift = wd - 1;
   if (ps + wd > 16)
   {
      lmsk = 0;
      for (i = 0; i < wd; i++)
      {
	 lmsk |= 0x80000000 >> (ps + i);
      }
      lval = getlong (sr, st, FALSE, SRC);
      lbit = lval & lmsk;
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "   lmsk = >%08X, lval = >%08X, lbit = >%08X\n",
	       lmsk, lval, lbit);
#endif
      lwrk = 0;
      for (i = 0; i < (wd/2)+1; i++)
      {
	 uint32 obit;
	 obit = (lbit & (0x8000000 >> ((wd + ps) - i - 1))) << shift;
	 lwrk |= obit | (lbit & (0x80000000 >> (ps + i)) >> shift);
	 shift -= 2;
      }
      lval = (lval & ~lmsk) | lwrk;
#ifdef DEBUGEINSTR
      fprintf (stderr, "   lval = >%08X\n", lval);
#endif
      putlong (sr, st, lval, SRC);
   }
   else
   {
      smsk = 0;
      for (i = 0; i < wd; i++)
      {
	 smsk |= 0x8000 >> (ps + i);
      }
      sval = getword (sr, st, FALSE, SRC);
      sbit = sval & smsk;
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "   smsk = >%04X, sval = >%04X, sbit = >%04X\n",
	       smsk, sval, sbit);
#endif
      swrk = 0;
      for (i = 0; i < (wd/2)+1; i++)
      {
	 uint16 obit;
	 obit = (sbit & (0x8000 >> ((wd + ps) - i - 1))) << shift;
	 swrk |= obit | ((sbit & (0x8000 >> (ps + i))) >> shift);
	 shift -= 2;
      }
      sval = (sval & ~smsk) | swrk;
#ifdef DEBUGEINSTR
      fprintf (stderr, "   sval = >%04X\n", sval);
#endif
      putword (sr, st, sval, SRC);
   }

   return (0);
}

/***********************************************************************
* proc16op - Process type 16 operations.
***********************************************************************/

int
proc16op (uint16 inst)
{
   int i;
   int sign;
   int longvals;
   uint32 lsval;
   uint32 ldval;
   uint32 lsmsk = 0;
   uint32 ldmsk = 0;
   uint16 inst1;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 ps;
   uint16 wd;
   uint16 sval;
   uint16 dval;
   uint16 smsk = 0;
   uint16 dmsk = 0;

   if (model < 12) return (-1);

   wd = inst & 0x000F;

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   ps = (inst1 & 0xF000) >> 12;
   if (wd == 0)
   {
      wd = R0 & 0x000F;
      if (wd == 0) wd = 16;
   }
   if (ps == 0)
   {
      ps = (R0 >> 8) & 0x000F;
   }
   if (ps + wd > 16)
   {
      longvals = TRUE;
      for (i = 0; i < wd; i++)
      {
	 lsmsk |= 1 << (16+i);
      }
      for (i = 0; i < wd; i++)
      {
	 ldmsk |= 0x80000000 >> (ps + i);
      }
   }
   else
   {
      longvals = FALSE;
      for (i = 0; i < wd; i++)
      {
	 smsk |= 1 << i;
      }
      for (i = 0; i < wd; i++)
      {
	 dmsk |= 0x8000 >> (ps + i);
      }
   }

   switch ((inst & 0x00F0) >> 4)
   {
   case 1: /* INSF */

      if (traceit) fprintf (tracefd, "INSF");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	   "INST = INSF sr = %d, st = %d, dr = %d, dt = %d, ps = %d, wd = %d\n",
	       sr, st, dr, dt, ps, wd);
      fprintf (stderr,
	       "  lsmsk = >%08X, ldmsk = >%08X\n",
	       lsmsk, ldmsk);
      fprintf (stderr,
	       "  smsk = >%04X, dmsk = >%04X\n",
	       smsk, dmsk);
      fprintf (stderr,"   stat = >%04X\n", statreg);
#endif
      if (longvals)
      {
	 int shfcnt;

	 shfcnt = 16 - ps - wd;
	 lsval = ((uint32)getword (sr, st, TRUE, SRC) << 16) & lsmsk;
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "   lsval = %08X\n", lsval);
#endif
	 ldval = getlong (dr, dt, FALSE, DST) & ~ldmsk;
	 if (shfcnt >= 0)
	    ldval |= lsval << shfcnt;
	 else
	    ldval |= lsval >> -shfcnt;
	 putlong (dr, dt, ldval, DST);
	 cmplongzero (lsval << (16 - wd));
#ifdef DEBUGEINSTR
	 fprintf (stderr,"   stat = >%04X\n", statreg);
	 fprintf (stderr,
		  "   ldval = %08X\n", ldval);
#endif
      }
      else
      {
	 int shfcnt;

	 shfcnt = 16 - ps - wd;
	 sval = getword (sr, st, TRUE, SRC) & smsk;
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "   sval = %04X\n", sval);
#endif
	 dval = getword (dr, dt, FALSE, DST) & ~dmsk;
	 if (shfcnt >= 0)
	    dval |= sval << shfcnt;
	 else
	    dval |= sval >> -shfcnt;
	 putword (dr, dt, dval, DST);
	 cmpwordzero (sval << (16 - wd));
#ifdef DEBUGEINSTR
	 fprintf (stderr,"   stat = >%04X\n", statreg);
	 fprintf (stderr,
		  "   dval = %04X\n", dval);
#endif
      }
      break;

   case 2: /* XV */

      if (traceit) fprintf (tracefd, "XV");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	    "INST = XV sr = %d, st = %d, dr = %d, dt = %d, ps = %d, wd = %d\n",
	       sr, st, dr, dt, ps, wd);
#endif
      sign = TRUE;
      goto EXTRACT;

   case 3: /* XF */

      if (traceit) fprintf (tracefd, "XF");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	    "INST = XF sr = %d, st = %d, dr = %d, dt = %d, ps = %d, wd = %d\n",
	       sr, st, dr, dt, ps, wd);
#endif
      sign = FALSE;

   EXTRACT:
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "  longvals = %s, ldmsk = >%08X, dmsk = >%04X\n",
	       longvals ? "TRUE" : "FALSE", ldmsk, dmsk);
      fprintf (stderr,"   stat = >%04X\n", statreg);
#endif
      if (longvals)
      {
	 lsval = getlong (sr, st, FALSE, SRC);
	 if (st == 3) PUTREG (sr, GETREG(sr) + 2);
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "   lsval = %08X\n", lsval);
#endif
	 lsval = lsval & ldmsk;
	 lsval <<= ps;
	 cmplongzero (lsval);
	 if (sign)
	    (int32)lsval >>= (16 - wd);
	 else
	    lsval >>= (16 - wd);
	 sval = (uint16)(lsval >> 16);
      }
      else
      {
	 sval = getword (sr, st, TRUE, SRC);
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "   sval = %04X\n", sval);
#endif
	 sval = sval & dmsk;
	 sval <<= ps;
	 cmpwordzero (sval);
	 if (sign)
	    (int16)sval >>= 16 - wd;
	 else
	    sval >>= 16 - wd;
      }
      putword (dr, dt, sval, DST);
#ifdef DEBUGEINSTR
      fprintf (stderr,"   stat = >%04X\n", statreg);
      fprintf (stderr,
	       "   val = %04X\n", sval);
#endif
      break;

   default:
      return (-1);
   }
   return (0);

}

/***********************************************************************
* proc17op - Process type 17 operations.
***********************************************************************/

int
proc17op (uint16 inst)
{
   uint16 inst1;
   uint16 ov;
   uint16 v;
   uint16 r;
   uint16 c;
   int16 wpc;
   int8 disp;

   if (model < 12) return (-1);


   inst1 = GETINST;
   pcreg += 2;

   wpc = (pcreg >> 1) & 0x7FFF;

   disp =  inst1 & 0x00FF;
   r = (inst1 & 0x0F00) >> 8;
   c = (inst1 & 0xF000) >> 12;

   if (c == 0) c = R0;

   switch ((inst & 0x000F) - 0x000C)
   {
   case 0: /* SRJ */

#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = SRJ disp = %d, r = %d, c = %d\n",
	       disp, r, c);
#endif
      if (traceit) fprintf (tracefd, "SRJ >%X R%d %d", disp & 0xFF,r,c);
      ov = GETREG(r);
      v = ov - c;
      PUTREG (r, v);
      if (traceit) fprintf (tracefd, " v = %d", v);
      if (v == 0 || ((int16)v < 0 && (int16)ov > 0)) return (0);
      break;

   case 1: /* ARJ */

#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = ARJ disp = %d, r = %d, c = %d\n",
	       disp, r, c);
#endif
      if (traceit) fprintf (tracefd, "ARJ >%X R%d %d", disp & 0xFF,r,c);
      ov = GETREG(r);
      v = ov + c;
      PUTREG (r, v);
      if (traceit) fprintf (tracefd, " v = %d", v);
      if (v == 0 || ((int16)v > 0 && (int16)ov < 0)) return (0);
      break;

   default:
      return (-1);
   }

   if ((uint8)disp == 0xFF)
   {
      idle = TRUE;
      if (GET_MASK == 0) run = FALSE;
   }
   else
      opcreg = pcreg;

   wpc += disp;
   pcreg = (wpc << 1) & 0xFFFE;
   GETINST; /* force mpcreg */
   return (0);
}

/***********************************************************************
* proc18op - Process type 18 operations.
***********************************************************************/

int
proc18op (uint16 inst)
{
   uint16 r;

   r = inst & 0x000F;

   switch (((inst & 0x00F0) >> 4) - 0x0003)
   {
   case 0: /* STPC */

      if (traceit) fprintf (tracefd, "STPC R%d", r);
      if (model < 12) return (-1);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = STPC r = %d\n", r);
#endif
      PUTREG (r, pcreg-2);
      break;

   case 4: /* LIM */

      if (traceit) fprintf (tracefd, "LIM R%d", r);
      if (model < 12) return (-1);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = LIM r = %d\n", r);
#endif
      if (IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr,
		  "%d: ERRINT requested: nonPRIV LIM", instcount);
#endif
	 tracememory (TRACE_PRIVINST, mpcreg-2);
	 errcrudata |= ERRPIN;
         if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      SET_MASK (GETREG(r));
      break;

   case 5: /* LST */

      if (traceit) fprintf (tracefd, "LST R%d", r);
      if (model < 11) return (-1);
#ifdef DEBUGEINSTR
      fprintf (stderr, 
	       "%08.8ld: PC %04X(%06X) WP %04X ST %04X INST %04X\n",
	       instcount, pcreg-2, mpcreg-2, wpreg, statreg, inst);
      fprintf (stderr,
	       "INST = LST r = %d\n", r);
      fprintf (stderr,
	       "   (r) = >%04X\n   stat = >%04X\n", GETREG(r), statreg);
      {
	 int pr = mapaddr (pcreg-2, GET_MAP, MEM_NOPROT, FALSE);
	 fprintf (stderr, "Code:\n");
	 HEXDUMP (stderr, &memory[pr-16], 32, pr-16);
      }
#endif
      if (IS_NONPRV)
      {
	 uint16 st;
	 st = statreg & 0x01DF;
         st |= GETREG(r) & 0xFC20;
	 statreg = st;
      }
      else
      {
         statreg = GETREG(r);
      }
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "   stat = >%04X\n", statreg);
#endif
      break;

   case 6: /* LWP */

      if (traceit) fprintf (tracefd, "LWP R%d", r);
      if (model < 11) return (-1);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = LWP r = %d\n", r);
#endif
      wpreg = GETREG(r) & 0xFFFE;
      break;

   case 7: /* LCS */

      if (traceit) fprintf (tracefd, "LCS");
      if (model < 12) return (-1);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = LCS r = %d\n", r);
#endif
      if (IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr,
		  "%d: ERRINT requested: nonPRIV LCS", instcount);
#endif
	 tracememory (TRACE_PRIVINST, mpcreg-2);
	 errcrudata |= ERRPIN;
         if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      /* NOP for now */
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc19op - Process type 19 operations.
***********************************************************************/

int
proc19op (uint16 inst)
{
   uint16 inst1;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 ad;

   if (model < 12) return (-1);

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;

   if (inst != 0x002B) return (-1);

#ifdef DEBUGEINSTR
   fprintf (stderr,
	    "INST = MOVA sr = %d, st = %d, dr = %d, dt = %d\n",
	    sr, st, dr, dt);
#endif
   if (traceit) fprintf (tracefd, "MOVA");
   ad = getaddr (sr, st, TRUE);
   putword (dr, dt, ad, DST);
   
   return (0);
}

/***********************************************************************
* proc20 - Process type 20 operations.
***********************************************************************/

int
proc20op (uint16 inst)
{
   uint16 inst1;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 cc;
   uint16 lscb[5];
   uint16 tbl[2];

   if (model < 12) return (-1);

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   cc = (inst1 & 0xF000) >> 12;

   switch (inst & 0x000F)
   {
   case 1: /* SLSL */

      if (traceit) fprintf (tracefd, "SLSL");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = SLSL sr = %d, st = %d, dr = %d, dt = %d, cc = %d\n",
	       sr, st, dr, dt, cc);
#endif
      getlscb (sr, st, TRUE, SRC, lscb);
      gettbl (dr, dt, FALSE, DST, tbl);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "   lscb = >%04X >%04X >%04x >%04X >%04X\n",
	       lscb[0],lscb[1],lscb[2],lscb[3],lscb[4]); 
      fprintf (stderr,
	       "   tbl = >%04X >%04X\n",
	       tbl[0],tbl[1]);
#endif
      CLR_EQ;
      do
      {
	 uint16 val;

         val = GETMEM ((int16)tbl[0]+(int16)lscb[1], DST);
	 if (chkvalcc (val, cc, lscb) == TRUE)
	 {
	    SET_EQ;
	    break;
	 }
	 tbl[1] = tbl[0];
	 tbl[0] = GETMEM ((int16)tbl[1]+(int16)lscb[0], DST);
      } while (tbl[0] != lscb[4]);

      puttbl (dr, dt, tbl, DST);
      break;

   case 2: /* SLSP */

      if (traceit) fprintf (tracefd, "SLSP");
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = SLSP sr = %d, st = %d, dr = %d, dt = %d, cc = %d\n",
	       sr, st, dr, dt, cc);
#endif
      if (IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr,
		  "%d: ERRINT requested: nonPRIV SLSP", instcount);
#endif
	 tracememory (TRACE_PRIVINST, mpcreg-2);
	 errcrudata |= ERRPIN;
         if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      getlscb (sr, st, TRUE, SRC, lscb);
      gettbl (dr, dt, FALSE, DST, tbl);
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "   lscb = >%04X >%04X >%04x >%04X >%04X\n",
	       lscb[0],lscb[1],lscb[2],lscb[3],lscb[4]); 
      fprintf (stderr,
	       "   tbl = >%04X >%04X\n",
	       tbl[0],tbl[1]);
      fprintf (stderr,
	       "   addr = ");
#endif
      CLR_EQ;
      do
      {
	 uint32 addr;
	 uint16 val;

	 if (IS_MAP)
	 {
	    addr = (int16)tbl[0]+(int16)lscb[1];
	    val = GETMEM ((uint16)addr, DST);
	 }
	 else
	 {
	    addr = (uint32)((int32)(tbl[0]<<5)+(int16)lscb[1]);
	    val = GETMEM0 (addr);
	 }
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  " >%06X", addr);
#endif
	 if (chkvalcc (val, cc, lscb) == TRUE)
	 {
	    SET_EQ;
	    break;
	 }
	 tbl[1] = tbl[0];
	 if (IS_MAP)
	 {
	    addr = (int16)tbl[1]+(int16)lscb[0];
	    tbl[0] = GETMEM ((uint16)addr, DST);
	 }
	 else
	 {
	    addr = (uint32)((int32)(tbl[1]<<5)+(int16)lscb[0]);
	    tbl[0] = GETMEM0 (addr);
	 }
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  ":>%06X\n", addr);
#endif
      } while (tbl[0] != lscb[4]);

      puttbl (dr, dt, tbl, DST);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc21 - Process type 21 operations.
***********************************************************************/

int
proc21op (uint16 inst)
{
   int i;
   uint16 inst1;
   uint16 dt;
   uint16 dr;
   uint16 dl;
   uint16 st;
   uint16 sr;
   uint16 sl;
   uint16 ct;
   uint16 sad;
   uint16 dad;
   uint8 carry, ocarry;
   uint8 sdat[18];

   if (model < 12) return (-1);

   if ((inst & 0xFFF0) != 0x03F0) return (-1);

   dl = inst & 0x000F;

   inst1 = GETINST;
   pcreg += 2;

   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   sl = (inst1 & 0xF000) >> 12;

   if (sl == 0)
   {
      sl = R0 & 0x000F;
      if (sl == 0) sl = 16;
   }
   if (dl == 0)
   {
      dl = (R0 >> 8) & 0x000F;
      if (dl == 0) dl = 16;
   }
#ifdef DEBUGEINSTR
   fprintf (stderr,
	    "INST = EP sr = %d, st = %d, sl = %d, dr = %d, dt = %d, dl = %d\n",
	    sr, st, sl, dr, dt, dl);
#endif
   if (traceit) fprintf (tracefd, "EP %d->%d", sl, dl);
   sad = getaddr (sr, st, FALSE);
   dad = getaddr (dr, dt, FALSE);
   putaddr (sr, st, sl);
   putaddr (dr, dt, dl);

   if (sl > dl)
   {
      SET_OVER;
      return (0);
   }
   else CLR_OVER;
   

   for (i = 0; i < sl; i++)
   {
      sdat[i] = GETMEMB (sad+i, SRC);
   }

   ct = (dl - sl) * 8;

   for (; ct; ct--)
   {
      ocarry = 0;
      for (i = 0; i < dl; i++)
      {
	 carry = sdat[i] & 1;
	 if (i == 0) sdat[i] = (int8)sdat[i] >> 1;
	 else sdat[i] = sdat[i] >> 1;
	 sdat[i] |= ocarry << 7;
	 ocarry = carry;
      }
   }

   ct = FALSE;
   for (i = 0; i < dl; i++)
   {
      PUTMEMB (dad+i, sdat[i], DST);
      if (sdat[i]) ct = TRUE;
   }

   CLR_AGT;
   CLR_LGT;

   if (!ct)
   {
      SET_EQ;
   }
   else
   {
      CLR_EQ;
      SET_LGT;
      if ((int8)sdat[0] >= 0) SET_AGT;
   }

   return (0);
}
