/************************************************************************
*
* lnk990 - Links objects from asm990
*
* Changes:
*   06/05/03   DGP   Original.
*   06/25/03   DGP   Changed to print listing like TXLINK.
*                    Added printsymbols function.
*                    Added partial link.
*                    Added multiply defined and undefined symbols.
*   07/08/03   DGP   Moved get date/time to always get it.
*   04/08/04   DGP   Added "Cassette" mode.
*   05/25/04   DGP   Added long ref/def support.
*	
************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#include "lnkdef.h"

FILE *lstfd = NULL;
int listmode = FALSE;
int pc; /* the linker pc */
int absolute = FALSE;
int partiallink = FALSE;
int symbolcount = 0;
int errcount = 0;
int pgmlength = 0;
int absentry = -1;
int relentry = -1;
int linecnt = MAXLINE;
int modcount = 0;
int undefs = FALSE;
int muldefs = FALSE;
int cassette = FALSE;
char errline[120];
char inbuf[MAXLINE];
char idtbuf[IDTSIZE+2];
struct tm *timeblk;

SymNode *symbols[MAXSYMBOLS];
Module modules[MAXMODULES];

uint8 memory[MEMSIZE];
Memory memctl[MEMSIZE];

static int pagenum = 0;
static char datebuf[48];

/***********************************************************************
* printheader - Print header on listing.
***********************************************************************/

void
printheader (FILE *lstfd)
{
   if (linecnt > LINESPAGE)
   {
      linecnt = 0;
      if (pagenum) fputc ('\f', lstfd);
      fprintf (lstfd, H1FORMAT,
	       VERSION,
	       idtbuf,
	       datebuf,
	       ++pagenum);
   }
}

/***********************************************************************
* printsymbols - Print the symbol table.
***********************************************************************/

static void
printsymbols (FILE *lstfd)
{
   int i, j;
   int longformat;
   char type;
   char type1;

   j = 0;
   longformat = FALSE;

   fprintf (lstfd, "\n                        D E F I N I T I O N S \n\n");
   linecnt++;

   for (i = 0; i < symbolcount; i++)
      if (symbols[i]->longsym) longformat = TRUE;

   for (i = 0; i < symbolcount; i++)
   {
      printheader (lstfd);

      if (partiallink)
      {
	 if (symbols[i]->global) type = 'G';
	 else if (symbols[i]->external) type = 'E';
      }
      else if (symbols[i]->relocatable) type = '\'';
      else type = ' ';

      if (symbols[i]->muldef)
      {
	 muldefs = TRUE;
	 type1 = 'M';
      }
      else if (symbols[i]->undef)
      {
	 if (!partiallink) errcount++;
	 undefs = TRUE;
	 type1 = 'U';
      }
      else type1 = ' ';

      if (longformat)
      {
	 fprintf (lstfd, LONGSYMFORMAT,
		  symbols[i]->symbol,
		  symbols[i]->value,
		  type, type1);
	 j = 4;
      }
      else
      {
	 fprintf (lstfd, SYMFORMAT,
		  symbols[i]->symbol,
		  symbols[i]->value,
		  type, type1);
	 j++;
      }
      if (j == 4)
      {
	 fprintf (lstfd, "\n");
	 linecnt++;
	 j = 0;
      }
   }
   fprintf (lstfd, "\n");
}

/***********************************************************************
* Main procedure
***********************************************************************/

main (int argc, char **argv)
{
   FILE *infd = NULL;
   FILE *outfd = NULL;
   char *infile[MAXFILES];
   char *outfile = NULL;
   char *lstfile = NULL;
   char *bp;
   int i;
   int incnt = 0;
   int status = 0;
   time_t curtime;
  
#ifdef DEBUG
   printf ("lnk990: Entered:\n");
   printf ("args:\n");
   for (i = 1; i < argc; i++)
   {
      printf ("   arg[%2d] = %s\n", i, argv[i]);
   }
#endif

   /*
   ** Clear files.
   */

   for (i = 0; i < MAXFILES; i++)
   {
      infile[i] = NULL;
   }

   /*
   ** Clear symboltable
   */

   for (i = 0; i < MAXSYMBOLS; i++)
   {
      symbols[i] = NULL;
   }

   /*
   ** Scan off the the args.
   */

   idtbuf[0] = '\0';

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'c':
	    cassette = TRUE;
	    break;

	 case 'i':
	    i++;
	    if (strlen(argv[i]) >= IDTSIZE) argv[i][IDTSIZE] = '\0';
	    strcpy (idtbuf, argv[i]);
	    break;

	 case 'l':
	    i++;
	    lstfile = argv[i];
	    listmode = TRUE;
	    break;

         case 'o':
            i++;
            outfile = argv[i];
            break;

	 case 'p':
	    partiallink = TRUE;
	    break;

         default:
      USAGE:
	    printf ("lnk990 - version %s\n", VERSION);
	    printf (
	 "usage: lnk990 [-options] -o file.bin infile.bin...\n");
            printf (" options:\n");
	    printf ("    -c           - Cassette mode\n");
            printf ("    -i IDT       - Define IDT name for link\n");
	    printf ("    -p           - Partial linking\n");
	    printf ("    -l listfile  - Generate map listing to listfile\n");
	    return (ABORT);
         }
      }

      else
      {
         infile[incnt++] = argv[i];
      }

   }

   if (!infile[0] && !outfile) goto USAGE;

#ifdef DEBUG
   printf (" outfile = %s\n", outfile);
   for (i = 0; i < incnt; i++)
      printf (" infile[%d] = %s\n", i, infile[i]);
   if (listmode)
      printf (" lstfile = %s\n", lstfile);
#endif

   /*
   ** Open the files.
   */

   if ((outfd = fopen (outfile, "w")) == NULL)
   {
      perror ("lnk990: Can't open output file");
      exit (1);
   }
   if (listmode) {
      if ((lstfd = fopen (lstfile, "w")) == NULL)
      {
	 perror ("lnk990: Can't open listing file");
	 exit (1);
      }

   }

   /*
   ** Get current date/time.
   */

   curtime = time(NULL);
   timeblk = localtime (&curtime);
   strcpy (datebuf, ctime(&curtime));
   *strchr (datebuf, '\n') = '\0';

   /*
   ** Clear memory.
   */

   memset (&memory, '\0', sizeof(memory));
   for (i = 0; i < MEMSIZE; i++)
   {
      memset (&memctl[i], '\0', sizeof (Memory));
   }

   /*
   ** Load the objects for pass 1.
   */

   for (i = 0; i < incnt; i++)
   {
      if ((infd = fopen (infile[i], "r")) == NULL)
      {
	 sprintf (inbuf, "lnk990: Can't open input file %s", infile[i]);
	 perror (inbuf);
	 exit (1);
      }

      if (pc & 1) pc ++;
      status = lnkloader (infd, pc, 1, infile[i]);

      fclose (infd);
   }

   /*
   ** Clear memory.
   */

   memset (&memory, '\0', sizeof(memory));
   for (i = 0; i < MEMSIZE; i++)
   {
      memset (&memctl[i], '\0', sizeof (Memory));
   }

   /*
   ** Print the modules
   */

   if (listmode)
   {
      printheader (lstfd);
      fprintf (lstfd,
	 "  MODULE    ORIGIN  LENGTH    DATE      TIME    CREATOR   FILE\n\n");
      linecnt += 2;
      for (i = 0; i < modcount; i++)
      {
	 printheader (lstfd);
         fprintf (lstfd, MODFORMAT, modules[i].name, modules[i].origin,
		  modules[i].length, modules[i].date, modules[i].time,
		  modules[i].creator, modules[i].objfile);
	 linecnt ++;
      }
   }

   /*
   ** reLoad the objects for pass 2.
   */

   pc = 0;
   for (i = 0; i < incnt; i++)
   {

      if ((infd = fopen (infile[i], "r")) == NULL)
      {
	 sprintf (inbuf, "lnk990: Can't open input file %s", infile[i]);
	 perror (inbuf);
	 exit (1);
      }

      if (pc & 1) pc ++;
      status = lnkloader (infd, pc, 2, infile[i]);

      fclose (infd);
   }

   if (listmode)
   {

      /*
      ** Print the symbol table
      */

      printsymbols (lstfd);

      printheader (lstfd);
      fprintf (lstfd, "\n");

      if (undefs)
	 fprintf (lstfd, "** There are undefined symbols - U\n");
      if (muldefs)
	 fprintf (lstfd, "** There are multiply defined symbols - M\n");

      fprintf (lstfd, "\n%04X Length of program \n", pc);
      if (relentry >= 0)
	 fprintf (lstfd, "%04X Program entry \n", relentry);
      fprintf (lstfd, "%d Errors\n", errcount);
   }

   if (errcount)
      printf ("lnk990: %d errors\n", errcount);

   /*
   ** Punch out linked object
   */

   if (!errcount)
      lnkpunch (outfd, partiallink);

   /*
   ** Close the files
   */

   fclose (outfd);
   if (listmode)
      fclose (lstfd);

   return (status == 0 ? NORMAL : ABORT);
}
