/*
 * Distributed V Kernel
 * Copyright (c) 1986 by Stanford University, All rights reserved.
 *
 * Kernel Rqdx disk driver, William Lees
 *
 * $Revision: 1.16.1.3 $
 * $Locker:  $
 * $State: Exp $
 */

/*
 * This source file contains the major entry points for RQDX device server.
 * These include the device power-up routine, and well as the subroutines
 * for manipulating device UIO objects:
 *	Create, Release, Read, Write, Modify and Query
 */

/* Common definitions */

#include <Vioprotocol.h>
#include <Vquerykernel.h>
#include "externals.h"
#include "process.h"
#include "uvaxmemory.h"
#include "memory.h"
#include "interrupt.h"
#include "devman.h"
#include "rqdx.h"
#include "mscp.h"

/* Exports - for Mscp module */

short			RqdxDriverState;	   /* Drive state (software) */
struct rqdx_unit_info	RqdxUnitInfo[NUM_RQDX_DISKS];	/* Info on each unit */

/* Forward */

ResponseCode RqdxCreate _TAKES(( Process *, DeviceInstance *));
ResponseCode RqdxReadWrite _TAKES
  (( Process *, DeviceInstance *));  /* Common code for reading and writing */
ResponseCode RqdxQuery _TAKES((Process *, DeviceInstance *, unsigned short));
ResponseCode RqdxModify _TAKES((Process *, DeviceInstance *, unsigned short));
ResponseCode RqdxRelease _TAKES(( Process *, DeviceInstance *));

unsigned PossibleRqdxQBusAddrs[] = POSSIBLE_RQDX_QBUS_ADDRESSES;
unsigned RqdxQBusAddr;

unsigned char FindRqdxDisks()
    /* Probe each of the possible RQDX Q-bus addresses, in order: */
  {
    int i;

    for (i = 0; i < sizeof(PossibleRqdxQBusAddrs)/sizeof(unsigned); ++i)
      {
	unsigned qBusAddr = PossibleRqdxQBusAddrs[i];
	
	if (Probe((short *)PhysRqdx(qBusAddr), PROBE_READ))
	  {
	    printx("Found an RQDX disk controller (Q bus addr = %08o)\n",
		    qBusAddr);
	    RqdxQBusAddr = qBusAddr;
	    return PRF_DISK_RQDX;
	  }
      }
    return PRF_NONE;
  }

void
BuildSysMap_Rqdx()
  {
    /*
     * We could ask for a slice of the kernel memory map, but at present we
     *   don't seem to need one.
     */
  }

/****************************************************************************
 * Routine name:	RqdxPowerup
 *
 * Description:
 *
 *	This routine is called at kernel startup to init the controller.
 *
 * Inputs:	None.
 *
 * Outputs:
 *
 *	ResponseCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	SavedIPL	Temp storage for hardware IPL
 *
 */

unsigned	SavedIPL;	/* Storage for IPL */

ResponseCode RqdxPowerup()
{
	int i;
	ResponseCode s;
	struct rqdx_unit_info *ui = &RqdxUnitInfo[0];

	for (i = 0; i < NUM_RQDX_DISKS; i++, ui++)
	    {
	    ui->unit_diskq.head = NULL;
	    ui->unit_diskq.tail = NULL;
	    ui->unit_diskq.type = DISK_QUEUE;
	    }
/*
 * Call MscpStartInit() to start controller port initialization sequence.  That
 *   routine lowers the interrupt priority level so that interrupts can occur;
 *   we then spin, processing interrupts until the (first phase of)
 *   initialization completes successfully or an error occurs.
 * We save and restore the interrupt priority level so that the kernel
 *   initialization code executed after us won't get a nasty surprise.
 */
	asm("	mfpr	$ipl, _SavedIPL");	/* Save old IPL */

	if ( ( s = MscpStartInit() ) != OK ) return(s);

	while ( (RqdxDriverState != S_IDLE) &&
		(RqdxDriverState != S_RUN) &&
		(s == OK ) ) ;

	asm("	mtpr	_SavedIPL, $ipl");	/* Restore old IPL */

	return( s );
};


/****************************************************************************
 * Routine name:	RqdxDoUnitSetup
 *
 * Description:
 *
 *	Called by the the Mscp interrupt-handler (yup, an up-call -- ugh) when
 *	it discovers a unit that seems to have a real device attached.  The
 *	device may not necessarily be online.
 *
 *	Does initialization which is required for this unit to become usable
 *	as a kernel device:  allocates a slice of the Q-bus map for it and adds
 *	its name to the device table.
 *
 * Inputs:
 *
 *	unit	Unit-number (0 <= unit < NUM_RQDX_DISKS)
 *
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxUnitInfo
 *	Kernel device table (in mi/dm.c)
 *
 */

void RqdxDoUnitSetup(unit)
	int unit;
{
	register struct rqdx_unit_info *ui;
	char Vname[6];

	if (unit < 0 || unit >= NUM_RQDX_DISKS) {
		printx("RqdxDoUnitSetup: got unexpected unit-number (0x%x)\n",
			unit);
		return;
	}
	ui = &RqdxUnitInfo[unit];
	if (ui->unit_done_setup)
		return;

	/* Hope it's OK to do this on an interrupt */
	AllocateDvmaSpace(RQDX_DISK_BUFFER_SIZE, &ui->unit_DMA_Address);
	Copy(Vname,"diskX", 6);
	Vname[4] = unit % 10 + '0';
	AddDevice(Vname, unit, RqdxCreate, 0, 0);
	printx("[device/%s has DEC device type %s, media name %s\n",
		Vname, ui->device_type, ui->media_name);
	ui->unit_done_setup = 1;
}

/****************************************************************************
 * Routine name:	RqdxCreate
 *
 * Description:
 *
 *	This routine creates a UIO object for one of the units on the
 *	RQDX controller.
 *
 * Inputs:	None.
 *
 * Outputs:
 *
 *	ResponseCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxDriverState
 *	RqdxUnitInfo
 *
 */

ResponseCode RqdxCreate( pd, desc )
	Process *pd;
	DeviceInstance *desc;
{
	CreateInstanceRequest *req = (CreateInstanceRequest *) &(pd->msg);

/* Checks.  Unit number must be in range, controller and unit must each
 * be on-line. 
 */
	if ( desc->unit < 0 || desc->unit >= NUM_RQDX_DISKS )
		return( BAD_ARGS );

	if ( RqdxDriverState != S_RUN ) return( DEVICE_ERROR );

	if ( RqdxUnitInfo[desc->unit].unit_online == 0 )
		return( DEVICE_ERROR );

/* Initialize the device instance descriptor. */

	desc->id |= UIO_READABLE|UIO_WRITEABLE|MULTI_BLOCK|UIO_RANDOM_ACCESS;
	if ( req->filemode == FREAD )
	    desc->id &= ~UIO_WRITEABLE;
	desc->owner = pd->pid;

	desc->readfunc = RqdxReadWrite;
	desc->writefunc = req->filemode == FREAD ? NotWriteable : RqdxReadWrite;
	desc->modifyfunc = RqdxModify;
	desc->queryfunc = RqdxQuery;
	desc->releasefunc = RqdxRelease;
	

/* Fill in unit specific info */

	desc->blocksize = SECSIZE * BLOCK_FACTOR;
	desc->lastblock =
		RqdxUnitInfo[desc->unit].unit_last_block / BLOCK_FACTOR;

	return( OK );
}


/****************************************************************************
 * Routine name:	RqdxReadWrite
 *
 * Description:
 *
 *	This routine is the common code to read or write a UIO object
 *
 * Inputs:
 *
 *	pd	Descriptor of requesting process
 *	desc	UIO
 *
 * Outputs:
 *
 *	ResponseCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxUnitInfo
 *
 */

ResponseCode RqdxReadWrite( pd, desc )
	Process *pd;
	DeviceInstance *desc;
{
	IoRequest *req = (IoRequest *) &(pd->msg);

/* Check that unit in question is online and that bytecount is valid.
 * Byte count must be even, smaller than disk buffer, and smaller than
 * disk size.
 */
	if ( RqdxUnitInfo[desc->unit].unit_online == 0 )
		return( DEVICE_ERROR );

	if ( (req->bytecount & 1) == 1 ) {
		req->bytecount = 0;
		return( BAD_BYTE_COUNT);
	}

	if ( req->bytecount > RQDX_DISK_BUFFER_SIZE ) {
		req->bytecount = 0;
		return ( BAD_BYTE_COUNT );
	}

	if ((req->blocknumber * BLOCK_FACTOR) >
			 RqdxUnitInfo[desc->unit].unit_last_block) {
		req->bytecount = 0;
		return( BAD_BLOCK_NO );
	};
	/*
	 * Check if the block number plus the bytecount is larger than the disk.
	 */
	if ((req->blocknumber * SECSIZE * BLOCK_FACTOR) + req->bytecount >
		(RqdxUnitInfo[desc->unit].unit_last_block *
			SECSIZE * BLOCK_FACTOR)) {
		req->bytecount = 0;
		return( BAD_BYTE_COUNT );
	};

	MscpUnitBlockIO(pd);

	return( NO_REPLY );	/* Interrupt routine will reply */
}

/****************************************************************************
 * Routine name:	RqdxRelease
 *
 * Description:
 *
 *	This routine disassociates a process from a UIO object, which
 *	is statically allocated and not disposed of.
 *
 * Inputs:
 *
 *	pd	Descriptor of reqesting process
 *	desc	UIO
 *
 * Outputs:
 *
 *	ResponseCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 */

ResponseCode RqdxRelease( pd, desc)
    Process *pd;
    DeviceInstance *desc;
  {
    IoRequest *req
	= (IoRequest *) &(pd->msg);
    if (req->releasemode & ABORT_CHANGES)
	return MODE_NOT_SUPPORTED; /* It's too late to back out now! */
    if (!(req->releasemode & RETAIN_UIO))
	desc->owner = 0;

    return OK;
  }

/****************************************************************************
 * Routine name:	RqdxQuery
 *
 * Description:
 *
 *	This routine is supposed to return information or statistics
 *	about the Rqdx device.  Currently it returns the maximum number 
 *	bytes per request.
 *
 * Inputs:
 *
 *	pd		Descriptor of requesting process
 *	desc		UIO
 *	dirIndex	
 *
 * Outputs:
 *
 *	ResponseCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 */

ResponseCode RqdxQuery( pd, desc, dirIndex )
	Process *pd;
	DeviceInstance *desc;
	unsigned short dirIndex;
{
	IoReply *reply = (IoReply *)&pd->msg;
	reply->bytecount = RQDX_BUFFER_BLOCKS * SECSIZE * BLOCK_FACTOR;
	return( OK );
}

/****************************************************************************
 * Routine name:	RqdxModify
 *
 * Description:
 *
 *	This routine allows one to modify characteristics of a UIO object.
 *	Currently a nop.
 *
 * Inputs:
 *
 *	pd		Descriptor of requesting process
 *	desc		UIO
 *	dirIndex	
 *
 * Outputs:
 *
 *	ResponseCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 */

ResponseCode RqdxModify( pd, desc, dirIndex)
	Process *pd;
	DeviceInstance *desc;
	unsigned short dirIndex;
{
	printx("[RqdxModify]\n");
	return( OK );
}

