/*
 * Distributed V Kernel
 * Copyright (c) 1986 by Stanford University, All rightrs reserved.
 *
 * Kernel MSCP disk driver, William Lees
 *
 * $Revision: 1.22.1.8 $
 * $Locker:  $
 * $State: Exp $
 */


/* This source file encapuslates the subroutines and data structures
 * required to speak the MSCP protocol.  There are three entry points:
 *
 *	o MscpStartInit		to start the port init sequence
 *	o MscpUnitBlockIO	to initiate a transfer
 *	o MscpInterrupt		when a controller interrupt occurs
 *
 * The hardware port init sequence used by the RQDX is the same as that
 * used by the UDA devices of larger VAXen.  The aspects of MSCP used will
 * be a superset of those employed here.  The major enhancement required
 * to this driver is to implementent separate data structures for each
 * controller, i.e. separate communications area, separate packets, etc
 *
 * In order for the controller to DMA to host memory it must be mapped by
 * the Qbus registers.  As of this writing only kernel memory is mapped.
 * This requires a double buffering scheme.  Note that VAX memory to memory
 * transfers are pretty fast.  Because large buffers are required, only one
 * transfer is allowed to use the single buffer at a time.  Unfortunately,
 * this prevents the controller from performing seek optimization. In the
 * future, Qbus registers should be allocated to map the user buffers.
 *
 * Speaking of addressing, Qbus addresses are only 22 bits.  Only kernel memory
 * is mapped in.  To convert a kernel address to a Qbus address suitable for
 * passing to the controller, just use the low order 22 bits.
 *
 * When concurrent transfers are implemented, note that each outstanding
 * command must have a unique command reference number.  Using the requestor's
 * pid or a pointer to an additional information data block works well.
 * The latter is a mechanism for passing software context to the response
 * handling routine.
 */

/* Common definitions */
#include <Vioprotocol.h>
#include <Venviron.h>
#include <Vprocesscommon.h>
#include "process.h"
#include "devman.h"
#include "externals.h"
#include "interrupt.h"
#include "uvaxmemory.h"
#include "rqdx.h"		/* RQDX controller */
#include "mscp.h"		/* MSCP protocol */

/* Routine Exports */

extern MscpUnitBlockIO();		/* Block transfer */


extern short RqdxDriverState;			/* State of controller */
extern struct rqdx_unit_info RqdxUnitInfo[3];	/* State of the units */

/* Assembly lnguage Forward */
_BEGIN_EXTERN_C
void  MscpInterrupt();	/* Interrupt-handler function */
void  Asm_MscpInterrupt();	/* Entry point to interrupt-handler */
_END_EXTERN_C

/* Forward */
void MscpCommunicationInit();	/* Initialize data structures */
void MscpControllerInit();	/* Set controller characteristics */

void MscpResponsePacket _TAKES((int)); /* Handle a response packet */
void MscpHandleErrorLogMsg
   _TAKES((struct  mslg *));     /* Handle a error log (datagram) msg */

void MscpUnitOnline _TAKES((short)); /* Send packet to bring unit on-line */
void MscpGetUnitStatus _TAKES((short));	/* Send packet to get unit status */

void MscpConvertMediaId
  _TAKES(( unsigned, char *, char *));  /* Decode media id field */

void MscpDecodeStatusEvent 
_TAKES(( struct rqdx_unit_info *, unsigned short)); /* Decode status / event code */

struct mscp * MscpGetCommandPacket();	/* Find next free command packet */

/* Private */

static void MscpBuildAndGo _TAKES((Process *));

short		Credits;	/* Transfer credits */
short		LastCommand;	/* Pointer into command ring */
short		LastResponse;	/* Pointer into response ring */
struct mscp	Response[NRSP];	/* Response packets */
struct mscp	Command[NCMD];	/* Command packets */

struct rqdx_ca Rqdx_ca;		/* RQDX Communications area */

/* Horrible hack: What's the real way to tell whether this is an optical disk?*/
/* (The Emulex UC04 MSCP controller is version 6, model 6.) */
#define isOpticalDisk(ui)  (strcmp((ui)->media_name, "UO00") == 0)


/****************************************************************************
 * Routine name:	Asm_MscpInterrupt
 *
 * Description:
 *
 *	This routine is a jacket which perserves registers before
 *	calling the C interrupt routine.  It is produced from a macro
 *	expansion.
 *
 * Inputs:	None.
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

Call_inthandler(MscpInterrupt);

/****************************************************************************
 * Routine Name:	MscpStartInit
 *
 * Description:
 *
 *	This routine starts the port init sequence for the controller.  It
 *	lowers the interrupt priority level so interrupts can occur; the rest
 *	of the initialization sequence is performed in MscpInterrupt().
 *
 * Inputs:	None.
 *
 * Outputs:
 *
 *	ResponseCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxDriverState
 *	Processor's IPL
 */

ResponseCode MscpStartInit()
{
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;

	printx("Initializing disk controller...");

	RqdxDriverState = S_IDLE;
	setexvec(Asm_MscpInterrupt, VecRqdx);	/* Load interrupt vector */
	asm("	mtpr	$15, $ipl");		/* Allow device interrupts */

	rqdx->ip = 0;	/* Writing to IP dev reg means initialize */

	while ( (rqdx->sa & UDA_STEP1) == 0) ;
	if (rqdx->sa & UDA_ERR) return( DEVICE_ERROR );

/* Tell it number of command packets, number of response packets, to
 * interrupt during this sequence, and our interrupt vector.
 */
	RqdxDriverState = S_STEP1;
	rqdx->sa = UDA_ERR | (NCMDL2<<11) | (NRSPL2<<8) |
			UDA_IE | RQDX_INTERRUPT;

	return( OK );	/* Sequence continues in interrupt routine */
}

/****************************************************************************
 * Routine name:	MscpInterrupt
 *
 * Description:
 *
 *	This routine is called by the assembly language jacket when a
 *	controller interrupt occurs.  This routine handles the remainder
 *	of the port initialization sequence as well as passing response
 *	packets off for handling.
 *
 * Inputs:	None.
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxDriverState
 *	Rqdx_ca
 */

void MscpInterrupt()
{
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;
	unsigned temp;
	int i;

	switch (RqdxDriverState) {

	case S_IDLE:
		printx("[Random RQDX interrupt ignored]\n");
		return;

/* We've received an interrupt and we were in Step 1.  If the Step 2 flag
 * isn't set then error.  Load the first half of the communication area
 * into SA.  Go to state step 2.  Note the two step assignment to the
 * address register.  It is required because of a compiler bug.
 */
	case S_STEP1:
		if ( (rqdx->sa & UDA_STEP2) == 0 ) {
			RqdxDriverState = S_IDLE;
			return;
		};
		temp = (unsigned) &Rqdx_ca.ResponseDesc[0];	/* Kludge */
		rqdx->sa = (short) temp;			/* Kludge */
		RqdxDriverState = S_STEP2;
		return;

/* We've received an interrupt and we were in Step 2.  If the Step 3 flag
 * isn't set then error.  Load the upper half of the communication area
 * into SA.  Go to state step 3.
 */
	case S_STEP2:
		if ( (rqdx->sa & UDA_STEP3) == 0 ) {
			RqdxDriverState = S_IDLE;
			return;
		};
		temp = ( ( (unsigned) &Rqdx_ca.ResponseDesc[0] ) >> 16) & 0x3f;
		rqdx->sa = (short) temp;			/* Kludge */
		RqdxDriverState = S_STEP3;
		return;

/* We've received an interrupt and we were in Step 3.  If the Step 4 flag
 * isn't set then error.  We made it!  Find out the version and model, then
 * init the data structures and send the controller its first packet!
 */
	case S_STEP3:
		if ( (rqdx->sa & UDA_STEP4) == 0 ) {
			RqdxDriverState = S_IDLE;
			return;
		};
		printx("...port init seq OK, version %d model %d.\n",
			(rqdx->sa & 0xf), ( (rqdx->sa>>4) & 0xf ) );

		rqdx->sa = UDA_GO;
		RqdxDriverState = S_SCHAR;

		MscpCommunicationInit();
		MscpControllerInit();
		return;    

	case S_SCHAR:
	case S_RUN:
		break;
	default:
		printx("[Random RQDX interrupt ignored]\n");
		return;
	}

/* Check for controller error */

	if (rqdx->sa & UDA_ERR)
		printx("Rqdx fatal error (sa = %x)\n", rqdx->sa);
	else

/* Check for buffer purge request */

	if (Rqdx_ca.ca_bdp) {
		printx("[UDA buffer purge]\n");
		Rqdx_ca.ca_bdp = 0;
		rqdx->sa = 0; /* Signal buffer purge complete */
	}
	else

/* Check for response ring transtion */

	if (Rqdx_ca.ResponseInterrupt) {
/*		printx("[ResponseInterrupt]\n"); */
		Rqdx_ca.ResponseInterrupt = 0;
		for (i = LastResponse;; i++) {
			i %= NRSP;
			if ( Rqdx_ca.ResponseDesc[i] & UDA_OWN ) break;
			MscpResponsePacket(i);
			Rqdx_ca.ResponseDesc[i] |= UDA_OWN;
		}
		LastResponse = i;
	}
	else

/* Check for command ring transition */

	if (Rqdx_ca.CommandInterrupt) {
		printx("[CommandInterrupt]\n");
		Rqdx_ca.CommandInterrupt = 0;
	}
	else
		printx("[Random RQDX interrupt ignored]\n");

};

/****************************************************************************
 * Routine Name:	MscpCommunicationInit
 *
 * Description:
 *
 *	This routine initializes the packet descriptors.
 *
 * Inputs:	None
 *
 * Outputs:	None
 *
 * Implicit Inputs/Outputs:
 *
 *	LastCommand
 *	LastResponse
 *	Credits
 *	Rqdx_ca
 */

void MscpCommunicationInit()
{
	int i;

	LastCommand = 1;
	LastResponse = 0;
	Credits = 0;

#define qbus(object) ( ( (unsigned) &(object) ) & 0x3fffff )

/* Mark the response descriptors as owned by the controller.  Not having
 * the INT flag set means its a new one for us to read.
 */
	for (i = 0; i < NRSP; i++) {
		Rqdx_ca.ResponseDesc[i] =
			(UDA_OWN | UDA_INT | qbus(Response[i].mscp_cmd_ref) );
		Response[i].mscp_header.uda_msglen = mscp_msglen;
	};

/* Mark the command descriptor as interrupt enable on completion. */

	for (i = 0; i < NCMD; i++) {
		Rqdx_ca.CommandDesc[i] =
			(UDA_INT | qbus( Command[i].mscp_cmd_ref ) );
		Command[i].mscp_header.uda_msglen = mscp_msglen;
	};
};


/****************************************************************************
 * Routine Name:	ControllerInit
 *
 * Description:
 *
 *	This routine sends the first packet to the controller.  It is not
 *	built using the GetCommandPacket subroutine because at this point
 *	no credits have been assigned.  This packet does a "set controller
 *	characteristics" to tell it what kind of datagrams we want.
 *
 * Inputs:	None.
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:
 *
 *	Command
 *	Rqdx_ca
 *
 */

void MscpControllerInit()
{
	short i;
	struct mscp *mp;
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;

	mp = &Command[0];

	mp->mscp_opcode = M_OP_STCON;
	mp->mscp_cmd_ref = 0;
	mp->mscp_flags = mp->mscp_modifier = 0;

	mp->mscp_version = 0;
	mp->mscp_cnt_flgs = M_CF_ATTN | M_CF_MISC | M_CF_THIS;
	mp->mscp_hst_tmo = mp->mscp_use_frac = 0;
	mp->mscp_time_low = mp->mscp_time_high = 0;
	mp->mscp_cnt_dep = 0;

/* Setting the OWN flag tells the controller it can have it */

	Rqdx_ca.CommandDesc[0] |= UDA_OWN | UDA_INT;

/* Reading from IP means start looking for commands */

	i = rqdx->ip;	/* Initiate polling */
};

/****************************************************************************
 * Routine Name:	ResponsePacket
 *
 * Description:
 *
 *	This routine processes a response packet found by the interrupt
 *	routine.  A response packet that indicates the completion of a
 *	command has the M_OP_END bit set in the opcode field.
 *
 * Inputs:
 *
 *	Index	Index of response packet to process
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:
 *
 *	Response
 *	RqdxDriverState
 *
 */
BooleanInt printErrorLogMsgs = TRUE;

void MscpResponsePacket(index)
	int index;
{
	register struct mscp *mp = &Response[index];
	register struct rqdx_unit_info *ui = &RqdxUnitInfo[mp->mscp_unit];
	unsigned short s = mp->mscp_status & M_ST_MASK;
	register Process *pd;
	IoReply *reply;
	Process *oldPd;
	int  i;

/* Give us some credit */

	mp->mscp_header.uda_msglen = mscp_msglen;
	Credits += credits(mp->mscp_header);
	if (msgtype(mp->mscp_header) > 0x10)
	    { printx("Unknown RQDX message type\n"); return; }

    /* Handle error log (datagram) msgs separately: */
	if (msgtype(mp->mscp_header) == 0x10)
	  {
	    if (printErrorLogMsgs)
		MscpHandleErrorLogMsg((struct mslg *)mp);
	    return;
	  }

	switch (mp->mscp_opcode)
	  {
/* After completion of the port init sequence the interrupt routine issues
 * a set controller characteristics request to the controller.  When this
 * completes here, send packets to check the status of each of the units.
 */
	case M_OP_STCON | M_OP_END:	
	    if (s != M_ST_SUCC)
	      {
		printx("RQDX MSCP STCON failure\n");
		MscpDecodeStatusEvent(ui, mp->mscp_status);
		RqdxDriverState = S_IDLE;
		return;
	      }
	    RqdxDriverState = S_RUN;

	    for (i = 0; i < NUM_RQDX_DISKS; i++)
		MscpGetUnitStatus(i);
	    break;

/* Handle completion of a get unit status request.  If the unit is available,
 * i.e. has media mounted, then send a packet to bring the unit on-line.
 */
	case M_OP_GTUNT | M_OP_END:
	    /* If the "unit id" contained in the reply is zero, then the unit
	     * probably doesn't exist at all, so don't bother to set it up:
             */
	    if (mp->mscp_unit_model == 0 && mp->mscp_unit_class == 0)
		break;
	    MscpConvertMediaId(mp->mscp_media_id, 
			       ui->device_type, ui->media_name);
	    RqdxDoUnitSetup(mp->mscp_unit);
	    if (s == M_ST_AVLBL)
		MscpUnitOnline(mp->mscp_unit);
	    else if (s != M_ST_SUCC)
	      {
		ui->unit_online = 0;
		printx("Device %s%d: %s is currently OFFLINE\n",
		       ui->device_type, mp->mscp_unit, ui->media_name);
		MscpDecodeStatusEvent(ui, mp->mscp_status);
	      }
	    break;

/* Handle completion of unit on-line command.  If it succeeds print a message
 * and mark the unit as on-line.
 */
	case M_OP_ONLIN | M_OP_END:
	    if (s != M_ST_SUCC)
	      {
		printx("Device %s%d: %s has failed to come ONLINE\n",
		       ui->device_type, mp->mscp_unit, ui->media_name);
		MscpDecodeStatusEvent(ui, mp->mscp_status);
		return;
	      }
	    printx("Device %s%d: %s is ONLINE with %d 512-byte blocks\n",
		   ui->device_type, mp->mscp_unit, 
		   ui->media_name, mp->mscp_unit_size);
	    ui->unit_online = 1;
	    ui->unit_last_block = mp->mscp_unit_size - 1;
	    break;

/* An unit attention response means a unit MAY be able to come on-line now.
 * Closing a floppy door, for example, will produce these.  Send a get unit
 * status packet to check the status of the unit now.
 */
	case M_OP_AVATN:
	    MscpGetUnitStatus(mp->mscp_unit);
	    break;

/* Handle completion of a read or write transfer.  If it doesn't succeed
 * mark the reply as a device error.  If it succeeds and the request was
 * a read then copy the user's data from the kernel buffer to the user's
 * buffer.  Wake up the requestor now that the transfer is complete.
 */
	case M_OP_READ | M_OP_END:
	case M_OP_WRITE | M_OP_END:
	    Lockq(&ui->unit_diskq);
	    pd = ui->unit_diskq.head;
	    if (pd != ui->CurrentPD)	/* stolen off the queue! */
	      {
		Unlockq( &ui->unit_diskq);
		goto check_waiting_req;
	      }
	    ui->unit_diskq.head = ui->unit_diskq.head->link;
	    if (ui->unit_diskq.head == NULL)
		ui->unit_diskq.tail = NULL;
	    Unlockq( &ui->unit_diskq);
	    pd->queuePtr = NULL;
	    pd->link = NULL;

	    reply = (IoReply *) &(pd->msg);
	    if (s == M_ST_SUCC)
		reply->replycode = OK;
	    else if (isOpticalDisk(ui))
	      { /* Special error conditions for write-once optical disks: */
		reply->bytecount = 0;
		if (mp->mscp_status == (M_ST_DATA | (3<<5)) &&
		    mp->mscp_opcode	== (M_OP_READ | M_OP_END))
		    /* An attempt to read an uninitialized block: */
		    reply->replycode = NO_BLOCK_DATA;
	     /* else if... (other special optical disk errors?) */
		else
		    goto true_error; /* hack */
	      }
	    else
	      {
	    true_error:
		reply->replycode = DEVICE_ERROR;
		reply->bytecount = 0;
		printx("%s%d (%s): I/O error (cmd ref # = %d), %d bytes",
		       ui->device_type, mp->mscp_unit, ui->media_name,
		       mp->mscp_cmd_ref, mp->mscp_byte_cnt);
		/* Check whether one or more error log msgs were also generated.
		 * If not, then we print more details of the error here.
		 */
		if ((mp->mscp_flags&M_EF_ERLOG) == 0 || !printErrorLogMsgs)
		  {
		    printx(", end msg flags 0%o, block #%d (MSCP)",
			   mp->mscp_flags, mp->mscp_lbn);
		    MscpDecodeStatusEvent(ui, mp->mscp_status);
		  }
		printx("\n");
	      }

	    Addready(pd);

	    /* start up a new operation if one is waiting: */
check_waiting_req:
	    if (ui->unit_diskq.head != NULL)
	      {
		pd = ui->unit_diskq.head;
		oldPd = GetAddressableProcess();
		SetAddressableProcess(pd);
		MscpBuildAndGo(pd);
		SetAddressableProcess(oldPd);
	      }
	    break;

	default:
	    printx("[RQDX unexpected 0%o response packet ignored]\n",
		    mp->mscp_opcode);
	}
}

/****************************************************************************
 * Routine Name:	MscpConvertMediaId
 *
 * Description:
 *
 *	This routine converts the MSCP media id field into two Digital
 *	standard names: the generic device type name (e.g. DU), and the
 *	media name (e.g. RD52).
 *
 * Inputs:
 *
 *	media_id	Longword containing encoded media information
 *
 * Outputs:
 *
 *	device_type	Pointer to string to receive device type
 *	media_name	Pointer to string to receive media name
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpConvertMediaId(media_id, device_type, media_name)
	unsigned media_id;
	char device_type[];
	char media_name[];
{
	char c;
	int p = 0;

/* Decode device type.  Its always two characters */

	c = (char) ( (media_id >> 27) & 0x1f );
	device_type[p++] = c + 64;
	c = (char) ( (media_id >> 22) & 0x1f );
	device_type[p++] = c + 64;
	device_type[p]='\0';

/* Decode media name.  Its 1-3 characters.  Its followed by a two-digit
 * number.
 */
	p = 0;
	c = (char) ( (media_id >> 17) & 0x1f );
	media_name[p++] = c + 64;
	c = (char) ( (media_id >> 12) & 0x1f );
	if (c != 0) media_name[p++] = c + 64;
	c = (char) ( (media_id >> 7) & 0x1f );
	if (c != 0) media_name[p++] = c + 64;

	c = (char) (media_id & 0x3f);
	media_name[p++] = (c / 10) + 48;
	media_name[p++] = (c % 10) + 48;

	media_name[p] = '\0';
}

/****************************************************************************
 * Routine Name:	MscpGetUnitStatus
 *
 * Description:
 *
 *	This routine builds and sends a "get unit status" packet.
 *
 * Inputs:
 *
 *	unit	Unit whose status is to be obtained
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpGetUnitStatus(unit)
	short unit;
{
	short i;
	struct mscp *mp;
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;

	mp = MscpGetCommandPacket();
	mp->mscp_unit = unit;
	mp->mscp_opcode = M_OP_GTUNT;
	
	i = rqdx->ip;	/* Initiate polling */
};

/****************************************************************************
 * Routine Name:	MscpUnitOnline
 *
 * Description:
 *
 *	This routine builds and sends a "unit on-line" command
 *
 * Inputs:
 *
 *	unit	Unit to be brought on-line.
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpUnitOnline(unit)
	short unit;
{
	short i;
	struct mscp *mp;
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;

	mp = MscpGetCommandPacket();
	mp->mscp_unit = unit;
	mp->mscp_opcode = M_OP_ONLIN;
	
	i = rqdx->ip;	/* Initiate polling */
};

/****************************************************************************
 * Routine Name:	MscpUnitBlockIO
 *
 * Description:
 *
 *	This routine builds a transfer command and sends it.
 *
 * Inputs:
 *
 *	pd		Desciptor of requesting process
 *
 *
 * Implicit Inputs/Outputs:
 *
 *	DiskRequestorBuffer
 */

void
MscpUnitBlockIO(pd)
    register Process *pd;
  {
    extern DeviceInstance *DeviceInstanceTable[];

    IoRequest *req = (IoRequest *)&(pd->msg);
    DeviceInstance *inst = DeviceInstanceTable[req->fileid & (MAX_DEVICES-1)];
    struct rqdx_unit_info *ui = &RqdxUnitInfo[inst->unit];

    /* Put all pds on the disk queue.  The current disk transfer
     * is for the pd at the head of the queue.
     */
    pd->link = NULL;
    pd->queuePtr = &ui->unit_diskq;
    pd->state = AWAITING_INT;
    Lockq(&ui->unit_diskq);
    if (ui->unit_diskq.head == NULL)
	ui->unit_diskq.head = pd;
    else
	ui->unit_diskq.tail->link = pd;
    ui->unit_diskq.tail = pd;
    Unlockq(&ui->unit_diskq);

    /* If the pd we were called with is at the head of the Diskq,
     * start the I/O.  
     */
    if (ui->unit_diskq.head == pd)
        MscpBuildAndGo(pd);
  }


static void MscpBuildAndGo(pd)
    register Process *pd;
  {
    extern DeviceInstance *DeviceInstanceTable[];

    register IoRequest *req = (IoRequest *)&(pd->msg);
    DeviceInstance *inst = DeviceInstanceTable[req->fileid & (MAX_DEVICES-1)];
    struct rqdx_unit_info *ui = &RqdxUnitInfo[inst->unit];
    register struct mscp *mp;
    unsigned user_buffer;
    static unsigned mscpCmdRef = 0; /* unique id for MSCP command packets */

    if (req->bytecount <= IO_MSG_BUFFER)
	user_buffer = (unsigned)req->shortbuffer;
    else if (req->requestcode == READ_INSTANCE)
	user_buffer = (unsigned)pd->dstSegment.ptr;
    else
	user_buffer = (unsigned)pd->srcSegment.ptr;

/*printx("MSCP command %x on buffer %x\n", req->requestcode, user_buffer);*/

if (user_buffer == 0){
PrintMessage(pd);
    Kabort("is zero");
}
    /* Map the user's buffer into the Q-bus memory address space: */
    MapDvma((char **)&ui->unit_DMA_Address, (char *)user_buffer, req->bytecount);

    ui->CurrentPD = pd;

    mp = MscpGetCommandPacket();
    mp->mscp_cmd_ref = ++mscpCmdRef;
    mp->mscp_unit = inst->unit;
    if (req->requestcode == READ_INSTANCE)
      {
	mp->mscp_opcode = M_OP_READ;
	mp->mscp_modifier = 0;
      }
    else
      {
	mp->mscp_opcode = M_OP_WRITE;
	/* RSF: The following appears to be necessary in order for the Emulex
	 * UC04 optical disk controller to detect (and prevent) attempts to 
	 * overwrite an already-written block on optical disks.  (The optical
	 * disk's relocation jumper must also be connected for this to occur.)
	 */
	mp->mscp_modifier = M_MD_COMP; /* verify the written data */
      }
    mp->mscp_byte_cnt = req->bytecount;
    mp->mscp_lbn = req->blocknumber * BLOCK_FACTOR;
    mp->mscp_buffer = ui->unit_DMA_Address + MD_PageOffs(user_buffer);

      {
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;
	short i;

	i = rqdx->ip;	/* Initiate polling */
      }
  }


/****************************************************************************
 * Routine Name:	MscpGetCommandPacket
 *
 * Description:
 *
 *	This routine finds a free command packet and initializes it.
 *
 * Inputs:	None.
 *
 * Outputs:
 *
 *	struct mscp *	Pointer to free mscp command packet
 *
 * Implicit Inputs/Outputs:
 *
 *	LastCommand
 *	Rqdx_ca
 *	Credits
 *
 */

struct mscp * MscpGetCommandPacket()
{
	struct mscp *mp;
	int i;

/* Busy wait until command packet found.  Must have atleast two credits and
 * packet must not be owned by controller. */

	while (1) {
		i = LastCommand;
		if ( ( (Rqdx_ca.CommandDesc[i] &
			(UDA_OWN|UDA_INT)) == UDA_INT ) && (Credits >= 2) ) {
			Credits--;
			mp = &Command[i];
			mp->mscp_cmd_ref = mp->mscp_unit = 0;
			mp->mscp_xxx1 = mp->mscp_opcode = 0;
			mp->mscp_flags = mp->mscp_modifier = 0;
			mp->mscp_byte_cnt = mp->mscp_buffer = 0;
			mp->mscp_mapbase = mp->mscp_xxx2 = 0;
			mp->mscp_lbn = mp->mscp_xxx3 = 0;

			Rqdx_ca.CommandDesc[i] |= UDA_OWN | UDA_INT;
			LastCommand = (i + 1) % NCMD;
			break;
		}
	};
	return(mp);
}

/****************************************************************************
 * Routine Name:	MscpHandleErrorLogMsg
 *
 * Description:
 *
 *	Handle a MSCP error log datagram.
 *	(At present, we `handle' it merely by printing it on the console.)
 *
 * Inputs:
 *
 *	mp	Pointer to datagram
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpHandleErrorLogMsg(mp)
	struct mslg *mp;
  {
    struct rqdx_unit_info *ui = &RqdxUnitInfo[mp->mslg_unit];

    printx("Error log msg (cmd ref # = %d), flags 0%o: ",
	   mp->mslg_cmdref, mp->mslg_flags);
    switch (mp->mslg_format)
      {
	case M_FM_CNTERR:
		printx("RQDX disk controller error");
		MscpDecodeStatusEvent(ui, mp->mslg_event);
		break;
	case M_FM_BUSADDR:
		printx("RQDX error accessing host memory");
		printx("at address 0x%x\n", mp->mslg_bus_addr);
		MscpDecodeStatusEvent(ui, mp->mslg_event);
		break;
	case M_FM_DISKTRN:
		printx("Disk xfer error: level %d, retry %d, block # %d (MSCP)",
		       mp->mslg_group&0xFF, (mp->mslg_group>>8)&0xFF,
		       mp->mslg_hdr);
		MscpDecodeStatusEvent(ui, mp->mslg_event);
		break;
	case M_FM_SDI:
		printx("SDI error");
		MscpDecodeStatusEvent(ui, mp->mslg_event);
		break;
	case M_FM_SMLDSK:
		printx("Small disk error");
		MscpDecodeStatusEvent(ui, mp->mslg_event);
		break;
	default:
		printx("RQDX unknown error ( event 0%o, unit %d, format %d )\n",
			mp->mslg_event, mp->mslg_unit, mp->mslg_format);
		MscpDecodeStatusEvent(ui, mp->mslg_event);
		break;
      };
  }

/****************************************************************************
 * Routine Name:	MscpDecodeStatusEvent
 *
 * Description:
 *
 *	This routine decodes an MSCP status or event code and prints
 *	the corresponding English error message.  MSCP codes have a 5 bit
 *	major error code and an 11 bit subcode.
 *
 * Inputs:
 *
 *	code	Word status code to be decoded
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void
MscpDecodeStatusEvent(ui, code)
    struct rqdx_unit_info *ui;
    unsigned short code;
{
	unsigned short major_code = (code & 0x1f);
	unsigned short sub_code = (code >> 5);
	char *m, *s;

	switch (major_code) {
	case M_ST_SUCC:
		m = "Success";
		if (sub_code == 0) s = "Normal";
		else
		if (sub_code == 1) s = "Spin-down ignored";
		else
		if (sub_code == 2) s = "Still connected";
		else
		if (sub_code == 4) s = "Duplicate unit number";
		else
		if (sub_code == 8) s = "Already online";
		else
		if (sub_code == 16) s = "Still online";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_ICMD:
		m = "Invalid command";
		if (sub_code == 0) s = "Invalid message length";
		else
			{
			printx("offset to field: 0x%x\n", code & 0xff00);
			s = "Field in error";
			}
		break;

	case M_ST_ABRTD:
		m = "Command aborted";
		s = "";
		break;

	case M_ST_OFFLN:
		m = "Unit offline";
		if (sub_code == 0) s = "Unit unknown or online to another";
		else
		if (sub_code == 1) s = "No volume mounted or drive disabled";
		else
		if (sub_code == 2) s = "Unit is inoperative";
		else
		if (sub_code == 4) s = "Duplicate unit number";
		else
		if (sub_code == 8) s = "Unit disabled by field service";
		else
			s = "Unknown erorr sub-code";
		break;

	case M_ST_AVLBL:
		m = "Unit available";
		s = "";
		break;

	case M_ST_MFMTE:
		m = "Media format error";
		if (sub_code == 2) s = "FCT unreadable - invalid sector head";
		else
		if (sub_code == 3) s = "FCT unreadable - data sync timeout";
		else
		if (sub_code == 5)
			s = "Disk isn't formatted with 512 byte sectors";
		else
		if (sub_code == 6) s = "Disk isn't formatted or FCT corrupted";
		else
		if (sub_code == 7)
			if (isOpticalDisk(ui))
			    s = "Uninitialized optical disk - Use Emulex diagnostic program to initialize RCT";
			else
			    s = "FCT unreadable - Uncorrectable ECC error";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_WRTPR:
		m = "Write protected";
		if (sub_code == 128) s = "Unit is software write protected";
		else
		if (sub_code == 256) s = "Unit is hardware write protected";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_COMP:
		m = "Compare error";
		s = "";
		break;

	case M_ST_DATA:
		m = "Data error";
		if (sub_code == 2) s = "Header compare error";
		else
		if (sub_code == 3)
			s = "Data sync not found (data sync timeout)";
		else
		if (sub_code == 7) s = "Uncorrectable ECC error";
		else
		if (sub_code == 8) s = "One symbol ECC error";
		else
		if (sub_code == 9) s = "Two symbol ECC error";
		else
		if (sub_code == 10) s = "Three symbol ECC error";
		else
		if (sub_code == 11) s = "Four symbol ECC error";
		else
		if (sub_code == 12) s = "Five symbol ECC error";
		else
		if (sub_code == 13) s = "Six symbol ECC error";
		else
		if (sub_code == 14) s = "Seven symbol ECC error";
		else
		if (sub_code == 15) s = "Eight symbol ECC error";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_HSTBF:
		m = "Host buffer access error";
		if (sub_code == 1) s = "Odd transfer address";
		else
		if (sub_code == 2) s = "Odd byte count";
		else
		if (sub_code == 3) s = "Non-existent memory error";
		else
		if (sub_code == 4) s = "Host memory parity error";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_CNTLR:
		m = "Controller error";
		if (sub_code == 1) s = "SERDES overrun error";
		else
		if (sub_code == 2) s = "EDC error";
		else
		if (sub_code == 3) s = "Inconsistent internal data structure";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_DRIVE:
		m = "Drive error";
		if (sub_code == 1) s = "SDI command timed out (no response)";
		else
		if (sub_code == 2) s = "Controller detected transmission error";
		else
		if (sub_code == 3) s = "Positioner error (mis-seek)";
		else
		if (sub_code == 4) s = "Lost read/write ready";
		else
		if (sub_code == 5) s = "Driver clock dropout";
		else
		if (sub_code == 6) s = "Lost receiver ready between sectors";
		else
		if (sub_code == 7) s = "Drive detected error";
		else
		if (sub_code == 8) s = "Controller detected pulse";
		else
			s = "Unknown error sub-code";
		break;

	default:
		m = "Unknown error code";
		s = "Unknown error sub-code";
		break;
	}

	printx("      \"%s - %s\"\n", m, s);
}

