/*
 * V Kernel - Copyright (c) 1985 by Stanford University, All rights reserved.
 *
 * VAX specific memory access routine(s) for debugger 
 *
 *   These routines do access checking to see if access is actually allowed.
 *   They also allow access based upon either physical or virtual memory.
 *
 * $Revision: 1.7.1.4 $
 * $Locker:  $
 * $State: Exp $
 */

#include <Venviron.h>
#include <Vquerykernel.h>
#include <Vexceptionprotocol.h>

#include "memaccess.h"
#include "memory.h"
#include "externals.h"
#include "uvaxmemory.h"
#include "interrupt.h"
#include "process.h"

/* Forward */
int vProbe _TAKES(( short *, unsigned));
void readmem _TAKES(( long *, long *));
void writemem _TAKES(( long *, long *));
int KernelCanWriteByte _TAKES(( char *));
int SetKernelWriteByte _TAKES(( char *, int));

int
memaccess(address,contents,type,mode)
long *address; 	/* memory location to test */
long *contents; /* new/old contents of memory location, as appropriate */
int	type;	/* type of memory access, either physical or virtual */
int mode; 	/* type of memory access requested, either read or write */
{
	int (*probe)_TAKES((short *, unsigned)); /* probe routine to call */
	/* vProbe is for VIRTUAL ONLY */
	/* Probe is for   PHYSICAL ONLY */
	extern MachineConfigurationReply MachineConfig;

	switch(type)
	{
		case(MEMACCESS_PHYS):
			probe = Probe;
			if (((unsigned long)address & 0x7FFFFFFF)
			    > (unsigned long)MachineConfig.fastMemory) 
			  return MEMACCESS_INVAL_ADDR;
			break;
		case(MEMACCESS_VIRT):
			probe = vProbe;
			break;
		default: return MEMACCESS_INVAL_TYPE;
			break;
	}

	switch(mode)
	{
		case MEMACCESS_READ: /* physical, read access */
			if((*probe)((short*)address,PROBE_READ) == 1)  /* access OK */
			{
				readmem(address,contents);
				return MEMACCESS_OK;
			}
			else return MEMACCESS_DENIED; /* access denied */
			break;
		case MEMACCESS_WRITE: /* physical, write access */
			if((*probe)((short *)address,PROBE_PRESERVE) == 1) /* access OK */
			{
				writemem(address,contents);
				return MEMACCESS_OK;
			}
			else return MEMACCESS_DENIED; /* access denied */
			break;
		default: return MEMACCESS_INVAL_MODE; /* bad mode - return error */
			break;
	}

	return MEMACCESS_UNKNOWN; /* the "impossible" error */
}

void
readmem(address,contents)
long	*address;
long	*contents;
{
	*contents = *address;
}

void
writemem(address,contents)
long	*address;
long	*contents;
{
	*address = *contents;
}

int
vProbe(address,mode)
short *address;  /* address to check */
unsigned mode; 		/* type of access, read or preserve */
{
	switch(mode)
	{
		case PROBE_READ: /* check read access */
			return KernelCanRead((unsigned)address, sizeof(long));
		case PROBE_PRESERVE: /* check write access */
			return KernelCanWrite((unsigned)address, sizeof(long));
		default: return -1; /* invalide mode */
	}
}

/* these two routines should be moved to memory.c when they are working. */
/* these routines are modified KernelCanRead routines... The same caveats
about assumptions apply here. */

int
KernelCanWrite(address, length)
    unsigned	address, length;
  {
    unsigned	testadr;
    /* Test one byte in each page from address to address+length-1 */
    /*   If we can read one byte in the page, we can read anything */
    for (testadr = address; (unsigned)(testadr-address) < length;
			     /* ^ Does this work for all cases? */
	 testadr = (testadr+MD_PAGESIZE) & (-MD_PAGESIZE) /*gag*/)
	if (! KernelCanWriteByte((char *)testadr))
	    return 0;
    return 1;
  }

int
KernelCanWriteByte(address)
    char	*address;
  {
    register unsigned r11 = MD_PageNumber(MD_SegOffs(address));
    register PageTableEntry *r10;

    switch((unsigned)address >> MD_SEGMENTBITS)
      {
	case 3:
	    ;asm("naughty:");
	    return 0;
	case 2:
	    asm("	mfpr	$slr,	r0");
	    asm("	cmpl	r11,	r0");
	    asm("	bgequ	naughty	  ");
	    r10 = &(SysPageTable[r11]);
	    break;
	case 1:	   /* We don't use P1 at present, but let's get this right */
	    asm("	mfpr	$p1lr,	r0");
	    asm("	cmpl	r11,	r0");	/* Bounds-check */
	    asm("	blssu	naughty	  ");
	    asm("	ashl	$2, r11,r10");	/* Form virtual address of */
	    asm("	mfpr	$p1br,	r0");   /*   P1 page table entry   */
	    asm("	addl2	r0,	r10");
		/* Check that this gives an address in System Space.  Should */
		/*   be, unless our software is really screwy or the hardware*/
		/*   is less clever than we thought.  Still, check it anyway;*/
		/*   if things went wrong we might otherwise recurse.	     */
	    asm("	bgeq	naughty   ");
	    if (! KernelCanReadByte((char *)r10))     /* We'll actually want to read*/
		return 0;		      /* a PTE (4 bytes) but this   */
	    break;			      /* gives the right answer     */
	case 0:
	    asm("	mfpr	$p0lr,	r0");
	    asm("	cmpl	r11,	r0");
	    asm("	bgequ	naughty   ");
	    asm("	ashl	$2, r11,r10");
	    asm("	mfpr	$p0br,	r0");
	    asm("	bgeq	naughty   "); /* Checks P0BR->System space  */
	    asm("	addl2	r0,	r10");
	    if (! KernelCanReadByte((char *)r10))
		return 0;
	    break;
      }
	printx("r10 = %x\n",r10);
	printx("protection = %d\n",r10->fields.protection);
    if (!r10->fields.valid || r10->fields.protection < 2 ||
		 (r10->fields.protection & 3) == 3)
		return 0;
    return 1;
  }

int
SetKernelWrite(address,length,mode)
    unsigned	address, length;
	int mode;
  {
    unsigned	testadr;
    /* Test one byte in each page from address to address+length-1 */
	/* Set it so we can read the page */
    for (testadr = address; (unsigned)(testadr-address) < length;
			     /* ^ Does this work for all cases? */
	 testadr = (testadr+MD_PAGESIZE) & (-MD_PAGESIZE) /*gag*/)
	if (! SetKernelWriteByte((char *)testadr, mode))
	    return 0;
    return 1;
  }

int
SetKernelWriteByte(address, mode)
    char	*address;
    int         mode;
  {
    register unsigned r11 = MD_PageNumber(MD_SegOffs(address));
    register PageTableEntry *r10;

    switch((unsigned)address >> MD_SEGMENTBITS)
      {
	case 3:
	    ;asm("naughty2:");
	    return 0;
	case 2:
	    asm("	mfpr	$slr,	r0");
	    asm("	cmpl	r11,	r0");
	    asm("	bgequ	naughty2  ");
	    r10 = &(SysPageTable[r11]);
	    break;
	case 1:	   /* We don't use P1 at present, but let's get this right */
	    asm("	mfpr	$p1lr,	r0");
	    asm("	cmpl	r11,	r0");	/* Bounds-check */
	    asm("	blssu	naughty2  ");
	    asm("	ashl	$2, r11,r10");	/* Form virtual address of */
	    asm("	mfpr	$p1br,	r0");   /*   P1 page table entry   */
	    asm("	addl2	r0,	r10");
		/* Check that this gives an address in System Space.  Should */
		/*   be, unless our software is really screwy or the hardware*/
		/*   is less clever than we thought.  Still, check it anyway;*/
		/*   if things went wrong we might otherwise recurse.	     */
	    asm("	bgeq	naughty2  ");
	    if (! KernelCanReadByte((char *)r10))     /* We'll actually want to read*/
		return 0;		      /* a PTE (4 bytes) but this   */
	    break;			      /* gives the right answer     */
	case 0:
	    asm("	mfpr	$p0lr,	r0");
	    asm("	cmpl	r11,	r0");
	    asm("	bgequ	naughty2  ");
	    asm("	ashl	$2, r11,r10");
	    asm("	mfpr	$p0br,	r0");
	    asm("	bgeq	naughty2  "); /* Checks P0BR->System space  */
	    asm("	addl2	r0,	r10");
	    if (! KernelCanReadByte((char *)r10))
		return 0;
	    break;
      }
	  r10->fields.protection = (mode & VM_PROTECTION_MASK);
	  return 1;
  }

