using System;
using System.Collections.Generic;
using System.Text;

namespace PERQDisk
{
    /// <summary>
    /// Represents a PartitionInformationBlock (PIB) structure, which defines a disk partition.
    /// It's very similar to a DIB, so there's some duplication of code here.  Meh.        
    /// 
    /// The structure of this table is described in Section 2.3 of the "File System Data Structures" documentation in the 
    /// POS manual.
    /// </summary>
    public class PartitionInformationBlock
    {
        public PartitionInformationBlock(Sector pibSector)
        {
            if (pibSector == null)
            {
                throw new ArgumentNullException("pibSector");
            }

            BuildData(pibSector);
        }

        private void BuildData(Sector pibSector)
        {
            _freeHead = pibSector.ReadDWord(0);
            _freeTail = pibSector.ReadDWord(4);
            _numberFree = pibSector.ReadDWord(8);
            _rootDirectoryID = pibSector.ReadDWord(12);
            _badSegmentID = pibSector.ReadDWord(16);
            _partitionName = pibSector.ReadString(228, 8);
            _partitionStart = pibSector.ReadDWord(236);
            _partitionEnd = pibSector.ReadDWord(240);
            _partitionRoot = pibSector.ReadDWord(500);
            _partitionType = pibSector.ReadWord(504);
        }

        public uint FreeHead
        {
            get { return _freeHead; }
        }

        public uint FreeTail
        {
            get { return _freeTail; }
        }

        public uint NumberFree
        {
            get { return _numberFree; }
        }

        public uint RootDirectoryID
        {
            get { return _rootDirectoryID; }
        }

        public uint BadSegmentID
        {
            get { return _badSegmentID; }
        }

        public string PartitionName
        {
            get { return _partitionName; }
        }

        public uint PartitionStart
        {
            get { return _partitionStart; }
        }

        public uint PartitionEnd
        {
            get { return _partitionEnd; }
        }

        public uint PartitionRoot
        {
            get { return _partitionRoot; }
        }

        public uint PartitionType
        {
            get { return _partitionType; }
        }

        private uint _freeHead;
        private uint _freeTail;
        private uint _numberFree;
        private uint _rootDirectoryID;
        private uint _badSegmentID;
        private string _partitionName;
        private uint _partitionStart;
        private uint _partitionEnd;
        private uint _partitionRoot;
        private ushort _partitionType;
    }    

    /// <summary>
    /// Represents a Partition on a POS Volume.
    /// </summary>
    public class Partition
    {
        public Partition(PhysicalDisk disk, PartitionInformationBlock pib)
        {
            if (disk == null)
            {
                throw new ArgumentNullException("disk");
            }

            if( pib == null )
            {
                throw new ArgumentNullException("pib");
            }

            _pib = pib;
            _disk = disk;

            BuildPartition();
        }

        public string Name
        {
            get { return _pib.PartitionName; }
        }

        public File Root
        {
            get { return _root; }
        }

        public void PrintPIBInfo()
        {
            Console.WriteLine(" Name: '{0}'\n  Start: {1}\n  End: {2}\n  Free: {3}\n  Root: {4}", 
                _pib.PartitionName, 
                _disk.LogicalBlockToPhysicalBlock(_disk.LDAToLogicalBlock(_pib.PartitionStart)),
                _disk.LogicalBlockToPhysicalBlock(_disk.LDAToLogicalBlock(_pib.PartitionEnd)),                
                _pib.NumberFree,
                _disk.LogicalBlockToPhysicalBlock(_disk.LDAToLogicalBlock(_pib.RootDirectoryID)));
        }

        private void BuildPartition()
        {
            //Find the Root directory and let it do its magic.
            Block rootFibBlock = _disk.LDAToLogicalBlock(_pib.RootDirectoryID);
            Sector rootFibSector = _disk.GetSector(rootFibBlock);
            _root = new File(_disk, new DirectoryEntry(Name), new FileInformationBlock( rootFibSector ) );                
        }

        private PartitionInformationBlock _pib;
        private File _root;
        private PhysicalDisk _disk;
    }
}
