using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace PERQDisk
{
    /// <summary>
    /// Represents a PERQ hard disk Block (sector) Header.
    /// </summary>
    public struct BlockHeader
    {
        uint SerialNum;
        ushort LogBlock;
        ushort Filler;
        uint NextAdr;
        uint PrevAdr;            
    }

    /// <summary>
    /// Represents a physical disk sector, which is where the actual
    /// data lives.  Woo!
    /// </summary>
    public class Sector
    {
        public Sector(int sectorID)
        {
            if (sectorID < 0)
            {
                throw new ArgumentOutOfRangeException("sectorID");
            }

            _sectorID = sectorID;

            _rawHeader = new byte[16];
            _sectorData = new byte[512];    //512 bytes per sector            
        }

        public byte[] Data
        {
            get { return _sectorData; }
        }

        /// <summary>
        /// Loads this sector's data from disk.
        /// This assumes that the incoming FileStream is currently
        /// pointing to the data for this sector.
        /// </summary>
        /// <param name="fs"></param>
        public void Load(FileStream fs)
        {
            
            // Read the header data
            if (fs.Read(_rawHeader, 0, 16) != 16)
            {
                throw new InvalidOperationException("Out of data when reading sector header.");
            }

            // Read the sector data
            if (fs.Read(_sectorData, 0, 512) != 512)
            {
                throw new InvalidOperationException("Out of data when reading sector data.");
            }

            //TODO: parse header into struct
        }

        public byte ReadByte(int offset)
        {
            if (offset < 0 || offset >= 512)
            {
                throw new ArgumentOutOfRangeException("offset");
            }

            return Helper.ReadByte(_sectorData, offset);
        }

        public ushort ReadWord(int offset)
        {
            if (offset < 0 || offset >= 511) //need two bytes
            {
                throw new ArgumentOutOfRangeException("offset");
            }

            return Helper.ReadWord(_sectorData, offset);
        }

        public uint ReadDWord(int offset)
        {
            if (offset < 0 || offset >= 509) //need 4 bytes
            {
                throw new ArgumentOutOfRangeException("offset");
            }

            return Helper.ReadDWord(_sectorData, offset);
        }

        public string ReadString(int offset, int count)
        {
            if (count < 0 || offset < 0 || offset + count >= 512)
            {
                throw new ArgumentOutOfRangeException("offset, count");
            }

            return Helper.ReadString(_sectorData, offset, count);
        }

        private BlockHeader _sectorHeader;
        private byte[] _rawHeader;
        private byte[] _sectorData;
        private int _sectorID;
    }
}
