using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace PERQDisk
{
    /// <summary>
    /// Represents a cyl,track,sector triad; additionally contains a bit indicating whether this
    /// block is "logical" (excluding boot sector) or "physical" (representing actual disk layout).
    /// This could (should) probably be factored into two classes, inheriting from a common abstract base class.
    /// But I'm lazy.
    /// </summary>
    public struct Block
    {
        public uint Cylinder;
        public uint Track;
        public uint Sector;
        public bool IsLogical;

        public override string ToString()
        {
            if (IsLogical)
            {
                return String.Format("Logical Cylinder {0}, Track {1}, Sector {2}", Cylinder, Track, Sector);
            }
            else
            {
                return String.Format("Physical Cylinder {0}, Track {1}, Sector {2}", Cylinder, Track, Sector);
            }
        }
    }

    /// <summary>
    /// Represents a physical disk drive, which contains one or more Cylinders.
    /// This is the primary interface for interaction with the disk image, and allows
    /// reading sectors of data.
    /// 
    /// NOTE: this abstraction is really unnecessary and could be done a lot better.
    /// </summary>
    public class PhysicalDisk
    {
        /// <summary>
        /// Constructs a new PhysicalDisk, given the number of Cylinders on the drive,
        /// Tracks per Cylinder and Sectors per Track.
        /// </summary>
        /// <param name="cylinders"></param>
        /// <param name="tracks"></param>
        /// <param name="sectors"></param>
        public PhysicalDisk(int cylinders, int tracks, int sectors)
        {
            if (cylinders < 0 || tracks < 0 || sectors < 0)
            {
                throw new ArgumentOutOfRangeException("Invalid chs specification, must be greater than zero.");
            }

            _cylinders = cylinders;
            _tracks = tracks;
            _sectors = sectors;

            _cylinderList = new Cylinder[_cylinders];

            CreateCylinders();

            _loaded = false;
        }


        public bool Loaded
        {
            get { return _loaded; }
        }

        /// <summary>
        /// Loads the disk image from the supplied filestream.
        /// </summary>
        /// <param name="fs"></param>
        public void Load(FileStream fs)
        {
            for (int cyl = 0; cyl < _cylinders; cyl++)
            {
                for (int track = 0; track < _tracks; track++)
                {
                    for (int sector = 0; sector < _sectors; sector++)
                    {
                        _cylinderList[cyl].LoadSector(fs, track, sector);
                    }
                }
            }

            _loaded = true;
        }

        public Sector GetSector(int cylinder, int track, int sector)
        {
            if (cylinder < 0 || cylinder >= _tracks)
            {
                throw new ArgumentOutOfRangeException("cylinder");
            }

            return _cylinderList[cylinder].GetSector(track, sector);
        }

        public Sector GetSector(Block block)
        {
            // Get physical block if this is a logical one.
            if (block.IsLogical)
            {
                block = LogicalBlockToPhysicalBlock(block);
            }
            
            return _cylinderList[block.Cylinder].GetSector((int)block.Track, (int)block.Sector);            
        }

        /// <summary>
        /// Converts a logical disk address into a Logical block expressed in Cyl,Track,Sector format.
        /// 
        /// This routine converts a LDA (logical disk address) to an LBN; this can then be directly converted
        /// to a Physical block address (LBN's cyl0,track0,sector0 is physical cyl0,track1,sector0)
        /// Summarized from the POS documentation, Page 1-7:       
        /// 
        /// Low word:
        ///     - Bits 0 through 7 are reserved for future use
        ///     - Bits 8 through 15 form the low 8 bits of the LBN
        /// 
        /// High word:
        ///     - Bits 0 through 10 form the high eleven bits of the LBN
        ///     - Bits 11 through 13 specify the volume (000 = hard disk, 100 = floppy)
        ///     - Bits 14 and 15 indicate whether the address is on disk or virtualized.
        /// 
        /// Currently we assume a hard disk image and so only the LBN-specific data is taken into account here.
        ///         
        /// 
        /// </summary>
        /// <param name="logicalAddress"></param>
        /// <returns></returns>
        public Block LDAToLogicalBlock(uint lda)
        {
            uint lba = ((lda & 0x0000ff00) >> 8) | ((lda & 0x07ff0000) >> 8);
     
            //Now we have the actual logical block number. 

            Block pb = new Block();
            pb.Cylinder = (uint)(lba / (Tracks * Sectors));
            pb.Track = (uint)((lba - (pb.Cylinder * Tracks * Sectors)) / Sectors);
            pb.Sector = (uint)(lba % Sectors);
            pb.IsLogical = true;

            return pb;

        }

        public Block PhysicalBlockToLogicalBlock(Block physBlock)
        {
            if (physBlock.IsLogical)
            {
                throw new InvalidOperationException("This block is already logical.");
            }

            // Move this back one track to compensate for the physical->logical conversion
            physBlock.Track = physBlock.Track - 1;

            if (physBlock.Track < 0)
            {
                physBlock.Track = (uint)(Tracks - 1);
                physBlock.Cylinder = physBlock.Cylinder - 1;

                if (physBlock.Cylinder < 0)
                {
                    throw new InvalidOperationException("Went off the beginning of the disk converting to logical.");
                }
            }

            physBlock.IsLogical = true;

            return physBlock;                        
        }

        public Block LogicalBlockToPhysicalBlock(Block physBlock)
        {
            if (!physBlock.IsLogical)
            {
                throw new InvalidOperationException("This block is already physical.");
            }

            // Move this forward one track to compensate for the logical->physical conversion
            physBlock.Track = physBlock.Track + 1;

            if (physBlock.Track >= Tracks)
            {
                physBlock.Track = 0;
                physBlock.Cylinder = physBlock.Cylinder + 1;

                if (physBlock.Cylinder >= Cylinders)
                {
                    throw new InvalidOperationException("Went off the end of the disk converting to physical.");
                }
            }

            physBlock.IsLogical = false;

            return physBlock;
        }
        
        public int Cylinders
        {
            get { return _cylinders; }
        }

        public int Tracks
        {
            get { return _tracks; }
        }

        public int Sectors
        {
            get { return _sectors; }
        }

        private void CreateCylinders()
        {
            for (int i = 0; i < _cylinders; i++)
            {
                _cylinderList[i] = new Cylinder(i, _tracks, _sectors);
            }
        }

        private Cylinder[] _cylinderList;

        private int _cylinders;
        private int _tracks;
        private int _sectors;

        private bool _loaded;
    }

}
