Module RLE;

Exports

Imports Util from Util;

Const   RLEVersion = 1;
        RLEMaxRows = 256;
        RLEMaxCols = 255;
        RLEDirectory = 'Sys:Part1>Images>';
        RLEExtension = '.RLE';
        RLEPattern = '*.RLE';

Type    Run = Packed Record
            Start: 0..255;
            Length: 0..255;
            End;

        RLEDataType = Array [1 .. MaxInt] of Run;
                                        { no need to disable array checking }
        pRLEData = ^RLEDataType;

        RLEHeader = Record
                RLEVers: Integer;
                Rows: Integer;
                Cols: Integer;
                NumRuns: Integer;
                RLEName: String [25];
                RUpLeft: UCoord;
                RTickNum: Integer;
                RTickRow: Integer;
                RTickColor: Boolean;
                RLEData: pRLEData;
                End;

        RLEImage = ^RLEHeader;

Procedure RLECreate (Var rim: RLEImage; MinBlocks: Integer);
Procedure RLEDispose (Var rim: RLEImage);

Function  RLERead (Name: String): RLEImage;
Procedure RLEWrite (rim: RLEImage);

Function  RLERect (rim: RLEImage): URect;
Procedure RLEDisplay (rim: RLEImage);

Procedure RLEStart (rim: RLEImage);
Procedure RLETick (rim: RLEImage);

Exception RLEExcept (reason: String);

Private

Imports Sail_String from Sail_String;
Imports PMatch from PMatch;
Imports Screen from Screen;
Imports FileSystem from FileSystem;
Imports MultiRead from MultiRead;

Const   RLEBits = 4096;

Procedure RLECreate (Var rim: RLEImage; MinBlocks: Integer);

        Var     pnt: UPointer;
                blocks: Integer;

        Begin
        UCheckVersion (UVersion, 'Recompile the RLE module.');
        blocks := IMax (MinBlocks, 1);
        pnt . segment := UCreateSegment (Blocks, 1, 256);
        pnt . offset := 0;
        RasterOp (RXor, 512, Blocks * 8, 0, 0, 32, pnt . P, 0, 0, 32, pnt . P);
        rim := ReCast (pnt . P, RLEImage);
        with rim^ do
            begin
            RLEVers := RLEVersion;
            RLEName := '';
            Rows := 1;
            Cols := 1;
            NumRuns := 0;
            pnt . offset := 256;
            RLEData := ReCast (pnt . P, pRLEData);
            end;
        End;

Function RLERead (Name: String): RLEImage;

        Var     rim: RLEImage;
                fid: FileID;
                Blks, Bits: Integer;
                FileName, stupid: String;
                SavePtr: pRLEData;

        Begin
        stupid := RLEPattern;
        if PattMatch (Name, stupid, True) then
            Adjust (Name, Length (Name) - Length (RLEExtension));
        FileName := RLEDirectory;
        AppendString (FileName, Name);
        AppendString (FileName, RLEExtension);
        fid := FSInternalLookup (FileName, Blks, Bits);
        if fid = 0 then
            Raise RLEExcept ('RLERead: file does not exist');
        if Bits <> RLEBits then
            Raise RLEExcept ('RLERead: bad number of bits in file');
        if Blks < 2 then
            Raise RLEExcept ('RLERead: not enough blocks in file');
        
        RLECreate (rim, Blks);
        SavePtr := rim^ . RLEData;
        MultiRead (fid, ReCast (rim, PDirBlk), 0, Blks);
        with rim^ do
            begin
            if RLEVers <> RLEVersion then
                Raise RLEExcept ('RLERead: incompatible version');
            if UBlocks (NumRuns) + 1 <> Blks then
                Raise RLEExcept ('RLERead: wrong size of file');
            if RLEName <> Name then
                Raise RLEExcept ('RLERead: names are different');
            RLEData := SavePtr;
            end;
        RLERead := rim;
        End;

Procedure RLEWrite (rim: RLEImage);

        Var     fid: FileID;
                pnt: UPointer;
                i, Blocks, Seg: Integer;
                FileName, stupid: String;

        Begin
        stupid := RLEPattern;
        with rim^ do
            begin
            if PattMatch (RLEName, stupid, True) then
                Adjust (RLEName, Length (RLEName) - Length (RLEExtension));
            FileName := RLEDirectory;
            AppendString (FileName, RLEName);
            end;
        AppendString (FileName, RLEExtension);
        fid := FSEnter (FileName);
        if fid = 0 then
            Raise RLEExcept ('RLEWrite: unable to create file');
        pnt := ReCast (rim, UPointer);
        Seg := pnt . segment;
        Blocks := UBlocks (rim^ . NumRuns) + 1;
        for i := 0 to Blocks - 1 do
            FSBlkWrite (fid, i, MakePtr (Seg, i * 256, pDirBlk));
        FSClose (fid, Blocks, RLEBits);
        End;

Procedure RLEDispose (Var rim: RLEImage);

        Begin
        UDispPSegment (rim);
        End;

Function  RLERect (rim: RLEImage): URect;

        Var     rec: URect;

        Begin
        with rec, rim^ do
            begin
            UUpLeft := RUpLeft;
            UHeight := Rows;
            UWidth := Cols;
            end;
        RLERect := rec;
        End;

Procedure RLEDisplay (rim: RLEImage);

        Var     row, col, i: Integer;
                IsObject: Boolean;
                Color: Array [Boolean] of LineStyle;

        Begin
        Color [True] := DrawLine;
        Color [False] := EraseLine;
        with rim^ do
            begin
            row := RUpLeft . UY - 1;
            for i := 1 to NumRuns do
                with RLEData^ [i] do
                    begin
                    if start = 0
                        then begin
                            IsObject := False;
                            row := row + 1;
                            end
                        else IsObject := not IsObject;
                    col := RUpLeft . UX + start;
                    Line (Color [IsObject], col, row,
                                        col + length, row, SScreenP);
                    end;
            end;
        End;

Var     Color: Array [Boolean] of LineStyle;

Procedure RLEStart (rim: RLEImage);

        Begin
        Color [True] := DrawLine;
        Color [False] := EraseLine;
        with rim^ do
            begin
            RTickColor := False;
            RTickNum := 1;
            RTickRow := RUpLeft . UY - 1;
            end;
        End;

Procedure RLETick (rim: RLEImage);

        Var     col: Integer;

        Begin
        with rim^ do
            begin
            if RTickNum <= NumRuns then
                begin
                with RLEData^ [RTickNum] do
                    begin
                    if start = 0
                        then begin
                            RTickColor := False;
                            RTickRow := RTickRow + 1;
                            end
                        else RTickColor := not RTickColor;
                    col := RUpLeft . UX + start;
                    Line (Color [RTickColor], col, RTickRow,
                                        col + length, RTickRow, SScreenP);
                    end;
                RTickNum := RTickNum + 1;
                end;
            end;
        End.
