Module EGroup;

{ ****************** Maintain a set of groups to listen to ****************** }
{
    The Perq Ethernet interface lets you receive from up to 5 groups, or
    from all groups.  This module lets you add and remove groups from the
    set of groups you are listening to, and allows you to listen to any
    number of groups by getting the hardware to listen to all groups if
    you have more than 5, and then rejecting ones you're not listening to.

    InitGroup initializes to receive from no groups.  It sets up the
    EtherAdRec.  You must give it your EtherAddress for the EtherAdRec.
    ***** You still must call E10Reset to cause it to take effect. *****

    GrpToAddr converts a group number to an EtherAddress.

    AddrToGrp converts an EtherAddress to a group number, or returns -1
    if the EtherAddress is not a group address.

    AddGroup adds a group to the set of groups we are listening to, setting
    up the EtherAdRec.
    ***** You still must call E10Reset to cause it to take effect. *****

    RmvGroup removes a group from the set of groups we are listening to,
    setting up the EtherAdRec.
    ***** You still must call E10Reset to cause it to take effect. *****

    GroupToMe tells you whether the given address is a group address that
    we are listening to.  If so, Group is set to the group number that it
    is from.
}

Exports

Imports Ether10IO from Ether10IO;

Type    GroupType = 0 .. 255;

Procedure InitGroup (MyAddr: EtherAddress; AdRec: pEtherAdRec);

Function GrpToAddr (Group: GroupType): EtherAddress;

Function AddrToGrp (Addr: EtherAddress): Integer;

Procedure AddGroup (Group: GroupType; AdRec: pEtherAdRec);

Procedure RmvGroup (Group: GroupType; AdRec: pEtherAdRec);

Function GroupToMe (Addr: EtherAddress; Var Group: GroupType): Boolean;

Private

Var     NumGroups: Integer;
        RcvGroups: Set of GroupType;

Procedure InitGroup (MyAddr: EtherAddress; AdRec: pEtherAdRec);

        Begin
        with AdRec^ do
            begin
            LowAddress := MyAddr . Low;
            MCB := MltCstNone;
            end;
        NumGroups := 0;               { don't receive from any groups }
        RcvGroups := [];
        End;

Function GrpToAddr (Group: GroupType): EtherAddress;

        Var     Addr: EtherAddress;

        Begin
        with Addr do
            begin
            High := 1;
            Mid := 0;
            Low := Shift (Group, 8);
            end;
        GrpToAddr := Addr;
        End;

Function AddrToGrp (Addr: EtherAddress): Integer;

        Begin
        if (Addr . High = 1) and (Addr . Mid = 0)
                and (LAnd (Addr . Low, 255) = 0)
            then AddrToGrp := Shift (Addr . Low, -8)
            else AddrToGrp := -1;
        End;

Procedure AddGroup (Group: GroupType; AdRec: pEtherAdRec);

        Begin
        if Group in RcvGroups then Exit (AddGroup);
        RcvGroups := RcvGroups + [Group];
        NumGroups := NumGroups + 1;
        Case NumGroups of
            1:  begin
                with AdRec^ do
                    begin
                    MCB := MltCstGrp;
                    MultCst1 := Group;
                    MultCst2 := Group;
                    MultCst3 := Group;
                    MultCst4 := Group;
                    MultCst5 := Group;
                    end;
                end;
            2:  begin
                AdRec^ . MultCst2 := Group;
                end;
            3:  begin
                AdRec^ . MultCst3 := Group;
                end;
            4:  begin
                AdRec^ . MultCst4 := Group;
                end;
            5:  begin
                AdRec^ . MultCst5 := Group;
                end;
            6:  begin
                AdRec^ . MCB := MltCstAll;
                end;
            Otherwise:  { no more to do }
            end;
        End;

Procedure RmvGroup (Group: GroupType; AdRec: pEtherAdRec);

        Label   3, 4, 5;

        Var     g: 0 .. 255;

        Begin
        if not (Group in RcvGroups) then Exit (RmvGroup);
        RcvGroups := RcvGroups - [Group];
        NumGroups := NumGroups - 1;
        if NumGroups <= 0 then
            begin
            AdRec^ . MCB := MltCstNone;
            end
        else if NumGroups <= 5 then
            with AdRec^ do
                begin
                MCB := MltCstGrp;
                g := 0;
                while not (g in RcvGroup) do g := g + 1;
                MultCst1 := g;
                if NumGroups > 1
                  then begin
                    while not (g in RcvGroup) do g := g + 1;
                    MultCst2 := g;
                    if NumGroups > 2
                      then begin
                        while not (g in RcvGroup) do g := g + 1;
                        MultCst3 := g;
                        if NumGroups > 3
                          then begin
                            while not (g in RcvGroup) do g := g + 1;
                            MultCst4 := g;
                            if NumGroups > 4
                              then begin
                                while not (g in RcvGroup) do g := g + 1;
                                MultCst5 := g;
                                end
                              else goto 5;
                            end
                          else goto 4;
                        end
                      else goto 3;
                    end
                  else begin
                    MultCst2 := MultCst1;
3:                  MultCst3 := MultCst1;
4:                  MultCst4 := MultCst1;
5:                  MultCst5 := MultCst1;
                    end;
                end;
        End;

Function GroupToMe (Addr: EtherAddress; Var Group: GroupType): Boolean;

        Var     g: Integer;

        Begin
        g := AddrToGrp (Addr);
        if g >= 0
            then begin
                Group := g;
                GroupToMe := g in RcvGroups;
                end
            else GroupToMe := False;
        End.
