program drc ;
{ design rule checker for sif files }

CONST
    MAXBOXES = 200;

    { constants for the layers.
      ZZZZ means an illegal layer. }
    ZZZZ = 0;   { illegal }
    ND = 1;     { diffusion }
    NP = 2;     { polysilicon }
    NC = 4;     { contact }
    NM = 8;     { metal }
    NI = 16;    { implant }
    NB = 32;    { buried contact -- note: this layer not allowed }
    NG = 64;    { glass }
TYPE
    boxtype = record
        tlx, tly, brx, bry : integer;
        mark : integer;
    end;
    boxaratype = record
        vec : array [0..MAXBOXES] of boxtype; { 0 index is for speed, not used}
        cursize : integer;
        maxsize : integer; { not used }
    end;
    siftype = record
        poly : boxaratype;
        diff : boxaratype;
        Metal : boxaratype;
        implant : boxaratype;
        contact : boxaratype;
        glass : boxaratype;
    end;
    
VAR
    DiffAndPoly : boxaratype;
    DiffNotPoly : boxaratype;
    Transistor : boxaratype;
    Depletion : boxaratype;
    Enhancement : boxaratype;

PROCEDURE ReadSIF (filename : string; VAR sif : siftype);
VAR
    mode : integer;
    inchan : text;
    contflag : boolean;
    ch : char;
    
  PROCEDURE skipblanks ;
  { skips blanks from inchan, stops on eof or non-blank }
  VAR
    keepgoing : boolean;
    ch : char;
  BEGIN
    keepgoing := TRUE;
    while (keepgoing) do BEGIN
        if (eoln (inchan)) then BEGIN
            if (not eof (inchan)) then readln (inchan)
            else keepgoing := FALSE; {reached eof}
        END
        else BEGIN
            ch := inchan^;
            if ((('0' <= ch) and (ch <= '9')) or
                (('A' <= ch) and (ch <= 'Z')) or
                (ch = '-') or
                (ch = '(') or
                (ch = ')') or
                (ch = ';')) then
                    keepgoing := FALSE
            else BEGIN
                read (inchan, ch);
{                writeln ('Skipping ''', ch, '''');}
            END;
        END;
    END;
  END;
  
  PROCEDURE SkipSep ;
  { skip blanks and upper case letters }
  VAR
    keepgoing : boolean;
    ch : char;
  BEGIN
    keepgoing := TRUE;
    while (keepgoing) do BEGIN
        SkipBlanks ;
        if (eof (inchan)) then keepgoing := FALSE
        else BEGIN
            ch := inchan^;
            if (('A' <= ch) and (ch <= 'Z')) then read (inchan, ch)
            else keepgoing := FALSE;
        END;
    END; {while}
  END;
  
  FUNCTION ReadInteger : integer;
  VAR
    ans : integer;
  BEGIN
    SkipSep ;
    read (inchan, ans);
    ReadInteger := ans;
  END;
  
  PROCEDURE FlushToSemi ;
  VAR
    ch : char;
  BEGIN
    while (inchan^ <> ';') do read (inchan, ch);
    read (inchan, ch); {remove semi}
  END;

VAR
    curlength, curwidth : integer;
    thisbox : boxtype;

BEGIN
    reset (inchan, filename);
    mode := ZZZZ;
    
    { skip blanks, get semi separated commands, end with 'E' and blanks }
    contflag := TRUE;
    while (contflag) do BEGIN
        skipblanks ;
        read (inchan, ch);
        case ch of
          'B' :
            BEGIN
              writeln ('Block');
              curlength := ReadInteger;
              curwidth := ReadInteger;
              thisbox.tlx := ReadInteger;
              thisbox.bry := ReadInteger;
              thisbox.tly := thisbox.bry + curwidth;
              thisbox.brx := thisbox.tlx + curlength;
              with sif do case mode of
                ND:
                  with diff do BEGIN
                      cursize := cursize + 1;
                      vec[cursize] := thisbox;
                  END;
                NP:
                  with poly do BEGIN
                      cursize := cursize + 1;
                      vec[cursize] := thisbox;
                  END;
                NC:
                  with contact do BEGIN
                      cursize := cursize + 1;
                      vec[cursize] := thisbox;
                  END;
                NM:
                  with Metal do BEGIN
                      cursize := cursize + 1;
                      vec[cursize] := thisbox;
                  END;
                NI:
                  with implant do BEGIN
                      cursize := cursize + 1;
                      vec[cursize] := thisbox;
                  END;
                NB: writeln ('Warning, Buried Contacts not implemented');
                NG:
                  with glass do BEGIN
                      cursize := cursize + 1;
                      vec[cursize] := thisbox;
                  END;
                ZZZZ : writeln ('Error, Box, No Layer Defined');
                otherwise : writeln ('Error, Box, Bad Layer');
              end; {case}
            END;
          'L' :
            BEGIN
              writeln ('Layer');
              SkipBlanks ;
              read (inchan, ch);
              if (ch <> 'N') then writeln ('Error, Layer, bad layer name')
              else BEGIN
                  read (inchan, ch);
                  case ch of
                    'D': mode := ND;
                    'P': mode := NP;
                    'C': mode := NC;
                    'M': mode := NM;
                    'I': mode := NI;
                    'B':
                      BEGIN
writeln ('Warning, Layer, Buried Contact not implemented');
writeln ('Layer not changed');
                      END;
                    'G': mode := NG;
                    otherwise : writeln ('Error, Layer, bad layer name');
                  end; {case}
              END;
            END;
          'E' :
            BEGIN
              writeln ('End');
              contflag := FALSE;
            END;
          otherwise:
            BEGIN
              writeln ('Foo, flush to semicolon');
            END;
        end; {case}
        if (contflag) then FlushToSemi ;
    END; {while}
    
    close (inchan);
    
END;

PROCEDURE InitBoxAra (VAR boxes : boxaratype);
BEGIN
    boxes.cursize := 0;
    boxes.maxsize := MAXBOXES;
END;

PROCEDURE zapsif (VAR sif : siftype);
BEGIN
    with sif do BEGIN
        InitBoxAra (poly);
        InitBoxAra (diff);
        InitBoxAra (Metal);
        InitBoxAra (implant);
        InitBoxAra (contact);
        InitBoxAra (glass);
    END;
END;

PROCEDURE PrintBox (box : boxtype);
BEGIN
    with box do writeln ('Length ', brx-tlx : 3,
           ' Width ', tly-bry : 3,
           ' BRX ', tlx : 5,
           ' BRY ', bry : 5);
END;

PROCEDURE PrintAra (var boxes : boxaratype);
VAR
    i : integer;
BEGIN
    for i := 1 to boxes.cursize do PrintBox (boxes.vec[i]);
END;

PROCEDURE DRCInit ;
BEGIN
END;

PROCEDURE DRCError (str : string);
BEGIN
    writeln ;
    writeln ('**ERROR: ', str);
END;

PROCEDURE DRCWarning (str : string);
BEGIN
    writeln ;
    writeln ('**WARNING: ', str);
END;

PROCEDURE DRCUnchecked (str : string);
BEGIN
    writeln ;
    writeln ('**UNCHECKED: ', str);
END;

PROCEDURE DRCPrintBox (box : boxtype);
BEGIN
    PrintBox (box);
END;

FUNCTION TestWidths (VAR boxes : boxaratype; minwidth : integer) : boolean;
{ returns TRUE if all ok, else FALSE }
VAR
    i : integer;
    ans : boolean;
    thisone : boolean;
BEGIN
    ans := TRUE;
    with boxes do BEGIN
        for i := 1 to cursize do with vec[i] do BEGIN
            thisone := TRUE;
            if (brx-tlx < minwidth) then thisone := FALSE;
            if (tly-bry < minwidth) then thisone := FALSE;
            if (thisone = FALSE) then BEGIN
                DRCUnchecked ('Bad Width');
                DRCPrintBox (vec[i]);
            END;
            if (thisone = FALSE) then ans := FALSE;
        END;
    END;
    TestWidths := ans;
END;

FUNCTION Inside (Box : boxtype; x, y : integer) : boolean;
BEGIN
    with box do BEGIN
        if ((tlx <= x) and (x <= brx) and (bry <= y) and (y <= tly)) then
            Inside := TRUE
        else Inside := FALSE;
    END;
END;

FUNCTION Intersect (VAR box1, box2 : boxtype; VAR didtouch : boolean):boxtype;
LABEL
    1111, 2222;
VAR
    ans : boxtype;
    x1, x2, x3, x4, y1, y2, y3, y4 : integer;
BEGIN
    didtouch := TRUE;

    y1 := box1.tly;
    y2 := box1.bry;
    y3 := box2.tly;
    y4 := box2.bry;
    if (y3 > y1) then BEGIN
        if (y4 > y1) then goto 1111; {no intersection }
        ans.tly := y1;
        if (y4 >= y2) then ans.bry := y4
        else ans.bry := y2;
    END
    else if (y3 >= y2) then BEGIN
        ans.tly := y3;
        if (y4 >= y2) then ans.bry := y4
        else ans.bry := y2;
    END
    else goto 1111;
    x1 := box1.brx;
    x2 := box1.tlx;
    x3 := box2.brx;
    x4 := box2.tlx;
    if (x3 > x1) then BEGIN
        ans.brx := x1;
        if (x4 > x1) then goto 1111;
        if (x4 >= x2) then ans.tlx := x4
        else ans.tlx := x2;
    END
    else if (x3 >= x2) then BEGIN
        ans.brx := x3;
        if (x4 >= x2) then ans.tlx := x4
        else ans.tlx := x2;
    END
    else goto 1111;
    goto 2222;
  1111: ;
    { boxes did not intersect at all, didtouch <== FALSE }
    didtouch := FALSE;
  2222: ;
    Intersect := ans;
END;

FUNCTION BoxIntersect (VAR box1, box2 : boxtype) : boxtype;
LABEL
    1111, 2222;
VAR
    ans : boxtype;
    x1, x2, x3, x4, y1, y2, y3, y4 : integer;
BEGIN
    y1 := box1.tly;
    y2 := box1.bry;
    y3 := box2.tly;
    y4 := box2.bry;
    if (y3 > y1) then BEGIN
        if (y4 > y1) then goto 1111; {no intersection }
        ans.tly := y1;
        if (y4 >= y2) then ans.bry := y4
        else ans.bry := y2;
    END
    else if (y3 >= y2) then BEGIN
        ans.tly := y3;
        if (y4 >= y2) then ans.bry := y4
        else ans.bry := y2;
    END
    else goto 1111;
    x1 := box1.brx;
    x2 := box1.tlx;
    x3 := box2.brx;
    x4 := box2.tlx;
    if (x3 > x1) then BEGIN
        ans.brx := x1;
        if (x4 > x1) then goto 1111;
        if (x4 >= x2) then ans.tlx := x4
        else ans.tlx := x2;
    END
    else if (x3 >= x2) then BEGIN
        ans.brx := x3;
        if (x4 >= x2) then ans.tlx := x4
        else ans.tlx := x2;
    END
    else goto 1111;
    goto 2222;
  1111: ;
    with ans do BEGIN
        tlx := -1000;
        tly := -1000;
        brx := -1000;
        bry := -1000;
    END;
  2222: ;
    BoxIntersect := ans;
END;

FUNCTION BoxWidth (var Box : boxtype) : integer;
BEGIN
    with Box do BoxWidth := tly - bry;
END;

FUNCTION BoxLength (var Box : boxtype) : integer;
BEGIN
    with Box do BoxLength := brx - tlx;
END;

FUNCTION BoxRight (var Box : boxtype) : integer;
BEGIN
    BoxRight := box.brx;
END;

FUNCTION BoxLeft (var Box : boxtype) : integer;
BEGIN
    BoxLeft := box.tlx;
END;

FUNCTION BoxTop (var Box : boxtype) : integer;
BEGIN
    BoxTop := box.tly;
END;

FUNCTION BoxBottom (var Box : boxtype) : integer;
BEGIN
    BoxBottom := box.bry;
END;

FUNCTION BoxArea (var box : boxtype) : integer;
BEGIN
    BoxArea := BoxLength (box) * BoxWidth (box);
END;

FUNCTION OverlapPred (var box1, box2 : boxtype) : boolean;
{ returns true if box1 and box2 intersect with positive area }
VAR
    interbox : boxtype;
    touched : boolean;
BEGIN
    OverlapPred := FALSE;
    interbox := Intersect (box1, box2, touched);
    if (touched) then if (BoxArea (interbox) > 0) then OverlapPred := TRUE;
END;

Function BoxDist (box1, box2 : boxtype; space : integer) : boolean;
VAR
    xdist, ydist : integer;
BEGIN
    if      (box1.tlx > box2.brx) then xdist := box1.tlx - box2.brx
    else if (box2.tlx > box1.brx) then xdist := box2.tlx - box1.brx
    else xdist := 0;
    if      (box1.bry > box2.tly) then ydist := box1.bry - box2.tly
    else if (box2.bry > box1.tly) then ydist := box2.bry - box1.tly
    else ydist := 0;
    
    if (xdist*xdist + ydist*ydist >= space*space) then BoxDist := TRUE
    else BoxDist := FALSE;
END;

{$IFC FALSE THEN}
FUNCTION BDkind (box1, box2 : boxtype; width, space : integer) : integer;
{ returns 0 if connected, 1 if disjoint, 2 if bad }
VAR
    InterBox : boxtype;
BEGIN
    if (BoxDist (box1, box2, space)) then BDKind := 1
    else BEGIN
        InterBox := BoxIntersect (box1, box2);
        with Interbox do BEGIN
            if (sqr (BoxWidth (interbox))+sqr (BoxLength (interbox)) >=
                width*width) then BDKind := 0
            else BDKind := 2;
        END;
    END;
END;
{$ENDC}

FUNCTION SubConnections (var box1, box2 : boxtype; width : integer) : boolean;
{ returns TRUE ifof box1 and box2 overlap with at least a min width }
VAR
    interbox : boxtype;
    touched : boolean;
    sqrdist : integer;
BEGIN
    SubConnections := FALSE;
    interbox := Intersect (box1, box2, touched);
    if (touched) then BEGIN
        sqrdist := sqr (BoxLength (interbox)) + sqr (BoxWidth (interbox));
        if (sqrdist >= sqr (width)) then SubConnections := TRUE;
    END;
END;

PROCEDURE Connections (var boxes : boxaratype; width : integer);
VAR
    curmark, nummarks, mergedmark : integer;
    i, j, k : integer;
BEGIN
    nummarks := 0; {strictly speaking: misnomer}
    for i := 1 to boxes.cursize do boxes.vec[i].mark := 0;
    with boxes do for i := 1 to cursize do BEGIN
        if (vec[i].mark <> 0) then curmark := vec[i].mark
        else BEGIN
            nummarks := nummarks + 1;
            curmark := nummarks;
            vec[i].mark := curmark;
        END;
        for j := i+1 to cursize do BEGIN
            if (SubConnections (vec[i], vec[j], width)) then BEGIN
                { connection }
                if (vec[j].mark = 0) then vec[j].mark := curmark
                else BEGIN
                    { merge marks curmark and vec[j].mark, etc. }
                    mergedmark := curmark;
                    curmark := vec[j].mark;
                    for k := 1 to cursize do BEGIN
                        if (vec[k].mark = mergedmark) then 
                            vec[k].mark := curmark;
                    END; {for k}
                END; {else}
            END; {if}
        END; {for j}
    END; {for i}
END;


PROCEDURE TestSpacing (VAR boxes1, boxes2 : boxaratype;
                       minspacing : integer);
CONST
    VERBOSE = FALSE;
VAR
    i, j, k : integer;
BEGIN
    { now check that no bad pairs of boxes are not marked the same }
    {$IFC VERBOSE THEN}
        PrintAra (boxes);
    {$ENDC}

    for i := 1 to boxes1.cursize do BEGIN
        for j := 1 to boxes2.cursize do BEGIN
            if (not boxdist(boxes1.vec[i],boxes2.vec[j],minspacing)) then BEGIN
                if (boxes1.vec[i].mark <> boxes2.vec[j].mark) then BEGIN
                    DRCUnchecked ('Bad Spacing');
                    DRCPrintBox (boxes1.vec[i]);
                    DRCPrintBox (boxes2.vec[j]);
                END;
            END;
        END;
    END;
END;

PROCEDURE TestOverlap (boxes : boxaratype);
VAR
    i, j : integer;
    interbox : boxtype;
    hit : boolean;
BEGIN
    for i := 1 to boxes.cursize do for j := 1 to i-1 do BEGIN
        interbox := Intersect (boxes.vec[i], boxes.vec[j], hit);
        if (hit) then
            if ((BoxWidth (interbox)>0) or (BoxLength (interbox)>0)) then BEGIN
                DRCWarning ('Boxes Intersect on this layer');
                DRCPrintBox (boxes.vec[i]);
                DRCPrintBox (boxes.vec[j]);
            END;
    END;
END;

PROCEDURE AndBoxAra (VAR outboxes : boxaratype; in1, in2 : boxaratype);
VAR
    i, j : integer;
    interbox : boxtype;
    touched : boolean;
BEGIN
    outboxes.cursize := 0;
    for i := 1 to in1.cursize do for j := 1 to in2.cursize do BEGIN
        interbox := Intersect (in1.vec[i], in2.vec[j], touched);
        if (touched) then if (BoxArea (interbox) > 0) then BEGIN
            if (outboxes.cursize < outboxes.maxsize) then BEGIN
                outboxes.cursize := outboxes.cursize + 1;
                outboxes.vec[outboxes.cursize] := interbox;
            END
            else writeln ('**Unchecked, andboxara');
        END;
    END;
END;

PROCEDURE AddBoxToAra (var boxes : boxaratype; box : boxtype);
BEGIN
    if (boxes.cursize < boxes.maxsize) then BEGIN
        boxes.cursize := boxes.cursize + 1;
        boxes.vec[boxes.cursize] := box;
    END
    else BEGIN
        writeln ('Error, AddBoxToAra');
    END;
END;

FUNCTION ChopBox (var box1 : boxtype;
               right : integer; useright : boolean;
               left : integer; useleft : boolean;
               top : integer; usetop : boolean;
               bottom : integer; usebottom : boolean;
               VAR boxexists : boolean) : boxtype;
VAR
    ans : boolean;
    ansbox : boxtype;
BEGIN
    ans := TRUE;
    ansbox := box1;
    if (ans and useright) then BEGIN
        if (BoxRight (box1) <= right) then {no-op}
        else if (boxLeft (box1) >= right) then ans := FALSE
        else ansbox.brx := right;
    END;
    if (ans and useleft) then BEGIN
        if (BoxLeft (box1) >= left) then {no-op}
        else if (BoxRight (box1) <= left) then ans := FALSE
        else ansbox.tlx := left;
    END;
    if (ans and usetop) then BEGIN
        if (BoxTop (box1) <= top) then {no-op}
        else if (boxBottom (box1) >= top) then ans := FALSE
        else ansbox.tly := top;
    END;
    if (ans and usebottom) then BEGIN
        if (BoxBottom (box1) >= bottom) then {no-op}
        else if (BoxTop (box1) <= bottom) then ans := FALSE
        else ansbox.bry := bottom;
    END;
    ChopBox := ansbox;
    boxexists := ans;
END;

{FUNCTION Mergeable (var box1, box2 : boxtype; VAR updown : boolean) : boolean;
BEGIN
    if (((BoxBottom (box1) = BoxBottom (box2)) and
        ((BoxTop (box1) = BoxTop (box2))) then BEGIN
        
    END;
END;}

FUNCTION Merge (var box1, box2 : boxtype; VAR mergeable : boolean) : boxtype;
VAR
    interbox : boxtype;
    touched : boolean;
    ans : boxtype;
BEGIN
    mergeable := TRUE;
    interbox := Intersect (box1, box2, touched);
    if (not touched) then BEGIN
        mergeable := FALSE;
        Merge := box1;
        exit (Merge);
    END;

    if ((BoxBottom (box1) = BoxBottom (box2)) and
        (BoxTop (box1) = BoxTop (box2))) then BEGIN
            ans.brx := BoxRight (box1);
            if (ans.brx < BoxRight (box2)) then
                ans.brx := BoxRight (box2);
            ans.tlx := BoxLeft (box1);
            if (ans.tlx > BoxLeft (box2)) then
                ans.tlx := BoxLeft (box2);
            ans.bry := BoxBottom (box1);
            ans.tly := BoxTop (box1);
            Merge := ans;
            exit (Merge);
    END;
    
    if ((BoxRight (box1) = BoxRight (box2)) and
        (BoxLeft (box1) = BoxLeft (box2))) then BEGIN
            ans.tly := BoxTop (box1);
            if (ans.tly < BoxTop (box2)) then
                ans.tly := BoxTop (box2);
            ans.bry := BoxBottom (box1);
            if (ans.bry > BoxBottom (box2)) then
                ans.bry := BoxBottom (box2);
            ans.brx := BoxRight (box1);
            ans.tlx := BoxLeft (box1);
            Merge := ans;
    END;
END;

PROCEDURE AndNotBoxAra (VAR outboxes : boxaratype; in1, in2 : boxaratype);
CONST
    VERBOSE = FALSE;
VAR
    i, j, k, n : integer;
    top, bottom, right, left : integer;
    boxlist : boxaratype;
    frozencursize : integer;
    boxes8 : array [0..2, 0..2] of boxtype;
    okboxes8 : array [0..2, 0..2] of boolean;
    corn00, corn02, corn20, corn22 : boxtype;
    okcorn00, okcorn02, okcorn20, okcorn22 : boolean;
    rboxok, lboxok, mboxok : boxtype;
BEGIN
    InitBoxAra (outboxes);
    for i := 1 to in1.cursize do BEGIN
        InitBoxAra (boxlist);
        AddBoxToAra (boxlist, in1.vec[i]);
        for j := 1 to in2.cursize do BEGIN
            if (OverlapPred (in1.vec[i], in2.vec[j])) then BEGIN
                frozencursize := boxlist.cursize;
                top := BoxTop (in2.vec[j]);
                bottom := BoxBottom (in2.vec[j]);
                right := BoxRight (in2.vec[j]);
                left := BoxLeft (in2.vec[j]);
                {$IFC VERBOSE THEN}
                    writeln (' right left top bottom ',right,left,top,bottom);
                {$ENDC}
                for k := 1 to frozencursize do BEGIN
                    { find the 8 chopped up boxes and then merge them and
                      add them to the boxlist, deleting the current box. }
                    {$IFC VERBOSE THEN}
                        PrintBox (boxlist.vec[k]);
                    {$ENDC}
                    boxes8[0, 0] := ChopBox (boxlist.vec[k],
                        left, TRUE,
                        0, FALSE,
                        bottom, TRUE,
                        0, FALSE,
                        okboxes8[0, 0]);

                    boxes8[1, 0] := ChopBox (boxlist.vec[k],
                        right, TRUE,
                        left, TRUE,
                        bottom, TRUE,
                        0, FALSE,
                        okboxes8[1, 0]);

                    boxes8[2, 0] := ChopBox (boxlist.vec[k],
                        0, FALSE,
                        right, TRUE,
                        bottom, TRUE,
                        0, FALSE,
                        okboxes8[2, 0]);

                    boxes8[0, 1] := ChopBox (boxlist.vec[k],
                        left, TRUE,
                        0, FALSE,
                        top, TRUE,
                        bottom, TRUE,
                        okboxes8[0, 1]);

                    boxes8[2, 1] := ChopBox (boxlist.vec[k],
                        0, FALSE,
                        right, TRUE,
                        top, TRUE,
                        bottom, TRUE,
                        okboxes8[2, 1]);

                    boxes8[0, 2] := ChopBox (boxlist.vec[k],
                        left, TRUE,
                        0, FALSE,
                        0, FALSE,
                        top, TRUE,
                        okboxes8[0, 2]);

                    boxes8[1, 2] := ChopBox (boxlist.vec[k],
                        right, TRUE,
                        left, TRUE,
                        0, FALSE,
                        top, TRUE,
                        okboxes8[1, 2]);

                    boxes8[2, 2] := ChopBox (boxlist.vec[k],
                        0, FALSE,
                        right, TRUE,
                        0, FALSE,
                        top, TRUE,
                        okboxes8[2, 2]);
                    
                    { merge boxes }
                    okcorn00 := FALSE;
                    okcorn02 := FALSE;
                    okcorn20 := FALSE;
                    okcorn22 := FALSE;
                    
                    if                (okboxes8[0, 0] and okboxes8[0, 1]) then
                        corn00 := Merge (boxes8[0, 0], boxes8[0, 1], okcorn00);
                    if                (okboxes8[0, 2] and okboxes8[1, 2]) then
                        corn02 := Merge (boxes8[0, 2], boxes8[1, 2], okcorn02);
                    if                (okboxes8[2, 0] and okboxes8[1, 0]) then
                        corn20 := Merge (boxes8[2, 0], boxes8[1, 0], okcorn20);
                    if                (okboxes8[2, 2] and okboxes8[2, 1]) then
                        corn22 := Merge (boxes8[2, 2], boxes8[2, 1], okcorn22);
                    
                    {$IFC VERBOSE THEN}
                        if (okboxes8[0, 0]) then BEGIN
                            write ('0, 0  TRUE ');
                            PrintBox (boxes8[0, 0]);
                        END
                        else writeln ('0, 0 FALSE');

                        if (okboxes8[1, 0]) then BEGIN
                            write ('1, 0  TRUE ');
                            PrintBox (boxes8[1, 0]);
                        END
                        else writeln ('1, 0 FALSE');

                        if (okboxes8[2, 0]) then BEGIN
                            write ('2, 0  TRUE ');
                            PrintBox (boxes8[2, 0]);
                        END
                        else writeln ('2, 0 FALSE');

                        if (okboxes8[0, 1]) then BEGIN
                            write ('0, 1  TRUE ');
                            PrintBox (boxes8[0, 1]);
                        END                    
                        else writeln ('0, 1 FALSE');

                        if (okboxes8[2, 1]) then BEGIN
                            write ('2, 1  TRUE ');
                            PrintBox (boxes8[2, 1]);
                        END
                        else writeln ('2, 1 FALSE');

                        if (okboxes8[0, 2]) then BEGIN
                            write ('0, 2  TRUE ');
                            PrintBox (boxes8[0, 2]);
                        END
                        else writeln ('0, 2 FALSE');

                        if (okboxes8[1, 2]) then BEGIN
                            write ('1, 2  TRUE ');
                            PrintBox (boxes8[1, 2]);
                        END
                        else writeln ('1, 2 FALSE');

                        if (okboxes8[2, 2]) then BEGIN
                            write ('2, 2  TRUE ');
                            PrintBox (boxes8[2, 2]);
                        END
                        else writeln ('2, 2 FALSE');
                        
                        if (okcorn00) then BEGIN
                            write ('corn 00  TRUE ');
                            PrintBox (corn00);
                        END
                        else writeln ('corn 00 FALSE');

                        if (okcorn02) then BEGIN
                            write ('corn 02  TRUE ');
                            PrintBox (corn02);
                        END
                        else writeln ('corn 02 FALSE');

                        if (okcorn20) then BEGIN
                            write ('corn 20  TRUE ');
                            PrintBox (corn20);
                        END
                        else writeln ('corn 20 FALSE');

                        if (okcorn22) then BEGIN
                            write ('corn 22  TRUE ');
                            PrintBox (corn22);
                        END
                        else writeln ('corn 22 FALSE');
                    {$ENDC}

                    { add boxes to boxlist }
                    if (okcorn00) then AddBoxToAra (boxlist, corn00)
                    else if (okboxes8[0, 1]) then 
                        AddBoxToAra (boxlist, boxes8[0, 1]);
                    if (okcorn02) then AddBoxToAra (boxlist, corn02)
                    else if (okboxes8[1, 2]) then 
                        AddBoxToAra (boxlist, boxes8[1, 2]);
                    if (okcorn20) then AddBoxToAra (boxlist, corn20)
                    else if (okboxes8[1, 0]) then 
                        AddBoxToAra (boxlist, boxes8[1, 0]);
                    if (okcorn22) then AddBoxToAra (boxlist, corn22)
                    else if (okboxes8[2, 1]) then 
                        AddBoxToAra (boxlist, boxes8[2, 1]);
                    
                    { remove the original box }
                    if (boxlist.cursize <= 0) then writeln ('Error, imp')
                    else BEGIN
                        boxlist.vec[k] := boxlist.vec[boxlist.cursize];
                        boxlist.cursize := boxlist.cursize - 1;
                    END;
                END; {for k}
            END; {if}
        END; {for j}
        {$IFC VERBOSE THEN}
            writeln ;
            PrintAra (boxlist);
            writeln ;
        {$ENDC}
        for j := 1 to boxlist.cursize do
            AddBoxToAra (outboxes, boxlist.vec[j]);
    END; {for i}
END;

PROCEDURE FindTransistor (VAR trans : boxaratype;
                          var proposed : boxaratype
                          var contact : boxaratype);
VAR
    workcopy : boxaratype;
    isoktrans : array [0..MAXBOXES] of boolean;
    i, j : integer;
BEGIN
    workcopy := proposed;
    Connections (workcopy, 2);
    for i := 1 to proposed.cursize do isoktrans[i] := TRUE;

    for i := 1 to contact.cursize do BEGIN
        for j := 1 to proposed.cursize do BEGIN
            if (OverlapPred (contact.vec[i], proposed.vec[j])) then BEGIN
                isoktrans[j] := FALSE;
            END;
        END;
    END;
    
    { transitively complete isoktrans under relation connected }
    for i := 1 to proposed.cursize do BEGIN
        if (NOT isoktrans[i]) then BEGIN
            { mark all its connected transistors }
            for j := 1 to proposed.cursize do BEGIN
                if (workcopy.vec[j].mark = workcopy.vec[i].mark) then
                    isoktrans[j] := FALSE;
            END;
        END;
    END;

    trans.cursize := 0;
    for i := 1 to proposed.cursize do BEGIN
        if (isoktrans[i]) then AddBoxToAra (trans, proposed.vec[i]);
    END;
END;


CONST
    OPTIONFILE = 'drc.options';
VAR
    sif : siftype;
    i : integer;
    inchan : text;
    redirect : boolean;

BEGIN {main}
    { get options from options file }
    reset (inchan, OPTIONFILE);
    readln (inchan, i);
    if (i = 0) then redirect := FALSE
    else redirect := TRUE;
    if (redirect) then BEGIN
        close (output);
        rewrite (output, 'drc.out');
    END;

    { inits }
    zapsif (sif);

    ReadSif ('test.sif', sif);
    
    { test atomic minimum widths for wires }
    with sif do BEGIN
        { test for boxes intersecting non-trivially on one color }
        writeln ;
        writeln ('2) Overlap Testing...');
        writeln ('Testing Poly...');
        TestOverlap (poly);
        writeln ('Testing Metal...');
        TestOverlap (Metal);
        writeln ('Testing Diff...');
        TestOverlap (diff);
        writeln ('Testing Contact...');
        TestOverlap (contact);
        writeln ('Testing Implant...');
        TestOverlap (implant);
        writeln ('Testing Glass...');
        TestOverlap (glass);
        writeln ;
        
        InitBoxAra (DiffAndPoly);
        InitBoxAra (DiffNotPoly);
        AndBoxAra (DiffandPoly, Diff, Poly);
        AndNotBoxAra (DiffnotPoly, Diff, Poly);

        {$IFC TRUE THEN}
            writeln ;
            writeln ('Diff boxes:');
            PrintAra (Diff);
            writeln ;
            writeln ;
            writeln ('Poly boxes:');
            PrintAra (Poly);
            writeln ;
            writeln ;
            writeln ('DiffnotPoly boxes:');
            PrintAra (DiffnotPoly);
            writeln ;
        {$ENDC}

        InitBoxAra (Transistor);
        FindTransistor (Transistor, DiffAndPoly, contact);
        
        {$IFC TRUE THEN}
            writeln ;
            writeln ('Transistor boxes:');
            for i := 1 to transistor.cursize do BEGIN
                PrintBox (transistor.vec[i]);
            END;
            writeln ;
        {$ENDC}
        
        InitBoxAra (Depletion);
        InitBoxAra (Enhancement);
        AndBoxAra (Depletion, Transistor, Implant);
        AndNotBoxAra (Enhancement, Transistor, Implant);
        
        Connections (Poly, 2);
        Connections (Diff, 2);
        Connections (DiffNotPoly, 2);
        Connections (Metal, 2);
        Connections (Contact, 2);
        Connections (Transistor, 2);
        Connections (Implant, 1);

        writeln ;
        writeln ('TESTING WIDTHS...');
        writeln ;

        writeln ('Testing poly...');
        if (not (TestWidths (poly, 2))) then ;
        writeln ('Testing diff...');
        if (not (TestWidths (diff, 2))) then ;
        writeln ('Testing Metal...');
        if (not (TestWidths (Metal, 3))) then ;
        writeln ('Testing contact...');
        if (not (TestWidths (contact, 2))) then ;
        writeln ('Testing DiffnotPoly...');
        if (not (TestWidths (DiffnotPoly, 2))) then ;
        
        writeln ;
        writeln ('TESTING SPACING');
        writeln ;

        { for spacing testing, make sure that connections are global for
          these layers: poly, diff, contact, transistor, implant, diffnotpoly }
        for i := 1 to diff.cursize do 
            diff.vec[i].mark := diff.vec[i].mark + MAXBOXES * 2;
        for i := 1 to metal.cursize do 
            metal.vec[i].mark := metal.vec[i].mark + MAXBOXES * 4;
        for i := 1 to contact.cursize do 
            contact.vec[i].mark := contact.vec[i].mark + MAXBOXES * 6;
        for i := 1 to transistor.cursize do 
            transistor.vec[i].mark := transistor.vec[i].mark + MAXBOXES * 8;
        for i := 1 to implant.cursize do 
            implant.vec[i].mark := implant.vec[i].mark + MAXBOXES * 10;
        for i := 1 to diffnotpoly.cursize do 
            diffnotpoly.vec[i].mark := diffnotpoly.vec[i].mark + MAXBOXES * 12;
            

        writeln ('Testing poly-poly Spacing...');
        TestSpacing (poly, poly, 2);
        writeln ('Testing diff-diff Spacing...');
        TestSpacing (diff, diff, 3);
        writeln ('Testing metal-metal Spacing...');
        TestSpacing (metal, metal, 3);
{        writeln ('Testing diff-poly Spacing...');
        TestSpacing (diff, poly, 1);}
        writeln ('Testing contact-contact Spacing...');
        TestSpacing (contact, contact, 2);
        writeln ('Testing contact-transistor Spacing...');
        TestSpacing (contact, transistor, 2);
        writeln ('Testing implant-enhancement Spacing...');
        TestSpacing (implant, enhancement, 2);

    END;
    
    if (redirect) then close (output);
END.
{ }
