module BufferUtilities;

{*****************************************}
{
{        Pepper:  Spice Interim Editor
{        Buffer Manipulation Utilities
{        Richard Cohn
{        April 1, 1981
{
{*****************************************}


exports

imports RegionEditor from RegionEdit;
imports Perq_String from Perq_String;

procedure MakeNewWindow (FileName:  PathName);
procedure DestroyWindow;
procedure ReplaceWindow (FileName:  PathName);
procedure ChangeToWindow (WinNum:  Windex);
procedure ExpandWindow;
procedure CompressWindow;
function  BufferModified:  boolean;
procedure QuitWindow (ReplaceWin:  boolean; var ExitWindow:  boolean);
procedure RefreshScreen (EdWin:  WinRange; WinTitle:  String);

function  InitFilled (IsFile:  boolean):  boolean;
procedure ReInitFilled;
procedure WriteFile (FileName:  PathName; var Message: PString);
procedure DoWriteFile;
procedure DoSaveFile;
procedure FileRename (OldName: PathName);


private

imports FileUtils from FileUtils;
imports ScreenUtilities from EdScreen;
imports TextUtilities from EdText;
imports ReplayUtilities from EdReplay;
imports DiskIO from DiskIO;
imports AllocDisk from AllocDisk;
imports Memory from Memory;


{****************************************************************}

procedure FileRename (OldName: PathName);

var FId: FileId;
    Ignore: Integer;
    NewName:  PathName;
 
{********************************}

procedure FileDelete (Name: PathName);

var FId: FileId;

begin { FileDelete }
FId := FSLocalLookUp (Name, Ignore, Ignore);
if FId <> 0 then
    FSDelete (Name)
end; { FileDelete }
 
{********************************}

function GetDollars (OldName:  PathName):  PathName;

{ Append $'s to a file until a name is found that doesn't exist.  }

var FID:  FileID;
    f:    FileIndex;
    NewName:  PathName;

begin { Dollars }
NewName := Concat (OldName, '$');
GetDollars := NewName;
FID := FSLocalLookUp (NewName, Ignore, Ignore);
if FID = 0 then
    exit (GetDollars);
f := MinFile;
while (f < Files) and (FID <> IdFile [f].Index) do
    f := f + 1;
if FID <> IdFile [f].Index then
    FileDelete (NewName)
else
    GetDollars := GetDollars (NewName)
end; { Dollars }

{********************************}

begin { FileRename }
FId := FSLocalLookUp (OldName,Ignore,Ignore);
if FId <> 0 then
    begin
    NewName := GetDollars (OldName);
    FSRename (OldName, NewName)
    end
end; { FileRename }


{****************************************************************}

function StripPath (Name:  PathName):  PathName;

const 
    DirChar = '>';
    StrChar = ':';

var p:  integer;

begin
p := RevPosC (Name, DirChar);
if p > 0 then
    Delete (Name, 1, p)
else
    begin
    p := PosC (Name, StrChar);
    if p > 0 then
        Delete (Name, 1, p)
    end;
StripPath := Name
end;  { StripPath }


{****************************************************************}

procedure AdjustYBounds (ChangeWin:  pTextWindow);

begin
with ChangeWin^, Bound^ do
    begin
    HTextBound := yMin + ThumbHeight;
    yTBarOrigin := yMin + yTBarOffset;
    yThumb := yTBarOrigin + TBarHeight;
    yHome := HTextBound + BoxMargin;
    LastLine := (yMax - yHome - BoxMargin) div LineHeight - 1;
    yLastLine := (LastLine + 1) * LineHeight + yHome;
    if CurLine > LastLine then
        CurLine := LastLine
    end
end; { AdjustYBounds }


{****************************************************************}

procedure MakeNewWindow (FileName:  PathName);

var 
    OldWin:  pTextWindow;
    NewWinBound:  pWindow;
    w:  integer;
    yOldMax:  integer;

begin
w := MinWindex;
while (Winary [w] <> nil) and (w <= MaxWindex) do
    w := w + 1;
if Winary [w] <> nil then
    begin
    Warn ('Maximum number of windows already in use');
    exit (MakeNewWindow)
    end;
new (NewWinBound);
with CurWin^, Bound^ do
    begin
    if yMax - yMin <= 2 * MinWindowHeight then
        begin
        Warn ('Current window too small to split');
        exit (MakeNewWindow)
        end;
    NewWinBound^ := Bound^;
    yOldMax := yMax;
    yMax := (yMin + yMax) div 2 - 1;
    NewWinBound^.yMin := yMax + 1
    end;
AdjustYBounds (CurWin);
OldWin := CurWin;
Winary[w] := SetUpWindow (NewWinBound, '', false, true, true, true, false);
dispose (NewWinBound);
CurWin := Winary [w];
OffPointer;
with CurWin^ do
    begin
    WinNum := w;
    Name := FileName;
    if not InitFilled (true) then
        begin
        Prompt ('Make window aborted');
        ChangeTextWindow (OldWin);
        CurWin^.Bound^.yMax := yOldMax;
        AdjustYBounds (CurWin);
        dispose (Winary [w]);
        Winary [w] := nil;
        OnPointer;
        exit (MakeNewWindow)
        end;
    ScreenFirst := Add (FilledFirst,2);
    ScreenLast := FilledLast;
    with Ln [-1] do
        begin
        Start := ScreenFirst;
        Finish := ScreenFirst;
        Length := 0
        end;
    Next := OldWin^.Next;
    Next^.Prev := CurWin;
    Prev := OldWin;
    OldWin^.Next := CurWin
    end;
ChangeTextWindow (OldWin);
RefreshTextWindow;
ChangeTextWindow (Winary[w]);
RefreshTextWindow;
end; { MakeNewWindow }


{****************************************************************}

procedure WriteFile (FileName:  PathName; var Message: PString);

const
    ChInBlock = MaxLength div 2;

var Buffer: array [MemPage] of pSwapBuffer;
      BufBlock: MemPage;
      Block: Integer;
      Filled: Integer;
      FileNumber: Integer;
      NChars: Integer;
      Ch: Char;
      Ignore: Integer;
      OldFirstScreen, OldLastScreen:  Position;
      w:  Windex;
      i:  integer;
   
{********************************}
   
procedure Send( B: pSwapBuffer; Start, Length: Integer );

var I, N: Integer;

begin { Send }
if Filled + Length > MaxLength then
    begin
    N := MaxLength - Filled;
    for I := 0 to N - 1 do
        Buffer [BufBlock]^[Filled + I] := B^[Start + I];
    if BufBlock = MaxMemPage then
        begin
        for i := 0 to MaxMemPage do
            FSBlkWrite (FileNumber, Block + i, Recast (Buffer [i], pDirBlk));
        Block := Block + MaxMemPage + 1;
        BufBlock := 0
        end
    else
        BufBlock := BufBlock + 1;
    Start := Start + N;
    Length := Length - N;
    Filled := 0;
    UpdateThumbBar
    end;
for I := 0 to Length - 1 do
    Buffer [BufBlock]^[Filled+I] := B^[Start+I];
Filled := Filled + Length
end; { Send }
   
{********************************}
   
begin { WriteFile }
if DEBUG [2] then Status (Concat ('Enter WriteFile:  FileName = ', FileName));
   Message := '';
   if FileName = '' then
    begin
     Message := Concat (Concat ('Can''t write to ', FileName), 
         '.  Exit with new name.');
     Exit(WriteFile)
    end;
   FileNumber := FSLookUp (FileName, Ignore, Ignore);
   if FileNumber <> 0 then
        begin
        for w := MinWindex to MaxWindex do
            if Winary [w] <> nil then
                if (FileNumber = IdFile [Winary [w]^.FIndex].Index) and
                    (w <> CurWin^.WinNum) then
                    begin
                    Message := Concat ('Can''t write to ', FileName);
                    Exit(WriteFile)
                    end;
        FileRename (FileName);
        end;
   FileNumber := FSEnter (FileName);
   if FileNumber = 0 then
        begin 
        Message := Concat (Concat ('Can''t write to ', FileName), 
            '.  Exit with new name.');
        Exit(WriteFile)
        end;
   Prompt (Concat ('Writing ',FileName));
   Attach (Cursor1, CurWin^.FilledFirst, ReadCursor);
   OldFirstScreen := CurWin^.ScreenFirst;
   OldLastScreen := CurWin^.ScreenLast;
   CurWin^.ScreenFirst := Cursor1.Pos;
   CurWin^.ScreenLast  := OldLastScreen;
   Add1C(Cursor1);
   Add1C(Cursor1);
   if BufferSeg = -1 then { not yet initialized }
        CreateSegment (BufferSeg, MaxMemPage + 1, 1, MaxMemPage + 1);
   BufBlock := 0;
   Block := 0;
   Filled := 0;
   for i := 0 to MaxMemPage do
        New (BufferSeg, ChInBlock, Buffer [i]);
   while not EOT(Cursor1.Pos) do
    begin NChars := Cursor1.Pos.Chunk^.Length - Cursor1.Pos.Offset;
     Send (Txt [Cursor1.ChPage].Buffer,
           Cursor1.Pos.Offset + Cursor1.Pos.Chunk^.First,
           NChars);
     AddC (Cursor1, NChars);
     CurWin^.ScreenLast := Cursor1.Pos;
    end;
   UpdateThumbBar;
   if (Filled <> 0) or (BufBlock <> 0) then
       begin
       for i := 0 to BufBlock do
           FSBlkWrite (FileNumber, Block + i, Recast (Buffer [i], pDirBlk));
       Block := Block + BufBlock + 1;
       end
   else
       Filled := MaxLength;
   FSClose(FileNumber,Block,Filled*8);
   Prompt (Concat(FileName, ' written'));
   for i := 0 to MaxMemPage do
        Dispose (Buffer [i]);
   Detach(Cursor1);
   CurWin^.ScreenFirst := OldFirstScreen;
   CurWin^.ScreenLast  := OldLastScreen;
   UpdateThumbBar;
   if DEBUG [2] then Status ('Exit WriteFile')
  end { WriteFile };


{****************************************************************}

function BufferModified:  boolean;

{ Checks if buffer in CurWin has been modified by comparing its chunk
{ structure to what a virgin chunk structure should look like.  }
    
var 
    P:          Position;
    C:          pChunk;
    Page:       DiskPage;
    F:          FileIndex;

begin
BufferModified := true;
with CurWin^ do
    begin
    if FilledLast.Chunk^.OrderC + FilledLast.Offset <> OrgLastC then
        exit (BufferModified);
    P := Add (CurWin^.FilledFirst, 2)
    end;
C := P.Chunk;
Page := C^.CPage;
F := GetPFile (Page);
if (F = SwapFile) or (Page <> IdFile [F].MinPage) then
    begin
    BufferModified := P <> CurWin^.FilledLast;
    exit (BufferModified)
    end;
while (C^.Length = MaxLength) and (C^.CPage = Page) do
    begin
    C := C^.Next;
    Page := Page + 1
    end;
{ Now either a change has been made or at end.  If at end, either at last
{ page, or if last page was of length MaxLength then at FilledLast. }
if C^.Next <> nil then
    begin
    if C^.CPage = Page then
        C := C^.Next
    end
else if Page <> IdFile [F].MaxPage + 1 then
    exit (BufferModified);
BufferModified := C^.Next <> nil
end; { BufferModified }


{****************************************************************}

procedure MergeWindow;

{ Merge the space occupied by CurWin with that of its upper or lower neighbor.
{ If CurWin has no neighbors then do nothing.  }

var NewWin:  pTextWindow;

begin { MergeWindow }
NewWin := nil;
if CurWin^.Prev^.Bound^.yMax < CurWin^.Bound^.yMax then
    { merge with window above }
    begin
    NewWin := CurWin^.Prev;
    NewWin^.Next := CurWin^.Next;
    CurWin^.Next^.Prev := NewWin;
    NewWin^.Bound^.yMax := CurWin^.Bound^.yMax;
    end
else if CurWin^.Next^.Bound^.yMin > CurWin^.Bound^.yMin then
    { merge with window below }
    begin
    NewWin := CurWin^.Next;
    NewWin^.prev := CurWin^.prev;
    CurWin^.Prev^.Next := NewWin;
    NewWin^.Bound^.yMin := CurWin^.Bound^.yMin;
    end;
if NewWin <> nil then
    AdjustYBounds (NewWin);
Winary [CurWin^.WinNum] := nil;
dispose (CurWin);
if NewWin = nil then
    CurWin := nil
else
    begin
    ChangeTextWindow (NewWin);
    RefreshTextWindow
    end
end; { MergeWindow }


{******************************************************************}

procedure QuitWindow (ReplaceWin:  boolean; var ExitWindow:  boolean);

{ called by DestroyWindow and EditQuit to dispose of windows }

      
var 
    OldWin:     pTextWindow;
    FileName:   PathName;
    Success:    boolean;
    Message:    PString;
    Ch:         char;
    w:          Windex;
    FID:        integer;
    b:          KillBMin..KillBMax;
    Double:     boolean;
    QuitSet:    set of char;        


{********************************}

procedure CleanUpWindow;

begin
if CurWin = SelectWindow then
    UnSelect;
with Bufary [KillB] do
    if first <> NilPosition then
        if GetChFile (first.Chunk, last.Chunk) = CurWin^.WinNum then
            begin
            PrevKill := NilPosition;
            NextKill := NilPosition
            end;
Collect (CurWin^.FilledFirst, CurWin^.FilledLast);
if not ReplaceWin then
    MergeWindow
end; { CleanUpWindow }


{********************************}

begin { QuitWindow }
if DEBUG [2] then
    Status (Concat ('Enter QuitWindow:  FullName = ', CurWin^.FullName));
OffPointer;
OldWin := CurWin;
ExitWindow := true;
if BufferModified then
    begin
    if Replay = NotReplaying then
        Ch := ' '
    else
        Ch := GetTrnByte;
    if Replay = NotReplaying then
        QuitSet := ['U', 'W', 'E', 'R']
    else
        QuitSet := ['U', 'W', 'E'];
    repeat 
    ChangeTextWindow (PromptWindow);
    if Ch <> 'R' then
        repeat
        ClearLine (0,0,0);
        if Replay = NotReplaying then
            write ('Quit Window', OldWin^.WinNum:2,
            ' (Update, Write new file,', ' Exit, Return):  [Update]')
        else
            write ('Quit Window', OldWin^.WinNum:2,
            ' (Update, Write new file,', ' Exit):  [Update]');
        repeat LookForCommand (CanvKeybd);
        until NewEvent.Cmd <> NullCmd;
        Ch := chr (LAND (#177, ord (NewEvent.Ch)));
        if Ch = CR then
            Ch := 'U'
        else if Ch = DEL then
            Ch := 'R'
        else if Ch in ['a'..'z'] then 
            Ch := Chr(Ord(Ch) - Ord('a') + Ord('A'));
        if not (Ch in QuitSet) then
            write (BEL)
        until Ch in QuitSet;
    Message := '';
    Success := true;
    if Ch = 'R' then
        begin
        OnPointer;
        ExitWindow := false;
        ImmedPrompt := true
        end
    else if Ch = 'W' then
        begin 
        Prompt ('Enter file name: ');
        MovePencil (0, 19);  { 19 characters in previous prompt }
        Readln(FileName);
        ChangeTextWindow (OldWin);
        if FileName = '' then
            begin
            Error ('Write file aborted');
            Success := false
            end
        else
            begin
            WriteFile(FileName, Message);
            if Message <> '' then
                begin
                Error (Message);
                Success := false
                end
            end
        end
    else if Ch = 'U' then
        begin
        FID := IdFile [OldWin^.FIndex].Index;
        if FID <> 0 then
            begin
            Double := false;
            for w := MinWindex to MaxWindex do
                if Winary [w] <> nil then
                    Double := Double or (FID=IdFile [Winary[w]^.FIndex].Index)
                        and (w <> OldWin^.WinNum);
            if Double then
                begin
                Prompt
                   ('Another window exists on this file--write this one? [Y]');
                repeat LookForCommand (CanvKeybd)
                until NewEvent.Cmd <> NullCmd;
                Success := NewEvent.Ch in ['y', 'Y', CR];
                if Success then
                    for w := MinWindex to MaxWindex do
                        if Winary [w] <> nil then
                            if (FID = IdFile [Winary[w]^.FIndex].Index) and
                                (w <> OldWin^.WinNum) then
                                begin
                                ChangeTextWindow (Winary [w]);
                                CleanUpWindow
                                end
                end { if Double }
            end; { if FID... }
        if Success then
            begin
            ChangeTextWindow (OldWin); 
            WriteFile(OldWin^.FullName, Message);
            if Message <> '' then
                begin
                Error (Message);
                Success := false
                end
            end { if Success... }
        end
    else if Ch = 'E' then
        begin
        Prompt 
          ('File has been modified--are you sure you want to exit? [Yes]');
        ImmedPrompt := true;
        repeat LookForCommand (CanvKeybd)
        until NewEvent.Cmd <> NullCmd;
        Success := (chr (LAND (#177,ord (NewEvent.Ch))) in ['y', 'Y', CR]);
        end
    until Success;
    if Replay = NotReplaying then
        if Ch = 'R' then
            SendTrnByte (Ch)
        else
            SendTrnByte (' ')
    end; { if BufferModified }
ChangeTextWindow (OldWin);
if ExitWindow then
    CleanUpWindow;
NeedPrompt := true;
end;  { QuitWindow }


{****************************************************************}

procedure DestroyWindow;

var
    w:           Windex;
    ExitWindow:  boolean;

begin
OffPointer;
QuitWindow (false, ExitWindow);
if not ExitWindow then
    begin
    Prompt ('Delete window aborted');
    OnPointer;
    exit (DestroyWindow)
    end;
if CurWin = nil then { no neighbors }
    begin
    w := MinWindex;
    while Winary [w] = nil do
        w := w + 1;
    if w <= MaxWindex then
        ChangeTextWindow (Winary [w])
    else
        ExitEdit := true
    end
end; { DestroyWindow }


{****************************************************************}

procedure ReplaceWindow (FileName:  PathName);

var
    OldWin, NewWin:  pTextWindow;
    ExitWindow:  boolean;
    w:  Windex;

begin
with CurWin^ do begin
OffPointer;
w := WinNum;
QuitWindow (true, ExitWindow);
if not ExitWindow then
    begin
    Prompt ('Replace window aborted');
    NeedPrompt := true;
    ImmedPrompt := true;
    OnPointer;
    exit (ReplaceWindow)
    end
else
    begin
    Name := FileName;
    if not InitFilled (true) then
        begin
        MergeWindow;
        if CurWin = nil then { no neighbors }
            begin
            w := MinWindex;
            while Winary [w] = nil do
                w := w + 1;
            ChangeTextWindow (Winary [w]);
            if w > MaxWindex then { w = PromptWindow^.WinNum }
                ExitEdit := true
            end;
        Prompt ('Read file aborted');
        NeedPrompt := true;
        ImmedPrompt := true;
        exit (ReplaceWindow)
        end;
    ScreenFirst := Add (FilledFirst,2);
    ScreenLast := FilledLast;
    with Ln [-1] do
        begin
        Start := ScreenFirst;
        Finish := ScreenFirst;
        end;
    nChanges := 0;
    nMoves := 0;
    Mark := NilPosition;
    RefreshTextWindow;
    MoveTextPointer (0,0)
    end;
end { with CurWin^ }
end; { ReplaceWindow }


{****************************************************************}

procedure ChangeToWindow (WinNum:  Windex);

var NewWin:  pTextWindow;

begin
NewWin := Winary [WinNum];
if NewWin = nil then
    begin
    Warn ('Window doesn''t exist');
    exit (ChangeToWindow)
    end;
ChangeTextWindow (NewWin)
end;  { ChangeToWindow }


{****************************************************************}

procedure ExpandWindow;

var ShrinkWin, BigWin:  pTextWindow;
    SizeChange:  integer;
    PrevHeight, NextHeight:  integer;

begin
SizeChange := RepeatCount * LineHeight;
ShrinkWin := nil;
with CurWin^, Bound^ do
    begin
    if Prev^.Bound^.yMax < yMax then
        { window exists above }
        PrevHeight := Prev^.Bound^.yMax - Prev^.Bound^.yMin - 
            MinWindowHeight
    else
        PrevHeight := 0;
    if Next^.Bound^.yMax > yMax then
        { window exists below }
        NextHeight := Next^.Bound^.yMax - Next^.Bound^.yMin - 
            MinWindowHeight
    else if PrevHeight = 0 then
        begin
        Warn ('Can''t expand window--no neighboring windows');
        exit (ExpandWindow)
        end
    else
        NextHeight := 0;
    if (PrevHeight >= NextHeight) and (PrevHeight > 0) then
        begin
        ShrinkWin := Prev;
        SizeChange := ShrinkWin^.Bound^.yMax - ShrinkWin^.Bound^.yMin - 
            MinWindowHeight;
        SizeChange := (SizeChange div LineHeight) * LineHeight;
        if SizeChange > RepeatCount * LineHeight then
            SizeChange := RepeatCount * LineHeight;
        ShrinkWin^.Bound^.yMax := ShrinkWin^.Bound^.yMax - SizeChange;
        yMin := yMin - SizeChange
        end
    else if NextHeight > 0 then
        begin
        ShrinkWin := Next;
        SizeChange := ShrinkWin^.Bound^.yMax - ShrinkWin^.Bound^.yMin - 
            MinWindowHeight;
        SizeChange := (SizeChange div LineHeight) * LineHeight;
        if SizeChange > RepeatCount * LineHeight then
            SizeChange := RepeatCount * LineHeight;
        ShrinkWin^.Bound^.yMin := ShrinkWin^.Bound^.yMin + SizeChange;
        yMax := yMax + SizeChange
        end
    else
        SizeChange := 0;
    if SizeChange = 0 then
        begin
        Warn ('Can''t expand window--neighboring windows too small');
        exit (ExpandWindow)
        end
    end; { with CurWin^, Bound^ }
AdjustYBounds (ShrinkWin);
AdjustYBounds (CurWin);
BigWin := CurWin;
ChangeTextWindow (ShrinkWin);
RefreshTextWindow;
ChangeTextWindow (BigWin);
RefreshTextWindow
end; { ExpandWindow }


{****************************************************************}

procedure CompressWindow;

var GrowWin, SmallWin:  pTextWindow;
    SizeChange:  integer;
    
begin
SizeChange := RepeatCount * LineHeight;
GrowWin := nil;
with CurWin^, Bound^ do
    begin
    if yMax - yMin < MinWindowHeight + SizeChange then
        begin
        SizeChange := yMax - yMin - MinWindowHeight;
        SizeChange := (SizeChange div LineHeight) * LineHeight;
        if SizeChange <= 0 then
            begin     
            Warn ('Window too small--can''t shrink it');
            exit (CompressWindow)
            end
        end;
    if Prev^.Bound^.yMax < yMax then
        { window exists above }
        begin
        GrowWin := Prev;
        GrowWin^.Bound^.yMax := GrowWin^.Bound^.yMax + SizeChange;
        Bound^.yMin := Bound^.yMin + SizeChange
        end
    else if Next^.Bound^.yMax > yMax then
        { window exists below }
        begin
        GrowWin := Next;
        GrowWin^.Bound^.yMin := GrowWin^.Bound^.yMin - SizeChange;
        Bound^.yMax := Bound^.yMax - SizeChange
        end
    else
        begin
        Warn ('Can''t shrink window--no neighboring windows');
        exit (CompressWindow)
        end;
    end;
AdjustYBounds (GrowWin);
AdjustYBounds (CurWin);
SmallWin := CurWin;
ChangeTextWindow (GrowWin);
RefreshTextWindow;
ChangeTextWindow (SmallWin);
RefreshTextWindow
end; { ExpandWindow }


{*****************************************************************}

procedure RefreshScreen (EdWin:  WinRange; WinTitle:  String);

var OldWin:  pTextWindow;
    w:  integer;

begin
if DEBUG [2] then Status  ('Enter RefreshScreen');
RefreshWindow (EdWin);
ChangeTitle (WinTitle);
RefreshPromptWindow;
OldWin := CurWin;
for w := MinWindex to MaxWindex do
    if Winary [w] <> nil then
        begin
        ChangeTextWindow (Winary[w]);
        RefreshTextWindow
        end;
ChangeTextWindow (OldWin);
if DEBUG [2] then Status  ('Exit RefreshScreen')
end; { RefreshScreen }
  

{*************************************************************************}
  
function InitFilled (IsFile:  boolean):  boolean;
  
var EmptyDoc:  boolean;
    OldWin:  pTextWindow;
    IdBlock: ptrDiskBuffer;
    Disk, Part: Integer;
    FileNumber: FileId;
    Blocks, Bits: Integer;


{****************************}

procedure InitReplayDoc;

begin { InitReplayDoc }
if DEBUG [2] then Status('Enter InitReplayDoc');
if GetTrnByte = DEL then
    begin
    Error ('Read file aborted');
    exit (InitFilled)
    end;
FileNumber := GetTrnWord;
if FileNumber = 0 then
    begin
    Error ('Replay transcript of file creation');
    CurWin^.Name := ''
    end
else
    begin
    Disk := WhichDisk(FileIdToSegId(FileNumber));
    if not DiskTable[Disk].InUse then
        begin
        Error ('Correct device no longer mounted');
        Exit (InitFilled)
        end;
    Part := WhichPartition(FileIdToSegId(FileNumber));
    if Part <> 0 then
        if not PartTable[Part].PartInUse then
            Part := 0;
    if Part = 0 then
        begin
        Error ('Correct partition no longer mounted');
        Exit (InitFilled)
        end;
    New(0,256,IdBlock);
    FSBlkRead(FileNumber,-1,Recast(IdBlock,pDirBlk));
    CurWin^.Name := DiskTable[Disk].RootPartition;
    AppendChar(CurWin^.Name, ':');
    AppendString(CurWin^.Name, PartTable[Part].PartName);
    AppendChar(CurWin^.Name, '>');
    AppendString(CurWin^.Name, IdBlock^.FSData.FileName);
    Blocks := IdBlock^.FSData.FileBlocks;
    Bits := IdBlock^.FSData.FileBits;
    Dispose(IdBlock);
    Error (Concat ('Replay transcript on ', CurWin^.Name))
    end; { else not (FileNumber = 0) }
if DEBUG [2] then Status('Enter InitReplayDoc')
end; { InitReplayDoc }

{*****************************} 

procedure ReadFile;

const nExtensions = 4;

var Block, i, j: integer;
    NameDotExtension, PathAndName: PathName;
    P, Q: Position;
    T: pChunk;
    TxtIndex: integer;

begin { ReadFile }
if DEBUG [2] then Status ('Enter ReadFile');
if Replay = NotReplaying then
    begin
    nameDotExtension := CurWin^.Name;
    fileNumber := FSExtSearch (FSSysSearchList, '  .Pas .Cmd .Mss ',
        nameDotExtension, blocks, bits);
    if DEBUG [2] then StatusNumber ('ReadFile:  FileNumber =', FileNumber);
    SendTrnWord (FileNumber)
    end { if Replay = NotReplaying ... }
else
    NameDotExtension := CurWin^.Name;
if FileNumber = 0 then
    with CurWin^ do 
        begin
        FullName := Name;
        FixFileName (FullName, true);
        Name := StripPath (Name);
        EmptyDoc := true;
        FIndex := 0;
        Blocks := 0
        end
else { not (FileNumber = 0) }
    begin
    FixFileName (NameDotExtension, true);
    FSRemoveDots (NameDotExtension);
    CurWin^.FullName := NameDotExtension;
    CurWin^.Name := StripPath (NameDotExtension);
    Prompt (Concat ('Reading ', CurWin^.FullName));
    i := MinFile;
    while (IdFile [i].Index <> FileNumber) and (i < Files) do
        i := i + 1;
    if IdFile [i].Index = FileNumber then
        CurWin^.FIndex := i
    else
        begin
        if Files = MaxFile then
            begin
            Error ('Reached max number of files--suggest quit');
            exit (InitFilled)
            end;
        Files := Files + 1;
        CurWin^.FIndex := Files;
        with IdFile [CurWin^.FIndex] do
            begin
            Index := FileNumber;
            MinPage := FilePages;
            FilePages := FilePages + Blocks;
            if Blocks = 0 then
                MaxPage := MinPage
            else
                MaxPage := FilePages - 1
            end; { with }
        if FilePages >= IdFile [SwapFile].MinPage then
            begin
            Error ('Out of storage--quit now!!');
            exit (InitFilled)
            end
        end { else }
    end; { else }
LeftPart.Chunk := nil;
RightPart.Chunk := nil;
for Block := 0 to Blocks - 1 do
    begin
    New(T);
    with T^ do
        begin
        CPage := IdFile [CurWin^.FIndex].MinPage + Block;
        First := 0;
        Length := 512;
        OrderP := 0;
        OrderC := 0;
        Next := nil;
        Prev := nil
        end;
    if LeftPart.Chunk = nil then
        begin
        LeftPart.Chunk := T;
        LeftPart.Offset := 0
        end
    else
        begin
        RightPart.Chunk^.Next := T;
        T^.Prev := RightPart.Chunk
        end;
    RightPart.Chunk := T;
    RightPart.Offset := 0;
    end;  { for Block... }
if LeftPart.Chunk = nil then
    begin
    LeftPart := EmptyFirst;
    RightPart := EmptyFirst
    end
else
    begin
    T^.Length := Bits div 8;
    RightPart.Offset := T^.Length - 1
    end;
EmptyDoc := EQ(LeftPart,RightPart);
if DEBUG [2] then Status ('Exit ReadFile')
end { ReadFile }; 
  

{*********************************}

 
begin { InitFilled }
if DEBUG [2] then Status('Enter InitFilled');
InitFilled := false;
if Replay <> NotReplaying then
    InitReplayDoc;
if CurWin^.Name = '' then
    begin
    OldWin := CurWin;
    ChangeTextWindow (PromptWindow);
    Prompt ('Enter name (<RETURN> aborts):');
    MovePencil (0,31);  { No. of char's in prompt + 2 }
    readln (OldWin^.Name);
    ChangeTextWindow (OldWin);
    if CurWin^.Name = '' then
        begin
        SendTrnByte (DEL);  { read file aborted }
        exit (InitFilled)
        end
    end;
SendTrnByte (INS);  { read file not aborted }
if IsFile then
    ReadFile
else
    begin
    Error ('ReadDoc not yet implemented');
    exit (InitFilled)
    end;
with CurWin^ do
    begin
    Attach(Cursor1,EmptyFirst,WriteCursor);
    FilledFirst := Cursor1.Pos;
    Cursor1.Ch := CR;
    Add1C(Cursor1);
    Cursor1.Ch := LF;
    Add1C(Cursor1);
    FilledLast := Cursor1.Pos;
    Cursor1.Ch := Etx;
    Add1C(Cursor1);
    EmptyFirst := Cursor1.Pos;
    Detach(Cursor1);
    Split(EmptyFirst);
    if EmptyDoc then
        Prompt (Concat ('Creating new document ', CurWin^.FullName))
    else
        begin
        Split(FilledLast);
        Join(FilledFirst,LeftPart);
        Join(RightPart,FilledLast)
        end;
    FilledFirst.Chunk^.OrderP := 0;
    FilledFirst.Chunk^.OrderC := 0;
    ReOrder(FilledFirst);
    OrgLastC := FilledLast.Chunk^.OrderC + FilledLast.Offset;
    Mark := NilPosition;
    ScreenFirst := Add(FilledFirst,2);
    ScreenLast  := FilledLast;
    with Ln [-1] do
        begin
        Start := ScreenFirst;
        Finish := ScreenFirst;
        Length := 0
        end
    end; { with }
InitFilled := true;
if DEBUG [2] then Status('Exit InitFilled')
end { InitFilled };
  

{*************************************************************************}

procedure ReInitFilled;

{ Fix up the positions that have been changed by writing out the file
{ under its current name.  }

var
    L:  LineIndex;
    b:  BufRange;
    SF, M, BSF, BSL:  Position;
    Ignore:  boolean;

{********************************}

function FixUp (ModP:  Position):  Position;

var P:  pChunk;
    Q:  Position;

begin { FixUp }
if (ModP.Chunk^ <> NilPosition.Chunk^)
    or (ModP.Offset <> NilPosition.Offset) then
    begin
    P := CurWin^.FilledFirst.Chunk;
    while P^.OrderP < ModP.Chunk^.OrderP do
        P := P^.Next;
    Q.Chunk := P;
    Q.Offset := 0;
    FixUp := Add (Q, ModP.Offset + ModP.Chunk^.OrderC - Q.Chunk^.OrderC)
    end
else
    FixUp := NilPosition
end; { FixUp }

{********************************}

begin { ReInitFilled }
NewChunk (SF.Chunk);
SF.Chunk^ := CurWin^.ScreenFirst.Chunk^;
SF.Offset := CurWin^.ScreenFirst.Offset;
NewChunk (M.Chunk);
M.Chunk^ := CurWin^.Mark.Chunk^;
M.Offset := CurWin^.Mark.Offset;
if SelectWindow = CurWin then
    begin
    NewChunk (BSF.Chunk);
    BSF.Chunk^ := Bufary [SelectB].First.Chunk^;
    BSF.Offset := Bufary [SelectB].First.Offset;
    NewChunk (BSL.Chunk);
    BSL.Chunk^ := Bufary [SelectB].Last.Chunk^;
    BSL.Offset := Bufary [SelectB].Last.Offset
    end;
with Bufary [KillB] do
    if first <> NilPosition then
        if GetChFile (first.Chunk, last.Chunk) = CurWin^.WinNum then
            begin
            PrevKill := NilPosition;
            NextKill := NilPosition
            end;
Collect (CurWin^.FilledFirst, CurWin^.FilledLast);
Ignore := InitFilled (true);
CurWin^.ScreenFirst := FixUp (SF);
CurWin^.Mark := FixUp (M);
Dispose (SF.Chunk);
Dispose (M.Chunk);
if SelectWindow = CurWin then
    begin
    Bufary [SelectB].First := FixUp (BSF);
    Bufary [SelectB].Last := FixUp (BSL);
    Dispose (BSF.Chunk);
    Dispose (BSL.Chunk)
    end;
RefreshTextWindow
end; { ReInitFilled }
  

{*************************************************************************}

procedure DoWriteFile;

var 
    FileName:  PathName;
    Message:   PString;
    OldWin:    pTextWindow;

begin { DoWriteFile }
OffPointer;
OldWin := CurWin;
if Replay = NotReplaying then
    begin
    ChangeTextWindow (PromptWindow);
    Prompt ('Enter file name: ');
    MovePencil (0, 19);  { 19 characters in previous prompt }
    Readln(FileName);
    ChangeTextWindow (OldWin);
    if FileName = '' then
        begin
        SendTrnByte (DEL);
        Prompt ('Write file aborted');
        NeedPrompt := true;
        OnPointer;
        exit (DoWriteFile)
        end;
    SendTrnByte (INS);
    WriteFile (FileName, Message)
    end
else
    begin
    if GetTrnByte = DEL then
        begin
        OnPointer;
        exit (DoWriteFile)
        end;
    Message := 'Replaying--file not written';
    ReInitFilled
    end;
if Message <> '' then
    Error (Message)
else
    begin
    if FileName <> CurWin^.Name then
        begin
        CurWin^.Name := FileName;
        FixFileName (FileName, true);
        CurWin^.FullName := FileName
        end;
    ReInitFilled
    end;
OnPointer;
ImmedPrompt := true;
NeedPrompt := true
end; { DoWriteFile }  
  

{*************************************************************************}

procedure DoSaveFile;

{ Save the file in the current window using its FullName.  }

var
    message:  PString;

begin { DoSaveFile }
CurWin^.nChanges := 0;
CurWin^.Name := CurWin^.FullName;  { for InitFilled to work }
if Replay = NotReplaying then
    WriteFile (CurWin^.FullName, Message)
else
    begin
    Message := '';
    Error ('Replaying--file not written')
    end;
if Message = '' then
    ReInitFilled
else
    Error (Message);
NeedPrompt := true;
ImmedPrompt := true
end. { DoSaveFile }
